# 23 "asllib/Lexer.mll"
 

open Tokens

open Error

module type CONFIG = sig
end

let reserved_keywords = []

let is_reserved_keyword: string -> bool =
  let tbl: (string, unit) Hashtbl.t = Hashtbl.create (List.length reserved_keywords) in
  let () = List.iter (fun s -> Hashtbl.add tbl s ()) reserved_keywords in
  fun s -> Hashtbl.mem tbl s

(* Get a Tokens.token from the string name of the token.
   A useful utility when working with cmly files.

   Note that this set only contains simple enumerative types, not compound types

   This function's exhaustiveness is guaranteed by the test in tests/Lexer. *)
let token_of_string =
 let s t = Some t in
 function
 | "ACCESSOR"           -> s ACCESSOR
 | "AND"                -> s AND
 | "ARRAY"              -> s ARRAY
 | "ARROW"              -> s ARROW
 | "AS"                 -> s AS
 | "ASSERT"             -> s ASSERT
 | "BAND"               -> s BAND
 | "BEGIN"              -> s BEGIN
 | "BEQ"                -> s BEQ
 | "BIT"                -> s BIT
 | "BITS"               -> s BITS
 | "BNOT"               -> s BNOT
 | "BOOLEAN"            -> s BOOLEAN
 | "BOR"                -> s BOR
 | "CASE"               -> s CASE
 | "CATCH"              -> s CATCH
 | "COLLECTION"         -> s COLLECTION
 | "COLON"              -> s COLON
 | "COLON_COLON"        -> s COLON_COLON
 | "COMMA"              -> s COMMA
 | "CONFIG"             -> s CONFIG
 | "CONSTANT"           -> s CONSTANT
 | "DEBUG"              -> s DEBUG
 | "DIV"                -> s DIV
 | "DIVRM"              -> s DIVRM
 | "DO"                 -> s DO
 | "DOT"                -> s DOT
 | "DOWNTO"             -> s DOWNTO
 | "ELSE"               -> s ELSE
 | "ELSIF"              -> s ELSIF
 | "END"                -> s END
 | "ENUMERATION"        -> s ENUMERATION
 | "EOF"                -> s EOF
 | "XOR"                -> s XOR
 | "EQ"                 -> s EQ
 | "EQ_EQ"              -> s EQ_EQ
 | "EXCEPTION"          -> s EXCEPTION
 | "FOR"                -> s FOR
 | "FUNC"               -> s FUNC
 | "GE"                 -> s GE
 | "GETTER"             -> s GETTER
 | "GT"                 -> s GT
 | "IF"                 -> s IF
 | "IMPLEMENTATION"     -> s IMPLEMENTATION
 | "IMPDEF"             -> s IMPDEF
 | "IMPL"               -> s IMPL
 | "IN"                 -> s IN
 | "INTEGER"            -> s INTEGER
 | "LBRACE"             -> s LBRACE
 | "LBRACKET"           -> s LBRACKET
 | "LE"                 -> s LE
 | "LET"                -> s LET
 | "LOOPLIMIT"          -> s LOOPLIMIT
 | "LPAR"               -> s LPAR
 | "LT"                 -> s LT
 | "MINUS"              -> s MINUS
 | "MOD"                -> s MOD
 | "MUL"                -> s MUL
 | "NE"                 -> s NE
 | "NORETURN"           -> s NORETURN
 | "NOT"                -> s NOT
 | "OF"                 -> s OF
 | "OR"                 -> s OR
 | "OTHERWISE"          -> s OTHERWISE
 | "PASS"               -> s PASS
 | "PLUS"               -> s PLUS
 | "PLUS_PLUS"          -> s PLUS_PLUS
 | "PLUS_COLON"         -> s PLUS_COLON
 | "POW"                -> s POW
 | "PRAGMA"             -> s PRAGMA
 | "PRINTLN"            -> s PRINTLN
 | "PRINT"              -> s PRINT
 | "PURE"               -> s PURE
 | "RBRACE"             -> s RBRACE
 | "RBRACKET"           -> s RBRACKET
 | "RDIV"               -> s RDIV
 | "READONLY"           -> s READONLY
 | "REAL"               -> s REAL
 | "RECORD"             -> s RECORD
 | "RECURSELIMIT"       -> s RECURSELIMIT
 | "REPEAT"             -> s REPEAT
 | "RETURN"             -> s RETURN
 | "RPAR"               -> s RPAR
 | "STAR_COLON"         -> s STAR_COLON
 | "SEMI_COLON"         -> s SEMI_COLON
 | "SETTER"             -> s SETTER
 | "SHL"                -> s SHL
 | "SHR"                -> s SHR
 | "SLICING"            -> s SLICING
 | "STRING"             -> s STRING
 | "SUBTYPES"           -> s SUBTYPES
 | "THEN"               -> s THEN
 | "THROW"              -> s THROW
 | "TO"                 -> s TO
 | "TRY"                -> s TRY
 | "TYPE"               -> s TYPE
 | "ARBITRARY"          -> s ARBITRARY
 | "UNREACHABLE"        -> s UNREACHABLE
 | "UNTIL"              -> s UNTIL
 | "VAR"                -> s VAR
 | "WHEN"               -> s WHEN
 | "WHERE"              -> s WHERE
 | "WHILE"              -> s WHILE
 | "WITH"               -> s WITH
 | "LLBRACKET"          -> s LLBRACKET
 | "RRBRACKET"          -> s RRBRACKET
 | _ -> None

(** Convert a lexical token to the symbol it lexes *)
let token_to_symbol = function
  | BNOT               -> "!"
  | COMMA              -> ","
  | LT                 -> "<"
  | SHR                -> ">>"
  | BAND               -> "&&"
  | IMPL               -> "==>"
  | SHL                -> "<<"
  | RBRACKET           -> "]"
  | RRBRACKET          -> "]]"
  | RPAR               -> ")"
  | SLICING            -> ".."
  | EQ                 -> "="
  | LBRACE             -> "{"
  | NE                -> "!="
  | MINUS              -> "-"
  | BEQ                -> "<=>"
  | LBRACKET           -> "["
  | LLBRACKET          -> "[["
  | LPAR               -> "("
  | DOT                -> "."
  | LE                -> "<="
  | POW                -> "^"
  | MUL                -> "*"
  | RDIV               -> "/"
  | EQ_EQ              -> "=="
  | BOR                -> "||"
  | PLUS               -> "+"
  | COLON              -> ":"
  | ARROW              -> "=>"
  | RBRACE             -> "}"
  | COLON_COLON        -> "::"
  | PLUS_PLUS          -> "++"
  | GT                 -> ">"
  | PLUS_COLON         -> "+:"
  | STAR_COLON         -> "*:"
  | SEMI_COLON         -> ";"
  | GE                -> ">="
  (* Keywords *)
  | ACCESSOR           -> "accessor"
  | AND                -> "AND"
  | ARRAY              -> "array"
  | AS                 -> "as"
  | ASSERT             -> "assert"
  | BEGIN              -> "begin"
  | BIT                -> "bit"
  | BITS               -> "bits"
  | BOOLEAN            -> "boolean"
  | CASE               -> "case"
  | CATCH              -> "catch"
  | COLLECTION         -> "collection"
  | CONFIG             -> "config"
  | CONSTANT           -> "constant"
  | DIV                -> "DIV"
  | DIVRM              -> "DIVRM"
  | DO                 -> "do"
  | DOWNTO             -> "downto"
  | ELSE               -> "else"
  | ELSIF              -> "elsif"
  | END                -> "end"
  | ENUMERATION        -> "enumeration"
  | XOR                -> "XOR"
  | EXCEPTION          -> "exception"
  | FOR                -> "for"
  | FUNC               -> "func"
  | GETTER             -> "getter"
  | IF                 -> "if"
  | IMPLEMENTATION     -> "implementation"
  | IMPDEF             -> "impdef"
  | IN                 -> "IN"
  | INTEGER            -> "integer"
  | LET                -> "let"
  | LOOPLIMIT          -> "looplimit"
  | MOD                -> "MOD"
  | NORETURN           -> "noreturn"
  | NOT                -> "NOT"
  | OF                 -> "of"
  | OR                 -> "OR"
  | OTHERWISE          -> "otherwise"
  | PASS               -> "pass"
  | PRAGMA             -> "pragma"
  | PRINTLN            -> "println"
  | PRINT              -> "print"
  | PURE               -> "pure"
  | READONLY           -> "readonly"
  | REAL               -> "real"
  | RECORD             -> "record"
  | RECURSELIMIT       -> "recurselimit"
  | REPEAT             -> "repeat"
  | RETURN             -> "return"
  | SETTER             -> "setter"
  | STRING             -> "string"
  | SUBTYPES           -> "subtypes"
  | THEN               -> "then"
  | THROW              -> "throw"
  | TO                 -> "to"
  | TRY                -> "try"
  | TYPE               -> "type"
  | ARBITRARY          -> "ARBITRARY"
  | UNREACHABLE        -> "unreachable"
  | UNTIL              -> "until"
  | VAR                -> "var"
  | WHEN               -> "when"
  | WHERE              -> "where"
  | WHILE              -> "while"
  | WITH               -> "with"
  | BOOL_LIT _
  | INT_LIT _
  | REAL_LIT _
  | STRING_LIT _
  | BITVECTOR_LIT _
  | MASK_LIT _
  | IDENTIFIER _
  | EOF
  | DEBUG -> assert false (* Complex tokens *)


module Make (Config : CONFIG) = struct

exception LexerError

let new_line lexbuf = Lexing.new_line lexbuf; lexbuf
let bitvector_lit lxm = BITVECTOR_LIT (Bitvector.of_string lxm)
let mask_lit lxm = MASK_LIT (Bitvector.mask_of_string lxm)
let reserved_err s = Error.fatal_unknown_pos @@ (Error.ReservedIdentifier s)

let fatal lexbuf desc =
  AST.
    {
      desc;
      version = V1;
      pos_start = Lexing.lexeme_start_p lexbuf;
      pos_end = Lexing.lexeme_end_p lexbuf;
    }
  |> Error.fatal

let tr_name s = match s with
| "accessor"      -> ACCESSOR
| "AND"           -> AND
| "array"         -> ARRAY
| "as"            -> AS
| "assert"        -> ASSERT
| "begin"         -> BEGIN
| "bit"           -> BIT
| "bits"          -> BITS
| "boolean"       -> BOOLEAN
| "case"          -> CASE
| "catch"         -> CATCH
| "collection"    -> COLLECTION
| "config"        -> CONFIG
| "constant"      -> CONSTANT
| "__debug__"
| "__DEBUG__"     -> DEBUG
| "DIV"           -> DIV
| "DIVRM"         -> DIVRM
| "do"            -> DO
| "downto"        -> DOWNTO
| "else"          -> ELSE
| "elsif"         -> ELSIF
| "end"           -> END
| "enumeration"   -> ENUMERATION
| "XOR"           -> XOR
| "exception"     -> EXCEPTION
| "FALSE"         -> BOOL_LIT false
| "for"           -> FOR
| "func"          -> FUNC
| "getter"        -> GETTER
| "if"            -> IF
| "impdef"        -> IMPDEF
| "implementation" -> IMPLEMENTATION
| "IN"            -> IN
| "integer"       -> INTEGER
| "let"           -> LET
| "looplimit"     -> LOOPLIMIT
| "MOD"           -> MOD
| "NOT"           -> NOT
| "noreturn"      -> NORETURN
| "of"            -> OF
| "OR"            -> OR
| "otherwise"     -> OTHERWISE
| "pass"          -> PASS
| "pragma"        -> PRAGMA
| "println"       -> PRINTLN
| "print"         -> PRINT
| "pure"          -> PURE
| "readonly"      -> READONLY
| "real"          -> REAL
| "record"        -> RECORD
| "recurselimit"  -> RECURSELIMIT
| "repeat"        -> REPEAT
| "return"        -> RETURN
| "setter"        -> SETTER
| "string"        -> STRING
| "subtypes"      -> SUBTYPES
| "then"          -> THEN
| "throw"         -> THROW
| "to"            -> TO
| "try"           -> TRY
| "TRUE"          -> BOOL_LIT true
| "type"          -> TYPE
| "ARBITRARY"     -> ARBITRARY
| "unreachable"   -> UNREACHABLE
| "until"         -> UNTIL
| "var"           -> VAR
| "when"          -> WHEN
| "where"         -> WHERE
| "while"         -> WHILE
| "with"          -> WITH
(* Reserved identifiers *)
| x when is_reserved_keyword x -> reserved_err x
| x when ASTUtils.string_starts_with ~prefix:"__" x -> reserved_err x
(* End of reserved identifiers *)
| x               -> IDENTIFIER x

# 351 "asllib/Lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\251\255\252\255\253\255\254\255\255\255\001\000\002\000\
    \253\255\254\255\255\255\003\000\004\000\253\255\000\000\255\255\
    \086\000\205\255\196\000\208\255\214\255\000\000\001\000\025\000\
    \002\000\222\255\225\255\000\000\232\255\002\000\235\255\004\000\
    \000\000\004\000\000\000\002\000\005\000\244\255\006\000\031\001\
    \248\255\022\000\039\001\009\000\007\000\255\255\252\255\031\000\
    \166\000\055\001\202\255\119\001\201\255\204\255\215\001\203\255\
    \005\000\050\002\247\255\246\255\057\000\231\255\007\000\018\000\
    \238\255\229\255\228\255\207\255\242\255\241\255\019\000\240\255\
    \215\255\020\000\239\255\236\255\234\255\226\255\209\255\218\255\
    \210\255\213\255\212\255";
  Lexing.lex_backtrk =
   "\255\255\255\255\255\255\255\255\255\255\255\255\004\000\003\000\
    \255\255\255\255\255\255\004\000\003\000\255\255\001\000\255\255\
    \055\000\255\255\049\000\255\255\255\255\039\000\038\000\255\255\
    \034\000\255\255\255\255\028\000\255\255\031\000\255\255\018\000\
    \022\000\025\000\255\255\044\000\012\000\255\255\010\000\255\255\
    \255\255\004\000\004\000\035\000\001\000\255\255\255\255\002\000\
    \255\255\255\255\255\255\005\000\255\255\255\255\006\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\032\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\036\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255";
  Lexing.lex_default =
   "\001\000\000\000\000\000\000\000\000\000\000\000\007\000\007\000\
    \000\000\000\000\000\000\012\000\012\000\000\000\255\255\000\000\
    \255\255\000\000\255\255\000\000\000\000\255\255\255\255\255\255\
    \255\255\000\000\000\000\255\255\000\000\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\000\000\000\000\047\000\
    \053\000\050\000\000\000\255\255\000\000\000\000\255\255\000\000\
    \255\255\255\255\000\000\000\000\255\255\000\000\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\008\000\255\255\013\000\255\255\000\000\
    \044\000\000\000\000\000\000\000\044\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\010\000\255\255\060\000\069\000\044\000\
    \000\000\255\255\000\000\000\000\081\000\014\000\255\255\015\000\
    \076\000\070\000\063\000\046\000\000\000\060\000\060\000\000\000\
    \047\000\000\000\082\000\080\000\078\000\073\000\072\000\067\000\
    \068\000\064\000\062\000\061\000\048\000\066\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \065\000\071\000\074\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\060\000\000\000\077\000\002\000\009\000\255\255\044\000\
    \045\000\075\000\057\000\044\000\000\000\000\000\000\000\000\000\
    \000\000\060\000\060\000\000\000\000\000\000\000\005\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\041\000\044\000\038\000\
    \040\000\000\000\000\000\000\000\034\000\039\000\026\000\030\000\
    \024\000\022\000\037\000\033\000\029\000\043\000\042\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \021\000\019\000\036\000\032\000\035\000\079\000\000\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\027\000\000\000\031\000\025\000\018\000\000\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\028\000\023\000\020\000\255\255\000\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\255\255\
    \000\000\000\000\000\000\018\000\000\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\039\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\058\000\056\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\039\000\
    \039\000\000\000\000\000\000\000\000\000\048\000\017\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\000\000\000\000\000\000\000\000\000\000\000\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\041\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\057\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\000\000\049\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\000\000\000\000\000\000\000\000\051\000\000\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\000\000\000\000\000\000\000\000\054\000\255\255\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\057\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\059\000\056\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\057\000\057\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\057\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\006\000\007\000\011\000\012\000\255\255\
    \044\000\255\255\255\255\255\255\044\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\006\000\007\000\056\000\034\000\044\000\
    \255\255\047\000\255\255\255\255\022\000\011\000\012\000\014\000\
    \029\000\033\000\036\000\043\000\255\255\056\000\056\000\255\255\
    \043\000\255\255\021\000\022\000\024\000\032\000\032\000\035\000\
    \035\000\036\000\036\000\038\000\041\000\062\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \063\000\070\000\073\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\060\000\255\255\027\000\000\000\006\000\007\000\016\000\
    \016\000\031\000\060\000\016\000\255\255\255\255\255\255\255\255\
    \255\255\060\000\060\000\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\000\000\041\000\016\000\016\000\
    \016\000\255\255\255\255\255\255\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\023\000\255\255\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\255\255\016\000\016\000\016\000\255\255\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\048\000\255\255\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\255\255\255\255\
    \000\000\006\000\007\000\011\000\012\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\047\000\
    \255\255\255\255\255\255\018\000\255\255\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\039\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\039\000\039\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\039\000\
    \039\000\255\255\255\255\255\255\255\255\042\000\016\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\255\255\255\255\255\255\255\255\255\255\255\255\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \049\000\049\000\049\000\049\000\049\000\049\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\042\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\039\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\255\255\042\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\048\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\255\255\255\255\255\255\255\255\051\000\255\255\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\255\255\255\255\255\255\255\255\054\000\049\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\057\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\057\000\057\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\057\000\057\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\057\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255";
  Lexing.lex_base_code =
   "";
  Lexing.lex_backtrk_code =
   "";
  Lexing.lex_default_code =
   "";
  Lexing.lex_trans_code =
   "";
  Lexing.lex_check_code =
   "";
  Lexing.lex_code =
   "";
}

let rec escaped_string_chars acc lexbuf =
   __ocaml_lex_escaped_string_chars_rec acc lexbuf 0
and __ocaml_lex_escaped_string_chars_rec acc lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 414 "asllib/Lexer.mll"
         ( Buffer.add_char acc '\n'; string_lit acc lexbuf )
# 618 "asllib/Lexer.ml"

  | 1 ->
# 415 "asllib/Lexer.mll"
         ( Buffer.add_char acc '\t'; string_lit acc lexbuf )
# 623 "asllib/Lexer.ml"

  | 2 ->
# 416 "asllib/Lexer.mll"
         ( Buffer.add_char acc '"'; string_lit acc lexbuf )
# 628 "asllib/Lexer.ml"

  | 3 ->
# 417 "asllib/Lexer.mll"
         ( Buffer.add_char acc '\\'; string_lit acc lexbuf )
# 633 "asllib/Lexer.ml"

  | 4 ->
# 418 "asllib/Lexer.mll"
                         ( raise LexerError )
# 638 "asllib/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_escaped_string_chars_rec acc lexbuf __ocaml_lex_state

and string_lit acc lexbuf =
   __ocaml_lex_string_lit_rec acc lexbuf 6
and __ocaml_lex_string_lit_rec acc lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 421 "asllib/Lexer.mll"
          ( STRING_LIT (Buffer.contents acc) )
# 650 "asllib/Lexer.ml"

  | 1 ->
# 422 "asllib/Lexer.mll"
          ( escaped_string_chars acc lexbuf )
# 655 "asllib/Lexer.ml"

  | 2 ->
# 423 "asllib/Lexer.mll"
          ( Buffer.add_char acc '\n'; new_line lexbuf |> string_lit acc )
# 660 "asllib/Lexer.ml"

  | 3 ->
let
# 424 "asllib/Lexer.mll"
                          lxm
# 666 "asllib/Lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 424 "asllib/Lexer.mll"
                              ( Buffer.add_string acc lxm; string_lit acc lexbuf )
# 670 "asllib/Lexer.ml"

  | 4 ->
# 425 "asllib/Lexer.mll"
          ( raise LexerError )
# 675 "asllib/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_string_lit_rec acc lexbuf __ocaml_lex_state

and c_comments lexbuf =
   __ocaml_lex_c_comments_rec lexbuf 11
and __ocaml_lex_c_comments_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 433 "asllib/Lexer.mll"
                  ( token      lexbuf )
# 687 "asllib/Lexer.ml"

  | 1 ->
# 434 "asllib/Lexer.mll"
                  ( c_comments lexbuf )
# 692 "asllib/Lexer.ml"

  | 2 ->
# 435 "asllib/Lexer.mll"
                  ( new_line lexbuf |> c_comments )
# 697 "asllib/Lexer.ml"

  | 3 ->
# 436 "asllib/Lexer.mll"
                  ( c_comments lexbuf )
# 702 "asllib/Lexer.ml"

  | 4 ->
# 437 "asllib/Lexer.mll"
                  ( raise LexerError  )
# 707 "asllib/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_c_comments_rec lexbuf __ocaml_lex_state

and token lexbuf =
   __ocaml_lex_token_rec lexbuf 16
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 445 "asllib/Lexer.mll"
                               ( new_line lexbuf |> token         )
# 719 "asllib/Lexer.ml"

  | 1 ->
# 446 "asllib/Lexer.mll"
                               ( token lexbuf                     )
# 724 "asllib/Lexer.ml"

  | 2 ->
# 447 "asllib/Lexer.mll"
                               ( token lexbuf                     )
# 729 "asllib/Lexer.ml"

  | 3 ->
# 448 "asllib/Lexer.mll"
                               ( c_comments lexbuf                )
# 734 "asllib/Lexer.ml"

  | 4 ->
let
# 449 "asllib/Lexer.mll"
                 lxm
# 740 "asllib/Lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 449 "asllib/Lexer.mll"
                               ( INT_LIT(Z.of_string lxm)         )
# 744 "asllib/Lexer.ml"

  | 5 ->
let
# 450 "asllib/Lexer.mll"
                 lxm
# 750 "asllib/Lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 450 "asllib/Lexer.mll"
                               ( INT_LIT(Z.of_string lxm)         )
# 754 "asllib/Lexer.ml"

  | 6 ->
let
# 451 "asllib/Lexer.mll"
                  lxm
# 760 "asllib/Lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 451 "asllib/Lexer.mll"
                               ( REAL_LIT(Q.of_string lxm)        )
# 764 "asllib/Lexer.ml"

  | 7 ->
# 452 "asllib/Lexer.mll"
                               ( string_lit (Buffer.create 16) lexbuf )
# 769 "asllib/Lexer.ml"

  | 8 ->
let
# 453 "asllib/Lexer.mll"
                    lxm
# 775 "asllib/Lexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_curr_pos + -1) in
# 453 "asllib/Lexer.mll"
                               ( bitvector_lit lxm                )
# 779 "asllib/Lexer.ml"

  | 9 ->
let
# 454 "asllib/Lexer.mll"
                    lxm
# 785 "asllib/Lexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_curr_pos + -1) in
# 454 "asllib/Lexer.mll"
                               ( mask_lit lxm                     )
# 789 "asllib/Lexer.ml"

  | 10 ->
# 455 "asllib/Lexer.mll"
                               ( BNOT                             )
# 794 "asllib/Lexer.ml"

  | 11 ->
# 456 "asllib/Lexer.mll"
                               ( COMMA                            )
# 799 "asllib/Lexer.ml"

  | 12 ->
# 457 "asllib/Lexer.mll"
                               ( LT                               )
# 804 "asllib/Lexer.ml"

  | 13 ->
# 458 "asllib/Lexer.mll"
                               ( SHR                              )
# 809 "asllib/Lexer.ml"

  | 14 ->
# 459 "asllib/Lexer.mll"
                               ( BAND                             )
# 814 "asllib/Lexer.ml"

  | 15 ->
# 460 "asllib/Lexer.mll"
                               ( fatal lexbuf (CannotParse (Some "Did you mean `==>`?")) )
# 819 "asllib/Lexer.ml"

  | 16 ->
# 461 "asllib/Lexer.mll"
                               ( IMPL                             )
# 824 "asllib/Lexer.ml"

  | 17 ->
# 462 "asllib/Lexer.mll"
                               ( SHL                              )
# 829 "asllib/Lexer.ml"

  | 18 ->
# 463 "asllib/Lexer.mll"
                               ( RBRACKET                         )
# 834 "asllib/Lexer.ml"

  | 19 ->
# 464 "asllib/Lexer.mll"
                               ( RRBRACKET                        )
# 839 "asllib/Lexer.ml"

  | 20 ->
# 465 "asllib/Lexer.mll"
                               ( RPAR                             )
# 844 "asllib/Lexer.ml"

  | 21 ->
# 466 "asllib/Lexer.mll"
                               ( SLICING                          )
# 849 "asllib/Lexer.ml"

  | 22 ->
# 467 "asllib/Lexer.mll"
                               ( EQ                               )
# 854 "asllib/Lexer.ml"

  | 23 ->
# 468 "asllib/Lexer.mll"
                               ( LBRACE                           )
# 859 "asllib/Lexer.ml"

  | 24 ->
# 469 "asllib/Lexer.mll"
                               ( NE                              )
# 864 "asllib/Lexer.ml"

  | 25 ->
# 470 "asllib/Lexer.mll"
                               ( MINUS                            )
# 869 "asllib/Lexer.ml"

  | 26 ->
# 471 "asllib/Lexer.mll"
                               ( fatal lexbuf (CannotParse (Some "Did you mean `<=>`?")) )
# 874 "asllib/Lexer.ml"

  | 27 ->
# 472 "asllib/Lexer.mll"
                               ( BEQ                              )
# 879 "asllib/Lexer.ml"

  | 28 ->
# 473 "asllib/Lexer.mll"
                               ( LBRACKET                         )
# 884 "asllib/Lexer.ml"

  | 29 ->
# 474 "asllib/Lexer.mll"
                               ( LLBRACKET                        )
# 889 "asllib/Lexer.ml"

  | 30 ->
# 475 "asllib/Lexer.mll"
                               ( LPAR                             )
# 894 "asllib/Lexer.ml"

  | 31 ->
# 476 "asllib/Lexer.mll"
                               ( DOT                              )
# 899 "asllib/Lexer.ml"

  | 32 ->
# 477 "asllib/Lexer.mll"
                               ( LE                              )
# 904 "asllib/Lexer.ml"

  | 33 ->
# 478 "asllib/Lexer.mll"
                               ( POW                              )
# 909 "asllib/Lexer.ml"

  | 34 ->
# 479 "asllib/Lexer.mll"
                               ( MUL                              )
# 914 "asllib/Lexer.ml"

  | 35 ->
# 480 "asllib/Lexer.mll"
                               ( RDIV                             )
# 919 "asllib/Lexer.ml"

  | 36 ->
# 481 "asllib/Lexer.mll"
                               ( EQ_EQ                            )
# 924 "asllib/Lexer.ml"

  | 37 ->
# 482 "asllib/Lexer.mll"
                               ( BOR                              )
# 929 "asllib/Lexer.ml"

  | 38 ->
# 483 "asllib/Lexer.mll"
                               ( PLUS                             )
# 934 "asllib/Lexer.ml"

  | 39 ->
# 484 "asllib/Lexer.mll"
                               ( COLON                            )
# 939 "asllib/Lexer.ml"

  | 40 ->
# 485 "asllib/Lexer.mll"
                               ( ARROW                            )
# 944 "asllib/Lexer.ml"

  | 41 ->
# 486 "asllib/Lexer.mll"
                               ( RBRACE                           )
# 949 "asllib/Lexer.ml"

  | 42 ->
# 487 "asllib/Lexer.mll"
                               ( PLUS_PLUS                        )
# 954 "asllib/Lexer.ml"

  | 43 ->
# 488 "asllib/Lexer.mll"
                               ( COLON_COLON                      )
# 959 "asllib/Lexer.ml"

  | 44 ->
# 489 "asllib/Lexer.mll"
                               ( GT                               )
# 964 "asllib/Lexer.ml"

  | 45 ->
# 490 "asllib/Lexer.mll"
                               ( PLUS_COLON                       )
# 969 "asllib/Lexer.ml"

  | 46 ->
# 491 "asllib/Lexer.mll"
                               ( STAR_COLON                       )
# 974 "asllib/Lexer.ml"

  | 47 ->
# 492 "asllib/Lexer.mll"
                               ( SEMI_COLON                       )
# 979 "asllib/Lexer.ml"

  | 48 ->
# 493 "asllib/Lexer.mll"
                               ( GE                              )
# 984 "asllib/Lexer.ml"

  | 49 ->
let
# 494 "asllib/Lexer.mll"
                    lxm
# 990 "asllib/Lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 494 "asllib/Lexer.mll"
                               ( tr_name lxm                      )
# 994 "asllib/Lexer.ml"

  | 50 ->
# 495 "asllib/Lexer.mll"
                               ( EOF                              )
# 999 "asllib/Lexer.ml"

  | 51 ->
# 496 "asllib/Lexer.mll"
                               ( raise LexerError                 )
# 1004 "asllib/Lexer.ml"

  | 52 ->
# 497 "asllib/Lexer.mll"
                               ( raise LexerError                 )
# 1009 "asllib/Lexer.ml"

  | 53 ->
# 498 "asllib/Lexer.mll"
                               ( raise LexerError                 )
# 1014 "asllib/Lexer.ml"

  | 54 ->
# 499 "asllib/Lexer.mll"
                               ( raise LexerError                 )
# 1019 "asllib/Lexer.ml"

  | 55 ->
# 500 "asllib/Lexer.mll"
                               ( raise LexerError                 )
# 1024 "asllib/Lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_token_rec lexbuf __ocaml_lex_state

;;

# 501 "asllib/Lexer.mll"
 
end

# 1035 "asllib/Lexer.ml"
