'use strict';

/* eslint-disable max-len */

const assert = require('../../utils/assert.js');
const preq   = require('preq');
const Server = require('../../utils/server.js');
const nock   = require('nock');

// TODO: add support for nocked tests
const NOCK_TESTS = false; // because of Parsoid/PHP which uses the same URI structure as the MW API

describe('Access checks', () => {
    const server = new Server();

    const deletedPageTitle = 'User:Pchelolo/Access_Check_Tests';
    const deletedPageOlderRevision = 409433;
    const deletedPageRevision = 409434;
    const emptyResponse = { 'batchcomplete': '', 'query': { 'badrevids': { '292466': { 'revid': '292466' } } } };

    function setUpNockResponse(api, title, revision) {
        if (!NOCK_TESTS) {
            return api;
        }
        return api.post('')
        .reply(200, {
            'batchcomplete': '',
            'query': {
                'pages': {
                    '49453581': {
                        'pageid': 49453581,
                        'ns': 0,
                        title,
                        'pagelanguage': 'en',
                        'touched': '2015-05-22T08:49:39Z',
                        'lastrevid': revision,
                        'length': 2941,
                        'revisions': [{
                            'revid': revision,
                            'user': 'Chuck Norris',
                            'userid': 3606755,
                            'timestamp': '2015-03-25T20:29:50Z',
                            'size': 2941,
                            'sha1': 'c47571122e00f28402d2a1b75cff77a22e7bfecd',
                            'comment': 'Test',
                            'tags': []
                        }]
                    }
                }
            }
        });
    }

    before(() => {
        if (!nock.isActive()) {
            nock.activate();
        }
        return server.start()
        // Do a preparation request to force siteinfo fetch so that we don't need to mock it
        .then(() => preq.get({
            uri: `${server.config.bucketURL()}/html/Main_Page`,
            headers: {
                'cache-control': 'no-cache'
            }
        }))
        // Load in the revisions
        .then(() => {
            let api = nock(server.config.apiURL());
            api = setUpNockResponse(api, deletedPageTitle, deletedPageOlderRevision);
            api = setUpNockResponse(api, deletedPageTitle, deletedPageRevision);
            return preq.get({
                uri: `${server.config.bucketURL()}/html/${encodeURIComponent(deletedPageTitle)}/${deletedPageOlderRevision}`,
                headers: {
                    'cache-control': 'no-cache'
                }
            })
            .then((res) => {
                assert.deepEqual(res.status, 200);
                return preq.get({
                    uri: `${server.config.bucketURL()}/html/${encodeURIComponent(deletedPageTitle)}/${deletedPageRevision}`,
                    headers: {
                        'cache-control': 'no-cache'
                    }
                });
            })
            .then(res => assert.deepEqual(res.status, 200))
            .then(() => api.done())
            .finally(() => nock.cleanAll());
        });
    });
    after(() =>  server.stop());

    describe('Deleting', () => {
        it('should understand the page was deleted', () => {
            const api = nock(server.config.apiURL())
            // Other requests return nothing as if the page is deleted.
            .post('').reply(200, emptyResponse);
            // Fetch the page
            return preq.get({
                uri: `${server.config.bucketURL()}/title/${encodeURIComponent(deletedPageTitle)}`,
                headers: {
                    'cache-control': 'no-cache'
                }
            })
            .then(() => {
                throw new Error('404 should have been returned for a deleted page');
            }, (e) => {
                assert.deepEqual(e.status, 404);
                assert.contentType(e, 'application/problem+json');
            })
            .then(() => api.done())
            .finally(() => nock.cleanAll());
        });
    });

    function testAccess(contentVariant, restrictionType, title, rev) {
        let name = `should restrict access to ${restrictionType} page `;
        name += rev ? 'older revision' : 'latest';
        name += ` ${contentVariant}`;
        it(name, () => {
            // Check that access is enforced to html
            let uri = `${server.config.bucketURL()}/${contentVariant}/${encodeURIComponent(title)}`;
            if (rev) {
                uri += `/${rev}`;
            }
            return preq.get({ uri })
            .then((res) => {
                throw new Error('404 should have been returned for a deleted page');
            }, (e) => {
                assert.deepEqual(e.status, 404);
                assert.contentType(e, 'application/problem+json');
            });
        });
    }

    describe('Checking deletions', () => {
        it('should restrict access to deleted page latest revision', () => {
            // This is only required until the hack for no-cache header is in place
            const api = nock(server.config.apiURL())
            .post('').reply(200, emptyResponse);

            return preq.get({
                uri: `${server.config.bucketURL()}/title/${encodeURIComponent(deletedPageTitle)}/${deletedPageRevision}`
            })
            .then(() => {
                throw new Error('404 should have been returned for a deleted page');
            }, (e) => {
                assert.deepEqual(e.status, 404);
                assert.contentType(e, 'application/problem+json');
            })
            .then(() => api.done())
            .finally(() => nock.cleanAll());
        });

        it('should restrict access to older revision of a deleted page', () => {
            // This is only required until the hack for no-cache header is in place
            const api = nock(server.config.apiURL())
            .post('').reply(200, emptyResponse);

            return preq.get({
                uri: `${server.config.bucketURL()}/title/${encodeURIComponent(deletedPageTitle)}/${deletedPageOlderRevision}`
            })
            .then(() => {
                throw new Error('404 should have been returned for a deleted page');
            }, (e) => {
                assert.deepEqual(e.status, 404);
                assert.contentType(e, 'application/problem+json');
            })
            .then(() => api.done())
            .finally(() => nock.cleanAll());
        });

        testAccess('html', 'deleted', deletedPageTitle);
        testAccess('data-parsoid', 'deleted', deletedPageTitle);
        testAccess('html', 'deleted', deletedPageTitle, deletedPageOlderRevision);
        testAccess('data-parsoid', 'deleted', deletedPageTitle, deletedPageOlderRevision);
        testAccess('summary', 'deleted', deletedPageTitle);
        testAccess('mobile-html', 'deleted', deletedPageTitle);
    });

    describe('Undeleting', () => {
        it('Should understand that the page was undeleted', () => {
            return preq.get({
                uri: `${server.config.bucketURL()}/title/${encodeURIComponent(deletedPageTitle)}`,
                headers: {
                    'cache-control': 'no-cache'
                }
            })
            .then((res) => {
                assert.deepEqual(res.status, 200);
                return preq.get({
                    uri: `${server.config.bucketURL()}/html/${encodeURIComponent(deletedPageTitle)}/${deletedPageOlderRevision}`,
                });
            })
            .then((res) => {
                assert.deepEqual(res.status, 200);
            });
        });

        it('should understand the page was deleted again', () => {
            const api = nock(server.config.apiURL())
            // Other requests return nothing as if the page is deleted.
            .post('').reply(200, emptyResponse);
            // Fetch the page
            return preq.get({
                uri: `${server.config.bucketURL()}/title/${encodeURIComponent(deletedPageTitle)}`,
                headers: {
                    'cache-control': 'no-cache'
                }
            })
            .then(() => {
                throw new Error('404 should have been returned for a deleted page');
            }, (e) => {
                assert.deepEqual(e.status, 404);
                assert.contentType(e, 'application/problem+json');
            })
            .then(() => api.done())
            .finally(() => nock.cleanAll());
        });

        it('Should understand that the page was undeleted base on html request', () => {
            return preq.get({
                uri: `${server.config.bucketURL()}/html/${encodeURIComponent(deletedPageTitle)}`,
                headers: {
                    'cache-control': 'no-cache'
                }
            })
            .then((res) => {
                assert.deepEqual(res.status, 200);
                return preq.get({
                    uri: `${server.config.bucketURL()}/html/${encodeURIComponent(deletedPageTitle)}/${deletedPageOlderRevision}`,
                });
            })
            .then((res) => {
                assert.deepEqual(res.status, 200);
            });
        });
    });

    describe('Restricting', () => {
        const pageTitle = 'User:Pchelolo/restriction_testing_mock';
        const pageRev = 301375;
        it('should correctly fetch updated restrictions', () => {
            const normalRev = {
                "revid": pageRev,
                "user": "Pchelolo",
                "userid": 6591,
                "timestamp": "2015-02-03T21:15:55Z",
                "size": 7700,
                "tags": []
            };
            const normalResponse = {
                "pageid": 152993,
                "ns": 3,
                "title": pageTitle,
                "pagelanguage": "en",
                "pagelanguagehtmlcode": "en",
                "pagelanguagedir": "ltr",
                "touched": "2015-12-10T23:41:54Z",
                "lastrevid": pageRev,
                "length": 23950,
                "revisions": [normalRev]
            };
            const restrictedRev = Object.assign({}, normalRev);
            restrictedRev.texthidden = true;
            restrictedRev.sha1hidden = true;
            const restrictedResponse = Object.assign({}, normalResponse);
            restrictedResponse.revisions = [restrictedRev];
            const api = nock(server.config.apiURL('en.wikipedia.beta.wmflabs.org'))
            .post('').reply(200, {
                "batchcomplete": "",
                "query": { "pages": { "45161196": normalResponse } }
            }).post('').reply(200, {
                "batchcomplete": "",
                "query": { "pages": { "45161196": restrictedResponse } }
            });

            // First fetch a non-restricted revision
            return preq.get({
                uri: `${server.config.bucketURL('en.wikipedia.beta.wmflabs.org')}/title/${encodeURIComponent(pageTitle)}`
            })
            .then((res) => {
                assert.deepEqual(res.status, 200);
                assert.deepEqual(res.body.items.length, 1);
                // Now fetch update with restrictions
                return preq.get({
                    uri: `${server.config.bucketURL('en.wikipedia.beta.wmflabs.org')}/title/${encodeURIComponent(pageTitle)}`,
                    headers: {
                        'cache-control': 'no-cache'
                    }
                });
            }).then(() => {
                throw new Error('403 should be thrown');
            }, (e) => {
                assert.deepEqual(e.status, 403);
            }).then(() => api.done())
            .finally(() => nock.cleanAll());
        });


        it('should store updated restrictions', () => {
            return preq.get({
                uri: `${server.config.bucketURL('en.wikipedia.beta.wmflabs.org')}/html/${encodeURIComponent(pageTitle)}`
            })
            .then(() => {
                throw new Error('403 should be thrown');
            }, (e) => {
                assert.deepEqual(e.status, 403);
            });
        });

        it('should restrict access to restricted revision html', () => {
            return preq.get({
                uri: `${server.config.bucketURL('en.wikipedia.beta.wmflabs.org')}/html/${encodeURIComponent(pageTitle)}/${pageRev}`
            })
            .then(() => {
                throw new Error('403 should have been returned for a deleted page');
            }, (e) => {
                assert.deepEqual(e.status, 403);
                assert.contentType(e, 'application/problem+json');
            });
        });

        it('should allow to view content if restrictions disappeared', () => {
            return preq.get({
                uri: `${server.config.bucketURL('en.wikipedia.beta.wmflabs.org')}/title/${encodeURIComponent(pageTitle)}`,
                headers: {
                    'cache-control': 'no-cache'
                }
            })
            .then((res) => {
                assert.deepEqual(res.status, 200);
                return preq.get({
                    uri: `${server.config.bucketURL('en.wikipedia.beta.wmflabs.org')}/html/${encodeURIComponent(pageTitle)}/${pageRev}`,
                });
            })
            .then((res) => {
                assert.deepEqual(res.status, 200);
            });
        });
    });
});
