/*
 * Decompiled with CFR 0.152.
 */
package org.jackhuang.hmcl.util.logging;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.jackhuang.hmcl.util.Pair;
import org.jackhuang.hmcl.util.logging.CallerFinder;
import org.jackhuang.hmcl.util.logging.Level;
import org.jackhuang.hmcl.util.logging.LogEvent;
import org.tukaani.xz.LZMA2Options;
import org.tukaani.xz.XZOutputStream;

public final class Logger {
    public static final Logger LOG = new Logger();
    private static volatile String[] accessTokens = new String[0];
    static final String PACKAGE_PREFIX = "org.jackhuang.hmcl.";
    static final String CLASS_NAME = Logger.class.getName();
    private static final DateTimeFormatter TIME_FORMATTER = DateTimeFormatter.ofPattern("HH:mm:ss").withZone(ZoneId.systemDefault());
    private final BlockingQueue<LogEvent> queue = new LinkedBlockingQueue<LogEvent>();
    private final StringBuilder builder = new StringBuilder(512);
    private Path logFile;
    private ByteArrayOutputStream rawLogs;
    private PrintWriter logWriter;
    private Thread loggerThread;
    private boolean shutdown = false;
    private int logRetention = 0;

    public static synchronized void registerAccessToken(String token) {
        String[] oldAccessTokens = accessTokens;
        String[] newAccessTokens = Arrays.copyOf(oldAccessTokens, oldAccessTokens.length + 1);
        newAccessTokens[oldAccessTokens.length] = token;
        accessTokens = newAccessTokens;
    }

    public static String filterForbiddenToken(String message) {
        for (String token : accessTokens) {
            message = message.replace(token, "<access token>");
        }
        return message;
    }

    public void setLogRetention(int logRetention) {
        this.logRetention = Math.max(0, logRetention);
    }

    private String format(LogEvent.DoLog event) {
        StringBuilder builder = this.builder;
        builder.setLength(0);
        builder.append('[');
        TIME_FORMATTER.formatTo(Instant.ofEpochMilli(event.time), builder);
        builder.append("] [");
        if (event.caller != null && event.caller.startsWith(PACKAGE_PREFIX)) {
            builder.append("@.").append(event.caller, PACKAGE_PREFIX.length(), event.caller.length());
        } else {
            builder.append(event.caller);
        }
        builder.append('/').append((Object)event.level).append("] ").append(Logger.filterForbiddenToken(event.message));
        return builder.toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handle(LogEvent event) {
        if (event instanceof LogEvent.DoLog) {
            String log = this.format((LogEvent.DoLog)event);
            Throwable exception = ((LogEvent.DoLog)event).exception;
            System.out.println(log);
            if (exception != null) {
                exception.printStackTrace(System.out);
            }
            this.logWriter.println(log);
            if (exception != null) {
                exception.printStackTrace(this.logWriter);
            }
        } else if (event instanceof LogEvent.ExportLog) {
            LogEvent.ExportLog exportEvent = (LogEvent.ExportLog)event;
            this.logWriter.flush();
            try {
                if (this.logFile != null) {
                    Files.copy(this.logFile, exportEvent.output);
                }
                this.rawLogs.writeTo(exportEvent.output);
            }
            catch (IOException e) {
                exportEvent.exception = e;
            }
            finally {
                exportEvent.latch.countDown();
            }
        } else if (event instanceof LogEvent.Shutdown) {
            this.shutdown = true;
        } else {
            throw new AssertionError((Object)("Unknown event: " + event));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void onExit() {
        this.shutdown();
        try {
            this.loggerThread.join();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        String caller = CLASS_NAME + ".onExit";
        if (this.logRetention > 0 && this.logFile != null) {
            ArrayList<Pair<Path, int[]>> list = new ArrayList<Pair<Path, int[]>>();
            Pattern fileNamePattern = Pattern.compile("(?<year>\\d{4})-(?<month>\\d{2})-(?<day>\\d{2})T(?<hour>\\d{2})-(?<minute>\\d{2})-(?<second>\\d{2})(\\.(?<n>\\d+))?\\.log(\\.(gz|xz))?");
            Path dir = this.logFile.getParent();
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(dir);){
                for (Path path : stream) {
                    Matcher matcher = fileNamePattern.matcher(path.getFileName().toString());
                    if (!matcher.matches() || !Files.isRegularFile(path, new LinkOption[0])) continue;
                    int year = Integer.parseInt(matcher.group("year"));
                    int month = Integer.parseInt(matcher.group("month"));
                    int day = Integer.parseInt(matcher.group("day"));
                    int hour = Integer.parseInt(matcher.group("hour"));
                    int minute = Integer.parseInt(matcher.group("minute"));
                    int second = Integer.parseInt(matcher.group("second"));
                    int n = Optional.ofNullable(matcher.group("n")).map(Integer::parseInt).orElse(0);
                    list.add(Pair.pair(path, new int[]{year, month, day, hour, minute, second, n}));
                }
            }
            catch (IOException e) {
                this.log(Level.WARNING, caller, "Failed to list log files in " + dir, e);
            }
            if (list.size() > this.logRetention) {
                list.sort((a, b) -> {
                    int[] v1 = (int[])a.getValue();
                    int[] v2 = (int[])b.getValue();
                    assert (v1.length == v2.length);
                    for (int i = 0; i < v1.length; ++i) {
                        int c = Integer.compare(v1[i], v2[i]);
                        if (c == 0) continue;
                        return c;
                    }
                    return 0;
                });
                int end = list.size() - this.logRetention;
                for (int i = 0; i < end; ++i) {
                    Path file = (Path)((Pair)list.get(i)).getKey();
                    try {
                        if (Files.isSameFile(file, this.logFile)) continue;
                        this.log(Level.INFO, caller, "Delete old log file " + file, null);
                        Files.delete(file);
                        continue;
                    }
                    catch (IOException e) {
                        this.log(Level.WARNING, caller, "Failed to delete log file " + file, e);
                    }
                }
            }
        }
        ArrayList logs = new ArrayList();
        this.queue.drainTo(logs);
        for (LogEvent log : logs) {
            this.handle(log);
        }
        if (this.logFile == null) {
            return;
        }
        boolean failed = false;
        Path xzFile = this.logFile.resolveSibling(this.logFile.getFileName() + ".xz");
        try (XZOutputStream output = new XZOutputStream(Files.newOutputStream(xzFile, new OpenOption[0]), new LZMA2Options());){
            this.logWriter.flush();
            Files.copy(this.logFile, output);
        }
        catch (IOException e) {
            failed = true;
            this.handle(new LogEvent.DoLog(System.currentTimeMillis(), caller, Level.WARNING, "Failed to dump log file to xz format", e));
        }
        finally {
            this.logWriter.close();
        }
        if (!failed) {
            try {
                Files.delete(this.logFile);
            }
            catch (IOException e) {
                System.err.println("An exception occurred while deleting raw log file");
                e.printStackTrace(System.err);
            }
        }
    }

    public void start(Path logFolder) {
        if (logFolder != null) {
            String time = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH-mm-ss"));
            try {
                Files.createDirectories(logFolder, new FileAttribute[0]);
                int n = 0;
                while (true) {
                    Path file = logFolder.resolve(time + (n == 0 ? "" : "." + n) + ".log").toAbsolutePath().normalize();
                    try {
                        this.logWriter = new PrintWriter(Files.newBufferedWriter(file, StandardCharsets.UTF_8, StandardOpenOption.CREATE_NEW));
                        this.logFile = file;
                    }
                    catch (FileAlreadyExistsException fileAlreadyExistsException) {
                        ++n;
                        continue;
                    }
                    break;
                }
            }
            catch (IOException e) {
                this.log(Level.WARNING, CLASS_NAME + ".start", "Failed to create log file", e);
            }
        }
        if (this.logWriter == null) {
            this.rawLogs = new ByteArrayOutputStream(262144);
            this.logWriter = new PrintWriter(new OutputStreamWriter((OutputStream)this.rawLogs, StandardCharsets.UTF_8));
        }
        this.loggerThread = new Thread(() -> {
            ArrayList logs = new ArrayList();
            try {
                while (!this.shutdown) {
                    if (this.queue.drainTo(logs) > 0) {
                        for (LogEvent log : logs) {
                            this.handle(log);
                        }
                        logs.clear();
                        continue;
                    }
                    this.logWriter.flush();
                    this.handle(this.queue.take());
                }
                while (this.queue.drainTo(logs) > 0) {
                    for (LogEvent log : logs) {
                        this.handle(log);
                    }
                    logs.clear();
                }
            }
            catch (InterruptedException e) {
                throw new AssertionError("This thread cannot be interrupted", e);
            }
        });
        this.loggerThread.setName("HMCL Logger Thread");
        this.loggerThread.start();
        Thread cleanerThread = new Thread(this::onExit);
        cleanerThread.setName("HMCL Logger Shutdown Hook");
        Runtime.getRuntime().addShutdownHook(cleanerThread);
    }

    public void shutdown() {
        this.queue.add(new LogEvent.Shutdown());
    }

    public Path getLogFile() {
        return this.logFile;
    }

    public void exportLogs(OutputStream output) throws IOException {
        Objects.requireNonNull(output);
        LogEvent.ExportLog event = new LogEvent.ExportLog(output);
        try {
            this.queue.put(event);
            event.await();
        }
        catch (InterruptedException e) {
            throw new AssertionError("This thread cannot be interrupted", e);
        }
        if (event.exception != null) {
            throw event.exception;
        }
    }

    public String getLogs() {
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        try {
            this.exportLogs(output);
            return output.toString("UTF-8");
        }
        catch (IOException e) {
            this.log(Level.WARNING, CLASS_NAME + ".getLogs", "Failed to export logs", e);
            return "";
        }
    }

    private void log(Level level, String caller, String msg, Throwable exception) {
        this.queue.add(new LogEvent.DoLog(System.currentTimeMillis(), caller, level, msg, exception));
    }

    public void log(Level level, String msg) {
        this.log(level, CallerFinder.getCaller(), msg, null);
    }

    public void log(Level level, String msg, Throwable exception) {
        this.log(level, CallerFinder.getCaller(), msg, exception);
    }

    public void error(String msg) {
        this.log(Level.ERROR, CallerFinder.getCaller(), msg, null);
    }

    public void error(String msg, Throwable exception) {
        this.log(Level.ERROR, CallerFinder.getCaller(), msg, exception);
    }

    public void warning(String msg) {
        this.log(Level.WARNING, CallerFinder.getCaller(), msg, null);
    }

    public void warning(String msg, Throwable exception) {
        this.log(Level.WARNING, CallerFinder.getCaller(), msg, exception);
    }

    public void info(String msg) {
        this.log(Level.INFO, CallerFinder.getCaller(), msg, null);
    }

    public void info(String msg, Throwable exception) {
        this.log(Level.INFO, CallerFinder.getCaller(), msg, exception);
    }

    public void debug(String msg) {
        this.log(Level.DEBUG, CallerFinder.getCaller(), msg, null);
    }

    public void debug(String msg, Throwable exception) {
        this.log(Level.DEBUG, CallerFinder.getCaller(), msg, exception);
    }

    public void trace(String msg) {
        this.log(Level.TRACE, CallerFinder.getCaller(), msg, null);
    }

    public void trace(String msg, Throwable exception) {
        this.log(Level.TRACE, CallerFinder.getCaller(), msg, exception);
    }
}

