var documenterSearchIndex = {"docs":
[{"title":"Constructors","page":"Constructors","location":"manual/constructors.html#man-constructors","category":"section","text":""},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"Constructors [1] are functions that create new objects – specifically, instances of Composite Types. In Julia, type objects also serve as constructor functions: they create new instances of themselves when applied to an argument tuple as a function. This much was already mentioned briefly when composite types were introduced. For example:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> struct Foo\n           bar\n           baz\n       end\n\njulia> foo = Foo(1, 2)\nFoo(1, 2)\n\njulia> foo.bar\n1\n\njulia> foo.baz\n2"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"For many types, forming new objects by binding their field values together is all that is ever needed to create instances. However, in some cases more functionality is required when creating composite objects. Sometimes invariants must be enforced, either by checking arguments or by transforming them. Recursive data structures, especially those that may be self-referential, often cannot be constructed cleanly without first being created in an incomplete state and then altered programmatically to be made whole, as a separate step from object creation. Sometimes, it's just convenient to be able to construct objects with fewer or different types of parameters than they have fields. Julia's system for object construction addresses all of these cases and more."},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"[1]: Nomenclature: while the term \"constructor\" generally refers to the entire function which constructs objects of a type, it is common to abuse terminology slightly and refer to specific constructor methods as \"constructors\". In such situations, it is generally clear from the context that the term is used to mean \"constructor method\" rather than \"constructor function\", especially as it is often used in the sense of singling out a particular method of the constructor from all of the others."},{"title":"Outer Constructor Methods","page":"Constructors","location":"manual/constructors.html#man-outer-constructor-methods","category":"section","text":""},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"A constructor is just like any other function in Julia in that its overall behavior is defined by the combined behavior of its methods. Accordingly, you can add functionality to a constructor by simply defining new methods. For example, let's say you want to add a constructor method for Foo objects that takes only one argument and uses the given value for both the bar and baz fields. This is simple:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> Foo(x) = Foo(x,x)\nFoo\n\njulia> Foo(1)\nFoo(1, 1)"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"You could also add a zero-argument Foo constructor method that supplies default values for both of the bar and baz fields:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> Foo() = Foo(0)\nFoo\n\njulia> Foo()\nFoo(0, 0)"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"Here the zero-argument constructor method calls the single-argument constructor method, which in turn calls the automatically provided two-argument constructor method. For reasons that will become clear very shortly, additional constructor methods declared as normal methods like this are called outer constructor methods. Outer constructor methods can only ever create a new instance by calling another constructor method, such as the automatically provided default ones."},{"title":"Inner Constructor Methods","page":"Constructors","location":"manual/constructors.html#man-inner-constructor-methods","category":"section","text":""},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"While outer constructor methods succeed in addressing the problem of providing additional convenience methods for constructing objects, they fail to address the other two use cases mentioned in the introduction of this chapter: enforcing invariants, and allowing construction of self-referential objects. For these problems, one needs inner constructor methods. An inner constructor method is like an outer constructor method, except for two differences:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"It is declared inside the block of a type declaration, rather than outside of it like normal methods.\nIt has access to a special locally existent function called new that creates objects of the block's type."},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"For example, suppose one wants to declare a type that holds a pair of real numbers, subject to the constraint that the first number is not greater than the second one. One could declare it like this:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> struct OrderedPair\n           x::Real\n           y::Real\n           OrderedPair(x,y) = x > y ? error(\"out of order\") : new(x,y)\n       end"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"Now OrderedPair objects can only be constructed such that x <= y:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> OrderedPair(1, 2)\nOrderedPair(1, 2)\n\njulia> OrderedPair(2,1)\nERROR: out of order\nStacktrace:\n [1] error at ./error.jl:33 [inlined]\n [2] OrderedPair(::Int64, ::Int64) at ./none:4\n [3] top-level scope"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"If the type were declared mutable, you could reach in and directly change the field values to violate this invariant. Of course, messing around with an object's internals uninvited is bad practice. You (or someone else) can also provide additional outer constructor methods at any later point, but once a type is declared, there is no way to add more inner constructor methods. Since outer constructor methods can only create objects by calling other constructor methods, ultimately, some inner constructor must be called to create an object. This guarantees that all objects of the declared type must come into existence by a call to one of the inner constructor methods provided with the type, thereby giving some degree of enforcement of a type's invariants."},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"If any inner constructor method is defined, no default constructor method is provided: it is presumed that you have supplied yourself with all the inner constructors you need. The default constructor is equivalent to writing your own inner constructor method that takes all of the object's fields as parameters (constrained to be of the correct type, if the corresponding field has a type), and passes them to new, returning the resulting object:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> struct Foo\n           bar\n           baz\n           Foo(bar,baz) = new(bar,baz)\n       end\n"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"This declaration has the same effect as the earlier definition of the Foo type without an explicit inner constructor method. The following two types are equivalent – one with a default constructor, the other with an explicit constructor:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> struct T1\n           x::Int64\n       end\n\njulia> struct T2\n           x::Int64\n           T2(x) = new(x)\n       end\n\njulia> T1(1)\nT1(1)\n\njulia> T2(1)\nT2(1)\n\njulia> T1(1.0)\nT1(1)\n\njulia> T2(1.0)\nT2(1)"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"It is good practice to provide as few inner constructor methods as possible: only those taking all arguments explicitly and enforcing essential error checking and transformation. Additional convenience constructor methods, supplying default values or auxiliary transformations, should be provided as outer constructors that call the inner constructors to do the heavy lifting. This separation is typically quite natural."},{"title":"Incomplete Initialization","page":"Constructors","location":"manual/constructors.html#Incomplete-Initialization","category":"section","text":""},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"The final problem which has still not been addressed is construction of self-referential objects, or more generally, recursive data structures. Since the fundamental difficulty may not be immediately obvious, let us briefly explain it. Consider the following recursive type declaration:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> mutable struct SelfReferential\n           obj::SelfReferential\n       end\n"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"This type may appear innocuous enough, until one considers how to construct an instance of it. If a is an instance of SelfReferential, then a second instance can be created by the call:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> b = SelfReferential(a)"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"But how does one construct the first instance when no instance exists to provide as a valid value for its obj field? The only solution is to allow creating an incompletely initialized instance of SelfReferential with an unassigned obj field, and using that incomplete instance as a valid value for the obj field of another instance, such as, for example, itself."},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"To allow for the creation of incompletely initialized objects, Julia allows the new function to be called with fewer than the number of fields that the type has, returning an object with the unspecified fields uninitialized. The inner constructor method can then use the incomplete object, finishing its initialization before returning it. Here, for example, is another attempt at defining the SelfReferential type, this time using a zero-argument inner constructor returning instances having obj fields pointing to themselves:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> mutable struct SelfReferential\n           obj::SelfReferential\n           SelfReferential() = (x = new(); x.obj = x)\n       end\n"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"We can verify that this constructor works and constructs objects that are, in fact, self-referential:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> x = SelfReferential();\n\njulia> x === x\ntrue\n\njulia> x === x.obj\ntrue\n\njulia> x === x.obj.obj\ntrue"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"Although it is generally a good idea to return a fully initialized object from an inner constructor, it is possible to return incompletely initialized objects:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> mutable struct Incomplete\n           data\n           Incomplete() = new()\n       end\n\njulia> z = Incomplete();"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"While you are allowed to create objects with uninitialized fields, any access to an uninitialized reference is an immediate error:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> z.data\nERROR: UndefRefError: access to undefined reference"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"This avoids the need to continually check for null values. However, not all object fields are references. Julia considers some types to be \"plain data\", meaning all of their data is self-contained and does not reference other objects. The plain data types consist of primitive types (e.g. Int) and immutable structs of other plain data types (see also: isbits, isbitstype). The initial contents of a plain data type is undefined:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> struct HasPlain\n           n::Int\n           HasPlain() = new()\n       end\n\njulia> HasPlain()\nHasPlain(438103441441)"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"Arrays of plain data types exhibit the same behavior."},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"You can pass incomplete objects to other functions from inner constructors to delegate their completion:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> mutable struct Lazy\n           data\n           Lazy(v) = complete_me(new(), v)\n       end"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"As with incomplete objects returned from constructors, if complete_me or any of its callees try to access the data field of the Lazy object before it has been initialized, an error will be thrown immediately."},{"title":"Parametric Constructors","page":"Constructors","location":"manual/constructors.html#Parametric-Constructors","category":"section","text":""},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"Parametric types add a few wrinkles to the constructor story. Recall from Parametric Types that, by default, instances of parametric composite types can be constructed either with explicitly given type parameters or with type parameters implied by the types of the arguments given to the constructor. Here are some examples:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> struct Point{T<:Real}\n           x::T\n           y::T\n       end\n\njulia> Point(1,2) ## implicit T ##\nPoint{Int64}(1, 2)\n\njulia> Point(1.0,2.5) ## implicit T ##\nPoint{Float64}(1.0, 2.5)\n\njulia> Point(1,2.5) ## implicit T ##\nERROR: MethodError: no method matching Point(::Int64, ::Float64)\nThe type `Point` exists, but no method is defined for this combination of argument types when trying to construct it.\n\nClosest candidates are:\n  Point(::T, ::T) where T<:Real at none:2\n\njulia> Point{Int64}(1, 2) ## explicit T ##\nPoint{Int64}(1, 2)\n\njulia> Point{Int64}(1.0,2.5) ## explicit T ##\nERROR: InexactError: Int64(2.5)\nStacktrace:\n[...]\n\njulia> Point{Float64}(1.0, 2.5) ## explicit T ##\nPoint{Float64}(1.0, 2.5)\n\njulia> Point{Float64}(1,2) ## explicit T ##\nPoint{Float64}(1.0, 2.0)"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"As you can see, for constructor calls with explicit type parameters, the arguments are converted to the implied field types: Point{Int64}(1,2) works, but Point{Int64}(1.0,2.5) raises an InexactError when converting 2.5 to Int64. When the type is implied by the arguments to the constructor call, as in Point(1,2), then the types of the arguments must agree – otherwise the T cannot be determined – but any pair of real arguments with matching type may be given to the generic Point constructor."},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"What's really going on here is that Point, Point{Float64} and Point{Int64} are all different constructor functions. In fact, Point{T} is a distinct constructor function for each type T. Without any explicitly provided inner constructors, the declaration of the composite type Point{T<:Real} automatically provides an inner constructor, Point{T}, for each possible type T<:Real, that behaves just like non-parametric default inner constructors do. It also provides a single general outer Point constructor that takes pairs of real arguments, which must be of the same type. This automatic provision of constructors is equivalent to the following explicit declaration:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> struct Point{T<:Real}\n           x::T\n           y::T\n           Point{T}(x,y) where {T<:Real} = new(x,y)\n       end\n\njulia> Point(x::T, y::T) where {T<:Real} = Point{T}(x,y);"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"Notice that each definition looks like the form of constructor call that it handles. The call Point{Int64}(1,2) will invoke the definition Point{T}(x,y) inside the struct block. The outer constructor declaration, on the other hand, defines a method for the general Point constructor which only applies to pairs of values of the same real type. This declaration makes constructor calls without explicit type parameters, like Point(1,2) and Point(1.0,2.5), work. Since the method declaration restricts the arguments to being of the same type, calls like Point(1,2.5), with arguments of different types, result in \"no method\" errors."},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"Suppose we wanted to make the constructor call Point(1,2.5) work by \"promoting\" the integer value 1 to the floating-point value 1.0. The simplest way to achieve this is to define the following additional outer constructor method:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> Point(x::Int64, y::Float64) = Point(convert(Float64,x),y);"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"This method uses the convert function to explicitly convert x to Float64 and then delegates construction to the general constructor for the case where both arguments are Float64. With this method definition what was previously a MethodError now successfully creates a point of type Point{Float64}:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> p = Point(1,2.5)\nPoint{Float64}(1.0, 2.5)\n\njulia> typeof(p)\nPoint{Float64}"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"However, other similar calls still don't work:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> Point(1.5,2)\nERROR: MethodError: no method matching Point(::Float64, ::Int64)\nThe type `Point` exists, but no method is defined for this combination of argument types when trying to construct it.\n\nClosest candidates are:\n  Point(::T, !Matched::T) where T<:Real\n   @ Main none:1\n  Point(!Matched::Int64, !Matched::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"For a more general way to make all such calls work sensibly, see Conversion and Promotion. At the risk of spoiling the suspense, we can reveal here that all it takes is the following outer method definition to make all calls to the general Point constructor work as one would expect:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> Point(x::Real, y::Real) = Point(promote(x,y)...);"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"The promote function converts all its arguments to a common type – in this case Float64. With this method definition, the Point constructor promotes its arguments the same way that numeric operators like + do, and works for all kinds of real numbers:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> Point(1.5,2)\nPoint{Float64}(1.5, 2.0)\n\njulia> Point(1,1//2)\nPoint{Rational{Int64}}(1//1, 1//2)\n\njulia> Point(1.0,1//2)\nPoint{Float64}(1.0, 0.5)"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"Thus, while the implicit type parameter constructors provided by default in Julia are fairly strict, it is possible to make them behave in a more relaxed but sensible manner quite easily. Moreover, since constructors can leverage all of the power of the type system, methods, and multiple dispatch, defining sophisticated behavior is typically quite simple."},{"title":"Case Study: Rational","page":"Constructors","location":"manual/constructors.html#Case-Study:-Rational","category":"section","text":""},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"Perhaps the best way to tie all these pieces together is to present a real world example of a parametric composite type and its constructor methods. To that end, we implement our own rational number type OurRational, similar to Julia's built-in Rational type, defined in rational.jl:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> struct OurRational{T<:Integer} <: Real\n           num::T\n           den::T\n           function OurRational{T}(num::T, den::T) where T<:Integer\n               if num == 0 && den == 0\n                    error(\"invalid rational: 0//0\")\n               end\n               num = flipsign(num, den)\n               den = flipsign(den, den)\n               g = gcd(num, den)\n               num = div(num, g)\n               den = div(den, g)\n               new(num, den)\n           end\n       end\n\njulia> OurRational(n::T, d::T) where {T<:Integer} = OurRational{T}(n,d)\nOurRational\n\njulia> OurRational(n::Integer, d::Integer) = OurRational(promote(n,d)...)\nOurRational\n\njulia> OurRational(n::Integer) = OurRational(n,one(n))\nOurRational\n\njulia> ⊘(n::Integer, d::Integer) = OurRational(n,d)\n⊘ (generic function with 1 method)\n\njulia> ⊘(x::OurRational, y::Integer) = x.num ⊘ (x.den*y)\n⊘ (generic function with 2 methods)\n\njulia> ⊘(x::Integer, y::OurRational) = (x*y.den) ⊘ y.num\n⊘ (generic function with 3 methods)\n\njulia> ⊘(x::Complex, y::Real) = complex(real(x) ⊘ y, imag(x) ⊘ y)\n⊘ (generic function with 4 methods)\n\njulia> ⊘(x::Real, y::Complex) = (x*y') ⊘ real(y*y')\n⊘ (generic function with 5 methods)\n\njulia> function ⊘(x::Complex, y::Complex)\n           xy = x*y'\n           yy = real(y*y')\n           complex(real(xy) ⊘ yy, imag(xy) ⊘ yy)\n       end\n⊘ (generic function with 6 methods)"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"The first line – struct OurRational{T<:Integer} <: Real – declares that OurRational takes one type parameter of an integer type, and is itself a real type. The field declarations num::T and den::T indicate that the data held in a OurRational{T} object are a pair of integers of type T, one representing the rational value's numerator and the other representing its denominator."},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"Now things get interesting. OurRational has a single inner constructor method which checks that num and den aren't both zero and ensures that every rational is constructed in \"lowest terms\" with a non-negative denominator. This is accomplished by first flipping the signs of numerator and denominator if the denominator is negative. Then, both are divided by their greatest common divisor (gcd always returns a non-negative number, regardless of the sign of its arguments). Because this is the only inner constructor for OurRational, we can be certain that OurRational objects are always constructed in this normalized form."},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"OurRational also provides several outer constructor methods for convenience. The first is the \"standard\" general constructor that infers the type parameter T from the type of the numerator and denominator when they have the same type. The second applies when the given numerator and denominator values have different types: it promotes them to a common type and then delegates construction to the outer constructor for arguments of matching type. The third outer constructor turns integer values into rationals by supplying a value of 1 as the denominator."},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"Following the outer constructor definitions, we defined a number of methods for the ⊘ operator, which provides a syntax for writing rationals (e.g. 1 ⊘ 2). Julia's Rational type uses the // operator for this purpose. Before these definitions, ⊘ is a completely undefined operator with only syntax and no meaning. Afterwards, it behaves just as described in Rational Numbers – its entire behavior is defined in these few lines. Note that the infix use of ⊘ works because Julia has a set of symbols that are recognized to be infix operators. The first and most basic definition just makes a ⊘ b construct a OurRational by applying the OurRational constructor to a and b when they are integers. When one of the operands of ⊘ is already a rational number, we construct a new rational for the resulting ratio slightly differently; this behavior is actually identical to division of a rational with an integer. Finally, applying ⊘ to complex integral values creates an instance of Complex{<:OurRational} – a complex number whose real and imaginary parts are rationals:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> z = (1 + 2im) ⊘ (1 - 2im);\n\njulia> typeof(z)\nComplex{OurRational{Int64}}\n\njulia> typeof(z) <: Complex{<:OurRational}\ntrue"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"Thus, although the ⊘ operator usually returns an instance of OurRational, if either of its arguments are complex integers, it will return an instance of Complex{<:OurRational} instead. The interested reader should consider perusing the rest of rational.jl: it is short, self-contained, and implements an entire basic Julia type."},{"title":"Outer-only constructors","page":"Constructors","location":"manual/constructors.html#Outer-only-constructors","category":"section","text":""},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"As we have seen, a typical parametric type has inner constructors that are called when type parameters are known; e.g. they apply to Point{Int} but not to Point. Optionally, outer constructors that determine type parameters automatically can be added, for example constructing a Point{Int} from the call Point(1,2). Outer constructors call inner constructors to actually make instances. However, in some cases one would rather not provide inner constructors, so that specific type parameters cannot be requested manually."},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"For example, say we define a type that stores a vector along with an accurate representation of its sum:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> struct SummedArray{T<:Number,S<:Number}\n           data::Vector{T}\n           sum::S\n       end\n\njulia> SummedArray(Int32[1; 2; 3], Int32(6))\nSummedArray{Int32, Int32}(Int32[1, 2, 3], 6)"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"The problem is that we want S to be a larger type than T, so that we can sum many elements with less information loss. For example, when T is Int32, we would like S to be Int64. Therefore we want to avoid an interface that allows the user to construct instances of the type SummedArray{Int32,Int32}. One way to do this is to provide a constructor only for SummedArray, but inside the struct definition block to suppress generation of default constructors:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> struct SummedArray{T<:Number,S<:Number}\n           data::Vector{T}\n           sum::S\n           function SummedArray(a::Vector{T}) where T\n               S = widen(T)\n               new{T,S}(a, sum(S, a))\n           end\n       end\n\njulia> SummedArray(Int32[1; 2; 3], Int32(6))\nERROR: MethodError: no method matching SummedArray(::Vector{Int32}, ::Int32)\nThe type `SummedArray` exists, but no method is defined for this combination of argument types when trying to construct it.\n\nClosest candidates are:\n  SummedArray(::Vector{T}) where T\n   @ Main none:4\n\nStacktrace:\n[...]"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"This constructor will be invoked by the syntax SummedArray(a). The syntax new{T,S} allows specifying parameters for the type to be constructed, i.e. this call will return a SummedArray{T,S}. new{T,S} can be used in any constructor definition, but for convenience the parameters to new{} are automatically derived from the type being constructed when possible."},{"title":"Constructors are just callable objects","page":"Constructors","location":"manual/constructors.html#Constructors-are-just-callable-objects","category":"section","text":""},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"An object of any type may be made callable by defining a method. This includes types, i.e., objects of type Type; and constructors may, in fact, be viewed as just callable type objects. For example, there are many methods defined on Bool and various supertypes of it:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"methods(Bool)"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"The usual constructor syntax is exactly equivalent to the function-like object syntax, so trying to define a method with each syntax will cause the first method to be overwritten by the next one:"},{"title":"Constructors","page":"Constructors","location":"manual/constructors.html","category":"page","text":"julia> struct S\n           f::Int\n       end\n\njulia> S() = S(7)\nS\n\njulia> (::Type{S})() = S(8)  # overwrites the previous constructor method\n\njulia> S()\nS(8)"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html#SubArrays","category":"section","text":""},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Julia's SubArray type is a container encoding a \"view\" of a parent AbstractArray. This page documents some of the design principles and implementation of SubArrays."},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"One of the major design goals is to ensure high performance for views of both IndexLinear and IndexCartesian arrays. Furthermore, views of IndexLinear arrays should themselves be IndexLinear to the extent that it is possible."},{"title":"Index replacement","page":"SubArrays","location":"devdocs/subarrays.html#Index-replacement","category":"section","text":""},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Consider making 2d slices of a 3d array:"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> A = rand(2,3,4);\n\njulia> S1 = view(A, :, 1, 2:3)\n2×2 view(::Array{Float64, 3}, :, 1, 2:3) with eltype Float64:\n 0.839622  0.711389\n 0.967143  0.103929\n\njulia> S2 = view(A, 1, :, 2:3)\n3×2 view(::Array{Float64, 3}, 1, :, 2:3) with eltype Float64:\n 0.839622  0.711389\n 0.789764  0.806704\n 0.566704  0.962715"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"view drops \"singleton\" dimensions (ones that are specified by an Int), so both S1 and S2 are two-dimensional SubArrays. Consequently, the natural way to index these is with S1[i,j]. To extract the value from the parent array A, the natural approach is to replace S1[i,j] with A[i,1,(2:3)[j]] and S2[i,j] with A[1,i,(2:3)[j]]."},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"The key feature of the design of SubArrays is that this index replacement can be performed without any runtime overhead."},{"title":"SubArray design","page":"SubArrays","location":"devdocs/subarrays.html#SubArray-design","category":"section","text":""},{"title":"Type parameters and fields","page":"SubArrays","location":"devdocs/subarrays.html#Type-parameters-and-fields","category":"section","text":""},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"The strategy adopted is first and foremost expressed in the definition of the type:"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"struct SubArray{T,N,P,I,L} <: AbstractArray{T,N}\n    parent::P\n    indices::I\n    offset1::Int       # for linear indexing and pointer, only valid when L==true\n    stride1::Int       # used only for linear indexing\n    ...\nend"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"SubArray has 5 type parameters. The first two are the standard element type and dimensionality.  The next is the type of the parent AbstractArray. The most heavily-used is the fourth parameter, a Tuple of the types of the indices for each dimension. The final one, L, is only provided as a convenience for dispatch; it's a boolean that represents whether the index types support fast linear indexing. More on that later."},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"If in our example above A is a Array{Float64, 3}, our S1 case above would be a SubArray{Float64,2,Array{Float64,3},Tuple{Base.Slice{Base.OneTo{Int64}},Int64,UnitRange{Int64}},false}. Note in particular the tuple parameter, which stores the types of the indices used to create S1. Likewise,"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> S1.indices\n(Base.Slice(Base.OneTo(2)), 1, 2:3)"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Storing these values allows index replacement, and having the types encoded as parameters allows one to dispatch to efficient algorithms."},{"title":"Index translation","page":"SubArrays","location":"devdocs/subarrays.html#Index-translation","category":"section","text":""},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Performing index translation requires that you do different things for different concrete SubArray types. For example, for S1, one needs to apply the i,j indices to the first and third dimensions of the parent array, whereas for S2 one needs to apply them to the second and third. The simplest approach to indexing would be to do the type-analysis at runtime:"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"parentindices = Vector{Any}()\nfor thisindex in S.indices\n    ...\n    if isa(thisindex, Int)\n        # Don't consume one of the input indices\n        push!(parentindices, thisindex)\n    elseif isa(thisindex, AbstractVector)\n        # Consume an input index\n        push!(parentindices, thisindex[inputindex[j]])\n        j += 1\n    elseif isa(thisindex, AbstractMatrix)\n        # Consume two input indices\n        push!(parentindices, thisindex[inputindex[j], inputindex[j+1]])\n        j += 2\n    elseif ...\nend\nS.parent[parentindices...]"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Unfortunately, this would be disastrous in terms of performance: each element access would allocate memory, and involves the running of a lot of poorly-typed code."},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"The better approach is to dispatch to specific methods to handle each type of stored index. That's what reindex does: it dispatches on the type of the first stored index and consumes the appropriate number of input indices, and then it recurses on the remaining indices. In the case of S1, this expands to"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Base.reindex(S1, S1.indices, (i, j)) == (i, S1.indices[2], S1.indices[3][j])"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"for any pair of indices (i,j) (except CartesianIndexs and arrays thereof, see below)."},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"This is the core of a SubArray; indexing methods depend upon reindex to do this index translation. Sometimes, though, we can avoid the indirection and make it even faster."},{"title":"Linear indexing","page":"SubArrays","location":"devdocs/subarrays.html#Linear-indexing","category":"section","text":""},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Linear indexing can be implemented efficiently when the entire array has a single stride that separates successive elements, starting from some offset. This means that we can pre-compute these values and represent linear indexing simply as an addition and multiplication, avoiding the indirection of reindex and (more importantly) the slow computation of the cartesian coordinates entirely."},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"For SubArray types, the availability of efficient linear indexing is based purely on the types of the indices, and does not depend on values like the size of the parent array. You can ask whether a given set of indices supports fast linear indexing with the internal Base.viewindexing function:"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> Base.viewindexing(S1.indices)\nIndexCartesian()\n\njulia> Base.viewindexing(S2.indices)\nIndexLinear()"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"This is computed during construction of the SubArray and stored in the L type parameter as a boolean that encodes fast linear indexing support. While not strictly necessary, it means that we can define dispatch directly on SubArray{T,N,A,I,true} without any intermediaries."},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Since this computation doesn't depend on runtime values, it can miss some cases in which the stride happens to be uniform:"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> A = reshape(1:4*2, 4, 2)\n4×2 reshape(::UnitRange{Int64}, 4, 2) with eltype Int64:\n 1  5\n 2  6\n 3  7\n 4  8\n\njulia> diff(A[2:2:4,:][:])\n3-element Vector{Int64}:\n 2\n 2\n 2"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"A view constructed as view(A, 2:2:4, :) happens to have uniform stride, and therefore linear indexing indeed could be performed efficiently. However, success in this case depends on the size of the array: if the first dimension instead were odd,"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> A = reshape(1:5*2, 5, 2)\n5×2 reshape(::UnitRange{Int64}, 5, 2) with eltype Int64:\n 1   6\n 2   7\n 3   8\n 4   9\n 5  10\n\njulia> diff(A[2:2:4,:][:])\n3-element Vector{Int64}:\n 2\n 3\n 2"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"then A[2:2:4,:] does not have uniform stride, so we cannot guarantee efficient linear indexing.  Since we have to base this decision based purely on types encoded in the parameters of the SubArray, S = view(A, 2:2:4, :) cannot implement efficient linear indexing."},{"title":"A few details","page":"SubArrays","location":"devdocs/subarrays.html#A-few-details","category":"section","text":""},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Note that the Base.reindex function is agnostic to the types of the input indices; it simply determines how and where the stored indices should be reindexed. It not only supports integer indices, but it supports non-scalar indexing, too. This means that views of views don't need two levels of indirection; they can simply re-compute the indices into the original parent array!\nHopefully by now it's fairly clear that supporting slices means that the dimensionality, given by the parameter N, is not necessarily equal to the dimensionality of the parent array or the length of the indices tuple. Neither do user-supplied indices necessarily line up with entries in the indices tuple (e.g., the second user-supplied index might correspond to the third dimension of the parent array, and the third element in the indices tuple).\nWhat might be less obvious is that the dimensionality of the stored parent array must be equal to the number of effective indices in the indices tuple. Some examples:\nA = reshape(1:35, 5, 7) # A 2d parent Array\nS = view(A, 2:7)         # A 1d view created by linear indexing\nS = view(A, :, :, 1:1)   # Appending extra indices is supported\nNaively, you'd think you could just set S.parent = A and S.indices = (:,:,1:1), but supporting this dramatically complicates the reindexing process, especially for views of views. Not only do you need to dispatch on the types of the stored indices, but you need to examine whether a given index is the final one and \"merge\" any remaining stored indices together. This is not an easy task, and even worse: it's slow since it implicitly depends upon linear indexing.\nFortunately, this is precisely the computation that ReshapedArray performs, and it does so linearly if possible. Consequently, view ensures that the parent array is the appropriate dimensionality for the given indices by reshaping it if needed. The inner SubArray constructor ensures that this invariant is satisfied.\nCartesianIndex and arrays thereof throw a nasty wrench into the reindex scheme. Recall that reindex simply dispatches on the type of the stored indices in order to determine how many passed indices should be used and where they should go. But with CartesianIndex, there's no longer a one-to-one correspondence between the number of passed arguments and the number of dimensions that they index into. If we return to the above example of Base.reindex(S1, S1.indices, (i, j)), you can see that the expansion is incorrect for i, j = CartesianIndex(), CartesianIndex(2,1). It should skip the CartesianIndex() entirely and return:\n(CartesianIndex(2,1)[1], S1.indices[2], S1.indices[3][CartesianIndex(2,1)[2]])\nInstead, though, we get:\n(CartesianIndex(), S1.indices[2], S1.indices[3][CartesianIndex(2,1)])\nDoing this correctly would require combined dispatch on both the stored and passed indices across all combinations of dimensionalities in an intractable manner. As such, reindex must never be called with CartesianIndex indices. Fortunately, the scalar case is easily handled by first flattening the CartesianIndex arguments to plain integers. Arrays of CartesianIndex, however, cannot be split apart into orthogonal pieces so easily. Before attempting to use reindex, view must ensure that there are no arrays of CartesianIndex in the argument list. If there are, it can simply \"punt\" by avoiding the reindex calculation entirely, constructing a nested SubArray with two levels of indirection instead."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html#High-level-Overview-of-the-Native-Code-Generation-Process","category":"section","text":""},{"title":"Representation of Pointers","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html#Representation-of-Pointers","category":"section","text":""},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"When emitting code to an object file, pointers will be emitted as relocations. The deserialization code will ensure any object that pointed to one of these constants gets recreated and contains the right runtime pointer."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"Otherwise, they will be emitted as literal constants."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"To emit one of these objects, call literal_pointer_val. It'll handle tracking the Julia value and the LLVM global, ensuring they are valid both for the current runtime and after deserialization."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"When emitted into the object file, these globals are stored as references in a large gvals table. This allows the deserializer to reference them by index, and implement a custom manual mechanism similar to a Global Offset Table (GOT) to restore them."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"Function pointers are handled similarly. They are stored as values in a large fvals table. Like globals, this allows the deserializer to reference them by index."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"Note that extern functions are handled separately, with names, via the usual symbol resolution mechanism in the linker."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"Note too that ccall functions are also handled separately, via a manual GOT and Procedure Linkage Table (PLT)."},{"title":"Representation of Intermediate Values","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html#Representation-of-Intermediate-Values","category":"section","text":""},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"Values are passed around in a jl_cgval_t struct. This represents an R-value, and includes enough information to determine how to assign or pass it somewhere."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"They are created via one of the helper constructors, usually: mark_julia_type (for immediate values) and mark_julia_slot (for pointers to values)."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"The function convert_julia_type can transform between any two types. It returns an R-value with cgval.typ set to typ. It'll cast the object to the requested representation, making heap boxes, allocating stack copies, and computing tagged unions as needed to change the representation."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"By contrast update_julia_type will change cgval.typ to typ, only if it can be done at zero-cost (i.e. without emitting any code)."},{"title":"Union representation","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html#Union-representation","category":"section","text":""},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"Inferred union types may be stack allocated via a tagged type representation."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"The primitive routines that need to be able to handle tagged unions are:"},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"mark-type\nload-local\nstore-local\nisa\nis\nemit_typeof\nemit_sizeof\nboxed\nunbox\nspecialized cc-ret"},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"Everything else should be possible to handle in inference by using these primitives to implement union-splitting."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"The representation of the tagged-union is as a pair of < void* union, byte selector >. The selector is fixed-size as byte & 0x7f, and will union-tag the first 126 isbits. It records the one-based depth-first count into the type-union of the isbits objects inside. An index of zero indicates that the union* is actually a tagged heap-allocated jl_value_t*, and needs to be treated as normal for a boxed object rather than as a tagged union."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"The high bit of the selector (byte & 0x80) can be tested to determine if the void* is actually a heap-allocated (jl_value_t*) box, thus avoiding the cost of re-allocating a box, while maintaining the ability to efficiently handle union-splitting based on the low bits."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"It is guaranteed that byte & 0x7f is an exact test for the type, if the value can be represented by a tag – it will never be marked byte = 0x80. It is not necessary to also test the type-tag when testing isa."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"The union* memory region may be allocated at any size. The only constraint is that it is big enough to contain the data currently specified by selector. It might not be big enough to contain the union of all types that could be stored there according to the associated Union type field. Use appropriate care when copying."},{"title":"Specialized Calling Convention Signature Representation","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html#Specialized-Calling-Convention-Signature-Representation","category":"section","text":""},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"A jl_returninfo_t object describes the specialized calling convention details of any callable. It can be generated from any (specTypes, rettype) pair, such as a CodeInstance, or other place they are declared. This is the expected calling convention for specptr, but other data may be stored there. Only if the function pointer stored there has the expected specialized calling convention will the corresponding flag be set in specsigflags to indicate it is useable."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"If any of the arguments or return type of a method can be represented unboxed, and none are unable to be represented unboxed (such as an unbounded vararg), it will be given an optimized calling convention signature based on the specTypes and rettype values."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"The general principles are that:"},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"Primitive types get passed in int/float registers.\nTuples of VecElement types get passed in vector registers.\nStructs get passed on the stack.\nReturn values are handled similarly to arguments, with a size-cutoff at which they will instead be returned via a hidden sret argument."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"The total logic for this is implemented by get_specsig_function and deserves_sret."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html","category":"page","text":"Additionally, if the return type is a union, it may be returned as a pair of values (a pointer and a tag). If the union values can be stack-allocated, then sufficient space to store them will also be passed as a hidden first argument. If the struct to return needs gc roots, space for those will be passed as a hidden second argument. It is up to the callee whether the returned pointer will point to this space, a boxed object, or even other constant memory."},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html#Code-changes","category":"section","text":""},{"title":"Contributing to core functionality or base libraries","page":"Code changes","location":"devdocs/contributing/code-changes.html#Contributing-to-core-functionality-or-base-libraries","category":"section","text":""},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"By contributing code to Julia, you are agreeing to release it under the MIT License."},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"The Julia community uses GitHub issues to track and discuss problems, feature requests, and pull requests (PR)."},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"Issues and pull requests should have self explanatory titles such that they can be understood from the list of PRs and Issues. i.e. Add {feature} and Fix {bug} are good, Fix #12345. Corrects the bug. is bad."},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"You can make pull requests for incomplete features to get code review. The convention is to open these as draft PRs and prefix the pull request title with \"WIP:\" for Work In Progress, or \"RFC:\" for Request for Comments when work is completed and ready for merging. This will prevent accidental merging of work that is in progress."},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"Note: These instructions are for adding to or improving functionality in the base library. Before getting started, it can be helpful to discuss the proposed changes or additions on the Julia Discourse forum or in a GitHub issue—it's possible your proposed change belongs in a package rather than the core language. Also, keep in mind that changing stuff in the base can potentially break a lot of things. Finally, because of the time required to build Julia, note that it's usually faster to develop your code in stand-alone files, get it working, and then migrate it into the base libraries."},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"Add new code to Julia's base libraries as follows (this is the \"basic\" approach; see a more efficient approach in the next section):"},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"Edit the appropriate file in the base/ directory, or add new files if necessary. Create tests for your functionality and add them to files in the test/ directory. If you're editing C or Scheme code, most likely it lives in src/ or one of its subdirectories, although some aspects of Julia's REPL initialization live in cli/.\nAdd any new files to sysimg.jl in order to build them into the Julia system image.\nAdd any necessary export symbols in exports.jl.\nInclude your tests in test/Makefile and test/choosetests.jl."},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"Build as usual, and do make clean testall to test your contribution. If your contribution includes changes to Makefiles or external dependencies, make sure you can build Julia from a clean tree using git clean -fdx or equivalent (be careful – this command will delete any files lying around that aren't checked into git)."},{"title":"Running specific tests","page":"Code changes","location":"devdocs/contributing/code-changes.html#Running-specific-tests","category":"section","text":""},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"There are make targets for running specific tests:"},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"make test-bitarray"},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"You can also use the runtests.jl script, e.g. to run test/bitarray.jl and test/math.jl:"},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"./usr/bin/julia test/runtests.jl bitarray math"},{"title":"Modifying base more efficiently with Revise.jl","page":"Code changes","location":"devdocs/contributing/code-changes.html#Modifying-base-more-efficiently-with-Revise.jl","category":"section","text":""},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"Revise is a package that tracks changes in source files and automatically updates function definitions in your running Julia session. Using it, you can make extensive changes to Base without needing to rebuild in order to test your changes."},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"Here is the standard procedure:"},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"If you are planning changes to any types or macros, make those changes and build julia using make. (This is necessary because Revise cannot handle changes to type definitions or macros.) Unless it's required to get Julia to build, you do not have to add any functionality based on the new types, just the type definitions themselves.\nStart a Julia REPL session. Then issue the following commands:"},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"using Revise    # if you aren't launching it in your `.julia/config/startup.jl`\nRevise.track(Base)"},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"Edit files in base/, save your edits, and test the functionality."},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"If you need to restart your Julia session, just start at step 2 above. Revise.track(Base) will note any changes from when Julia was last built and incorporate them automatically. You only need to rebuild Julia if you made code-changes that Revise cannot handle."},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"For convenience, there are also test-revise-* targets for every test-* target that use Revise to load any modifications to Base into the current system image before running the corresponding test. This can be useful as a shortcut on the command line (since tests aren't always designed to be run outside the runtest harness)."},{"title":"Contributing to the standard library","page":"Code changes","location":"devdocs/contributing/code-changes.html#Contributing-to-the-standard-library","category":"section","text":""},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"The standard library (stdlib) packages are baked into the Julia system image. When running the ordinary test workflow on the stdlib packages, the system image version overrides the version you are developing. To test stdlib packages, you can do the following steps:"},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"Edit the UUID field of the Project.toml in the stdlib package\nChange the current directory to the directory of the stdlib you are developing\nStart julia with julia --project=.\nYou can now test the package by running pkg> test in Pkg mode."},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"Because you changed the UUID, the package manager treats the stdlib package as different from the one in the system image, and the system image version will not override the package."},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"Be sure to change the UUID value back before making the pull request."},{"title":"News-worthy changes","page":"Code changes","location":"devdocs/contributing/code-changes.html#News-worthy-changes","category":"section","text":""},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"For new functionality and other substantial changes, add a brief summary to NEWS.md. The news item should cross reference the pull request (PR) parenthetically, in the form ([#pr]). To add the PR reference number, first create the PR, then push an additional commit updating NEWS.md with the PR reference number. We periodically run ./julia doc/NEWS-update.jl from the julia directory to update the cross-reference links, but this should not be done in a typical PR in order to avoid conflicting commits."},{"title":"Annotations for new features, deprecations and behavior changes","page":"Code changes","location":"devdocs/contributing/code-changes.html#Annotations-for-new-features,-deprecations-and-behavior-changes","category":"section","text":""},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"API additions and deprecations, and minor behavior changes are allowed in minor version releases. For documented features that are part of the public API, a compatibility note should be added into the manual or the docstring. It should state the Julia minor version that changed the behavior and have a brief message describing the change."},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"At the moment, this should always be done with the following compat admonition (so that it would be possible to programmatically find the annotations in the future):"},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html","category":"page","text":"!!! compat \"Julia 1.X\"       This method was added in Julia 1.X."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Mathematical-Operations-and-Elementary-Functions","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Julia provides a complete collection of basic arithmetic and bitwise operators across all of its numeric primitive types, as well as providing portable, efficient implementations of a comprehensive collection of standard mathematical functions."},{"title":"Arithmetic Operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Arithmetic-Operators","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The following arithmetic operators are supported on all primitive numeric types:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Expression Name Description\n+x unary plus the identity operation\n-x unary minus maps values to their additive inverses\nx + y binary plus performs addition\nx - y binary minus performs subtraction\nx * y times performs multiplication\nx / y divide performs division\nx ÷ y integer divide x / y, truncated to an integer\nx \\ y inverse divide equivalent to y / x\nx ^ y power raises x to the yth power\nx % y remainder equivalent to rem(x, y)"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"A numeric literal placed directly before an identifier or parentheses, e.g. 2x or 2(x + y), is treated as a multiplication, except with higher precedence than other binary operations. See Numeric Literal Coefficients for details."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Julia's promotion system makes arithmetic operations on mixtures of argument types \"just work\" naturally and automatically. See Conversion and Promotion for details of the promotion system."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The ÷ sign can be conveniently typed by writing \\div<tab> to the REPL or Julia IDE. See the manual section on Unicode input for more information."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Here are some simple examples using arithmetic operators:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> 1 + 2 + 3\n6\n\njulia> 1 - 2\n-1\n\njulia> 3*2/12\n0.5"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"(By convention, we tend to space operators more tightly if they get applied before other nearby operators. For instance, we would generally write -x + 2 to reflect that first x gets negated, and then 2 is added to that result.)"},{"title":"Boolean Operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Boolean-Operators","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The following Boolean operators are supported on Bool types:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Expression Name\n!x negation\nx && y short-circuiting and\nx || y short-circuiting or"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Negation changes true to false and vice versa. The short-circuiting operations are explained on the linked page."},{"title":"Arithmetic operations with Bool values","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Arithmetic-operations-with-Bool-values","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Note that Bool is an integer type, such that false is numerically equal to 0 and true is numerically equal to 1. All the usual promotion rules and numeric operators are also defined on it, with a special behavior of arithmetic (non-Boolean) operations when all the arguments are Bool: in those cases, the arguments are promoted to Int instead of keeping their type. Compare e.g. the following equivalent operations with Bool and with a different numeric type (UInt8):"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> true - true\n0\n\njulia> 0x01 - 0x01\n0x00"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Also, when used in multiplication, false acts as a strong zero:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> NaN * false\n0.0\n\njulia> false * Inf\n0.0"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"This is useful for preventing the propagation of NaN values in quantities that are known to be zero. See Knuth (1992) for motivation."},{"title":"Bitwise Operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Bitwise-Operators","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The following bitwise operators are supported on all primitive integer types:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Expression Name\n~x bitwise not\nx & y bitwise and\nx | y bitwise or\nx ⊻ y bitwise xor (exclusive or)\nx ⊼ y bitwise nand (not and)\nx ⊽ y bitwise nor (not or)\nx >>> y logical shift right\nx >> y arithmetic shift right\nx << y logical/arithmetic shift left"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Here are some examples with bitwise operators:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> ~123\n-124\n\njulia> 123 & 234\n106\n\njulia> 123 | 234\n251\n\njulia> 123 ⊻ 234\n145\n\njulia> xor(123, 234)\n145\n\njulia> nand(123, 123)\n-124\n\njulia> 123 ⊼ 123\n-124\n\njulia> nor(123, 124)\n-128\n\njulia> 123 ⊽ 124\n-128\n\njulia> ~UInt32(123)\n0xffffff84\n\njulia> ~UInt8(123)\n0x84"},{"title":"Updating operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Updating-operators","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Every binary arithmetic and bitwise operator also has an updating version that assigns the result of the operation back into its left operand. The updating version of the binary operator is formed by placing a = immediately after the operator. For example, writing x += 3 is equivalent to writing x = x + 3:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> x = 1\n1\n\njulia> x += 3\n4\n\njulia> x\n4"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The updating versions of all the binary arithmetic and bitwise operators are:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"+=  -=  *=  /=  \\=  ÷=  %=  ^=  &=  |=  ⊻=  >>>=  >>=  <<="},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"note: Note\nAn updating operator rebinds the variable on the left-hand side. As a result, the type of the variable may change.julia> x = 0x01; typeof(x)\nUInt8\n\njulia> x *= 2 # Same as x = x * 2\n2\n\njulia> typeof(x)\nInt64"},{"title":"Vectorized \"dot\" operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#man-dot-operators","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"For every binary operation like ^, there is a corresponding \"dot\" operation .^ that is automatically defined to perform ^ element-by-element on arrays. For example, [1, 2, 3] ^ 3 is not defined, since there is no standard mathematical meaning to \"cubing\" a (non-square) array, but [1, 2, 3] .^ 3 is defined as computing the elementwise (or \"vectorized\") result [1^3, 2^3, 3^3]. Similarly for unary operators like ! or √, there is a corresponding .√ that applies the operator elementwise."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> [1, 2, 3] .^ 3\n3-element Vector{Int64}:\n  1\n  8\n 27"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"More specifically, a .^ b is parsed as the \"dot\" call (^).(a,b), which performs a broadcast operation: it can combine arrays and scalars, arrays of the same size (performing the operation elementwise), and even arrays of different shapes (e.g. combining row and column vectors to produce a matrix). Moreover, like all vectorized \"dot calls,\" these \"dot operators\" are fusing. For example, if you compute 2 .* A.^2 .+ sin.(A) (or equivalently @. 2A^2 + sin(A), using the @. macro) for an array A, it performs a single loop over A, computing 2a^2 + sin(a) for each element a of A. In particular, nested dot calls like f.(g.(x)) are fused, and \"adjacent\" binary operators like x .+ 3 .* x.^2 are equivalent to nested dot calls (+).(x, (*).(3, (^).(x, 2)))."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Furthermore, \"dotted\" updating operators like a .+= b (or @. a += b) are parsed as a .= a .+ b, where .= is a fused in-place assignment operation (see the dot syntax documentation)."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Note the dot syntax is also applicable to user-defined operators. For example, if you define ⊗(A, B) = kron(A, B) to give a convenient infix syntax A ⊗ B for Kronecker products (kron), then [A, B] .⊗ [C, D] will compute [A⊗C, B⊗D] with no additional coding."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Combining dot operators with numeric literals can be ambiguous. For example, it is not clear whether 1.+x means 1. + x or 1 .+ x. Therefore this syntax is disallowed, and spaces must be used around the operator in such cases."},{"title":"Numeric Comparisons","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Numeric-Comparisons","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Standard comparison operations are defined for all the primitive numeric types:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Operator Name\n== equality\n!=, ≠ inequality\n< less than\n<=, ≤ less than or equal to\n> greater than\n>=, ≥ greater than or equal to"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Here are some simple examples:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> 1 == 1\ntrue\n\njulia> 1 == 2\nfalse\n\njulia> 1 != 2\ntrue\n\njulia> 1 == 1.0\ntrue\n\njulia> 1 < 2\ntrue\n\njulia> 1.0 > 3\nfalse\n\njulia> 1 >= 1.0\ntrue\n\njulia> -1 <= 1\ntrue\n\njulia> -1 <= -1\ntrue\n\njulia> -1 <= -2\nfalse\n\njulia> 3 < -0.5\nfalse"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Integers are compared in the standard manner – by comparison of bits. Floating-point numbers are compared according to the IEEE 754 standard:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Finite numbers are ordered in the usual manner.\nPositive zero is equal but not greater than negative zero.\nInf is equal to itself and greater than everything else except NaN.\n-Inf is equal to itself and less than everything else except NaN.\nNaN is not equal to, not less than, and not greater than anything, including itself."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The last point is potentially surprising and thus worth noting:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> NaN == NaN\nfalse\n\njulia> NaN != NaN\ntrue\n\njulia> NaN < NaN\nfalse\n\njulia> NaN > NaN\nfalse"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"and can cause headaches when working with arrays:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> [1 NaN] == [1 NaN]\nfalse"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Julia provides additional functions to test numbers for special values, which can be useful in situations like hash key comparisons:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Function Tests if\nisequal(x, y) x and y are identical\nisfinite(x) x is a finite number\nisinf(x) x is infinite\nisnan(x) x is not a number"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"isequal considers NaNs equal to each other:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> isequal(NaN, NaN)\ntrue\n\njulia> isequal([1 NaN], [1 NaN])\ntrue\n\njulia> isequal(NaN, NaN32)\ntrue"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"isequal can also be used to distinguish signed zeros:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> -0.0 == 0.0\ntrue\n\njulia> isequal(-0.0, 0.0)\nfalse"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Mixed-type comparisons between signed integers, unsigned integers, and floats can be tricky. A great deal of care has been taken to ensure that Julia does them correctly."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"For other types, isequal defaults to calling ==, so if you want to define equality for your own types then you only need to add a == method. If you define your own equality function, you should probably define a corresponding hash method to ensure that isequal(x,y) implies hash(x) == hash(y)."},{"title":"Chaining comparisons","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Chaining-comparisons","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Unlike most languages, with the notable exception of Python, comparisons can be arbitrarily chained:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> 1 < 2 <= 2 < 3 == 3 > 2 >= 1 == 1 < 3 != 5\ntrue"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Chaining comparisons is often quite convenient in numerical code. Chained comparisons use the && operator for scalar comparisons, and the & operator for elementwise comparisons, which allows them to work on arrays. For example, 0 .< A .< 1 gives a boolean array whose entries are true where the corresponding elements of A are between 0 and 1."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Note the evaluation behavior of chained comparisons:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> v(x) = (println(x); x)\nv (generic function with 1 method)\n\njulia> v(1) < v(2) <= v(3)\n2\n1\n3\ntrue\n\njulia> v(1) > v(2) <= v(3)\n2\n1\nfalse"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The middle expression is only evaluated once, rather than twice as it would be if the expression were written as v(1) < v(2) && v(2) <= v(3). However, the order of evaluations in a chained comparison is undefined. It is strongly recommended not to use expressions with side effects (such as printing) in chained comparisons. If side effects are required, the short-circuit && operator should be used explicitly (see Short-Circuit Evaluation)."},{"title":"Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Elementary-Functions","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Julia provides a comprehensive collection of mathematical functions and operators. These mathematical operations are defined over as broad a class of numerical values as permit sensible definitions, including integers, floating-point numbers, rationals, and complex numbers, wherever such definitions make sense."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Moreover, these functions (like any Julia function) can be applied in \"vectorized\" fashion to arrays and other collections with the dot syntax f.(A), e.g. sin.(A) will compute the sine of each element of an array A."},{"title":"Operator Precedence and Associativity","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Operator-Precedence-and-Associativity","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Julia applies the following order and associativity of operations, from highest precedence to lowest:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Category Operators Associativity\nSyntax . followed by :: Left\nExponentiation ^ Right\nUnary + - ! ~ ¬ √ ∛ ∜ ⋆ ± ∓ <: >: Right[1]\nBitshifts << >> >>> Left\nFractions // Left\nMultiplication * / % & \\ ÷ Left[2]\nAddition + - | ⊻ Left[2]\nSyntax : .. Left\nSyntax |> Left\nSyntax <| Right\nComparisons > < >= <= == === != !== <: Non-associative\nControl flow && followed by || followed by ? Right\nPair => Right\nAssignments = += -= *= /= //= \\= ^= ÷= %= |= &= ⊻= <<= >>= >>>= Right"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"[1]: The unary operators + and - require explicit parentheses around their argument to disambiguate them from the operator ++, etc. Other compositions of unary operators are parsed with right-associativity, e. g., √√-a as √(√(-a))."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"[2]: The operators +, ++ and * are non-associative. a + b + c is parsed as +(a, b, c) not +(+(a, b), c). However, the fallback methods for +(a, b, c, d...) and *(a, b, c, d...) both default to left-associative evaluation."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"For a complete list of every Julia operator's precedence, see the top of this file: src/julia-parser.scm. Note that some of the operators there are not defined in the Base module but may be given definitions by standard libraries, packages or user code."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"You can also find the numerical precedence for any given operator via the built-in function Base.operator_precedence, where higher numbers take precedence:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> Base.operator_precedence(:+), Base.operator_precedence(:*), Base.operator_precedence(:.)\n(11, 12, 17)\n\njulia> Base.operator_precedence(:sin), Base.operator_precedence(:+=), Base.operator_precedence(:(=))  # (Note the necessary parens on `:(=)`)\n(0, 1, 1)"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"A symbol representing the operator associativity can also be found by calling the built-in function Base.operator_associativity:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> Base.operator_associativity(:-), Base.operator_associativity(:+), Base.operator_associativity(:^)\n(:left, :none, :right)\n\njulia> Base.operator_associativity(:⊗), Base.operator_associativity(:sin), Base.operator_associativity(:→)\n(:left, :none, :right)"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Note that symbols such as :sin return precedence 0. This value represents invalid operators and not operators of lowest precedence. Similarly, such operators are assigned associativity :none."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Numeric literal coefficients, e.g. 2x, are treated as multiplications with higher precedence than any other binary operation, with the exception of ^ where they have higher precedence only as the exponent."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> x = 3; 2x^2\n18\n\njulia> x = 3; 2^2x\n64"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Juxtaposition parses like a unary operator, which has the same natural asymmetry around exponents: -x^y and 2x^y parse as -(x^y) and 2(x^y) whereas x^-y and x^2y parse as x^(-y) and x^(2y)."},{"title":"Numerical Conversions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Numerical-Conversions","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Julia supports three forms of numerical conversion, which differ in their handling of inexact conversions."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The notation T(x) or convert(T, x) converts x to a value of type T.\nIf T is a floating-point type, the result is the nearest representable value, which could be positive or negative infinity.\nIf T is an integer type, an InexactError is raised if x is not representable by T.\nx % T converts an integer x to a value of integer type T congruent to x modulo 2^n, where n is the number of bits in T. In other words, the binary representation is truncated to fit.\nThe Rounding functions take a type T as an optional argument. For example, round(Int,x) is a shorthand for Int(round(x))."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The following examples show the different forms."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> Int8(127)\n127\n\njulia> Int8(128)\nERROR: InexactError: trunc(Int8, 128)\nStacktrace:\n[...]\n\njulia> Int8(127.0)\n127\n\njulia> Int8(3.14)\nERROR: InexactError: Int8(3.14)\nStacktrace:\n[...]\n\njulia> Int8(128.0)\nERROR: InexactError: Int8(128.0)\nStacktrace:\n[...]\n\njulia> 127 % Int8\n127\n\njulia> 128 % Int8\n-128\n\njulia> round(Int8,127.4)\n127\n\njulia> round(Int8,127.6)\nERROR: InexactError: Int8(128.0)\nStacktrace:\n[...]"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"See Conversion and Promotion for how to define your own conversions and promotions."},{"title":"Rounding functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Rounding-functions","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Function Description Return type\nround(x) round x to the nearest integer typeof(x)\nround(T, x) round x to the nearest integer T\nfloor(x) round x towards -Inf typeof(x)\nfloor(T, x) round x towards -Inf T\nceil(x) round x towards +Inf typeof(x)\nceil(T, x) round x towards +Inf T\ntrunc(x) round x towards zero typeof(x)\ntrunc(T, x) round x towards zero T"},{"title":"Division functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Division-functions","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Function Description\ndiv(x, y), x÷y truncated division; quotient rounded towards zero\nfld(x, y) floored division; quotient rounded towards -Inf\ncld(x, y) ceiling division; quotient rounded towards +Inf\nrem(x, y), x%y remainder; satisfies x == div(x, y)*y + rem(x, y); sign matches x\nmod(x, y) modulus; satisfies x == fld(x, y)*y + mod(x, y); sign matches y\nmod1(x, y) mod with offset 1; returns r∈(0, y] for y>0 or r∈[y, 0) for y<0, where mod(r, y) == mod(x, y)\nmod2pi(x) modulus with respect to 2pi;  0 <= mod2pi(x) < 2pi\ndivrem(x, y) returns (div(x, y),rem(x, y))\nfldmod(x, y) returns (fld(x, y), mod(x, y))\ngcd(x, y...) greatest positive common divisor of x, y,...\nlcm(x, y...) least positive common multiple of x, y,..."},{"title":"Sign and absolute value functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Sign-and-absolute-value-functions","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Function Description\nabs(x) a positive value with the magnitude of x\nabs2(x) the squared magnitude of x\nsign(x) indicates the sign of x, returning -1, 0, or +1\nsignbit(x) indicates whether the sign bit is on (true) or off (false)\ncopysign(x, y) a value with the magnitude of x and the sign of y\nflipsign(x, y) a value with the magnitude of x and the sign of x*y"},{"title":"Powers, logs and roots","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Powers,-logs-and-roots","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Function Description\nsqrt(x), √x square root of x\ncbrt(x), ∛x cube root of x\nfourthroot(x), ∜x fourth root of x\nhypot(x, y) hypotenuse of right-angled triangle with other sides of length x and y\nexp(x) natural exponential function at x\nexpm1(x) accurate exp(x) - 1 for x near zero\nldexp(x, n) x * 2^n computed efficiently for integer values of n\nlog(x) natural logarithm of x\nlog(b, x) base b logarithm of x\nlog2(x) base 2 logarithm of x\nlog10(x) base 10 logarithm of x\nlog1p(x) accurate log(1 + x) for x near zero\nexponent(x) binary exponent of x\nsignificand(x) binary significand (a.k.a. mantissa) of a floating-point number x"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"For an overview of why functions like hypot, expm1, and log1p are necessary and useful, see John D. Cook's excellent pair of blog posts on the subject: expm1, log1p, erfc, and hypot."},{"title":"Trigonometric and hyperbolic functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Trigonometric-and-hyperbolic-functions","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"All the standard trigonometric and hyperbolic functions are also defined:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"sin    cos    tan    cot    sec    csc\nsinh   cosh   tanh   coth   sech   csch\nasin   acos   atan   acot   asec   acsc\nasinh  acosh  atanh  acoth  asech  acsch\nsinc   cosc"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"These are all single-argument functions, with atan also accepting two arguments corresponding to a traditional atan2 function."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Additionally, sinpi(x) and cospi(x) are provided for more accurate computations of sin(pi * x) and cos(pi * x) respectively."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"In order to compute trigonometric functions with degrees instead of radians, suffix the function with d. For example, sind(x) computes the sine of x where x is specified in degrees. The complete list of trigonometric functions with degree variants is:"},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"sind   cosd   tand   cotd   secd   cscd\nasind  acosd  atand  acotd  asecd  acscd"},{"title":"Special functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Special-functions","category":"section","text":""},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Many other special mathematical functions are provided by the package SpecialFunctions.jl."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html#markdown_stdlib","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"This section describes Julia's markdown syntax, which is enabled by the Markdown standard library. The following Markdown elements are supported:"},{"title":"Inline elements","page":"Markdown","location":"stdlib/Markdown.html#Inline-elements","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Here \"inline\" refers to elements that can be found within blocks of text, i.e. paragraphs. These include the following elements."},{"title":"Bold","page":"Markdown","location":"stdlib/Markdown.html#Bold","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Surround words with two asterisks, **, to display the enclosed text in boldface."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A paragraph containing a **bold** word."},{"title":"Italics","page":"Markdown","location":"stdlib/Markdown.html#Italics","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Surround words with one asterisk, *, to display the enclosed text in italics."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A paragraph containing an *italicized* word."},{"title":"Literals","page":"Markdown","location":"stdlib/Markdown.html#Literals","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Surround text that should be displayed exactly as written with single backticks, ` ."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A paragraph containing a `literal` word."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Literals should be used when writing text that refers to names of variables, functions, or other parts of a Julia program."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"tip: Tip\nTo include a backtick character within literal text use three backticks rather than one to enclose the text.A paragraph containing ``` `backtick` characters ```.By extension any odd number of backticks may be used to enclose a lesser number of backticks."},{"title":"LaTeX","page":"Markdown","location":"stdlib/Markdown.html#\\LaTeX","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Surround text that should be displayed as mathematics using LaTeX syntax with double backticks, `` ."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A paragraph containing some ``\\LaTeX`` markup."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"tip: Tip\nAs with literals in the previous section, if literal backticks need to be written within double backticks use an even number greater than two. Note that if a single literal backtick needs to be included within LaTeX markup then two enclosing backticks is sufficient."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"note: Note\nThe \\ character should be escaped appropriately if the text is embedded in a Julia source code, for example, \"``\\\\LaTeX`` syntax in a docstring.\", since it is interpreted as a string literal. Alternatively, in order to avoid escaping, it is possible to use the raw string macro together with the @doc macro:@doc raw\"``\\LaTeX`` syntax in a docstring.\" functionname"},{"title":"Links","page":"Markdown","location":"stdlib/Markdown.html#Links","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Links to either external or internal targets can be written using the following syntax, where the text enclosed in square brackets, [ ], is the name of the link and the text enclosed in parentheses, ( ), is the URL."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A paragraph containing a link to [Julia](https://www.julialang.org)."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"It's also possible to add cross-references to other documented functions/methods/variables within the Julia documentation itself. For example:"},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"\"\"\"\n    tryparse(type, str; base)\n\nLike [`parse`](@ref), but returns either a value of the requested type,\nor [`nothing`](@ref) if the string does not contain a valid number.\n\"\"\""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"This will create a link in the generated docs to the parse documentation (which has more information about what this function actually does), and to the nothing documentation. It's good to include cross references to mutating/non-mutating versions of a function, or to highlight a difference between two similar-seeming functions."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"note: Note\nThe above cross referencing is not a Markdown feature, and relies on Documenter.jl, which is used to build base Julia's documentation."},{"title":"Footnote references","page":"Markdown","location":"stdlib/Markdown.html#Footnote-references","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Named and numbered footnote references can be written using the following syntax. A footnote name must be a single alphanumeric word containing no punctuation."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A paragraph containing a numbered footnote [^1] and a named one [^named]."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"note: Note\nThe text associated with a footnote can be written anywhere within the same page as the footnote reference. The syntax used to define the footnote text is discussed in the Footnotes section below."},{"title":"Toplevel elements","page":"Markdown","location":"stdlib/Markdown.html#Toplevel-elements","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"The following elements can be written either at the \"toplevel\" of a document or within another \"toplevel\" element."},{"title":"Paragraphs","page":"Markdown","location":"stdlib/Markdown.html#Paragraphs","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A paragraph is a block of plain text, possibly containing any number of inline elements defined in the Inline elements section above, with one or more blank lines above and below it."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"This is a paragraph.\n\nAnd this is *another* paragraph containing some emphasized text.\nA new line, but still part of the same paragraph."},{"title":"Headers","page":"Markdown","location":"stdlib/Markdown.html#Headers","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A document can be split up into different sections using headers. Headers use the following syntax:"},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"# Level One\n## Level Two\n### Level Three\n#### Level Four\n##### Level Five\n###### Level Six"},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A header line can contain any inline syntax in the same way as a paragraph can."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"tip: Tip\nTry to avoid using too many levels of header within a single document. A heavily nested document may be indicative of a need to restructure it or split it into several pages covering separate topics."},{"title":"Code blocks","page":"Markdown","location":"stdlib/Markdown.html#Code-blocks","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Source code can be displayed as a literal block using an indent of four spaces or one tab as shown in the following example."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"This is a paragraph.\n\n    function func(x)\n        # ...\n    end\n\nAnother paragraph."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Additionally, code blocks can be enclosed using triple backticks with an optional \"language\" to specify how a block of code should be highlighted."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A code block without a \"language\":\n\n```\nfunction func(x)\n    # ...\nend\n```\n\nand another one with the \"language\" specified as `julia`:\n\n```julia\nfunction func(x)\n    # ...\nend\n```"},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"note: Note\n\"Fenced\" code blocks, as shown in the last example, should be preferred over indented code blocks since there is no way to specify what language an indented code block is written in."},{"title":"Block quotes","page":"Markdown","location":"stdlib/Markdown.html#Block-quotes","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Text from external sources, such as quotations from books or websites, can be quoted using > characters prepended to each line of the quote as follows."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Here's a quote:\n\n> Julia is a high-level, high-performance dynamic programming language for\n> technical computing, with syntax that is familiar to users of other\n> technical computing environments."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Note that a single space must appear after the > character on each line. Quoted blocks may themselves contain other toplevel or inline elements."},{"title":"Images","page":"Markdown","location":"stdlib/Markdown.html#Images","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"The syntax for images is similar to the link syntax mentioned above. Prepending a ! character to a link will display an image from the specified URL rather than a link to it."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"![alternative text](link/to/image.png)"},{"title":"Lists","page":"Markdown","location":"stdlib/Markdown.html#Lists","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Unordered lists can be written by prepending each item in a list with either *, +, or -."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A list of items:\n\n  * item one\n  * item two\n  * item three"},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Note the two spaces before each * and the single space after each one."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Lists can contain other nested toplevel elements such as lists, code blocks, or quoteblocks. A blank line should be left between each list item when including any toplevel elements within a list."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Another list:\n\n  * item one\n\n  * item two\n\n    ```\n    f(x) = x\n    ```\n\n  * And a sublist:\n\n      + sub-item one\n      + sub-item two"},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"note: Note\nThe contents of each item in the list must line up with the first line of the item. In the above example the fenced code block must be indented by four spaces to align with the i in item two."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Ordered lists are written by replacing the \"bullet\" character, either *, +, or -, with a positive integer followed by either . or )."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Two ordered lists:\n\n 1. item one\n 2. item two\n 3. item three\n\n 5) item five\n 6) item six\n 7) item seven"},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"An ordered list may start from a number other than one, as in the second list of the above example, where it is numbered from five. As with unordered lists, ordered lists can contain nested toplevel elements."},{"title":"Display equations","page":"Markdown","location":"stdlib/Markdown.html#Display-equations","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Large LaTeX equations that do not fit inline within a paragraph may be written as display equations using a fenced code block with the \"language\" math as in the example below."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"```math\nf(a) = \\frac{1}{2\\pi}\\int_{0}^{2\\pi} (\\alpha+R\\cos(\\theta))d\\theta\n```"},{"title":"Footnotes","page":"Markdown","location":"stdlib/Markdown.html#Footnotes","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"This syntax is paired with the inline syntax for Footnote references. Make sure to read that section as well."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Footnote text is defined using the following syntax, which is similar to footnote reference syntax, aside from the : character that is appended to the footnote label."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"[^1]: Numbered footnote text.\n\n[^note]:\n\n    Named footnote text containing several toplevel elements\n    indented by 4 spaces or one tab.\n\n      * item one\n      * item two\n      * item three\n\n    ```julia\n    function func(x)\n        # ...\n    end\n    ```"},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"note: Note\nNo checks are done during parsing to make sure that all footnote references have matching footnotes."},{"title":"Horizontal rules","page":"Markdown","location":"stdlib/Markdown.html#Horizontal-rules","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"The equivalent of an <hr> HTML tag can be achieved using three hyphens (---). For example:"},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Text above the line.\n\n---\n\nAnd text below the line."},{"title":"Tables","page":"Markdown","location":"stdlib/Markdown.html#Tables","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Basic tables can be written using the syntax described below. Note that markdown tables have limited features and cannot contain nested toplevel elements unlike other elements discussed above – only inline elements are allowed. Tables must always contain a header row with column names. Cells cannot span multiple rows or columns of the table."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"| Column One | Column Two | Column Three |\n|:---------- | ---------- |:------------:|\n| Row `1`    | Column `2` |              |\n| *Row* 2    | **Row** 2  | Column ``3`` |"},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"note: Note\nAs illustrated in the above example each column of | characters must be aligned vertically.A : character on either end of a column's header separator (the row containing - characters) specifies whether the row is left-aligned, right-aligned, or (when : appears on both ends) center-aligned. Providing no : characters will default to right-aligning the column."},{"title":"Admonitions","page":"Markdown","location":"stdlib/Markdown.html#Admonitions","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Specially formatted blocks, known as admonitions, can be used to highlight particular remarks. They can be defined using the following !!! syntax:"},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"!!! note\n\n    This is the content of the note.\n    It is indented by 4 spaces. A tab would work as well.\n\n!!! warning \"Beware!\"\n\n    And this is another one.\n\n    This warning admonition has a custom title: `\"Beware!\"`."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"The first word after !!! declares the type of the admonition. There are standard admonition types that should produce special styling. Namely (in order of decreasing severity): danger, warning, info/note, and tip."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"You can also use your own admonition types, as long as the type name only contains lowercase Latin characters (a-z). For example, you could have a terminology block like this:"},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"!!! terminology \"julia vs Julia\"\n\n    Strictly speaking, \"Julia\" refers to the language,\n    and \"julia\" to the standard implementation."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"However, unless the code rendering the Markdown special-cases that particular admonition type, it will get the default styling."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A custom title for the box can be provided as a string (in double quotes) after the admonition type. If no title text is specified after the admonition type, then the type name will be used as the title (e.g. \"Note\" for the note admonition)."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Admonitions, like most other toplevel elements, can contain other toplevel elements (e.g. lists, images)."},{"title":"Markdown String Literals","page":"Markdown","location":"stdlib/Markdown.html#stdlib-markdown-literals","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"The md\"\" macro allows you to embed Markdown strings directly into your Julia code. This macro is designed to simplify the inclusion of Markdown-formatted text within your Julia source files."},{"title":"Usage","page":"Markdown","location":"stdlib/Markdown.html#Usage","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"result = md\"This is a **custom** Markdown string with [a link](http://example.com).\""},{"title":"Markdown Syntax Extensions","page":"Markdown","location":"stdlib/Markdown.html#Markdown-Syntax-Extensions","category":"section","text":""},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Julia's markdown supports interpolation in a very similar way to basic string literals, with the difference that it will store the object itself in the Markdown tree (as opposed to converting it to a string). When the Markdown content is rendered the usual show methods will be called, and these can be overridden as usual. This design allows the Markdown to be extended with arbitrarily complex features (such as references) without cluttering the basic syntax."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"In principle, the Markdown parser itself can also be arbitrarily extended by packages, or an entirely custom flavour of Markdown can be used, but this should generally be unnecessary."},{"title":"API reference","page":"Markdown","location":"stdlib/Markdown.html#stdlib-markdown-api","category":"section","text":""},{"title":"Markdown.MD","page":"Markdown","location":"stdlib/Markdown.html#Markdown.MD","category":"type","text":"MD\n\nMD represents a Markdown document. Note that the MD constructor should not generally be used directly, since it constructs the internal data structures. Instead, you can construct MD objects using the exported macros @md_str and @doc_str.\n\n\n\n\n\n"},{"title":"Markdown.@md_str","page":"Markdown","location":"stdlib/Markdown.html#Markdown.@md_str","category":"macro","text":"@md_str -> MD\n\nParse the given string as Markdown text and return a corresponding MD object.\n\nSee also Markdown.parse.\n\nExamples\n\njulia> s = md\"# Hello, world!\"\n  Hello, world!\n  ≡≡≡≡≡≡≡≡≡≡≡≡≡\n\njulia> typeof(s)\nMarkdown.MD\n\n\n\n\n\n\n"},{"title":"Markdown.@doc_str","page":"Markdown","location":"stdlib/Markdown.html#Markdown.@doc_str","category":"macro","text":"@doc_str -> MD\n\nParse the given string as Markdown text, add line and module information and return a corresponding MD object.\n\n@doc_str can be used in conjunction with the Base.Docs module. Please also refer to the manual section on documentation for more information.\n\nExamples\n\njulia> s = doc\"f(x) = 2*x\"\n  f(x) = 2*x\n\njulia> typeof(s)\nMarkdown.MD\n\n\n\n\n\n\n"},{"title":"Markdown.parse","page":"Markdown","location":"stdlib/Markdown.html#Markdown.parse","category":"function","text":"parse(stream::IO)::MD\n\nParse the content of stream as Julia-flavored Markdown text and return the corresponding MD object.\n\n\n\n\n\nMarkdown.parse(markdown::AbstractString)::MD\n\nParse markdown as Julia-flavored Markdown text and return the corresponding MD object.\n\nSee also @md_str.\n\n\n\n\n\n"},{"title":"Markdown.html","page":"Markdown","location":"stdlib/Markdown.html#Markdown.html","category":"function","text":"html([io::IO], md)\n\nOutput the contents of the Markdown object md in HTML format, either writing to an (optional) io stream or returning a string.\n\nOne can alternatively use show(io, \"text/html\", md) or repr(\"text/html\", md), which differ in that they wrap the output in a <div class=\"markdown\"> ... </div> element.\n\nExamples\n\njulia> html(md\"hello _world_\")\n\"<p>hello <em>world</em></p>\\n\"\n\n\n\n\n\n"},{"title":"Markdown.latex","page":"Markdown","location":"stdlib/Markdown.html#Markdown.latex","category":"function","text":"latex([io::IO], md)\n\nOutput the contents of the Markdown object md in LaTeX format, either writing to an (optional) io stream or returning a string.\n\nOne can alternatively use show(io, \"text/latex\", md) or repr(\"text/latex\", md).\n\nExamples\n\njulia> latex(md\"hello _world_\")\n\"hello \\\\emph{world}\\n\\n\"\n\n\n\n\n\n"},{"title":"TOML","page":"TOML","location":"stdlib/TOML.html#TOML","category":"section","text":""},{"title":"TOML","page":"TOML","location":"stdlib/TOML.html","category":"page","text":"TOML.jl is a Julia standard library for parsing and writing TOML v1.0 files."},{"title":"Parsing TOML data","page":"TOML","location":"stdlib/TOML.html#Parsing-TOML-data","category":"section","text":""},{"title":"TOML","page":"TOML","location":"stdlib/TOML.html","category":"page","text":"julia> using TOML\n\njulia> data = \"\"\"\n           [database]\n           server = \"192.168.1.1\"\n           ports = [ 8001, 8001, 8002 ]\n       \"\"\";\n\njulia> TOML.parse(data)\nDict{String, Any} with 1 entry:\n  \"database\" => Dict{String, Any}(\"server\"=>\"192.168.1.1\", \"ports\"=>[8001, 8001…"},{"title":"TOML","page":"TOML","location":"stdlib/TOML.html","category":"page","text":"To parse a file, use TOML.parsefile. If the file has a syntax error, an exception is thrown:"},{"title":"TOML","page":"TOML","location":"stdlib/TOML.html","category":"page","text":"julia> using TOML\n\njulia> TOML.parse(\"\"\"\n           value = 0.0.0\n       \"\"\")\nERROR: TOML Parser error:\nnone:1:16 error: failed to parse value\n      value = 0.0.0\n                 ^\n[...]"},{"title":"TOML","page":"TOML","location":"stdlib/TOML.html","category":"page","text":"There are other versions of the parse functions (TOML.tryparse and TOML.tryparsefile) that instead of throwing exceptions on parser error returns a TOML.ParserError with information:"},{"title":"TOML","page":"TOML","location":"stdlib/TOML.html","category":"page","text":"julia> using TOML\n\njulia> err = TOML.tryparse(\"\"\"\n           value = 0.0.0\n       \"\"\");\n\njulia> err.type\nErrGenericValueError::ErrorType = 14\n\njulia> err.line\n1\n\njulia> err.column\n16"},{"title":"Exporting data to TOML file","page":"TOML","location":"stdlib/TOML.html#Exporting-data-to-TOML-file","category":"section","text":""},{"title":"TOML","page":"TOML","location":"stdlib/TOML.html","category":"page","text":"The TOML.print function is used to print (or serialize) data into TOML format."},{"title":"TOML","page":"TOML","location":"stdlib/TOML.html","category":"page","text":"julia> using TOML\n\njulia> data = Dict(\n          \"names\" => [\"Julia\", \"Julio\"],\n          \"age\" => [10, 20],\n       );\n\njulia> TOML.print(data)\nnames = [\"Julia\", \"Julio\"]\nage = [10, 20]\n\njulia> fname = tempname();\n\njulia> open(fname, \"w\") do io\n           TOML.print(io, data)\n       end\n\njulia> TOML.parsefile(fname)\nDict{String, Any} with 2 entries:\n  \"names\" => [\"Julia\", \"Julio\"]\n  \"age\"   => [10, 20]"},{"title":"TOML","page":"TOML","location":"stdlib/TOML.html","category":"page","text":"Keys can be sorted according to some value"},{"title":"TOML","page":"TOML","location":"stdlib/TOML.html","category":"page","text":"julia> using TOML\n\njulia> TOML.print(Dict(\n       \"abc\"  => 1,\n       \"ab\"   => 2,\n       \"abcd\" => 3,\n       ); sorted=true, by=length)\nab = 2\nabc = 1\nabcd = 3"},{"title":"TOML","page":"TOML","location":"stdlib/TOML.html","category":"page","text":"For custom structs, pass a function that converts the struct to a supported type"},{"title":"TOML","page":"TOML","location":"stdlib/TOML.html","category":"page","text":"julia> using TOML\n\njulia> struct MyStruct\n           a::Int\n           b::String\n       end\n\njulia> TOML.print(Dict(\"foo\" => MyStruct(5, \"bar\"))) do x\n           x isa MyStruct && return [x.a, x.b]\n           error(\"unhandled type $(typeof(x))\")\n       end\nfoo = [5, \"bar\"]"},{"title":"References","page":"TOML","location":"stdlib/TOML.html#References","category":"section","text":""},{"title":"TOML.parse","page":"TOML","location":"stdlib/TOML.html#TOML.parse","category":"function","text":"parse(x::Union{AbstractString, IO})\nparse(p::Parser, x::Union{AbstractString, IO})\n\nParse the string  or stream x, and return the resulting table (dictionary). Throw a ParserError upon failure.\n\nSee also TOML.tryparse.\n\n\n\n\n\n"},{"title":"TOML.parsefile","page":"TOML","location":"stdlib/TOML.html#TOML.parsefile","category":"function","text":"parsefile(f::AbstractString)\nparsefile(p::Parser, f::AbstractString)\n\nParse file f and return the resulting table (dictionary). Throw a ParserError upon failure.\n\nSee also TOML.tryparsefile.\n\n\n\n\n\n"},{"title":"TOML.tryparse","page":"TOML","location":"stdlib/TOML.html#TOML.tryparse","category":"function","text":"tryparse(x::Union{AbstractString, IO})\ntryparse(p::Parser, x::Union{AbstractString, IO})\n\nParse the string or stream x, and return the resulting table (dictionary). Return a ParserError upon failure.\n\nSee also TOML.parse.\n\n\n\n\n\n"},{"title":"TOML.tryparsefile","page":"TOML","location":"stdlib/TOML.html#TOML.tryparsefile","category":"function","text":"tryparsefile(f::AbstractString)\ntryparsefile(p::Parser, f::AbstractString)\n\nParse file f and return the resulting table (dictionary). Return a ParserError upon failure.\n\nSee also TOML.parsefile.\n\n\n\n\n\n"},{"title":"TOML.print","page":"TOML","location":"stdlib/TOML.html#TOML.print","category":"function","text":"print([to_toml::Function], io::IO [=stdout], data::AbstractDict; sorted=false, by=identity, inline_tables::IdSet{<:AbstractDict})\n\nWrite data as TOML syntax to the stream io. If the keyword argument sorted is set to true, sort tables according to the function given by the keyword argument by. If the keyword argument inline_tables is given, it should be a set of tables that should be printed \"inline\".\n\ncompat: Julia 1.11\nThe inline_tables keyword argument is supported by Julia 1.11 or later.\n\nThe following data types are supported: AbstractDict, AbstractVector, AbstractString, Integer, AbstractFloat, Bool, Dates.DateTime, Dates.Time, Dates.Date. Note that the integers and floats need to be convertible to Float64 and Int64 respectively. For other data types, pass the function to_toml that takes the data types and returns a value of a supported type.\n\n\n\n\n\n"},{"title":"TOML.Parser","page":"TOML","location":"stdlib/TOML.html#TOML.Parser","category":"type","text":"Parser()\n\nConstructor for a TOML Parser.  Note that in most cases one does not need to explicitly create a Parser but instead one directly uses TOML.parsefile or TOML.parse.  Using an explicit parser will however reuse some internal data structures which can be beneficial for performance if a larger number of small files are parsed.\n\n\n\n\n\n"},{"title":"TOML.ParserError","page":"TOML","location":"stdlib/TOML.html#TOML.ParserError","category":"type","text":"ParserError\n\nType that is returned from tryparse and tryparsefile when parsing fails. It contains (among others) the following fields:\n\npos, the position in the string when the error happened\ntable, the result that so far was successfully parsed\ntype, an error type, different for different types of errors\n\n\n\n\n\n"},{"title":"Mathematics","page":"Mathematics","location":"base/math.html#Mathematics","category":"section","text":""},{"title":"Mathematical Operators","page":"Mathematics","location":"base/math.html#math-ops","category":"section","text":""},{"title":"Base.:-","page":"Mathematics","location":"base/math.html#Base.:--Tuple{Any}","category":"method","text":"-(x)\n\nUnary minus operator.\n\nSee also: abs, flipsign.\n\nExamples\n\njulia> -1\n-1\n\njulia> -(2)\n-2\n\njulia> -[1 2; 3 4]\n2×2 Matrix{Int64}:\n -1  -2\n -3  -4\n\njulia> -(true)  # promotes to Int\n-1\n\njulia> -(0x003)\n0xfffd\n\n\n\n\n\n"},{"title":"Base.:+","page":"Mathematics","location":"base/math.html#Base.:+","category":"function","text":"+(x, y...)\n\nAddition operator.\n\nInfix x+y+z+... calls this function with all arguments, i.e. +(x, y, z, ...), which by default then calls (x+y) + z + ... starting from the left.\n\nNote that overflow is possible for most integer types, including the default Int, when adding large numbers.\n\nExamples\n\njulia> 1 + 20 + 4\n25\n\njulia> +(1, 20, 4)\n25\n\njulia> [1,2] + [3,4]\n2-element Vector{Int64}:\n 4\n 6\n\njulia> typemax(Int) + 1 < 0\ntrue\n\n\n\n\n\ndt::Date + t::Time -> DateTime\n\nThe addition of a Date with a Time produces a DateTime. The hour, minute, second, and millisecond parts of the Time are used along with the year, month, and day of the Date to create the new DateTime. Non-zero microseconds or nanoseconds in the Time type will result in an InexactError being thrown.\n\n\n\n\n\n"},{"title":"Base.:-","page":"Mathematics","location":"base/math.html#Base.:--Tuple{Any, Any}","category":"method","text":"-(x, y)\n\nSubtraction operator.\n\nExamples\n\njulia> 2 - 3\n-1\n\njulia> -(2, 4.5)\n-2.5\n\n\n\n\n\n"},{"title":"Base.:*","page":"Mathematics","location":"base/math.html#Base.:*-Tuple{Any, Vararg{Any}}","category":"method","text":"*(x, y...)\n\nMultiplication operator.\n\nInfix x*y*z*... calls this function with all arguments, i.e. *(x, y, z, ...), which by default then calls (x*y) * z * ... starting from the left.\n\nJuxtaposition such as 2pi also calls *(2, pi). Note that this operation has higher precedence than a literal *. Note also that juxtaposition \"0x...\" (integer zero times a variable whose name starts with x) is forbidden as it clashes with unsigned integer literals: 0x01 isa UInt8.\n\nNote that overflow is possible for most integer types, including the default Int, when multiplying large numbers.\n\nExamples\n\njulia> 2 * 7 * 8\n112\n\njulia> *(2, 7, 8)\n112\n\njulia> [2 0; 0 3] * [1, 10]  # matrix * vector\n2-element Vector{Int64}:\n  2\n 30\n\njulia> 1/2pi, 1/2*pi  # juxtaposition has higher precedence\n(0.15915494309189535, 1.5707963267948966)\n\njulia> x = [1, 2]; x'x  # adjoint vector * vector\n5\n\n\n\n\n\n"},{"title":"Base.:/","page":"Mathematics","location":"base/math.html#Base.:/","category":"function","text":"/(x, y)\n\nRight division operator: multiplication of x by the inverse of y on the right.\n\nGives floating-point results for integer arguments. See ÷ for integer division, or // for Rational results.\n\nExamples\n\njulia> 1/2\n0.5\n\njulia> 4/2\n2.0\n\njulia> 4.5/2\n2.25\n\n\n\n\n\nA / B\n\nMatrix right-division: A / B is equivalent to (B' \\ A')' where \\ is the left-division operator. For square matrices, the result X is such that A == X*B.\n\nSee also: rdiv!.\n\nExamples\n\njulia> A = Float64[1 4 5; 3 9 2]; B = Float64[1 4 2; 3 4 2; 8 7 1];\n\njulia> X = A / B\n2×3 Matrix{Float64}:\n -0.65   3.75  -1.2\n  3.25  -2.75   1.0\n\njulia> isapprox(A, X*B)\ntrue\n\njulia> isapprox(X, A*pinv(B))\ntrue\n\n\n\n\n\n"},{"title":"Base.:\\","page":"Mathematics","location":"base/math.html#Base.:\\-Tuple{Any, Any}","category":"method","text":"\\(x, y)\n\nLeft division operator: multiplication of y by the inverse of x on the left. Gives floating-point results for integer arguments.\n\nExamples\n\njulia> 3 \\ 6\n2.0\n\njulia> inv(3) * 6\n2.0\n\njulia> A = [4 3; 2 1]; x = [5, 6];\n\njulia> A \\ x\n2-element Vector{Float64}:\n  6.5\n -7.0\n\njulia> inv(A) * x\n2-element Vector{Float64}:\n  6.5\n -7.0\n\n\n\n\n\n"},{"title":"Base.:^","page":"Mathematics","location":"base/math.html#Base.:^-Tuple{Number, Number}","category":"method","text":"^(x, y)\n\nExponentiation operator.\n\nIf x and y are integers, the result may overflow. To enter numbers in scientific notation, use Float64 literals such as 1.2e3 rather than 1.2 * 10^3.\n\nIf y is an Int literal (e.g. 2 in x^2 or -3 in x^-3), the Julia code x^y is transformed by the compiler to Base.literal_pow(^, x, Val(y)), to enable compile-time specialization on the value of the exponent. (As a default fallback we have Base.literal_pow(^, x, Val(y)) = ^(x,y), where usually ^ == Base.^ unless ^ has been defined in the calling namespace.) If y is a negative integer literal, then Base.literal_pow transforms the operation to inv(x)^-y by default, where -y is positive.\n\nSee also exp2, <<.\n\nExamples\n\njulia> 3^5\n243\n\njulia> 3^-1  # uses Base.literal_pow\n0.3333333333333333\n\njulia> p = -1;\n\njulia> 3^p\nERROR: DomainError with -1:\nCannot raise an integer x to a negative power -1.\n[...]\n\njulia> 3.0^p\n0.3333333333333333\n\njulia> 10^19 > 0  # integer overflow\nfalse\n\njulia> big(10)^19 == 1e19\ntrue\n\n\n\n\n\n"},{"title":"Base.fma","page":"Mathematics","location":"base/math.html#Base.fma","category":"function","text":"fma(x, y, z)\n\nComputes x*y+z without rounding the intermediate result x*y. On some systems this is significantly more expensive than x*y+z. fma is used to improve accuracy in certain algorithms. See muladd.\n\n\n\n\n\n"},{"title":"Base.muladd","page":"Mathematics","location":"base/math.html#Base.muladd","category":"function","text":"muladd(x, y, z)\n\nCombined multiply-add: computes x*y+z, but allowing the add and multiply to be merged with each other or with surrounding operations for performance. For example, this may be implemented as an fma if the hardware supports it efficiently. The result can be different on different machines and can also be different on the same machine due to constant propagation or other optimizations. See fma.\n\nExamples\n\njulia> muladd(3, 2, 1)\n7\n\njulia> 3 * 2 + 1\n7\n\n\n\n\n\nmuladd(A, y, z)\n\nCombined multiply-add, A*y .+ z, for matrix-matrix or matrix-vector multiplication. The result is always the same size as A*y, but z may be smaller, or a scalar.\n\ncompat: Julia 1.6\nThese methods require Julia 1.6 or later.\n\nExamples\n\njulia> A=[1.0 2.0; 3.0 4.0]; B=[1.0 1.0; 1.0 1.0]; z=[0, 100];\n\njulia> muladd(A, B, z)\n2×2 Matrix{Float64}:\n   3.0    3.0\n 107.0  107.0\n\n\n\n\n\n"},{"title":"Base.inv","page":"Mathematics","location":"base/math.html#Base.inv-Tuple{Number}","category":"method","text":"inv(x)\n\nReturn the multiplicative inverse of x, such that x*inv(x) or inv(x)*x yields one(x) (the multiplicative identity) up to roundoff errors.\n\nIf x is a number, this is essentially the same as one(x)/x, but for some types inv(x) may be slightly more efficient.\n\nExamples\n\njulia> inv(2)\n0.5\n\njulia> inv(1 + 2im)\n0.2 - 0.4im\n\njulia> inv(1 + 2im) * (1 + 2im)\n1.0 + 0.0im\n\njulia> inv(2//3)\n3//2\n\ncompat: Julia 1.2\ninv(::Missing) requires at least Julia 1.2.\n\n\n\n\n\n"},{"title":"Base.div","page":"Mathematics","location":"base/math.html#Base.div","category":"function","text":"div(x, y)\n÷(x, y)\n\nThe quotient from Euclidean (integer) division. Generally equivalent to a mathematical operation x/y without a fractional part.\n\nSee also: cld, fld, rem, divrem.\n\nExamples\n\njulia> 9 ÷ 4\n2\n\njulia> -5 ÷ 3\n-1\n\njulia> 5.0 ÷ 2\n2.0\n\njulia> div.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -1  -1  -1  0  0  0  0  0  1  1  1\n\n\n\n\n\n"},{"title":"Base.div","page":"Mathematics","location":"base/math.html#Base.div-Tuple{Any, Any, RoundingMode}","category":"method","text":"div(x, y, r::RoundingMode=RoundToZero)\n\nThe quotient from Euclidean (integer) division. Computes x / y, rounded to an integer according to the rounding mode r. In other words, the quantity\n\nround(x / y, r)\n\nwithout any intermediate rounding.\n\ncompat: Julia 1.4\nThe three-argument method taking a RoundingMode requires Julia 1.4 or later.\n\nSee also fld and cld, which are special cases of this function.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9.\n\nExamples:\n\njulia> div(4, 3, RoundToZero) # Matches div(4, 3)\n1\njulia> div(4, 3, RoundDown) # Matches fld(4, 3)\n1\njulia> div(4, 3, RoundUp) # Matches cld(4, 3)\n2\njulia> div(5, 2, RoundNearest)\n2\njulia> div(5, 2, RoundNearestTiesAway)\n3\njulia> div(-5, 2, RoundNearest)\n-2\njulia> div(-5, 2, RoundNearestTiesAway)\n-3\njulia> div(-5, 2, RoundNearestTiesUp)\n-2\njulia> div(4, 3, RoundFromZero)\n2\njulia> div(-4, 3, RoundFromZero)\n-2\n\nBecause div(x, y) implements strictly correct truncated rounding based on the true value of floating-point numbers, unintuitive situations can arise. For example:\n\njulia> div(6.0, 0.1)\n59.0\njulia> 6.0 / 0.1\n60.0\njulia> 6.0 / big(0.1)\n59.99999999999999666933092612453056361837965690217069245739573412231113406246995\n\nWhat is happening here is that the true value of the floating-point number written as 0.1 is slightly larger than the numerical value 1/10 while 6.0 represents the number 6 precisely. Therefore the true value of 6.0 / 0.1 is slightly less than 60. When doing division, this is rounded to precisely 60.0, but div(6.0, 0.1, RoundToZero) always truncates the true value, so the result is 59.0.\n\n\n\n\n\n"},{"title":"Base.fld","page":"Mathematics","location":"base/math.html#Base.fld","category":"function","text":"fld(x, y)\n\nLargest integer less than or equal to x / y. Equivalent to div(x, y, RoundDown).\n\nSee also div, cld, fld1.\n\nExamples\n\njulia> fld(7.3, 5.5)\n1.0\n\njulia> fld.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -2  -2  -1  -1  -1  0  0  0  1  1  1\n\nBecause fld(x, y) implements strictly correct floored rounding based on the true value of floating-point numbers, unintuitive situations can arise. For example:\n\njulia> fld(6.0, 0.1)\n59.0\njulia> 6.0 / 0.1\n60.0\njulia> 6.0 / big(0.1)\n59.99999999999999666933092612453056361837965690217069245739573412231113406246995\n\nWhat is happening here is that the true value of the floating-point number written as 0.1 is slightly larger than the numerical value 1/10 while 6.0 represents the number 6 precisely. Therefore the true value of 6.0 / 0.1 is slightly less than 60. When doing division, this is rounded to precisely 60.0, but fld(6.0, 0.1) always takes the floor of the true value, so the result is 59.0.\n\n\n\n\n\n"},{"title":"Base.cld","page":"Mathematics","location":"base/math.html#Base.cld","category":"function","text":"cld(x, y)\n\nSmallest integer larger than or equal to x / y. Equivalent to div(x, y, RoundUp).\n\nSee also div, fld.\n\nExamples\n\njulia> cld(5.5, 2.2)\n3.0\n\njulia> cld.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -1  -1  -1  0  0  0  1  1  1  2  2\n\n\n\n\n\n"},{"title":"Base.mod","page":"Mathematics","location":"base/math.html#Base.mod","category":"function","text":"rem(x::Integer, T::Type{<:Integer})::T\nmod(x::Integer, T::Type{<:Integer})::T\n%(x::Integer, T::Type{<:Integer})::T\n\nFind y::T such that x ≡ y (mod n), where n is the number of integers representable in T, and y is an integer in [typemin(T),typemax(T)]. If T can represent any integer (e.g. T == BigInt), then this operation corresponds to a conversion to T.\n\nExamples\n\njulia> x = 129 % Int8\n-127\n\njulia> typeof(x)\nInt8\n\njulia> x = 129 % BigInt\n129\n\njulia> typeof(x)\nBigInt\n\n\n\n\n\nmod(x, y)\nrem(x, y, RoundDown)\n\nThe reduction of x modulo y, or equivalently, the remainder of x after floored division by y, i.e. x - y*fld(x,y) if computed without intermediate rounding.\n\nThe result will have the same sign as y if isfinite(y), and magnitude less than abs(y) (with some exceptions, see note below).\n\nnote: Note\nWhen used with floating point values, the exact result may not be representable by the type, and so rounding error may occur. In particular, if the exact result is very close to y, then it may be rounded to y.\n\nSee also: rem, div, fld, mod1, invmod.\n\njulia> mod(8, 3)\n2\n\njulia> mod(9, 3)\n0\n\njulia> mod(8.9, 3)\n2.9000000000000004\n\njulia> mod(eps(), 3)\n2.220446049250313e-16\n\njulia> mod(-eps(), 3)\n3.0\n\njulia> mod.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n 1  2  0  1  2  0  1  2  0  1  2\n\n\n\n\n\nmod(x::Integer, r::AbstractUnitRange)\n\nFind y in the range r such that x ≡ y (mod n), where n = length(r), i.e. y = mod(x - first(r), n) + first(r).\n\nSee also mod1.\n\nExamples\n\njulia> mod(0, Base.OneTo(3))  # mod1(0, 3)\n3\n\njulia> mod(3, 0:2)  # mod(3, 3)\n0\n\ncompat: Julia 1.3\nThis method requires at least Julia 1.3.\n\n\n\n\n\n"},{"title":"Base.rem","page":"Mathematics","location":"base/math.html#Base.rem","category":"function","text":"rem(x, y)\n%(x, y)\n\nRemainder from Euclidean division, returning a value of the same sign as x, and smaller in magnitude than y. This value is always exact.\n\nSee also: div, mod, mod1, divrem.\n\nExamples\n\njulia> x = 15; y = 4;\n\njulia> x % y\n3\n\njulia> x == div(x, y) * y + rem(x, y)\ntrue\n\njulia> rem.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -2  -1  0  -2  -1  0  1  2  0  1  2\n\n\n\n\n\n"},{"title":"Base.rem","page":"Mathematics","location":"base/math.html#Base.rem-Tuple{Any, Any, RoundingMode}","category":"method","text":"rem(x, y, r::RoundingMode=RoundToZero)\n\nCompute the remainder of x after integer division by y, with the quotient rounded according to the rounding mode r. In other words, the quantity\n\nx - y * round(x / y, r)\n\nwithout any intermediate rounding.\n\nif r == RoundNearest, then the result is exact, and in the interval -y  2 y  2. See also RoundNearest.\nif r == RoundToZero (default), then the result is exact, and in the interval 0 y) if x is positive, or (-y 0 otherwise. See also RoundToZero.\nif r == RoundDown, then the result is in the interval 0 y) if y is positive, or (y 0 otherwise. The result may not be exact if x and y have different signs, and abs(x) < abs(y). See also RoundDown.\nif r == RoundUp, then the result is in the interval (-y 0 if y is positive, or 0 -y) otherwise. The result may not be exact if x and y have the same sign, and abs(x) < abs(y). See also RoundUp.\nif r == RoundFromZero, then the result is in the interval (-y 0 if y is positive, or 0 -y) otherwise. The result may not be exact if x and y have the same sign, and abs(x) < abs(y). See also RoundFromZero.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9.\n\nExamples:\n\njulia> x = 9; y = 4;\n\njulia> x % y  # same as rem(x, y)\n1\n\njulia> x ÷ y  # same as div(x, y)\n2\n\njulia> x == div(x, y) * y + rem(x, y)\ntrue\n\n\n\n\n\n"},{"title":"Base.Math.rem2pi","page":"Mathematics","location":"base/math.html#Base.Math.rem2pi","category":"function","text":"rem2pi(x, r::RoundingMode)\n\nCompute the remainder of x after integer division by 2π, with the quotient rounded according to the rounding mode r. In other words, the quantity\n\nx - 2π*round(x/(2π),r)\n\nwithout any intermediate rounding. This internally uses a high precision approximation of 2π, and so will give a more accurate result than rem(x,2π,r)\n\nif r == RoundNearest, then the result is in the interval -π π. This will generally be the most accurate result. See also RoundNearest.\nif r == RoundToZero, then the result is in the interval 0 2π if x is positive,. or -2π 0 otherwise. See also RoundToZero.\nif r == RoundDown, then the result is in the interval 0 2π. See also RoundDown.\nif r == RoundUp, then the result is in the interval -2π 0. See also RoundUp.\n\nExamples\n\njulia> rem2pi(7pi/4, RoundNearest)\n-0.7853981633974485\n\njulia> rem2pi(7pi/4, RoundDown)\n5.497787143782138\n\n\n\n\n\n"},{"title":"Base.Math.mod2pi","page":"Mathematics","location":"base/math.html#Base.Math.mod2pi","category":"function","text":"mod2pi(x)\n\nModulus after division by 2π, returning in the range 02π).\n\nThis function computes a floating point representation of the modulus after division by numerically exact 2π, and is therefore not exactly the same as mod(x,2π), which would compute the modulus of x relative to division by the floating-point number 2π.\n\nnote: Note\nDepending on the format of the input value, the closest representable value to 2π may be less than 2π. For example, the expression mod2pi(2π) will not return 0, because the intermediate value of 2*π is a Float64 and 2*Float64(π) < 2*big(π). See rem2pi for more refined control of this behavior.\n\nExamples\n\njulia> mod2pi(9*pi/4)\n0.7853981633974481\n\n\n\n\n\n"},{"title":"Base.divrem","page":"Mathematics","location":"base/math.html#Base.divrem","category":"function","text":"divrem(x, y, r::RoundingMode=RoundToZero)\n\nThe quotient and remainder from Euclidean division. Equivalent to (div(x, y, r), rem(x, y, r)). Equivalently, with the default value of r, this call is equivalent to (x ÷ y, x % y).\n\nSee also: fldmod, cld.\n\nExamples\n\njulia> divrem(3, 7)\n(0, 3)\n\njulia> divrem(7, 3)\n(2, 1)\n\n\n\n\n\n"},{"title":"Base.fldmod","page":"Mathematics","location":"base/math.html#Base.fldmod","category":"function","text":"fldmod(x, y)\n\nThe floored quotient and modulus after division. A convenience wrapper for divrem(x, y, RoundDown). Equivalent to (fld(x, y), mod(x, y)).\n\nSee also: fld, cld, fldmod1.\n\n\n\n\n\n"},{"title":"Base.fld1","page":"Mathematics","location":"base/math.html#Base.fld1","category":"function","text":"fld1(x, y)\n\nFlooring division, returning a value consistent with mod1(x,y)\n\nSee also mod1, fldmod1.\n\nExamples\n\njulia> x = 15; y = 4;\n\njulia> fld1(x, y)\n4\n\njulia> x == fld(x, y) * y + mod(x, y)\ntrue\n\njulia> x == (fld1(x, y) - 1) * y + mod1(x, y)\ntrue\n\n\n\n\n\n"},{"title":"Base.mod1","page":"Mathematics","location":"base/math.html#Base.mod1","category":"function","text":"mod1(x, y)\n\nModulus after flooring division, returning a value r such that mod(r, y) == mod(x, y) in the range (0 y for positive y and in the range y0) for negative y.\n\nWith integer arguments and positive y, this is equal to mod(x, 1:y), and hence natural for 1-based indexing. By comparison, mod(x, y) == mod(x, 0:y-1) is natural for computations with offsets or strides.\n\nSee also mod, fld1, fldmod1.\n\nExamples\n\njulia> mod1(4, 2)\n2\n\njulia> mod1.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n 1  2  3  1  2  3  1  2  3  1  2\n\njulia> mod1.([-0.1, 0, 0.1, 1, 2, 2.9, 3, 3.1]', 3)\n1×8 Matrix{Float64}:\n 2.9  3.0  0.1  1.0  2.0  2.9  3.0  0.1\n\n\n\n\n\n"},{"title":"Base.fldmod1","page":"Mathematics","location":"base/math.html#Base.fldmod1","category":"function","text":"fldmod1(x, y)\n\nReturn (fld1(x,y), mod1(x,y)).\n\nSee also fld1, mod1.\n\n\n\n\n\n"},{"title":"Base.://","page":"Mathematics","location":"base/math.html#Base.://","category":"function","text":"//(num, den)\n\nDivide two integers or rational numbers, giving a Rational result. More generally, // can be used for exact rational division of other numeric types with integer or rational components, such as complex numbers with integer components.\n\nNote that floating-point (AbstractFloat) arguments are not permitted by // (even if the values are rational). The arguments must be subtypes of Integer, Rational, or composites thereof.\n\nExamples\n\njulia> 3 // 5\n3//5\n\njulia> (3 // 5) // (2 // 1)\n3//10\n\njulia> (1+2im) // (3+4im)\n11//25 + 2//25*im\n\njulia> 1.0 // 2\nERROR: MethodError: no method matching //(::Float64, ::Int64)\n[...]\n\n\n\n\n\n"},{"title":"Base.rationalize","page":"Mathematics","location":"base/math.html#Base.rationalize","category":"function","text":"rationalize([T<:Integer=Int,] x; tol::Real=eps(x))\n\nApproximate floating point number x as a Rational number with components of the given integer type. The result will differ from x by no more than tol.\n\nExamples\n\njulia> rationalize(5.6)\n28//5\n\njulia> a = rationalize(BigInt, 10.3)\n103//10\n\njulia> typeof(numerator(a))\nBigInt\n\n\n\n\n\n"},{"title":"Base.numerator","page":"Mathematics","location":"base/math.html#Base.numerator","category":"function","text":"numerator(x)\n\nNumerator of the rational representation of x.\n\nExamples\n\njulia> numerator(2//3)\n2\n\njulia> numerator(4)\n4\n\n\n\n\n\n"},{"title":"Base.denominator","page":"Mathematics","location":"base/math.html#Base.denominator","category":"function","text":"denominator(x)\n\nDenominator of the rational representation of x.\n\nExamples\n\njulia> denominator(2//3)\n3\n\njulia> denominator(4)\n1\n\n\n\n\n\n"},{"title":"Base.:<<","page":"Mathematics","location":"base/math.html#Base.:<<","category":"function","text":"<<(B::BitVector, n)::BitVector\n\nLeft bit shift operator, B << n. For n >= 0, the result is B with elements shifted n positions backwards, filling with false values. If n < 0, elements are shifted forwards. Equivalent to B >> -n.\n\nExamples\n\njulia> B = BitVector([true, false, true, false, false])\n5-element BitVector:\n 1\n 0\n 1\n 0\n 0\n\njulia> B << 1\n5-element BitVector:\n 0\n 1\n 0\n 0\n 0\n\njulia> B << -1\n5-element BitVector:\n 0\n 1\n 0\n 1\n 0\n\n\n\n\n\n<<(x, n)\n\nLeft bit shift operator, x << n. For n >= 0, the result is x shifted left by n bits, filling with 0s. This is equivalent to x * 2^n. For n < 0, this is equivalent to x >> -n.\n\nExamples\n\njulia> Int8(3) << 2\n12\n\njulia> bitstring(Int8(3))\n\"00000011\"\n\njulia> bitstring(Int8(12))\n\"00001100\"\n\nSee also >>, >>>, exp2, ldexp.\n\n\n\n\n\n"},{"title":"Base.:>>","page":"Mathematics","location":"base/math.html#Base.:>>","category":"function","text":">>(B::BitVector, n)::BitVector\n\nRight bit shift operator, B >> n. For n >= 0, the result is B with elements shifted n positions forward, filling with false values. If n < 0, elements are shifted backwards. Equivalent to B << -n.\n\nExamples\n\njulia> B = BitVector([true, false, true, false, false])\n5-element BitVector:\n 1\n 0\n 1\n 0\n 0\n\njulia> B >> 1\n5-element BitVector:\n 0\n 1\n 0\n 1\n 0\n\njulia> B >> -1\n5-element BitVector:\n 0\n 1\n 0\n 0\n 0\n\n\n\n\n\n>>(x, n)\n\nRight bit shift operator, x >> n. For n >= 0, the result is x shifted right by n bits, filling with 0s if x >= 0, 1s if x < 0, preserving the sign of x. This is equivalent to fld(x, 2^n). For n < 0, this is equivalent to x << -n.\n\nExamples\n\njulia> Int8(13) >> 2\n3\n\njulia> bitstring(Int8(13))\n\"00001101\"\n\njulia> bitstring(Int8(3))\n\"00000011\"\n\njulia> Int8(-14) >> 2\n-4\n\njulia> bitstring(Int8(-14))\n\"11110010\"\n\njulia> bitstring(Int8(-4))\n\"11111100\"\n\nSee also >>>, <<.\n\n\n\n\n\n"},{"title":"Base.:>>>","page":"Mathematics","location":"base/math.html#Base.:>>>","category":"function","text":">>>(B::BitVector, n)::BitVector\n\nUnsigned right bitshift operator, B >>> n. Equivalent to B >> n. See >> for details and examples.\n\n\n\n\n\n>>>(x, n)\n\nUnsigned right bit shift operator, x >>> n. For n >= 0, the result is x shifted right by n bits, filling with 0s. For n < 0, this is equivalent to x << -n.\n\nFor Unsigned integer types, this is equivalent to >>. For Signed integer types, this is equivalent to signed(unsigned(x) >> n).\n\nExamples\n\njulia> Int8(-14) >>> 2\n60\n\njulia> bitstring(Int8(-14))\n\"11110010\"\n\njulia> bitstring(Int8(60))\n\"00111100\"\n\nBigInts are treated as if having infinite size, so no filling is required and this is equivalent to >>.\n\nSee also >>, <<.\n\n\n\n\n\n"},{"title":"Base.bitrotate","page":"Mathematics","location":"base/math.html#Base.bitrotate","category":"function","text":"bitrotate(x::Base.BitInteger, k::Integer)\n\nbitrotate(x, k) implements bitwise rotation. It returns the value of x with its bits rotated left k times. A negative value of k will rotate to the right instead.\n\ncompat: Julia 1.5\nThis function requires Julia 1.5 or later.\n\nSee also: <<, circshift, BitArray.\n\njulia> bitrotate(UInt8(114), 2)\n0xc9\n\njulia> bitstring(bitrotate(0b01110010, 2))\n\"11001001\"\n\njulia> bitstring(bitrotate(0b01110010, -2))\n\"10011100\"\n\njulia> bitstring(bitrotate(0b01110010, 8))\n\"01110010\"\n\n\n\n\n\n"},{"title":"Base.::","page":"Mathematics","location":"base/math.html#Base.::-Tuple{Any, Any, Any}","category":"method","text":"(:)(start, [step], stop)\n\nRange operator. a:b constructs a range from a to b with a step size equal to +1, which produces:\n\na UnitRange when a and b are integers, or\na StepRange when a and b are characters, or\na StepRangeLen when a and/or b are floating-point.\n\na:s:b is similar but uses a step size of s (a StepRange or StepRangeLen). See also range for more control.\n\nTo create a descending range, use reverse(a:b) or a negative step size, e.g. b:-1:a. Otherwise, when b < a, an empty range will be constructed and normalized to a:a-1.\n\nThe operator : is also used in indexing to select whole dimensions, e.g. in A[:, 1].\n\n: is also used to quote code, e.g. :(x + y) isa Expr and :x isa Symbol. Since :2 isa Int, it does not create a range in indexing: v[:2] == v[2] != v[begin:2].\n\n\n\n\n\n"},{"title":"Base.::","page":"Mathematics","location":"base/math.html#Base.::-Tuple{CartesianIndex, CartesianIndex, CartesianIndex}","category":"method","text":"(:)(start::CartesianIndex, [step::CartesianIndex], stop::CartesianIndex)\n\nConstruct CartesianIndices from two CartesianIndex and an optional step.\n\ncompat: Julia 1.1\nThis method requires at least Julia 1.1.\n\ncompat: Julia 1.6\nThe step range method start:step:stop requires at least Julia 1.6.\n\nExamples\n\njulia> I = CartesianIndex(2,1);\n\njulia> J = CartesianIndex(3,3);\n\njulia> I:J\nCartesianIndices((2:3, 1:3))\n\njulia> I:CartesianIndex(1, 2):J\nCartesianIndices((2:1:3, 1:2:3))\n\n\n\n\n\n"},{"title":"Base.range","page":"Mathematics","location":"base/math.html#Base.range","category":"function","text":"range(start, stop, length)\nrange(start, stop; length, step)\nrange(start; length, stop, step)\nrange(;start, length, stop, step)\n\nConstruct a specialized array with evenly spaced elements and optimized storage (an AbstractRange) from the arguments. Mathematically a range is uniquely determined by any three of start, step, stop and length. Valid invocations of range are:\n\nCall range with any three of start, step, stop, length.\nCall range with two of start, stop, length. In this case step will be assumed to be positive one. If both arguments are Integers, a UnitRange will be returned.\nCall range with one of stop or length. start and step will be assumed to be positive one.\n\nTo construct a descending range, specify a negative step size, e.g. range(5, 1; step = -1) => [5,4,3,2,1]. Otherwise, a stop value less than the start value, with the default step of +1, constructs an empty range. Empty ranges are normalized such that the stop is one less than the start, e.g. range(5, 1) == 5:4.\n\nSee Extended Help for additional details on the returned type. See also logrange for logarithmically spaced points.\n\nExamples\n\njulia> range(1, length=100)\n1:100\n\njulia> range(1, stop=100)\n1:100\n\njulia> range(1, step=5, length=100)\n1:5:496\n\njulia> range(1, step=5, stop=100)\n1:5:96\n\njulia> range(1, 10, length=101)\n1.0:0.09:10.0\n\njulia> range(1, 100, step=5)\n1:5:96\n\njulia> range(stop=10, length=5)\n6:10\n\njulia> range(stop=10, step=1, length=5)\n6:1:10\n\njulia> range(start=1, step=1, stop=10)\n1:1:10\n\njulia> range(; length = 10)\nBase.OneTo(10)\n\njulia> range(; stop = 6)\nBase.OneTo(6)\n\njulia> range(; stop = 6.5)\n1.0:1.0:6.0\n\nIf length is not specified and stop - start is not an integer multiple of step, a range that ends before stop will be produced.\n\njulia> range(1, 3.5, step=2)\n1.0:2.0:3.0\n\nSpecial care is taken to ensure intermediate values are computed rationally. To avoid this induced overhead, see the LinRange constructor.\n\ncompat: Julia 1.1\nstop as a positional argument requires at least Julia 1.1.\n\ncompat: Julia 1.7\nThe versions without keyword arguments and start as a keyword argument require at least Julia 1.7.\n\ncompat: Julia 1.8\nThe versions with stop as a sole keyword argument, or length as a sole keyword argument require at least Julia 1.8.\n\nExtended Help\n\nrange will produce a Base.OneTo when the arguments are Integers and\n\nOnly length is provided\nOnly stop is provided\n\nrange will produce a UnitRange when the arguments are Integers and\n\nOnly start  and stop are provided\nOnly length and stop are provided\n\nA UnitRange is not produced if step is provided even if specified as one.\n\n\n\n\n\n"},{"title":"Base.OneTo","page":"Mathematics","location":"base/math.html#Base.OneTo","category":"type","text":"Base.OneTo(n)\n\nDefine an AbstractUnitRange that behaves like 1:n, with the added distinction that the lower limit is guaranteed (by the type system) to be 1.\n\n\n\n\n\n"},{"title":"Base.StepRangeLen","page":"Mathematics","location":"base/math.html#Base.StepRangeLen","category":"type","text":"StepRangeLen(         ref::R, step::S, len, [offset=1]) where {  R,S}\nStepRangeLen{T,R,S}(  ref::R, step::S, len, [offset=1]) where {T,R,S}\nStepRangeLen{T,R,S,L}(ref::R, step::S, len, [offset=1]) where {T,R,S,L}\n\nA range r where r[i] produces values of type T (in the first form, T is deduced automatically), parameterized by a reference value, a step, and the length. By default ref is the starting value r[1], but alternatively you can supply it as the value of r[offset] for some other index 1 <= offset <= len. The syntax a:b or a:b:c, where any of a, b, or c are floating-point numbers, creates a StepRangeLen.\n\ncompat: Julia 1.7\nThe 4th type parameter L requires at least Julia 1.7.\n\n\n\n\n\n"},{"title":"Base.logrange","page":"Mathematics","location":"base/math.html#Base.logrange","category":"function","text":"logrange(start, stop, length)\nlogrange(start, stop; length)\n\nConstruct a specialized array whose elements are spaced logarithmically between the given endpoints. That is, the ratio of successive elements is a constant, calculated from the length.\n\nThis is similar to geomspace in Python. Unlike PowerRange in Mathematica, you specify the number of elements not the ratio. Unlike logspace in Python and Matlab, the start and stop arguments are always the first and last elements of the result, not powers applied to some base.\n\nExamples\n\njulia> logrange(10, 4000, length=3)\n3-element Base.LogRange{Float64, Base.TwicePrecision{Float64}}:\n 10.0, 200.0, 4000.0\n\njulia> ans[2] ≈ sqrt(10 * 4000)  # middle element is the geometric mean\ntrue\n\njulia> range(10, 40, length=3)[2] ≈ (10 + 40)/2  # arithmetic mean\ntrue\n\njulia> logrange(1f0, 32f0, 11)\n11-element Base.LogRange{Float32, Float64}:\n 1.0, 1.41421, 2.0, 2.82843, 4.0, 5.65685, 8.0, 11.3137, 16.0, 22.6274, 32.0\n\njulia> logrange(1, 1000, length=4) ≈ 10 .^ (0:3)\ntrue\n\nSee the LogRange type for further details.\n\nSee also range for linearly spaced points.\n\ncompat: Julia 1.11\nThis function requires at least Julia 1.11.\n\n\n\n\n\n"},{"title":"Base.LogRange","page":"Mathematics","location":"base/math.html#Base.LogRange","category":"type","text":"LogRange{T}(start, stop, len) <: AbstractVector{T}\n\nA range whose elements are spaced logarithmically between start and stop, with spacing controlled by len. Returned by logrange.\n\nLike LinRange, the first and last elements will be exactly those provided, but intermediate values may have small floating-point errors. These are calculated using the logs of the endpoints, which are stored on construction, often in higher precision than T.\n\nExamples\n\njulia> logrange(1, 4, length=5)\n5-element Base.LogRange{Float64, Base.TwicePrecision{Float64}}:\n 1.0, 1.41421, 2.0, 2.82843, 4.0\n\njulia> Base.LogRange{Float16}(1, 4, 5)\n5-element Base.LogRange{Float16, Float64}:\n 1.0, 1.414, 2.0, 2.828, 4.0\n\njulia> logrange(1e-310, 1e-300, 11)[1:2:end]\n6-element Vector{Float64}:\n 1.0e-310\n 9.999999999999974e-309\n 9.999999999999981e-307\n 9.999999999999988e-305\n 9.999999999999994e-303\n 1.0e-300\n\njulia> prevfloat(1e-308, 5) == ans[2]\ntrue\n\nNote that integer eltype T is not allowed. Use for instance round.(Int, xs), or explicit powers of some integer base:\n\njulia> xs = logrange(1, 512, 4)\n4-element Base.LogRange{Float64, Base.TwicePrecision{Float64}}:\n 1.0, 8.0, 64.0, 512.0\n\njulia> 2 .^ (0:3:9) |> println\n[1, 8, 64, 512]\n\ncompat: Julia 1.11\nThis type requires at least Julia 1.11.\n\n\n\n\n\n"},{"title":"Base.:==","page":"Mathematics","location":"base/math.html#Base.:==","category":"function","text":"==(x, y)\n\nGeneric equality operator. Falls back to ===. Should be implemented for all types with a notion of equality, based on the abstract value that an instance represents. For example, all numeric types are compared by numeric value, ignoring type. Strings are compared as sequences of characters, ignoring encoding. Collections of the same type generally compare their key sets, and if those are ==, then compare the values for each of those keys, returning true if all such pairs are ==. Other properties are typically not taken into account (such as the exact type).\n\nThis operator follows IEEE semantics for floating-point numbers: 0.0 == -0.0 and NaN != NaN.\n\nThe result is of type Bool, except when one of the operands is missing, in which case missing is returned (three-valued logic). Collections generally implement three-valued logic akin to all, returning missing if any operands contain missing values and all other pairs are equal. Use isequal or === to always get a Bool result.\n\nImplementation\n\nNew numeric types should implement this function for two arguments of the new type, and handle comparison to other types via promotion rules where possible.\n\nEquality and hashing are intimately related; two values that are considered isequal must have the same hash and by default isequal falls back to ==. If a type customizes the behavior of == and/or isequal, then hash must be similarly implemented to ensure isequal and hash agree. Sets, Dicts, and many other internal implementations assume that this invariant holds.\n\nIf some type defines ==, isequal, and isless then it should also implement < to ensure consistency of comparisons.\n\n\n\n\n\n"},{"title":"Base.:!=","page":"Mathematics","location":"base/math.html#Base.:!=","category":"function","text":"!=(x)\n\nCreate a function that compares its argument to x using !=, i.e. a function equivalent to y -> y != x. The returned function is of type Base.Fix2{typeof(!=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n!=(x, y)\n≠(x,y)\n\nNot-equals comparison operator. Always gives the opposite answer as ==.\n\nImplementation\n\nNew types should generally not implement this, and rely on the fallback definition !=(x,y) = !(x==y) instead.\n\nExamples\n\njulia> 3 != 2\ntrue\n\njulia> \"foo\" ≠ \"foo\"\nfalse\n\n\n\n\n\n"},{"title":"Core.:!==","page":"Mathematics","location":"base/math.html#Core.:!==","category":"function","text":"!==(x, y)\n≢(x,y)\n\nAlways gives the opposite answer as ===.\n\nExamples\n\njulia> a = [1, 2]; b = [1, 2];\n\njulia> a ≢ b\ntrue\n\njulia> a ≢ a\nfalse\n\n\n\n\n\n"},{"title":"Base.:<","page":"Mathematics","location":"base/math.html#Base.:<","category":"function","text":"<(x)\n\nCreate a function that compares its argument to x using <, i.e. a function equivalent to y -> y < x. The returned function is of type Base.Fix2{typeof(<)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n<(x, y)\n\nLess-than comparison operator. Falls back to isless. Because of the behavior of floating-point NaN values, this operator implements a partial order.\n\nImplementation\n\nNew types with a canonical partial order should implement this function for two arguments of the new type. Types with a canonical total order should implement isless instead.\n\nSee also isunordered.\n\nExamples\n\njulia> 'a' < 'b'\ntrue\n\njulia> \"abc\" < \"abd\"\ntrue\n\njulia> 5 < 3\nfalse\n\n\n\n\n\n"},{"title":"Base.:<=","page":"Mathematics","location":"base/math.html#Base.:<=","category":"function","text":"<=(x)\n\nCreate a function that compares its argument to x using <=, i.e. a function equivalent to y -> y <= x. The returned function is of type Base.Fix2{typeof(<=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n<=(x, y)\n≤(x,y)\n\nLess-than-or-equals comparison operator. Falls back to (x < y) | (x == y).\n\nExamples\n\njulia> 'a' <= 'b'\ntrue\n\njulia> 7 ≤ 7 ≤ 9\ntrue\n\njulia> \"abc\" ≤ \"abc\"\ntrue\n\njulia> 5 <= 3\nfalse\n\n\n\n\n\n"},{"title":"Base.:>","page":"Mathematics","location":"base/math.html#Base.:>","category":"function","text":">(x)\n\nCreate a function that compares its argument to x using >, i.e. a function equivalent to y -> y > x. The returned function is of type Base.Fix2{typeof(>)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n>(x, y)\n\nGreater-than comparison operator. Falls back to y < x.\n\nImplementation\n\nGenerally, new types should implement < instead of this function, and rely on the fallback definition >(x, y) = y < x.\n\nExamples\n\njulia> 'a' > 'b'\nfalse\n\njulia> 7 > 3 > 1\ntrue\n\njulia> \"abc\" > \"abd\"\nfalse\n\njulia> 5 > 3\ntrue\n\n\n\n\n\n"},{"title":"Base.:>=","page":"Mathematics","location":"base/math.html#Base.:>=","category":"function","text":">=(x)\n\nCreate a function that compares its argument to x using >=, i.e. a function equivalent to y -> y >= x. The returned function is of type Base.Fix2{typeof(>=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n>=(x, y)\n≥(x,y)\n\nGreater-than-or-equals comparison operator. Falls back to y <= x.\n\nImplementation\n\nNew types should prefer to implement <= instead of this function, and rely on the fallback definition >=(x, y) = y <= x.\n\nFurthermore, in many cases it is enough to implement just < and ==, relying on the fallback definitions of both <= and >=.\n\nExamples\n\njulia> 'a' >= 'b'\nfalse\n\njulia> 7 ≥ 7 ≥ 3\ntrue\n\njulia> \"abc\" ≥ \"abc\"\ntrue\n\njulia> 5 >= 3\ntrue\n\n\n\n\n\n"},{"title":"Base.cmp","page":"Mathematics","location":"base/math.html#Base.cmp","category":"function","text":"cmp(a::AbstractString, b::AbstractString)::Int\n\nCompare two strings. Return 0 if both strings have the same length and the character at each index is the same in both strings. Return -1 if a is a prefix of b, or if a comes before b in alphabetical order. Return 1 if b is a prefix of a, or if b comes before a in alphabetical order (technically, lexicographical order by Unicode code points).\n\nExamples\n\njulia> cmp(\"abc\", \"abc\")\n0\n\njulia> cmp(\"ab\", \"abc\")\n-1\n\njulia> cmp(\"abc\", \"ab\")\n1\n\njulia> cmp(\"ab\", \"ac\")\n-1\n\njulia> cmp(\"ac\", \"ab\")\n1\n\njulia> cmp(\"α\", \"a\")\n1\n\njulia> cmp(\"b\", \"β\")\n-1\n\n\n\n\n\ncmp(<, x, y)\n\nReturn -1, 0, or 1 depending on whether x is less than, equal to, or greater than y, respectively. The first argument specifies a less-than comparison function to use.\n\n\n\n\n\ncmp(x,y)\n\nReturn -1, 0, or 1 depending on whether x is less than, equal to, or greater than y, respectively. Uses the total order implemented by isless.\n\nExamples\n\njulia> cmp(1, 2)\n-1\n\njulia> cmp(2, 1)\n1\n\njulia> cmp(2+im, 3-im)\nERROR: MethodError: no method matching isless(::Complex{Int64}, ::Complex{Int64})\n[...]\n\n\n\n\n\n"},{"title":"Base.:~","page":"Mathematics","location":"base/math.html#Base.:~","category":"function","text":"~(x)\n\nBitwise not.\n\nSee also: !, &, |.\n\nExamples\n\njulia> ~4\n-5\n\njulia> ~10\n-11\n\njulia> ~true\nfalse\n\n\n\n\n\n"},{"title":"Base.:&","page":"Mathematics","location":"base/math.html#Base.:&","category":"function","text":"x & y\n\nBitwise and. Implements three-valued logic, returning missing if one operand is missing and the other is true. Add parentheses for function application form: (&)(x, y).\n\nSee also: |, xor, &&.\n\nExamples\n\njulia> 4 & 10\n0\n\njulia> 4 & 12\n4\n\njulia> true & missing\nmissing\n\njulia> false & missing\nfalse\n\n\n\n\n\n"},{"title":"Base.:|","page":"Mathematics","location":"base/math.html#Base.:|","category":"function","text":"x | y\n\nBitwise or. Implements three-valued logic, returning missing if one operand is missing and the other is false.\n\nSee also: &, xor, ||.\n\nExamples\n\njulia> 4 | 10\n14\n\njulia> 4 | 1\n5\n\njulia> true | missing\ntrue\n\njulia> false | missing\nmissing\n\n\n\n\n\n"},{"title":"Base.xor","page":"Mathematics","location":"base/math.html#Base.xor","category":"function","text":"xor(x, y)\n⊻(x, y)\n\nBitwise exclusive or of x and y. Implements three-valued logic, returning missing if one of the arguments is missing.\n\nThe infix operation a ⊻ b is a synonym for xor(a,b), and ⊻ can be typed by tab-completing \\xor or \\veebar in the Julia REPL.\n\nExamples\n\njulia> xor(true, false)\ntrue\n\njulia> xor(true, true)\nfalse\n\njulia> xor(true, missing)\nmissing\n\njulia> false ⊻ false\nfalse\n\njulia> [true; true; false] .⊻ [true; false; false]\n3-element BitVector:\n 0\n 1\n 0\n\n\n\n\n\n"},{"title":"Base.nand","page":"Mathematics","location":"base/math.html#Base.nand","category":"function","text":"nand(x, y)\n⊼(x, y)\n\nBitwise nand (not and) of x and y. Implements three-valued logic, returning missing if one of the arguments is missing.\n\nThe infix operation a ⊼ b is a synonym for nand(a,b), and ⊼ can be typed by tab-completing \\nand or \\barwedge in the Julia REPL.\n\nExamples\n\njulia> nand(true, false)\ntrue\n\njulia> nand(true, true)\nfalse\n\njulia> nand(true, missing)\nmissing\n\njulia> false ⊼ false\ntrue\n\njulia> [true; true; false] .⊼ [true; false; false]\n3-element BitVector:\n 0\n 1\n 1\n\n\n\n\n\n"},{"title":"Base.nor","page":"Mathematics","location":"base/math.html#Base.nor","category":"function","text":"nor(x, y)\n⊽(x, y)\n\nBitwise nor (not or) of x and y. Implements three-valued logic, returning missing if one of the arguments is missing and the other is not true.\n\nThe infix operation a ⊽ b is a synonym for nor(a,b), and ⊽ can be typed by tab-completing \\nor or \\barvee in the Julia REPL.\n\nExamples\n\njulia> nor(true, false)\nfalse\n\njulia> nor(true, true)\nfalse\n\njulia> nor(true, missing)\nfalse\n\njulia> false ⊽ false\ntrue\n\njulia> false ⊽ missing\nmissing\n\njulia> [true; true; false] .⊽ [true; false; false]\n3-element BitVector:\n 0\n 0\n 1\n\n\n\n\n\n"},{"title":"Base.:!","page":"Mathematics","location":"base/math.html#Base.:!","category":"function","text":"!f::Function\n\nPredicate function negation: when the argument of ! is a function, it returns a composed function which computes the boolean negation of f.\n\nSee also ∘.\n\nExamples\n\njulia> str = \"∀ ε > 0, ∃ δ > 0: |x-y| < δ ⇒ |f(x)-f(y)| < ε\"\n\"∀ ε > 0, ∃ δ > 0: |x-y| < δ ⇒ |f(x)-f(y)| < ε\"\n\njulia> filter(isletter, str)\n\"εδxyδfxfyε\"\n\njulia> filter(!isletter, str)\n\"∀  > 0, ∃  > 0: |-| <  ⇒ |()-()| < \"\n\ncompat: Julia 1.9\nStarting with Julia 1.9, !f returns a ComposedFunction instead of an anonymous function.\n\n\n\n\n\n!(x)\n\nBoolean not. Implements three-valued logic, returning missing if x is missing.\n\nSee also ~ for bitwise not.\n\nExamples\n\njulia> !true\nfalse\n\njulia> !false\ntrue\n\njulia> !missing\nmissing\n\njulia> .![true false true]\n1×3 BitMatrix:\n 0  1  0\n\n\n\n\n\n"},{"title":"&&","page":"Mathematics","location":"base/math.html#&&","category":"keyword","text":"x && y\n\nShort-circuiting boolean AND.\n\nThis is equivalent to x ? y : false: it returns false if x is false and the result of evaluating y if x is true. Note that if y is an expression, it is only evaluated when x is true, which is called \"short-circuiting\" behavior.\n\nAlso, y does not need to have a boolean value.  This means that (condition) && (statement) can be used as shorthand for if condition; statement; end for an arbitrary statement.\n\nSee also &, the ternary operator ? :, and the manual section on control flow.\n\nExamples\n\njulia> x = 3;\n\njulia> x > 1 && x < 10 && x isa Int\ntrue\n\njulia> x < 0 && error(\"expected positive x\")\nfalse\n\njulia> x > 0 && \"not a boolean\"\n\"not a boolean\"\n\n\n\n\n\n"},{"title":"||","page":"Mathematics","location":"base/math.html#||","category":"keyword","text":"x || y\n\nShort-circuiting boolean OR.\n\nThis is equivalent to x ? true : y: it returns true if x is true and the result of evaluating y if x is false. Note that if y is an expression, it is only evaluated when x is false, which is called \"short-circuiting\" behavior.\n\nAlso, y does not need to have a boolean value.  This means that (condition) || (statement) can be used as shorthand for if !(condition); statement; end for an arbitrary statement.\n\nSee also: |, xor, &&.\n\nExamples\n\njulia> pi < 3 || ℯ < 3\ntrue\n\njulia> false || true || println(\"neither is true!\")\ntrue\n\njulia> pi < 3 || \"not a boolean\"\n\"not a boolean\"\n\n\n\n\n\n"},{"title":"Mathematical Functions","page":"Mathematics","location":"base/math.html#Mathematical-Functions","category":"section","text":""},{"title":"Base.isapprox","page":"Mathematics","location":"base/math.html#Base.isapprox","category":"function","text":"isapprox(x; kwargs...) / ≈(x; kwargs...)\n\nCreate a function that compares its argument to x using ≈, i.e. a function equivalent to y -> y ≈ x.\n\nThe keyword arguments supported here are the same as those in the 2-argument isapprox.\n\ncompat: Julia 1.5\nThis method requires Julia 1.5 or later.\n\n\n\n\n\nisapprox(x, y; atol::Real=0, rtol::Real=atol>0 ? 0 : √eps, nans::Bool=false[, norm::Function])\n\nInexact equality comparison. Two numbers compare equal if their relative distance or their absolute distance is within tolerance bounds: isapprox returns true if norm(x-y) <= max(atol, rtol*max(norm(x), norm(y))). The default atol (absolute tolerance) is zero and the default rtol (relative tolerance) depends on the types of x and y. The keyword argument nans determines whether or not NaN values are considered equal (defaults to false).\n\nFor real or complex floating-point values, if an atol > 0 is not specified, rtol defaults to the square root of eps of the type of x or y, whichever is bigger (least precise). This corresponds to requiring equality of about half of the significant digits. Otherwise, e.g. for integer arguments or if an atol > 0 is supplied, rtol defaults to zero.\n\nThe norm keyword defaults to abs for numeric (x,y) and to LinearAlgebra.norm for arrays (where an alternative norm choice is sometimes useful). When x and y are arrays, if norm(x-y) is not finite (i.e. ±Inf or NaN), the comparison falls back to checking whether all elements of x and y are approximately equal component-wise.\n\nThe binary operator ≈ is equivalent to isapprox with the default arguments, and x ≉ y is equivalent to !isapprox(x,y).\n\nNote that x ≈ 0 (i.e., comparing to zero with the default tolerances) is equivalent to x == 0 since the default atol is 0.  In such cases, you should either supply an appropriate atol (or use norm(x) ≤ atol) or rearrange your code (e.g. use x ≈ y rather than x - y ≈ 0).   It is not possible to pick a nonzero atol automatically because it depends on the overall scaling (the \"units\") of your problem: for example, in x - y ≈ 0, atol=1e-9 is an absurdly small tolerance if x is the radius of the Earth in meters, but an absurdly large tolerance if x is the radius of a Hydrogen atom in meters.\n\ncompat: Julia 1.6\nPassing the norm keyword argument when comparing numeric (non-array) arguments requires Julia 1.6 or later.\n\nExamples\n\njulia> isapprox(0.1, 0.15; atol=0.05)\ntrue\n\njulia> isapprox(0.1, 0.15; rtol=0.34)\ntrue\n\njulia> isapprox(0.1, 0.15; rtol=0.33)\nfalse\n\njulia> 0.1 + 1e-10 ≈ 0.1\ntrue\n\njulia> 1e-10 ≈ 0\nfalse\n\njulia> isapprox(1e-10, 0, atol=1e-8)\ntrue\n\njulia> isapprox([10.0^9, 1.0], [10.0^9, 2.0]) # using `norm`\ntrue\n\n\n\n\n\n"},{"title":"Base.sin","page":"Mathematics","location":"base/math.html#Base.sin-Tuple{Number}","category":"method","text":"sin(x::T) where {T <: Number} -> float(T)\n\nCompute sine of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also sind, sinpi, sincos, cis, asin.\n\nExamples\n\njulia> round.(sin.(range(0, 2pi, length=9)'), digits=3)\n1×9 Matrix{Float64}:\n 0.0  0.707  1.0  0.707  0.0  -0.707  -1.0  -0.707  -0.0\n\njulia> sind(45)\n0.7071067811865476\n\njulia> sinpi(1/4)\n0.7071067811865475\n\njulia> round.(sincos(pi/6), digits=3)\n(0.5, 0.866)\n\njulia> round(cis(pi/6), digits=3)\n0.866 + 0.5im\n\njulia> round(exp(im*pi/6), digits=3)\n0.866 + 0.5im\n\n\n\n\n\n"},{"title":"Base.cos","page":"Mathematics","location":"base/math.html#Base.cos-Tuple{Number}","category":"method","text":"cos(x::T) where {T <: Number} -> float(T)\n\nCompute cosine of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also cosd, cospi, sincos, cis.\n\n\n\n\n\n"},{"title":"Base.Math.sincos","page":"Mathematics","location":"base/math.html#Base.Math.sincos-Tuple{Float64}","category":"method","text":"sincos(x::T) where T -> Tuple{float(T),float(T)}\n\nSimultaneously compute the sine and cosine of x, where x is in radians, returning a tuple (sine, cosine).\n\nThrow a DomainError if isinf(x), return a (T(NaN), T(NaN)) if isnan(x).\n\nSee also cis, sincospi, sincosd.\n\n\n\n\n\n"},{"title":"Base.tan","page":"Mathematics","location":"base/math.html#Base.tan-Tuple{Number}","category":"method","text":"tan(x::T) where {T <: Number} -> float(T)\n\nCompute tangent of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also tanh.\n\n\n\n\n\n"},{"title":"Base.Math.sind","page":"Mathematics","location":"base/math.html#Base.Math.sind","category":"function","text":"sind(x::T) where T -> float(T)\n\nCompute sine of x, where x is in degrees. If x is a matrix, x needs to be a square matrix.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.cosd","page":"Mathematics","location":"base/math.html#Base.Math.cosd","category":"function","text":"cosd(x::T) where T -> float(T)\n\nCompute cosine of x, where x is in degrees. If x is a matrix, x needs to be a square matrix.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.tand","page":"Mathematics","location":"base/math.html#Base.Math.tand","category":"function","text":"tand(x::T) where T -> float(T)\n\nCompute tangent of x, where x is in degrees. If x is a matrix, x needs to be a square matrix.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.sincosd","page":"Mathematics","location":"base/math.html#Base.Math.sincosd","category":"function","text":"sincosd(x::T) where T -> Tuple{float(T),float(T)}\n\nSimultaneously compute the sine and cosine of x, where x is in degrees, returning a tuple (sine, cosine).\n\nThrow a DomainError if isinf(x), return a (T(NaN), T(NaN)) tuple if isnan(x).\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n"},{"title":"Base.Math.sinpi","page":"Mathematics","location":"base/math.html#Base.Math.sinpi","category":"function","text":"sinpi(x::T) where T -> float(T)\n\nCompute sin(pi x) more accurately than sin(pi*x), especially for large x.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also sind, cospi, sincospi.\n\n\n\n\n\n"},{"title":"Base.Math.cospi","page":"Mathematics","location":"base/math.html#Base.Math.cospi","category":"function","text":"cospi(x::T) where T -> float(T)\n\nCompute cos(pi x) more accurately than cos(pi*x), especially for large x.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also: cispi, sincosd, sinpi.\n\n\n\n\n\n"},{"title":"Base.Math.tanpi","page":"Mathematics","location":"base/math.html#Base.Math.tanpi","category":"function","text":"tanpi(x::T) where T -> float(T)\n\nCompute tan(pi x) more accurately than tan(pi*x), especially for large x.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\nSee also tand, sinpi, cospi, sincospi.\n\n\n\n\n\n"},{"title":"Base.Math.sincospi","page":"Mathematics","location":"base/math.html#Base.Math.sincospi","category":"function","text":"sincospi(x::T) where T -> Tuple{float(T),float(T)}\n\nSimultaneously compute sinpi(x) and cospi(x) (the sine and cosine of π*x, where x is in radians), returning a tuple (sine, cosine).\n\nThrow a DomainError if isinf(x), return a (T(NaN), T(NaN)) tuple if isnan(x).\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\nSee also: cispi, sincosd, sinpi.\n\n\n\n\n\n"},{"title":"Base.sinh","page":"Mathematics","location":"base/math.html#Base.sinh-Tuple{Number}","category":"method","text":"sinh(x)\n\nCompute hyperbolic sine of x.\n\nSee also sin.\n\n\n\n\n\n"},{"title":"Base.cosh","page":"Mathematics","location":"base/math.html#Base.cosh-Tuple{Number}","category":"method","text":"cosh(x)\n\nCompute hyperbolic cosine of x.\n\nSee also cos.\n\n\n\n\n\n"},{"title":"Base.tanh","page":"Mathematics","location":"base/math.html#Base.tanh-Tuple{Number}","category":"method","text":"tanh(x)\n\nCompute hyperbolic tangent of x.\n\nSee also tan, atanh.\n\nExamples\n\njulia> tanh.(-3:3f0)  # Here 3f0 isa Float32\n7-element Vector{Float32}:\n -0.9950548\n -0.9640276\n -0.7615942\n  0.0\n  0.7615942\n  0.9640276\n  0.9950548\n\njulia> tan.(im .* (1:3))\n3-element Vector{ComplexF64}:\n 0.0 + 0.7615941559557649im\n 0.0 + 0.9640275800758169im\n 0.0 + 0.9950547536867306im\n\n\n\n\n\n"},{"title":"Base.asin","page":"Mathematics","location":"base/math.html#Base.asin-Tuple{Number}","category":"method","text":"asin(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse sine of x, where the output is in radians.\n\nReturn a T(NaN) if isnan(x).\n\nSee also asind for output in degrees.\n\nExamples\n\njulia> asin.((0, 1/2, 1))\n(0.0, 0.5235987755982989, 1.5707963267948966)\n\njulia> asind.((0, 1/2, 1))\n(0.0, 30.000000000000004, 90.0)\n\n\n\n\n\n"},{"title":"Base.acos","page":"Mathematics","location":"base/math.html#Base.acos-Tuple{Number}","category":"method","text":"acos(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse cosine of x, where the output is in radians.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.atan","page":"Mathematics","location":"base/math.html#Base.atan-Tuple{Number}","category":"method","text":"atan(y)\natan(y, x)\n\nCompute the inverse tangent of y or y/x, respectively.\n\nFor one real argument, this is the angle in radians between the positive x-axis and the point (1, y), returning a value in the interval -pi2 pi2.\n\nFor two arguments, this is the angle in radians between the positive x-axis and the point (x, y), returning a value in the interval -pi pi. This corresponds to a standard atan2 function. Note that by convention atan(0.0,x) is defined as pi and atan(-0.0,x) is defined as -pi when x < 0.\n\nSee also atand for degrees.\n\nExamples\n\njulia> rad2deg(atan(-1/√3))\n-30.000000000000004\n\njulia> rad2deg(atan(-1, √3))\n-30.000000000000004\n\njulia> rad2deg(atan(1, -√3))\n150.0\n\n\n\n\n\n"},{"title":"Base.Math.asind","page":"Mathematics","location":"base/math.html#Base.Math.asind","category":"function","text":"asind(x)\n\nCompute the inverse sine of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.acosd","page":"Mathematics","location":"base/math.html#Base.Math.acosd","category":"function","text":"acosd(x)\n\nCompute the inverse cosine of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.atand","page":"Mathematics","location":"base/math.html#Base.Math.atand","category":"function","text":"atand(y::T) where T -> float(T)\natand(y::T, x::S) where {T,S} -> promote_type(T,S)\natand(y::AbstractMatrix{T}) where T -> AbstractMatrix{Complex{float(T)}}\n\nCompute the inverse tangent of y or y/x, respectively, where the output is in degrees.\n\nReturn a NaN if isnan(y) or isnan(x). The returned NaN is either a T in the single argument version, or a promote_type(T,S) in the two argument version.\n\ncompat: Julia 1.7\nThe one-argument method supports square matrix arguments as of Julia 1.7.\n\n\n\n\n\n"},{"title":"Base.Math.sec","page":"Mathematics","location":"base/math.html#Base.Math.sec-Tuple{Number}","category":"method","text":"sec(x::T) where {T <: Number} -> float(T)\n\nCompute the secant of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.csc","page":"Mathematics","location":"base/math.html#Base.Math.csc-Tuple{Number}","category":"method","text":"csc(x::T) where {T <: Number} -> float(T)\n\nCompute the cosecant of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.cot","page":"Mathematics","location":"base/math.html#Base.Math.cot-Tuple{Number}","category":"method","text":"cot(x::T) where {T <: Number} -> float(T)\n\nCompute the cotangent of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.secd","page":"Mathematics","location":"base/math.html#Base.Math.secd","category":"function","text":"secd(x::T) where {T <: Number} -> float(T)\n\nCompute the secant of x, where x is in degrees.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.cscd","page":"Mathematics","location":"base/math.html#Base.Math.cscd","category":"function","text":"cscd(x::T) where {T <: Number} -> float(T)\n\nCompute the cosecant of x, where x is in degrees.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.cotd","page":"Mathematics","location":"base/math.html#Base.Math.cotd","category":"function","text":"cotd(x::T) where {T <: Number} -> float(T)\n\nCompute the cotangent of x, where x is in degrees.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.asec","page":"Mathematics","location":"base/math.html#Base.Math.asec-Tuple{Number}","category":"method","text":"asec(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse secant of x, where the output is in radians.\n\n\n\n\n\n"},{"title":"Base.Math.acsc","page":"Mathematics","location":"base/math.html#Base.Math.acsc-Tuple{Number}","category":"method","text":"acsc(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse cosecant of x, where the output is in radians.\n\n\n\n\n\n"},{"title":"Base.Math.acot","page":"Mathematics","location":"base/math.html#Base.Math.acot-Tuple{Number}","category":"method","text":"acot(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse cotangent of x, where the output is in radians.\n\n\n\n\n\n"},{"title":"Base.Math.asecd","page":"Mathematics","location":"base/math.html#Base.Math.asecd","category":"function","text":"asecd(x)\n\nCompute the inverse secant of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.acscd","page":"Mathematics","location":"base/math.html#Base.Math.acscd","category":"function","text":"acscd(x)\n\nCompute the inverse cosecant of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.acotd","page":"Mathematics","location":"base/math.html#Base.Math.acotd","category":"function","text":"acotd(x)\n\nCompute the inverse cotangent of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.sech","page":"Mathematics","location":"base/math.html#Base.Math.sech-Tuple{Number}","category":"method","text":"sech(x::T) where {T <: Number} -> float(T)\n\nCompute the hyperbolic secant of x.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.csch","page":"Mathematics","location":"base/math.html#Base.Math.csch-Tuple{Number}","category":"method","text":"csch(x::T) where {T <: Number} -> float(T)\n\nCompute the hyperbolic cosecant of x.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.coth","page":"Mathematics","location":"base/math.html#Base.Math.coth-Tuple{Number}","category":"method","text":"coth(x::T) where {T <: Number} -> float(T)\n\nCompute the hyperbolic cotangent of x.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.asinh","page":"Mathematics","location":"base/math.html#Base.asinh-Tuple{Number}","category":"method","text":"asinh(x)\n\nCompute the inverse hyperbolic sine of x.\n\n\n\n\n\n"},{"title":"Base.acosh","page":"Mathematics","location":"base/math.html#Base.acosh-Tuple{Number}","category":"method","text":"acosh(x)\n\nCompute the inverse hyperbolic cosine of x.\n\n\n\n\n\n"},{"title":"Base.atanh","page":"Mathematics","location":"base/math.html#Base.atanh-Tuple{Number}","category":"method","text":"atanh(x)\n\nCompute the inverse hyperbolic tangent of x.\n\n\n\n\n\n"},{"title":"Base.Math.asech","page":"Mathematics","location":"base/math.html#Base.Math.asech-Tuple{Number}","category":"method","text":"asech(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse hyperbolic secant of x.\n\n\n\n\n\n"},{"title":"Base.Math.acsch","page":"Mathematics","location":"base/math.html#Base.Math.acsch-Tuple{Number}","category":"method","text":"acsch(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse hyperbolic cosecant of x.\n\n\n\n\n\n"},{"title":"Base.Math.acoth","page":"Mathematics","location":"base/math.html#Base.Math.acoth-Tuple{Number}","category":"method","text":"acoth(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse hyperbolic cotangent of x.\n\n\n\n\n\n"},{"title":"Base.Math.sinc","page":"Mathematics","location":"base/math.html#Base.Math.sinc","category":"function","text":"sinc(x::T) where {T <: Number} -> float(T)\n\nCompute normalized sinc function operatornamesinc(x) = sin(pi x)  (pi x) if x neq 0, and 1 if x = 0.\n\nReturn a T(NaN) if isnan(x).\n\nSee also cosc, its derivative.\n\n\n\n\n\n"},{"title":"Base.Math.cosc","page":"Mathematics","location":"base/math.html#Base.Math.cosc","category":"function","text":"cosc(x::T) where {T <: Number} -> float(T)\n\nCompute cos(pi x)  x - sin(pi x)  (pi x^2) if x neq 0, and 0 if x = 0. This is the derivative of sinc(x).\n\nReturn a T(NaN) if isnan(x).\n\nSee also sinc.\n\n\n\n\n\n"},{"title":"Base.Math.deg2rad","page":"Mathematics","location":"base/math.html#Base.Math.deg2rad","category":"function","text":"deg2rad(x)\n\nConvert x from degrees to radians.\n\nSee also rad2deg, sind, pi.\n\nExamples\n\njulia> deg2rad(90)\n1.5707963267948966\n\n\n\n\n\n"},{"title":"Base.Math.rad2deg","page":"Mathematics","location":"base/math.html#Base.Math.rad2deg","category":"function","text":"rad2deg(x)\n\nConvert x from radians to degrees.\n\nSee also deg2rad.\n\nExamples\n\njulia> rad2deg(pi)\n180.0\n\n\n\n\n\n"},{"title":"Base.Math.hypot","page":"Mathematics","location":"base/math.html#Base.Math.hypot","category":"function","text":"hypot(x, y)\n\nCompute the hypotenuse sqrtx^2+y^2 avoiding overflow and underflow.\n\nThis code is an implementation of the algorithm described in: An Improved Algorithm for hypot(a,b) by Carlos F. Borges The article is available online at arXiv at the link   https://arxiv.org/abs/1904.09481\n\nhypot(x...)\n\nCompute the hypotenuse sqrtsum x_i^2 avoiding overflow and underflow.\n\nSee also norm in the LinearAlgebra standard library.\n\nExamples\n\njulia> a = Int64(10)^10;\n\njulia> hypot(a, a)\n1.4142135623730951e10\n\njulia> √(a^2 + a^2) # a^2 overflows\nERROR: DomainError with -2.914184810805068e18:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\njulia> hypot(3, 4im)\n5.0\n\njulia> hypot(-5.7)\n5.7\n\njulia> hypot(3, 4im, 12.0)\n13.0\n\njulia> using LinearAlgebra\n\njulia> norm([a, a, a, a]) == hypot(a, a, a, a)\ntrue\n\n\n\n\n\n"},{"title":"Base.log","page":"Mathematics","location":"base/math.html#Base.log-Tuple{Number}","category":"method","text":"log(x)\n\nCompute the natural logarithm of x.\n\nThrow a DomainError for negative Real arguments. Use Complex arguments to obtain Complex results.\n\nnote: Branch cut\nlog has a branch cut along the negative real axis; -0.0im is taken to be below the axis.\n\nSee also ℯ, log1p, log2, log10.\n\nExamples\n\njulia> log(2)\n0.6931471805599453\n\njulia> log(-3)\nERROR: DomainError with -3.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\njulia> log(-3 + 0im)\n1.0986122886681098 + 3.141592653589793im\n\njulia> log(-3 - 0.0im)\n1.0986122886681098 - 3.141592653589793im\n\njulia> log.(exp.(-1:1))\n3-element Vector{Float64}:\n -1.0\n  0.0\n  1.0\n\n\n\n\n\n"},{"title":"Base.log","page":"Mathematics","location":"base/math.html#Base.log-Tuple{Number, Number}","category":"method","text":"log(b,x)\n\nCompute the base b logarithm of x. Throw a DomainError for negative Real arguments.\n\nExamples\n\njulia> log(4,8)\n1.5\n\njulia> log(4,2)\n0.5\n\njulia> log(-2, 3)\nERROR: DomainError with -2.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\njulia> log(2, -3)\nERROR: DomainError with -3.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\nnote: Note\nIf b is a power of 2 or 10, log2 or log10 should be used, as these will typically be faster and more accurate. For example,julia> log(100,1000000)\n2.9999999999999996\n\njulia> log10(1000000)/2\n3.0\n\n\n\n\n\n"},{"title":"Base.log2","page":"Mathematics","location":"base/math.html#Base.log2","category":"function","text":"log2(x)\n\nCompute the logarithm of x to base 2. Throw a DomainError for negative Real arguments.\n\nSee also: exp2, ldexp, ispow2.\n\nExamples\n\njulia> log2(4)\n2.0\n\njulia> log2(10)\n3.321928094887362\n\njulia> log2(-2)\nERROR: DomainError with -2.0:\nlog2 was called with a negative real argument but will only return a complex result if called with a complex argument. Try log2(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(f::Symbol, x::Float64) at ./math.jl:31\n[...]\n\njulia> log2.(2.0 .^ (-1:1))\n3-element Vector{Float64}:\n -1.0\n  0.0\n  1.0\n\n\n\n\n\n"},{"title":"Base.log10","page":"Mathematics","location":"base/math.html#Base.log10","category":"function","text":"log10(x)\n\nCompute the logarithm of x to base 10. Throw a DomainError for negative Real arguments.\n\nExamples\n\njulia> log10(100)\n2.0\n\njulia> log10(2)\n0.3010299956639812\n\njulia> log10(-2)\nERROR: DomainError with -2.0:\nlog10 was called with a negative real argument but will only return a complex result if called with a complex argument. Try log10(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(f::Symbol, x::Float64) at ./math.jl:31\n[...]\n\n\n\n\n\n"},{"title":"Base.log1p","page":"Mathematics","location":"base/math.html#Base.log1p","category":"function","text":"log1p(x)\n\nAccurate natural logarithm of 1+x. Throw a DomainError for Real arguments less than -1.\n\nExamples\n\njulia> log1p(-0.5)\n-0.6931471805599453\n\njulia> log1p(0)\n0.0\n\njulia> log1p(-2)\nERROR: DomainError with -2.0:\nlog1p was called with a real argument < -1 but will only return a complex result if called with a complex argument. Try log1p(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\n\n\n\n\n"},{"title":"Base.Math.frexp","page":"Mathematics","location":"base/math.html#Base.Math.frexp","category":"function","text":"frexp(val)\n\nReturn (x,exp) such that x has a magnitude in the interval 12 1) or 0, and val is equal to x times 2^exp.\n\nSee also significand, exponent, ldexp.\n\nExamples\n\njulia> frexp(6.0)\n(0.75, 3)\n\njulia> significand(6.0), exponent(6.0)  # interval [1, 2) instead\n(1.5, 2)\n\njulia> frexp(0.0), frexp(NaN), frexp(-Inf)  # exponent would give an error\n((0.0, 0), (NaN, 0), (-Inf, 0))\n\n\n\n\n\n"},{"title":"Base.exp","page":"Mathematics","location":"base/math.html#Base.exp-Tuple{Float64}","category":"method","text":"exp(x)\n\nCompute the natural base exponential of x, in other words ℯ^x.\n\nSee also exp2, exp10 and cis.\n\nExamples\n\njulia> exp(1.0)\n2.718281828459045\n\njulia> exp(im * pi) ≈ cis(pi)\ntrue\n\n\n\n\n\n"},{"title":"Base.exp2","page":"Mathematics","location":"base/math.html#Base.exp2","category":"function","text":"exp2(x)\n\nCompute the base 2 exponential of x, in other words 2^x.\n\nSee also ldexp, <<.\n\nExamples\n\njulia> exp2(5)\n32.0\n\njulia> 2^5\n32\n\njulia> exp2(63) > typemax(Int)\ntrue\n\n\n\n\n\n"},{"title":"Base.exp10","page":"Mathematics","location":"base/math.html#Base.exp10","category":"function","text":"exp10(x)\n\nCompute the base 10 exponential of x, in other words 10^x.\n\nExamples\n\njulia> exp10(2)\n100.0\n\njulia> 10^2\n100\n\n\n\n\n\n"},{"title":"Base.Math.ldexp","page":"Mathematics","location":"base/math.html#Base.Math.ldexp","category":"function","text":"ldexp(x, n)\n\nCompute x times 2^n.\n\nSee also frexp, exponent.\n\nExamples\n\njulia> ldexp(5.0, 2)\n20.0\n\n\n\n\n\n"},{"title":"Base.Math.modf","page":"Mathematics","location":"base/math.html#Base.Math.modf","category":"function","text":"modf(x)\n\nReturn a tuple (fpart, ipart) of the fractional and integral parts of a number. Both parts have the same sign as the argument.\n\nExamples\n\njulia> modf(3.5)\n(0.5, 3.0)\n\njulia> modf(-3.5)\n(-0.5, -3.0)\n\n\n\n\n\n"},{"title":"Base.expm1","page":"Mathematics","location":"base/math.html#Base.expm1","category":"function","text":"expm1(x)\n\nAccurately compute e^x-1. It avoids the loss of precision involved in the direct evaluation of exp(x) - 1 for small values of x.\n\nExamples\n\njulia> expm1(1e-16)\n1.0e-16\n\njulia> exp(1e-16) - 1\n0.0\n\n\n\n\n\n"},{"title":"Base.round","page":"Mathematics","location":"base/math.html#Base.round","category":"function","text":"round([T,] x, [r::RoundingMode])\nround(x, [r::RoundingMode]; digits::Integer=0, base = 10)\nround(x, [r::RoundingMode]; sigdigits::Integer, base = 10)\n\nRounds the number x.\n\nWithout keyword arguments, x is rounded to an integer value, returning a value of type T, or of the same type of x if no T is provided. An InexactError will be thrown if the value is not representable by T, similar to convert.\n\nIf the digits keyword argument is provided, it rounds to the specified number of digits after the decimal place (or before if digits is negative), in base base.\n\nIf the sigdigits keyword argument is provided, it rounds to the specified number of significant digits, in base base.\n\nThe RoundingMode r controls the direction of the rounding; the default is RoundNearest, which rounds to the nearest integer, with ties (fractional values of 0.5) being rounded to the nearest even integer. Note that round may give incorrect results if the global rounding mode is changed (see rounding).\n\nWhen rounding to a floating point type, will round to integers representable by that type (and Inf) rather than true integers. Inf is treated as one ulp greater than the floatmax(T) for purposes of determining \"nearest\", similar to convert.\n\nExamples\n\njulia> round(1.7)\n2.0\n\njulia> round(Int, 1.7)\n2\n\njulia> round(1.5)\n2.0\n\njulia> round(2.5)\n2.0\n\njulia> round(pi; digits=2)\n3.14\n\njulia> round(pi; digits=3, base=2)\n3.125\n\njulia> round(123.456; sigdigits=2)\n120.0\n\njulia> round(357.913; sigdigits=4, base=2)\n352.0\n\njulia> round(Float16, typemax(UInt128))\nInf16\n\njulia> floor(Float16, typemax(UInt128))\nFloat16(6.55e4)\n\nnote: Note\nRounding to specified digits in bases other than 2 can be inexact when operating on binary floating point numbers. For example, the Float64 value represented by 1.15 is actually less than 1.15, yet will be rounded to 1.2. For example:julia> x = 1.15\n1.15\n\njulia> big(1.15)\n1.149999999999999911182158029987476766109466552734375\n\njulia> x < 115//100\ntrue\n\njulia> round(x, digits=1)\n1.2\n\nExtensions\n\nTo extend round to new numeric types, it is typically sufficient to define Base.round(x::NewType, r::RoundingMode).\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundingMode","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundingMode","category":"type","text":"RoundingMode\n\nA type used for controlling the rounding mode of floating point operations (via rounding/setrounding functions), or as optional arguments for rounding to the nearest integer (via the round function).\n\nCurrently supported rounding modes are:\n\nRoundNearest (default)\nRoundNearestTiesAway\nRoundNearestTiesUp\nRoundToZero\nRoundFromZero\nRoundUp\nRoundDown\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9. Prior versions support RoundFromZero for BigFloats only.\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundNearest","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundNearest","category":"constant","text":"RoundNearest\n\nThe default rounding mode. Rounds to the nearest integer, with ties (fractional values of 0.5) being rounded to the nearest even integer.\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundNearestTiesAway","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundNearestTiesAway","category":"constant","text":"RoundNearestTiesAway\n\nRounds to nearest integer, with ties rounded away from zero (C/C++ round behaviour).\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundNearestTiesUp","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundNearestTiesUp","category":"constant","text":"RoundNearestTiesUp\n\nRounds to nearest integer, with ties rounded toward positive infinity (Java/JavaScript round behaviour).\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundToZero","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundToZero","category":"constant","text":"RoundToZero\n\nround using this rounding mode is an alias for trunc.\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundFromZero","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundFromZero","category":"constant","text":"RoundFromZero\n\nRounds away from zero.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9. Prior versions support RoundFromZero for BigFloats only.\n\nExamples\n\njulia> BigFloat(\"1.0000000000000001\", 5, RoundFromZero)\n1.06\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundUp","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundUp","category":"constant","text":"RoundUp\n\nround using this rounding mode is an alias for ceil.\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundDown","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundDown","category":"constant","text":"RoundDown\n\nround using this rounding mode is an alias for floor.\n\n\n\n\n\n"},{"title":"Base.round","page":"Mathematics","location":"base/math.html#Base.round-Tuple{Complex{<:AbstractFloat}, RoundingMode, RoundingMode}","category":"method","text":"round(z::Complex[, RoundingModeReal, [RoundingModeImaginary]])\nround(z::Complex[, RoundingModeReal, [RoundingModeImaginary]]; digits=0, base=10)\nround(z::Complex[, RoundingModeReal, [RoundingModeImaginary]]; sigdigits, base=10)\n\nReturn the nearest integral value of the same type as the complex-valued z to z, breaking ties using the specified RoundingModes. The first RoundingMode is used for rounding the real components while the second is used for rounding the imaginary components.\n\nRoundingModeReal and RoundingModeImaginary default to RoundNearest, which rounds to the nearest integer, with ties (fractional values of 0.5) being rounded to the nearest even integer.\n\nExamples\n\njulia> round(3.14 + 4.5im)\n3.0 + 4.0im\n\njulia> round(3.14 + 4.5im, RoundUp, RoundNearestTiesUp)\n4.0 + 5.0im\n\njulia> round(3.14159 + 4.512im; digits = 1)\n3.1 + 4.5im\n\njulia> round(3.14159 + 4.512im; sigdigits = 3)\n3.14 + 4.51im\n\n\n\n\n\n"},{"title":"Base.ceil","page":"Mathematics","location":"base/math.html#Base.ceil","category":"function","text":"ceil([T,] x)\nceil(x; digits::Integer= [, base = 10])\nceil(x; sigdigits::Integer= [, base = 10])\n\nceil(x) returns the nearest integral value of the same type as x that is greater than or equal to x.\n\nceil(T, x) converts the result to type T, throwing an InexactError if the ceiled value is not representable as a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support ceil for a new type, define Base.round(x::NewType, ::RoundingMode{:Up}).\n\n\n\n\n\n"},{"title":"Base.floor","page":"Mathematics","location":"base/math.html#Base.floor","category":"function","text":"floor([T,] x)\nfloor(x; digits::Integer= [, base = 10])\nfloor(x; sigdigits::Integer= [, base = 10])\n\nfloor(x) returns the nearest integral value of the same type as x that is less than or equal to x.\n\nfloor(T, x) converts the result to type T, throwing an InexactError if the floored value is not representable a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support floor for a new type, define Base.round(x::NewType, ::RoundingMode{:Down}).\n\n\n\n\n\n"},{"title":"Base.trunc","page":"Mathematics","location":"base/math.html#Base.trunc","category":"function","text":"trunc([T,] x)\ntrunc(x; digits::Integer= [, base = 10])\ntrunc(x; sigdigits::Integer= [, base = 10])\n\ntrunc(x) returns the nearest integral value of the same type as x whose absolute value is less than or equal to the absolute value of x.\n\ntrunc(T, x) converts the result to type T, throwing an InexactError if the truncated value is not representable a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support trunc for a new type, define Base.round(x::NewType, ::RoundingMode{:ToZero}).\n\nSee also: %, floor, unsigned, unsafe_trunc.\n\nExamples\n\njulia> trunc(2.22)\n2.0\n\njulia> trunc(-2.22, digits=1)\n-2.2\n\njulia> trunc(Int, -2.22)\n-2\n\n\n\n\n\n"},{"title":"Base.unsafe_trunc","page":"Mathematics","location":"base/math.html#Base.unsafe_trunc","category":"function","text":"unsafe_trunc(T, x)\n\nReturn the nearest integral value of type T whose absolute value is less than or equal to the absolute value of x. If the value is not representable by T, an arbitrary value will be returned. See also trunc.\n\nExamples\n\njulia> unsafe_trunc(Int, -2.2)\n-2\n\njulia> unsafe_trunc(Int, NaN) isa Int\ntrue\n\n\n\n\n\n"},{"title":"Base.min","page":"Mathematics","location":"base/math.html#Base.min","category":"function","text":"min(x, y, ...)\n\nReturn the minimum of the arguments, with respect to isless. If any of the arguments is missing, return missing. See also the minimum function to take the minimum element from a collection.\n\nExamples\n\njulia> min(2, 5, 1)\n1\n\njulia> min(4, missing, 6)\nmissing\n\n\n\n\n\n"},{"title":"Base.max","page":"Mathematics","location":"base/math.html#Base.max","category":"function","text":"max(x, y, ...)\n\nReturn the maximum of the arguments, with respect to isless. If any of the arguments is missing, return missing. See also the maximum function to take the maximum element from a collection.\n\nExamples\n\njulia> max(2, 5, 1)\n5\n\njulia> max(5, missing, 6)\nmissing\n\n\n\n\n\n"},{"title":"Base.minmax","page":"Mathematics","location":"base/math.html#Base.minmax","category":"function","text":"minmax(x, y)\n\nReturn (min(x,y), max(x,y)).\n\nSee also extrema that returns (minimum(x), maximum(x)).\n\nExamples\n\njulia> minmax('c','b')\n('b', 'c')\n\n\n\n\n\n"},{"title":"Base.clamp","page":"Mathematics","location":"base/math.html#Base.clamp","category":"function","text":"clamp(x::Integer, r::AbstractUnitRange)\n\nClamp x to lie within range r.\n\ncompat: Julia 1.6\nThis method requires at least Julia 1.6.\n\n\n\n\n\nclamp(x, T)::T\n\nClamp x between typemin(T) and typemax(T) and convert the result to type T.\n\nSee also trunc.\n\nExamples\n\njulia> clamp(200, Int8)\n127\n\njulia> clamp(-200, Int8)\n-128\n\njulia> trunc(Int, 4pi^2)\n39\n\n\n\n\n\nclamp(x, lo, hi)\n\nReturn x if lo <= x <= hi. If x > hi, return hi. If x < lo, return lo. Arguments are promoted to a common type.\n\nSee also clamp!, min, max.\n\ncompat: Julia 1.3\nmissing as the first argument requires at least Julia 1.3.\n\nExamples\n\njulia> clamp.([pi, 1.0, big(10)], 2.0, 9.0)\n3-element Vector{BigFloat}:\n 3.141592653589793238462643383279502884197169399375105820974944592307816406286198\n 2.0\n 9.0\n\njulia> clamp.([11, 8, 5], 10, 6)  # an example where lo > hi\n3-element Vector{Int64}:\n  6\n  6\n 10\n\n\n\n\n\n"},{"title":"Base.clamp!","page":"Mathematics","location":"base/math.html#Base.clamp!","category":"function","text":"clamp!(array::AbstractArray, lo, hi)\n\nRestrict values in array to the specified range, in-place. See also clamp.\n\ncompat: Julia 1.3\nmissing entries in array require at least Julia 1.3.\n\nExamples\n\njulia> row = collect(-4:4)';\n\njulia> clamp!(row, 0, Inf)\n1×9 adjoint(::Vector{Int64}) with eltype Int64:\n 0  0  0  0  0  1  2  3  4\n\njulia> clamp.((-4:4)', 0, Inf)\n1×9 Matrix{Float64}:\n 0.0  0.0  0.0  0.0  0.0  1.0  2.0  3.0  4.0\n\n\n\n\n\n"},{"title":"Base.abs","page":"Mathematics","location":"base/math.html#Base.abs","category":"function","text":"abs(x)\n\nThe absolute value of x.\n\nWhen abs is applied to signed integers, overflow may occur, resulting in the return of a negative value. This overflow occurs only when abs is applied to the minimum representable value of a signed integer. That is, when x == typemin(typeof(x)), abs(x) == x < 0, not -x as might be expected.\n\nSee also: abs2, unsigned, sign.\n\nExamples\n\njulia> abs(-3)\n3\n\njulia> abs(1 + im)\n1.4142135623730951\n\njulia> abs.(Int8[-128 -127 -126 0 126 127])  # overflow at typemin(Int8)\n1×6 Matrix{Int8}:\n -128  127  126  0  126  127\n\njulia> maximum(abs, [1, -2, 3, -4])\n4\n\n\n\n\n\n"},{"title":"Base.Checked","page":"Mathematics","location":"base/math.html#Base.Checked","category":"module","text":"Checked\n\nThe Checked module provides arithmetic functions for the built-in signed and unsigned Integer types which throw an error when an overflow occurs. They are named like checked_sub, checked_div, etc. In addition, add_with_overflow, sub_with_overflow, mul_with_overflow return both the unchecked results and a boolean value denoting the presence of an overflow.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_abs","page":"Mathematics","location":"base/math.html#Base.Checked.checked_abs","category":"function","text":"Base.checked_abs(x)\n\nCalculates abs(x), checking for overflow errors where applicable. For example, standard two's complement signed integers (e.g. Int) cannot represent abs(typemin(Int)), thus leading to an overflow.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_neg","page":"Mathematics","location":"base/math.html#Base.Checked.checked_neg","category":"function","text":"Base.checked_neg(x)\n\nCalculates -x, checking for overflow errors where applicable. For example, standard two's complement signed integers (e.g. Int) cannot represent -typemin(Int), thus leading to an overflow.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_add","page":"Mathematics","location":"base/math.html#Base.Checked.checked_add","category":"function","text":"Base.checked_add(x, y)\n\nCalculates x+y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_sub","page":"Mathematics","location":"base/math.html#Base.Checked.checked_sub","category":"function","text":"Base.checked_sub(x, y)\n\nCalculates x-y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_mul","page":"Mathematics","location":"base/math.html#Base.Checked.checked_mul","category":"function","text":"Base.checked_mul(x, y)\n\nCalculates x*y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_div","page":"Mathematics","location":"base/math.html#Base.Checked.checked_div","category":"function","text":"Base.checked_div(x, y)\n\nCalculates div(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_rem","page":"Mathematics","location":"base/math.html#Base.Checked.checked_rem","category":"function","text":"Base.checked_rem(x, y)\n\nCalculates x%y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_fld","page":"Mathematics","location":"base/math.html#Base.Checked.checked_fld","category":"function","text":"Base.checked_fld(x, y)\n\nCalculates fld(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_mod","page":"Mathematics","location":"base/math.html#Base.Checked.checked_mod","category":"function","text":"Base.checked_mod(x, y)\n\nCalculates mod(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_cld","page":"Mathematics","location":"base/math.html#Base.Checked.checked_cld","category":"function","text":"Base.checked_cld(x, y)\n\nCalculates cld(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_pow","page":"Mathematics","location":"base/math.html#Base.Checked.checked_pow","category":"function","text":"Base.checked_pow(x, y)\n\nCalculates ^(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.add_with_overflow","page":"Mathematics","location":"base/math.html#Base.Checked.add_with_overflow","category":"function","text":"Base.add_with_overflow(x, y) -> (r, f)\n\nCalculates r = x+y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n"},{"title":"Base.Checked.sub_with_overflow","page":"Mathematics","location":"base/math.html#Base.Checked.sub_with_overflow","category":"function","text":"Base.sub_with_overflow(x, y) -> (r, f)\n\nCalculates r = x-y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n"},{"title":"Base.Checked.mul_with_overflow","page":"Mathematics","location":"base/math.html#Base.Checked.mul_with_overflow","category":"function","text":"Base.mul_with_overflow(x, y) -> (r, f)\n\nCalculates r = x*y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n"},{"title":"Base.abs2","page":"Mathematics","location":"base/math.html#Base.abs2","category":"function","text":"abs2(x)\n\nSquared absolute value of x.\n\nThis can be faster than abs(x)^2, especially for complex numbers where abs(x) requires a square root via hypot.\n\nSee also abs, conj, real.\n\nExamples\n\njulia> abs2(-3)\n9\n\njulia> abs2(3.0 + 4.0im)\n25.0\n\njulia> sum(abs2, [1+2im, 3+4im])  # LinearAlgebra.norm(x)^2\n30\n\n\n\n\n\n"},{"title":"Base.copysign","page":"Mathematics","location":"base/math.html#Base.copysign","category":"function","text":"copysign(x, y) -> z\n\nReturn z which has the magnitude of x and the same sign as y.\n\nExamples\n\njulia> copysign(1, -2)\n-1\n\njulia> copysign(-1, 2)\n1\n\n\n\n\n\n"},{"title":"Base.sign","page":"Mathematics","location":"base/math.html#Base.sign","category":"function","text":"sign(x)\n\nReturn zero if x==0 and xx otherwise (i.e., ±1 for real x).\n\nSee also signbit, zero, copysign, flipsign.\n\nExamples\n\njulia> sign(-4.0)\n-1.0\n\njulia> sign(99)\n1\n\njulia> sign(-0.0)\n-0.0\n\njulia> sign(0 + im)\n0.0 + 1.0im\n\n\n\n\n\n"},{"title":"Base.signbit","page":"Mathematics","location":"base/math.html#Base.signbit","category":"function","text":"signbit(x)\n\nReturn true if the value of the sign of x is negative, otherwise false.\n\nSee also sign and copysign.\n\nExamples\n\njulia> signbit(-4)\ntrue\n\njulia> signbit(5)\nfalse\n\njulia> signbit(5.5)\nfalse\n\njulia> signbit(-4.1)\ntrue\n\n\n\n\n\n"},{"title":"Base.flipsign","page":"Mathematics","location":"base/math.html#Base.flipsign","category":"function","text":"flipsign(x, y)\n\nReturn x with its sign flipped if y is negative. For example abs(x) = flipsign(x,x).\n\nExamples\n\njulia> flipsign(5, 3)\n5\n\njulia> flipsign(5, -3)\n-5\n\n\n\n\n\n"},{"title":"Base.sqrt","page":"Mathematics","location":"base/math.html#Base.sqrt-Tuple{Number}","category":"method","text":"sqrt(x)\n\nReturn sqrtx.\n\nThrow a DomainError for negative Real arguments. Use Complex negative arguments instead to obtain a Complex result.\n\nThe prefix operator √ is equivalent to sqrt.\n\nnote: Branch cut\nsqrt has a branch cut along the negative real axis; -0.0im is taken to be below the axis.\n\nSee also: hypot.\n\nExamples\n\njulia> sqrt(big(81))\n9.0\n\njulia> sqrt(big(-81))\nERROR: DomainError with -81.0:\nNaN result for non-NaN input.\nStacktrace:\n [1] sqrt(::BigFloat) at ./mpfr.jl:501\n[...]\n\njulia> sqrt(big(complex(-81)))\n0.0 + 9.0im\n\njulia> sqrt(-81 - 0.0im)  # -0.0im is below the branch cut\n0.0 - 9.0im\n\njulia> .√(1:4)\n4-element Vector{Float64}:\n 1.0\n 1.4142135623730951\n 1.7320508075688772\n 2.0\n\n\n\n\n\n"},{"title":"Base.isqrt","page":"Mathematics","location":"base/math.html#Base.isqrt","category":"function","text":"isqrt(n::Integer)\n\nInteger square root: the largest integer m such that m*m <= n.\n\njulia> isqrt(5)\n2\n\n\n\n\n\n"},{"title":"Base.Math.cbrt","page":"Mathematics","location":"base/math.html#Base.Math.cbrt-Tuple{AbstractFloat}","category":"method","text":"cbrt(x::Real)\n\nReturn the cube root of x, i.e. x^13. Negative values are accepted (returning the negative real root when x  0).\n\nThe prefix operator ∛ is equivalent to cbrt.\n\nExamples\n\njulia> cbrt(big(27))\n3.0\n\njulia> cbrt(big(-27))\n-3.0\n\n\n\n\n\n"},{"title":"Base.Math.fourthroot","page":"Mathematics","location":"base/math.html#Base.Math.fourthroot-Tuple{Number}","category":"method","text":"fourthroot(x)\n\nReturn the fourth root of x by applying sqrt twice successively.\n\n\n\n\n\n"},{"title":"Base.real","page":"Mathematics","location":"base/math.html#Base.real","category":"function","text":"real(A::AbstractArray)\n\nReturn an array containing the real part of each entry in array A.\n\nEquivalent to real.(A), except that when eltype(A) <: Real A is returned without copying, and that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> real([1, 2im, 3 + 4im])\n3-element Vector{Int64}:\n 1\n 0\n 3\n\njulia> real(fill(2 - im))\n0-dimensional Array{Int64, 0}:\n2\n\n\n\n\n\nreal(T::Type)\n\nReturn the type that represents the real part of a value of type T. e.g: for T == Complex{R}, returns R. Equivalent to typeof(real(zero(T))).\n\nExamples\n\njulia> real(Complex{Int})\nInt64\n\njulia> real(Float64)\nFloat64\n\n\n\n\n\nreal(z)\n\nReturn the real part of the complex number z.\n\nSee also: imag, reim, complex, isreal, Real.\n\nExamples\n\njulia> real(1 + 3im)\n1\n\n\n\n\n\n"},{"title":"Base.imag","page":"Mathematics","location":"base/math.html#Base.imag","category":"function","text":"imag(A::AbstractArray)\n\nReturn an array containing the imaginary part of each entry in array A.\n\nEquivalent to imag.(A), except that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> imag([1, 2im, 3 + 4im])\n3-element Vector{Int64}:\n 0\n 2\n 4\n\njulia> imag(fill(2 - im))\n0-dimensional Array{Int64, 0}:\n-1\n\n\n\n\n\nimag(z)\n\nReturn the imaginary part of the complex number z.\n\nSee also: conj, reim, adjoint, angle.\n\nExamples\n\njulia> imag(1 + 3im)\n3\n\n\n\n\n\n"},{"title":"Base.reim","page":"Mathematics","location":"base/math.html#Base.reim","category":"function","text":"reim(A::AbstractArray)\n\nReturn a tuple of two arrays containing respectively the real and the imaginary part of each entry in A.\n\nEquivalent to (real.(A), imag.(A)), except that when eltype(A) <: Real A is returned without copying to represent the real part, and that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> reim([1, 2im, 3 + 4im])\n([1, 0, 3], [0, 2, 4])\n\njulia> reim(fill(2 - im))\n(fill(2), fill(-1))\n\n\n\n\n\nreim(z)\n\nReturn a tuple of the real and imaginary parts of the complex number z.\n\nExamples\n\njulia> reim(1 + 3im)\n(1, 3)\n\n\n\n\n\n"},{"title":"Base.conj","page":"Mathematics","location":"base/math.html#Base.conj","category":"function","text":"conj(A::AbstractArray)\n\nReturn an array containing the complex conjugate of each entry in array A.\n\nEquivalent to conj.(A), except that when eltype(A) <: Real A is returned without copying, and that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> conj([1, 2im, 3 + 4im])\n3-element Vector{Complex{Int64}}:\n 1 + 0im\n 0 - 2im\n 3 - 4im\n\njulia> conj(fill(2 - im))\n0-dimensional Array{Complex{Int64}, 0}:\n2 + 1im\n\n\n\n\n\nconj(z)\n\nCompute the complex conjugate of a complex number z.\n\nSee also: angle, adjoint.\n\nExamples\n\njulia> conj(1 + 3im)\n1 - 3im\n\n\n\n\n\n"},{"title":"Base.angle","page":"Mathematics","location":"base/math.html#Base.angle","category":"function","text":"angle(z)\n\nCompute the phase angle in radians of a complex number z.\n\nReturns a number -pi ≤ angle(z) ≤ pi, and is thus discontinuous along the negative real axis.\n\nSee also: atan, cis, rad2deg.\n\nExamples\n\njulia> rad2deg(angle(1 + im))\n45.0\n\njulia> rad2deg(angle(1 - im))\n-45.0\n\njulia> rad2deg(angle(-1 + 1e-20im))\n180.0\n\njulia> rad2deg(angle(-1 - 1e-20im))\n-180.0\n\n\n\n\n\n"},{"title":"Base.cis","page":"Mathematics","location":"base/math.html#Base.cis","category":"function","text":"cis(x)\n\nMore efficient method for exp(im*x) by using Euler's formula: cos(x) + i sin(x) = exp(i x).\n\nSee also cispi, sincos, exp, angle.\n\nExamples\n\njulia> cis(π) ≈ -1\ntrue\n\n\n\n\n\n"},{"title":"Base.cispi","page":"Mathematics","location":"base/math.html#Base.cispi","category":"function","text":"cispi(x)\n\nMore accurate method for cis(pi*x) (especially for large x).\n\nSee also cis, sincospi, exp, angle.\n\nExamples\n\njulia> cispi(10000)\n1.0 + 0.0im\n\njulia> cispi(0.25 + 1im)\n0.030556854645954562 + 0.03055685464595456im\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n"},{"title":"Base.binomial","page":"Mathematics","location":"base/math.html#Base.binomial","category":"function","text":"binomial(x::Number, k::Integer)\n\nThe generalized binomial coefficient, defined for k ≥ 0 by the polynomial\n\nfrac1k prod_j=0^k-1 (x - j)\n\nWhen k < 0 it returns zero.\n\nFor the case of integer x, this is equivalent to the ordinary integer binomial coefficient\n\nbinomnk = fracnk (n-k)\n\nFurther generalizations to non-integer k are mathematically possible, but involve the Gamma function and/or the beta function, which are not provided by the Julia standard library but are available in external packages such as SpecialFunctions.jl.\n\nExternal links\n\nBinomial coefficient on Wikipedia.\n\n\n\n\n\nbinomial(n::Integer, k::Integer)\n\nThe binomial coefficient binomnk, being the coefficient of the kth term in the polynomial expansion of (1+x)^n.\n\nIf n is non-negative, then it is the number of ways to choose k out of n items:\n\nbinomnk = fracnk (n-k)\n\nwhere n is the factorial function.\n\nIf n is negative, then it is defined in terms of the identity\n\nbinomnk = (-1)^k binomk-n-1k\n\nSee also factorial.\n\nExamples\n\njulia> binomial(5, 3)\n10\n\njulia> factorial(5) ÷ (factorial(5-3) * factorial(3))\n10\n\njulia> binomial(-5, 3)\n-35\n\nExternal links\n\nBinomial coefficient on Wikipedia.\n\n\n\n\n\n"},{"title":"Base.factorial","page":"Mathematics","location":"base/math.html#Base.factorial","category":"function","text":"factorial(n::Integer)\n\nFactorial of n. If n is an Integer, the factorial is computed as an integer (promoted to at least 64 bits). Note that this may overflow if n is not small, but you can use factorial(big(n)) to compute the result exactly in arbitrary precision.\n\nSee also binomial.\n\nExamples\n\njulia> factorial(6)\n720\n\njulia> factorial(21)\nERROR: OverflowError: 21 is too large to look up in the table; consider using `factorial(big(21))` instead\nStacktrace:\n[...]\n\njulia> factorial(big(21))\n51090942171709440000\n\nExternal links\n\nFactorial on Wikipedia.\n\n\n\n\n\n"},{"title":"Base.gcd","page":"Mathematics","location":"base/math.html#Base.gcd","category":"function","text":"gcd(x, y...)\n\nGreatest common (positive) divisor (or zero if all arguments are zero). The arguments may be integer and rational numbers.\n\na is a divisor of b if there exists an integer m such that ma=b.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\nExamples\n\njulia> gcd(6, 9)\n3\n\njulia> gcd(6, -9)\n3\n\njulia> gcd(6, 0)\n6\n\njulia> gcd(0, 0)\n0\n\njulia> gcd(1//3, 2//3)\n1//3\n\njulia> gcd(1//3, -2//3)\n1//3\n\njulia> gcd(1//3, 2)\n1//3\n\njulia> gcd(0, 0, 10, 15)\n5\n\n\n\n\n\n"},{"title":"Base.lcm","page":"Mathematics","location":"base/math.html#Base.lcm","category":"function","text":"lcm(x, y...)\n\nLeast common (positive) multiple (or zero if any argument is zero). The arguments may be integer and rational numbers.\n\na is a multiple of b if there exists an integer m such that a=mb.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\nExamples\n\njulia> lcm(2, 3)\n6\n\njulia> lcm(-2, 3)\n6\n\njulia> lcm(0, 3)\n0\n\njulia> lcm(0, 0)\n0\n\njulia> lcm(1//3, 2//3)\n2//3\n\njulia> lcm(1//3, -2//3)\n2//3\n\njulia> lcm(1//3, 2)\n2//1\n\njulia> lcm(1, 3, 5, 7)\n105\n\n\n\n\n\n"},{"title":"Base.gcdx","page":"Mathematics","location":"base/math.html#Base.gcdx","category":"function","text":"gcdx(a, b...)\n\nComputes the greatest common (positive) divisor of a and b and their Bézout coefficients, i.e. the integer coefficients u and v that satisfy u*a + v*b = d = gcd(a b). gcdx(a b) returns (d u v).\n\nFor more arguments than two, i.e., gcdx(a, b, c, ...) the Bézout coefficients are computed recursively, returning a solution (d, u, v, w, ...) to u*a + v*b + w*c +  = d = gcd(a b c ).\n\nThe arguments may be integer and rational numbers.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\ncompat: Julia 1.12\nMore or fewer arguments than two require Julia 1.12 or later.\n\nExamples\n\njulia> gcdx(12, 42)\n(6, -3, 1)\n\njulia> gcdx(240, 46)\n(2, -9, 47)\n\njulia> gcdx(15, 12, 20)\n(1, 7, -7, -1)\n\nnote: Note\nBézout coefficients are not uniquely defined. gcdx returns the minimal Bézout coefficients that are computed by the extended Euclidean algorithm. (Ref: D. Knuth, TAoCP, 2/e, p. 325, Algorithm X.) For signed integers, these coefficients u and v are minimal in the sense that u  bd and v  ad. Furthermore, the signs of u and v are chosen so that d is positive. For unsigned integers, the coefficients u and v might be near their typemax, and the identity then holds only via the unsigned integers' modulo arithmetic.\n\n\n\n\n\n"},{"title":"Base.ispow2","page":"Mathematics","location":"base/math.html#Base.ispow2","category":"function","text":"ispow2(n::Number)::Bool\n\nTest whether n is an integer power of two.\n\nSee also count_ones, prevpow, nextpow.\n\nExamples\n\njulia> ispow2(4)\ntrue\n\njulia> ispow2(5)\nfalse\n\njulia> ispow2(4.5)\nfalse\n\njulia> ispow2(0.25)\ntrue\n\njulia> ispow2(1//8)\ntrue\n\ncompat: Julia 1.6\nSupport for non-Integer arguments was added in Julia 1.6.\n\n\n\n\n\n"},{"title":"Base.nextpow","page":"Mathematics","location":"base/math.html#Base.nextpow","category":"function","text":"nextpow(a, x)\n\nThe smallest a^n not less than x, where n is a non-negative integer. a must be greater than 1, and x must be greater than 0.\n\nSee also prevpow.\n\nExamples\n\njulia> nextpow(2, 7)\n8\n\njulia> nextpow(2, 9)\n16\n\njulia> nextpow(5, 20)\n25\n\njulia> nextpow(4, 16)\n16\n\n\n\n\n\n"},{"title":"Base.prevpow","page":"Mathematics","location":"base/math.html#Base.prevpow","category":"function","text":"prevpow(a, x)\n\nThe largest a^n not greater than x, where n is a non-negative integer. a must be greater than 1, and x must not be less than 1.\n\nSee also nextpow, isqrt.\n\nExamples\n\njulia> prevpow(2, 7)\n4\n\njulia> prevpow(2, 9)\n8\n\njulia> prevpow(5, 20)\n5\n\njulia> prevpow(4, 16)\n16\n\n\n\n\n\n"},{"title":"Base.nextprod","page":"Mathematics","location":"base/math.html#Base.nextprod","category":"function","text":"nextprod(factors::Union{Tuple,AbstractVector}, n)\n\nNext integer greater than or equal to n that can be written as prod k_i^p_i for integers p_1, p_2, etcetera, for factors k_i in factors.\n\nExamples\n\njulia> nextprod((2, 3), 105)\n108\n\njulia> 2^2 * 3^3\n108\n\ncompat: Julia 1.6\nThe method that accepts a tuple requires Julia 1.6 or later.\n\n\n\n\n\n"},{"title":"Base.invmod","page":"Mathematics","location":"base/math.html#Base.invmod","category":"function","text":"invmod(n::Integer, T) where {T <: Base.BitInteger}\ninvmod(n::T) where {T <: Base.BitInteger}\n\nCompute the modular inverse of n in the integer ring of type T, i.e. modulo 2^N where N = 8*sizeof(T) (e.g. N = 32 for Int32). In other words, these methods satisfy the following identities:\n\nn * invmod(n) == 1\n(n * invmod(n, T)) % T == 1\n(n % T) * invmod(n, T) == 1\n\nNote that * here is modular multiplication in the integer ring, T.  This will throw an error if n is even, because then it is not relatively prime with 2^N and thus has no such inverse.\n\nSpecifying the modulus implied by an integer type as an explicit value is often inconvenient since the modulus is by definition too big to be represented by the type.\n\nThe modular inverse is computed much more efficiently than the general case using the algorithm described in https://arxiv.org/pdf/2204.04342.pdf.\n\ncompat: Julia 1.11\nThe invmod(n) and invmod(n, T) methods require Julia 1.11 or later.\n\n\n\n\n\ninvmod(n::Integer, m::Integer)\n\nTake the inverse of n modulo m: y such that n y = 1 pmod m, and div(ym) = 0. This will throw an error if m = 0, or if gcd(nm) neq 1.\n\nExamples\n\njulia> invmod(2, 5)\n3\n\njulia> invmod(2, 3)\n2\n\njulia> invmod(5, 6)\n5\n\n\n\n\n\n"},{"title":"Base.powermod","page":"Mathematics","location":"base/math.html#Base.powermod","category":"function","text":"powermod(x::Integer, p::Integer, m)\n\nCompute x^p pmod m.\n\nExamples\n\njulia> powermod(2, 6, 5)\n4\n\njulia> mod(2^6, 5)\n4\n\njulia> powermod(5, 2, 20)\n5\n\njulia> powermod(5, 2, 19)\n6\n\njulia> powermod(5, 3, 19)\n11\n\n\n\n\n\n"},{"title":"Base.ndigits","page":"Mathematics","location":"base/math.html#Base.ndigits","category":"function","text":"ndigits(n::Integer; base::Integer=10, pad::Integer=1)\n\nCompute the number of digits in integer n written in base base (base must not be in [-1, 0, 1]), optionally padded with zeros to a specified size (the result will never be less than pad).\n\nSee also digits, count_ones.\n\nExamples\n\njulia> ndigits(0)\n1\n\njulia> ndigits(12345)\n5\n\njulia> ndigits(1022, base=16)\n3\n\njulia> string(1022, base=16)\n\"3fe\"\n\njulia> ndigits(123, pad=5)\n5\n\njulia> ndigits(-123)\n3\n\n\n\n\n\n"},{"title":"Base.add_sum","page":"Mathematics","location":"base/math.html#Base.add_sum","category":"function","text":"Base.add_sum(x, y)\n\nThe reduction operator used in sum. The main difference from + is that small integers are promoted to Int/UInt.\n\n\n\n\n\n"},{"title":"Base.uabs","page":"Mathematics","location":"base/math.html#Base.uabs","category":"function","text":"Base.uabs(x::Integer)\n\nReturn the absolute value of x, possibly returning a different type should the operation be susceptible to overflow. This typically arises when x is a two's complement signed integer, so that abs(typemin(x)) == typemin(x) < 0, in which case the result of uabs(x) will be an unsigned integer of the same size.\n\n\n\n\n\n"},{"title":"Base.widemul","page":"Mathematics","location":"base/math.html#Base.widemul","category":"function","text":"widemul(x, y)\n\nMultiply x and y, giving the result as a larger type.\n\nSee also promote, Base.add_sum.\n\nExamples\n\njulia> widemul(Float32(3.0), 4.0) isa BigFloat\ntrue\n\njulia> typemax(Int8) * typemax(Int8)\n1\n\njulia> widemul(typemax(Int8), typemax(Int8))  # == 127^2\n16129\n\n\n\n\n\n"},{"title":"Base.Math.evalpoly","page":"Mathematics","location":"base/math.html#Base.Math.evalpoly","category":"function","text":"evalpoly(x, p)\n\nEvaluate the polynomial sum_k x^k-1 pk for the coefficients p[1], p[2], ...; that is, the coefficients are given in ascending order by power of x. Loops are unrolled at compile time if the number of coefficients is statically known, i.e. when p is a Tuple. This function generates efficient code using Horner's method if x is real, or using a Goertzel-like [DK62] algorithm if x is complex.\n\n[DK62]: Donald Knuth, Art of Computer Programming, Volume 2: Seminumerical Algorithms, Sec. 4.6.4.\n\ncompat: Julia 1.4\nThis function requires Julia 1.4 or later.\n\nExamples\n\njulia> evalpoly(2, (1, 2, 3))\n17\n\n\n\n\n\n"},{"title":"Base.Math.@evalpoly","page":"Mathematics","location":"base/math.html#Base.Math.@evalpoly","category":"macro","text":"@evalpoly(z, c...)\n\nEvaluate the polynomial sum_k z^k-1 ck for the coefficients c[1], c[2], ...; that is, the coefficients are given in ascending order by power of z.  This macro expands to efficient inline code that uses either Horner's method or, for complex z, a more efficient Goertzel-like algorithm.\n\nSee also evalpoly.\n\nExamples\n\njulia> @evalpoly(3, 1, 0, 1)\n10\n\njulia> @evalpoly(2, 1, 0, 1)\n5\n\njulia> @evalpoly(2, 1, 1, 1)\n7\n\n\n\n\n\n"},{"title":"Base.FastMath.@fastmath","page":"Mathematics","location":"base/math.html#Base.FastMath.@fastmath","category":"macro","text":"@fastmath expr\n\nExecute a transformed version of the expression, which calls functions that may violate strict IEEE semantics. This allows the fastest possible operation, but results are undefined – be careful when doing this, as it may change numerical results.\n\nThis sets the LLVM Fast-Math flags, and corresponds to the -ffast-math option in clang. See the notes on performance annotations for more details.\n\nExamples\n\njulia> @fastmath 1+2\n3\n\njulia> @fastmath(sin(3))\n0.1411200080598672\n\n\n\n\n\n"},{"title":"Customizable binary operators","page":"Mathematics","location":"base/math.html#Customizable-binary-operators","category":"section","text":""},{"title":"Mathematics","page":"Mathematics","location":"base/math.html","category":"page","text":"Some unicode characters can be used to define new binary operators that support infix notation. For example ⊗(x,y) = kron(x,y) defines the ⊗ (otimes) function to be the Kronecker product, and one can call it as binary operator using infix syntax: C = A ⊗ B as well as with the usual prefix syntax C = ⊗(A,B)."},{"title":"Mathematics","page":"Mathematics","location":"base/math.html","category":"page","text":"Other characters that support such extensions include \\odot ⊙ and \\oplus ⊕"},{"title":"Mathematics","page":"Mathematics","location":"base/math.html","category":"page","text":"The complete list is in the parser code: https://github.com/JuliaLang/julia/blob/master/src/julia-parser.scm"},{"title":"Mathematics","page":"Mathematics","location":"base/math.html","category":"page","text":"Those that are parsed like * (in terms of precedence) include * / ÷ % & ⋅ ∘ × |\\\\| ∩ ∧ ⊗ ⊘ ⊙ ⊚ ⊛ ⊠ ⊡ ⊓ ∗ ∙ ∤ ⅋ ≀ ⊼ ⋄ ⋆ ⋇ ⋉ ⋊ ⋋ ⋌ ⋏ ⋒ ⟑ ⦸ ⦼ ⦾ ⦿ ⧶ ⧷ ⨇ ⨰ ⨱ ⨲ ⨳ ⨴ ⨵ ⨶ ⨷ ⨸ ⨻ ⨼ ⨽ ⩀ ⩃ ⩄ ⩋ ⩍ ⩎ ⩑ ⩓ ⩕ ⩘ ⩚ ⩜ ⩞ ⩟ ⩠ ⫛ ⊍ ▷ ⨝ ⟕ ⟖ ⟗ and those that are parsed like + include + - |\\|| ⊕ ⊖ ⊞ ⊟ |++| ∪ ∨ ⊔ ± ∓ ∔ ∸ ≏ ⊎ ⊻ ⊽ ⋎ ⋓ ⟇ ⧺ ⧻ ⨈ ⨢ ⨣ ⨤ ⨥ ⨦ ⨧ ⨨ ⨩ ⨪ ⨫ ⨬ ⨭ ⨮ ⨹ ⨺ ⩁ ⩂ ⩅ ⩊ ⩌ ⩏ ⩐ ⩒ ⩔ ⩖ ⩗ ⩛ ⩝ ⩡ ⩢ ⩣ There are many others that are related to arrows, comparisons, and powers."},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html#External-Profiler-Support","category":"section","text":""},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"Julia provides explicit support for some external tracing profilers, enabling you to obtain a high-level overview of the runtime's execution behavior."},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"The currently supported profilers are:"},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"Tracy\nIntel VTune (ITTAPI)"},{"title":"Adding New Zones","page":"External Profiler Support","location":"devdocs/external_profilers.html#Adding-New-Zones","category":"section","text":""},{"title":"From C/C++ code","page":"External Profiler Support","location":"devdocs/external_profilers.html#From-C/C-code","category":"section","text":""},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"To add new zones, use the JL_TIMING macro. You can find numerous examples throughout the codebase by searching for JL_TIMING. To add a new type of zone you add it to JL_TIMING_OWNERS (and possibly JL_TIMING_EVENTS)."},{"title":"From Julia code","page":"External Profiler Support","location":"devdocs/external_profilers.html#From-Julia-code","category":"section","text":""},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"The Compiler.@zone macro can be used to add a zone from Julia code, it is used as:"},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"Compiler.@zone \"ZONE NAME\" begin\n    ...\nend"},{"title":"Dynamically Enabling and Disabling Zones","page":"External Profiler Support","location":"devdocs/external_profilers.html#Dynamically-Enabling-and-Disabling-Zones","category":"section","text":""},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"The JULIA_TIMING_SUBSYSTEMS environment variable allows you to enable or disable zones for a specific Julia run. For instance, setting the variable to +GC,-INFERENCE will enable the GC zones and disable the INFERENCE zones."},{"title":"Tracy Profiler","page":"External Profiler Support","location":"devdocs/external_profilers.html#Tracy-Profiler","category":"section","text":""},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"Tracy  is a flexible profiler that can be optionally integrated with Julia."},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"A typical Tracy session might look like this:"},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"(Image: Typical Tracy usage)"},{"title":"Building Julia with Tracy","page":"External Profiler Support","location":"devdocs/external_profilers.html#Building-Julia-with-Tracy","category":"section","text":""},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"To enable Tracy integration, build Julia with the extra option WITH_TRACY=1 in the Make.user file."},{"title":"Installing the Tracy Profile Viewer","page":"External Profiler Support","location":"devdocs/external_profilers.html#Installing-the-Tracy-Profile-Viewer","category":"section","text":""},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"The easiest way to obtain the profile viewer is by adding the TracyProfiler_jll package and launching the profiler with:"},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"run(TracyProfiler_jll.tracy())"},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"note: Note\nOn macOS, you may want to set the TRACY_DPI_SCALE environment variable to 1.0 if the UI elements in the profiler appear excessively large."},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"To run a \"headless\" instance that saves the trace to disk, use"},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"run(`$(TracyProfiler_jll.capture()) -o mytracefile.tracy`)"},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"instead."},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"For information on using the Tracy UI, refer to the Tracy manual."},{"title":"Profiling Julia with Tracy","page":"External Profiler Support","location":"devdocs/external_profilers.html#Profiling-Julia-with-Tracy","category":"section","text":""},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"A typical workflow for profiling Julia with Tracy involves starting Julia using:"},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"JULIA_WAIT_FOR_TRACY=1 ./julia -e '...'"},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"The environment variable ensures that Julia waits until it has successfully connected to the Tracy profiler before continuing execution. Afterward, use the Tracy profiler UI, click Connect, and Julia execution should resume and profiling should start."},{"title":"Profiling package precompilation with Tracy","page":"External Profiler Support","location":"devdocs/external_profilers.html#Profiling-package-precompilation-with-Tracy","category":"section","text":""},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"To profile a package precompilation process it is easiest to explicitly call into Base.compilecache with the package you want to precompile:"},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"pkg = Base.identify_package(\"SparseArrays\")\nwithenv(\"JULIA_WAIT_FOR_TRACY\" => 1, \"TRACY_PORT\" => 9001) do\n    Base.compilecache(pkg)\nend"},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"Here, we use a custom port for tracy which makes it easier to find the correct client in the Tracy UI to connect to."},{"title":"Adding metadata to zones","page":"External Profiler Support","location":"devdocs/external_profilers.html#Adding-metadata-to-zones","category":"section","text":""},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"The various jl_timing_show_* and jl_timing_printf functions can be used to attach a string (or strings) to a zone. For example, the trace zone for inference shows the method instance that is being inferred."},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"The TracyCZoneColor function can be used to set the color of a certain zone. Search through the codebase to see how it is used."},{"title":"Viewing Tracy files in your browser","page":"External Profiler Support","location":"devdocs/external_profilers.html#Viewing-Tracy-files-in-your-browser","category":"section","text":""},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"Visit https://topolarity.github.io/trace-viewer/ for an (experimental) web viewer for Tracy traces."},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"You can open a local .tracy file or provide a URL from the web (e.g. a file in a Github repo). If you load a trace file from the web, you can also share the page URL directly with others, enabling them to view the same trace."},{"title":"Enabling stack trace samples","page":"External Profiler Support","location":"devdocs/external_profilers.html#Enabling-stack-trace-samples","category":"section","text":""},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"To enable call stack sampling in Tracy, build Julia with these options in your Make.user file:"},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"WITH_TRACY := 1\nWITH_TRACY_CALLSTACKS := 1\nUSE_BINARYBUILDER_LIBTRACYCLIENT := 0"},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"You may also need to run make -C deps clean-libtracyclient to force a re-build of Tracy."},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"This feature has a significant impact on trace size and profiling overhead, so it is recommended to leave call stack sampling off when possible, especially if you intend to share your trace files online."},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"Note that the Julia JIT runtime does not yet have integration for Tracy's symbolification, so Julia functions will typically be unknown in these stack traces."},{"title":"Intel VTune (ITTAPI) Profiler","page":"External Profiler Support","location":"devdocs/external_profilers.html#Intel-VTune-(ITTAPI)-Profiler","category":"section","text":""},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html","category":"page","text":"This section is yet to be written."},{"title":"FreeBSD","page":"FreeBSD","location":"devdocs/build/freebsd.html#FreeBSD","category":"section","text":""},{"title":"FreeBSD","page":"FreeBSD","location":"devdocs/build/freebsd.html","category":"page","text":"Clang is the default compiler on FreeBSD 11.0-RELEASE and above. The remaining build tools are available from the Ports Collection, and can be installed using pkg install git gcc gmake cmake pkgconf. To build Julia, simply run gmake. (Note that gmake must be used rather than make, since make on FreeBSD corresponds to the incompatible BSD Make rather than GNU Make.)"},{"title":"FreeBSD","page":"FreeBSD","location":"devdocs/build/freebsd.html","category":"page","text":"As mentioned above, it is important to note that the USE_SYSTEM_* flags should be used with caution on FreeBSD. This is because many system libraries, and even libraries from the Ports Collection, link to the system's libgcc_s.so.1, or to another library which links to the system libgcc_s. This library declares its GCC version to be 4.6, which is too old to build Julia, and conflicts with other libraries when linking. Thus it is highly recommended to simply allow Julia to build all of its dependencies. If you do choose to use the USE_SYSTEM_* flags, note that /usr/local is not on the compiler path by default, so you may need to add LDFLAGS=-L/usr/local/lib and CPPFLAGS=-I/usr/local/include to your Make.user, though doing so may interfere with other dependencies."},{"title":"FreeBSD","page":"FreeBSD","location":"devdocs/build/freebsd.html","category":"page","text":"Note that the x86 architecture does not support threading due to lack of compiler runtime library support, so you may need to set JULIA_THREADS=0 in your Make.user if you're on a 32-bit system."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html#Julia-ASTs","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Julia has two representations of code. First there is a surface syntax AST returned by the parser (e.g. the Meta.parse function), and manipulated by macros. It is a structured representation of code as it is written, constructed by julia-parser.scm from a character stream. Next there is a lowered form, or IR (intermediate representation), which is used by type inference and code generation. In the lowered form there are fewer types of nodes, all macros are expanded, and all control flow is converted to explicit branches and sequences of statements. The lowered form is constructed by julia-syntax.scm."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"First we will focus on the AST, since it is needed to write macros."},{"title":"Surface syntax AST","page":"Julia ASTs","location":"devdocs/ast.html#Surface-syntax-AST","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Front end ASTs consist almost entirely of Exprs and atoms (e.g. symbols, numbers). There is generally a different expression head for each visually distinct syntactic form. Examples will be given in s-expression syntax. Each parenthesized list corresponds to an Expr, where the first element is the head. For example (call f x) corresponds to Expr(:call, :f, :x) in Julia."},{"title":"Calls","page":"Julia ASTs","location":"devdocs/ast.html#Calls","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Input AST\nf(x) (call f x)\nf(x, y=1, z=2) (call f x (kw y 1) (kw z 2))\nf(x; y=1) (call f (parameters (kw y 1)) x)\nf(x...) (call f (... x))"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"do syntax:"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"f(x) do a,b\n    body\nend"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"parses as (do (call f x) (-> (tuple a b) (block body)))."},{"title":"Operators","page":"Julia ASTs","location":"devdocs/ast.html#Operators","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Most uses of operators are just function calls, so they are parsed with the head call. However some operators are special forms (not necessarily function calls), and in those cases the operator itself is the expression head. In julia-parser.scm these are referred to as \"syntactic operators\". Some operators (+ and *) use N-ary parsing; chained calls are parsed as a single N-argument call. Finally, chains of comparisons have their own special expression structure."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Input AST\nx+y (call + x y)\na+b+c+d (call + a b c d)\n2x (call * 2 x)\na&&b (&& a b)\nx += 1 (+= x 1)\na ? 1 : 2 (if a 1 2)\na,b (tuple a b)\na==b (call == a b)\n1<i<=n (comparison 1 < i <= n)\na.b (. a (quote b))\na.(b) (. a (tuple b))"},{"title":"Bracketed forms","page":"Julia ASTs","location":"devdocs/ast.html#Bracketed-forms","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Input AST\na[i] (ref a i)\nt[i;j] (typed_vcat t i j)\nt[i j] (typed_hcat t i j)\nt[a b; c d] (typed_vcat t (row a b) (row c d))\nt[a b;;; c d] (typed_ncat t 3 (row a b) (row c d))\na{b} (curly a b)\na{b;c} (curly a (parameters c) b)\n[x] (vect x)\n[x,y] (vect x y)\n[x;y] (vcat x y)\n[x y] (hcat x y)\n[x y; z t] (vcat (row x y) (row z t))\n[x;y;; z;t;;;] (ncat 3 (nrow 2 (nrow 1 x y) (nrow 1 z t)))\n[x for y in z, a in b] (comprehension (generator x (= y z) (= a b)))\nT[x for y in z] (typed_comprehension T (generator x (= y z)))\n(a, b, c) (tuple a b c)\n(a; b; c) (block a b c)"},{"title":"Macros","page":"Julia ASTs","location":"devdocs/ast.html#Macros","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Input AST\n@m x y (macrocall @m (line) x y)\nBase.@m x y (macrocall (. Base (quote @m)) (line) x y)\n@Base.m x y (macrocall (. Base (quote @m)) (line) x y)"},{"title":"Strings","page":"Julia ASTs","location":"devdocs/ast.html#Strings","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Input AST\n\"a\" \"a\"\nx\"y\" (macrocall @x_str (line) \"y\")\nx\"y\"z (macrocall @x_str (line) \"y\" \"z\")\n\"x = $x\" (string \"x = \" x)\n`a b c` (macrocall @cmd (line) \"a b c\")"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Doc string syntax:"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"\"some docs\"\nf(x) = x"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"parses as (macrocall (|.| Core '@doc) (line) \"some docs\" (= (call f x) (block x)))."},{"title":"Imports and such","page":"Julia ASTs","location":"devdocs/ast.html#Imports-and-such","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Input AST\nimport a (import (. a))\nimport a.b.c (import (. a b c))\nimport ...a (import (. . . . a))\nimport a.b, c.d (import (. a b) (. c d))\nimport Base: x (import (: (. Base) (. x)))\nimport Base: x, y (import (: (. Base) (. x) (. y)))\nexport a, b (export a b)\npublic a, b (public a b)"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"using has the same representation as import, but with expression head :using instead of :import."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"To programmatically create a public statement, you can use Expr(:public, :a, :b) or, closer to regular code, Meta.parse(\"public a, b\"). This approach is necessary due to current limitations on public. The public keyword is only recognized at the syntactic top level within a file (parse_stmts) or module. This restriction was implemented to prevent breaking existing code that used public as an identifier when it was introduced in Julia 1.11."},{"title":"Numbers","page":"Julia ASTs","location":"devdocs/ast.html#Numbers","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Julia supports more number types than many scheme implementations, so not all numbers are represented directly as scheme numbers in the AST."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Input AST\n11111111111111111111 (macrocall @int128_str nothing \"11111111111111111111\")\n0xfffffffffffffffff (macrocall @uint128_str nothing \"0xfffffffffffffffff\")\n1111...many digits... (macrocall @big_str nothing \"1111....\")"},{"title":"Block forms","page":"Julia ASTs","location":"devdocs/ast.html#Block-forms","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A block of statements is parsed as (block stmt1 stmt2 ...)."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"If statement:"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"if a\n    b\nelseif c\n    d\nelse\n    e\nend"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"parses as:"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"(if a (block (line 2) b)\n    (elseif (block (line 3) c) (block (line 4) d)\n            (block (line 6) e)))"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A while loop parses as (while condition body)."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A for loop parses as (for (= var iter) body). If there is more than one iteration specification, they are parsed as a block: (for (block (= v1 iter1) (= v2 iter2)) body)."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"break and continue are parsed as 0-argument expressions (break) and (continue)."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"let is parsed as (let (= var val) body) or (let (block (= var1 val1) (= var2 val2) ...) body), like for loops."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A basic function definition is parsed as (function (call f x) body). A more complex example:"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"function f(x::T; k = 1) where T\n    return x+1\nend"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"parses as:"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"(function (where (call f (parameters (kw k 1))\n                       (:: x T))\n                 T)\n          (block (line 2) (return (call + x 1))))"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Type definition:"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"mutable struct Foo{T<:S}\n    x::T\nend"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"parses as:"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"(struct true (curly Foo (<: T S))\n        (block (line 2) (:: x T)))"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"The first argument is a boolean telling whether the type is mutable."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"try blocks parse as (try try_block var catch_block finally_block). If no variable is present after catch, var is #f. If there is no finally clause, then the last argument is not present."},{"title":"Quote expressions","page":"Julia ASTs","location":"devdocs/ast.html#Quote-expressions","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Julia source syntax forms for code quoting (quote and :( )) support interpolation with $. In Lisp terminology, this means they are actually \"backquote\" or \"quasiquote\" forms. Internally, there is also a need for code quoting without interpolation. In Julia's scheme code, non-interpolating quote is represented with the expression head inert."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"inert expressions are converted to Julia QuoteNode objects. These objects wrap a single value of any type, and when evaluated simply return that value."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A quote expression whose argument is an atom also gets converted to a QuoteNode."},{"title":"Line numbers","page":"Julia ASTs","location":"devdocs/ast.html#Line-numbers","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Source location information is represented as (line line_num file_name) where the third component is optional (and omitted when the current line number, but not file name, changes)."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"These expressions are represented as LineNumberNodes in Julia."},{"title":"Macros","page":"Julia ASTs","location":"devdocs/ast.html#Macros-2","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Macro hygiene is represented through the expression head pair escape and hygienic-scope. The result of a macro expansion is automatically wrapped in (hygienic-scope block module [lno]), to represent the result of the new scope. The user can insert (escape block) inside to interpolate code from the caller. The lno is the __source__ argument of the macro, if included."},{"title":"Lowered form","page":"Julia ASTs","location":"devdocs/ast.html#Lowered-form","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Lowered form (IR) is more important to the compiler, since it is used for type inference, optimizations like inlining, and code generation. It is also less obvious to the human, since it results from a significant rearrangement of the input syntax."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"In addition to Symbols and some number types, the following data types exist in lowered form:"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Expr\nHas a node type indicated by the head field, and an args field which is a Vector{Any} of subexpressions. While almost every part of a surface AST is represented by an Expr, the IR uses only a limited number of Exprs, mostly for calls and some top-level-only forms.\nSlotNumber\nIdentifies arguments and local variables by consecutive numbering. It has an integer-valued id field giving the slot index. The types of these slots can be found in the slottypes field of their CodeInfo object.\nArgument\nThe same as SlotNumber, but appears only post-optimization. Indicates that the referenced slot is an argument of the enclosing function.\nCodeInfo\nWraps the IR of a group of statements. Its code field is an array of expressions to execute.\nGotoNode\nUnconditional branch. The argument is the branch target, represented as an index in the code array to jump to.\nGotoIfNot\nConditional branch. If the cond field evaluates to false, goes to the index identified by the dest field.\nReturnNode\nReturns its argument (the val field) as the value of the enclosing function. If the val field is undefined, then this represents an unreachable statement.\nQuoteNode\nWraps an arbitrary value to reference as data. For example, the function f() = :a contains a QuoteNode whose value field is the symbol a, in order to return the symbol itself instead of evaluating it.\nGlobalRef\nRefers to global variable name in module mod.\nSSAValue\nRefers to a consecutively-numbered (starting at 1) static single assignment (SSA) variable inserted by the compiler. The number (id) of an SSAValue is the code array index of the expression whose value it represents.\nNewvarNode\nMarks a point where a variable (slot) is created. This has the effect of resetting a variable to undefined."},{"title":"Expr types","page":"Julia ASTs","location":"devdocs/ast.html#Expr-types","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"These symbols appear in the head field of Exprs in lowered form."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"call\nFunction call (dynamic dispatch). args[1] is the function to call, args[2:end] are the arguments.\ninvoke\nFunction call (static dispatch). args[1] is the MethodInstance to call, args[2:end] are the arguments (including the function that is being called, at args[2]).\nstatic_parameter\nReference a static parameter by index.\n=\nAssignment. In the IR, the first argument is always a SlotNumber or a GlobalRef.\nmethod\nAdds a method to a generic function and assigns the result if necessary.\nHas a 1-argument form and a 3-argument form. The 1-argument form arises from the syntax function foo end. In the 1-argument form, the argument is a symbol. If this symbol already names a function in the current scope, nothing happens. If the symbol is undefined, a new function is created and assigned to the identifier specified by the symbol. If the symbol is defined but names a non-function, an error is raised. The definition of \"names a function\" is that the binding is constant, and refers to an object of singleton type. The rationale for this is that an instance of a singleton type uniquely identifies the type to add the method to. When the type has fields, it wouldn't be clear whether the method was being added to the instance or its type.\nThe 3-argument form has the following arguments:\nargs[1]\nA function name, or nothing if unknown or unneeded. If a symbol, then the expression first behaves like the 1-argument form above. This argument is ignored from then on. It can be nothing when methods are added strictly by type, (::T)(x) = x, or when a method is being added to an existing function, MyModule.f(x) = x.\nargs[2]\nA SimpleVector of argument type data. args[2][1] is a SimpleVector of the argument types, and args[2][2] is a SimpleVector of type variables corresponding to the method's static parameters.\nargs[3]\nA CodeInfo of the method itself. For \"out of scope\" method definitions (adding a method to a function that also has methods defined in different scopes) this is an expression that evaluates to a :lambda expression.\nstruct_type\nA 7-argument expression that defines a new struct:\nargs[1]\nThe name of the struct\nargs[2]\nA call expression that creates a SimpleVector specifying its parameters\nargs[3]\nA call expression that creates a SimpleVector specifying its fieldnames\nargs[4]\nA Symbol, GlobalRef, or Expr specifying the supertype (e.g., :Integer, GlobalRef(Core, :Any), or :(Core.apply_type(AbstractArray, T, N)))\nargs[5]\nA call expression that creates a SimpleVector specifying its fieldtypes\nargs[6]\nA Bool, true if mutable\nargs[7]\nThe number of arguments to initialize. This will be the number of fields, or the minimum number of fields called by an inner constructor's new statement.\nabstract_type\nA 3-argument expression that defines a new abstract type. The arguments are the same as arguments 1, 2, and 4 of struct_type expressions.\nprimitive_type\nA 4-argument expression that defines a new primitive type. Arguments 1, 2, and 4 are the same as struct_type. Argument 3 is the number of bits.\ncompat: Julia 1.5\nstruct_type, abstract_type, and primitive_type were removed in Julia 1.5 and replaced by calls to new builtins.\nglobal\nDeclares a global binding.\nconst\nDeclares a (global) variable as constant.\nnew\nAllocates a new struct-like object. First argument is the type. The new pseudo-function is lowered to this, and the type is always inserted by the compiler. This is very much an internal-only feature, and does no checking. Evaluating arbitrary new expressions can easily segfault.\nsplatnew\nSimilar to new, except field values are passed as a single tuple. Works similarly to splat(new) if new were a first-class function, hence the name.\nisdefined\nExpr(:isdefined, :x) returns a Bool indicating whether x has already been defined in the current scope.\nthe_exception\nYields the caught exception inside a catch block, as returned by jl_current_exception(ct).\nenter\nEnters an exception handler (setjmp). args[1] is the label of the catch block to jump to on error. Yields a token which is consumed by pop_exception.\nleave\nPop exception handlers. args[1] is the number of handlers to pop.\npop_exception\nPop the stack of current exceptions back to the state at the associated enter when leaving a catch block. args[1] contains the token from the associated enter.\ncompat: Julia 1.1\npop_exception is new in Julia 1.1.\ninbounds\nControls turning bounds checks on or off. A stack is maintained; if the first argument of this expression is true or false (true means bounds checks are disabled), it is pushed onto the stack. If the first argument is :pop, the stack is popped.\nboundscheck\nHas the value false if inlined into a section of code marked with @inbounds, otherwise has the value true.\nloopinfo\nMarks the end of the a loop. Contains metadata that is passed to LowerSimdLoop to either mark the inner loop of @simd expression, or to propagate information to LLVM loop passes.\ncopyast\nPart of the implementation of quasi-quote. The argument is a surface syntax AST that is simply copied recursively and returned at run time.\nmeta\nMetadata. args[1] is typically a symbol specifying the kind of metadata, and the rest of the arguments are free-form. The following kinds of metadata are commonly used:\n:inline and :noinline: Inlining hints.\nforeigncall\nStatically-computed container for ccall information. The fields are:\nargs[1] : name\nThe expression that'll be parsed for the foreign function.\nargs[2]::Type : RT\nThe (literal) return type, computed statically when the containing method was defined.\nargs[3]::SimpleVector (of Types) : AT\nThe (literal) vector of argument types, computed statically when the containing method was defined.\nargs[4]::Int : nreq\nThe number of required arguments for a varargs function definition.\nargs[5]::QuoteNode{<:Union{Symbol,Tuple{Symbol,UInt16}, Tuple{Symbol,UInt16,Bool}}: calling convention\nThe calling convention for the call, optionally with effects, and gc_safe (safe to execute concurrently to GC.).\nargs[6:5+length(args[3])] : arguments\nThe values for all the arguments (with types of each given in args[3]).\nargs[6+length(args[3])+1:end] : gc-roots\nThe additional objects that may need to be gc-rooted for the duration of the call. See Working with LLVM for where these are derived from and how they get handled.\nnew_opaque_closure\nConstructs a new opaque closure. The fields are:\nargs[1] : signature\nThe function signature of the opaque closure. Opaque closures don't participate in dispatch, but the input types can be restricted.\nargs[2] : lb\nLower bound on the output type. (Defaults to Union{})\nargs[3] : ub\nUpper bound on the output type. (Defaults to Any)\nargs[4] : constprop\nIndicates whether the opaque closure's identity may be used for constant propagation. The @opaque macro enables this by default, but this will cause additional inference which may be undesirable and prevents the code from running during precompile. If args[4] is a method, the argument is considered skipped.\nargs[5] : method\nThe actual method as an opaque_closure_method expression.\nargs[6:end] : captures\nThe values captured by the opaque closure.\ncompat: Julia 1.7\nOpaque closures were added in Julia 1.7"},{"title":"Method","page":"Julia ASTs","location":"devdocs/ast.html#ast-lowered-method","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A unique'd container describing the shared metadata for a single method."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"name, module, file, line, sig\nMetadata to uniquely identify the method for the computer and the human.\nambig\nCache of other methods that may be ambiguous with this one.\nspecializations\nCache of all MethodInstance ever created for this Method, used to ensure uniqueness. Uniqueness is required for efficiency, especially for incremental precompile and tracking of method invalidation.\nsource\nThe original source code (if available, usually compressed).\ngenerator\nA callable object which can be executed to get specialized source for a specific method signature.\nroots\nPointers to non-AST things that have been interpolated into the AST, required by compression of the AST, type-inference, or the generation of native code.\nnargs, isva, called, is_for_opaque_closure,\nDescriptive bit-fields for the source code of this Method.\nprimary_world\nThe world age that \"owns\" this Method."},{"title":"MethodInstance","page":"Julia ASTs","location":"devdocs/ast.html#MethodInstance","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A unique'd container describing a single callable signature for a Method. See especially Proper maintenance and care of multi-threading locks for important details on how to modify these fields safely."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"specTypes\nThe primary key for this MethodInstance. Uniqueness is guaranteed through a def.specializations lookup.\ndef\nThe Method that this function describes a specialization of. Or a Module, if this is a top-level Lambda expanded in Module, and which is not part of a Method.\nsparam_vals\nThe values of the static parameters in specTypes. For the MethodInstance at Method.unspecialized, this is the empty SimpleVector. But for a runtime MethodInstance from the MethodTable cache, this will always be defined and indexable.\nbackedges\nWe store the reverse-list of cache dependencies for efficient tracking of incremental reanalysis/recompilation work that may be needed after a new method definitions. This works by keeping a list of the other MethodInstance that have been inferred or optimized to contain a possible call to this MethodInstance. Those optimization results might be stored somewhere in the cache, or it might have been the result of something we didn't want to cache, such as constant propagation. Thus we merge all of those backedges to various cache entries here (there's almost always only the one applicable cache entry with a sentinel value for max_world anyways).\ncache\nCache of CodeInstance objects that share this template instantiation."},{"title":"CodeInstance","page":"Julia ASTs","location":"devdocs/ast.html#CodeInstance","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"def\nThe MethodInstance that this cache entry is derived from.\nowner\nA token that represents the owner of this CodeInstance. Will use jl_egal to match."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"rettype/rettype_const\nThe inferred return type for the specFunctionObject field, which (in most cases) is also the computed return type for the function in general.\ninferred\nMay contain a cache of the inferred source for this function, or it could be set to nothing to just indicate rettype is inferred.\nftpr\nThe generic jlcall entry point.\njlcall_api\nThe ABI to use when calling fptr. Some significant ones include:\n0 - Not compiled yet\n1 - JL_CALLABLE jl_value_t *(*)(jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\n2 - Constant (value stored in rettype_const)\n3 - With Static-parameters forwarded jl_value_t *(*)(jl_svec_t *sparams, jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\n4 - Run in interpreter jl_value_t *(*)(jl_method_instance_t *meth, jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\nmin_world / max_world\nThe range of world ages for which this method instance is valid to be called. If max_world is the special token value -1, the value is not yet known. It may continue to be used until we encounter a backedge that requires us to reconsider.\nTiming fields\ntime_infer_total: Total cost of computing inferred originally as wall-time from start to finish.\ntime_infer_cache_saved: The cost saved from time_infer_total by having caching. Adding this to time_infer_total should give a stable estimate for comparing the cost of two implementations or one implementation over time. This is generally an over-estimate of the time to infer something, since the cache is frequently effective at handling repeated work.\ntime_infer_self: Self cost of julia inference for inferred (a portion of time_infer_total). This is simply the incremental cost of compiling this one method, if given a fully populated cache of all call targets, even including constant inference results and LimitedAccuracy results, which generally are not in a cache.\ntime_compile: Self cost of llvm JIT compilation (e.g. of computing invoke from inferred). A total cost estimate can be computed by walking all of the edges contents and summing those, while accounting for cycles and duplicates. (This field currently does not include any measured AOT compile times.)"},{"title":"CodeInfo","page":"Julia ASTs","location":"devdocs/ast.html#CodeInfo","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"A (usually temporary) container for holding lowered (and possibly inferred) source code."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"code\nAn Any array of statements\nslotnames\nAn array of symbols giving names for each slot (argument or local variable).\nslotflags\nA UInt8 array of slot properties, represented as bit flags:\n0x02 - assigned (only false if there are no assignment statements with this var on the left)\n0x08 - used (if there is any read or write of the slot)\n0x10 - statically assigned once\n0x20 - might be used before assigned. This flag is only valid after type inference.\nssavaluetypes\nEither an array or an Int.\nIf an Int, it gives the number of compiler-inserted temporary locations in the function (the length of code array). If an array, specifies a type for each location.\nssaflags\nStatement-level 32 bits flags for each expression in the function. See the definition of jl_code_info_t in julia.h for more details."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"These are only populated after inference (or by generated functions in some cases):"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"debuginfo\nAn object to retrieve source information for each statements, see How to interpret line numbers in a CodeInfo object.\nrettype\nThe inferred return type of the lowered form (IR). Default value is Any. This is mostly present for convenience, as (due to the way OpaqueClosures work) it is not necessarily the rettype used by codegen.\nparent\nThe MethodInstance that \"owns\" this object (if applicable).\nedges\nForward edges to method instances that must be invalidated.\nmin_world/max_world\nThe range of world ages for which this code was valid at the time when it had been inferred."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Optional Fields:"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"slottypes\nAn array of types for the slots.\nmethod_for_inference_limit_heuristics\nThe method_for_inference_heuristics will expand the given method's generator if necessary during inference."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Boolean properties:"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"propagate_inbounds\nWhether this should propagate @inbounds when inlined for the purpose of eliding @boundscheck blocks."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"UInt8 settings:"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"constprop, inlineable\n0 = use heuristic\n1 = aggressive\n2 = none\npurity Constructed from 5 bit flags:\n0x01 << 0 = this method is guaranteed to return or terminate consistently (:consistent)\n0x01 << 1 = this method is free from externally semantically visible side effects (:effect_free)\n0x01 << 2 = this method is guaranteed to not throw an exception (:nothrow)\n0x01 << 3 = this method is guaranteed to terminate (:terminates_globally)\n0x01 << 4 = the syntactic control flow within this method is guaranteed to terminate (:terminates_locally)\nSee the documentation of Base.@assume_effects for more details."},{"title":"How to interpret line numbers in a CodeInfo object","page":"Julia ASTs","location":"devdocs/ast.html#How-to-interpret-line-numbers-in-a-CodeInfo-object","category":"section","text":""},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"There are 2 common forms for this data: one used internally that compresses the data somewhat and one used in the compiler. They contain the same basic info, but the compiler version is all mutable while the version used internally is not."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"Many consumers may be able to call Base.IRShow.buildLineInfoNode, Base.IRShow.append_scopes!, or Stacktraces.lookup(::InterpreterIP) to avoid needing to (re-)implement these details specifically."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"The definitions of each of these are:"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"struct Core.DebugInfo\n    @noinline\n    def::Union{Method,MethodInstance,Symbol}\n    linetable::Union{Nothing,DebugInfo}\n    edges::SimpleVector{DebugInfo}\n    codelocs::String # compressed data\nend\nmutable struct Core.Compiler.DebugInfoStream\n    def::Union{Method,MethodInstance,Symbol}\n    linetable::Union{Nothing,DebugInfo}\n    edges::Vector{DebugInfo}\n    firstline::Int32 # the starting line for this block (specified by an index of 0)\n    codelocs::Vector{Int32} # for each statement:\n        # index into linetable (if defined), else a line number (in the file represented by def)\n        # then index into edges\n        # then index into edges[linetable]\nend"},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html","category":"page","text":"def : where this DebugInfo was defined (the Method, MethodInstance, or Symbol of file scope, for example)\nlinetable\nAnother DebugInfo that this was derived from, which contains the actual line numbers, such that this DebugInfo contains only the indexes into it. This avoids making copies, as well as makes it possible to track how each individual statement transformed from source to optimized, not just the separate line numbers. If def is not a Symbol, then that object replaces the current function object for the metadata on what function is conceptually being executed (e.g. think Cassette transforms here). The codelocs values described below also are interpreted as an index into the codelocs in this object, instead of being a line number itself.\nedges : Vector of the unique DebugInfo for every function inlined into this (which recursively have the edges for everything inlined into them).\nfirstline (when uncompressed to DebugInfoStream)\nThe line number associated with the begin statement (or other keyword such as function or quote) that delineates where this code definition \"starts\".\ncodelocs (when uncompressed to DebugInfoStream)\nA vector of indices, with 3 values for each statement in the IR plus one for the starting point of the block, that describe the stacktrace from that point:\nthe integer index into the linetable.codelocs field, giving the original location associated with each statement (including its syntactic edges), or zero indicating no change to the line number from the previously executed statement (which is not necessarily syntactic or lexical prior), or the line number itself if the linetable field is nothing.\nthe integer index into edges, giving the DebugInfo inlined there, or zero if there are no edges.\n(if entry 2 is non-zero) the integer index into edges[].codelocs, to interpret recursively for each function in the inlining stack, or zero indicating to use edges[].firstline as the line number.\nSpecial codes include:\n(zero, zero, *): no change to the line number or edges from the previous statement (you may choose to interpret this either syntactically or lexically). The inlining depth also might have changed, though most callers should ignore that.\n(zero, non-zero, *) : no line number, just edges (usually because of macro-expansion into top-level code)."},{"title":"Methods","page":"Methods","location":"manual/methods.html#Methods","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Recall from Functions that a function is an object that maps a tuple of arguments to a return value, or throws an exception if no appropriate value can be returned. It is common for the same conceptual function or operation to be implemented quite differently for different types of arguments: adding two integers is very different from adding two floating-point numbers, both of which are distinct from adding an integer to a floating-point number. Despite their implementation differences, these operations all fall under the general concept of \"addition\". Accordingly, in Julia, these behaviors all belong to a single object: the + function."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"To facilitate using many different implementations of the same concept smoothly, functions need not be defined all at once, but can rather be defined piecewise by providing specific behaviors for certain combinations of argument types and counts. A definition of one possible behavior for a function is called a method. Thus far, we have presented only examples of functions defined with a single method, applicable to all types of arguments. However, the signatures of method definitions can be annotated to indicate the types of arguments in addition to their number, and more than a single method definition may be provided. When a function is applied to a particular tuple of arguments, the most specific method applicable to those arguments is applied. Thus, the overall behavior of a function is a patchwork of the behaviors of its various method definitions. If the patchwork is well designed, even though the implementations of the methods may be quite different, the outward behavior of the function will appear seamless and consistent."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"The choice of which method to execute when a function is applied is called dispatch. Julia allows the dispatch process to choose which of a function's methods to call based on the number of arguments given, and on the types of all of the function's arguments. This is different than traditional object-oriented languages, where dispatch occurs based only on the first argument, which often has a special argument syntax, and is sometimes implied rather than explicitly written as an argument. [1] Using all of a function's arguments to choose which method should be invoked, rather than just the first, is known as multiple dispatch. Multiple dispatch is particularly useful for mathematical code, where it makes little sense to artificially deem the operations to \"belong\" to one argument more than any of the others: does the addition operation in x + y belong to x any more than it does to y? The implementation of a mathematical operator generally depends on the types of all of its arguments. Even beyond mathematical operations, however, multiple dispatch ends up being a powerful and convenient paradigm for structuring and organizing programs."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"[1]: In C++ or Java, for example, in a method call like obj.meth(arg1,arg2), the object obj \"receives\" the method call and is implicitly passed to the method via the this keyword, rather than as an explicit method argument. When the current this object is the receiver of a method call, it can be omitted altogether, writing just meth(arg1,arg2), with this implied as the receiving object."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"note: Note\nAll the examples in this chapter assume that you are defining methods for a function in the same module. If you want to add methods to a function in another module, you have to import it or use the name qualified with module names. See the section on namespace management."},{"title":"Defining Methods","page":"Methods","location":"manual/methods.html#Defining-Methods","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Until now, we have, in our examples, defined only functions with a single method having unconstrained argument types. Such functions behave just like they would in traditional dynamically typed languages. Nevertheless, we have used multiple dispatch and methods almost continually without being aware of it: all of Julia's standard functions and operators, like the aforementioned + function, have many methods defining their behavior over various possible combinations of argument type and count."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"When defining a function, one can optionally constrain the types of parameters it is applicable to, using the :: type-assertion operator, introduced in the section on Composite Types:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> f(x::Float64, y::Float64) = 2x + y\nf (generic function with 1 method)"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"This function definition applies only to calls where x and y are both values of type Float64:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> f(2.0, 3.0)\n7.0"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Applying it to any other types of arguments will result in a MethodError:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> f(2.0, 3)\nERROR: MethodError: no method matching f(::Float64, ::Int64)\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(::Float64, !Matched::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> f(Float32(2.0), 3.0)\nERROR: MethodError: no method matching f(::Float32, ::Float64)\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(!Matched::Float64, ::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> f(2.0, \"3.0\")\nERROR: MethodError: no method matching f(::Float64, ::String)\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(::Float64, !Matched::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> f(\"2.0\", \"3.0\")\nERROR: MethodError: no method matching f(::String, ::String)\nThe function `f` exists, but no method is defined for this combination of argument types."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"As you can see, the arguments must be precisely of type Float64. Other numeric types, such as integers or 32-bit floating-point values, are not automatically converted to 64-bit floating-point, nor are strings parsed as numbers. Because Float64 is a concrete type and concrete types cannot be subclassed in Julia, such a definition can only be applied to arguments that are exactly of type Float64. It may often be useful, however, to write more general methods where the declared parameter types are abstract:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> f(x::Number, y::Number) = 2x - y\nf (generic function with 2 methods)\n\njulia> f(2.0, 3)\n1.0"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"This method definition applies to any pair of arguments that are instances of Number. They need not be of the same type, so long as they are each numeric values. The problem of handling disparate numeric types is delegated to the arithmetic operations in the expression 2x - y."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"To define a function with multiple methods, one simply defines the function multiple times, with different numbers and types of arguments. The first method definition for a function creates the function object, and subsequent method definitions add new methods to the existing function object. The most specific method definition matching the number and types of the arguments will be executed when the function is applied. Thus, the two method definitions above, taken together, define the behavior for f over all pairs of instances of the abstract type Number – but with a different behavior specific to pairs of Float64 values. If one of the arguments is a 64-bit float but the other one is not, then the f(Float64,Float64) method cannot be called and the more general f(Number,Number) method must be used:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> f(2.0, 3.0)\n7.0\n\njulia> f(2, 3.0)\n1.0\n\njulia> f(2.0, 3)\n1.0\n\njulia> f(2, 3)\n1"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"The 2x + y definition is only used in the first case, while the 2x - y definition is used in the others. No automatic casting or conversion of function arguments is ever performed: all conversion in Julia is non-magical and completely explicit. Conversion and Promotion, however, shows how clever application of sufficiently advanced technology can be indistinguishable from magic. [Clarke61]"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"For non-numeric values, and for fewer or more than two arguments, the function f remains undefined, and applying it will still result in a MethodError:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> f(\"foo\", 3)\nERROR: MethodError: no method matching f(::String, ::Int64)\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(!Matched::Number, ::Number)\n   @ Main none:1\n  f(!Matched::Float64, !Matched::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> f()\nERROR: MethodError: no method matching f()\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(!Matched::Float64, !Matched::Float64)\n   @ Main none:1\n  f(!Matched::Number, !Matched::Number)\n   @ Main none:1\n\nStacktrace:\n[...]"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"You can easily see which methods exist for a function by entering the function object itself in an interactive session:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> f\nf (generic function with 2 methods)"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"This output tells us that f is a function object with two methods. To find out what the signatures of those methods are, use the methods function:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> methods(f)\n# 2 methods for generic function \"f\" from Main:\n [1] f(x::Float64, y::Float64)\n     @ none:1\n [2] f(x::Number, y::Number)\n     @ none:1"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"which shows that f has two methods, one taking two Float64 arguments and one taking arguments of type Number. It also indicates the file and line number where the methods were defined: because these methods were defined at the REPL, we get the apparent line number none:1."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"In the absence of a type declaration with ::, the type of a method parameter is Any by default, meaning that it is unconstrained since all values in Julia are instances of the abstract type Any. Thus, we can define a catch-all method for f like so:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> f(x,y) = println(\"Whoa there, Nelly.\")\nf (generic function with 3 methods)\n\njulia> methods(f)\n# 3 methods for generic function \"f\" from Main:\n [1] f(x::Float64, y::Float64)\n     @ none:1\n [2] f(x::Number, y::Number)\n     @ none:1\n [3] f(x, y)\n     @ none:1\n\njulia> f(\"foo\", 1)\nWhoa there, Nelly."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"This catch-all is less specific than any other possible method definition for a pair of parameter values, so it will only be called on pairs of arguments to which no other method definition applies."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Note that in the signature of the third method, there is no type specified for the arguments x and y. This is a shortened way of expressing f(x::Any, y::Any)."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Although it seems a simple concept, multiple dispatch on the types of values is perhaps the single most powerful and central feature of the Julia language. Core operations typically have dozens of methods:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> methods(+)\n# 180 methods for generic function \"+\":\n[1] +(x::Bool, z::Complex{Bool}) in Base at complex.jl:227\n[2] +(x::Bool, y::Bool) in Base at bool.jl:89\n[3] +(x::Bool) in Base at bool.jl:86\n[4] +(x::Bool, y::T) where T<:AbstractFloat in Base at bool.jl:96\n[5] +(x::Bool, z::Complex) in Base at complex.jl:234\n[6] +(a::Float16, b::Float16) in Base at float.jl:373\n[7] +(x::Float32, y::Float32) in Base at float.jl:375\n[8] +(x::Float64, y::Float64) in Base at float.jl:376\n[9] +(z::Complex{Bool}, x::Bool) in Base at complex.jl:228\n[10] +(z::Complex{Bool}, x::Real) in Base at complex.jl:242\n[11] +(x::Char, y::Integer) in Base at char.jl:40\n[12] +(c::BigInt, x::BigFloat) in Base.MPFR at mpfr.jl:307\n[13] +(a::BigInt, b::BigInt, c::BigInt, d::BigInt, e::BigInt) in Base.GMP at gmp.jl:392\n[14] +(a::BigInt, b::BigInt, c::BigInt, d::BigInt) in Base.GMP at gmp.jl:391\n[15] +(a::BigInt, b::BigInt, c::BigInt) in Base.GMP at gmp.jl:390\n[16] +(x::BigInt, y::BigInt) in Base.GMP at gmp.jl:361\n[17] +(x::BigInt, c::Union{UInt16, UInt32, UInt64, UInt8}) in Base.GMP at gmp.jl:398\n...\n[180] +(a, b, c, xs...) in Base at operators.jl:424"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Multiple dispatch together with the flexible parametric type system give Julia its ability to abstractly express high-level algorithms decoupled from implementation details."},{"title":"Method specializations","page":"Methods","location":"manual/methods.html#man-method-specializations","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"When you create multiple methods of the same function, this is sometimes called \"specialization.\" In this case, you're specializing the function by adding additional methods to it: each new method is a new specialization of the function. As shown above, these specializations are returned by methods."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"There's another kind of specialization that occurs without programmer intervention: Julia's compiler can automatically specialize the method for the specific argument types used. Such specializations are not listed by methods, as this doesn't create new Methods, but tools like @code_typed allow you to inspect such specializations."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"For example, if you create a method"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"mysum(x::Real, y::Real) = x + y"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"you've given the function mysum one new method (possibly its only method), and that method takes any pair of Real number inputs. But if you then execute"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> mysum(1, 2)\n3\n\njulia> mysum(1.0, 2.0)\n3.0"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Julia will compile mysum twice, once for x::Int, y::Int and again for x::Float64, y::Float64. The point of compiling twice is performance: the methods that get called for + (which mysum uses) vary depending on the specific types of x and y, and by compiling different specializations Julia can do all the method lookup ahead of time. This allows the program to run much more quickly, since it does not have to bother with method lookup while it is running. Julia's automatic specialization allows you to write generic algorithms and expect that the compiler will generate efficient, specialized code to handle each case you need."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"In cases where the number of potential specializations might be effectively unlimited, Julia may avoid this default specialization. See Be aware of when Julia avoids specializing for more information."},{"title":"Method Ambiguities","page":"Methods","location":"manual/methods.html#man-ambiguities","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"It is possible to define a set of function methods such that there is no unique most specific method applicable to some combinations of arguments:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> g(x::Float64, y) = 2x + y\ng (generic function with 1 method)\n\njulia> g(x, y::Float64) = x + 2y\ng (generic function with 2 methods)\n\njulia> g(2.0, 3)\n7.0\n\njulia> g(2, 3.0)\n8.0\n\njulia> g(2.0, 3.0)\nERROR: MethodError: g(::Float64, ::Float64) is ambiguous.\n\nCandidates:\n  g(x, y::Float64)\n    @ Main none:1\n  g(x::Float64, y)\n    @ Main none:1\n\nPossible fix, define\n  g(::Float64, ::Float64)\n\nStacktrace:\n[...]"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Here the call g(2.0, 3.0) could be handled by either the g(::Float64, ::Any) or the g(::Any, ::Float64) method. The order in which the methods are defined does not matter and neither is more specific than the other. In such cases, Julia raises a MethodError rather than arbitrarily picking a method. You can avoid method ambiguities by specifying an appropriate method for the intersection case:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> g(x::Float64, y::Float64) = 2x + 2y\ng (generic function with 3 methods)\n\njulia> g(2.0, 3)\n7.0\n\njulia> g(2, 3.0)\n8.0\n\njulia> g(2.0, 3.0)\n10.0"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"It is recommended that the disambiguating method be defined first, since otherwise the ambiguity exists, if transiently, until the more specific method is defined."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"In more complex cases, resolving method ambiguities involves a certain element of design; this topic is explored further below."},{"title":"Parametric Methods","page":"Methods","location":"manual/methods.html#Parametric-Methods","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Method definitions can optionally have type parameters qualifying the signature:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> same_type(x::T, y::T) where {T} = true\nsame_type (generic function with 1 method)\n\njulia> same_type(x,y) = false\nsame_type (generic function with 2 methods)"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"The first method applies whenever both arguments are of the same concrete type, regardless of what type that is, while the second method acts as a catch-all, covering all other cases. Thus, overall, this defines a boolean function that checks whether its two arguments are of the same type:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> same_type(1, 2)\ntrue\n\njulia> same_type(1, 2.0)\nfalse\n\njulia> same_type(1.0, 2.0)\ntrue\n\njulia> same_type(\"foo\", 2.0)\nfalse\n\njulia> same_type(\"foo\", \"bar\")\ntrue\n\njulia> same_type(Int32(1), Int64(2))\nfalse"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Such definitions correspond to methods whose type signatures are UnionAll types (see UnionAll Types)."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"This kind of definition of function behavior by dispatch is quite common – idiomatic, even – in Julia. Method type parameters are not restricted to being used as the types of arguments: they can be used anywhere a value would be in the signature of the function or body of the function. Here's an example where the method type parameter T is used as the type parameter to the parametric type Vector{T} in the method signature:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> function myappend(v::Vector{T}, x::T) where {T}\n           return [v..., x]\n       end\nmyappend (generic function with 1 method)"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"The type parameter T in this example ensures that the added element x is a subtype of the existing eltype of the vector v. The where keyword introduces a list of those constraints after the method signature definition. This works the same for one-line definitions, as seen above, and must appear before the return type declaration, if present, as illustrated below:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> (myappend(v::Vector{T}, x::T)::Vector) where {T} = [v..., x]\nmyappend (generic function with 1 method)\n\njulia> myappend([1,2,3],4)\n4-element Vector{Int64}:\n 1\n 2\n 3\n 4\n\njulia> myappend([1,2,3],2.5)\nERROR: MethodError: no method matching myappend(::Vector{Int64}, ::Float64)\nThe function `myappend` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  myappend(::Vector{T}, !Matched::T) where T\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> myappend([1.0,2.0,3.0],4.0)\n4-element Vector{Float64}:\n 1.0\n 2.0\n 3.0\n 4.0\n\njulia> myappend([1.0,2.0,3.0],4)\nERROR: MethodError: no method matching myappend(::Vector{Float64}, ::Int64)\nThe function `myappend` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  myappend(::Vector{T}, !Matched::T) where T\n   @ Main none:1\n\nStacktrace:\n[...]"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"If the type of the appended element does not match the element type of the vector it is appended to, a MethodError is raised. In the following example, the method's type parameter T is used as the return value:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> mytypeof(x::T) where {T} = T\nmytypeof (generic function with 1 method)\n\njulia> mytypeof(1)\nInt64\n\njulia> mytypeof(1.0)\nFloat64"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Just as you can put subtype constraints on type parameters in type declarations (see Parametric Types), you can also constrain type parameters of methods:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> same_type_numeric(x::T, y::T) where {T<:Number} = true\nsame_type_numeric (generic function with 1 method)\n\njulia> same_type_numeric(x::Number, y::Number) = false\nsame_type_numeric (generic function with 2 methods)\n\njulia> same_type_numeric(1, 2)\ntrue\n\njulia> same_type_numeric(1, 2.0)\nfalse\n\njulia> same_type_numeric(1.0, 2.0)\ntrue\n\njulia> same_type_numeric(\"foo\", 2.0)\nERROR: MethodError: no method matching same_type_numeric(::String, ::Float64)\nThe function `same_type_numeric` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  same_type_numeric(!Matched::T, ::T) where T<:Number\n   @ Main none:1\n  same_type_numeric(!Matched::Number, ::Number)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> same_type_numeric(\"foo\", \"bar\")\nERROR: MethodError: no method matching same_type_numeric(::String, ::String)\nThe function `same_type_numeric` exists, but no method is defined for this combination of argument types.\n\njulia> same_type_numeric(Int32(1), Int64(2))\nfalse"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"The same_type_numeric function behaves much like the same_type function defined above, but is only defined for pairs of numbers."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Parametric methods allow the same syntax as where expressions used to write types (see UnionAll Types). If there is only a single parameter, the enclosing curly braces (in where {T}) can be omitted, but are often preferred for clarity. Multiple parameters can be separated with commas, e.g. where {T, S<:Real}, or written using nested where, e.g. where S<:Real where T."},{"title":"Redefining Methods","page":"Methods","location":"manual/methods.html#Redefining-Methods","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"When redefining a method or adding new methods, it is important to realize that these changes don't take effect immediately. This is key to Julia's ability to statically infer and compile code to run fast, without the usual JIT tricks and overhead. Indeed, any new method definition won't be visible to the current runtime environment, including Tasks and Threads (and any previously defined @generated functions). Let's start with an example to see what this means:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> function tryeval()\n           @eval newfun() = 1\n           newfun()\n       end\ntryeval (generic function with 1 method)\n\njulia> tryeval()\nERROR: MethodError: no method matching newfun()\nThe applicable method may be too new: running in world age xxxx1, while current world is xxxx2.\nClosest candidates are:\n  newfun() at none:1 (method too new to be called from this world context.)\n in tryeval() at none:1\n ...\n\njulia> newfun()\n1"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"In this example, observe that the new definition for newfun has been created, but can't be immediately called. The new global is immediately visible to the tryeval function, so you could write return newfun (without parentheses). But neither you, nor any of your callers, nor the functions they call, or etc. can call this new method definition!"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"But there's an exception: future calls to newfun from the REPL work as expected, being able to both see and call the new definition of newfun."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"However, future calls to tryeval will continue to see the definition of newfun as it was at the previous statement at the REPL, and thus before that call to tryeval."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"You may want to try this for yourself to see how it works."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"The implementation of this behavior is a \"world age counter\", which is further described in the World Age manual chapter."},{"title":"Design Patterns with Parametric Methods","page":"Methods","location":"manual/methods.html#Design-Patterns-with-Parametric-Methods","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"While complex dispatch logic is not required for performance or usability, sometimes it can be the best way to express some algorithm. Here are a few common design patterns that come up sometimes when using dispatch in this way."},{"title":"Extracting the type parameter from a super-type","page":"Methods","location":"manual/methods.html#Extracting-the-type-parameter-from-a-super-type","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Here is a correct code template for returning the element-type T of any arbitrary subtype of AbstractArray that has well-defined element type:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"abstract type AbstractArray{T, N} end\neltype(::Type{<:AbstractArray{T}}) where {T} = T"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"using so-called triangular dispatch. Note that UnionAll types, for example eltype(AbstractArray{T} where T <: Integer), do not match the above method. The implementation of eltype in Base adds a fallback method to Any for such cases."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"One common mistake is to try and get the element-type by using introspection:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"eltype_wrong(::Type{A}) where {A<:AbstractArray} = A.parameters[1]"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"However, it is not hard to construct cases where this will fail:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"struct BitVector <: AbstractArray{Bool, 1}; end"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Here we have created a type BitVector which has no parameters, but where the element-type is still fully specified, with T equal to Bool!"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Another mistake is to try to walk up the type hierarchy using supertype:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"eltype_wrong(::Type{AbstractArray{T}}) where {T} = T\neltype_wrong(::Type{AbstractArray{T, N}}) where {T, N} = T\neltype_wrong(::Type{A}) where {A<:AbstractArray} = eltype_wrong(supertype(A))"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"While this works for declared types, it fails for types without supertypes:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> eltype_wrong(Union{Vector{Int}, Matrix{Int}})\nERROR: MethodError: no method matching supertype(::Type{VecOrMat{Int64}})\n\nClosest candidates are:\n  supertype(::UnionAll)\n   @ Base operators.jl:44\n  supertype(::DataType)\n   @ Base operators.jl:43"},{"title":"Building a similar type with a different type parameter","page":"Methods","location":"manual/methods.html#Building-a-similar-type-with-a-different-type-parameter","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"When building generic code, there is often a need for constructing a similar object with some change made to the layout of the type, also necessitating a change of the type parameters. For instance, you might have some sort of abstract array with an arbitrary element type and want to write your computation on it with a specific element type. We must implement a method for each AbstractArray{T} subtype that describes how to compute this type transform. There is no general transform of one subtype into another subtype with a different parameter."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"The subtypes of AbstractArray typically implement two methods to achieve this: A method to convert the input array to a subtype of a specific AbstractArray{T, N} abstract type; and a method to make a new uninitialized array with a specific element type. Sample implementations of these can be found in Julia Base. Here is a basic example usage of them, guaranteeing that input and output are of the same type:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"input = convert(AbstractArray{Eltype}, input)\noutput = similar(input, Eltype)"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"As an extension of this, in cases where the algorithm needs a copy of the input array, convert is insufficient as the return value may alias the original input. Combining similar (to make the output array) and copyto! (to fill it with the input data) is a generic way to express the requirement for a mutable copy of the input argument:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"copy_with_eltype(input, Eltype) = copyto!(similar(input, Eltype), input)"},{"title":"Iterated dispatch","page":"Methods","location":"manual/methods.html#Iterated-dispatch","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"In order to dispatch a multi-level parametric argument list, often it is best to separate each level of dispatch into distinct functions. This may sound similar in approach to single-dispatch, but as we shall see below, it is still more flexible."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"For example, trying to dispatch on the element-type of an array will often run into ambiguous situations. Instead, common code will dispatch first on the container type, then recurse down to a more specific method based on eltype. In most cases, the algorithms lend themselves conveniently to this hierarchical approach, while in other cases, this rigor must be resolved manually. This dispatching branching can be observed, for example, in the logic to sum two matrices:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"# First dispatch selects the map algorithm for element-wise summation.\n+(a::Matrix, b::Matrix) = map(+, a, b)\n# Then dispatch handles each element and selects the appropriate\n# common element type for the computation.\n+(a, b) = +(promote(a, b)...)\n# Once the elements have the same type, they can be added.\n# For example, via primitive operations exposed by the processor.\n+(a::Float64, b::Float64) = Core.add(a, b)"},{"title":"Trait-based dispatch","page":"Methods","location":"manual/methods.html#Trait-based-dispatch","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"A natural extension to the iterated dispatch above is to add a layer to method selection that allows to dispatch on sets of types which are independent from the sets defined by the type hierarchy. We could construct such a set by writing out a Union of the types in question, but then this set would not be extensible as Union-types cannot be altered after creation. However, such an extensible set can be programmed with a design pattern often referred to as a \"Holy-trait\"."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"This pattern is implemented by defining a generic function which computes a different singleton value (or type) for each trait-set to which the function arguments may belong to. If this function is pure there is no impact on performance compared to normal dispatch."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"The example in the previous section glossed over the implementation details of map and promote, which both operate in terms of these traits. When iterating over a matrix, such as in the implementation of map, one important question is what order to use to traverse the data. When AbstractArray subtypes implement the Base.IndexStyle trait, other functions such as map can dispatch on this information to pick the best algorithm (see Abstract Array Interface). This means that each subtype does not need to implement a custom version of map, since the generic definitions + trait classes will enable the system to select the fastest version. Here is a toy implementation of map illustrating the trait-based dispatch:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"map(f, a::AbstractArray, b::AbstractArray) = map(Base.IndexStyle(a, b), f, a, b)\n# generic implementation:\nmap(::Base.IndexCartesian, f, a::AbstractArray, b::AbstractArray) = ...\n# linear-indexing implementation (faster)\nmap(::Base.IndexLinear, f, a::AbstractArray, b::AbstractArray) = ..."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"This trait-based approach is also present in the promote mechanism employed by the scalar +. It uses promote_type, which returns the optimal common type to compute the operation given the two types of the operands. This makes it possible to reduce the problem of implementing every function for every pair of possible type arguments, to the much smaller problem of implementing a conversion operation from each type to a common type, plus a table of preferred pair-wise promotion rules."},{"title":"Output-type computation","page":"Methods","location":"manual/methods.html#Output-type-computation","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"The discussion of trait-based promotion provides a transition into our next design pattern: computing the output element type for a matrix operation."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"For implementing primitive operations, such as addition, we use the promote_type function to compute the desired output type. (As before, we saw this at work in the promote call in the call to +)."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"For more complex functions on matrices, it may be necessary to compute the expected return type for a more complex sequence of operations. This is often performed by the following steps:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Write a small function op that expresses the set of operations performed by the kernel of the algorithm.\nCompute the element type R of the result matrix as promote_op(op, argument_types...), where argument_types is computed from eltype applied to each input array.\nBuild the output matrix as similar(R, dims), where dims are the desired dimensions of the output array."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"For a more specific example, a generic square-matrix multiply pseudo-code might look like:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"function matmul(a::AbstractMatrix, b::AbstractMatrix)\n    op = (ai, bi) -> ai * bi + ai * bi\n\n    ## this is insufficient because it assumes `one(eltype(a))` is constructable:\n    # R = typeof(op(one(eltype(a)), one(eltype(b))))\n\n    ## this fails because it assumes `a[1]` exists and is representative of all elements of the array\n    # R = typeof(op(a[1], b[1]))\n\n    ## this is incorrect because it assumes that `+` calls `promote_type`\n    ## but this is not true for some types, such as Bool:\n    # R = promote_type(ai, bi)\n\n    # this is wrong, since depending on the return value\n    # of type-inference is very brittle (as well as not being optimizable):\n    # R = Base.return_types(op, (eltype(a), eltype(b)))\n\n    ## but, finally, this works:\n    R = promote_op(op, eltype(a), eltype(b))\n    ## although sometimes it may give a larger type than desired\n    ## it will always give a correct type\n\n    output = similar(b, R, (size(a, 1), size(b, 2)))\n    if size(a, 2) > 0\n        for j in 1:size(b, 2)\n            for i in 1:size(a, 1)\n                ## here we don't use `ab = zero(R)`,\n                ## since `R` might be `Any` and `zero(Any)` is not defined\n                ## we also must declare `ab::R` to make the type of `ab` constant in the loop,\n                ## since it is possible that typeof(a * b) != typeof(a * b + a * b) == R\n                ab::R = a[i, 1] * b[1, j]\n                for k in 2:size(a, 2)\n                    ab += a[i, k] * b[k, j]\n                end\n                output[i, j] = ab\n            end\n        end\n    end\n    return output\nend"},{"title":"Separate convert and kernel logic","page":"Methods","location":"manual/methods.html#Separate-convert-and-kernel-logic","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"One way to significantly cut down on compile-times and testing complexity is to isolate the logic for converting to the desired type and the computation. This lets the compiler specialize and inline the conversion logic independent from the rest of the body of the larger kernel."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"This is a common pattern seen when converting from a larger class of types to the one specific argument type that is actually supported by the algorithm:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"complexfunction(arg::Int) = ...\ncomplexfunction(arg::Any) = complexfunction(convert(Int, arg))\n\nmatmul(a::T, b::T) = ...\nmatmul(a, b) = matmul(promote(a, b)...)"},{"title":"Parametrically-constrained Varargs methods","page":"Methods","location":"manual/methods.html#Parametrically-constrained-Varargs-methods","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Function parameters can also be used to constrain the number of arguments that may be supplied to a \"varargs\" function (Varargs Functions). The notation Vararg{T,N} is used to indicate such a constraint. For example:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> bar(a,b,x::Vararg{Any,2}) = (a,b,x)\nbar (generic function with 1 method)\n\njulia> bar(1,2,3)\nERROR: MethodError: no method matching bar(::Int64, ::Int64, ::Int64)\nThe function `bar` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  bar(::Any, ::Any, ::Any, !Matched::Any)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> bar(1,2,3,4)\n(1, 2, (3, 4))\n\njulia> bar(1,2,3,4,5)\nERROR: MethodError: no method matching bar(::Int64, ::Int64, ::Int64, ::Int64, ::Int64)\nThe function `bar` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  bar(::Any, ::Any, ::Any, ::Any)\n   @ Main none:1\n\nStacktrace:\n[...]"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"More usefully, it is possible to constrain varargs methods by a parameter. For example:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"function getindex(A::AbstractArray{T,N}, indices::Vararg{Number,N}) where {T,N}"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"would be called only when the number of indices matches the dimensionality of the array."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"When only the type of supplied arguments needs to be constrained Vararg{T} can be equivalently written as T.... For instance f(x::Int...) = x is a shorthand for f(x::Vararg{Int}) = x."},{"title":"Note on Optional and keyword Arguments","page":"Methods","location":"manual/methods.html#Note-on-Optional-and-keyword-Arguments","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"As mentioned briefly in Functions, optional arguments are implemented as syntax for multiple method definitions. For example, this definition:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"f(a=1,b=2) = a+2b"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"translates to the following three methods:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"f(a,b) = a+2b\nf(a) = f(a,2)\nf() = f(1,2)"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"This means that calling f() is equivalent to calling f(1,2). In this case the result is 5, because f(1,2) invokes the first method of f above. However, this need not always be the case. If you define a fourth method that is more specialized for integers:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"f(a::Int,b::Int) = a-2b"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"then the result of both f() and f(1,2) is -3. In other words, optional arguments are tied to a function, not to any specific method of that function. It depends on the types of the optional arguments which method is invoked. When optional arguments are defined in terms of a global variable, the type of the optional argument may even change at run-time."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Keyword arguments behave quite differently from ordinary positional arguments. In particular, they do not participate in method dispatch. Methods are dispatched based only on positional arguments, with keyword arguments processed after the matching method is identified."},{"title":"Function-like objects","page":"Methods","location":"manual/methods.html#Function-like-objects","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Methods are associated with types, so it is possible to make any arbitrary Julia object \"callable\" by adding methods to its type."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"For example, you can define a type that stores the coefficients of a polynomial, but behaves like a function evaluating the polynomial:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> struct Polynomial{R}\n           coeffs::Vector{R}\n       end\n\njulia> function (p::Polynomial)(x)\n           v = p.coeffs[end]\n           for i = (length(p.coeffs)-1):-1:1\n               v = v*x + p.coeffs[i]\n           end\n           return v\n       end\n\njulia> (p::Polynomial)() = p(5)"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Notice that the function is specified by type instead of by name. As with normal functions there is a terse syntax form. In the function body, p will refer to the object that was called. A Polynomial can be used as follows:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> poly = Polynomial([1,10,100])\nPolynomial{Int64}([1, 10, 100])\n\njulia> poly(3)\n931\n\njulia> poly()\n2551"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"This mechanism is also the key to how type constructors and closures (inner functions that refer to their surrounding environment) work in Julia."},{"title":"Empty generic functions","page":"Methods","location":"manual/methods.html#Empty-generic-functions","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Occasionally it is useful to introduce a generic function without yet adding methods. This can be used to separate interface definitions from implementations. It might also be done for the purpose of documentation or code readability. The syntax for this is an empty function block without a tuple of arguments:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"function emptyfunc end"},{"title":"Method design and the avoidance of ambiguities","page":"Methods","location":"manual/methods.html#man-method-design-ambiguities","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Julia's method polymorphism is one of its most powerful features, yet exploiting this power can pose design challenges. In particular, in more complex method hierarchies it is not uncommon for ambiguities to arise."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Above, it was pointed out that one can resolve ambiguities like"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"f(x, y::Int) = 1\nf(x::Int, y) = 2"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"by defining a method"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"f(x::Int, y::Int) = 3"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"This is often the right strategy; however, there are circumstances where following this advice mindlessly can be counterproductive. In particular, the more methods a generic function has, the more possibilities there are for ambiguities. When your method hierarchies get more complicated than this simple example, it can be worth your while to think carefully about alternative strategies."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Below we discuss particular challenges and some alternative ways to resolve such issues."},{"title":"Tuple and NTuple arguments","page":"Methods","location":"manual/methods.html#Tuple-and-NTuple-arguments","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Tuple (and NTuple) arguments present special challenges. For example,"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"f(x::NTuple{N,Int}) where {N} = 1\nf(x::NTuple{N,Float64}) where {N} = 2"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"are ambiguous because of the possibility that N == 0: there are no elements to determine whether the Int or Float64 variant should be called. To resolve the ambiguity, one approach is define a method for the empty tuple:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"f(x::Tuple{}) = 3"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Alternatively, for all methods but one you can insist that there is at least one element in the tuple:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"f(x::NTuple{N,Int}) where {N} = 1           # this is the fallback\nf(x::Tuple{Float64, Vararg{Float64}}) = 2   # this requires at least one Float64"},{"title":"Orthogonalize your design","page":"Methods","location":"manual/methods.html#man-methods-orthogonalize","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"When you might be tempted to dispatch on two or more arguments, consider whether a \"wrapper\" function might make for a simpler design. For example, instead of writing multiple variants:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"f(x::A, y::A) = ...\nf(x::A, y::B) = ...\nf(x::B, y::A) = ...\nf(x::B, y::B) = ..."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"you might consider defining"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"f(x::A, y::A) = ...\nf(x, y) = f(g(x), g(y))"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"where g converts the argument to type A. This is a very specific example of the more general principle of orthogonal design, in which separate concepts are assigned to separate methods. Here, g will most likely need a fallback definition"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"g(x::A) = x"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"A related strategy exploits promote to bring x and y to a common type:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"f(x::T, y::T) where {T} = ...\nf(x, y) = f(promote(x, y)...)"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"One risk with this design is the possibility that if there is no suitable promotion method converting x and y to the same type, the second method will recurse on itself infinitely and trigger a stack overflow."},{"title":"Dispatch on one argument at a time","page":"Methods","location":"manual/methods.html#Dispatch-on-one-argument-at-a-time","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"If you need to dispatch on multiple arguments, and there are many fallbacks with too many combinations to make it practical to define all possible variants, then consider introducing a \"name cascade\" where (for example) you dispatch on the first argument and then call an internal method:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"f(x::A, y) = _fA(x, y)\nf(x::B, y) = _fB(x, y)"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Then the internal methods _fA and _fB can dispatch on y without concern about ambiguities with each other with respect to x."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Be aware that this strategy has at least one major disadvantage: in many cases, it is not possible for users to further customize the behavior of f by defining further specializations of your exported function f. Instead, they have to define specializations for your internal methods _fA and _fB, and this blurs the lines between exported and internal methods."},{"title":"Abstract containers and element types","page":"Methods","location":"manual/methods.html#Abstract-containers-and-element-types","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Where possible, try to avoid defining methods that dispatch on specific element types of abstract containers. For example,"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"-(A::AbstractArray{T}, b::Date) where {T<:Date}"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"generates ambiguities for anyone who defines a method"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"-(A::MyArrayType{T}, b::T) where {T}"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"The best approach is to avoid defining either of these methods: instead, rely on a generic method -(A::AbstractArray, b) and make sure this method is implemented with generic calls (like similar and -) that do the right thing for each container type and element type separately. This is just a more complex variant of the advice to orthogonalize your methods."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"When this approach is not possible, it may be worth starting a discussion with other developers about resolving the ambiguity; just because one method was defined first does not necessarily mean that it can't be modified or eliminated. As a last resort, one developer can define the \"band-aid\" method"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"-(A::MyArrayType{T}, b::Date) where {T<:Date} = ..."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"that resolves the ambiguity by brute force."},{"title":"Complex method \"cascades\" with default arguments","page":"Methods","location":"manual/methods.html#Complex-method-\"cascades\"-with-default-arguments","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"If you are defining a method \"cascade\" that supplies defaults, be careful about dropping any arguments that correspond to potential defaults. For example, suppose you're writing a digital filtering algorithm and you have a method that handles the edges of the signal by applying padding:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"function myfilter(A, kernel, ::Replicate)\n    Apadded = replicate_edges(A, size(kernel))\n    myfilter(Apadded, kernel)  # now perform the \"real\" computation\nend"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"This will run afoul of a method that supplies default padding:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"myfilter(A, kernel) = myfilter(A, kernel, Replicate()) # replicate the edge by default"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"Together, these two methods generate an infinite recursion with A constantly growing bigger."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"The better design would be to define your call hierarchy like this:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"struct NoPad end  # indicate that no padding is desired, or that it's already applied\n\nmyfilter(A, kernel) = myfilter(A, kernel, Replicate())  # default boundary conditions\n\nfunction myfilter(A, kernel, ::Replicate)\n    Apadded = replicate_edges(A, size(kernel))\n    myfilter(Apadded, kernel, NoPad())  # indicate the new boundary conditions\nend\n\n# other padding methods go here\n\nfunction myfilter(A, kernel, ::NoPad)\n    # Here's the \"real\" implementation of the core computation\nend"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"NoPad is supplied in the same argument position as any other kind of padding, so it keeps the dispatch hierarchy well organized and with reduced likelihood of ambiguities. Moreover, it extends the \"public\" myfilter interface: a user who wants to control the padding explicitly can call the NoPad variant directly."},{"title":"Defining methods in local scope","page":"Methods","location":"manual/methods.html#Defining-methods-in-local-scope","category":"section","text":""},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"You can define methods within a local scope, for example"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"julia> function f(x)\n           g(y::Int) = y + x\n           g(y) = y - x\n           g\n       end\nf (generic function with 1 method)\n\njulia> h = f(3);\n\njulia> h(4)\n7\n\njulia> h(4.0)\n1.0"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"However, you should not define local methods conditionally or subject to control flow, as in"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"function f2(inc)\n    if inc\n        g(x) = x + 1\n    else\n        g(x) = x - 1\n    end\nend\n\nfunction f3()\n    function g end\n    return g\n    g() = 0\nend"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"as it is not clear what function will end up getting defined. In the future, it might be an error to define local methods in this manner."},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"For cases like this use anonymous functions instead:"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"function f2(inc)\n    g = if inc\n        x -> x + 1\n    else\n        x -> x - 1\n    end\nend"},{"title":"Methods","page":"Methods","location":"manual/methods.html","category":"page","text":"[Clarke61]: Arthur C. Clarke, Profiles of the Future (1961): Clarke's Third Law."},{"title":"Multi-Threading","page":"Multi-Threading","location":"base/multi-threading.html#lib-multithreading","category":"section","text":""},{"title":"Base.Threads.@threads","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.@threads","category":"macro","text":"Threads.@threads [schedule] for ... end\n\nA macro to execute a for loop in parallel. The iteration space is distributed to coarse-grained tasks. This policy can be specified by the schedule argument. The execution of the loop waits for the evaluation of all iterations.\n\nTasks spawned by @threads are scheduled on the :default threadpool. This means that @threads will not use threads from the :interactive threadpool, even if called from the main thread or from a task in the interactive pool. The :default threadpool is intended for compute-intensive parallel workloads.\n\nSee also: @spawn and pmap in Distributed. For more information on threadpools, see the chapter on threadpools.\n\nExtended help\n\nSemantics\n\nUnless stronger guarantees are specified by the scheduling option, the loop executed by @threads macro have the following semantics.\n\nThe @threads macro executes the loop body in an unspecified order and potentially concurrently. It does not specify the exact assignments of the tasks and the worker threads. The assignments can be different for each execution. The loop body code (including any code transitively called from it) must not make any assumptions about the distribution of iterations to tasks or the worker thread in which they are executed. The loop body for each iteration must be able to make forward progress independent of other iterations and be free from data races. As such, invalid synchronizations across iterations may deadlock while unsynchronized memory accesses may result in undefined behavior.\n\nFor example, the above conditions imply that:\n\nA lock taken in an iteration must be released within the same iteration.\nCommunicating between iterations using blocking primitives like Channels is incorrect.\nWrite only to locations not shared across iterations (unless a lock or atomic operation is used).\nUnless the :static schedule is used, the value of threadid() may change even within a single iteration. See Task Migration.\n\nSchedulers\n\nWithout the scheduler argument, the exact scheduling is unspecified and varies across Julia releases. Currently, :dynamic is used when the scheduler is not specified.\n\ncompat: Julia 1.5\nThe schedule argument is available as of Julia 1.5.\n\n:dynamic (default)\n\n:dynamic scheduler executes iterations dynamically to available worker threads. Current implementation assumes that the workload for each iteration is uniform. However, this assumption may be removed in the future.\n\nThis scheduling option is merely a hint to the underlying execution mechanism. However, a few properties can be expected. The number of Tasks used by :dynamic scheduler is bounded by a small constant multiple of the number of available worker threads (Threads.threadpoolsize()). Each task processes contiguous regions of the iteration space. Thus, @threads :dynamic for x in xs; f(x); end is typically more efficient than @sync for x in xs; @spawn f(x); end if length(xs) is significantly larger than the number of the worker threads and the run-time of f(x) is relatively smaller than the cost of spawning and synchronizing a task (typically less than 10 microseconds).\n\ncompat: Julia 1.8\nThe :dynamic option for the schedule argument is available and the default as of Julia 1.8.\n\n:greedy\n\n:greedy scheduler spawns up to Threads.threadpoolsize() tasks, each greedily working on the given iterated values as they are produced. As soon as one task finishes its work, it takes the next value from the iterator. Work done by any individual task is not necessarily on contiguous values from the iterator. The given iterator may produce values forever, only the iterator interface is required (no indexing).\n\nThis scheduling option is generally a good choice if the workload of individual iterations is not uniform/has a large spread.\n\ncompat: Julia 1.11\nThe :greedy option for the schedule argument is available as of Julia 1.11.\n\n:static\n\n:static scheduler creates one task per thread and divides the iterations equally among them, assigning each task specifically to each thread. In particular, the value of threadid() is guaranteed to be constant within one iteration. Specifying :static is an error if used from inside another @threads loop or from a thread other than 1.\n\nnote: Note\n:static scheduling exists for supporting transition of code written before Julia 1.3. In newly written library functions, :static scheduling is discouraged because the functions using this option cannot be called from arbitrary worker threads.\n\nExamples\n\nTo illustrate of the different scheduling strategies, consider the following function busywait containing a non-yielding timed loop that runs for a given number of seconds.\n\njulia> function busywait(seconds)\n            tstart = time_ns()\n            while (time_ns() - tstart) / 1e9 < seconds\n            end\n        end\n\njulia> @time begin\n            Threads.@spawn busywait(5)\n            Threads.@threads :static for i in 1:Threads.threadpoolsize()\n                busywait(1)\n            end\n        end\n6.003001 seconds (16.33 k allocations: 899.255 KiB, 0.25% compilation time)\n\njulia> @time begin\n            Threads.@spawn busywait(5)\n            Threads.@threads :dynamic for i in 1:Threads.threadpoolsize()\n                busywait(1)\n            end\n        end\n2.012056 seconds (16.05 k allocations: 883.919 KiB, 0.66% compilation time)\n\nThe :dynamic example takes 2 seconds since one of the non-occupied threads is able to run two of the 1-second iterations to complete the for loop.\n\n\n\n\n\n"},{"title":"Base.Threads.foreach","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.foreach","category":"function","text":"Threads.foreach(f, channel::Channel;\n                schedule::Threads.AbstractSchedule=Threads.FairSchedule(),\n                ntasks=Threads.threadpoolsize())\n\nSimilar to foreach(f, channel), but iteration over channel and calls to f are split across ntasks tasks spawned by Threads.@spawn. This function will wait for all internally spawned tasks to complete before returning.\n\nIf schedule isa FairSchedule, Threads.foreach will attempt to spawn tasks in a manner that enables Julia's scheduler to more freely load-balance work items across threads. This approach generally has higher per-item overhead, but may perform better than StaticSchedule in concurrence with other multithreaded workloads.\n\nIf schedule isa StaticSchedule, Threads.foreach will spawn tasks in a manner that incurs lower per-item overhead than FairSchedule, but is less amenable to load-balancing. This approach thus may be more suitable for fine-grained, uniform workloads, but may perform worse than FairSchedule in concurrence with other multithreaded workloads.\n\nExamples\n\njulia> n = 20\n\njulia> c = Channel{Int}(ch -> foreach(i -> put!(ch, i), 1:n), 1)\n\njulia> d = Channel{Int}(n) do ch\n           f = i -> put!(ch, i^2)\n           Threads.foreach(f, c)\n       end\n\njulia> collect(d)\ncollect(d) = [1, 4, 9, 16, 25, 36, 49, 64, 81, 100, 121, 144, 169, 196, 225, 256, 289, 324, 361, 400]\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n"},{"title":"Base.Threads.@spawn","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.@spawn","category":"macro","text":"Threads.@spawn [:default|:interactive|:samepool] expr\n\nCreate a Task and schedule it to run on any available thread in the specified threadpool: :default, :interactive, or :samepool to use the same as the caller. :default is used if unspecified. The task is allocated to a thread once one becomes available. To wait for the task to finish, call wait on the result of this macro, or call fetch to wait and then obtain its return value.\n\nValues can be interpolated into @spawn via $, which copies the value directly into the constructed underlying closure. This allows you to insert the value of a variable, isolating the asynchronous code from changes to the variable's value in the current task.\n\nnote: Note\nThe thread that the task runs on may change if the task yields, therefore threadid() should not be treated as constant for a task. See Task Migration, and the broader multi-threading manual for further important caveats. See also the chapter on threadpools.\n\ncompat: Julia 1.3\nThis macro is available as of Julia 1.3.\n\ncompat: Julia 1.4\nInterpolating values via $ is available as of Julia 1.4.\n\ncompat: Julia 1.9\nA threadpool may be specified as of Julia 1.9.\n\ncompat: Julia 1.12\nThe same threadpool may be specified as of Julia 1.12.\n\nExamples\n\njulia> t() = println(\"Hello from \", Threads.threadid());\n\njulia> tasks = fetch.([Threads.@spawn t() for i in 1:4]);\nHello from 1\nHello from 1\nHello from 3\nHello from 4\n\n\n\n\n\n"},{"title":"Base.Threads.threadid","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.threadid","category":"function","text":"Threads.threadid([t::Task])::Int\n\nGet the ID number of the current thread of execution, or the thread of task t. The master thread has ID 1.\n\nExamples\n\njulia> Threads.threadid()\n1\n\njulia> Threads.@threads for i in 1:4\n          println(Threads.threadid())\n       end\n4\n2\n5\n4\n\njulia> Threads.threadid(Threads.@spawn \"foo\")\n2\n\nnote: Note\nThe thread that a task runs on may change if the task yields, which is known as Task Migration. For this reason in most cases it is not safe to use threadid([task]) to index into, say, a vector of buffers or stateful objects.\n\n\n\n\n\n"},{"title":"Base.Threads.maxthreadid","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.maxthreadid","category":"function","text":"Threads.maxthreadid()::Int\n\nGet a lower bound on the number of threads (across all thread pools) available to the Julia process, with atomic-acquire semantics. The result will always be greater than or equal to threadid() as well as threadid(task) for any task you were able to observe before calling maxthreadid.\n\n\n\n\n\n"},{"title":"Base.Threads.nthreads","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.nthreads","category":"function","text":"Threads.nthreads(:default | :interactive)::Int\n\nGet the current number of threads within the specified thread pool. The threads in :interactive have id numbers 1:nthreads(:interactive), and the threads in :default have id numbers in nthreads(:interactive) .+ (1:nthreads(:default)).\n\nSee also BLAS.get_num_threads and BLAS.set_num_threads in the LinearAlgebra standard library, and nprocs() in the Distributed standard library and Threads.maxthreadid().\n\n\n\n\n\n"},{"title":"Base.Threads.threadpool","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.threadpool","category":"function","text":"Threads.threadpool(tid = threadid())::Symbol\n\nReturns the specified thread's threadpool; either :default, :interactive, or :foreign.\n\n\n\n\n\n"},{"title":"Base.Threads.nthreadpools","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.nthreadpools","category":"function","text":"Threads.nthreadpools()::Int\n\nReturns the number of threadpools currently configured.\n\n\n\n\n\n"},{"title":"Base.Threads.threadpoolsize","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.threadpoolsize","category":"function","text":"Threads.threadpoolsize(pool::Symbol = :default)::Int\n\nGet the number of threads available to the default thread pool (or to the specified thread pool).\n\nSee also: BLAS.get_num_threads and BLAS.set_num_threads in the LinearAlgebra standard library, and nprocs() in the Distributed standard library.\n\n\n\n\n\n"},{"title":"Base.Threads.ngcthreads","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.ngcthreads","category":"function","text":"Threads.ngcthreads()::Int\n\nReturns the number of GC threads currently configured. This includes both mark threads and concurrent sweep threads.\n\n\n\n\n\n"},{"title":"Multi-Threading","page":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"See also Multi-Threading."},{"title":"Atomic operations","page":"Multi-Threading","location":"base/multi-threading.html#Atomic-operations","category":"section","text":""},{"title":"atomic","page":"Multi-Threading","location":"base/multi-threading.html#atomic","category":"keyword","text":"Unsafe pointer operations are compatible with loading and storing pointers declared with _Atomic and std::atomic type in C11 and C++23 respectively. An error may be thrown if there is not support for atomically loading the Julia type T.\n\nSee also: unsafe_load, unsafe_modify!, unsafe_replace!, unsafe_store!, unsafe_swap!\n\n\n\n\n\n"},{"title":"Base.@atomic","page":"Multi-Threading","location":"base/multi-threading.html#Base.@atomic","category":"macro","text":"@atomic var\n@atomic order ex\n\nMark var or ex as being performed atomically, if ex is a supported expression. If no order is specified it defaults to :sequentially_consistent.\n\n@atomic a.b.x = new\n@atomic a.b.x += addend\n@atomic :release a.b.x = new\n@atomic :acquire_release a.b.x += addend\n@atomic m[idx] = new\n@atomic m[idx] += addend\n@atomic :release m[idx] = new\n@atomic :acquire_release m[idx] += addend\n\nPerform the store operation expressed on the right atomically and return the new value.\n\nWith assignment (=), this operation translates to a setproperty!(a.b, :x, new) or, in case of reference, to a setindex_atomic!(m, order, new, idx) call, with order defaulting to :sequentially_consistent.\n\nWith any modifying operator this operation translates to a modifyproperty!(a.b, :x, op, addend)[2] or, in case of reference, to a modifyindex_atomic!(m, order, op, addend, idx...)[2] call, with order defaulting to :sequentially_consistent.\n\n@atomic a.b.x max arg2\n@atomic a.b.x + arg2\n@atomic max(a.b.x, arg2)\n@atomic :acquire_release max(a.b.x, arg2)\n@atomic :acquire_release a.b.x + arg2\n@atomic :acquire_release a.b.x max arg2\n@atomic m[idx] max arg2\n@atomic m[idx] + arg2\n@atomic max(m[idx], arg2)\n@atomic :acquire_release max(m[idx], arg2)\n@atomic :acquire_release m[idx] + arg2\n@atomic :acquire_release m[idx] max arg2\n\nPerform the binary operation expressed on the right atomically. Store the result into the field or the reference in the first argument, and return the values (old, new).\n\nThis operation translates to a modifyproperty!(a.b, :x, func, arg2) or, in case of reference to a modifyindex_atomic!(m, order, func, arg2, idx) call, with order defaulting to :sequentially_consistent.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n1\n\njulia> @atomic :sequentially_consistent a.x = 2 # set field x of a, with sequential consistency\n2\n\njulia> @atomic a.x += 1 # increment field x of a, with sequential consistency\n3\n\njulia> @atomic a.x + 1 # increment field x of a, with sequential consistency\n3 => 4\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n4\n\njulia> @atomic max(a.x, 10) # change field x of a to the max value, with sequential consistency\n4 => 10\n\njulia> @atomic a.x max 5 # again change field x of a to the max value, with sequential consistency\n10 => 10\n\njulia> mem = AtomicMemory{Int}(undef, 2);\n\njulia> @atomic mem[1] = 2 # set mem[1] to value 2 with sequential consistency\n2\n\njulia> @atomic :monotonic mem[1] # fetch the first value of mem, with monotonic consistency\n2\n\njulia> @atomic mem[1] += 1 # increment the first value of mem, with sequential consistency\n3\n\njulia> @atomic mem[1] + 1 # increment the first value of mem, with sequential consistency\n3 => 4\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n4\n\njulia> @atomic max(mem[1], 10) # change the first value of mem to the max value, with sequential consistency\n4 => 10\n\njulia> @atomic mem[1] max 5 # again change the first value of mem to the max value, with sequential consistency\n10 => 10\n\ncompat: Julia 1.7\nAtomic fields functionality requires at least Julia 1.7.\n\ncompat: Julia 1.12\nAtomic reference functionality requires at least Julia 1.12.\n\n\n\n\n\n"},{"title":"Base.@atomicswap","page":"Multi-Threading","location":"base/multi-threading.html#Base.@atomicswap","category":"macro","text":"@atomicswap a.b.x = new\n@atomicswap :sequentially_consistent a.b.x = new\n@atomicswap m[idx] = new\n@atomicswap :sequentially_consistent m[idx] = new\n\nStores new into a.b.x (m[idx] in case of reference) and returns the old value of a.b.x (the old value stored at m[idx], respectively).\n\nThis operation translates to a swapproperty!(a.b, :x, new) or, in case of reference, swapindex_atomic!(mem, order, new, idx) call, with order defaulting to :sequentially_consistent.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomicswap a.x = 2+2 # replace field x of a with 4, with sequential consistency\n1\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n4\n\njulia> mem = AtomicMemory{Int}(undef, 2);\n\njulia> @atomic mem[1] = 1;\n\njulia> @atomicswap mem[1] = 4 # replace the first value of `mem` with 4, with sequential consistency\n1\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n4\n\ncompat: Julia 1.7\nAtomic fields functionality requires at least Julia 1.7.\n\ncompat: Julia 1.12\nAtomic reference functionality requires at least Julia 1.12.\n\n\n\n\n\n"},{"title":"Base.@atomicreplace","page":"Multi-Threading","location":"base/multi-threading.html#Base.@atomicreplace","category":"macro","text":"@atomicreplace a.b.x expected => desired\n@atomicreplace :sequentially_consistent a.b.x expected => desired\n@atomicreplace :sequentially_consistent :monotonic a.b.x expected => desired\n@atomicreplace m[idx] expected => desired\n@atomicreplace :sequentially_consistent m[idx] expected => desired\n@atomicreplace :sequentially_consistent :monotonic m[idx] expected => desired\n\nPerform the conditional replacement expressed by the pair atomically, returning the values (old, success::Bool). Where success indicates whether the replacement was completed.\n\nThis operation translates to a replaceproperty!(a.b, :x, expected, desired) or, in case of reference, to a replaceindex_atomic!(mem, success_order, fail_order, expected, desired, idx) call, with both orders defaulting to :sequentially_consistent.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomicreplace a.x 1 => 2 # replace field x of a with 2 if it was 1, with sequential consistency\n(old = 1, success = true)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n2\n\njulia> @atomicreplace a.x 1 => 3 # replace field x of a with 2 if it was 1, with sequential consistency\n(old = 2, success = false)\n\njulia> xchg = 2 => 0; # replace field x of a with 0 if it was 2, with sequential consistency\n\njulia> @atomicreplace a.x xchg\n(old = 2, success = true)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n0\n\njulia> mem = AtomicMemory{Int}(undef, 2);\n\njulia> @atomic mem[1] = 1;\n\njulia> @atomicreplace mem[1] 1 => 2 # replace the first value of mem with 2 if it was 1, with sequential consistency\n(old = 1, success = true)\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n2\n\njulia> @atomicreplace mem[1] 1 => 3 # replace field x of a with 2 if it was 1, with sequential consistency\n(old = 2, success = false)\n\njulia> xchg = 2 => 0; # replace field x of a with 0 if it was 2, with sequential consistency\n\njulia> @atomicreplace mem[1] xchg\n(old = 2, success = true)\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n0\n\ncompat: Julia 1.7\nAtomic fields functionality requires at least Julia 1.7.\n\ncompat: Julia 1.12\nAtomic reference functionality requires at least Julia 1.12.\n\n\n\n\n\n"},{"title":"Base.@atomiconce","page":"Multi-Threading","location":"base/multi-threading.html#Base.@atomiconce","category":"macro","text":"@atomiconce a.b.x = value\n@atomiconce :sequentially_consistent a.b.x = value\n@atomiconce :sequentially_consistent :monotonic a.b.x = value\n@atomiconce m[idx] = value\n@atomiconce :sequentially_consistent m[idx] = value\n@atomiconce :sequentially_consistent :monotonic m[idx] = value\n\nPerform the conditional assignment of value atomically if it was previously unset. Returned value success::Bool indicates whether the assignment was completed.\n\nThis operation translates to a setpropertyonce!(a.b, :x, value) or, in case of reference, to a setindexonce_atomic!(m, success_order, fail_order, value, idx) call, with both orders defaulting to :sequentially_consistent.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct AtomicOnce\n           @atomic x\n           AtomicOnce() = new()\n       end\n\njulia> a = AtomicOnce()\nAtomicOnce(#undef)\n\njulia> @atomiconce a.x = 1 # set field x of a to 1, if unset, with sequential consistency\ntrue\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n1\n\njulia> @atomiconce :monotonic a.x = 2 # set field x of a to 1, if unset, with monotonic consistence\nfalse\n\njulia> mem = AtomicMemory{Vector{Int}}(undef, 1);\n\njulia> isassigned(mem, 1)\nfalse\n\njulia> @atomiconce mem[1] = [1] # set the first value of mem to [1], if unset, with sequential consistency\ntrue\n\njulia> isassigned(mem, 1)\ntrue\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n1-element Vector{Int64}:\n 1\n\njulia> @atomiconce :monotonic mem[1] = [2] # set the first value of mem to [2], if unset, with monotonic\nfalse\n\njulia> @atomic mem[1]\n1-element Vector{Int64}:\n 1\n\ncompat: Julia 1.11\nAtomic fields functionality requires at least Julia 1.11.\n\ncompat: Julia 1.12\nAtomic reference functionality requires at least Julia 1.12.\n\n\n\n\n\n"},{"title":"Core.AtomicMemory","page":"Multi-Threading","location":"base/multi-threading.html#Core.AtomicMemory","category":"type","text":"AtomicMemory{T} == GenericMemory{:atomic, T, Core.CPU}\n\nFixed-size DenseVector{T}. Fetching of any of its individual elements is performed atomically (with :monotonic ordering by default).\n\nwarning: Warning\nThe access to AtomicMemory must be done by either using the @atomic macro or the lower level interface functions: Base.getindex_atomic, Base.setindex_atomic!, Base.setindexonce_atomic!, Base.swapindex_atomic!, Base.modifyindex_atomic!, and Base.replaceindex_atomic!.\n\nFor details, see Atomic Operations as well as macros @atomic, @atomiconce, @atomicswap, and @atomicreplace.\n\ncompat: Julia 1.11\nThis type requires Julia 1.11 or later.\n\ncompat: Julia 1.12\nLower level interface functions or @atomic macro requires Julia 1.12 or later.\n\n\n\n\n\n"},{"title":"Multi-Threading","page":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"There are also optional memory ordering parameters for the unsafe set of functions, that select the C/C++-compatible versions of these atomic operations, if that parameter is specified to unsafe_load, unsafe_store!, unsafe_swap!, unsafe_replace!, and unsafe_modify!."},{"title":"Multi-Threading","page":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"warning: Warning\nThe following APIs are deprecated, though support for them is likely to remain for several releases."},{"title":"Base.Threads.Atomic","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.Atomic","category":"type","text":"Threads.Atomic{T}\n\nHolds a reference to an object of type T, ensuring that it is only accessed atomically, i.e. in a thread-safe manner.\n\nNew atomic objects can be created from a non-atomic values; if none is specified, the atomic object is initialized with zero.\n\nAtomic objects can be accessed using the [] notation:\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> x[] = 1\n1\n\njulia> x[]\n1\n\nAtomic operations use an atomic_ prefix, such as atomic_add!, atomic_xchg!, etc.\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_cas!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_cas!","category":"function","text":"Threads.atomic_cas!(x::Atomic{T}, cmp::T, newval::T) where T\n\nAtomically compare-and-set x\n\nAtomically compares the value in x with cmp. If equal, write newval to x. Otherwise, leaves x unmodified. Returns the old value in x. By comparing the returned value to cmp (via ===) one knows whether x was modified and now holds the new value newval.\n\nFor further details, see LLVM's cmpxchg instruction.\n\nThis function can be used to implement transactional semantics. Before the transaction, one records the value in x. After the transaction, the new value is stored only if x has not been modified in the mean time.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_cas!(x, 4, 2);\n\njulia> x\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_cas!(x, 3, 2);\n\njulia> x\nBase.Threads.Atomic{Int64}(2)\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_xchg!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_xchg!","category":"function","text":"Threads.atomic_xchg!(x::Atomic{T}, newval::T) where T\n\nAtomically exchange the value in x\n\nAtomically exchanges the value in x with newval. Returns the old value.\n\nFor further details, see LLVM's atomicrmw xchg instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_xchg!(x, 2)\n3\n\njulia> x[]\n2\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_add!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_add!","category":"function","text":"Threads.atomic_add!(x::Atomic{T}, val::T) where T <: ArithmeticTypes\n\nAtomically add val to x\n\nPerforms x[] += val atomically. Returns the old value. Not defined for Atomic{Bool}.\n\nFor further details, see LLVM's atomicrmw add instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_add!(x, 2)\n3\n\njulia> x[]\n5\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_sub!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_sub!","category":"function","text":"Threads.atomic_sub!(x::Atomic{T}, val::T) where T <: ArithmeticTypes\n\nAtomically subtract val from x\n\nPerforms x[] -= val atomically. Returns the old value. Not defined for Atomic{Bool}.\n\nFor further details, see LLVM's atomicrmw sub instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_sub!(x, 2)\n3\n\njulia> x[]\n1\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_and!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_and!","category":"function","text":"Threads.atomic_and!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-and x with val\n\nPerforms x[] &= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw and instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_and!(x, 2)\n3\n\njulia> x[]\n2\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_nand!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_nand!","category":"function","text":"Threads.atomic_nand!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-nand (not-and) x with val\n\nPerforms x[] = ~(x[] & val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw nand instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_nand!(x, 2)\n3\n\njulia> x[]\n-3\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_or!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_or!","category":"function","text":"Threads.atomic_or!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-or x with val\n\nPerforms x[] |= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw or instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_or!(x, 7)\n5\n\njulia> x[]\n7\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_xor!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_xor!","category":"function","text":"Threads.atomic_xor!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-xor (exclusive-or) x with val\n\nPerforms x[] $= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw xor instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_xor!(x, 7)\n5\n\njulia> x[]\n2\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_max!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_max!","category":"function","text":"Threads.atomic_max!(x::Atomic{T}, val::T) where T\n\nAtomically store the maximum of x and val in x\n\nPerforms x[] = max(x[], val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw max instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_max!(x, 7)\n5\n\njulia> x[]\n7\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_min!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_min!","category":"function","text":"Threads.atomic_min!(x::Atomic{T}, val::T) where T\n\nAtomically store the minimum of x and val in x\n\nPerforms x[] = min(x[], val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw min instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(7)\nBase.Threads.Atomic{Int64}(7)\n\njulia> Threads.atomic_min!(x, 5)\n7\n\njulia> x[]\n5\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_fence","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_fence","category":"function","text":"Threads.atomic_fence()\n\nInsert a sequential-consistency memory fence\n\nInserts a memory fence with sequentially-consistent ordering semantics. There are algorithms where this is needed, i.e. where an acquire/release ordering is insufficient.\n\nThis is likely a very expensive operation. Given that all other atomic operations in Julia already have acquire/release semantics, explicit fences should not be necessary in most cases.\n\nFor further details, see LLVM's fence instruction.\n\n\n\n\n\n"},{"title":"ccall using a libuv threadpool (Experimental)","page":"Multi-Threading","location":"base/multi-threading.html#ccall-using-a-libuv-threadpool-(Experimental)","category":"section","text":""},{"title":"Base.@threadcall","page":"Multi-Threading","location":"base/multi-threading.html#Base.@threadcall","category":"macro","text":"@threadcall((cfunc, clib), rettype, (argtypes...), argvals...)\n\nThe @threadcall macro is called in the same way as ccall but does the work in a different thread. This is useful when you want to call a blocking C function without causing the current julia thread to become blocked. Concurrency is limited by size of the libuv thread pool, which defaults to 4 threads but can be increased by setting the UV_THREADPOOL_SIZE environment variable and restarting the julia process.\n\nNote that the called function should never call back into Julia.\n\n\n\n\n\n"},{"title":"Low-level synchronization primitives","page":"Multi-Threading","location":"base/multi-threading.html#Low-level-synchronization-primitives","category":"section","text":""},{"title":"Multi-Threading","page":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"These building blocks are used to create the regular synchronization objects."},{"title":"Base.Threads.AbstractSpinLock","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.AbstractSpinLock","category":"type","text":"abstract type AbstractSpinLock <: AbstractLock end\n\nA non-reentrant, test-and-test-and-set spin lock. Recursive use will result in a deadlock. This kind of lock should only be used around code that takes little time to execute and does not block (e.g. perform I/O). In general, ReentrantLock should be used instead.\n\nEach lock must be matched with an unlock. If !islocked(lck::AbstractSpinLock) holds, trylock(lck) succeeds unless there are other tasks attempting to hold the lock \"at the same time.\"\n\nTest-and-test-and-set spin locks are quickest up to about 30ish contending threads. If you have more contention than that, different synchronization approaches should be considered.\n\n\n\n\n\n"},{"title":"Base.Threads.SpinLock","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.SpinLock","category":"type","text":"SpinLock() <: AbstractSpinLock\n\nSpinlocks are not padded, and so may suffer from false sharing. See also PaddedSpinLock.\n\nSee the documentation for AbstractSpinLock regarding correct usage.\n\n\n\n\n\n"},{"title":"Base.Threads.PaddedSpinLock","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.PaddedSpinLock","category":"type","text":"PaddedSpinLock() <: AbstractSpinLock\n\nPaddedSpinLocks are padded so that each is guaranteed to be on its own cache line, to avoid false sharing. See also SpinLock.\n\nSee the documentation for AbstractSpinLock regarding correct usage.\n\n\n\n\n\n"},{"title":"Task metrics (Experimental)","page":"Multi-Threading","location":"base/multi-threading.html#Task-metrics-(Experimental)","category":"section","text":""},{"title":"Base.Experimental.task_metrics","page":"Multi-Threading","location":"base/multi-threading.html#Base.Experimental.task_metrics","category":"function","text":"Base.Experimental.task_metrics(::Bool)\n\nEnable or disable the collection of per-task metrics. A Task created when Base.Experimental.task_metrics(true) is in effect will have Base.Experimental.task_running_time_ns and Base.Experimental.task_wall_time_ns timing information available.\n\nnote: Note\nTask metrics can be enabled at start-up via the --task-metrics=yes command line option.\n\n\n\n\n\n"},{"title":"Base.Experimental.task_running_time_ns","page":"Multi-Threading","location":"base/multi-threading.html#Base.Experimental.task_running_time_ns","category":"function","text":"Base.Experimental.task_running_time_ns(t::Task)::Union{UInt64, Nothing}\n\nReturn the total nanoseconds that the task t has spent running. This metric is only updated when t yields or completes unless t is the current task, in which it will be updated continuously. See also Base.Experimental.task_wall_time_ns.\n\nReturns nothing if task timings are not enabled. See Base.Experimental.task_metrics.\n\nnote: This metric is from the Julia scheduler\nA task may be running on an OS thread that is descheduled by the OS scheduler, this time still counts towards the metric.\n\ncompat: Julia 1.12\nThis method was added in Julia 1.12.\n\n\n\n\n\n"},{"title":"Base.Experimental.task_wall_time_ns","page":"Multi-Threading","location":"base/multi-threading.html#Base.Experimental.task_wall_time_ns","category":"function","text":"Base.Experimental.task_wall_time_ns(t::Task)::Union{UInt64, Nothing}\n\nReturn the total nanoseconds that the task t was runnable. This is the time since the task first entered the run queue until the time at which it completed, or until the current time if the task has not yet completed. See also Base.Experimental.task_running_time_ns.\n\nReturns nothing if task timings are not enabled. See Base.Experimental.task_metrics.\n\ncompat: Julia 1.12\nThis method was added in Julia 1.12.\n\n\n\n\n\n"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html#Metaprogramming","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The strongest legacy of Lisp in the Julia language is its metaprogramming support. Like Lisp, Julia represents its own code as a data structure of the language itself. Since code is represented by objects that can be created and manipulated from within the language, it is possible for a program to transform and generate its own code. This allows sophisticated code generation without extra build steps, and also allows true Lisp-style macros operating at the level of abstract syntax trees. In contrast, preprocessor \"macro\" systems, like that of C and C++, perform textual manipulation and substitution before any actual parsing or interpretation occurs. Because all data types and code in Julia are represented by Julia data structures, powerful reflection capabilities are available to explore the internals of a program and its types just like any other data."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"warning: Warning\nMetaprogramming is a powerful tool, but it introduces complexity that can make code more difficult to understand. For example, it can be surprisingly hard to get scope rules correct. Metaprogramming should typically be used only when other approaches such as higher order functions and closures cannot be applied.eval and defining new macros should be typically used as a last resort. It is almost never a good idea to use Meta.parse or convert an arbitrary string into Julia code. For manipulating Julia code, use the Expr data structure directly to avoid the complexity of how Julia syntax is parsed.The best uses of metaprogramming often implement most of their functionality in runtime helper functions, striving to minimize the amount of code they generate."},{"title":"Program representation","page":"Metaprogramming","location":"manual/metaprogramming.html#Program-representation","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Every Julia program starts life as a string:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> prog = \"1 + 1\"\n\"1 + 1\""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"What happens next?"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The next step is to parse each string into an object called an expression, represented by the Julia type Expr:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1 = Meta.parse(prog)\n:(1 + 1)\n\njulia> typeof(ex1)\nExpr"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Expr objects contain two parts:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"a Symbol identifying the kind of expression. A symbol is an interned string identifier (more discussion below)."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1.head\n:call"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"the expression arguments, which may be symbols, other expressions, or literal values:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1.args\n3-element Vector{Any}:\n  :+\n 1\n 1"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Expressions may also be constructed directly in prefix notation:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex2 = Expr(:call, :+, 1, 1)\n:(1 + 1)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The two expressions constructed above – by parsing and by direct construction – are equivalent:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1 == ex2\ntrue"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The key point here is that Julia code is internally represented as a data structure that is accessible from the language itself."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The dump function provides indented and annotated display of Expr objects:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> dump(ex2)\nExpr\n  head: Symbol call\n  args: Array{Any}((3,))\n    1: Symbol +\n    2: Int64 1\n    3: Int64 1"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Expr objects may also be nested:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex3 = Meta.parse(\"(4 + 4) / 2\")\n:((4 + 4) / 2)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Another way to view expressions is with Meta.show_sexpr, which displays the S-expression form of a given Expr, which may look very familiar to users of Lisp. Here's an example illustrating the display on a nested Expr:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> Meta.show_sexpr(ex3)\n(:call, :/, (:call, :+, 4, 4), 2)"},{"title":"Symbols","page":"Metaprogramming","location":"manual/metaprogramming.html#Symbols","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The : character has two syntactic purposes in Julia. The first form creates a Symbol, an interned string used as one building-block of expressions, from valid identifiers:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> s = :foo\n:foo\n\njulia> typeof(s)\nSymbol"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The Symbol constructor takes any number of arguments and creates a new symbol by concatenating their string representations together:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> :foo === Symbol(\"foo\")\ntrue\n\njulia> Symbol(\"1foo\") # `:1foo` would not work, as `1foo` is not a valid identifier\nSymbol(\"1foo\")\n\njulia> Symbol(\"func\",10)\n:func10\n\njulia> Symbol(:var,'_',\"sym\")\n:var_sym"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In the context of an expression, symbols are used to indicate access to variables; when an expression is evaluated, a symbol is replaced with the value bound to that symbol in the appropriate scope."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Sometimes extra parentheses around the argument to : are needed to avoid ambiguity in parsing:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> :(:)\n:(:)\n\njulia> :(::)\n:(::)"},{"title":"Expressions and evaluation","page":"Metaprogramming","location":"manual/metaprogramming.html#Expressions-and-evaluation","category":"section","text":""},{"title":"Quoting","page":"Metaprogramming","location":"manual/metaprogramming.html#Quoting","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The second syntactic purpose of the : character is to create expression objects without using the explicit Expr constructor. This is referred to as quoting. The : character, followed by paired parentheses around a single statement of Julia code, produces an Expr object based on the enclosed code. Here is an example of the short form used to quote an arithmetic expression:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = :(a+b*c+1)\n:(a + b * c + 1)\n\njulia> typeof(ex)\nExpr"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"(to view the structure of this expression, try ex.head and ex.args, or use dump as above or Meta.@dump)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note that equivalent expressions may be constructed using Meta.parse or the direct Expr form:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia>      :(a + b*c + 1)       ==\n       Meta.parse(\"a + b*c + 1\") ==\n       Expr(:call, :+, :a, Expr(:call, :*, :b, :c), 1)\ntrue"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Expressions provided by the parser generally only have symbols, other expressions, and literal values as their args, whereas expressions constructed by Julia code can have arbitrary run-time values without literal forms as args. In this specific example, + and a are symbols, *(b,c) is a subexpression, and 1 is a literal 64-bit signed integer."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"There is a second syntactic form of quoting for multiple expressions: blocks of code enclosed in quote ... end."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = quote\n           x = 1\n           y = 2\n           x + y\n       end\nquote\n    #= none:2 =#\n    x = 1\n    #= none:3 =#\n    y = 2\n    #= none:4 =#\n    x + y\nend\n\njulia> typeof(ex)\nExpr"},{"title":"Interpolation","page":"Metaprogramming","location":"manual/metaprogramming.html#man-expression-interpolation","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Direct construction of Expr objects with value arguments is powerful, but Expr constructors can be tedious compared to \"normal\" Julia syntax. As an alternative, Julia allows interpolation of literals or expressions into quoted expressions. Interpolation is indicated by a prefix $."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In this example, the value of variable a is interpolated:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> a = 1;\n\njulia> ex = :($a + b)\n:(1 + b)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Interpolating into an unquoted expression is not supported and will cause a compile-time error:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> $a + b\nERROR: syntax: \"$\" expression outside quote"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In this example, the tuple (1,2,3) is interpolated as an expression into a conditional test:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = :(a in $:((1,2,3)) )\n:(a in (1, 2, 3))"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The use of $ for expression interpolation is intentionally reminiscent of string interpolation and command interpolation. Expression interpolation allows convenient, readable programmatic construction of complex Julia expressions."},{"title":"Splatting interpolation","page":"Metaprogramming","location":"manual/metaprogramming.html#Splatting-interpolation","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Notice that the $ interpolation syntax allows inserting only a single expression into an enclosing expression. Occasionally, you have an array of expressions and need them all to become arguments of the surrounding expression. This can be done with the syntax $(xs...). For example, the following code generates a function call where the number of arguments is determined programmatically:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> args = [:x, :y, :z];\n\njulia> :(f(1, $(args...)))\n:(f(1, x, y, z))"},{"title":"Nested quote","page":"Metaprogramming","location":"manual/metaprogramming.html#Nested-quote","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Naturally, it is possible for quote expressions to contain other quote expressions. Understanding how interpolation works in these cases can be a bit tricky. Consider this example:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> x = :(1 + 2);\n\njulia> e = quote quote $x end end\nquote\n    #= none:1 =#\n    $(Expr(:quote, quote\n    #= none:1 =#\n    $(Expr(:$, :x))\nend))\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Notice that the result contains $x, which means that x has not been evaluated yet. In other words, the $ expression \"belongs to\" the inner quote expression, and so its argument is only evaluated when the inner quote expression is:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> eval(e)\nquote\n    #= none:1 =#\n    1 + 2\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"However, the outer quote expression is able to interpolate values inside the $ in the inner quote. This is done with multiple $s:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> e = quote quote $$x end end\nquote\n    #= none:1 =#\n    $(Expr(:quote, quote\n    #= none:1 =#\n    $(Expr(:$, :(1 + 2)))\nend))\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Notice that (1 + 2) now appears in the result instead of the symbol x. Evaluating this expression yields an interpolated 3:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> eval(e)\nquote\n    #= none:1 =#\n    3\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The intuition behind this behavior is that x is evaluated once for each $: one $ works similarly to eval(:x), giving x's value, while two $s do the equivalent of eval(eval(:x))."},{"title":"QuoteNode","page":"Metaprogramming","location":"manual/metaprogramming.html#man-quote-node","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The usual representation of a quote form in an AST is an Expr with head :quote:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> dump(Meta.parse(\":(1+2)\"))\nExpr\n  head: Symbol quote\n  args: Array{Any}((1,))\n    1: Expr\n      head: Symbol call\n      args: Array{Any}((3,))\n        1: Symbol +\n        2: Int64 1\n        3: Int64 2"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"As we have seen, such expressions support interpolation with $. However, in some situations it is necessary to quote code without performing interpolation. This kind of quoting does not yet have syntax, but is represented internally as an object of type QuoteNode:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> eval(Meta.quot(Expr(:$, :(1+2))))\n3\n\njulia> eval(QuoteNode(Expr(:$, :(1+2))))\n:($(Expr(:$, :(1 + 2))))"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The parser yields QuoteNodes for simple quoted items like symbols:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> dump(Meta.parse(\":x\"))\nQuoteNode\n  value: Symbol x"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"QuoteNode can also be used for certain advanced metaprogramming tasks."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note that while it does not support $, it also does not prevent it, nor does it preserve the identity of the wrapped object:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> b = 2; eval(Expr(:quote, QuoteNode(Expr(:$, :b))))\n:($(QuoteNode(2)))"},{"title":"Evaluating expressions","page":"Metaprogramming","location":"manual/metaprogramming.html#Evaluating-expressions","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Given an expression object, one can cause Julia to evaluate (execute) it at global scope using eval:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1 = :(1 + 2)\n:(1 + 2)\n\njulia> eval(ex1)\n3\n\njulia> ex = :(a + b)\n:(a + b)\n\njulia> eval(ex)\nERROR: UndefVarError: `b` not defined in `Main`\n[...]\n\njulia> a = 1; b = 2;\n\njulia> eval(ex)\n3"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Every module has its own eval function that evaluates expressions in its global scope. Expressions passed to eval are not limited to returning values – they can also have side-effects that alter the state of the enclosing module's environment:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = :(x = 1)\n:(x = 1)\n\njulia> x\nERROR: UndefVarError: `x` not defined in `Main`\n\njulia> eval(ex)\n1\n\njulia> x\n1"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here, the evaluation of an expression object causes a value to be assigned to the global variable x."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Since expressions are just Expr objects which can be constructed programmatically and then evaluated, it is possible to dynamically generate arbitrary code which can then be run using eval. Here is a simple example:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> a = 1;\n\njulia> ex = Expr(:call, :+, a, :b)\n:(1 + b)\n\njulia> a = 0; b = 2;\n\njulia> eval(ex)\n3"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The value of a is used to construct the expression ex which applies the + function to the value 1 and the variable b. Note the important distinction between the way a and b are used:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The value of the variable a at expression construction time is used as an immediate value in the expression. Thus, the value of a when the expression is evaluated no longer matters: the value in the expression is already 1, independent of whatever the value of a might be.\nOn the other hand, the symbol :b is used in the expression construction, so the value of the variable b at that time is irrelevant – :b is just a symbol and the variable b need not even be defined. At expression evaluation time, however, the value of the symbol :b is resolved by looking up the value of the variable b."},{"title":"Functions on Expressions","page":"Metaprogramming","location":"manual/metaprogramming.html#Functions-on-Expressions","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"As hinted above, one extremely useful feature of Julia is the capability to generate and manipulate Julia code within Julia itself. We have already seen one example of a function returning Expr objects: the Meta.parse function, which takes a string of Julia code and returns the corresponding Expr. A function can also take one or more Expr objects as arguments, and return another Expr. Here is a simple, motivating example:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> function math_expr(op, op1, op2)\n           expr = Expr(:call, op, op1, op2)\n           return expr\n       end\nmath_expr (generic function with 1 method)\n\njulia>  ex = math_expr(:+, 1, Expr(:call, :*, 4, 5))\n:(1 + 4 * 5)\n\njulia> eval(ex)\n21"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"As another example, here is a function that doubles any numeric argument, but leaves expressions alone:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> function make_expr2(op, opr1, opr2)\n           opr1f, opr2f = map(x -> isa(x, Number) ? 2*x : x, (opr1, opr2))\n           retexpr = Expr(:call, op, opr1f, opr2f)\n           return retexpr\n       end\nmake_expr2 (generic function with 1 method)\n\njulia> make_expr2(:+, 1, 2)\n:(2 + 4)\n\njulia> ex = make_expr2(:+, 1, Expr(:call, :*, 5, 8))\n:(2 + 5 * 8)\n\njulia> eval(ex)\n42"},{"title":"Macros","page":"Metaprogramming","location":"manual/metaprogramming.html#man-macros","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros provide a mechanism to include generated code in the final body of a program. A macro maps a tuple of arguments to a returned expression, and the resulting expression is compiled directly rather than requiring a runtime eval call. Macro arguments may include expressions, literal values, and symbols."},{"title":"Basics","page":"Metaprogramming","location":"manual/metaprogramming.html#Basics","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here is an extraordinarily simple macro:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro sayhello()\n           return :( println(\"Hello, world!\") )\n       end\n@sayhello (macro with 1 method)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros have a dedicated character in Julia's syntax: the @ (at-sign), followed by the unique name declared in a macro NAME ... end block. In this example, the compiler will replace all instances of @sayhello with:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":":( println(\"Hello, world!\") )"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"When @sayhello is entered in the REPL, the expression executes immediately, thus we only see the evaluation result:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @sayhello()\nHello, world!"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Now, consider a slightly more complex macro:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro sayhello(name)\n           return :( println(\"Hello, \", $name) )\n       end\n@sayhello (macro with 1 method)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This macro takes one argument: name. When @sayhello is encountered, the quoted expression is expanded to interpolate the value of the argument into the final expression:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @sayhello(\"human\")\nHello, human"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We can view the quoted return expression using the function macroexpand (important note: this is an extremely useful tool for debugging macros):"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = macroexpand(Main, :(@sayhello(\"human\")) )\n:(Main.println(\"Hello, \", \"human\"))\n\njulia> typeof(ex)\nExpr"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We can see that the \"human\" literal has been interpolated into the expression."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"There also exists a macro @macroexpand that is perhaps a bit more convenient than the macroexpand function:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @macroexpand @sayhello \"human\"\n:(println(\"Hello, \", \"human\"))"},{"title":"Hold up: why macros?","page":"Metaprogramming","location":"manual/metaprogramming.html#Hold-up:-why-macros?","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We have already seen a function f(::Expr...) -> Expr in a previous section. In fact, macroexpand is also such a function. So, why do macros exist?"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros are necessary because they execute when code is parsed, therefore, macros allow the programmer to generate and include fragments of customized code before the full program is run. To illustrate the difference, consider the following example:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro twostep(arg)\n           println(\"I execute at parse time. The argument is: \", arg)\n           return :(println(\"I execute at runtime. The argument is: \", $arg))\n       end\n@twostep (macro with 1 method)\n\njulia> ex = macroexpand(Main, :(@twostep :(1, 2, 3)) );\nI execute at parse time. The argument is: :((1, 2, 3))"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The first call to println is executed when macroexpand is called. The resulting expression contains only the second println:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> typeof(ex)\nExpr\n\njulia> ex\n:(println(\"I execute at runtime. The argument is: \", $(Expr(:copyast, :($(QuoteNode(:((1, 2, 3)))))))))\n\njulia> eval(ex)\nI execute at runtime. The argument is: (1, 2, 3)"},{"title":"Macro invocation","page":"Metaprogramming","location":"manual/metaprogramming.html#Macro-invocation","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros are invoked with the following general syntax:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"@name expr1 expr2 ...\n@name(expr1, expr2, ...)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note the distinguishing @ before the macro name and the lack of commas between the argument expressions in the first form, and the lack of whitespace after @name in the second form. The two styles should not be mixed. For example, the following syntax is different from the examples above; it passes the tuple (expr1, expr2, ...) as one argument to the macro:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"@name (expr1, expr2, ...)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"An alternative way to invoke a macro over an array literal (or comprehension) is to juxtapose both without using parentheses. In this case, the array will be the only expression fed to the macro. The following syntax is equivalent (and different from @name [a b] * v):"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"@name[a b] * v\n@name([a b]) * v"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"It is important to emphasize that macros receive their arguments as expressions, literals, or symbols. One way to explore macro arguments is to call the show function within the macro body:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro showarg(x)\n           show(x)\n           # ... remainder of macro, returning an expression\n       end\n@showarg (macro with 1 method)\n\njulia> @showarg(a)\n:a\n\njulia> @showarg(1+1)\n:(1 + 1)\n\njulia> @showarg(println(\"Yo!\"))\n:(println(\"Yo!\"))\n\njulia> @showarg(1)        # Numeric literal\n1\n\njulia> @showarg(\"Yo!\")    # String literal\n\"Yo!\"\n\njulia> @showarg(\"Yo! $(\"hello\")\")    # String with interpolation is an Expr rather than a String\n:(\"Yo! $(\"hello\")\")"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In addition to the given argument list, every macro is passed extra arguments named __source__ and __module__."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The argument __source__ provides information (in the form of a LineNumberNode object) about the parser location of the @ sign from the macro invocation. This allows macros to include better error diagnostic information, and is commonly used by logging, string-parser macros, and docs, for example, as well as to implement the @__LINE__, @__FILE__, and @__DIR__ macros."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The location information can be accessed by referencing __source__.line and __source__.file:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro __LOCATION__(); return QuoteNode(__source__); end\n@__LOCATION__ (macro with 1 method)\n\njulia> dump(\n            @__LOCATION__(\n       ))\nLineNumberNode\n  line: Int64 2\n  file: Symbol none"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The argument __module__ provides information (in the form of a Module object) about the expansion context of the macro invocation. This allows macros to look up contextual information, such as existing bindings, or to insert the value as an extra argument to a runtime function call doing self-reflection in the current module."},{"title":"Building an advanced macro","page":"Metaprogramming","location":"manual/metaprogramming.html#Building-an-advanced-macro","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here is a simplified definition of Julia's @assert macro:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro assert(ex)\n           return :( $ex ? nothing : throw(AssertionError($(string(ex)))) )\n       end\n@assert (macro with 1 method)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This macro can be used like this:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @assert 1 == 1.0\n\njulia> @assert 1 == 0\nERROR: AssertionError: 1 == 0"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In place of the written syntax, the macro call is expanded at parse time to its returned result. This is equivalent to writing:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"1 == 1.0 ? nothing : throw(AssertionError(\"1 == 1.0\"))\n1 == 0 ? nothing : throw(AssertionError(\"1 == 0\"))"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"That is, in the first call, the expression :(1 == 1.0) is spliced into the test condition slot, while the value of string(:(1 == 1.0)) is spliced into the assertion message slot. The entire expression, thus constructed, is placed into the syntax tree where the @assert macro call occurs. Then at execution time, if the test expression evaluates to true, then nothing is returned, whereas if the test is false, an error is raised indicating the asserted expression that was false. Notice that it would not be possible to write this as a function, since only the value of the condition is available and it would be impossible to display the expression that computed it in the error message."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The actual definition of @assert in Julia Base is more complicated. It allows the user to optionally specify their own error message, instead of just printing the failed expression. Just like in functions with a variable number of arguments (Varargs Functions), this is specified with an ellipses following the last argument:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro assert(ex, msgs...)\n           msg_body = isempty(msgs) ? ex : msgs[1]\n           msg = string(msg_body)\n           return :($ex ? nothing : throw(AssertionError($msg)))\n       end\n@assert (macro with 1 method)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Now @assert has two modes of operation, depending upon the number of arguments it receives! If there's only one argument, the tuple of expressions captured by msgs will be empty and it will behave the same as the simpler definition above. But now if the user specifies a second argument, it is printed in the message body instead of the failing expression. You can inspect the result of a macro expansion with the aptly named @macroexpand macro:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @macroexpand @assert a == b\n:(if Main.a == Main.b\n        Main.nothing\n    else\n        Main.throw(Main.AssertionError(\"a == b\"))\n    end)\n\njulia> @macroexpand @assert a==b \"a should equal b!\"\n:(if Main.a == Main.b\n        Main.nothing\n    else\n        Main.throw(Main.AssertionError(\"a should equal b!\"))\n    end)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"There is yet another case that the actual @assert macro handles: what if, in addition to printing \"a should equal b,\" we wanted to print their values? One might naively try to use string interpolation in the custom message, e.g., @assert a==b \"a ($a) should equal b ($b)!\", but this won't work as expected with the above macro. Can you see why? Recall from string interpolation that an interpolated string is rewritten to a call to string. Compare:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> typeof(:(\"a should equal b\"))\nString\n\njulia> typeof(:(\"a ($a) should equal b ($b)!\"))\nExpr\n\njulia> dump(:(\"a ($a) should equal b ($b)!\"))\nExpr\n  head: Symbol string\n  args: Array{Any}((5,))\n    1: String \"a (\"\n    2: Symbol a\n    3: String \") should equal b (\"\n    4: Symbol b\n    5: String \")!\""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"So now instead of getting a plain string in msg_body, the macro is receiving a full expression that will need to be evaluated in order to display as expected. This can be spliced directly into the returned expression as an argument to the string call; see error.jl for the complete implementation."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The @assert macro makes great use of splicing into quoted expressions to simplify the manipulation of expressions inside the macro body."},{"title":"Hygiene","page":"Metaprogramming","location":"manual/metaprogramming.html#Hygiene","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"An issue that arises in more complex macros is that of hygiene. In short, macros must ensure that the variables they introduce in their returned expressions do not accidentally clash with existing variables in the surrounding code they expand into. Conversely, the expressions that are passed into a macro as arguments are often expected to evaluate in the context of the surrounding code, interacting with and modifying the existing variables. Another concern arises from the fact that a macro may be called in a different module from where it was defined. In this case we need to ensure that all global variables are resolved to the correct module. Julia already has a major advantage over languages with textual macro expansion (like C) in that it only needs to consider the returned expression. All the other variables (such as msg in @assert above) follow the normal scoping block behavior."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"To demonstrate these issues, let us consider writing a @time macro that takes an expression as its argument, records the time, evaluates the expression, records the time again, prints the difference between the before and after times, and then has the value of the expression as its final value. The macro might look like this:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro time(ex)\n    return quote\n        local t0 = time_ns()\n        local val = $ex\n        local t1 = time_ns()\n        println(\"elapsed time: \", (t1-t0)/1e9, \" seconds\")\n        val\n    end\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here, we want t0, t1, and val to be private temporary variables, and we want time_ns to refer to the time_ns function in Julia Base, not to any time_ns variable the user might have (the same applies to println). Imagine the problems that could occur if the user expression ex also contained assignments to a variable called t0, or defined its own time_ns variable. We might get errors, or mysteriously incorrect behavior."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Julia's macro expander solves these problems in the following way. First, variables within a macro result are classified as either local or global. A variable is considered local if it is assigned to (and not declared global), declared local, or used as a function argument name. Otherwise, it is considered global. Local variables are then renamed to be unique (using the gensym function, which generates new symbols), and global variables are resolved within the macro definition environment. Therefore both of the above concerns are handled; the macro's locals will not conflict with any user variables, and time_ns and println will refer to the Julia Base definitions."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"One problem remains however. Consider the following use of this macro:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"module MyModule\nimport Base.@time\n\ntime_ns() = ... # compute something\n\n@time time_ns()\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here the user expression ex is a call to time_ns, but not the same time_ns function that the macro uses. It clearly refers to MyModule.time_ns. Therefore we must arrange for the code in ex to be resolved in the macro call environment. This is done by \"escaping\" the expression with esc:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro time(ex)\n    ...\n    local val = $(esc(ex))\n    ...\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"An expression wrapped in this manner is left alone by the macro expander and simply pasted into the output verbatim. Therefore it will be resolved in the macro call environment."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This escaping mechanism can be used to \"violate\" hygiene when necessary, in order to introduce or manipulate user variables. For example, the following macro sets x to zero in the call environment:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro zerox()\n           return esc(:(x = 0))\n       end\n@zerox (macro with 1 method)\n\njulia> function foo()\n           x = 1\n           @zerox\n           return x # is zero\n       end\nfoo (generic function with 1 method)\n\njulia> foo()\n0"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This kind of manipulation of variables should be used judiciously, but is occasionally quite handy."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Getting the hygiene rules correct can be a formidable challenge. Before using a macro, you might want to consider whether a function closure would be sufficient. Another useful strategy is to defer as much work as possible to runtime. For example, many macros simply wrap their arguments in a QuoteNode or other similar Expr. Some examples of this include @task body which simply returns schedule(Task(() -> $body)), and @eval expr, which simply returns eval(QuoteNode(expr))."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"To demonstrate, we might rewrite the @time example above as:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro time(expr)\n    return :(timeit(() -> $(esc(expr))))\nend\nfunction timeit(f)\n    t0 = time_ns()\n    val = f()\n    t1 = time_ns()\n    println(\"elapsed time: \", (t1-t0)/1e9, \" seconds\")\n    return val\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"However, we don't do this for a good reason: wrapping the expr in a new scope block (the anonymous function) also slightly changes the meaning of the expression (the scope of any variables in it), while we want @time to be usable with minimum impact on the wrapped code."},{"title":"Macros and dispatch","page":"Metaprogramming","location":"manual/metaprogramming.html#Macros-and-dispatch","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros, just like Julia functions, are generic. This means they can also have multiple method definitions, thanks to multiple dispatch:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro m end\n@m (macro with 0 methods)\n\njulia> macro m(args...)\n           println(\"$(length(args)) arguments\")\n       end\n@m (macro with 1 method)\n\njulia> macro m(x,y)\n           println(\"Two arguments\")\n       end\n@m (macro with 2 methods)\n\njulia> @m \"asd\"\n1 arguments\n\njulia> @m 1 2\nTwo arguments"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"However one should keep in mind, that macro dispatch is based on the types of AST that are handed to the macro, not the types that the AST evaluates to at runtime:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro m(::Int)\n           println(\"An Integer\")\n       end\n@m (macro with 3 methods)\n\njulia> @m 2\nAn Integer\n\njulia> x = 2\n2\n\njulia> @m x\n1 arguments"},{"title":"Code Generation","page":"Metaprogramming","location":"manual/metaprogramming.html#Code-Generation","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"When a significant amount of repetitive boilerplate code is required, it is common to generate it programmatically to avoid redundancy. In most languages, this requires an extra build step, and a separate program to generate the repetitive code. In Julia, expression interpolation and eval allow such code generation to take place in the normal course of program execution. For example, consider the following custom type"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"struct MyNumber\n    x::Float64\nend\n# output\n"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for which we want to add a number of methods to. We can do this programmatically in the following loop:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for op = (:sin, :cos, :tan, :log, :exp)\n    eval(quote\n        Base.$op(a::MyNumber) = MyNumber($op(a.x))\n    end)\nend\n# output\n"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"and we can now use those functions with our custom type:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> x = MyNumber(π)\nMyNumber(3.141592653589793)\n\njulia> sin(x)\nMyNumber(1.2246467991473532e-16)\n\njulia> cos(x)\nMyNumber(-1.0)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In this manner, Julia acts as its own preprocessor, and allows code generation from inside the language. The above code could be written slightly more tersely using the : prefix quoting form:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for op = (:sin, :cos, :tan, :log, :exp)\n    eval(:(Base.$op(a::MyNumber) = MyNumber($op(a.x))))\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This sort of in-language code generation, however, using the eval(quote(...)) pattern, is common enough that Julia comes with a macro to abbreviate this pattern:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for op = (:sin, :cos, :tan, :log, :exp)\n    @eval Base.$op(a::MyNumber) = MyNumber($op(a.x))\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The @eval macro rewrites this call to be precisely equivalent to the above longer versions. For longer blocks of generated code, the expression argument given to @eval can be a block:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"@eval begin\n    # multiple lines\nend"},{"title":"Non-Standard String Literals","page":"Metaprogramming","location":"manual/metaprogramming.html#meta-non-standard-string-literals","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Recall from Strings that string literals prefixed by an identifier are called non-standard string literals, and can have different semantics than un-prefixed string literals. For example:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"r\"^\\s*(?:#|$)\" produces a regular expression object rather than a string\nb\"DATA\\xff\\u2200\" is a byte array literal for [68,65,84,65,255,226,136,128]."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Perhaps surprisingly, these behaviors are not hard-coded into the Julia parser or compiler. Instead, they are custom behaviors provided by a general mechanism that anyone can use: prefixed string literals are parsed as calls to specially-named macros. For example, the regular expression macro is just the following:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro r_str(p)\n    Regex(p)\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"That's all. This macro says that the literal contents of the string literal r\"^\\s*(?:#|$)\" should be passed to the @r_str macro and the result of that expansion should be placed in the syntax tree where the string literal occurs. In other words, the expression r\"^\\s*(?:#|$)\" is equivalent to placing the following object directly into the syntax tree:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Regex(\"^\\\\s*(?:#|\\$)\")"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Not only is the string literal form shorter and far more convenient, but it is also more efficient: since the regular expression is compiled and the Regex object is actually created when the code is compiled, the compilation occurs only once, rather than every time the code is executed. Consider if the regular expression occurs in a loop:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for line = lines\n    m = match(r\"^\\s*(?:#|$)\", line)\n    if m === nothing\n        # non-comment\n    else\n        # comment\n    end\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Since the regular expression r\"^\\s*(?:#|$)\" is compiled and inserted into the syntax tree when this code is parsed, the expression is only compiled once instead of each time the loop is executed. In order to accomplish this without macros, one would have to write this loop like this:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"re = Regex(\"^\\\\s*(?:#|\\$)\")\nfor line = lines\n    m = match(re, line)\n    if m === nothing\n        # non-comment\n    else\n        # comment\n    end\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Moreover, if the compiler could not determine that the regex object was constant over all loops, certain optimizations might not be possible, making this version still less efficient than the more convenient literal form above. Of course, there are still situations where the non-literal form is more convenient: if one needs to interpolate a variable into the regular expression, one must take this more verbose approach; in cases where the regular expression pattern itself is dynamic, potentially changing upon each loop iteration, a new regular expression object must be constructed on each iteration. In the vast majority of use cases, however, regular expressions are not constructed based on run-time data. In this majority of cases, the ability to write regular expressions as compile-time values is invaluable."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The mechanism for user-defined string literals is deeply, profoundly powerful. Not only are Julia's non-standard literals implemented using it, but the command literal syntax (`echo \"Hello, $person\"`) is also implemented using the following innocuous-looking macro:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro cmd(str)\n    :(cmd_gen($(shell_parse(str)[1])))\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Of course, a large amount of complexity is hidden in the functions used in this macro definition, but they are just functions, written entirely in Julia. You can read their source and see precisely what they do – and all they do is construct expression objects to be inserted into your program's syntax tree."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Like string literals, command literals can also be prefixed by an identifier to form what are called non-standard command literals. These command literals are parsed as calls to specially-named macros. For example, the syntax custom`literal` is parsed as @custom_cmd \"literal\". Julia itself does not contain any non-standard command literals, but packages can make use of this syntax. Aside from the different syntax and the _cmd suffix instead of the _str suffix, non-standard command literals behave exactly like non-standard string literals."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In the event that two modules provide non-standard string or command literals with the same name, it is possible to qualify the string or command literal with a module name. For instance, if both Foo and Bar provide non-standard string literal @x_str, then one can write Foo.x\"literal\" or Bar.x\"literal\" to disambiguate between the two."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Another way to define a macro would be like this:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro foo_str(str, flag)\n    # do stuff\nend"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This macro can then be called with the following syntax:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"foo\"str\"flag"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The type of flag in the above mentioned syntax would be a String with contents of whatever trails after the string literal."},{"title":"Generated functions","page":"Metaprogramming","location":"manual/metaprogramming.html#Generated-functions","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"A very special macro is @generated, which allows you to define so-called generated functions. These have the capability to generate specialized code depending on the types of their arguments with more flexibility and/or less code than what can be achieved with multiple dispatch. While macros work with expressions at parse time and cannot access the types of their inputs, a generated function gets expanded at a time when the types of the arguments are known, but the function is not yet compiled."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Instead of performing some calculation or action, a generated function declaration returns a quoted expression which then forms the body for the method corresponding to the types of the arguments. When a generated function is called, the expression it returns is compiled and then run. To make this efficient, the result is usually cached. And to make this inferable, only a limited subset of the language is usable. Thus, generated functions provide a flexible way to move work from run time to compile time, at the expense of greater restrictions on allowed constructs."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"When defining generated functions, there are five main differences to ordinary functions:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"You annotate the function declaration with the @generated macro. This adds some information to the AST that lets the compiler know that this is a generated function.\nIn the body of the generated function you only have access to the types of the arguments – not their values.\nInstead of calculating something or performing some action, you return a quoted expression which, when evaluated, does what you want.\nGenerated functions are only permitted to call functions that were defined before the definition of the generated function. (Failure to follow this may result in getting MethodErrors referring to functions from a future world-age.)\nGenerated functions must not mutate or observe any non-constant global state (including, for example, IO, locks, non-local dictionaries, or using hasmethod). This means they can only read global constants, and cannot have any side effects. In other words, they must be completely pure. Due to an implementation limitation, this also means that they currently cannot define a closure or generator."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"It's easiest to illustrate this with an example. We can declare a generated function foo as"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated function foo(x)\n           Core.println(x)\n           return :(x * x)\n       end\nfoo (generic function with 1 method)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note that the body returns a quoted expression, namely :(x * x), rather than just the value of x * x."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"From the caller's perspective, this is identical to a regular function; in fact, you don't have to know whether you're calling a regular or generated function. Let's see how foo behaves:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> x = foo(2); # note: output is from println() statement in the body\nInt64\n\njulia> x           # now we print x\n4\n\njulia> y = foo(\"bar\");\nString\n\njulia> y\n\"barbar\""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"So, we see that in the body of the generated function, x is the type of the passed argument, and the value returned by the generated function, is the result of evaluating the quoted expression we returned from the definition, now with the value of x."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"What happens if we evaluate foo again with a type that we have already used?"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> foo(4)\n16"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note that there is no printout of Int64. We can see that the body of the generated function was only executed once here, for the specific set of argument types, and the result was cached. After that, for this example, the expression returned from the generated function on the first invocation was re-used as the method body. However, the actual caching behavior is an implementation-defined performance optimization, so it is invalid to depend too closely on this behavior."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The number of times a generated function is generated might be only once, but it might also be more often, or appear to not happen at all. As a consequence, you should never write a generated function with side effects - when, and how often, the side effects occur is undefined. (This is true for macros too - and just like for macros, the use of eval in a generated function is a sign that you're doing something the wrong way.) However, unlike macros, the runtime system cannot correctly handle a call to eval, so it is disallowed."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"It is also important to see how @generated functions interact with method redefinition. Following the principle that a correct @generated function must not observe any mutable state or cause any mutation of global state, we see the following behavior. Observe that the generated function cannot call any method that was not defined prior to the definition of the generated function itself."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Initially f(x) has one definition"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> f(x) = \"original definition\";"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Define other operations that use f(x):"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> g(x) = f(x);\n\njulia> @generated gen1(x) = f(x);\n\njulia> @generated gen2(x) = :(f(x));"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We now add some new definitions for f(x):"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> f(x::Int) = \"definition for Int\";\n\njulia> f(x::Type{Int}) = \"definition for Type{Int}\";"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"and compare how these results differ:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> f(1)\n\"definition for Int\"\n\njulia> g(1)\n\"definition for Int\"\n\njulia> gen1(1)\n\"original definition\"\n\njulia> gen2(1)\n\"definition for Int\""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Each method of a generated function has its own view of defined functions:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated gen1(x::Real) = f(x);\n\njulia> gen1(1)\n\"definition for Type{Int}\""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The example generated function foo above did not do anything a normal function foo(x) = x * x could not do (except printing the type on the first invocation, and incurring higher overhead). However, the power of a generated function lies in its ability to compute different quoted expressions depending on the types passed to it:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated function bar(x)\n           if x <: Integer\n               return :(x ^ 2)\n           else\n               return :(x)\n           end\n       end\nbar (generic function with 1 method)\n\njulia> bar(4)\n16\n\njulia> bar(\"baz\")\n\"baz\""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"(although of course this contrived example would be more easily implemented using multiple dispatch...)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Abusing this will corrupt the runtime system and cause undefined behavior:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated function baz(x)\n           if rand() < .9\n               return :(x^2)\n           else\n               return :(\"boo!\")\n           end\n       end\nbaz (generic function with 1 method)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Since the body of the generated function is non-deterministic, its behavior, and the behavior of all subsequent code is undefined."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Don't copy these examples!"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"These examples are hopefully helpful to illustrate how generated functions work, both in the definition end and at the call site; however, don't copy them, for the following reasons:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"the foo function has side-effects (the call to Core.println), and it is undefined exactly when, how often or how many times these side-effects will occur\nthe bar function solves a problem that is better solved with multiple dispatch - defining bar(x) = x and bar(x::Integer) = x ^ 2 will do the same thing, but it is both simpler and faster.\nthe baz function is pathological"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note that the set of operations that should not be attempted in a generated function is unbounded, and the runtime system can currently only detect a subset of the invalid operations. There are many other operations that will simply corrupt the runtime system without notification, usually in subtle ways not obviously connected to the bad definition. Because the function generator is run during inference, it must respect all of the limitations of that code."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Some operations that should not be attempted include:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Caching of native pointers.\nInteracting with the contents or methods of Core.Compiler in any way.\nObserving any mutable state.\nInference on the generated function may be run at any time, including while your code is attempting to observe or mutate this state.\nTaking any locks: C code you call out to may use locks internally, (for example, it is not problematic to call malloc, even though most implementations require locks internally) but don't attempt to hold or acquire any while executing Julia code.\nCalling any function that is defined after the body of the generated function. This condition is relaxed for incrementally-loaded precompiled modules to allow calling any function in the module."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Alright, now that we have a better understanding of how generated functions work, let's use them to build some more advanced (and valid) functionality..."},{"title":"An advanced example","page":"Metaprogramming","location":"manual/metaprogramming.html#An-advanced-example","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Julia's base library has an internal sub2ind function to calculate a linear index into an n-dimensional array, based on a set of n multilinear indices - in other words, to calculate the index i that can be used to index into an array A using A[i], instead of A[x,y,z,...]. One possible implementation is the following:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> function sub2ind_loop(dims::NTuple{N}, I::Integer...) where N\n           ind = I[N] - 1\n           for i = N-1:-1:1\n               ind = I[i]-1 + dims[i]*ind\n           end\n           return ind + 1\n       end;\n\njulia> sub2ind_loop((3, 5), 1, 2)\n4"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The same thing can be done using recursion:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> sub2ind_rec(dims::Tuple{}) = 1;\n\njulia> sub2ind_rec(dims::Tuple{}, i1::Integer, I::Integer...) =\n           i1 == 1 ? sub2ind_rec(dims, I...) : throw(BoundsError());\n\njulia> sub2ind_rec(dims::Tuple{Integer, Vararg{Integer}}, i1::Integer) = i1;\n\njulia> sub2ind_rec(dims::Tuple{Integer, Vararg{Integer}}, i1::Integer, I::Integer...) =\n           i1 + dims[1] * (sub2ind_rec(Base.tail(dims), I...) - 1);\n\njulia> sub2ind_rec((3, 5), 1, 2)\n4"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Both these implementations, although different, do essentially the same thing: a runtime loop over the dimensions of the array, collecting the offset in each dimension into the final index."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"However, all the information we need for the loop is embedded in the type information of the arguments. This allows the compiler to move the iteration to compile time and eliminate the runtime loops altogether. We can utilize generated functions to achieve a similar effect; in compiler parlance, we use generated functions to manually unroll the loop. The body becomes almost identical, but instead of calculating the linear index, we build up an expression that calculates the index:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end;\n\njulia> sub2ind_gen((3, 5), 1, 2)\n4"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"What code will this generate?"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"An easy way to find out is to extract the body into another (regular) function:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> function sub2ind_gen_impl(dims::Type{T}, I...) where T <: NTuple{N,Any} where N\n           length(I) == N || return :(error(\"partial indexing is unsupported\"))\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end;\n\njulia> @generated function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           return sub2ind_gen_impl(dims, I...)\n       end;\n\njulia> sub2ind_gen((3, 5), 1, 2)\n4"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We can now execute sub2ind_gen_impl and examine the expression it returns:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> sub2ind_gen_impl(Tuple{Int,Int}, Int, Int)\n:(((I[1] - 1) + dims[1] * (I[2] - 1)) + 1)"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"So, the method body that will be used here doesn't include a loop at all - just indexing into the two tuples, multiplication and addition/subtraction. All the looping is performed compile-time, and we avoid looping during execution entirely. Thus, we only loop once per type, in this case once per N (except in edge cases where the function is generated more than once - see disclaimer above)."},{"title":"Optionally-generated functions","page":"Metaprogramming","location":"manual/metaprogramming.html#Optionally-generated-functions","category":"section","text":""},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Generated functions can achieve high efficiency at run time, but come with a compile time cost: a new function body must be generated for every combination of concrete argument types. Typically, Julia is able to compile \"generic\" versions of functions that will work for any arguments, but with generated functions this is impossible. This means that programs making heavy use of generated functions might be impossible to statically compile."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"To solve this problem, the language provides syntax for writing normal, non-generated alternative implementations of generated functions. Applied to the sub2ind example above, it would look like this:"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> function sub2ind_gen_impl(dims::Type{T}, I...) where T <: NTuple{N,Any} where N\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end;\n\njulia> function sub2ind_gen_fallback(dims::NTuple{N}, I) where N\n           ind = I[N] - 1\n           for i = (N - 1):-1:1\n               ind = I[i] - 1 + dims[i]*ind\n           end\n           return ind + 1\n       end;\n\njulia> function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           length(I) == N || error(\"partial indexing is unsupported\")\n           if @generated\n               return sub2ind_gen_impl(dims, I...)\n           else\n               return sub2ind_gen_fallback(dims, I)\n           end\n       end;\n\njulia> sub2ind_gen((3, 5), 1, 2)\n4"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Internally, this code creates two implementations of the function: a generated one where the first block in if @generated is used, and a normal one where the else block is used. Inside the then part of the if @generated block, code has the same semantics as other generated functions: argument names refer to types, and the code should return an expression. Multiple if @generated blocks may occur, in which case the generated implementation uses all of the then blocks and the alternate implementation uses all of the else blocks."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Notice that we added an error check to the top of the function. This code will be common to both versions, and is run-time code in both versions (it will be quoted and returned as an expression from the generated version). That means that the values and types of local variables are not available at code generation time — the code-generation code can only see the types of arguments."},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In this style of definition, the code generation feature is essentially an optional optimization. The compiler will use it if convenient, but otherwise may choose to use the normal implementation instead. This style is preferred, since it allows the compiler to make more decisions and compile programs in more ways, and since normal code is more readable than code-generating code. However, which implementation is used depends on compiler implementation details, so it is essential for the two implementations to behave identically."},{"title":"Git workflow recommendations","page":"Git workflow recommendations","location":"devdocs/contributing/git-workflow.html#Git-workflow-recommendations","category":"section","text":""},{"title":"Git Recommendations For Pull Requests","page":"Git workflow recommendations","location":"devdocs/contributing/git-workflow.html#Git-Recommendations-For-Pull-Requests","category":"section","text":""},{"title":"Git workflow recommendations","page":"Git workflow recommendations","location":"devdocs/contributing/git-workflow.html","category":"page","text":"Avoid working from the master branch of your fork. Create a new branch as it will make it easier to update your pull request if Julia's master changes.\nTry to squash together small commits that make repeated changes to the same section of code, so your pull request is easier to review. A reasonable number of separate well-factored commits is fine, especially for larger changes.\nIf any conflicts arise due to changes in Julia's master, prefer updating your pull request branch with git rebase versus git merge or git pull, since the latter will introduce merge commits that clutter the git history with noise that makes your changes more difficult to review.\nDescriptive commit messages are good.\nUsing git add -p or git add -i can be useful to avoid accidentally committing unrelated changes.\nWhen linking to specific lines of code in discussion of an issue or pull request, hit the y key while viewing code on GitHub to reload the page with a URL that includes the specific version that you're viewing. That way any lines of code that you refer to will still make sense in the future, even if the content of the file changes.\nWhitespace can be automatically removed from existing commits with git rebase.\nTo remove whitespace for the previous commit, run git rebase --whitespace=fix HEAD~1.\nTo remove whitespace relative to the master branch, run git rebase --whitespace=fix master."},{"title":"Git Recommendations For Pull Request Reviewers","page":"Git workflow recommendations","location":"devdocs/contributing/git-workflow.html#Git-Recommendations-For-Pull-Request-Reviewers","category":"section","text":""},{"title":"Git workflow recommendations","page":"Git workflow recommendations","location":"devdocs/contributing/git-workflow.html","category":"page","text":"When merging, we generally like squash+merge. Unless it is the rare case of a PR with carefully staged individual commits that you want in the history separately, in which case merge is acceptable, but usually prefer squash+merge."},{"title":"Punctuation","page":"Punctuation","location":"base/punctuation.html#man-punctuation","category":"section","text":""},{"title":"Punctuation","page":"Punctuation","location":"base/punctuation.html","category":"page","text":"Extended documentation for mathematical symbols & functions is here."},{"title":"Punctuation","page":"Punctuation","location":"base/punctuation.html","category":"page","text":"symbol meaning\n@ the at-sign marks a macro invocation; optionally followed by an argument list\n! an exclamation mark is a prefix operator for logical negation (\"not\")\na! function names that end with an exclamation mark modify one or more of their arguments by convention\n# the number sign (or hash or pound) character begins single line comments\n#= when followed by an equals sign, it begins a multi-line comment (these are nestable)\n=# end a multi-line comment by immediately preceding the number sign with an equals sign\n$ the dollar sign is used for string and expression interpolation\n% the percent symbol is the remainder operator\n^ the caret is the exponentiation operator\n& single ampersand is bitwise and\n&& double ampersands is short-circuiting boolean and\n| single pipe character is bitwise or\n|| double pipe characters is short-circuiting boolean or\n⊻ the unicode xor character is bitwise exclusive or\n~ the tilde is an operator for bitwise not\n' a trailing apostrophe is the adjoint (that is, the complex transpose) operator Aᴴ\n* the asterisk is used for multiplication, including matrix multiplication and string concatenation\n/ forward slash divides the argument on its left by the one on its right\n// double forward slash performs exact, rational division\n\\ backslash operator divides the argument on its right by the one on its left, commonly used to solve matrix equations\n() parentheses with no arguments constructs an empty Tuple\n(a,...) parentheses with comma-separated arguments constructs a tuple containing its arguments\n(a=1,...) parentheses with comma-separated assignments constructs a NamedTuple\n(x;y) parentheses can also be used to group one or more semicolon separated expressions\na[] array indexing (calling getindex or setindex!)\n[,] vector literal constructor (calling vect)\n[;] vertical concatenation (calling vcat or hvcat)\n[    ] with space-separated expressions, horizontal concatenation (calling hcat or hvcat)\nT{ } curly braces following a type list that type's parameters\n{} curly braces can also be used to group multiple where expressions in function declarations\n; semicolons separate statements, begin a list of keyword arguments in function declarations or calls, or are used to separate array literals for vertical concatenation\n, commas separate function arguments or tuple or array components\n? the question mark delimits the ternary conditional operator (used like: conditional ? if_true : if_false)\n\" \" the single double-quote character delimits String literals\n\"\"\" \"\"\" three double-quote characters delimits string literals that may contain \" and ignore leading indentation\n' ' the single-quote character delimits Char (that is, character) literals\n` ` the backtick character delimits external process (Cmd) literals\nA... triple periods are a postfix operator that \"splat\" their arguments' contents into many arguments of a function call or declare a varargs function that \"slurps\" up many arguments into a single tuple\na.b single periods access named fields in objects/modules (calling getproperty or setproperty!)\nf.() periods may also prefix parentheses (like f.(...)) or infix operators (like .+) to perform the function element-wise (calling broadcast)\na:b colons (:) used as a binary infix operator construct a range from a to b (inclusive) with fixed step size 1\na:s:b colons (:) used as a ternary infix operator construct a range from a to b (inclusive) with step size s\n: when used by themselves, Colons represent all indices within a dimension, frequently combined with indexing\n:: double-colons represent a type annotation or typeassert, depending on context, frequently used when declaring function arguments\n:( ) quoted expression\n:a Symbol a\n<: subtype operator\n>: supertype operator (reverse of subtype operator)\n= single equals sign is assignment\n== double equals sign is value equality comparison\n=== triple equals sign is programmatically identical equality comparison\n=> right arrow using an equals sign defines a Pair typically used to populate dictionaries\n-> right arrow using a hyphen defines an anonymous function on a single line\n|> pipe operator passes output from the left argument to input of the right argument, usually a function\n∘ function composition operator (typed with \\circ{tab}) combines two functions as though they are a single larger function\n_ underscores may be assigned values which will not be saved, often used to ignore multiple return values or create repetitive comprehensions"},{"title":"Talking to the compiler (the :meta mechanism)","page":"Talking to the compiler (the :meta mechanism)","location":"devdocs/meta.html#Talking-to-the-compiler-(the-:meta-mechanism)","category":"section","text":""},{"title":"Talking to the compiler (the :meta mechanism)","page":"Talking to the compiler (the :meta mechanism)","location":"devdocs/meta.html","category":"page","text":"In some circumstances, one might wish to provide hints or instructions that a given block of code has special properties: you might always want to inline it, or you might want to turn on special compiler optimization passes. Starting with version 0.4, Julia has a convention that these instructions can be placed inside a :meta expression, which is typically (but not necessarily) the first expression in the body of a function."},{"title":"Talking to the compiler (the :meta mechanism)","page":"Talking to the compiler (the :meta mechanism)","location":"devdocs/meta.html","category":"page","text":":meta expressions are created with macros. As an example, consider the implementation of the @inline macro:"},{"title":"Talking to the compiler (the :meta mechanism)","page":"Talking to the compiler (the :meta mechanism)","location":"devdocs/meta.html","category":"page","text":"macro inline(ex)\n    esc(isa(ex, Expr) ? pushmeta!(ex, :inline) : ex)\nend"},{"title":"Talking to the compiler (the :meta mechanism)","page":"Talking to the compiler (the :meta mechanism)","location":"devdocs/meta.html","category":"page","text":"Here, ex is expected to be an expression defining a function. A statement like this:"},{"title":"Talking to the compiler (the :meta mechanism)","page":"Talking to the compiler (the :meta mechanism)","location":"devdocs/meta.html","category":"page","text":"@inline function myfunction(x)\n    x*(x+3)\nend"},{"title":"Talking to the compiler (the :meta mechanism)","page":"Talking to the compiler (the :meta mechanism)","location":"devdocs/meta.html","category":"page","text":"gets turned into an expression like this:"},{"title":"Talking to the compiler (the :meta mechanism)","page":"Talking to the compiler (the :meta mechanism)","location":"devdocs/meta.html","category":"page","text":"quote\n    function myfunction(x)\n        Expr(:meta, :inline)\n        x*(x+3)\n    end\nend"},{"title":"Talking to the compiler (the :meta mechanism)","page":"Talking to the compiler (the :meta mechanism)","location":"devdocs/meta.html","category":"page","text":"Base.pushmeta!(ex, tag::Union{Symbol,Expr}) appends :tag to the end of the :meta expression, creating a new :meta expression if necessary."},{"title":"Talking to the compiler (the :meta mechanism)","page":"Talking to the compiler (the :meta mechanism)","location":"devdocs/meta.html","category":"page","text":"To use the metadata, you have to parse these :meta expressions. If your implementation can be performed within Julia, Base.popmeta! is very handy: Base.popmeta!(body, :symbol) will scan a function body expression (one without the function signature) for the first :meta expression containing :symbol, extract any arguments, and return a tuple (found::Bool, args::Array{Any}). If the metadata did not have any arguments, or :symbol was not found, the args array will be empty."},{"title":"Talking to the compiler (the :meta mechanism)","page":"Talking to the compiler (the :meta mechanism)","location":"devdocs/meta.html","category":"page","text":"Not yet provided is a convenient infrastructure for parsing :meta expressions from C++."},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#conversion-and-promotion","category":"section","text":""},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Julia has a system for promoting arguments of mathematical operators to a common type, which has been mentioned in various other sections, including Integers and Floating-Point Numbers, Mathematical Operations and Elementary Functions, Types, and Methods. In this section, we explain how this promotion system works, as well as how to extend it to new types and apply it to functions besides built-in mathematical operators. Traditionally, programming languages fall into two camps with respect to promotion of arithmetic arguments:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Automatic promotion for built-in arithmetic types and operators. In most languages, built-in numeric types, when used as operands to arithmetic operators with infix syntax, such as +, -, *, and /, are automatically promoted to a common type to produce the expected results. C, Java, Perl, and Python, to name a few, all correctly compute the sum 1 + 1.5 as the floating-point value 2.5, even though one of the operands to + is an integer. These systems are convenient and designed carefully enough that they are generally all-but-invisible to the programmer: hardly anyone consciously thinks of this promotion taking place when writing such an expression, but compilers and interpreters must perform conversion before addition since integers and floating-point values cannot be added as-is. Complex rules for such automatic conversions are thus inevitably part of specifications and implementations for such languages.\nNo automatic promotion. This camp includes Ada and ML – very \"strict\" statically typed languages. In these languages, every conversion must be explicitly specified by the programmer. Thus, the example expression 1 + 1.5 would be a compilation error in both Ada and ML. Instead one must write real(1) + 1.5, explicitly converting the integer 1 to a floating-point value before performing addition. Explicit conversion everywhere is so inconvenient, however, that even Ada has some degree of automatic conversion: integer literals are promoted to the expected integer type automatically, and floating-point literals are similarly promoted to appropriate floating-point types."},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"In a sense, Julia falls into the \"no automatic promotion\" category: mathematical operators are just functions with special syntax, and the arguments of functions are never automatically converted. However, one may observe that applying mathematical operations to a wide variety of mixed argument types is just an extreme case of polymorphic multiple dispatch – something which Julia's dispatch and type systems are particularly well-suited to handle. \"Automatic\" promotion of mathematical operands simply emerges as a special application: Julia comes with pre-defined catch-all dispatch rules for mathematical operators, invoked when no specific implementation exists for some combination of operand types. These catch-all rules first promote all operands to a common type using user-definable promotion rules, and then invoke a specialized implementation of the operator in question for the resulting values, now of the same type. User-defined types can easily participate in this promotion system by defining methods for conversion to and from other types, and providing a handful of promotion rules defining what types they should promote to when mixed with other types."},{"title":"Conversion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Conversion","category":"section","text":""},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"The standard way to obtain a value of a certain type T is to call the type's constructor, T(x). However, there are cases where it's convenient to convert a value from one type to another without the programmer asking for it explicitly. One example is assigning a value into an array: if A is a Vector{Float64}, the expression A[1] = 2 should work by automatically converting the 2 from Int to Float64, and storing the result in the array. This is done via the convert function."},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"The convert function generally takes two arguments: the first is a type object and the second is a value to convert to that type. The returned value is the value converted to an instance of given type. The simplest way to understand this function is to see it in action:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"julia> x = 12\n12\n\njulia> typeof(x)\nInt64\n\njulia> xu = convert(UInt8, x)\n0x0c\n\njulia> typeof(xu)\nUInt8\n\njulia> xf = convert(AbstractFloat, x)\n12.0\n\njulia> typeof(xf)\nFloat64\n\njulia> a = Any[1 2 3; 4 5 6]\n2×3 Matrix{Any}:\n 1  2  3\n 4  5  6\n\njulia> convert(Array{Float64}, a)\n2×3 Matrix{Float64}:\n 1.0  2.0  3.0\n 4.0  5.0  6.0"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Conversion isn't always possible, in which case a MethodError is thrown indicating that convert doesn't know how to perform the requested conversion:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"julia> convert(AbstractFloat, \"foo\")\nERROR: MethodError: Cannot `convert` an object of type String to an object of type AbstractFloat\n[...]"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Some languages consider parsing strings as numbers or formatting numbers as strings to be conversions (many dynamic languages will even perform conversion for you automatically). This is not the case in Julia. Even though some strings can be parsed as numbers, most strings are not valid representations of numbers, and only a very limited subset of them are. Therefore in Julia the dedicated parse function must be used to perform this operation, making it more explicit."},{"title":"When is convert called?","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#When-is-convert-called?","category":"section","text":""},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"The following language constructs call convert:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Assigning to an array converts to the array's element type.\nAssigning to a field of an object converts to the declared type of the field.\nConstructing an object with new converts to the object's declared field types.\nAssigning to a variable with a declared type (e.g. local x::T) converts to that type.\nA function with a declared return type converts its return value to that type.\nPassing a value to ccall converts it to the corresponding argument type."},{"title":"Conversion vs. Construction","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Conversion-vs.-Construction","category":"section","text":""},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Note that the behavior of convert(T, x) appears to be nearly identical to T(x). Indeed, it usually is. However, there is a key semantic difference: since convert can be called implicitly, its methods are restricted to cases that are considered \"safe\" or \"unsurprising\". convert will only convert between types that represent the same basic kind of thing (e.g. different representations of numbers, or different string encodings). It is also usually lossless; converting a value to a different type and back again should result in the exact same value."},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"There are four general kinds of cases where constructors differ from convert:"},{"title":"Constructors for types unrelated to their arguments","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Constructors-for-types-unrelated-to-their-arguments","category":"section","text":""},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Some constructors don't implement the concept of \"conversion\". For example, Timer(2) creates a 2-second timer, which is not really a \"conversion\" from an integer to a timer."},{"title":"Mutable collections","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Mutable-collections","category":"section","text":""},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"convert(T, x) is expected to return the original x if x is already of type T. In contrast, if T is a mutable collection type then T(x) should always make a new collection (copying elements from x)."},{"title":"Wrapper types","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Wrapper-types","category":"section","text":""},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"For some types which \"wrap\" other values, the constructor may wrap its argument inside a new object even if it is already of the requested type. For example Some(x) wraps x to indicate that a value is present (in a context where the result might be a Some or nothing). However, x itself might be the object Some(y), in which case the result is Some(Some(y)), with two levels of wrapping. convert(Some, x), on the other hand, would just return x since it is already a Some."},{"title":"Constructors that don't return instances of their own type","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Constructors-that-don't-return-instances-of-their-own-type","category":"section","text":""},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"In very rare cases it might make sense for the constructor T(x) to return an object not of type T. This could happen if a wrapper type is its own inverse (e.g. Flip(Flip(x)) === x), or to support an old calling syntax for backwards compatibility when a library is restructured. But convert(T, x) should always return a value of type T."},{"title":"Defining New Conversions","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Defining-New-Conversions","category":"section","text":""},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"When defining a new type, initially all ways of creating it should be defined as constructors. If it becomes clear that implicit conversion would be useful, and that some constructors meet the above \"safety\" criteria, then convert methods can be added. These methods are typically quite simple, as they only need to call the appropriate constructor. Such a definition might look like this:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"import Base: convert\nconvert(::Type{MyType}, x) = MyType(x)"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"The type of the first argument of this method is Type{MyType}, the only instance of which is MyType. Thus, this method is only invoked when the first argument is the type value MyType. Notice the syntax used for the first argument: the argument name is omitted prior to the :: symbol, and only the type is given. This is the syntax in Julia for a function argument whose type is specified but whose value does not need to be referenced by name."},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"All instances of some abstract types are by default considered \"sufficiently similar\" that a universal convert definition is provided in Julia Base. For example, this definition states that it's valid to convert any Number type to any other by calling a 1-argument constructor:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"convert(::Type{T}, x::Number) where {T<:Number} = T(x)::T"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"This means that new Number types only need to define constructors, since this definition will handle convert for them. An identity conversion is also provided to handle the case where the argument is already of the requested type:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"convert(::Type{T}, x::T) where {T<:Number} = x"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Similar definitions exist for AbstractString, AbstractArray, and AbstractDict."},{"title":"Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Promotion","category":"section","text":""},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Promotion refers to converting values of mixed types to a single common type. Although it is not strictly necessary, it is generally implied that the common type to which the values are converted can faithfully represent all of the original values. In this sense, the term \"promotion\" is appropriate since the values are converted to a \"greater\" type – i.e. one which can represent all of the input values in a single common type. It is important, however, not to confuse this with object-oriented (structural) super-typing, or Julia's notion of abstract super-types: promotion has nothing to do with the type hierarchy, and everything to do with converting between alternate representations. For instance, although every Int32 value can also be represented as a Float64 value, Int32 is not a subtype of Float64."},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Promotion to a common \"greater\" type is performed in Julia by the promote function, which takes any number of arguments, and returns a tuple of the same number of values, converted to a common type, or throws an exception if promotion is not possible. The most common use case for promotion is to convert numeric arguments to a common type:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"julia> promote(1, 2.5)\n(1.0, 2.5)\n\njulia> promote(1, 2.5, 3)\n(1.0, 2.5, 3.0)\n\njulia> promote(2, 3//4)\n(2//1, 3//4)\n\njulia> promote(1, 2.5, 3, 3//4)\n(1.0, 2.5, 3.0, 0.75)\n\njulia> promote(1.5, im)\n(1.5 + 0.0im, 0.0 + 1.0im)\n\njulia> promote(1 + 2im, 3//4)\n(1//1 + 2//1*im, 3//4 + 0//1*im)"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Floating-point values are promoted to the largest of the floating-point argument types. Integer values are promoted to the largest of the integer argument types. If the types are the same size but differ in signedness, the unsigned type is chosen. Mixtures of integers and floating-point values are promoted to a floating-point type big enough to hold all the values. Integers mixed with rationals are promoted to rationals. Rationals mixed with floats are promoted to floats. Complex values mixed with real values are promoted to the appropriate kind of complex value."},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"That is really all there is to using promotions. The rest is just a matter of clever application, the most typical \"clever\" application being the definition of catch-all methods for numeric operations like the arithmetic operators +, -, * and /. Here are some of the catch-all method definitions given in promotion.jl:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"+(x::Number, y::Number) = +(promote(x,y)...)\n-(x::Number, y::Number) = -(promote(x,y)...)\n*(x::Number, y::Number) = *(promote(x,y)...)\n/(x::Number, y::Number) = /(promote(x,y)...)"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"These method definitions say that in the absence of more specific rules for adding, subtracting, multiplying and dividing pairs of numeric values, promote the values to a common type and then try again. That's all there is to it: nowhere else does one ever need to worry about promotion to a common numeric type for arithmetic operations – it just happens automatically. There are definitions of catch-all promotion methods for a number of other arithmetic and mathematical functions in promotion.jl, but beyond that, there are hardly any calls to promote required in Julia Base. The most common usages of promote occur in outer constructors methods, provided for convenience, to allow constructor calls with mixed types to delegate to an inner type with fields promoted to an appropriate common type. For example, recall that rational.jl provides the following outer constructor method:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Rational(n::Integer, d::Integer) = Rational(promote(n,d)...)"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"This allows calls like the following to work:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"julia> x = Rational(Int8(15),Int32(-5))\n-3//1\n\njulia> typeof(x)\nRational{Int32}"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"For most user-defined types, it is better practice to require programmers to supply the expected types to constructor functions explicitly, but sometimes, especially for numeric problems, it can be convenient to do promotion automatically."},{"title":"Defining Promotion Rules","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Defining-Promotion-Rules","category":"section","text":""},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Although one could, in principle, define methods for the promote function directly, this would require many redundant definitions for all possible permutations of argument types. Instead, the behavior of promote is defined in terms of an auxiliary function called promote_rule, which one can provide methods for. The promote_rule function takes a pair of type objects and returns another type object, such that instances of the argument types will be promoted to the returned type. Thus, by defining the rule:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"import Base: promote_rule\npromote_rule(::Type{Float64}, ::Type{Float32}) = Float64"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"one declares that when 64-bit and 32-bit floating-point values are promoted together, they should be promoted to 64-bit floating-point. The promotion type does not need to be one of the argument types. For example, the following promotion rules both occur in Julia Base:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"promote_rule(::Type{BigInt}, ::Type{Float64}) = BigFloat\npromote_rule(::Type{BigInt}, ::Type{Int8}) = BigInt"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"In the latter case, the result type is BigInt since BigInt is the only type large enough to hold integers for arbitrary-precision integer arithmetic. Also note that one does not need to define both promote_rule(::Type{A}, ::Type{B}) and promote_rule(::Type{B}, ::Type{A}) – the symmetry is implied by the way promote_rule is used in the promotion process."},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"The promote_rule function is used as a building block to define a second function called promote_type, which, given any number of type objects, returns the common type to which those values, as arguments to promote should be promoted. Thus, if one wants to know, in absence of actual values, what type a collection of values of certain types would promote to, one can use promote_type:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"julia> promote_type(Int8, Int64)\nInt64"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Note that we do not overload promote_type directly: we overload promote_rule instead. promote_type uses promote_rule, and adds the symmetry. Overloading it directly can cause ambiguity errors. We overload promote_rule to define how things should be promoted, and we use promote_type to query that."},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Internally, promote_type is used inside of promote to determine what type argument values should be converted to for promotion. The curious reader can read the code in promotion.jl, which defines the complete promotion mechanism in about 35 lines."},{"title":"Case Study: Rational Promotions","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Case-Study:-Rational-Promotions","category":"section","text":""},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"Finally, we finish off our ongoing case study of Julia's rational number type, which makes relatively sophisticated use of the promotion mechanism with the following promotion rules:"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"import Base: promote_rule\npromote_rule(::Type{Rational{T}}, ::Type{S}) where {T<:Integer,S<:Integer} = Rational{promote_type(T,S)}\npromote_rule(::Type{Rational{T}}, ::Type{Rational{S}}) where {T<:Integer,S<:Integer} = Rational{promote_type(T,S)}\npromote_rule(::Type{Rational{T}}, ::Type{S}) where {T<:Integer,S<:AbstractFloat} = promote_type(T,S)"},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"The first rule says that promoting a rational number with any other integer type promotes to a rational type whose numerator/denominator type is the result of promotion of its numerator/denominator type with the other integer type. The second rule applies the same logic to two different types of rational numbers, resulting in a rational of the promotion of their respective numerator/denominator types. The third and final rule dictates that promoting a rational with a float results in the same type as promoting the numerator/denominator type with the float."},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html","category":"page","text":"This small handful of promotion rules, together with the type's constructors and the default convert method for numbers, are sufficient to make rational numbers interoperate completely naturally with all of Julia's other numeric types – integers, floating-point numbers, and complex numbers. By providing appropriate conversion methods and promotion rules in the same manner, any user-defined numeric type can interoperate just as naturally with Julia's predefined numerics."},{"title":"Strings","page":"Strings","location":"manual/strings.html#man-strings","category":"section","text":""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Strings are finite sequences of characters. Of course, the real trouble comes when one asks what a character is. The characters that English speakers are familiar with are the letters A, B, C, etc., together with numerals and common punctuation symbols. These characters are standardized together with a mapping to integer values between 0 and 127 by the ASCII standard. There are, of course, many other characters used in non-English languages, including variants of the ASCII characters with accents and other modifications, related scripts such as Cyrillic and Greek, and scripts completely unrelated to ASCII and English, including Arabic, Chinese, Hebrew, Hindi, Japanese, and Korean. The Unicode standard tackles the complexities of what exactly a character is, and is generally accepted as the definitive standard addressing this problem. Depending on your needs, you can either ignore these complexities entirely and just pretend that only ASCII characters exist, or you can write code that can handle any of the characters or encodings that one may encounter when handling non-ASCII text. Julia makes dealing with plain ASCII text simple and efficient, and handling Unicode is as simple and efficient as possible. In particular, you can write C-style string code to process ASCII strings, and they will work as expected, both in terms of performance and semantics. If such code encounters non-ASCII text, it will gracefully fail with a clear error message, rather than silently introducing corrupt results. When this happens, modifying the code to handle non-ASCII data is straightforward."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"There are a few noteworthy high-level features about Julia's strings:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"The built-in concrete type used for strings (and string literals) in Julia is String. This supports the full range of Unicode characters via the UTF-8 encoding. (A transcode function is provided to convert to/from other Unicode encodings.)\nAll string types are subtypes of the abstract type AbstractString, and external packages define additional AbstractString subtypes (e.g. for other encodings). If you define a function expecting a string argument, you should declare the type as AbstractString in order to accept any string type.\nLike C and Java, but unlike most dynamic languages, Julia has a first-class type for representing a single character, called AbstractChar. The built-in Char subtype of AbstractChar is a 32-bit primitive type that can represent any Unicode character (and which is based on the UTF-8 encoding).\nAs in Java, strings are immutable: the value of an AbstractString object cannot be changed. To construct a different string value, you construct a new string from parts of other strings.\nConceptually, a string is a partial function from indices to characters: for some index values, no character value is returned, and instead an exception is thrown. This allows for efficient indexing into strings by the byte index of an encoded representation rather than by a character index, which cannot be implemented both efficiently and simply for variable-width encodings of Unicode strings."},{"title":"Characters","page":"Strings","location":"manual/strings.html#man-characters","category":"section","text":""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"A Char value represents a single character: it is just a 32-bit primitive type with a special literal representation and appropriate arithmetic behaviors, and which can be converted to a numeric value representing a Unicode code point.  (Julia packages may define other subtypes of AbstractChar, e.g. to optimize operations for other text encodings.) Here is how Char values are input and shown (note that character literals are delimited with single quotes, not double quotes):"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> c = 'x'\n'x': ASCII/Unicode U+0078 (category Ll: Letter, lowercase)\n\njulia> typeof(c)\nChar"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"You can easily convert a Char to its integer value, i.e. code point:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> c = Int('x')\n120\n\njulia> typeof(c)\nInt64"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"On 32-bit architectures, typeof(c) will be Int32. You can convert an integer value back to a Char just as easily:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> Char(120)\n'x': ASCII/Unicode U+0078 (category Ll: Letter, lowercase)"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Not all integer values are valid Unicode code points, but for performance, the Char conversion does not check that every character value is valid. If you want to check that each converted value is a valid code point, use the isvalid function:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> Char(0x110000)\n'\\U110000': Unicode U+110000 (category In: Invalid, too high)\n\njulia> isvalid(Char, 0x110000)\nfalse"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"As of this writing, the valid Unicode code points are U+0000 through U+D7FF and U+E000 through U+10FFFF. These have not all been assigned intelligible meanings yet, nor are they necessarily interpretable by applications, but all of these values are considered to be valid Unicode characters."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"You can input any Unicode character in single quotes using \\u followed by up to four hexadecimal digits or \\U followed by up to eight hexadecimal digits (the longest valid value only requires six):"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> '\\u0'\n'\\0': ASCII/Unicode U+0000 (category Cc: Other, control)\n\njulia> '\\u78'\n'x': ASCII/Unicode U+0078 (category Ll: Letter, lowercase)\n\njulia> '\\u2200'\n'∀': Unicode U+2200 (category Sm: Symbol, math)\n\njulia> '\\U10ffff'\n'\\U10ffff': Unicode U+10FFFF (category Cn: Other, not assigned)"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Julia uses your system's locale and language settings to determine which characters can be printed as-is and which must be output using the generic, escaped \\u or \\U input forms. In addition to these Unicode escape forms, all of C's traditional escaped input forms can also be used:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> Int('\\0')\n0\n\njulia> Int('\\t')\n9\n\njulia> Int('\\n')\n10\n\njulia> Int('\\e')\n27\n\njulia> Int('\\x7f')\n127\n\njulia> Int('\\177')\n127"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"You can do comparisons and a limited amount of arithmetic with Char values:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> 'A' < 'a'\ntrue\n\njulia> 'A' <= 'a' <= 'Z'\nfalse\n\njulia> 'A' <= 'X' <= 'Z'\ntrue\n\njulia> 'x' - 'a'\n23\n\njulia> 'A' + 1\n'B': ASCII/Unicode U+0042 (category Lu: Letter, uppercase)"},{"title":"String Basics","page":"Strings","location":"manual/strings.html#String-Basics","category":"section","text":""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"String literals are delimited by double quotes or triple double quotes (not single quotes):"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> str = \"Hello, world.\\n\"\n\"Hello, world.\\n\"\n\njulia> \"\"\"Contains \"quote\" characters\"\"\"\n\"Contains \\\"quote\\\" characters\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Long lines in strings can be broken up by preceding the newline with a backslash (\\):"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> \"This is a long \\\n       line\"\n\"This is a long line\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"If you want to extract a character from a string, you index into it:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> str[begin]\n'H': ASCII/Unicode U+0048 (category Lu: Letter, uppercase)\n\njulia> str[1]\n'H': ASCII/Unicode U+0048 (category Lu: Letter, uppercase)\n\njulia> str[6]\n',': ASCII/Unicode U+002C (category Po: Punctuation, other)\n\njulia> str[end]\n'\\n': ASCII/Unicode U+000A (category Cc: Other, control)"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Many Julia objects, including strings, can be indexed with integers. The index of the first element (the first character of a string) is returned by firstindex(str), and the index of the last element (character) with lastindex(str). The keywords begin and end can be used inside an indexing operation as shorthand for the first and last indices, respectively, along the given dimension. String indexing, like most indexing in Julia, is 1-based: firstindex always returns 1 for any AbstractString. As we will see below, however, lastindex(str) is not in general the same as length(str) for a string, because some Unicode characters can occupy multiple \"code units\"."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"You can perform arithmetic and other operations with end, just like a normal value:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> str[end-1]\n'.': ASCII/Unicode U+002E (category Po: Punctuation, other)\n\njulia> str[end÷2]\n' ': ASCII/Unicode U+0020 (category Zs: Separator, space)"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Using an index less than begin (1) or greater than end raises an error:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> str[begin-1]\nERROR: BoundsError: attempt to access 14-codeunit String at index [0]\n[...]\n\njulia> str[end+1]\nERROR: BoundsError: attempt to access 14-codeunit String at index [15]\n[...]"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"You can also extract a substring using range indexing:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> str[4:9]\n\"lo, wo\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Notice that the expressions str[k] and str[k:k] do not give the same result:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> str[6]\n',': ASCII/Unicode U+002C (category Po: Punctuation, other)\n\njulia> str[6:6]\n\",\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"The former is a single character value of type Char, while the latter is a string value that happens to contain only a single character. In Julia these are very different things."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Range indexing makes a copy of the selected part of the original string. Alternatively, it is possible to create a view into a string using the type SubString. More simply, using the @views macro on a block of code converts all string slices into substrings. For example:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> str = \"long string\"\n\"long string\"\n\njulia> substr = SubString(str, 1, 4)\n\"long\"\n\njulia> typeof(substr)\nSubString{String}\n\njulia> @views typeof(str[1:4]) # @views converts slices to SubStrings\nSubString{String}"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Several standard functions like chop, chomp or strip return a SubString."},{"title":"Unicode and UTF-8","page":"Strings","location":"manual/strings.html#Unicode-and-UTF-8","category":"section","text":""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Julia fully supports Unicode characters and strings. As discussed above, in character literals, Unicode code points can be represented using Unicode \\u and \\U escape sequences, as well as all the standard C escape sequences. These can likewise be used to write string literals:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> s = \"\\u2200 x \\u2203 y\"\n\"∀ x ∃ y\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Whether these Unicode characters are displayed as escapes or shown as special characters depends on your terminal's locale settings and its support for Unicode. String literals are encoded using the UTF-8 encoding. UTF-8 is a variable-width encoding, meaning that not all characters are encoded in the same number of bytes (\"code units\"). In UTF-8, ASCII characters — i.e. those with code points less than 0x80 (128) – are encoded as they are in ASCII, using a single byte, while code points 0x80 and above are encoded using multiple bytes — up to four per character."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"String indices in Julia refer to code units (= bytes for UTF-8), the fixed-width building blocks that are used to encode arbitrary characters (code points). This means that not every index into a String is necessarily a valid index for a character. If you index into a string at such an invalid byte index, an error is thrown:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> s[1]\n'∀': Unicode U+2200 (category Sm: Symbol, math)\n\njulia> s[2]\nERROR: StringIndexError: invalid index [2], valid nearby indices [1]=>'∀', [4]=>' '\nStacktrace:\n[...]\n\njulia> s[3]\nERROR: StringIndexError: invalid index [3], valid nearby indices [1]=>'∀', [4]=>' '\nStacktrace:\n[...]\n\njulia> s[4]\n' ': ASCII/Unicode U+0020 (category Zs: Separator, space)"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"In this case, the character ∀ is a three-byte character, so the indices 2 and 3 are invalid and the next character's index is 4; this next valid index can be computed by nextind(s,1), and the next index after that by nextind(s,4) and so on."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Since end is always the last valid index into a collection, end-1 references an invalid byte index if the second-to-last character is multibyte."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> s[end-1]\n' ': ASCII/Unicode U+0020 (category Zs: Separator, space)\n\njulia> s[end-2]\nERROR: StringIndexError: invalid index [9], valid nearby indices [7]=>'∃', [10]=>' '\nStacktrace:\n[...]\n\njulia> s[prevind(s, end, 2)]\n'∃': Unicode U+2203 (category Sm: Symbol, math)"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"The first case works, because the last character y and the space are one-byte characters, whereas end-2 indexes into the middle of the ∃ multibyte representation. The correct way for this case is using prevind(s, lastindex(s), 2) or, if you're using that value to index into s you can write s[prevind(s, end, 2)] and end expands to lastindex(s)."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Extraction of a substring using range indexing also expects valid byte indices or an error is thrown:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> s[1:1]\n\"∀\"\n\njulia> s[1:2]\nERROR: StringIndexError: invalid index [2], valid nearby indices [1]=>'∀', [4]=>' '\nStacktrace:\n[...]\n\njulia> s[1:4]\n\"∀ \""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Because of variable-length encodings, the number of characters in a string (given by length(s)) is not always the same as the last index. If you iterate through the indices 1 through lastindex(s) and index into s, the sequence of characters returned when errors aren't thrown is the sequence of characters comprising the string s. Thus length(s) <= lastindex(s), since each character in a string must have its own index. The following is an inefficient and verbose way to iterate through the characters of s:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> for i = firstindex(s):lastindex(s)\n           try\n               println(s[i])\n           catch\n               # ignore the index error\n           end\n       end\n∀\n\nx\n\n∃\n\ny"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"The blank lines actually have spaces on them. Fortunately, the above awkward idiom is unnecessary for iterating through the characters in a string, since you can just use the string as an iterable object, no exception handling required:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> for c in s\n           println(c)\n       end\n∀\n\nx\n\n∃\n\ny"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"If you need to obtain valid indices for a string, you can use the nextind and prevind functions to increment/decrement to the next/previous valid index, as mentioned above. You can also use the eachindex function to iterate over the valid character indices:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> collect(eachindex(s))\n7-element Vector{Int64}:\n  1\n  4\n  5\n  6\n  7\n 10\n 11"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"To access the raw code units (bytes for UTF-8) of the encoding, you can use the codeunit(s,i) function, where the index i runs consecutively from 1 to ncodeunits(s). The codeunits(s) function returns an AbstractVector{UInt8} wrapper that lets you access these raw codeunits (bytes) as an array."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Strings in Julia can contain invalid UTF-8 code unit sequences. This convention allows to treat any byte sequence as a String. In such situations a rule is that when parsing a sequence of code units from left to right characters are formed by the longest sequence of 8-bit code units that matches the start of one of the following bit patterns (each x can be 0 or 1):"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"0xxxxxxx;\n110xxxxx 10xxxxxx;\n1110xxxx 10xxxxxx 10xxxxxx;\n11110xxx 10xxxxxx 10xxxxxx 10xxxxxx;\n10xxxxxx;\n11111xxx."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"In particular this means that overlong and too-high code unit sequences and prefixes thereof are treated as a single invalid character rather than multiple invalid characters. This rule may be best explained with an example:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> s = \"\\xc0\\xa0\\xe2\\x88\\xe2|\"\n\"\\xc0\\xa0\\xe2\\x88\\xe2|\"\n\njulia> foreach(display, s)\n'\\xc0\\xa0': [overlong] ASCII/Unicode U+0020 (category Zs: Separator, space)\n'\\xe2\\x88': Malformed UTF-8 (category Ma: Malformed, bad data)\n'\\xe2': Malformed UTF-8 (category Ma: Malformed, bad data)\n'|': ASCII/Unicode U+007C (category Sm: Symbol, math)\n\njulia> isvalid.(collect(s))\n4-element BitArray{1}:\n 0\n 0\n 0\n 1\n\njulia> s2 = \"\\xf7\\xbf\\xbf\\xbf\"\n\"\\U1fffff\"\n\njulia> foreach(display, s2)\n'\\U1fffff': Unicode U+1FFFFF (category In: Invalid, too high)"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"We can see that the first two code units in the string s form an overlong encoding of space character. It is invalid, but is accepted in a string as a single character. The next two code units form a valid start of a three-byte UTF-8 sequence. However, the fifth code unit \\xe2 is not its valid continuation. Therefore code units 3 and 4 are also interpreted as malformed characters in this string. Similarly code unit 5 forms a malformed character because | is not a valid continuation to it. Finally the string s2 contains one too high code point."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Julia uses the UTF-8 encoding by default, and support for new encodings can be added by packages. For example, the LegacyStrings.jl package implements UTF16String and UTF32String types. Additional discussion of other encodings and how to implement support for them is beyond the scope of this document for the time being. For further discussion of UTF-8 encoding issues, see the section below on byte array literals. The transcode function is provided to convert data between the various UTF-xx encodings, primarily for working with external data and libraries."},{"title":"Concatenation","page":"Strings","location":"manual/strings.html#man-concatenation","category":"section","text":""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"One of the most common and useful string operations is concatenation:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> greet = \"Hello\"\n\"Hello\"\n\njulia> whom = \"world\"\n\"world\"\n\njulia> string(greet, \", \", whom, \".\\n\")\n\"Hello, world.\\n\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"It's important to be aware of potentially dangerous situations such as concatenation of invalid UTF-8 strings. The resulting string may contain different characters than the input strings, and its number of characters may be lower than sum of numbers of characters of the concatenated strings, e.g.:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> a, b = \"\\xe2\\x88\", \"\\x80\"\n(\"\\xe2\\x88\", \"\\x80\")\n\njulia> c = string(a, b)\n\"∀\"\n\njulia> collect.([a, b, c])\n3-element Vector{Vector{Char}}:\n ['\\xe2\\x88']\n ['\\x80']\n ['∀']\n\njulia> length.([a, b, c])\n3-element Vector{Int64}:\n 1\n 1\n 1"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"This situation can happen only for invalid UTF-8 strings. For valid UTF-8 strings concatenation preserves all characters in strings and additivity of string lengths."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Julia also provides * for string concatenation:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> greet * \", \" * whom * \".\\n\"\n\"Hello, world.\\n\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"While * may seem like a surprising choice to users of languages that provide + for string concatenation, this use of * has precedent in mathematics, particularly in abstract algebra."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"In mathematics, + usually denotes a commutative operation, where the order of the operands does not matter. An example of this is matrix addition, where A + B == B + A for any matrices A and B that have the same shape. In contrast, * typically denotes a noncommutative operation, where the order of the operands does matter. An example of this is matrix multiplication, where in general A * B != B * A. As with matrix multiplication, string concatenation is noncommutative: greet * whom != whom * greet. As such, * is a more natural choice for an infix string concatenation operator, consistent with common mathematical use."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"More precisely, the set of all finite-length strings S together with the string concatenation operator * forms a free monoid (S, *). The identity element of this set is the empty string, \"\". Whenever a free monoid is not commutative, the operation is typically represented as \\cdot, *, or a similar symbol, rather than +, which as stated usually implies commutativity."},{"title":"Interpolation","page":"Strings","location":"manual/strings.html#string-interpolation","category":"section","text":""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Constructing strings using concatenation can become a bit cumbersome, however. To reduce the need for these verbose calls to string or repeated multiplications, Julia allows interpolation into string literals using $, as in Perl:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> greet = \"Hello\"; whom = \"world\";\n\njulia> \"$greet, $whom.\\n\"\n\"Hello, world.\\n\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"This is more readable and convenient and equivalent to the above string concatenation – the system rewrites this apparent single string literal into the call string(greet, \", \", whom, \".\\n\")."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"The shortest complete expression after the $ is taken as the expression whose value is to be interpolated into the string. Thus, you can interpolate any expression into a string using parentheses:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> \"1 + 2 = $(1 + 2)\"\n\"1 + 2 = 3\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Both concatenation and string interpolation call string to convert objects into string form. However, string actually just returns the output of print, so new types should add methods to print or show instead of string."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Most non-AbstractString objects are converted to strings closely corresponding to how they are entered as literal expressions:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> v = [1,2,3]\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> \"v: $v\"\n\"v: [1, 2, 3]\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"string is the identity for AbstractString and AbstractChar values, so these are interpolated into strings as themselves, unquoted and unescaped:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> c = 'x'\n'x': ASCII/Unicode U+0078 (category Ll: Letter, lowercase)\n\njulia> \"hi, $c\"\n\"hi, x\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"To include a literal $ in a string literal, escape it with a backslash:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> print(\"I have \\$100 in my account.\\n\")\nI have $100 in my account."},{"title":"Triple-Quoted String Literals","page":"Strings","location":"manual/strings.html#Triple-Quoted-String-Literals","category":"section","text":""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"When strings are created using triple-quotes (\"\"\"...\"\"\") they have some special behavior that can be useful for creating longer blocks of text."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"First, triple-quoted strings are also dedented to the level of the least-indented line. This is useful for defining strings within code that is indented. For example:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> str = \"\"\"\n           Hello,\n           world.\n         \"\"\"\n\"  Hello,\\n  world.\\n\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"In this case the final (empty) line before the closing \"\"\" sets the indentation level."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"The dedentation level is determined as the longest common starting sequence of spaces or tabs in all lines, excluding the line following the opening \"\"\" and lines containing only spaces or tabs (the line containing the closing \"\"\" is always included). Then for all lines, excluding the text following the opening \"\"\", the common starting sequence is removed (including lines containing only spaces and tabs if they start with this sequence), e.g.:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> \"\"\"    This\n         is\n           a test\"\"\"\n\"    This\\nis\\n  a test\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Next, if the opening \"\"\" is followed by a newline, the newline is stripped from the resulting string."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"\"\"\"hello\"\"\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"is equivalent to"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"\"\"\"\nhello\"\"\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"but"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"\"\"\"\n\nhello\"\"\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"will contain a literal newline at the beginning."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Stripping of the newline is performed after the dedentation. For example:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> \"\"\"\n         Hello,\n         world.\"\"\"\n\"Hello,\\nworld.\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"If the newline is removed using a backslash, dedentation will be respected as well:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> \"\"\"\n         Averylong\\\n         word\"\"\"\n\"Averylongword\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Trailing whitespace is left unaltered."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Triple-quoted string literals can contain \" characters without escaping."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Note that line breaks in literal strings, whether single- or triple-quoted, result in a newline (LF) character \\n in the string, even if your editor uses a carriage return \\r (CR) or CRLF combination to end lines. To include a CR in a string, use an explicit escape \\r; for example, you can enter the literal string \"a CRLF line ending\\r\\n\"."},{"title":"Common Operations","page":"Strings","location":"manual/strings.html#Common-Operations","category":"section","text":""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"You can lexicographically compare strings using the standard comparison operators:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> \"abracadabra\" < \"xylophone\"\ntrue\n\njulia> \"abracadabra\" == \"xylophone\"\nfalse\n\njulia> \"Hello, world.\" != \"Goodbye, world.\"\ntrue\n\njulia> \"1 + 2 = 3\" == \"1 + 2 = $(1 + 2)\"\ntrue"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"You can search for the index of a particular character using the findfirst and findlast functions:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> findfirst('o', \"xylophone\")\n4\n\njulia> findlast('o', \"xylophone\")\n7\n\njulia> findfirst('z', \"xylophone\")"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"You can start the search for a character at a given offset by using the functions findnext and findprev:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> findnext('o', \"xylophone\", 1)\n4\n\njulia> findnext('o', \"xylophone\", 5)\n7\n\njulia> findprev('o', \"xylophone\", 5)\n4\n\njulia> findnext('o', \"xylophone\", 8)"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"You can use the occursin function to check if a substring is found within a string:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> occursin(\"world\", \"Hello, world.\")\ntrue\n\njulia> occursin(\"o\", \"Xylophon\")\ntrue\n\njulia> occursin(\"a\", \"Xylophon\")\nfalse\n\njulia> occursin('o', \"Xylophon\")\ntrue"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"The last example shows that occursin can also look for a character literal."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Two other handy string functions are repeat and join:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> repeat(\".:Z:.\", 10)\n\".:Z:..:Z:..:Z:..:Z:..:Z:..:Z:..:Z:..:Z:..:Z:..:Z:.\"\n\njulia> join([\"apples\", \"bananas\", \"pineapples\"], \", \", \" and \")\n\"apples, bananas and pineapples\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Some other useful functions include:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"firstindex(str) gives the minimal (byte) index that can be used to index into str (always 1 for strings, not necessarily true for other containers).\nlastindex(str) gives the maximal (byte) index that can be used to index into str.\nlength(str) the number of characters in str.\nlength(str, i, j) the number of valid character indices in str from i to j.\nncodeunits(str) number of code units in a string.\ncodeunit(str, i) gives the code unit value in the string str at index i.\nthisind(str, i) given an arbitrary index into a string find the first index of the character into which the index points.\nnextind(str, i, n=1) find the start of the nth character starting after index i.\nprevind(str, i, n=1) find the start of the nth character starting before index i."},{"title":"Non-Standard String Literals","page":"Strings","location":"manual/strings.html#non-standard-string-literals","category":"section","text":""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"There are situations when you want to construct a string or use string semantics, but the behavior of the standard string construct is not quite what is needed. For these kinds of situations, Julia provides non-standard string literals. A non-standard string literal looks like a regular double-quoted string literal, but is immediately prefixed by an identifier, and may behave differently from a normal string literal."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Regular expressions, byte array literals, and version number literals, as described below, are some examples of non-standard string literals. Users and packages may also define new non-standard string literals. Further documentation is given in the Metaprogramming section."},{"title":"Regular Expressions","page":"Strings","location":"manual/strings.html#man-regex-literals","category":"section","text":""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Sometimes you are not looking for an exact string, but a particular pattern. For example, suppose you are trying to extract a single date from a large text file. You don’t know what that date is (that’s why you are searching for it), but you do know it will look something like YYYY-MM-DD. Regular expressions allow you to specify these patterns and search for them."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Julia uses version 2 of Perl-compatible regular expressions (regexes), as provided by the PCRE library (see the PCRE2 syntax description for more details). Regular expressions are related to strings in two ways: the obvious connection is that regular expressions are used to find regular patterns in strings; the other connection is that regular expressions are themselves input as strings, which are parsed into a state machine that can be used to efficiently search for patterns in strings. In Julia, regular expressions are input using non-standard string literals prefixed with various identifiers beginning with r. The most basic regular expression literal without any options turned on just uses r\"...\":"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> re = r\"^\\s*(?:#|$)\"\nr\"^\\s*(?:#|$)\"\n\njulia> typeof(re)\nRegex"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"To check if a regex matches a string, use occursin:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> occursin(r\"^\\s*(?:#|$)\", \"not a comment\")\nfalse\n\njulia> occursin(r\"^\\s*(?:#|$)\", \"# a comment\")\ntrue"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"As one can see here, occursin simply returns true or false, indicating whether a match for the given regex occurs in the string. Commonly, however, one wants to know not just whether a string matched, but also how it matched. To capture this information about a match, use the match function instead:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> match(r\"^\\s*(?:#|$)\", \"not a comment\")\n\njulia> match(r\"^\\s*(?:#|$)\", \"# a comment\")\nRegexMatch(\"#\")"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"If the regular expression does not match the given string, match returns nothing – a special value that does not print anything at the interactive prompt. Other than not printing, it is a completely normal value and you can test for it programmatically:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"m = match(r\"^\\s*(?:#|$)\", line)\nif m === nothing\n    println(\"not a comment\")\nelse\n    println(\"blank or comment\")\nend"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"If a regular expression does match, the value returned by match is a RegexMatch object. These objects record how the expression matches, including the substring that the pattern matches and any captured substrings, if there are any. This example only captures the portion of the substring that matches, but perhaps we want to capture any non-blank text after the comment character. We could do the following:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> m = match(r\"^\\s*(?:#\\s*(.*?)\\s*$)\", \"# a comment \")\nRegexMatch(\"# a comment \", 1=\"a comment\")"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"When calling match, you have the option to specify an index at which to start the search. For example:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> m = match(r\"[0-9]\",\"aaaa1aaaa2aaaa3\",1)\nRegexMatch(\"1\")\n\njulia> m = match(r\"[0-9]\",\"aaaa1aaaa2aaaa3\",6)\nRegexMatch(\"2\")\n\njulia> m = match(r\"[0-9]\",\"aaaa1aaaa2aaaa3\",11)\nRegexMatch(\"3\")"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"You can extract the following info from a RegexMatch object:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"the entire substring matched: m.match\nthe captured substrings as an array of strings: m.captures\nthe offset at which the whole match begins: m.offset\nthe offsets of the captured substrings as a vector: m.offsets"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"For when a capture doesn't match, instead of a substring, m.captures contains nothing in that position, and m.offsets has a zero offset (recall that indices in Julia are 1-based, so a zero offset into a string is invalid). Here is a pair of somewhat contrived examples:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> m = match(r\"(a|b)(c)?(d)\", \"acd\")\nRegexMatch(\"acd\", 1=\"a\", 2=\"c\", 3=\"d\")\n\njulia> m.match\n\"acd\"\n\njulia> m.captures\n3-element Vector{Union{Nothing, SubString{String}}}:\n \"a\"\n \"c\"\n \"d\"\n\njulia> m.offset\n1\n\njulia> m.offsets\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> m = match(r\"(a|b)(c)?(d)\", \"ad\")\nRegexMatch(\"ad\", 1=\"a\", 2=nothing, 3=\"d\")\n\njulia> m.match\n\"ad\"\n\njulia> m.captures\n3-element Vector{Union{Nothing, SubString{String}}}:\n \"a\"\n nothing\n \"d\"\n\njulia> m.offset\n1\n\njulia> m.offsets\n3-element Vector{Int64}:\n 1\n 0\n 2"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"It is convenient to have captures returned as an array so that one can use destructuring syntax to bind them to local variables. As a convenience, the RegexMatch object implements iterator methods that pass through to the captures field, so you can destructure the match object directly:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> first, second, third = m; first\n\"a\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Captures can also be accessed by indexing the RegexMatch object with the number or name of the capture group:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> m=match(r\"(?<hour>\\d+):(?<minute>\\d+)\",\"12:45\")\nRegexMatch(\"12:45\", hour=\"12\", minute=\"45\")\n\njulia> m[:minute]\n\"45\"\n\njulia> m[2]\n\"45\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Captures can be referenced in a substitution string when using replace by using \\n to refer to the nth capture group and prefixing the substitution string with s. Capture group 0 refers to the entire match object. Named capture groups can be referenced in the substitution with \\g<groupname>. For example:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> replace(\"first second\", r\"(\\w+) (?<agroup>\\w+)\" => s\"\\g<agroup> \\1\")\n\"second first\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Numbered capture groups can also be referenced as \\g<n> for disambiguation, as in:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> replace(\"a\", r\".\" => s\"\\g<0>1\")\n\"a1\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"You can modify the behavior of regular expressions by some combination of the flags i, m, s, and x after the closing double quote mark. These flags have the same meaning as they do in Perl, as explained in this excerpt from the perlre manpage:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"i   Do case-insensitive pattern matching.\n\n    If locale matching rules are in effect, the case map is taken\n    from the current locale for code points less than 255, and\n    from Unicode rules for larger code points. However, matches\n    that would cross the Unicode rules/non-Unicode rules boundary\n    (ords 255/256) will not succeed.\n\nm   Treat string as multiple lines. That is, change \"^\" and \"$\"\n    from matching the start or end of the string to matching the\n    start or end of any line anywhere within the string.\n\ns   Treat string as single line. That is, change \".\" to match any\n    character whatsoever, even a newline, which normally it would\n    not match.\n\n    Used together, as r\"\"ms, they let the \".\" match any character\n    whatsoever, while still allowing \"^\" and \"$\" to match,\n    respectively, just after and just before newlines within the\n    string.\n\nx   Tells the regular expression parser to ignore most whitespace\n    that is neither backslashed nor within a character class. You\n    can use this to break up your regular expression into\n    (slightly) more readable parts. The '#' character is also\n    treated as a metacharacter introducing a comment, just as in\n    ordinary code."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"For example, the following regex has all three flags turned on:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> r\"a+.*b+.*d$\"ism\nr\"a+.*b+.*d$\"ims\n\njulia> match(r\"a+.*b+.*d$\"ism, \"Goodbye,\\nOh, angry,\\nBad world\\n\")\nRegexMatch(\"angry,\\nBad world\")"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"The r\"...\" literal is constructed without interpolation and unescaping (except for quotation mark \" which still has to be escaped). Here is an example showing the difference from standard string literals:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> x = 10\n10\n\njulia> r\"$x\"\nr\"$x\"\n\njulia> \"$x\"\n\"10\"\n\njulia> r\"\\x\"\nr\"\\x\"\n\njulia> \"\\x\"\nERROR: ParseError:\n# Error @ none:1:2\n\"\\x\"\n#└┘ ── invalid hex escape sequence"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Triple-quoted regex strings, of the form r\"\"\"...\"\"\", are also supported (and may be convenient for regular expressions containing quotation marks or newlines)."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"The Regex() constructor may be used to create a valid regex string programmatically. This permits using the contents of string variables and other string operations when constructing the regex string. Any of the regex codes above can be used within the single string argument to Regex(). Here are some examples:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> using Dates\n\njulia> d = Date(1962,7,10)\n1962-07-10\n\njulia> regex_d = Regex(\"Day \" * string(day(d)))\nr\"Day 10\"\n\njulia> match(regex_d, \"It happened on Day 10\")\nRegexMatch(\"Day 10\")\n\njulia> name = \"Jon\"\n\"Jon\"\n\njulia> regex_name = Regex(\"[\\\"( ]\\\\Q$name\\\\E[\\\") ]\")  # interpolate value of name\nr\"[\\\"( ]\\QJon\\E[\\\") ]\"\n\njulia> match(regex_name, \" Jon \")\nRegexMatch(\" Jon \")\n\njulia> match(regex_name, \"[Jon]\") === nothing\ntrue"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Note the use of the \\Q...\\E escape sequence. All characters between the \\Q and the \\E are interpreted as literal characters. This is convenient for matching characters that would otherwise be regex metacharacters. However, caution is needed when using this feature together with string interpolation, since the interpolated string might itself contain the \\E sequence, unexpectedly terminating literal matching. User inputs need to be sanitized before inclusion in a regex."},{"title":"Byte Array Literals","page":"Strings","location":"manual/strings.html#man-byte-array-literals","category":"section","text":""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Another useful non-standard string literal is the byte-array string literal: b\"...\". This form lets you use string notation to express read only literal byte arrays – i.e. arrays of UInt8 values. The type of those objects is CodeUnits{UInt8, String}. The rules for byte array literals are the following:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"ASCII characters and ASCII escapes produce a single byte.\n\\x and octal escape sequences produce the byte corresponding to the escape value.\nUnicode escape sequences produce a sequence of bytes encoding that code point in UTF-8."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"There is some overlap between these rules since the behavior of \\x and octal escapes less than 0x80 (128) are covered by both of the first two rules, but here these rules agree. Together, these rules allow one to easily use ASCII characters, arbitrary byte values, and UTF-8 sequences to produce arrays of bytes. Here is an example using all three:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> b\"DATA\\xff\\u2200\"\n8-element Base.CodeUnits{UInt8, String}:\n 0x44\n 0x41\n 0x54\n 0x41\n 0xff\n 0xe2\n 0x88\n 0x80"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"The ASCII string \"DATA\" corresponds to the bytes 68, 65, 84, 65. \\xff produces the single byte 255. The Unicode escape \\u2200 is encoded in UTF-8 as the three bytes 226, 136, 128. Note that the resulting byte array does not correspond to a valid UTF-8 string:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> isvalid(\"DATA\\xff\\u2200\")\nfalse"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"As it was mentioned CodeUnits{UInt8, String} type behaves like read only array of UInt8 and if you need a standard vector you can convert it using Vector{UInt8}:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> x = b\"123\"\n3-element Base.CodeUnits{UInt8, String}:\n 0x31\n 0x32\n 0x33\n\njulia> x[1]\n0x31\n\njulia> x[1] = 0x32\nERROR: CanonicalIndexError: setindex! not defined for Base.CodeUnits{UInt8, String}\n[...]\n\njulia> Vector{UInt8}(x)\n3-element Vector{UInt8}:\n 0x31\n 0x32\n 0x33"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Also observe the significant distinction between \\xff and \\uff: the former escape sequence encodes the byte 255, whereas the latter escape sequence represents the code point 255, which is encoded as two bytes in UTF-8:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> b\"\\xff\"\n1-element Base.CodeUnits{UInt8, String}:\n 0xff\n\njulia> b\"\\uff\"\n2-element Base.CodeUnits{UInt8, String}:\n 0xc3\n 0xbf"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Character literals use the same behavior."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"For code points less than \\u80, it happens that the UTF-8 encoding of each code point is just the single byte produced by the corresponding \\x escape, so the distinction can safely be ignored. For the escapes \\x80 through \\xff as compared to \\u80 through \\uff, however, there is a major difference: the former escapes all encode single bytes, which – unless followed by very specific continuation bytes – do not form valid UTF-8 data, whereas the latter escapes all represent Unicode code points with two-byte encodings."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"If this is all extremely confusing, try reading \"The Absolute Minimum Every Software Developer Absolutely, Positively Must Know About Unicode and Character Sets\". It's an excellent introduction to Unicode and UTF-8, and may help alleviate some confusion regarding the matter."},{"title":"Version Number Literals","page":"Strings","location":"manual/strings.html#man-version-number-literals","category":"section","text":""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Version numbers can easily be expressed with non-standard string literals of the form v\"...\". Version number literals create VersionNumber objects which follow the specifications of semantic versioning 2.0.0-rc2, and therefore are composed of major, minor and patch numeric values, followed by pre-release and build alphanumeric annotations. For example, v\"0.2.1-rc1+win64\" is broken into major version 0, minor version 2, patch version 1, pre-release rc1 and build win64. When entering a version literal, everything except the major version number is optional, therefore e.g.  v\"0.2\" is equivalent to v\"0.2.0\" (with empty pre-release/build annotations), v\"2\" is equivalent to v\"2.0.0\", and so on."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"VersionNumber objects are mostly useful to easily and correctly compare two (or more) versions. For example, the constant VERSION holds Julia version number as a VersionNumber object, and therefore one can define some version-specific behavior using simple statements as:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"if v\"0.2\" <= VERSION < v\"0.3-\"\n    # do something specific to 0.2 release series\nend"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Note that in the above example the non-standard version number v\"0.3-\" is used, with a trailing -: this notation is a Julia extension of the standard, and it's used to indicate a version which is lower than any 0.3 release, including all of its pre-releases. So in the above example the code would only run with stable 0.2 versions, and exclude such versions as v\"0.3.0-rc1\". In order to also allow for unstable (i.e. pre-release) 0.2 versions, the lower bound check should be modified like this: v\"0.2-\" <= VERSION."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Another non-standard version specification extension allows one to use a trailing + to express an upper limit on build versions, e.g.  VERSION > v\"0.2-rc1+\" can be used to mean any version above 0.2-rc1 and any of its builds: it will return false for version v\"0.2-rc1+win64\" and true for v\"0.2-rc2\"."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"It is good practice to use such special versions in comparisons (particularly, the trailing - should always be used on upper bounds unless there's a good reason not to), but they must not be used as the actual version number of anything, as they are invalid in the semantic versioning scheme."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Besides being used for the VERSION constant, VersionNumber objects are widely used in the Pkg module, to specify packages versions and their dependencies."},{"title":"Raw String Literals","page":"Strings","location":"manual/strings.html#man-raw-string-literals","category":"section","text":""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Raw strings without interpolation or unescaping can be expressed with non-standard string literals of the form raw\"...\". Raw string literals create ordinary String objects which contain the enclosed contents exactly as entered with no interpolation or unescaping. This is useful for strings which contain code or markup in other languages which use $ or \\ as special characters."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"The exception is that quotation marks still must be escaped, e.g. raw\"\\\"\" is equivalent to \"\\\"\". To make it possible to express all strings, backslashes then also must be escaped, but only when appearing right before a quote character:"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> println(raw\"\\\\ \\\\\\\"\")\n\\\\ \\\""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"Notice that the first two backslashes appear verbatim in the output, since they do not precede a quote character. However, the next backslash character escapes the backslash that follows it, and the last backslash escapes a quote, since these backslashes appear before a quote."},{"title":"Annotated Strings","page":"Strings","location":"manual/strings.html#man-annotated-strings","category":"section","text":""},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"note: Note\nThe API for AnnotatedStrings is considered experimental and is subject to change between Julia versions."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"It is sometimes useful to be able to hold metadata relating to regions of a string. A AnnotatedString wraps another string and allows for regions of it to be annotated with labelled values (:label => value). All generic string operations are applied to the underlying string. However, when possible, styling information is preserved. This means you can manipulate a AnnotatedString —taking substrings, padding them, concatenating them with other strings— and the metadata annotations will \"come along for the ride\"."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"This string type is fundamental to the StyledStrings stdlib, which uses :face-labelled annotations to hold styling information."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"When concatenating a AnnotatedString, take care to use annotatedstring instead of string if you want to keep the string annotations."},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"julia> str = Base.AnnotatedString(\"hello there\",\n               [(1:5, :word, :greeting), (7:11, :label, 1)])\n\"hello there\"\n\njulia> length(str)\n11\n\njulia> lpad(str, 14)\n\"   hello there\"\n\njulia> typeof(lpad(str, 7))\nBase.AnnotatedString{String}\n\njulia> str2 = Base.AnnotatedString(\" julia\", [(2:6, :face, :magenta)])\n\" julia\"\n\njulia> Base.annotatedstring(str, str2)\n\"hello there julia\"\n\njulia> str * str2 == Base.annotatedstring(str, str2) # *-concatenation still works\ntrue"},{"title":"Strings","page":"Strings","location":"manual/strings.html","category":"page","text":"The annotations of a AnnotatedString can be accessed and modified via the annotations and annotate! functions."},{"title":"Unicode Input","page":"Unicode Input","location":"manual/unicode-input.html#Unicode-Input","category":"section","text":""},{"title":"Unicode Input","page":"Unicode Input","location":"manual/unicode-input.html","category":"page","text":"The following table lists Unicode characters that can be entered via tab completion of LaTeX-like abbreviations in the Julia REPL (and in various other editing environments). You can also get information on how to type a symbol by entering it in the REPL help, i.e. by typing ? and then entering the symbol in the REPL (e.g., by copy-paste from somewhere you saw the symbol)."},{"title":"Unicode Input","page":"Unicode Input","location":"manual/unicode-input.html","category":"page","text":"warning: Warning\nThis table may appear to contain missing characters in the second column, or even show characters that are inconsistent with the characters as they are rendered in the Julia REPL. In these cases, users are strongly advised to check their choice of fonts in their browser and REPL environment, as there are known issues with glyphs in many fonts."},{"title":"Unicode Input","page":"Unicode Input","location":"manual/unicode-input.html","category":"page","text":"#\n# Generate a table containing all LaTeX and Emoji tab completions available in the REPL.\n#\nimport REPL, Markdown\nconst NBSP = '\\u00A0'\n\nfunction tab_completions(symbols...)\n    completions = Dict{String, Vector{String}}()\n    for each in symbols, (k, v) in each\n        completions[v] = push!(get!(completions, v, String[]), k)\n    end\n    return completions\nend\n\nfunction unicode_data()\n    file = normpath(@__DIR__, \"..\", \"..\", \"..\", \"..\", \"..\", \"doc\", \"UnicodeData.txt\")\n    names = Dict{UInt32, String}()\n    open(file) do unidata\n        for line in readlines(unidata)\n            id, name, desc = split(line, \";\")[[1, 2, 11]]\n            codepoint = parse(UInt32, \"0x$id\")\n            names[codepoint] = titlecase(lowercase(\n                name == \"\" ? desc : desc == \"\" ? name : \"$name / $desc\"))\n        end\n    end\n    return names\nend\n\n# Surround combining characters with no-break spaces (i.e '\\u00A0'). Follows the same format\n# for how unicode is displayed on the unicode.org website:\n# https://util.unicode.org/UnicodeJsps/character.jsp?a=0300\nfunction fix_combining_chars(char)\n    cat = Base.Unicode.category_code(char)\n    return cat == 6 || cat == 8 ? \"$NBSP$char$NBSP\" : \"$char\"\nend\n\nfunction table_entries(completions, unicode_dict)\n    entries = Any[Any[\n        [\"Code point(s)\"],\n        [\"Character(s)\"],\n        [\"Tab completion sequence(s)\"],\n        [\"Unicode name(s)\"],\n    ]]\n    for (chars, inputs) in sort!(collect(completions), by = first)\n        code_points, unicode_names, characters = String[], String[], String[]\n        for char in chars\n            push!(code_points, \"U+$(uppercase(string(UInt32(char), base = 16, pad = 5)))\")\n            push!(unicode_names, get(unicode_dict, UInt32(char), \"(No Unicode name)\"))\n            push!(characters, isempty(characters) ? fix_combining_chars(char) : \"$char\")\n        end\n        inputs_md = []\n        for (i, input) in enumerate(inputs)\n            i > 1 && push!(inputs_md, \", \")\n            push!(inputs_md, Markdown.Code(\"\", input))\n        end\n        push!(entries, [\n            [join(code_points, \" + \")],\n            [join(characters)],\n            inputs_md,\n            [join(unicode_names, \" + \")],\n        ])\n    end\n    table = Markdown.Table(entries, [:l, :c, :l, :l])\n    # We also need to wrap the Table in a Markdown.MD \"document\"\n    return Markdown.MD([table])\nend\n\ntable_entries(\n    tab_completions(\n        REPL.REPLCompletions.latex_symbols,\n        REPL.REPLCompletions.emoji_symbols\n    ),\n    unicode_data()\n)"},{"title":"Contributing to patch releases","page":"Contributing to patch releases","location":"devdocs/contributing/patch-releases.html#Contributing-to-patch-releases","category":"section","text":""},{"title":"Contributing to patch releases","page":"Contributing to patch releases","location":"devdocs/contributing/patch-releases.html","category":"page","text":"The process of creating a patch release is roughly as follows:"},{"title":"Contributing to patch releases","page":"Contributing to patch releases","location":"devdocs/contributing/patch-releases.html","category":"page","text":"Create a new branch (e.g. backports-release-1.10) against the relevant minor release branch (e.g. release-1.10). Usually a corresponding pull request is created as well.\nAdd commits, nominally from master (hence \"backports\"), to that branch. See below for more information on this process.\nRun the BaseBenchmarks.jl benchmark suite and PkgEval.jl package ecosystem exerciser against that branch. Nominally BaseBenchmarks.jl and PkgEval.jl are invoked via Nanosoldier.jl from the pull request associated with the backports branch. Fix any issues.\nOnce all test and benchmark reports look good, merge the backports branch into the corresponding release branch (e.g. merge backports-release-1.10 into release-1.10).\nOpen a pull request that bumps the version of the relevant minor release to the next patch version, e.g. as in this pull request.\nPing @JuliaLang/releases to tag the patch release and update the website.\nOpen a pull request that bumps the version of the relevant minor release to the next prerelease patch version, e.g. as in this pull request."},{"title":"Contributing to patch releases","page":"Contributing to patch releases","location":"devdocs/contributing/patch-releases.html","category":"page","text":"Step 2 above, i.e. backporting commits to the backports-release-X.Y branch, has largely been automated via Backporter: Backporter searches for merged pull requests with the relevant backport-X.Y tag, and attempts to cherry-pick the commits from those pull requests onto the backports-release-X.Y branch. Some commits apply successfully without intervention, others not so much. The latter commits require \"manual\" backporting, with which help is generally much appreciated. Backporter generates a report identifying those commits it managed to backport automatically and those that require manual backporting; this report is usually copied into the first post of the pull request associated with backports-release-X.Y and maintained as additional commits are automatically and/or manually backported."},{"title":"Contributing to patch releases","page":"Contributing to patch releases","location":"devdocs/contributing/patch-releases.html","category":"page","text":"When contributing a manual backport, if you have the necessary permissions, please push the backport directly to the backports-release-X.Y branch. If you lack the relevant permissions, please open a pull request against the backports-release-X.Y branch with the manual backport. Once the manual backport is live on the backports-release-X.Y branch, please remove the backport-X.Y tag from the originating pull request for the commits."},{"title":"Memory-mapped I/O","page":"Memory-mapped I/O","location":"stdlib/Mmap.html#Memory-mapped-I/O","category":"section","text":""},{"title":"Memory-mapped I/O","page":"Memory-mapped I/O","location":"stdlib/Mmap.html","category":"page","text":"Low level module for mmap (memory mapping of files)."},{"title":"Mmap.Anonymous","page":"Memory-mapped I/O","location":"stdlib/Mmap.html#Mmap.Anonymous","category":"type","text":"Mmap.Anonymous(name::AbstractString=\"\", readonly::Bool=false, create::Bool=true)\n\nCreate an IO-like object for creating zeroed-out mmapped-memory that is not tied to a file for use in mmap. Used by SharedArray for creating shared memory arrays.\n\nExamples\n\njulia> using Mmap\n\njulia> anon = Mmap.Anonymous();\n\njulia> isreadable(anon)\ntrue\n\njulia> iswritable(anon)\ntrue\n\njulia> isopen(anon)\ntrue\n\n\n\n\n\n"},{"title":"Mmap.mmap","page":"Memory-mapped I/O","location":"stdlib/Mmap.html#Mmap.mmap","category":"function","text":"mmap(io::Union{IOStream,AbstractString,Mmap.AnonymousMmap}[, type::Type{Array{T,N}}, dims, offset]; grow::Bool=true, shared::Bool=true)\nmmap(type::Type{Array{T,N}}, dims)\n\nCreate an Array whose values are linked to a file, using memory-mapping. This provides a convenient way of working with data too large to fit in the computer's memory.\n\nThe type is an Array{T,N} with a bits-type element of T and dimension N that determines how the bytes of the array are interpreted. Note that the file must be stored in binary format, and no format conversions are possible (this is a limitation of operating systems, not Julia).\n\ndims is a tuple or single Integer specifying the size or length of the array.\n\nThe file is passed via the stream argument, either as an open IOStream or filename string. When you initialize the stream, use \"r\" for a \"read-only\" array, and \"w+\" to create a new array used to write values to disk.\n\nIf no type argument is specified, the default is Vector{UInt8}.\n\nOptionally, you can specify an offset (in bytes) if, for example, you want to skip over a header in the file. The default value for the offset is the current stream position for an IOStream.\n\nThe grow keyword argument specifies whether the disk file should be grown to accommodate the requested size of array (if the total file size is < requested array size). Write privileges are required to grow the file.\n\nThe shared keyword argument specifies whether the resulting Array and changes made to it will be visible to other processes mapping the same file.\n\nFor example, the following code\n\n# Create a file for mmapping\n# (you could alternatively use mmap to do this step, too)\nusing Mmap\nA = rand(1:20, 5, 30)\ns = open(\"/tmp/mmap.bin\", \"w+\")\n# We'll write the dimensions of the array as the first two Ints in the file\nwrite(s, size(A,1))\nwrite(s, size(A,2))\n# Now write the data\nwrite(s, A)\nclose(s)\n\n# Test by reading it back in\ns = open(\"/tmp/mmap.bin\")   # default is read-only\nm = read(s, Int)\nn = read(s, Int)\nA2 = mmap(s, Matrix{Int}, (m,n))\n\ncreates a m-by-n Matrix{Int}, linked to the file associated with stream s.\n\nA more portable file would need to encode the word size – 32 bit or 64 bit – and endianness information in the header. In practice, consider encoding binary data using standard formats like HDF5 (which can be used with memory-mapping).\n\n\n\n\n\nmmap(io, BitArray, [dims, offset])\n\nCreate a BitArray whose values are linked to a file, using memory-mapping; it has the same purpose, works in the same way, and has the same arguments, as mmap, but the byte representation is different.\n\nExamples\n\njulia> using Mmap\n\njulia> io = open(\"mmap.bin\", \"w+\");\n\njulia> B = mmap(io, BitArray, (25,30000));\n\njulia> B[3, 4000] = true;\n\njulia> Mmap.sync!(B);\n\njulia> close(io);\n\njulia> io = open(\"mmap.bin\", \"r+\");\n\njulia> C = mmap(io, BitArray, (25,30000));\n\njulia> C[3, 4000]\ntrue\n\njulia> C[2, 4000]\nfalse\n\njulia> close(io)\n\njulia> rm(\"mmap.bin\")\n\nThis creates a 25-by-30000 BitArray, linked to the file associated with stream io.\n\n\n\n\n\n"},{"title":"Mmap.sync!","page":"Memory-mapped I/O","location":"stdlib/Mmap.html#Mmap.sync!","category":"function","text":"Mmap.sync!(array)\n\nForces synchronization between the in-memory version of a memory-mapped Array or BitArray and the on-disk version.\n\n\n\n\n\n"},{"title":"macOS","page":"macOS","location":"devdocs/build/macos.html#macOS","category":"section","text":""},{"title":"macOS","page":"macOS","location":"devdocs/build/macos.html","category":"page","text":"You need to have the current Xcode command line utilities installed: run xcode-select --install in the terminal. You will need to rerun this terminal command after each macOS update, otherwise you may run into errors involving missing libraries or headers."},{"title":"macOS","page":"macOS","location":"devdocs/build/macos.html","category":"page","text":"The dependent libraries are now built with BinaryBuilder and will be automatically downloaded. This is the preferred way to build Julia source. In case you want to build them all on your own, you will need a 64-bit gfortran to compile Julia dependencies."},{"title":"macOS","page":"macOS","location":"devdocs/build/macos.html","category":"page","text":"brew install gcc"},{"title":"macOS","page":"macOS","location":"devdocs/build/macos.html","category":"page","text":"If you have set LD_LIBRARY_PATH or DYLD_LIBRARY_PATH in your .bashrc or equivalent, Julia may be unable to find various libraries that come bundled with it. These environment variables need to be unset for Julia to work."},{"title":"Serialization","page":"Serialization","location":"stdlib/Serialization.html#Serialization","category":"section","text":""},{"title":"Serialization","page":"Serialization","location":"stdlib/Serialization.html","category":"page","text":"Provides serialization of Julia objects."},{"title":"Serialization.serialize","page":"Serialization","location":"stdlib/Serialization.html#Serialization.serialize","category":"function","text":"serialize(stream::IO, value)\n\nWrite an arbitrary value to a stream in an opaque format, such that it can be read back by deserialize. The read-back value will be as identical as possible to the original, but note that Ptr values are serialized as all-zero bit patterns (NULL).\n\nAn 8-byte identifying header is written to the stream first. To avoid writing the header, construct a Serializer and use it as the first argument to serialize instead. See also Serialization.writeheader.\n\nThe data format can change in minor (1.x) Julia releases, but files written by prior 1.x versions will remain readable. The main exception to this is when the definition of a type in an external package changes. If that occurs, it may be necessary to specify an explicit compatible version of the affected package in your environment. Renaming functions, even private functions, inside packages can also put existing files out of sync. Anonymous functions require special care: because their names are automatically generated, minor code changes can cause them to be renamed. Serializing anonymous functions should be avoided in files intended for long-term storage.\n\nIn some cases, the word size (32- or 64-bit) of the reading and writing machines must match. In rarer cases the OS or architecture must also match, for example when using packages that contain platform-dependent code.\n\n\n\n\n\nserialize(filename::AbstractString, value)\n\nOpen a file and serialize the given value to it.\n\ncompat: Julia 1.1\nThis method is available as of Julia 1.1.\n\n\n\n\n\n"},{"title":"Serialization.deserialize","page":"Serialization","location":"stdlib/Serialization.html#Serialization.deserialize","category":"function","text":"deserialize(stream)\n\nRead a value written by serialize. deserialize assumes the binary data read from stream is correct and has been serialized by a compatible implementation of serialize. deserialize is designed for simplicity and performance, and so does not validate the data read. Malformed data can result in process termination. The caller must ensure the integrity and correctness of data read from stream.\n\n\n\n\n\ndeserialize(filename::AbstractString)\n\nOpen a file and deserialize its contents.\n\ncompat: Julia 1.1\nThis method is available as of Julia 1.1.\n\n\n\n\n\n"},{"title":"Serialization.writeheader","page":"Serialization","location":"stdlib/Serialization.html#Serialization.writeheader","category":"function","text":"Serialization.writeheader(s::AbstractSerializer)\n\nWrite an identifying header to the specified serializer. The header consists of 8 bytes as follows:\n\nOffset Description\n0 tag byte (0x37)\n1-2 signature bytes \"JL\"\n3 protocol version\n4 bits 0-1: endianness: 0 = little, 1 = big\n4 bits 2-3: platform: 0 = 32-bit, 1 = 64-bit\n5-7 reserved\n\n\n\n\n\n"},{"title":"Recommended File Extension","page":"Serialization","location":"stdlib/Serialization.html#Recommended-File-Extension","category":"section","text":""},{"title":"Serialization","page":"Serialization","location":"stdlib/Serialization.html","category":"page","text":"While the Serialization module does not mandate a specific file extension, the Julia community commonly uses the .jls extension for serialized Julia files."},{"title":"Serialization","page":"Serialization","location":"stdlib/Serialization.html","category":"page","text":"Example:"},{"title":"Serialization","page":"Serialization","location":"stdlib/Serialization.html","category":"page","text":"open(\"model.jls\", \"w\") do io\n    serialize(io, my_model)\nend"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html#Binary-distributions","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"These notes are for those wishing to compile a binary distribution of Julia for distribution on various platforms. We love users spreading Julia as far and wide as they can, trying it out on as wide an array of operating systems and hardware configurations as possible. As each platform has specific gotchas and processes that must be followed in order to create a portable, working Julia distribution, we have separated most of the notes by OS."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Note that while the code for Julia is MIT-licensed, with a few exceptions, the distribution created by the techniques described herein will be GPL licensed, as various dependent libraries such as SuiteSparse are GPL licensed. We do hope to have a non-GPL distribution of Julia in the future."},{"title":"Versioning and Git","page":"Binary distributions","location":"devdocs/build/distributing.html#Versioning-and-Git","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"The Makefile uses both the VERSION file and commit hashes and tags from the git repository to generate the base/version_git.jl with information we use to fill the splash screen and the versioninfo() output. If you for some reason don't want to have the git repository available when building you should pregenerate the base/version_git.jl file with:"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"make -C base version_git.jl.phony"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Julia has lots of build dependencies where we use patched versions that has not yet been included by the popular package managers. These dependencies will usually be automatically downloaded when you build, but if you want to be able to build Julia on a computer without internet access you should create a full-source-dist archive with the special make target"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"make full-source-dist"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"that creates a julia-version-commit.tar.gz archive with all required dependencies."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"When compiling a tagged release in the git repository, we don't display the branch/commit hash info in the splash screen. You can use this line to show a release description of up to 45 characters. To set this line you have to create a Make.user file containing:"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"override TAGGED_RELEASE_BANNER = \"my-package-repository build\""},{"title":"Target Architectures","page":"Binary distributions","location":"devdocs/build/distributing.html#Target-Architectures","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"By default, Julia optimizes its system image to the native architecture of the build machine. This is usually not what you want when building packages, as it will make Julia fail at startup on any machine with incompatible CPUs (in particular older ones with more restricted instruction sets)."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"We therefore recommend that you pass the MARCH variable when calling make, setting it to the baseline target you intend to support. This will determine the target CPU for both the Julia executable and libraries, and the system image (the latter can also be set using JULIA_CPU_TARGET). Typically useful values for x86 CPUs are x86-64 and core2 (for 64-bit builds) and pentium4 (for 32-bit builds). Unfortunately, CPUs older than Pentium 4 are currently not supported (see this issue)."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"The full list of CPU targets supported by LLVM can be obtained by running llc -mattr=help."},{"title":"Linux","page":"Binary distributions","location":"devdocs/build/distributing.html#Linux","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"On Linux, make binary-dist creates a tarball that contains a fully functional Julia installation. If you wish to create a distribution package such as a .deb, or .rpm, some extra effort is needed. See the julia-debian repository for an example of what metadata is needed for creating .deb packages for Debian and Ubuntu-based systems. See the Fedora package for RPM-based distributions. Although we have not yet experimented with it, Alien could be used to generate Julia packages for various Linux distributions."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Julia supports overriding standard installation directories via prefix and other environment variables you can pass when calling make and make install. See Make.inc for their list. DESTDIR can also be used to force the installation into a temporary directory."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"By default, Julia loads $prefix/etc/julia/startup.jl as an installation-wide initialization file. This file can be used by distribution managers to set up custom paths or initialization code. For Linux distribution packages, if $prefix is set to /usr, there is no /usr/etc to look into. This requires the path to Julia's private etc directory to be changed. This can be done via the sysconfdir make variable when building. Simply pass sysconfdir=/etc to make when building and Julia will first check /etc/julia/startup.jl before trying $prefix/etc/julia/startup.jl."},{"title":"OS X","page":"Binary distributions","location":"devdocs/build/distributing.html#OS-X","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"To create a binary distribution on OSX, build Julia first, then cd to contrib/mac/app, and run make with the same makevars that were used with make when building Julia proper. This will then create a .dmg file in the contrib/mac/app directory holding a completely self-contained Julia.app."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Alternatively, Julia may be built as a framework by invoking make with the darwinframework target and DARWIN_FRAMEWORK=1 set. For example, make DARWIN_FRAMEWORK=1 darwinframework."},{"title":"Windows","page":"Binary distributions","location":"devdocs/build/distributing.html#Windows","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Instructions for creating a Julia distribution on Windows are described in the build devdocs for Windows."},{"title":"Notes on BLAS and LAPACK","page":"Binary distributions","location":"devdocs/build/distributing.html#Notes-on-BLAS-and-LAPACK","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Julia builds OpenBLAS by default, which includes the BLAS and LAPACK libraries. On 32-bit architectures, Julia builds OpenBLAS to use 32-bit integers, while on 64-bit architectures, Julia builds OpenBLAS to use 64-bit integers (ILP64). It is essential that all Julia functions that call BLAS and LAPACK API routines use integers of the correct width."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Most BLAS and LAPACK distributions provided on linux distributions, and even commercial implementations ship libraries that use 32-bit APIs. In many cases, a 64-bit API is provided as a separate library."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"When using vendor provided or OS provided libraries, a make option called USE_BLAS64 is available as part of the Julia build. When doing make USE_BLAS64=0, Julia will call BLAS and LAPACK assuming a 32-bit API, where all integers are 32-bit wide, even on a 64-bit architecture."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Other libraries that Julia uses, such as SuiteSparse also use BLAS and LAPACK internally. The APIs need to be consistent across all libraries that depend on BLAS and LAPACK. The Julia build process will build all these libraries correctly, but when overriding defaults and using system provided libraries, this consistency must be ensured."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Also note that Linux distributions sometimes ship several versions of OpenBLAS, some of which enable multithreading, and others only working in a serial fashion. For example, in Fedora, libopenblasp.so is threaded, but libopenblas.so is not. We recommend using the former for optimal performance. To choose an OpenBLAS library whose name is different from the default libopenblas.so, pass LIBBLAS=-l$(YOURBLAS) and LIBBLASNAME=lib$(YOURBLAS) to make, replacing $(YOURBLAS) with the name of your library. You can also add .so.0 to the name of the library if you want your package to work without requiring the unversioned .so symlink."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Finally, OpenBLAS includes its own optimized version of LAPACK. If you set USE_SYSTEM_BLAS=1 and USE_SYSTEM_LAPACK=1, you should also set LIBLAPACK=-l$(YOURBLAS) and LIBLAPACKNAME=lib$(YOURBLAS). Else, the reference LAPACK will be used and performance will typically be much lower."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Starting with Julia 1.7, Julia uses libblastrampoline to pick a different BLAS at runtime."},{"title":"Point releasing 101","page":"Binary distributions","location":"devdocs/build/distributing.html#Point-releasing-101","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Creating a point/patch release consists of several distinct steps."},{"title":"Backporting commits","page":"Binary distributions","location":"devdocs/build/distributing.html#Backporting-commits","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Some pull requests are labeled \"backport pending x.y\", e.g. \"backport pending 0.6\". This designates that the next subsequent release tagged from the release-x.y branch should include the commit(s) in that pull request. Once the pull request is merged into master, each of the commits should be cherry picked to a dedicated branch that will ultimately be merged into release-x.y."},{"title":"Creating a backports branch","page":"Binary distributions","location":"devdocs/build/distributing.html#Creating-a-backports-branch","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"First, create a new branch based on release-x.y. The typical convention for Julia branches is to prefix the branch name with your initials if it's intended to be a personal branch. For the sake of example, we'll say that the author of the branch is Jane Smith."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"git fetch origin\ngit checkout release-x.y\ngit rebase origin/release-x.y\ngit checkout -b js/backport-x.y"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"This ensures that your local copy of release-x.y is up to date with origin before you create a new branch from it."},{"title":"Cherry picking commits","page":"Binary distributions","location":"devdocs/build/distributing.html#Cherry-picking-commits","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Now we do the actual backporting. Find all merged pull requests labeled \"backport pending x.y\" in the GitHub web UI. For each of these, scroll to the bottom where it says \"someperson merged commit 123abc into master XX minutes ago\". Note that the commit name is a link; if you click it, you'll be shown the contents of the commit. If this page shows that 123abc is a merge commit, go back to the PR page—we don't want merge commits, we want the actual commits. However, if this does not show a merge commit, it means that the PR was squash-merged. In that case, use the git SHA of the commit, listed next to commit on this page."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Once you have the SHA of the commit, cherry-pick it onto the backporting branch:"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"git cherry-pick -x -e <sha>"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"There may be conflicts which need to be resolved manually. Once conflicts are resolved (if applicable), add a reference to the GitHub pull request that introduced the commit in the body of the commit message."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"After all of the relevant commits are on the backports branch, push the branch to GitHub."},{"title":"Checking for performance regressions","page":"Binary distributions","location":"devdocs/build/distributing.html#Checking-for-performance-regressions","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Point releases should never introduce performance regressions. Luckily the Julia benchmarking bot, Nanosoldier, can run benchmarks against any branch, not just master. In this case we want to check the benchmark results of js/backport-x.y against release-x.y. To do this, awaken the Nanosoldier from his robotic slumber using a comment on your backporting pull request:"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"@nanosoldier `runbenchmarks(ALL, vs=\":release-x.y\")`"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"This will run all registered benchmarks on release-x.y and js/backport-x.y and produce a summary of results, marking all improvements and regressions."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"If Nanosoldier finds any regressions, try verifying locally and rerun Nanosoldier if necessary. If the regressions are deemed to be real rather than just noise, you'll have to find a commit on master to backport that fixes it if one exists, otherwise you should determine what caused the regression and submit a patch (or get someone who knows the code to submit a patch) to master, then backport the commit once that's merged. (Or submit a patch directly to the backport branch if appropriate.)"},{"title":"Building test binaries","page":"Binary distributions","location":"devdocs/build/distributing.html#Building-test-binaries","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"After the backport PR has been merged into the release-x.y branch, update your local clone of Julia, then get the SHA of the branch using"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"git rev-parse origin/release-x.y"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Keep that handy, as it's what you'll enter in the \"Revision\" field in the buildbot UI."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"For now, all you need are binaries for Linux x86-64, since this is what's used for running PackageEvaluator. Go to https://buildog.julialang.org, submit a job for nuke_linux64, then queue up a job for package_linux64, providing the SHA as the revision. When the packaging job completes, it will upload the binary to the julialang2 bucket on AWS. Retrieve the URL, as it will be used for PackageEvaluator."},{"title":"Checking for package breakages","page":"Binary distributions","location":"devdocs/build/distributing.html#Checking-for-package-breakages","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Point releases should never break packages, with the possible exception of packages that are doing some seriously questionable hacks using Base internals that are not intended to be user-facing. (In those cases, maybe have a word with the package author.)"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Checking whether changes made in the forthcoming new version will break packages can be accomplished using PackageEvaluator, often called \"PkgEval\" for short. PkgEval is what populates the status badges on GitHub repos and on pkg.julialang.org. It typically runs on one of the non-benchmarking nodes of Nanosoldier and uses Vagrant to perform its duties in separate, parallel VirtualBox virtual machines."},{"title":"Setting up PackageEvaluator","page":"Binary distributions","location":"devdocs/build/distributing.html#Setting-up-PackageEvaluator","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Clone PackageEvaluator and create a branch called backport-x.y.z, and check it out. Note that the required changes are a little hacky and confusing, and hopefully that will be addressed in a future version of PackageEvaluator. The changes to make will be modeled off of this commit."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"The setup script takes its first argument as the version of Julia to run and the second as the range of package names (AK for packages named A-K, LZ for L-Z). The basic idea is that we're going to tweak that a bit to run only two versions of Julia, the current x.y release and our backport version, each with three ranges of packages."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"In the linked diff, we're saying that if the second argument is LZ, use the binaries built from our backport branch, otherwise (AK) use the release binaries. Then we're using the first argument to run a section of the package list: A-F for input 0.4, G-N for 0.5, and O-Z for 0.6."},{"title":"Running PackageEvaluator","page":"Binary distributions","location":"devdocs/build/distributing.html#Running-PackageEvaluator","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"To run PkgEval, find a hefty enough machine (such as Nanosoldier node 1), then run"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"git clone https://github.com/JuliaCI/PackageEvaluator.jl.git\ncd PackageEvaluator.jl/scripts\ngit checkout backport-x.y.z\n./runvagrant.sh"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"This produces some folders in the scripts/ directory. The folder names and their contents are decoded below:"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Folder name Julia version Package range\n0.4AK Release A-F\n0.4LZ Backport A-F\n0.5AK Release G-N\n0.5LZ Backport G-N\n0.6AK Release O-Z\n0.6LZ Backport O-Z"},{"title":"Investigating results","page":"Binary distributions","location":"devdocs/build/distributing.html#Investigating-results","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Once that's done, you can use ./summary.sh from that same directory to produce a summary report of the findings. We'll do so for each of the folders to aggregate overall results by version."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"./summary.sh 0.4AK/*.json > summary_release.txt\n./summary.sh 0.5AK/*.json >> summary_release.txt\n./summary.sh 0.6AK/*.json >> summary_release.txt\n./summary.sh 0.4LZ/*.json > summary_backport.txt\n./summary.sh 0.5LZ/*.json >> summary_backport.txt\n./summary.sh 0.6LZ/*.json >> summary_backport.txt"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Now we have two files, summary_release.txt and summary_backport.txt, containing the PackageEvaluator test results (pass/fail) for each package for the two versions."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"To make these easier to ingest into a Julia, we'll convert them into CSV files then use the DataFrames package to process the results. To convert to CSV, copy each .txt file to a corresponding .csv file, then enter Vim and execute ggVGI\"<esc> then :%s/\\.json /\",/g. (You don't have to use Vim; this just is one way to do it.) Now process the results with Julia code similar to the following."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"using DataFrames\n\nrelease = readtable(\"summary_release.csv\", header=false, names=[:package, :release])\nbackport = readtable(\"summary_backport.csv\", header=false, names=[:package, :backport])\n\nresults = join(release, backport, on=:package, kind=:outer)\n\nfor result in eachrow(results)\n    a = result[:release]\n    b = result[:backport]\n    if (isna(a) && !isna(b)) || (isna(b) && !isna(a))\n        color = :yellow\n    elseif a != b && occursin(\"pass\", b)\n        color = :green\n    elseif a != b\n        color = :red\n    else\n        continue\n    end\n    printstyled(result[:package], \": Release \", a, \" -> Backport \", b, \"\\n\", color=color)\nend"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"This will write color-coded lines to stdout. All lines in red must be investigated as they signify potential breakages caused by the backport version. Lines in yellow should be looked into since it means a package ran on one version but not on the other for some reason. If you find that your backported branch is causing breakages, use git bisect to identify the problematic commits, git revert those commits, and repeat the process."},{"title":"Merging backports into the release branch","page":"Binary distributions","location":"devdocs/build/distributing.html#Merging-backports-into-the-release-branch","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"After you have ensured that"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"the backported commits pass all of Julia's unit tests,\nthere are no performance regressions introduced by the backported commits as compared to the release branch, and\nthe backported commits do not break any registered packages,"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"then the backport branch is ready to be merged into release-x.y. Once it's merged, go through and remove the \"backport pending x.y\" label from all pull requests containing the commits that have been backported. Do not remove the label from PRs that have not been backported."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"The release-x.y branch should now contain all of the new commits. The last thing we want to do to the branch is to adjust the version number. To do this, submit a PR against release-x.y that edits the VERSION file to remove -pre from the version number. Once that's merged, we're ready to tag."},{"title":"Tagging the release","page":"Binary distributions","location":"devdocs/build/distributing.html#Tagging-the-release","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"It's time! Check out the release-x.y branch and make sure that your local copy of the branch is up to date with the remote branch. At the command line, run"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"git tag v$(cat VERSION)\ngit push --tags"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"This creates the tag locally and pushes it to GitHub."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"After tagging the release, submit another PR to release-x.y to bump the patch number and add -pre back to the end. This denotes that the branch state reflects a prerelease version of the next point release in the x.y series."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Follow the remaining directions in the Makefile."},{"title":"Signing binaries","page":"Binary distributions","location":"devdocs/build/distributing.html#Signing-binaries","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Some of these steps will require secure passwords. To obtain the appropriate passwords, contact Elliot Saba (staticfloat) or Alex Arslan (ararslan). Note that code signing for each platform must be performed on that platform (e.g. Windows signing must be done on Windows, etc.)."},{"title":"Linux","page":"Binary distributions","location":"devdocs/build/distributing.html#Linux-2","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Code signing must be done manually on Linux, but it's quite simple. First obtain the file julia.key from the CodeSigning folder in the juliasecure AWS bucket. Add this to your GnuPG keyring using"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"gpg --import julia.key"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"This will require entering a password that you must obtain from Elliot or Alex. Next, set the trust level for the key to maximum. Start by entering a gpg session:"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"gpg --edit-key julia"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"At the prompt, type trust, then when asked for a trust level, provide the maximum available (likely 5). Exit GnuPG."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Now, for each of the Linux tarballs that were built on the buildbots, enter"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"gpg -u julia --armor --detach-sig julia-x.y.z-linux-<arch>.tar.gz"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"This will produce a corresponding .asc file for each tarball. And that's it!"},{"title":"macOS","page":"Binary distributions","location":"devdocs/build/distributing.html#macOS","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Code signing should happen automatically on the macOS buildbots. However, it's important to verify that it was successful. On a system or virtual machine running macOS, download the .dmg file that was built on the buildbots. For the sake of example, say that the .dmg file is called julia-x.y.z-osx.dmg. Run"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"mkdir ./jlmnt\nhdiutil mount -readonly -mountpoint ./jlmnt julia-x.y.z-osx.dmg\ncodesign -v jlmnt/Julia-x.y.app"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Be sure to note the name of the mounted disk listed when mounting! For the sake of example, we'll assume this is disk3. If the code signing verification exited successfully, there will be no output from the codesign step. If it was indeed successful, you can detach the .dmg now:"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"hdiutil eject /dev/disk3\nrm -rf ./jlmnt"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"If you get a message like"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Julia-x.y.app: code object is not signed at all"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"then you'll need to sign manually."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"To sign manually, first retrieve the OS X certificates from the CodeSigning folder in the juliasecure bucket on AWS. Add the .p12 file to your keychain using Keychain.app. Ask Elliot Saba (staticfloat) or Alex Arslan (ararslan) for the password for the key. Now run"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"hdiutil convert julia-x.y.z-osx.dmg -format UDRW -o julia-x.y.z-osx_writable.dmg\nmkdir ./jlmnt\nhdiutil mount -mountpoint julia-x.y.z-osx_writable.dmg\ncodesign -s \"AFB379C0B4CBD9DB9A762797FC2AB5460A2B0DBE\" --deep jlmnt/Julia-x.y.app"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"This may fail with a message like"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Julia-x.y.app: resource fork, Finder information, or similar detritus not allowed"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"If that's the case, you'll need to remove extraneous attributes:"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"xattr -cr jlmnt/Julia-x.y.app"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Then retry code signing. If that produces no errors, retry verification. If all is now well, unmount the writable .dmg and convert it back to read-only:"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"hdiutil eject /dev/disk3\nrm -rf ./jlmnt\nhdiutil convert julia-x.y.z-osx_writable.dmg -format UDZO -o julia-x.y.z-osx_fixed.dmg"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Verify that the resulting .dmg is in fact fixed by double clicking it. If everything looks good, eject it then drop the _fixed suffix from the name. And that's it!"},{"title":"Windows","page":"Binary distributions","location":"devdocs/build/distributing.html#Windows-2","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Signing must be performed manually on Windows. First obtain the Windows 10 SDK, which contains the necessary signing utilities, from the Microsoft website. We need the SignTool utility which should have been installed somewhere like C:\\Program Files (x86)\\Windows Kits\\10\\App Certification Kit. Grab the Windows certificate files from CodeSigning on juliasecure and put them in the same directory as the executables. Open a Windows CMD window, cd to where all the files are, and run"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"set PATH=%PATH%;C:\\Program Files (x86)\\Windows Kits\\10\\App Certification Kit;\nsigntool sign /f julia-windows-code-sign_2017.p12 /p \"PASSWORD\" ^\n   /t http://timestamp.verisign.com/scripts/timstamp.dll ^\n   /v julia-x.y.z-win32.exe"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Note that ^ is a line continuation character in Windows CMD and PASSWORD is a placeholder for the password for this certificate. As usual, contact Elliot or Alex for passwords. If there are no errors, we're all good!"},{"title":"Uploading binaries","page":"Binary distributions","location":"devdocs/build/distributing.html#Uploading-binaries","category":"section","text":""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Now that everything is signed, we need to upload the binaries to AWS. You can use a program like Cyberduck or the aws command line utility. The binaries should go in the julialang2 bucket in the appropriate folders. For example, Linux x86-64 goes in julialang2/bin/linux/x.y. Be sure to delete the current julia-x.y-latest-linux-<arch>.tar.gz file and replace it with a duplicate of julia-x.y.z-linux-<arch>.tar.gz."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"We also need to upload the checksums for everything we've built, including the source tarballs and all release binaries. This is simple:"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"shasum -a 256 julia-x.y.z* | grep -v -e sha256 -e md5 -e asc > julia-x.y.z.sha256\nmd5sum julia-x.y.z* | grep -v -e sha256 -e md5 -e asc > julia-x.y.z.md5"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Note that if you're running those commands on macOS, you'll get very slightly different output, which can be reformatted by looking at an existing file. Mac users will also need to use md5 -r instead of md5sum. Upload the .md5 and .sha256 files to julialang2/bin/checksums on AWS."},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Ensure that the permissions on AWS for all uploaded files are set to \"Everyone: READ.\""},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"For each file we've uploaded, we need to purge the Fastly cache so that the links on the website point to the updated files. As an example:"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"curl -X PURGE https://julialang-s3.julialang.org/bin/checksums/julia-x.y.z.sha256"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html","category":"page","text":"Sometimes this isn't necessary but it's good to do anyway."},{"title":"Iteration utilities","page":"Iteration utilities","location":"base/iterators.html#Iteration-utilities","category":"section","text":""},{"title":"Base.Iterators.Stateful","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.Stateful","category":"type","text":"Stateful(itr)\n\nThere are several different ways to think about this iterator wrapper:\n\nIt provides a mutable wrapper around an iterator and its iteration state.\nIt turns an iterator-like abstraction into a Channel-like abstraction.\nIt's an iterator that mutates to become its own rest iterator whenever an item is produced.\n\nStateful provides the regular iterator interface. Like other mutable iterators (e.g. Base.Channel), if iteration is stopped early (e.g. by a break in a for loop), iteration can be resumed from the same spot by continuing to iterate over the same iterator object (in contrast, an immutable iterator would restart from the beginning).\n\nExamples\n\njulia> a = Iterators.Stateful(\"abcdef\");\n\njulia> isempty(a)\nfalse\n\njulia> popfirst!(a)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> collect(Iterators.take(a, 3))\n3-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n 'd': ASCII/Unicode U+0064 (category Ll: Letter, lowercase)\n\njulia> collect(a)\n2-element Vector{Char}:\n 'e': ASCII/Unicode U+0065 (category Ll: Letter, lowercase)\n 'f': ASCII/Unicode U+0066 (category Ll: Letter, lowercase)\n\njulia> Iterators.reset!(a); popfirst!(a)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> Iterators.reset!(a, \"hello\"); popfirst!(a)\n'h': ASCII/Unicode U+0068 (category Ll: Letter, lowercase)\n\njulia> a = Iterators.Stateful([1,1,1,2,3,4]);\n\njulia> for x in a; x == 1 || break; end\n\njulia> peek(a)\n3\n\njulia> sum(a) # Sum the remaining elements\n7\n\n\n\n\n\n"},{"title":"Base.Iterators.zip","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.zip","category":"function","text":"zip(iters...)\n\nRun multiple iterators at the same time, until any of them is exhausted. The value type of the zip iterator is a tuple of values of its subiterators.\n\nnote: Note\nzip orders the calls to its subiterators in such a way that stateful iterators will not advance when another iterator finishes in the current iteration.\n\nnote: Note\nzip() with no arguments yields an infinite iterator of empty tuples.\n\nSee also: enumerate, Base.splat.\n\nExamples\n\njulia> a = 1:5\n1:5\n\njulia> b = [\"e\",\"d\",\"b\",\"c\",\"a\"]\n5-element Vector{String}:\n \"e\"\n \"d\"\n \"b\"\n \"c\"\n \"a\"\n\njulia> c = zip(a,b)\nzip(1:5, [\"e\", \"d\", \"b\", \"c\", \"a\"])\n\njulia> length(c)\n5\n\njulia> first(c)\n(1, \"e\")\n\n\n\n\n\n"},{"title":"Base.Iterators.enumerate","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.enumerate","category":"function","text":"enumerate(iter)\n\nAn iterator that yields (i, x) where i is a counter starting at 1, and x is the ith value from the given iterator. It's useful when you need not only the values x over which you are iterating, but also the number of iterations so far.\n\nNote that i may not be valid for indexing iter, or may index a different element. This will happen if iter has indices that do not start at 1, and may happen for strings, dictionaries, etc. See the pairs(IndexLinear(), iter) method if you want to ensure that i is an index.\n\nExamples\n\njulia> a = [\"a\", \"b\", \"c\"];\n\njulia> for (index, value) in enumerate(a)\n           println(\"$index $value\")\n       end\n1 a\n2 b\n3 c\n\njulia> str = \"naïve\";\n\njulia> for (i, val) in enumerate(str)\n           print(\"i = \", i, \", val = \", val, \", \")\n           try @show(str[i]) catch e println(e) end\n       end\ni = 1, val = n, str[i] = 'n'\ni = 2, val = a, str[i] = 'a'\ni = 3, val = ï, str[i] = 'ï'\ni = 4, val = v, StringIndexError(\"naïve\", 4)\ni = 5, val = e, str[i] = 'v'\n\n\n\n\n\n"},{"title":"Base.Iterators.rest","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.rest","category":"function","text":"rest(iter, state)\n\nAn iterator that yields the same elements as iter, but starting at the given state, which must be a state obtainable via a sequence of one or more calls to iterate(iter[, state])\n\nSee also: Iterators.drop, Iterators.peel, Base.rest.\n\nExamples\n\njulia> iter = [1,2,3,4];\n\njulia> val, state = iterate(iter)\n(1, 2)\n\njulia> collect(Iterators.rest(iter, state))\n3-element Vector{Int64}:\n 2\n 3\n 4\n\n\n\n\n\n"},{"title":"Base.Iterators.countfrom","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.countfrom","category":"function","text":"countfrom(start=1, step=1)\n\nAn iterator that counts forever, starting at start and incrementing by step.\n\nExamples\n\njulia> for v in Iterators.countfrom(5, 2)\n           v > 10 && break\n           println(v)\n       end\n5\n7\n9\n\n\n\n\n\n"},{"title":"Base.Iterators.take","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.take","category":"function","text":"take(iter, n)\n\nAn iterator that generates at most the first n elements of iter.\n\nSee also: drop, peel, first, Base.take!.\n\nExamples\n\njulia> a = 1:2:11\n1:2:11\n\njulia> collect(a)\n6-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n\njulia> collect(Iterators.take(a,3))\n3-element Vector{Int64}:\n 1\n 3\n 5\n\n\n\n\n\n"},{"title":"Base.Iterators.takewhile","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.takewhile","category":"function","text":"takewhile(pred, iter)\n\nAn iterator that generates element from iter as long as predicate pred is true, afterwards, drops every element.\n\ncompat: Julia 1.4\nThis function requires at least Julia 1.4.\n\nExamples\n\njulia> s = collect(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> collect(Iterators.takewhile(<(3),s))\n2-element Vector{Int64}:\n 1\n 2\n\n\n\n\n\n"},{"title":"Base.Iterators.drop","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.drop","category":"function","text":"drop(iter, n)\n\nAn iterator that generates all but the first n elements of iter.\n\nExamples\n\njulia> a = 1:2:11\n1:2:11\n\njulia> collect(a)\n6-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n\njulia> collect(Iterators.drop(a,4))\n2-element Vector{Int64}:\n  9\n 11\n\n\n\n\n\n"},{"title":"Base.Iterators.dropwhile","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.dropwhile","category":"function","text":"dropwhile(pred, iter)\n\nAn iterator that drops element from iter as long as predicate pred is true, afterwards, returns every element.\n\ncompat: Julia 1.4\nThis function requires at least Julia 1.4.\n\nExamples\n\njulia> s = collect(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> collect(Iterators.dropwhile(<(3),s))\n3-element Vector{Int64}:\n 3\n 4\n 5\n\n\n\n\n\n"},{"title":"Base.Iterators.findeach","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.findeach","category":"function","text":"findeach(f, it)\nfindeach(it)\n\nAn iterator that generates every key from the key/value pairs of pairs(it), where f(value) returns true.\n\nIf f is not specified, default to identity.\n\nIterators.findeach is the lazy equivalent of findall.\n\ncompat: Julia 1.13\nfindeach requires at least Julia 1.13.\n\nExamples\n\njulia> collect(Iterators.findeach(isodd, Dict(2 => 3, 3 => 2)))\n1-element Vector{Int64}:\n 2\n\njulia> only(Iterators.findeach(==(1), [3,6,2,1]))\n4\n\n\n\n\n\n"},{"title":"Base.Iterators.cycle","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.cycle","category":"function","text":"cycle(iter[, n::Int])\n\nAn iterator that cycles through iter forever. If n is specified, then it cycles through iter that many times. When iter is empty, so are cycle(iter) and cycle(iter, n).\n\nIterators.cycle(iter, n) is the lazy equivalent of Base.repeat(vector, n), while Iterators.repeated(iter, n) is the lazy Base.fill(item, n).\n\ncompat: Julia 1.11\nThe method cycle(iter, n) was added in Julia 1.11.\n\nExamples\n\njulia> for (i, v) in enumerate(Iterators.cycle(\"hello\"))\n           print(v)\n           i > 10 && break\n       end\nhellohelloh\n\njulia> foreach(print, Iterators.cycle(['j', 'u', 'l', 'i', 'a'], 3))\njuliajuliajulia\n\njulia> repeat([1,2,3], 4) == collect(Iterators.cycle([1,2,3], 4))\ntrue\n\njulia> fill([1,2,3], 4) == collect(Iterators.repeated([1,2,3], 4))\ntrue\n\n\n\n\n\n"},{"title":"Base.Iterators.repeated","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.repeated","category":"function","text":"repeated(x[, n::Int])\n\nAn iterator that generates the value x forever. If n is specified, generates x that many times (equivalent to take(repeated(x), n)).\n\nSee also fill, and compare Iterators.cycle.\n\nExamples\n\njulia> a = Iterators.repeated([1 2], 4);\n\njulia> collect(a)\n4-element Vector{Matrix{Int64}}:\n [1 2]\n [1 2]\n [1 2]\n [1 2]\n\njulia> ans == fill([1 2], 4)\ntrue\n\njulia> Iterators.cycle([1 2], 4) |> collect |> println\n[1, 2, 1, 2, 1, 2, 1, 2]\n\n\n\n\n\n"},{"title":"Base.Iterators.product","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.product","category":"function","text":"product(iters...)\n\nReturn an iterator over the product of several iterators. Each generated element is a tuple whose ith element comes from the ith argument iterator. The first iterator changes the fastest.\n\nSee also: zip, Iterators.flatten.\n\nExamples\n\njulia> collect(Iterators.product(1:2, 3:5))\n2×3 Matrix{Tuple{Int64, Int64}}:\n (1, 3)  (1, 4)  (1, 5)\n (2, 3)  (2, 4)  (2, 5)\n\njulia> ans == [(x,y) for x in 1:2, y in 3:5]  # collects a generator involving Iterators.product\ntrue\n\n\n\n\n\n"},{"title":"Base.Iterators.flatten","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.flatten","category":"function","text":"flatten(iter)\n\nGiven an iterator that yields iterators, return an iterator that yields the elements of those iterators. Put differently, the elements of the argument iterator are concatenated.\n\nExamples\n\njulia> collect(Iterators.flatten((1:2, 8:9)))\n4-element Vector{Int64}:\n 1\n 2\n 8\n 9\n\njulia> [(x,y) for x in 0:1 for y in 'a':'c']  # collects generators involving Iterators.flatten\n6-element Vector{Tuple{Int64, Char}}:\n (0, 'a')\n (0, 'b')\n (0, 'c')\n (1, 'a')\n (1, 'b')\n (1, 'c')\n\n\n\n\n\n"},{"title":"Base.Iterators.flatmap","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.flatmap","category":"function","text":"Iterators.flatmap(f, iterators...)\n\nEquivalent to flatten(map(f, iterators...)).\n\nSee also Iterators.flatten, Iterators.map.\n\ncompat: Julia 1.9\nThis function was added in Julia 1.9.\n\nExamples\n\njulia> Iterators.flatmap(n -> -n:2:n, 1:3) |> collect\n9-element Vector{Int64}:\n -1\n  1\n -2\n  0\n  2\n -3\n -1\n  1\n  3\n\njulia> stack(n -> -n:2:n, 1:3)\nERROR: DimensionMismatch: stack expects uniform slices, got axes(x) == (1:3,) while first had (1:2,)\n[...]\n\njulia> Iterators.flatmap(n -> (-n, 10n), 1:2) |> collect\n4-element Vector{Int64}:\n -1\n 10\n -2\n 20\n\njulia> ans == vec(stack(n -> (-n, 10n), 1:2))\ntrue\n\n\n\n\n\n"},{"title":"Base.Iterators.partition","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.partition","category":"function","text":"partition(collection, n)\n\nIterate over a collection n elements at a time.\n\nExamples\n\njulia> collect(Iterators.partition([1,2,3,4,5], 2))\n3-element Vector{SubArray{Int64, 1, Vector{Int64}, Tuple{UnitRange{Int64}}, true}}:\n [1, 2]\n [3, 4]\n [5]\n\n\n\n\n\n"},{"title":"Base.Iterators.map","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.map","category":"function","text":"Iterators.map(f, iterators...)\n\nCreate a lazy mapping.  This is another syntax for writing (f(args...) for args in zip(iterators...)).\n\ncompat: Julia 1.6\nThis function requires at least Julia 1.6.\n\nExamples\n\njulia> collect(Iterators.map(x -> x^2, 1:3))\n3-element Vector{Int64}:\n 1\n 4\n 9\n\n\n\n\n\n"},{"title":"Base.Iterators.filter","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.filter","category":"function","text":"Iterators.filter(flt, itr)\n\nGiven a predicate function flt and an iterable object itr, return an iterable object which upon iteration yields the elements x of itr that satisfy flt(x). The order of the original iterator is preserved.\n\nThis function is lazy; that is, it is guaranteed to return in Θ(1) time and use Θ(1) additional space, and flt will not be called by an invocation of filter. Calls to flt will be made when iterating over the returned iterable object. These calls are not cached and repeated calls will be made when reiterating.\n\nwarning: Warning\nSubsequent lazy transformations on the iterator returned from filter, such as those performed by Iterators.reverse or cycle, will also delay calls to flt until collecting or iterating over the returned iterable object. If the filter predicate is nondeterministic or its return values depend on the order of iteration over the elements of itr, composition with lazy transformations may result in surprising behavior. If this is undesirable, either ensure that flt is a pure function or collect intermediate filter iterators before further transformations.\n\nSee Base.filter for an eager implementation of filtering for arrays.\n\nExamples\n\njulia> f = Iterators.filter(isodd, [1, 2, 3, 4, 5])\nBase.Iterators.Filter{typeof(isodd), Vector{Int64}}(isodd, [1, 2, 3, 4, 5])\n\njulia> foreach(println, f)\n1\n3\n5\n\njulia> [x for x in [1, 2, 3, 4, 5] if isodd(x)]  # collects a generator over Iterators.filter\n3-element Vector{Int64}:\n 1\n 3\n 5\n\n\n\n\n\n"},{"title":"Base.Iterators.accumulate","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.accumulate","category":"function","text":"Iterators.accumulate(f, itr; [init])\n\nGiven a 2-argument function f and an iterator itr, return a new iterator that successively applies f to the previous value and the next element of itr.\n\nThis is effectively a lazy version of Base.accumulate.\n\ncompat: Julia 1.5\nKeyword argument init is added in Julia 1.5.\n\nExamples\n\njulia> a = Iterators.accumulate(+, [1,2,3,4]);\n\njulia> foreach(println, a)\n1\n3\n6\n10\n\njulia> b = Iterators.accumulate(/, (2, 5, 2, 5); init = 100);\n\njulia> collect(b)\n4-element Vector{Float64}:\n 50.0\n 10.0\n  5.0\n  1.0\n\n\n\n\n\n"},{"title":"Base.Iterators.reverse","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.reverse","category":"function","text":"Iterators.reverse(itr)\n\nGiven an iterator itr, then reverse(itr) is an iterator over the same collection but in the reverse order. This iterator is \"lazy\" in that it does not make a copy of the collection in order to reverse it; see Base.reverse for an eager implementation.\n\n(By default, this returns an Iterators.Reverse object wrapping itr, which is iterable if the corresponding iterate methods are defined, but some itr types may implement more specialized Iterators.reverse behaviors.)\n\nNot all iterator types T support reverse-order iteration.  If T doesn't, then iterating over Iterators.reverse(itr::T) will throw a MethodError because of the missing iterate methods for Iterators.Reverse{T}. (To implement these methods, the original iterator itr::T can be obtained from an r::Iterators.Reverse{T} object by r.itr; more generally, one can use Iterators.reverse(r).)\n\nExamples\n\njulia> foreach(println, Iterators.reverse(1:5))\n5\n4\n3\n2\n1\n\n\n\n\n\n"},{"title":"Base.Iterators.only","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.only","category":"function","text":"only(x)\n\nReturn the one and only element of collection x, or throw an ArgumentError if the collection has zero or multiple elements.\n\nSee also first, last.\n\ncompat: Julia 1.4\nThis method requires at least Julia 1.4.\n\nExamples\n\njulia> only([\"a\"])\n\"a\"\n\njulia> only(\"a\")\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> only(())\nERROR: ArgumentError: Tuple contains 0 elements, must contain exactly 1 element\nStacktrace:\n[...]\n\njulia> only(('a', 'b'))\nERROR: ArgumentError: Tuple contains 2 elements, must contain exactly 1 element\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"Base.Iterators.peel","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.peel","category":"function","text":"peel(iter)\n\nReturns the first element and an iterator over the remaining elements.\n\nIf the iterator is empty return nothing (like iterate).\n\ncompat: Julia 1.7\nPrior versions throw a BoundsError if the iterator is empty.\n\nSee also: Iterators.drop, Iterators.take.\n\nExamples\n\njulia> (a, rest) = Iterators.peel(\"abc\");\n\njulia> a\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> collect(rest)\n2-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n\n\n\n\n\n"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html#Environment-Variables","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Julia can be configured with a number of environment variables, set either in the usual way for each operating system, or in a portable way from within Julia. Supposing that you want to set the environment variable JULIA_EDITOR to vim, you can type ENV[\"JULIA_EDITOR\"] = \"vim\" (for instance, in the REPL) to make this change on a case by case basis, or add the same to the user configuration file ~/.julia/config/startup.jl in the user's home directory to have a permanent effect. The current value of the same environment variable can be determined by evaluating ENV[\"JULIA_EDITOR\"]."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The environment variables that Julia uses generally start with JULIA. If InteractiveUtils.versioninfo is called with the keyword verbose=true, then the output will list any defined environment variables relevant for Julia, including those which include JULIA in their names."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nIt is recommended to avoid changing environment variables during runtime, such as within a ~/.julia/config/startup.jl.One reason is that some julia language variables, such as JULIA_NUM_THREADS and JULIA_PROJECT, need to be set before Julia starts.Similarly, __init__() functions of user modules in the sysimage (via PackageCompiler) are run before startup.jl, so setting environment variables in a startup.jl may be too late for user code.Further, changing environment variables during runtime can introduce data races into otherwise benign code.In Bash, environment variables can either be set manually by running, e.g., export JULIA_NUM_THREADS=4 before starting Julia, or by adding the same command to ~/.bashrc or ~/.bash_profile to set the variable each time Bash is started."},{"title":"File locations","page":"Environment Variables","location":"manual/environment-variables.html#File-locations","category":"section","text":""},{"title":"JULIA_BINDIR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_BINDIR","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The absolute path of the directory containing the Julia executable, which sets the global variable Sys.BINDIR. If $JULIA_BINDIR is not set, then Julia determines the value Sys.BINDIR at run-time."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The executable itself is one of"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"$JULIA_BINDIR/julia\n$JULIA_BINDIR/julia-debug"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"by default."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The global variable Base.DATAROOTDIR determines a relative path from Sys.BINDIR to the data directory associated with Julia. Then the path"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"$JULIA_BINDIR/$DATAROOTDIR/julia/base"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"determines the directory in which Julia initially searches for source files (via Base.find_source_file())."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Likewise, the global variable Base.SYSCONFDIR determines a relative path to the configuration file directory. Then Julia searches for a startup.jl file at"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"$JULIA_BINDIR/$SYSCONFDIR/julia/startup.jl\n$JULIA_BINDIR/../etc/julia/startup.jl"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"by default (via Base.load_julia_startup())."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"For example, a Linux installation with a Julia executable located at /bin/julia, a DATAROOTDIR of ../share, and a SYSCONFDIR of ../etc will have JULIA_BINDIR set to /bin, a source-file search path of"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"/share/julia/base"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"and a global configuration search path of"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"/etc/julia/startup.jl"},{"title":"JULIA_PROJECT","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PROJECT","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"A directory path that indicates which project should be the initial active project. Setting this environment variable has the same effect as specifying the --project start-up option, but --project has higher precedence. If the variable is set to @. (note the trailing dot) then Julia tries to find a project directory that contains Project.toml or JuliaProject.toml file from the current directory and its parents. See also the chapter on Code Loading."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nJULIA_PROJECT must be defined before starting julia; defining it in startup.jl is too late in the startup process."},{"title":"JULIA_LOAD_PATH","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_LOAD_PATH","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The JULIA_LOAD_PATH environment variable is used to populate the global Julia LOAD_PATH variable, which determines which packages can be loaded via import and using (see Code Loading)."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Unlike the shell PATH variable, empty entries in JULIA_LOAD_PATH are expanded to the default value of LOAD_PATH, [\"@\", \"@v#.#\", \"@stdlib\"] when populating LOAD_PATH. This allows easy appending, prepending, etc. of the load path value in shell scripts regardless of whether JULIA_LOAD_PATH is already set or not. For example, to prepend the directory /foo/bar to LOAD_PATH just do"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"export JULIA_LOAD_PATH=\"/foo/bar:$JULIA_LOAD_PATH\""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If the JULIA_LOAD_PATH environment variable is already set, its old value will be prepended with /foo/bar. On the other hand, if JULIA_LOAD_PATH is not set, then it will be set to /foo/bar: which will expand to a LOAD_PATH value of [\"/foo/bar\", \"@\", \"@v#.#\", \"@stdlib\"]. If JULIA_LOAD_PATH is set to the empty string, it expands to an empty LOAD_PATH array. In other words, the empty string is interpreted as a zero-element array, not a one-element array of the empty string. This behavior was chosen so that it would be possible to set an empty load path via the environment variable. If you want the default load path, either unset the environment variable or if it must have a value, set it to the string :."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nOn Windows, path elements are separated by the ; character, as is the case with most path lists on Windows. Replace : with ; in the above paragraph."},{"title":"JULIA_DEPOT_PATH","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_DEPOT_PATH","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The JULIA_DEPOT_PATH environment variable is used to populate the global Julia DEPOT_PATH variable, which controls where the package manager, as well as Julia's code loading mechanisms, look for package registries, installed packages, named environments, repo clones, cached compiled package images, configuration files, and the default location of the REPL's history file."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Unlike the shell PATH variable but similar to JULIA_LOAD_PATH, empty entries in JULIA_DEPOT_PATH have special behavior:"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"At the end, it is expanded to the default value of DEPOT_PATH, excluding the user depot.\nAt the start, it is expanded to the default value of DEPOT_PATH, including the user depot."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"This allows easy overriding of the user depot, while still retaining access to resources that are bundled with Julia, like cache files, artifacts, etc. For example, to switch the user depot to /foo/bar use a trailing :"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"export JULIA_DEPOT_PATH=\"/foo/bar:\""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"All package operations, like cloning registries or installing packages, will now write to /foo/bar, but since the empty entry is expanded to the default system depot, any bundled resources will still be available. If you really only want to use the depot at /foo/bar, and not load any bundled resources, simply set the environment variable to /foo/bar without the trailing colon."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"To append a depot at the end of the full default list, including the default user depot, use a leading :"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"export JULIA_DEPOT_PATH=\":/foo/bar\""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"There are two exceptions to the above rule. First, if JULIA_DEPOT_PATH is set to the empty string, it expands to an empty DEPOT_PATH array. In other words, the empty string is interpreted as a zero-element array, not a one-element array of the empty string. This behavior was chosen so that it would be possible to set an empty depot path via the environment variable."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Second, if no user depot is specified in JULIA_DEPOT_PATH, then the empty entry is expanded to the default depot including the user depot. This makes it possible to use the default depot, as if the environment variable was unset, by setting it to the string :."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nOn Windows, path elements are separated by the ; character, as is the case with most path lists on Windows. Replace : with ; in the above paragraph."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nJULIA_DEPOT_PATH must be defined before starting julia; defining it in startup.jl is too late in the startup process; at that point you can instead directly modify the DEPOT_PATH array, which is populated from the environment variable."},{"title":"JULIA_HISTORY","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_HISTORY","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The absolute path REPL.find_hist_file() of the REPL's history file. If $JULIA_HISTORY is not set, then REPL.find_hist_file() defaults to"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"$(DEPOT_PATH[1])/logs/repl_history.jl"},{"title":"JULIA_MAX_NUM_PRECOMPILE_FILES","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_MAX_NUM_PRECOMPILE_FILES","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Sets the maximum number of different instances of a single package that are to be stored in the precompile cache (default = 10)."},{"title":"JULIA_VERBOSE_LINKING","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_VERBOSE_LINKING","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to true, linker commands will be displayed during precompilation."},{"title":"Pkg.jl","page":"Environment Variables","location":"manual/environment-variables.html#Pkg.jl","category":"section","text":""},{"title":"JULIA_CI","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_CI","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to true, this indicates to the package server that any package operations are part of a continuous integration (CI) system for the purposes of gathering package usage statistics."},{"title":"JULIA_NUM_PRECOMPILE_TASKS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_NUM_PRECOMPILE_TASKS","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The number of parallel tasks to use when precompiling packages. See Pkg.precompile."},{"title":"JULIA_PKG_DEVDIR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_DEVDIR","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The default directory used by Pkg.develop for downloading packages."},{"title":"JULIA_PKG_IGNORE_HASHES","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_IGNORE_HASHES","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to 1, this will ignore incorrect hashes in artifacts. This should be used carefully, as it disables verification of downloads, but can resolve issues when moving files across different types of file systems. See Pkg.jl issue #2317 for more details."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.6\nThis is only supported in Julia 1.6 and above."},{"title":"JULIA_PKG_OFFLINE","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_OFFLINE","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to true, this will enable offline mode: see Pkg.offline."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.5\nPkg's offline mode requires Julia 1.5 or later."},{"title":"JULIA_PKG_PRECOMPILE_AUTO","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_PRECOMPILE_AUTO","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to 0, this will disable automatic precompilation by package actions which change the manifest. See Pkg.precompile."},{"title":"JULIA_PKG_SERVER","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_SERVER","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Specifies the URL of the package registry to use. By default, Pkg uses https://pkg.julialang.org to fetch Julia packages. In addition, you can disable the use of the PkgServer protocol, and instead access the packages directly from their hosts (GitHub, GitLab, etc.) by setting: export JULIA_PKG_SERVER=\"\""},{"title":"JULIA_PKG_SERVER_REGISTRY_PREFERENCE","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_SERVER_REGISTRY_PREFERENCE","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Specifies the preferred registry flavor. Currently supported values are conservative (the default), which will only publish resources that have been processed by the storage server (and thereby have a higher probability of being available from the PkgServers), whereas eager will publish registries whose resources have not necessarily been processed by the storage servers. Users behind restrictive firewalls that do not allow downloading from arbitrary servers should not use the eager flavor."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.7\nThis only affects Julia 1.7 and above."},{"title":"JULIA_PKG_UNPACK_REGISTRY","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_UNPACK_REGISTRY","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to true, this will unpack the registry instead of storing it as a compressed tarball."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.7\nThis only affects Julia 1.7 and above. Earlier versions will always unpack the registry."},{"title":"JULIA_PKG_USE_CLI_GIT","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_USE_CLI_GIT","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to true, Pkg operations which use the git protocol will use an external git executable instead of the default libgit2 library."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.7\nUse of the git executable is only supported on Julia 1.7 and above."},{"title":"JULIA_PKGRESOLVE_ACCURACY","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKGRESOLVE_ACCURACY","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The accuracy of the package resolver. This should be a positive integer, the default is 1."},{"title":"JULIA_PKG_PRESERVE_TIERED_INSTALLED","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_PRESERVE_TIERED_INSTALLED","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Change the default package installation strategy to Pkg.PRESERVE_TIERED_INSTALLED to let the package manager try to install versions of packages while keeping as many versions of packages already installed as possible."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.9\nThis only affects Julia 1.9 and above."},{"title":"JULIA_PKG_GC_AUTO","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_GC_AUTO","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to false, automatic garbage collection of packages and artifacts will be disabled; see Pkg.gc for more details."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.12\nThis environment variable is only supported on Julia 1.12 and above."},{"title":"Network transport","page":"Environment Variables","location":"manual/environment-variables.html#Network-transport","category":"section","text":""},{"title":"JULIA_NO_VERIFY_HOSTS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_NO_VERIFY_HOSTS","category":"section","text":""},{"title":"JULIA_SSL_NO_VERIFY_HOSTS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_SSL_NO_VERIFY_HOSTS","category":"section","text":""},{"title":"JULIA_SSH_NO_VERIFY_HOSTS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_SSH_NO_VERIFY_HOSTS","category":"section","text":""},{"title":"JULIA_ALWAYS_VERIFY_HOSTS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_ALWAYS_VERIFY_HOSTS","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Specify hosts whose identity should or should not be verified for specific transport layers. See NetworkOptions.verify_host"},{"title":"JULIA_SSL_CA_ROOTS_PATH","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_SSL_CA_ROOTS_PATH","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Specify the file or directory containing the certificate authority roots. See NetworkOptions.ca_roots"},{"title":"External applications","page":"Environment Variables","location":"manual/environment-variables.html#External-applications","category":"section","text":""},{"title":"JULIA_SHELL","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_SHELL","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The absolute path of the shell with which Julia should execute external commands (via Base.repl_cmd()). Defaults to the environment variable $SHELL, and falls back to /bin/sh if $SHELL is unset."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nOn Windows, this environment variable is ignored, and external commands are executed directly."},{"title":"JULIA_EDITOR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_EDITOR","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The editor returned by InteractiveUtils.editor() and used in, e.g., InteractiveUtils.edit, referring to the command of the preferred editor, for instance vim."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"$JULIA_EDITOR takes precedence over $VISUAL, which in turn takes precedence over $EDITOR. If none of these environment variables is set, then the editor is taken to be open on Windows and OS X, or /etc/alternatives/editor if it exists, or emacs otherwise."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"To use Visual Studio Code on Windows, set $JULIA_EDITOR to code.cmd."},{"title":"Parallelization","page":"Environment Variables","location":"manual/environment-variables.html#Parallelization","category":"section","text":""},{"title":"JULIA_CPU_THREADS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_CPU_THREADS","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Overrides the global variable Base.Sys.CPU_THREADS, the number of logical CPU cores available."},{"title":"JULIA_WORKER_TIMEOUT","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_WORKER_TIMEOUT","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"A Float64 that sets the value of Distributed.worker_timeout() (default: 60.0). This function gives the number of seconds a worker process will wait for a master process to establish a connection before dying."},{"title":"JULIA_NUM_THREADS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_NUM_THREADS","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"An unsigned 64-bit integer (uint64_t) or string that sets the maximum number of threads available to Julia. If $JULIA_NUM_THREADS is not set or is a non-positive integer, or if the number of CPU threads cannot be determined through system calls, then the number of threads is set to 1."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If $JULIA_NUM_THREADS is set to auto, then the number of threads will be set to the number of CPU threads. It can also be set to a comma-separated string to specify the size of the :default and :interactive threadpools, respectively:"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"# 5 threads in the :default pool and 2 in the :interactive pool\nexport JULIA_NUM_THREADS=5,2\n\n# `auto` threads in the :default pool and 1 in the :interactive pool\nexport JULIA_NUM_THREADS=auto,1"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nJULIA_NUM_THREADS must be defined before starting Julia; defining it in startup.jl is too late in the startup process."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.5\nIn Julia 1.5 and above the number of threads can also be specified on startup using the -t/--threads command line argument."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.7\nThe auto value for $JULIA_NUM_THREADS requires Julia 1.7 or above."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.9\nThe x,y format for threadpools requires Julia 1.9 or above."},{"title":"JULIA_THREAD_SLEEP_THRESHOLD","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_THREAD_SLEEP_THRESHOLD","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to a string that starts with the case-insensitive substring \"infinite\", then spinning threads never sleep. Otherwise, $JULIA_THREAD_SLEEP_THRESHOLD is interpreted as an unsigned 64-bit integer (uint64_t) and gives, in nanoseconds, the amount of time after which spinning threads should sleep."},{"title":"JULIA_NUM_GC_THREADS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_NUM_GC_THREADS","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Sets the number of threads used by Garbage Collection. If unspecified is set to the number of worker threads."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"compat: Julia 1.10\nThe environment variable was added in 1.10"},{"title":"JULIA_IMAGE_THREADS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_IMAGE_THREADS","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"An unsigned 32-bit integer that sets the number of threads used by image compilation in this Julia process. The value of this variable may be ignored if the module is a small module. If left unspecified, the smaller of the value of JULIA_CPU_THREADS or half the number of logical CPU cores is used in its place."},{"title":"JULIA_IMAGE_TIMINGS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_IMAGE_TIMINGS","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"A boolean value that determines if detailed timing information is printed during during image compilation. Defaults to 0."},{"title":"JULIA_EXCLUSIVE","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_EXCLUSIVE","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to anything besides 0, then Julia's thread policy is consistent with running on a dedicated machine: each thread in the default threadpool is affinitized.  Interactive threads remain under the control of the operating system scheduler."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Otherwise, Julia lets the operating system handle thread policy."},{"title":"Garbage Collection","page":"Environment Variables","location":"manual/environment-variables.html#Garbage-Collection","category":"section","text":""},{"title":"JULIA_HEAP_SIZE_HINT","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_HEAP_SIZE_HINT","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Environment variable equivalent to the --heap-size-hint=<size>[<unit>] command line option."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Forces garbage collection if memory usage is higher than the given value. The value may be specified as a number of bytes, optionally in units of:"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"- B  (bytes)\n- K  (kibibytes)\n- M  (mebibytes)\n- G  (gibibytes)\n- T  (tebibytes)\n- %  (percentage of physical memory)"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"For example, JULIA_HEAP_SIZE_HINT=1G would provide a 1 GB heap size hint to the garbage collector."},{"title":"REPL formatting","page":"Environment Variables","location":"manual/environment-variables.html#REPL-formatting","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Environment variables that determine how REPL output should be formatted at the terminal. The JULIA_*_COLOR variables should be set to ANSI terminal escape sequences. Julia provides a high-level interface with much of the same functionality; see the section on The Julia REPL."},{"title":"JULIA_ERROR_COLOR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_ERROR_COLOR","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The formatting Base.error_color() (default: light red, \"\\033[91m\") that errors should have at the terminal."},{"title":"JULIA_WARN_COLOR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_WARN_COLOR","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The formatting Base.warn_color() (default: yellow, \"\\033[93m\") that warnings should have at the terminal."},{"title":"JULIA_INFO_COLOR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_INFO_COLOR","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The formatting Base.info_color() (default: cyan, \"\\033[36m\") that info should have at the terminal."},{"title":"JULIA_INPUT_COLOR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_INPUT_COLOR","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The formatting Base.input_color() (default: normal, \"\\033[0m\") that input should have at the terminal."},{"title":"JULIA_ANSWER_COLOR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_ANSWER_COLOR","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The formatting Base.answer_color() (default: normal, \"\\033[0m\") that output should have at the terminal."},{"title":"NO_COLOR","page":"Environment Variables","location":"manual/environment-variables.html#NO_COLOR","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"When this variable is present and not an empty string (regardless of its value) then colored text will be disabled on the REPL. Can be overridden with the flag --color=yes or with the environment variable FORCE_COLOR. This environment variable is commonly recognized by command-line applications."},{"title":"FORCE_COLOR","page":"Environment Variables","location":"manual/environment-variables.html#FORCE_COLOR","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"When this variable is present and not an empty string (regardless of its value) then colored text will be enabled on the REPL. Can be overridden with the flag --color=no. This environment variable is commonly recognized by command-line applications."},{"title":"System and Package Image Building","page":"Environment Variables","location":"manual/environment-variables.html#System-and-Package-Image-Building","category":"section","text":""},{"title":"JULIA_CPU_TARGET","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_CPU_TARGET","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Modify the target machine architecture for (pre)compiling system and package images. JULIA_CPU_TARGET only affects machine code image generation being output to a disk cache. Unlike the --cpu-target, or -C, command line option, it does not influence just-in-time (JIT) code generation within a Julia session where machine code is only stored in memory."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Valid values for JULIA_CPU_TARGET can be obtained by executing julia -C help."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"To get the CPU target string that was used to build the current system image, use Sys.sysimage_target(). This can be useful for reproducing the same system image or understanding what CPU features were enabled during compilation."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Setting JULIA_CPU_TARGET is important for heterogeneous compute systems where processors of distinct types or features may be present. This is commonly encountered in high performance computing (HPC) clusters since the component nodes may be using distinct processors. In this case, you may want to use the sysimage CPU target to maintain the same configuration as the sysimage. See below for more details."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"The CPU target string is a list of strings separated by ; each string starts with a CPU or architecture name and followed by an optional list of features separated by ,. A generic or empty CPU name means the basic required feature set of the target ISA which is at least the architecture the C/C++ runtime is compiled with. Each string is interpreted by LLVM."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nPackage images can only target the same or more specific CPU features than their base system image."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"A few special features are supported:"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"sysimage\nA special keyword that can be used as a CPU target name, which will be replaced   with the CPU target string that was used to build the current system image. This allows   you to specify CPU targets that build upon or extend the current sysimage's target, which   is particularly helpful for creating package images that are as flexible as the sysimage.\nclone_all\nThis forces the target to have all functions in sysimg cloned.   When used in negative form (i.e. -clone_all), this disables full clone that's   enabled by default for certain targets.\nbase([0-9]*)\nThis specifies the (0-based) base target index. The base target is the target   that the current target is based on, i.e. the functions that are not being cloned   will use the version in the base target. This option causes the base target to be   fully cloned (as if clone_all is specified for it) if it is not the default target (0).   The index can only be smaller than the current index.\nopt_size\nOptimize for size with minimum performance impact. Clang/GCC's -Os.\nmin_size\nOptimize only for size. Clang's -Oz."},{"title":"Debugging and profiling","page":"Environment Variables","location":"manual/environment-variables.html#Debugging-and-profiling","category":"section","text":""},{"title":"JULIA_DEBUG","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_DEBUG","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Enable debug logging for a file or module, see Logging for more information."},{"title":"CI Debug Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html#CI-Debug-Environment-Variables","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Julia automatically enables verbose debugging options when certain continuous integration (CI) debug environment variables are set. This improves the debugging experience when CI jobs are re-run with debug logging enabled, by automatically:"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Enabling --trace-eval (location mode) to show expressions being evaluated\nSetting JULIA_TEST_VERBOSE=true to enable verbose test output"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"This allows developers to get detailed debugging information from CI runs without modifying their scripts or workflow files."},{"title":"JULIA_PROFILE_PEEK_HEAP_SNAPSHOT","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PROFILE_PEEK_HEAP_SNAPSHOT","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Enable collecting of a heap snapshot during execution via the profiling peek mechanism. See Triggered During Execution."},{"title":"JULIA_TIMING_SUBSYSTEMS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_TIMING_SUBSYSTEMS","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Allows you to enable or disable zones for a specific Julia run. For instance, setting the variable to +GC,-INFERENCE will enable the GC zones and disable the INFERENCE zones. See Dynamically Enabling and Disabling Zones."},{"title":"JULIA_GC_WAIT_FOR_DEBUGGER","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_GC_WAIT_FOR_DEBUGGER","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to anything besides 0, then the Julia garbage collector will wait for a debugger to attach instead of aborting whenever there's a critical error."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nThis environment variable only has an effect if Julia was compiled with garbage-collection debugging (that is, if WITH_GC_DEBUG_ENV is set to 1 in the build configuration)."},{"title":"ENABLE_JITPROFILING","page":"Environment Variables","location":"manual/environment-variables.html#ENABLE_JITPROFILING","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to anything besides 0, then the compiler will create and register an event listener for just-in-time (JIT) profiling."},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"note: Note\nThis environment variable only has an effect if Julia was compiled with JIT profiling support, using eitherIntel's VTune™ Amplifier (USE_INTEL_JITEVENTS set to 1 in the build configuration), or\nOProfile (USE_OPROFILE_JITEVENTS set to 1 in the build configuration).\nPerf (USE_PERF_JITEVENTS set to 1 in the build configuration). This integration is enabled by default."},{"title":"ENABLE_GDBLISTENER","page":"Environment Variables","location":"manual/environment-variables.html#ENABLE_GDBLISTENER","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"If set to anything besides 0 enables GDB registration of Julia code on release builds. On debug builds of Julia this is always enabled. Recommended to use with -g 2."},{"title":"JULIA_LLVM_ARGS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_LLVM_ARGS","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Arguments to be passed to the LLVM backend."},{"title":"JULIA_FALLBACK_REPL","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_FALLBACK_REPL","category":"section","text":""},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html","category":"page","text":"Forces the fallback repl instead of REPL.jl."},{"title":"Writing tests","page":"Writing tests","location":"devdocs/contributing/tests.html#Writing-tests","category":"section","text":""},{"title":"Writing tests","page":"Writing tests","location":"devdocs/contributing/tests.html","category":"page","text":"There are never enough tests. Track code coverage at Codecov, and help improve it."},{"title":"Writing tests","page":"Writing tests","location":"devdocs/contributing/tests.html","category":"page","text":"Go visit https://codecov.io/github/JuliaLang/julia.\nBrowse through the source files and find some untested functionality (highlighted in red) that you think you might be able to write a test for.\nWrite a test that exercises this functionality—you can add your test to one of the existing files, or start a new one, whichever seems most appropriate to you. If you're adding a new test file, make sure you include it in the list of tests in test/choosetests.jl. https://docs.julialang.org/en/v1/stdlib/Test/ may be helpful in explaining how the testing infrastructure works.\nRun make test-all to rebuild Julia and run your new test(s). If you had to fix a bug or add functionality in base, this will ensure that your test passes and that you have not introduced extraneous whitespace.\nSubmit the test as a pull request (PR)."},{"title":"Writing tests","page":"Writing tests","location":"devdocs/contributing/tests.html","category":"page","text":"Code for the buildbot configuration is maintained at: https://github.com/staticfloat/julia-buildbot\nYou can see the current buildbot setup at: https://build.julialang.org/builders\nIssue 9493 and issue 11885 have more detailed discussion on code coverage."},{"title":"Writing tests","page":"Writing tests","location":"devdocs/contributing/tests.html","category":"page","text":"Code coverage shows functionality that still needs \"proof of concept\" tests. These are important, as are tests for tricky edge cases, such as converting between integer types when the number to convert is near the maximum of the range of one of the integer types. Even if a function already has some coverage on Codecov, it may still benefit from tests for edge cases."},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian","category":"section","text":""},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"The (non-exported) Cartesian module provides macros that facilitate writing multidimensional algorithms. Most often you can write such algorithms with straightforward techniques; however, there are a few cases where Base.Cartesian is still useful or necessary."},{"title":"Principles of usage","page":"Base.Cartesian","location":"devdocs/cartesian.html#Principles-of-usage","category":"section","text":""},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"A simple example of usage is:"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"@nloops 3 i A begin\n    s += @nref 3 A i\nend"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"which generates the following code:"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"for i_3 = axes(A, 3)\n    for i_2 = axes(A, 2)\n        for i_1 = axes(A, 1)\n            s += A[i_1, i_2, i_3]\n        end\n    end\nend"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"In general, Cartesian allows you to write generic code that contains repetitive elements, like the nested loops in this example. Other applications include repeated expressions (e.g., loop unwinding) or creating function calls with variable numbers of arguments without using the \"splat\" construct (i...)."},{"title":"Basic syntax","page":"Base.Cartesian","location":"devdocs/cartesian.html#Basic-syntax","category":"section","text":""},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"The (basic) syntax of @nloops is as follows:"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"The first argument must be an integer (not a variable) specifying the number of loops.\nThe second argument is the symbol-prefix used for the iterator variable. Here we used i, and variables i_1, i_2, i_3 were generated.\nThe third argument specifies the range for each iterator variable. If you use a variable (symbol) here, it's taken as axes(A, dim). More flexibly, you can use the anonymous-function expression syntax described below.\nThe last argument is the body of the loop. Here, that's what appears between the begin...end."},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"There are some additional features of @nloops described in the reference section."},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"@nref follows a similar pattern, generating A[i_1,i_2,i_3] from @nref 3 A i. The general practice is to read from left to right, which is why @nloops is @nloops 3 i A expr (as in for i_2 = axes(A, 2), where i_2 is to the left and the range is to the right) whereas @nref is @nref 3 A i (as in A[i_1,i_2,i_3], where the array comes first)."},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"If you're developing code with Cartesian, you may find that debugging is easier when you examine the generated code, using @macroexpand:"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"julia> @macroexpand @nref 2 A i\n:(A[i_1, i_2])"},{"title":"Supplying the number of expressions","page":"Base.Cartesian","location":"devdocs/cartesian.html#Supplying-the-number-of-expressions","category":"section","text":""},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"The first argument to both of these macros is the number of expressions, which must be an integer. When you're writing a function that you intend to work in multiple dimensions, this may not be something you want to hard-code. The recommended approach is to use a @generated function. Here's an example:"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"@generated function mysum(A::Array{T,N}) where {T,N}\n    quote\n        s = zero(T)\n        @nloops $N i A begin\n            s += @nref $N A i\n        end\n        s\n    end\nend"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"Naturally, you can also prepare expressions or perform calculations before the quote block."},{"title":"Anonymous-function expressions as macro arguments","page":"Base.Cartesian","location":"devdocs/cartesian.html#Anonymous-function-expressions-as-macro-arguments","category":"section","text":""},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"Perhaps the single most powerful feature in Cartesian is the ability to supply anonymous-function expressions that get evaluated at parsing time. Let's consider a simple example:"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"@nexprs 2 j->(i_j = 1)"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"@nexprs generates n expressions that follow a pattern. This code would generate the following statements:"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"i_1 = 1\ni_2 = 1"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"In each generated statement, an \"isolated\" j (the variable of the anonymous function) gets replaced by values in the range 1:2. Generally speaking, Cartesian employs a LaTeX-like syntax. This allows you to do math on the index j. Here's an example computing the strides of an array:"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"s_1 = 1\n@nexprs 3 j->(s_{j+1} = s_j * size(A, j))"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"would generate expressions"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"s_1 = 1\ns_2 = s_1 * size(A, 1)\ns_3 = s_2 * size(A, 2)\ns_4 = s_3 * size(A, 3)"},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html","category":"page","text":"Anonymous-function expressions have many uses in practice."},{"title":"Macro reference","page":"Base.Cartesian","location":"devdocs/cartesian.html#dev-cartesian-reference","category":"section","text":""},{"title":"Base.Cartesian.@nloops","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@nloops","category":"macro","text":"@nloops N itersym rangeexpr bodyexpr\n@nloops N itersym rangeexpr preexpr bodyexpr\n@nloops N itersym rangeexpr preexpr postexpr bodyexpr\n\nGenerate N nested loops, using itersym as the prefix for the iteration variables. rangeexpr may be an anonymous-function expression, or a simple symbol var in which case the range is axes(var, d) for dimension d.\n\nOptionally, you can provide \"pre\" and \"post\" expressions. These get executed first and last, respectively, in the body of each loop. For example:\n\n@nloops 2 i A d -> j_d = min(i_d, 5) begin\n    s += @nref 2 A j\nend\n\nwould generate:\n\nfor i_2 = axes(A, 2)\n    j_2 = min(i_2, 5)\n    for i_1 = axes(A, 1)\n        j_1 = min(i_1, 5)\n        s += A[j_1, j_2]\n    end\nend\n\nIf you want just a post-expression, supply nothing for the pre-expression. Using parentheses and semicolons, you can supply multi-statement expressions.\n\n\n\n\n\n"},{"title":"Base.Cartesian.@nref","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@nref","category":"macro","text":"@nref N A indexexpr\n\nGenerate expressions like A[i_1, i_2, ...]. indexexpr can either be an iteration-symbol prefix, or an anonymous-function expression.\n\nExamples\n\njulia> @macroexpand Base.Cartesian.@nref 3 A i\n:(A[i_1, i_2, i_3])\n\n\n\n\n\n"},{"title":"Base.Cartesian.@nextract","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@nextract","category":"macro","text":"@nextract N esym isym\n\nGenerate N variables esym_1, esym_2, ..., esym_N to extract values from isym. isym can be either a Symbol or anonymous-function expression.\n\n@nextract 2 x y would generate\n\nx_1 = y[1]\nx_2 = y[2]\n\nwhile @nextract 3 x d->y[2d-1] yields\n\nx_1 = y[1]\nx_2 = y[3]\nx_3 = y[5]\n\n\n\n\n\n"},{"title":"Base.Cartesian.@nexprs","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@nexprs","category":"macro","text":"@nexprs N expr\n\nGenerate N expressions. expr should be an anonymous-function expression.\n\nExamples\n\njulia> @macroexpand Base.Cartesian.@nexprs 4 i -> y[i] = A[i+j]\nquote\n    y[1] = A[1 + j]\n    y[2] = A[2 + j]\n    y[3] = A[3 + j]\n    y[4] = A[4 + j]\nend\n\n\n\n\n\n"},{"title":"Base.Cartesian.@ncall","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@ncall","category":"macro","text":"@ncall N f sym...\n\nGenerate a function call expression. sym represents any number of function arguments, the last of which may be an anonymous-function expression and is expanded into N arguments.\n\nFor example, @ncall 3 func a generates\n\nfunc(a_1, a_2, a_3)\n\nwhile @ncall 2 func a b i->c[i] yields\n\nfunc(a, b, c[1], c[2])\n\n\n\n\n\n"},{"title":"Base.Cartesian.@ncallkw","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@ncallkw","category":"macro","text":"@ncallkw N f kw sym...\n\nGenerate a function call expression with keyword arguments kw.... As in the case of @ncall, sym represents any number of function arguments, the last of which may be an anonymous-function expression and is expanded into N arguments.\n\nExamples\n\njulia> using Base.Cartesian\n\njulia> f(x...; a, b = 1, c = 2, d = 3) = +(x..., a, b, c, d);\n\njulia> x_1, x_2 = (-1, -2); b = 0; kw = (c = 0, d = 0);\n\njulia> @ncallkw 2 f (; a = 0, b, kw...) x\n-3\n\n\n\n\n\n\n"},{"title":"Base.Cartesian.@ntuple","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@ntuple","category":"macro","text":"@ntuple N expr\n\nGenerates an N-tuple. @ntuple 2 i would generate (i_1, i_2), and @ntuple 2 k->k+1 would generate (2,3).\n\n\n\n\n\n"},{"title":"Base.Cartesian.@nall","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@nall","category":"macro","text":"@nall N expr\n\nCheck whether all of the expressions generated by the anonymous-function expression expr evaluate to true.\n\n@nall 3 d->(i_d > 1) would generate the expression (i_1 > 1 && i_2 > 1 && i_3 > 1). This can be convenient for bounds-checking.\n\n\n\n\n\n"},{"title":"Base.Cartesian.@nany","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@nany","category":"macro","text":"@nany N expr\n\nCheck whether any of the expressions generated by the anonymous-function expression expr evaluate to true.\n\n@nany 3 d->(i_d > 1) would generate the expression (i_1 > 1 || i_2 > 1 || i_3 > 1).\n\n\n\n\n\n"},{"title":"Base.Cartesian.@nif","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@nif","category":"macro","text":"@nif N conditionexpr expr\n@nif N conditionexpr expr elseexpr\n\nGenerates a sequence of if ... elseif ... else ... end statements. For example:\n\n@nif 3 d->(i_d >= size(A,d)) d->(error(\"Dimension \", d, \" too big\")) d->println(\"All OK\")\n\nwould generate:\n\nif i_1 > size(A, 1)\n    error(\"Dimension \", 1, \" too big\")\nelseif i_2 > size(A, 2)\n    error(\"Dimension \", 2, \" too big\")\nelse\n    println(\"All OK\")\nend\n\n\n\n\n\n"},{"title":"Pkg","page":"Pkg","location":"stdlib/Pkg.html#Pkg","category":"section","text":""},{"title":"Pkg","page":"Pkg","location":"stdlib/Pkg.html","category":"page","text":"Pkg is Julia's built-in package manager, and handles operations such as installing, updating and removing packages."},{"title":"Pkg","page":"Pkg","location":"stdlib/Pkg.html","category":"page","text":"note: Note\nWhat follows is a very brief introduction to Pkg. For more information on Project.toml files, Manifest.toml files, package version compatibility ([compat]), environments, registries, etc., it is highly recommended to read the full manual, which is available here: https://pkgdocs.julialang.org."},{"title":"Pkg","page":"Pkg","location":"stdlib/Pkg.html","category":"page","text":"import Markdown\nfile = joinpath(Sys.STDLIB, \"Pkg\", \"docs\", \"src\", \"getting-started.md\")\nstr = read(file, String)\nstr = replace(str, r\"^#.*$\"m => \"\")\nstr = replace(str, \"[API Reference](@ref)\" => \"[API Reference](https://pkgdocs.julialang.org/v1/api/)\")\nstr = replace(str, \"(@ref Working-with-Environments)\" => \"(https://pkgdocs.julialang.org/v1/environments/)\")\nstr = replace(str, \"(@ref Managing-Packages)\" => \"(https://pkgdocs.julialang.org/v1/managing-packages/)\")\nMarkdown.parse(str)"},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html#Using-AI-agents-to-work-on-Julia","category":"section","text":""},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html","category":"page","text":"![WARNING] You are responsible for the code you submit in PRs. Do not submit PRs containing AI-generated code that you do not understand or that does not meet the ordinary quality bar for PRs to julia."},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html","category":"page","text":"This page documents best practices for setting up AI agents to work with Julia. If you find additional prompt instructions that work well for common tasks, consider submitting a PR to add these to AGENTS.md."},{"title":"Google Jules","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html#Google-Jules","category":"section","text":""},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html","category":"page","text":"Use the following for your Initial Setup configuration."},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html","category":"page","text":"curl -fsSL https://install.julialang.org | sh -s -- -y --default-channel nightly\n. /home/swebot/.profile"},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html","category":"page","text":"Jules has access to the internet, so you can give it links to issues or additional documentation in your prompting."},{"title":"OpenAI Codex","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html#OpenAI-Codex","category":"section","text":""},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html","category":"page","text":"Configure the following:"},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html","category":"page","text":"Setup Script"},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html","category":"page","text":"apt update\napt install less\ncurl -fsSL https://install.julialang.org | sh -s -- -y --default-channel nightly\nsource /root/.bashrc\nmake -C /workspace/julia/doc alldeps JULIA_EXECUTABLE=\"/root/.juliaup/bin/julia\"\nmake -C /workspace/julia/test install-revise-deps JULIA_EXECUTABLE=\"/root/.juliaup/bin/julia\""},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html","category":"page","text":"Environment Variables"},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html","category":"page","text":"JULIA_PKG_OFFLINE=true"},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html","category":"page","text":"Codex does not have internet access after initial setup, so you cannot give it additional information as links - you will need to copy any relevant text into the prompt."},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html","category":"page","text":"Note that Codex rebuilds the environment after every invocation. This can add significant latency. Codex work best for well-defined tasks that can be solved in a single shot."},{"title":"SIMD Support","page":"SIMD Support","location":"base/simd-types.html#SIMD-Support","category":"section","text":""},{"title":"SIMD Support","page":"SIMD Support","location":"base/simd-types.html","category":"page","text":"Type VecElement{T} is intended for building libraries of SIMD operations. Practical use of it requires using llvmcall. The type is defined as:"},{"title":"SIMD Support","page":"SIMD Support","location":"base/simd-types.html","category":"page","text":"struct VecElement{T}\n    value::T\nend"},{"title":"SIMD Support","page":"SIMD Support","location":"base/simd-types.html","category":"page","text":"It has a special compilation rule: a homogeneous tuple of VecElement{T} maps to an LLVM vector type when T is a primitive bits type."},{"title":"SIMD Support","page":"SIMD Support","location":"base/simd-types.html","category":"page","text":"At -O3, the compiler might automatically vectorize operations on such tuples. For example, the following program, when compiled with julia -O3 generates two SIMD addition instructions (addps) on x86 systems:"},{"title":"SIMD Support","page":"SIMD Support","location":"base/simd-types.html","category":"page","text":"const m128 = NTuple{4,VecElement{Float32}}\n\nfunction add(a::m128, b::m128)\n    (VecElement(a[1].value+b[1].value),\n     VecElement(a[2].value+b[2].value),\n     VecElement(a[3].value+b[3].value),\n     VecElement(a[4].value+b[4].value))\nend\n\ntriple(c::m128) = add(add(c,c),c)\n\ncode_native(triple,(m128,))"},{"title":"SIMD Support","page":"SIMD Support","location":"base/simd-types.html","category":"page","text":"However, since the automatic vectorization cannot be relied upon, future use will mostly be via libraries that use llvmcall."},{"title":"Julia Syntax Highlighting","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html#Julia-Syntax-Highlighting","category":"section","text":""},{"title":"Julia Syntax Highlighting","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html","category":"page","text":"The JuliaSyntaxHighlighting library serves as a small convenience package to syntax highlight Julia code using JuliaSyntax and StyledStrings."},{"title":"Julia Syntax Highlighting","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html","category":"page","text":"It is intended for use across the standard library, and the wider ecosystem."},{"title":"Functions","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html#stdlib-jsh-api","category":"section","text":""},{"title":"JuliaSyntaxHighlighting.highlight","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html#JuliaSyntaxHighlighting.highlight","category":"function","text":"highlight(content::Union{AbstractString, IO},\n          ast::JuliaSyntax.GreenNode = <parsed content>;\n          syntax_errors::Bool = false) -> AnnotatedString{String}\n\nApply syntax highlighting to content using JuliaSyntax.\n\nBy default, JuliaSyntax.parseall is used to generate to ast with the ignore_errors keyword argument set to true. Alternatively, one may provide a pre-generated ast.\n\nWhen syntax_errors is set, the julia_error face is applied to detected syntax errors.\n\nwarning: Warning\nNote that the particular faces used by JuliaSyntax, and the way they are applied, is subject to change.\n\nExamples\n\njulia> JuliaSyntaxHighlighting.highlight(\"sum(1:8)\")\n\"sum(1:8)\"\n\njulia> JuliaSyntaxHighlighting.highlight(\"sum(1:8)\") |> Base.annotations\n6-element Vector{@NamedTuple{region::UnitRange{Int64}, label::Symbol, value}}:\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((1:3, :face, :julia_funcall))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((4:4, :face, :julia_rainbow_paren_1))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((5:5, :face, :julia_number))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((6:6, :face, :julia_operator))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((7:7, :face, :julia_number))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((8:8, :face, :julia_rainbow_paren_1))\n\n\n\n\n\n"},{"title":"JuliaSyntaxHighlighting.highlight!","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html#JuliaSyntaxHighlighting.highlight!","category":"function","text":"highlight!(content::Union{AnnotatedString, SubString{AnnotatedString}},\n           ast::JuliaSyntax.GreenNode = <parsed content>;\n           syntax_errors::Bool = false) -> content\n\nModify content by applying syntax highlighting using JuliaSyntax.\n\nBy default, JuliaSyntax.parseall is used to generate to ast with the ignore_errors keyword argument set to true. Alternatively, one may provide a pre-generated ast.\n\nWhen syntax_errors is set, the julia_error face is applied to detected syntax errors.\n\nwarning: Warning\nNote that the particular faces used by JuliaSyntax, and the way they are applied, is subject to change.\n\nExamples\n\njulia> str = Base.AnnotatedString(\"sum(1:8)\")\n\"sum(1:8)\"\n\njulia> JuliaSyntaxHighlighting.highlight!(str)\n\"sum(1:8)\"\n\njulia> Base.annotations(str)\n6-element Vector{@NamedTuple{region::UnitRange{Int64}, label::Symbol, value}}:\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((1:3, :face, :julia_funcall))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((4:4, :face, :julia_rainbow_paren_1))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((5:5, :face, :julia_number))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((6:6, :face, :julia_operator))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((7:7, :face, :julia_number))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((8:8, :face, :julia_rainbow_paren_1))\n\n\n\n\n\n"},{"title":"Faces","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html#stdlib-jsh-faces","category":"section","text":""},{"title":"Julia Syntax Highlighting","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html","category":"page","text":"The highlight/highlight! methods work by applying custom faces to Julia code. As part of the standard library, these faces use privileged face names, of the form julia_*. These can be re-used in other packages, and customised with faces.toml configuration."},{"title":"Julia Syntax Highlighting","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html","category":"page","text":"warning: Unstable faces\nThe particular faces used by JuliaSyntaxHighlighting are liable to change without warning in point releases. As the syntax highlighting rules are refined over time, changes should become less and less frequent though."},{"title":"Julia Syntax Highlighting","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html","category":"page","text":"The current set of faces, and their default values are as follows:"},{"title":"Julia Syntax Highlighting","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html","category":"page","text":"julia_macro: magenta\njulia_symbol: magenta\njulia_singleton_identifier: inherits from julia_symbol\njulia_type: yellow\njulia_typedec: bright blue\njulia_comment: grey\njulia_string: green\njulia_regex: inherits from julia_string\njulia_backslash_literal: magenta, inherits from julia_string\njulia_string_delim: bright green\njulia_cmdstring: inherits from julia_string\njulia_char: inherits from julia_string\njulia_char_delim: inherits from julia_string_delim\njulia_number: bright magenta\njulia_bool: inherits from julia_number\njulia_funcall: cyan\njulia_funcdef: cyan\njulia_broadcast: bright blue, bold\njulia_builtin: bright blue\njulia_operator: blue\njulia_comparator: inherits from julia_operator\njulia_assignment: bright red\njulia_keyword: red\njulia_parentheses: unstyled\njulia_unpaired_parentheses: inherit from julia_error and julia_parentheses\njulia_error: red background\njulia_rainbow_paren_1: bright green, inherits from julia_parentheses\njulia_rainbow_paren_2: bright blue, inherits from julia_parentheses\njulia_rainbow_paren_3: bright red, inherits from julia_parentheses\njulia_rainbow_paren_4: inherits from julia_rainbow_paren_1\njulia_rainbow_paren_5: inherits from julia_rainbow_paren_2\njulia_rainbow_paren_6: inherits from julia_rainbow_paren_3\njulia_rainbow_bracket_1: blue, inherits from julia_parentheses\njulia_rainbow_bracket_2: brightmagenta, inherits from `juliaparentheses`\njulia_rainbow_bracket_3: inherits from julia_rainbow_bracket_1\njulia_rainbow_bracket_4: inherits from julia_rainbow_bracket_2\njulia_rainbow_bracket_5: inherits from julia_rainbow_bracket_1\njulia_rainbow_bracket_6: inherits from julia_rainbow_bracket_2\njulia_rainbow_curly_1: bright yellow, inherits from julia_parentheses\njulia_rainbow_curly_2: yellow, inherits from julia_parentheses\njulia_rainbow_curly_3: inherits from julia_rainbow_curly_1\njulia_rainbow_curly_4: inherits from julia_rainbow_curly_2\njulia_rainbow_curly_5: inherits from julia_rainbow_curly_1\njulia_rainbow_curly_6: inherits from julia_rainbow_curly_2"},{"title":"Tasks","page":"Tasks","location":"base/parallel.html#Tasks","category":"section","text":""},{"title":"Core.Task","page":"Tasks","location":"base/parallel.html#Core.Task","category":"type","text":"Task(func[, reserved_stack::Int])\n\nCreate a Task (i.e. coroutine) to execute the given function func (which must be callable with no arguments). The task exits when this function returns. The task will run in the \"world age\" from the parent at construction when scheduled.\n\nThe optional reserved_stack argument specifies the size of the stack available for this task, in bytes. The default, 0, uses the system-dependent stack size default.\n\nwarning: Warning\nBy default tasks will have the sticky bit set to true t.sticky. This models the historic default for @async. Sticky tasks can only be run on the worker thread they are first scheduled on, and when scheduled will make the task that they were scheduled from sticky. To obtain the behavior of Threads.@spawn set the sticky bit manually to false.\n\nExamples\n\njulia> a() = sum(i for i in 1:1000);\n\njulia> b = Task(a);\n\nIn this example, b is a runnable Task that hasn't started yet.\n\n\n\n\n\n"},{"title":"Base.@task","page":"Tasks","location":"base/parallel.html#Base.@task","category":"macro","text":"@task\n\nWrap an expression in a Task without executing it, and return the Task. This only creates a task, and does not run it.\n\nwarning: Warning\nBy default tasks will have the sticky bit set to true t.sticky. This models the historic default for @async. Sticky tasks can only be run on the worker thread they are first scheduled on, and when scheduled will make the task that they were scheduled from sticky. To obtain the behavior of Threads.@spawn set the sticky bit manually to false.\n\nExamples\n\njulia> a1() = sum(i for i in 1:1000);\n\njulia> b = @task a1();\n\njulia> istaskstarted(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n"},{"title":"Base.@async","page":"Tasks","location":"base/parallel.html#Base.@async","category":"macro","text":"@async\n\nWrap an expression in a Task and add it to the local machine's scheduler queue.\n\nValues can be interpolated into @async via $, which copies the value directly into the constructed underlying closure. This allows you to insert the value of a variable, isolating the asynchronous code from changes to the variable's value in the current task.\n\nwarning: Warning\nIt is strongly encouraged to favor Threads.@spawn over @async always even when no parallelism is required especially in publicly distributed libraries.  This is because a use of @async disables the migration of the parent task across worker threads in the current implementation of Julia.  Thus, seemingly innocent use of @async in a library function can have a large impact on the performance of very different parts of user applications.\n\ncompat: Julia 1.4\nInterpolating values via $ is available as of Julia 1.4.\n\n\n\n\n\n"},{"title":"Base.asyncmap","page":"Tasks","location":"base/parallel.html#Base.asyncmap","category":"function","text":"asyncmap(f, c...; ntasks=0, batch_size=nothing)\n\nUses multiple concurrent tasks to map f over a collection (or multiple equal length collections). For multiple collection arguments, f is applied elementwise.\n\nThe output is guaranteed to be the same order as the elements of the collection(s) c.\n\nntasks specifies the number of tasks to run concurrently. Depending on the length of the collections, if ntasks is unspecified, up to 100 tasks will be used for concurrent mapping.\n\nntasks can also be specified as a zero-arg function. In this case, the number of tasks to run in parallel is checked before processing every element and a new task started if the value of ntasks_func is greater than the current number of tasks.\n\nIf batch_size is specified, the collection is processed in batch mode. f must then be a function that must accept a Vector of argument tuples and must return a vector of results. The input vector will have a length of batch_size or less.\n\nThe following examples highlight execution in different tasks by returning the objectid of the tasks in which the mapping function is executed.\n\nFirst, with ntasks undefined, each element is processed in a different task.\n\njulia> tskoid() = objectid(current_task());\n\njulia> asyncmap(x->tskoid(), 1:5)\n5-element Vector{UInt64}:\n 0x6e15e66c75c75853\n 0x440f8819a1baa682\n 0x9fb3eeadd0c83985\n 0xebd3e35fe90d4050\n 0x29efc93edce2b961\n\njulia> length(unique(asyncmap(x->tskoid(), 1:5)))\n5\n\nWith ntasks=2 all elements are processed in 2 tasks.\n\njulia> asyncmap(x->tskoid(), 1:5; ntasks=2)\n5-element Vector{UInt64}:\n 0x027ab1680df7ae94\n 0xa23d2f80cd7cf157\n 0x027ab1680df7ae94\n 0xa23d2f80cd7cf157\n 0x027ab1680df7ae94\n\njulia> length(unique(asyncmap(x->tskoid(), 1:5; ntasks=2)))\n2\n\nWith batch_size defined, the mapping function needs to be changed to accept an array of argument tuples and return an array of results. map is used in the modified mapping function to achieve this.\n\njulia> batch_func(input) = map(x->string(\"args_tuple: \", x, \", element_val: \", x[1], \", task: \", tskoid()), input)\nbatch_func (generic function with 1 method)\n\njulia> asyncmap(batch_func, 1:5; ntasks=2, batch_size=2)\n5-element Vector{String}:\n \"args_tuple: (1,), element_val: 1, task: 9118321258196414413\"\n \"args_tuple: (2,), element_val: 2, task: 4904288162898683522\"\n \"args_tuple: (3,), element_val: 3, task: 9118321258196414413\"\n \"args_tuple: (4,), element_val: 4, task: 4904288162898683522\"\n \"args_tuple: (5,), element_val: 5, task: 9118321258196414413\"\n\n\n\n\n\n"},{"title":"Base.asyncmap!","page":"Tasks","location":"base/parallel.html#Base.asyncmap!","category":"function","text":"asyncmap!(f, results, c...; ntasks=0, batch_size=nothing)\n\nLike asyncmap, but stores output in results rather than returning a collection.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\n\n\n\n\n"},{"title":"Base.current_task","page":"Tasks","location":"base/parallel.html#Base.current_task","category":"function","text":"current_task()\n\nGet the currently running Task.\n\n\n\n\n\n"},{"title":"Base.istaskdone","page":"Tasks","location":"base/parallel.html#Base.istaskdone","category":"function","text":"istaskdone(t::Task)::Bool\n\nDetermine whether a task has exited.\n\nExamples\n\njulia> a2() = sum(i for i in 1:1000);\n\njulia> b = Task(a2);\n\njulia> istaskdone(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n"},{"title":"Base.istaskstarted","page":"Tasks","location":"base/parallel.html#Base.istaskstarted","category":"function","text":"istaskstarted(t::Task)::Bool\n\nDetermine whether a task has started executing.\n\nExamples\n\njulia> a3() = sum(i for i in 1:1000);\n\njulia> b = Task(a3);\n\njulia> istaskstarted(b)\nfalse\n\n\n\n\n\n"},{"title":"Base.istaskfailed","page":"Tasks","location":"base/parallel.html#Base.istaskfailed","category":"function","text":"istaskfailed(t::Task)::Bool\n\nDetermine whether a task has exited because an exception was thrown.\n\nExamples\n\njulia> a4() = error(\"task failed\");\n\njulia> b = Task(a4);\n\njulia> istaskfailed(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskfailed(b)\ntrue\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n"},{"title":"Base.task_local_storage","page":"Tasks","location":"base/parallel.html#Base.task_local_storage-Tuple{Any}","category":"method","text":"task_local_storage(key)\n\nLook up the value of a key in the current task's task-local storage.\n\n\n\n\n\n"},{"title":"Base.task_local_storage","page":"Tasks","location":"base/parallel.html#Base.task_local_storage-Tuple{Any, Any}","category":"method","text":"task_local_storage(key, value)\n\nAssign a value to a key in the current task's task-local storage.\n\n\n\n\n\n"},{"title":"Base.task_local_storage","page":"Tasks","location":"base/parallel.html#Base.task_local_storage-Tuple{Function, Any, Any}","category":"method","text":"task_local_storage(body, key, value)\n\nCall the function body with a modified task-local storage, in which value is assigned to key; the previous value of key, or lack thereof, is restored afterwards. Useful for emulating dynamic scoping.\n\n\n\n\n\n"},{"title":"Core.ConcurrencyViolationError","page":"Tasks","location":"base/parallel.html#Core.ConcurrencyViolationError","category":"type","text":"ConcurrencyViolationError(msg) <: Exception\n\nAn error thrown when a detectable violation of concurrent semantics has occurred.\n\nA non-exhaustive list of examples of when this is used include:\n\nThrowing when a deadlock has been detected (e.g. wait(current_task()))\nKnown-unsafe behavior is attempted (e.g. yield(current_task))\nA known non-threadsafe datastructure is attempted to be modified from multiple concurrent tasks\nA lock is being unlocked that wasn't locked by this task\n\n\n\n\n\n"},{"title":"Scheduling","page":"Tasks","location":"base/parallel.html#Scheduling","category":"section","text":""},{"title":"Base.yield","page":"Tasks","location":"base/parallel.html#Base.yield","category":"function","text":"yield(t::Task, arg = nothing)\n\nA fast, unfair-scheduling version of schedule(t, arg); yield() which immediately yields to t before calling the scheduler.\n\nThrows a ConcurrencyViolationError if t is the currently running task.\n\n\n\n\n\nyield()\n\nSwitch to the scheduler to allow another scheduled task to run. A task that calls this function is still runnable, and will be restarted immediately if there are no other runnable tasks.\n\n\n\n\n\n"},{"title":"Base.yieldto","page":"Tasks","location":"base/parallel.html#Base.yieldto","category":"function","text":"yieldto(t::Task, arg = nothing)\n\nSwitch to the given task. The first time a task is switched to, the task's function is called with no arguments. On subsequent switches, arg is returned from the task's last call to yieldto. This is a low-level call that only switches tasks, not considering states or scheduling in any way. Its use is discouraged.\n\n\n\n\n\n"},{"title":"Base.sleep","page":"Tasks","location":"base/parallel.html#Base.sleep","category":"function","text":"sleep(seconds)\n\nBlock the current task for a specified number of seconds. The minimum sleep time is 1 millisecond or input of 0.001.\n\n\n\n\n\n"},{"title":"Base.schedule","page":"Tasks","location":"base/parallel.html#Base.schedule","category":"function","text":"schedule(t::Task, [val]; error=false)\n\nAdd a Task to the scheduler's queue. This causes the task to run constantly when the system is otherwise idle, unless the task performs a blocking operation such as wait.\n\nIf a second argument val is provided, it will be passed to the task (via the return value of yieldto) when it runs again. If error is true, the value is raised as an exception in the woken task.\n\nwarning: Warning\nIt is incorrect to use schedule on an arbitrary Task that has already been started. See the API reference for more information.\n\nwarning: Warning\nBy default tasks will have the sticky bit set to true t.sticky. This models the historic default for @async. Sticky tasks can only be run on the worker thread they are first scheduled on, and when scheduled will make the task that they were scheduled from sticky. To obtain the behavior of Threads.@spawn set the sticky bit manually to false.\n\nExamples\n\njulia> a5() = sum(i for i in 1:1000);\n\njulia> b = Task(a5);\n\njulia> istaskstarted(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskstarted(b)\ntrue\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n"},{"title":"Synchronization","page":"Tasks","location":"base/parallel.html#lib-task-sync","category":"section","text":""},{"title":"Base.errormonitor","page":"Tasks","location":"base/parallel.html#Base.errormonitor","category":"function","text":"errormonitor(t::Task)\n\nPrint an error log to stderr if task t fails.\n\nExamples\n\njulia> wait(errormonitor(Threads.@spawn error(\"task failed\")); throw = false)\nUnhandled Task ERROR: task failed\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"Base.@sync","page":"Tasks","location":"base/parallel.html#Base.@sync","category":"macro","text":"@sync\n\nWait until all lexically-enclosed uses of @async, @spawn, Distributed.@spawnat and Distributed.@distributed are complete. All exceptions thrown by enclosed async operations are collected and thrown as a CompositeException.\n\nExamples\n\njulia> Threads.nthreads()\n4\n\njulia> @sync begin\n           Threads.@spawn println(\"Thread-id $(Threads.threadid()), task 1\")\n           Threads.@spawn println(\"Thread-id $(Threads.threadid()), task 2\")\n       end;\nThread-id 3, task 1\nThread-id 1, task 2\n\n\n\n\n\n"},{"title":"Base.wait","page":"Tasks","location":"base/parallel.html#Base.wait","category":"function","text":"wait([x])\n\nBlock the current task until some event occurs.\n\nChannel: Wait for a value to be appended to the channel.\nCondition: Wait for notify on a condition and return the val parameter passed to notify. See the Condition-specific docstring of wait for the exact behavior.\nProcess: Wait for a process or process chain to exit. The exitcode field of a process can be used to determine success or failure.\nTask: Wait for a Task to finish. See the Task-specific docstring of wait for the exact behavior.\nRawFD: Wait for changes on a file descriptor (see the FileWatching package).\n\nIf no argument is passed, the task blocks for an undefined period. A task can only be restarted by an explicit call to schedule or yieldto.\n\nOften wait is called within a while loop to ensure a waited-for condition is met before proceeding.\n\n\n\n\n\n"},{"title":"Base.waitany","page":"Tasks","location":"base/parallel.html#Base.waitany","category":"function","text":"waitany(tasks; throw=true) -> (done_tasks, remaining_tasks)\n\nWait until at least one of the given tasks have been completed.\n\nIf throw is true, throw CompositeException when one of the completed tasks completes with an exception.\n\nThe return value consists of two task vectors. The first one consists of completed tasks, and the other consists of uncompleted tasks.\n\nwarning: Warning\nThis may scale poorly compared to writing code that uses multiple individual tasks that each runs serially, since this needs to scan the list of tasks each time and synchronize with each one every time this is called. Or consider using waitall(tasks; failfast=true) instead.\n\ncompat: Julia 1.12\nThis function requires at least Julia 1.12.\n\n\n\n\n\n"},{"title":"Base.waitall","page":"Tasks","location":"base/parallel.html#Base.waitall","category":"function","text":"waitall(tasks; failfast=true, throw=true) -> (done_tasks, remaining_tasks)\n\nWait until all the given tasks have been completed.\n\nIf failfast is true, the function will return when at least one of the given tasks is finished by exception. If throw is true, throw CompositeException when one of the completed tasks has failed.\n\nfailfast and throw keyword arguments work independently; when only throw=true is specified, this function waits for all the tasks to complete.\n\nThe return value consists of two task vectors. The first one consists of completed tasks, and the other consists of uncompleted tasks.\n\ncompat: Julia 1.12\nThis function requires at least Julia 1.12.\n\n\n\n\n\n"},{"title":"Base.fetch","page":"Tasks","location":"base/parallel.html#Base.fetch-Tuple{Task}","category":"method","text":"fetch(t::Task)\n\nWait for a Task to finish, then return its result value. If the task fails with an exception, a TaskFailedException (which wraps the failed task) is thrown.\n\n\n\n\n\n"},{"title":"Base.fetch","page":"Tasks","location":"base/parallel.html#Base.fetch-Tuple{Any}","category":"method","text":"fetch(x::Any)\n\nReturn x.\n\n\n\n\n\n"},{"title":"Base.timedwait","page":"Tasks","location":"base/parallel.html#Base.timedwait","category":"function","text":"timedwait(testcb, timeout::Real; pollint::Real=0.1)\n\nWait until testcb() returns true or timeout seconds have passed, whichever is earlier. The test function is polled every pollint seconds. The minimum value for pollint is 0.001 seconds, that is, 1 millisecond.\n\nReturn :ok or :timed_out.\n\nExamples\n\njulia> cb() = (sleep(5); return);\n\njulia> t = @async cb();\n\njulia> timedwait(()->istaskdone(t), 1)\n:timed_out\n\njulia> timedwait(()->istaskdone(t), 6.5)\n:ok\n\n\n\n\n\n"},{"title":"Base.Condition","page":"Tasks","location":"base/parallel.html#Base.Condition","category":"type","text":"Condition()\n\nCreate an edge-triggered event source that tasks can wait for. Tasks that call wait on a Condition are suspended and queued. Tasks are woken up when notify is later called on the Condition. Waiting on a condition can return a value or raise an error if the optional arguments of notify are used. Edge triggering means that only tasks waiting at the time notify is called can be woken up. For level-triggered notifications, you must keep extra state to keep track of whether a notification has happened. The Channel and Threads.Event types do this, and can be used for level-triggered events.\n\nThis object is NOT thread-safe. See Threads.Condition for a thread-safe version.\n\n\n\n\n\n"},{"title":"Base.Threads.Condition","page":"Tasks","location":"base/parallel.html#Base.Threads.Condition","category":"type","text":"Threads.Condition([lock])\n\nA thread-safe version of Base.Condition.\n\nTo call wait or notify on a Threads.Condition, you must first call lock on it. When wait is called, the lock is atomically released during blocking, and will be reacquired before wait returns. Therefore idiomatic use of a Threads.Condition c looks like the following:\n\nlock(c)\ntry\n    while !thing_we_are_waiting_for\n        wait(c)\n    end\nfinally\n    unlock(c)\nend\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n"},{"title":"Base.Event","page":"Tasks","location":"base/parallel.html#Base.Event","category":"type","text":"Event([autoreset=false])\n\nCreate a level-triggered event source. Tasks that call wait on an Event are suspended and queued until notify is called on the Event. After notify is called, the Event remains in a signaled state and tasks will no longer block when waiting for it, until reset is called.\n\nIf autoreset is true, at most one task will be released from wait for each call to notify.\n\nThis provides an acquire & release memory ordering on notify/wait.\n\ncompat: Julia 1.1\nThis functionality requires at least Julia 1.1.\n\ncompat: Julia 1.8\nThe autoreset functionality and memory ordering guarantee requires at least Julia 1.8.\n\n\n\n\n\n"},{"title":"Base.notify","page":"Tasks","location":"base/parallel.html#Base.notify","category":"function","text":"notify(condition, val=nothing; all=true, error=false)\n\nWake up tasks waiting for a condition, passing them val. If all is true (the default), all waiting tasks are woken, otherwise only one is. If error is true, the passed value is raised as an exception in the woken tasks.\n\nReturn the count of tasks woken up. Return 0 if no tasks are waiting on condition.\n\n\n\n\n\n"},{"title":"Base.reset","page":"Tasks","location":"base/parallel.html#Base.reset-Tuple{Base.Event}","category":"method","text":"reset(::Event)\n\nReset an Event back into an un-set state. Then any future calls to wait will block until notify is called again.\n\n\n\n\n\n"},{"title":"Base.Semaphore","page":"Tasks","location":"base/parallel.html#Base.Semaphore","category":"type","text":"Semaphore(sem_size)\n\nCreate a counting semaphore that allows at most sem_size acquires to be in use at any time. Each acquire must be matched with a release.\n\nThis provides a acquire & release memory ordering on acquire/release calls.\n\n\n\n\n\n"},{"title":"Base.acquire","page":"Tasks","location":"base/parallel.html#Base.acquire","category":"function","text":"acquire(f, s::Semaphore)\n\nExecute f after acquiring from Semaphore s, and release on completion or error.\n\nFor example, a do-block form that ensures only 2 calls of foo will be active at the same time:\n\ns = Base.Semaphore(2)\n@sync for _ in 1:100\n    Threads.@spawn begin\n        Base.acquire(s) do\n            foo()\n        end\n    end\nend\n\ncompat: Julia 1.8\nThis method requires at least Julia 1.8.\n\n\n\n\n\nacquire(s::Semaphore)\n\nWait for one of the sem_size permits to be available, blocking until one can be acquired.\n\n\n\n\n\n"},{"title":"Base.release","page":"Tasks","location":"base/parallel.html#Base.release","category":"function","text":"release(s::Semaphore)\n\nReturn one permit to the pool, possibly allowing another task to acquire it and resume execution.\n\n\n\n\n\n"},{"title":"Base.AbstractLock","page":"Tasks","location":"base/parallel.html#Base.AbstractLock","category":"type","text":"AbstractLock\n\nAbstract supertype describing types that implement the synchronization primitives: lock, trylock, unlock, and islocked.\n\n\n\n\n\n"},{"title":"Base.lock","page":"Tasks","location":"base/parallel.html#Base.lock","category":"function","text":"lock(f::Function, l::Lockable)\n\nAcquire the lock associated with l, execute f with the lock held, and release the lock when f returns. f will receive one positional argument: the value wrapped by l. If the lock is already locked by a different task/thread, wait for it to become available. When this function returns, the lock has been released, so the caller should not attempt to unlock it.\n\ncompat: Julia 1.11\nRequires at least Julia 1.11.\n\n\n\n\n\nlock(f::Function, lock)\n\nAcquire the lock, execute f with the lock held, and release the lock when f returns. If the lock is already locked by a different task/thread, wait for it to become available.\n\nWhen this function returns, the lock has been released, so the caller should not attempt to unlock it.\n\nSee also: @lock.\n\ncompat: Julia 1.7\nUsing a Channel as the second argument requires Julia 1.7 or later.\n\n\n\n\n\nlock(lock)\n\nAcquire the lock when it becomes available. If the lock is already locked by a different task/thread, wait for it to become available.\n\nEach lock must be matched by an unlock.\n\n\n\n\n\n"},{"title":"Base.unlock","page":"Tasks","location":"base/parallel.html#Base.unlock","category":"function","text":"unlock(lock)\n\nReleases ownership of the lock.\n\nIf this is a recursive lock which has been acquired before, decrement an internal counter and return immediately.\n\n\n\n\n\n"},{"title":"Base.trylock","page":"Tasks","location":"base/parallel.html#Base.trylock","category":"function","text":"trylock(lock) -> Success (Boolean)\n\nAcquire the lock if it is available, and return true if successful. If the lock is already locked by a different task/thread, return false.\n\nEach successful trylock must be matched by an unlock.\n\nFunction trylock combined with islocked can be used for writing the test-and-test-and-set or exponential backoff algorithms if it is supported by the typeof(lock) (read its documentation).\n\n\n\n\n\n"},{"title":"Base.islocked","page":"Tasks","location":"base/parallel.html#Base.islocked","category":"function","text":"islocked(lock) -> Status (Boolean)\n\nCheck whether the lock is held by any task/thread. This function alone should not be used for synchronization. However, islocked combined with trylock can be used for writing the test-and-test-and-set or exponential backoff algorithms if it is supported by the typeof(lock) (read its documentation).\n\nExtended help\n\nFor example, an exponential backoff can be implemented as follows if the lock implementation satisfied the properties documented below.\n\nnspins = 0\nwhile true\n    while islocked(lock)\n        GC.safepoint()\n        nspins += 1\n        nspins > LIMIT && error(\"timeout\")\n    end\n    trylock(lock) && break\n    backoff()\nend\n\nImplementation\n\nA lock implementation is advised to define islocked with the following properties and note it in its docstring.\n\nislocked(lock) is data-race-free.\nIf islocked(lock) returns false, an immediate invocation of trylock(lock) must succeed (returns true) if there is no interference from other tasks.\n\n\n\n\n\n"},{"title":"Base.ReentrantLock","page":"Tasks","location":"base/parallel.html#Base.ReentrantLock","category":"type","text":"ReentrantLock()\n\nCreates a re-entrant lock for synchronizing Tasks. The same task can acquire the lock as many times as required (this is what the \"Reentrant\" part of the name means). Each lock must be matched with an unlock.\n\nCalling lock will also inhibit running of finalizers on that thread until the corresponding unlock. Use of the standard lock pattern illustrated below should naturally be supported, but beware of inverting the try/lock order or missing the try block entirely (e.g. attempting to return with the lock still held):\n\nThis provides a acquire/release memory ordering on lock/unlock calls.\n\nlock(l)\ntry\n    <atomic work>\nfinally\n    unlock(l)\nend\n\nIf !islocked(lck::ReentrantLock) holds, trylock(lck) succeeds unless there are other tasks attempting to hold the lock \"at the same time.\"\n\n\n\n\n\n"},{"title":"Base.@lock","page":"Tasks","location":"base/parallel.html#Base.@lock","category":"macro","text":"@lock l expr\n\nMacro version of lock(f, l::AbstractLock) but with expr instead of f function. Expands to:\n\nlock(l)\ntry\n    expr\nfinally\n    unlock(l)\nend\n\nThis is similar to using lock with a do block, but avoids creating a closure and thus can improve the performance.\n\ncompat: Compat\n@lock was added in Julia 1.3, and exported in Julia 1.10.\n\n\n\n\n\n"},{"title":"Base.Lockable","page":"Tasks","location":"base/parallel.html#Base.Lockable","category":"type","text":"Lockable(value, lock = ReentrantLock())\n\nCreates a Lockable object that wraps value and associates it with the provided lock. This object supports @lock, lock, trylock, unlock. To access the value, index the lockable object while holding the lock.\n\ncompat: Julia 1.11\nRequires at least Julia 1.11.\n\nExample\n\njulia> locked_list = Base.Lockable(Int[]);\n\njulia> @lock(locked_list, push!(locked_list[], 1)) # must hold the lock to access the value\n1-element Vector{Int64}:\n 1\n\njulia> lock(summary, locked_list)\n\"1-element Vector{Int64}\"\n\n\n\n\n\n"},{"title":"Channels","page":"Tasks","location":"base/parallel.html#Channels","category":"section","text":""},{"title":"Base.AbstractChannel","page":"Tasks","location":"base/parallel.html#Base.AbstractChannel","category":"type","text":"AbstractChannel{T}\n\nRepresentation of a channel passing objects of type T.\n\n\n\n\n\n"},{"title":"Base.Channel","page":"Tasks","location":"base/parallel.html#Base.Channel","category":"type","text":"Channel{T=Any}(size::Int=0)\n\nConstructs a Channel with an internal buffer that can hold a maximum of size objects of type T. put! calls on a full channel block until an object is removed with take!.\n\nChannel(0) constructs an unbuffered channel. put! blocks until a matching take! is called. And vice-versa.\n\nOther constructors:\n\nChannel(): default constructor, equivalent to Channel{Any}(0)\nChannel(Inf): equivalent to Channel{Any}(typemax(Int))\nChannel(sz): equivalent to Channel{Any}(sz)\n\ncompat: Julia 1.3\nThe default constructor Channel() and default size=0 were added in Julia 1.3.\n\n\n\n\n\n"},{"title":"Base.Channel","page":"Tasks","location":"base/parallel.html#Base.Channel-Tuple{Function}","category":"method","text":"Channel{T=Any}(func::Function, size=0; taskref=nothing, spawn=false, threadpool=nothing)\n\nCreate a new task from func, bind it to a new channel of type T and size size, and schedule the task, all in a single call. The channel is automatically closed when the task terminates.\n\nfunc must accept the bound channel as its only argument.\n\nIf you need a reference to the created task, pass a Ref{Task} object via the keyword argument taskref.\n\nIf spawn=true, the Task created for func may be scheduled on another thread in parallel, equivalent to creating a task via Threads.@spawn.\n\nIf spawn=true and the threadpool argument is not set, it defaults to :default.\n\nIf the threadpool argument is set (to :default or :interactive), this implies that spawn=true and the new Task is spawned to the specified threadpool.\n\nReturn a Channel.\n\nExamples\n\njulia> chnl = Channel() do ch\n           foreach(i -> put!(ch, i), 1:4)\n       end;\n\njulia> typeof(chnl)\nChannel{Any}\n\njulia> for i in chnl\n           @show i\n       end;\ni = 1\ni = 2\ni = 3\ni = 4\n\nReferencing the created task:\n\njulia> taskref = Ref{Task}();\n\njulia> chnl = Channel(taskref=taskref) do ch\n           println(take!(ch))\n       end;\n\njulia> istaskdone(taskref[])\nfalse\n\njulia> put!(chnl, \"Hello\");\nHello\n\njulia> istaskdone(taskref[])\ntrue\n\ncompat: Julia 1.3\nThe spawn= parameter was added in Julia 1.3. This constructor was added in Julia 1.3. In earlier versions of Julia, Channel used keyword arguments to set size and T, but those constructors are deprecated.\n\ncompat: Julia 1.9\nThe threadpool= argument was added in Julia 1.9.\n\njulia> chnl = Channel{Char}(1, spawn=true) do ch\n           for c in \"hello world\"\n               put!(ch, c)\n           end\n       end;\n\njulia> String(collect(chnl))\n\"hello world\"\n\n\n\n\n\n"},{"title":"Base.put!","page":"Tasks","location":"base/parallel.html#Base.put!-Tuple{Channel, Any}","category":"method","text":"put!(c::Channel, v)\n\nAppend an item v to the channel c. Blocks if the channel is full.\n\nFor unbuffered channels, blocks until a take! is performed by a different task.\n\ncompat: Julia 1.1\nv now gets converted to the channel's type with convert as put! is called.\n\n\n\n\n\n"},{"title":"Base.take!","page":"Tasks","location":"base/parallel.html#Base.take!-Tuple{Channel}","category":"method","text":"take!(c::Channel)\n\nRemoves and returns a value from a Channel in order. Blocks until data is available. For unbuffered channels, blocks until a put! is performed by a different task.\n\nExamples\n\nBuffered channel\n\njulia> c = Channel(1);\n\njulia> put!(c, 1);\n\njulia> take!(c)\n1\n\nUnbuffered channel\n\njulia> c = Channel(0);\n\njulia> task = Task(() -> put!(c, 1));\n\njulia> schedule(task);\n\njulia> take!(c)\n1\n\n\n\n\n\n"},{"title":"Base.isfull","page":"Tasks","location":"base/parallel.html#Base.isfull-Tuple{Channel}","category":"method","text":"isfull(c::Channel)\n\nDetermines if a Channel is full, in the sense that calling put!(c, some_value) would have blocked. Returns immediately, does not block.\n\nNote that it may frequently be the case that put! will not block after this returns true. Users must take precautions not to accidentally create live-lock bugs in their code by calling this method, as these are generally harder to debug than deadlocks. It is also possible that put! will block after this call returns false, if there are multiple producer tasks calling put! in parallel.\n\nExamples\n\nBuffered channel\n\njulia> c = Channel(1); # capacity = 1\n\njulia> isfull(c)\nfalse\n\njulia> put!(c, 1);\n\njulia> isfull(c)\ntrue\n\nUnbuffered channel\n\njulia> c = Channel(); # capacity = 0\n\njulia> isfull(c) # unbuffered channel is always full\ntrue\n\n\n\n\n\n"},{"title":"Base.isready","page":"Tasks","location":"base/parallel.html#Base.isready-Tuple{Channel}","category":"method","text":"isready(c::Channel)\n\nDetermines whether a Channel has a value stored in it. Returns immediately, does not block.\n\nFor unbuffered channels, return true if there are tasks waiting on a put!.\n\nExamples\n\nBuffered channel\n\njulia> c = Channel(1);\n\njulia> isready(c)\nfalse\n\njulia> put!(c, 1);\n\njulia> isready(c)\ntrue\n\nUnbuffered channel\n\njulia> c = Channel();\n\njulia> isready(c)  # no tasks waiting to put!\nfalse\n\njulia> task = Task(() -> put!(c, 1));\n\njulia> schedule(task);  # schedule a put! task\n\njulia> isready(c)\ntrue\n\n\n\n\n\n"},{"title":"Base.isopen","page":"Tasks","location":"base/parallel.html#Base.isopen-Tuple{Channel}","category":"method","text":"isopen(c::Channel)\n\nDetermines whether a Channel is open for new put! operations. Notice that a Channel can be closed and still have buffered elements which can be consumed with take!.\n\nExamples\n\nBuffered channel with task\n\njulia> c = Channel(ch -> put!(ch, 1), 1);\n\njulia> isopen(c) # The channel is closed to new `put!`s\nfalse\n\njulia> isready(c) # The channel is closed but still contains elements\ntrue\n\njulia> take!(c)\n1\n\njulia> isready(c)\nfalse\n\nUnbuffered channel\n\njulia> c = Channel{Int}();\n\njulia> isopen(c)\ntrue\n\njulia> close(c)\n\njulia> isopen(c)\nfalse\n\n\n\n\n\n"},{"title":"Base.fetch","page":"Tasks","location":"base/parallel.html#Base.fetch-Tuple{Channel}","category":"method","text":"fetch(c::Channel)\n\nWaits for and returns (without removing) the first available item from the Channel. Note: fetch is unsupported on an unbuffered (0-size) Channel.\n\nExamples\n\nBuffered channel\n\njulia> c = Channel(3) do ch\n           foreach(i -> put!(ch, i), 1:3)\n       end;\n\njulia> fetch(c)\n1\n\njulia> collect(c)  # item is not removed\n3-element Vector{Any}:\n 1\n 2\n 3\n\n\n\n\n\n"},{"title":"Base.close","page":"Tasks","location":"base/parallel.html#Base.close-Tuple{Channel}","category":"method","text":"close(c::Channel[, excp::Exception])\n\nClose a channel. An exception (optionally given by excp), is thrown by:\n\nput! on a closed channel.\ntake! and fetch on an empty, closed channel.\n\n\n\n\n\n"},{"title":"Base.bind","page":"Tasks","location":"base/parallel.html#Base.bind-Tuple{Channel, Task}","category":"method","text":"bind(chnl::Channel, task::Task)\n\nAssociate the lifetime of chnl with a task. Channel chnl is automatically closed when the task terminates. Any uncaught exception in the task is propagated to all waiters on chnl.\n\nThe chnl object can be explicitly closed independent of task termination. Terminating tasks have no effect on already closed Channel objects.\n\nWhen a channel is bound to multiple tasks, the first task to terminate will close the channel. When multiple channels are bound to the same task, termination of the task will close all of the bound channels.\n\nExamples\n\njulia> c = Channel(0);\n\njulia> task = @async foreach(i->put!(c, i), 1:4);\n\njulia> bind(c,task);\n\njulia> for i in c\n           @show i\n       end;\ni = 1\ni = 2\ni = 3\ni = 4\n\njulia> isopen(c)\nfalse\n\njulia> c = Channel(0);\n\njulia> task = @async (put!(c, 1); error(\"foo\"));\n\njulia> bind(c, task);\n\njulia> take!(c)\n1\n\njulia> put!(c, 1);\nERROR: TaskFailedException\nStacktrace:\n[...]\n    nested task error: foo\n[...]\n\n\n\n\n\n"},{"title":"Low-level synchronization using schedule and wait","page":"Tasks","location":"base/parallel.html#low-level-schedule-wait","category":"section","text":""},{"title":"Tasks","page":"Tasks","location":"base/parallel.html","category":"page","text":"The easiest correct use of schedule is on a Task that is not started (scheduled) yet. However, it is possible to use schedule and wait as a very low-level building block for constructing synchronization interfaces. A crucial pre-condition of calling schedule(task) is that the caller must \"own\" the task; i.e., it must know that the call to wait in the given task is happening at the locations known to the code calling schedule(task). One strategy for ensuring such pre-condition is to use atomics, as demonstrated in the following example:"},{"title":"Tasks","page":"Tasks","location":"base/parallel.html","category":"page","text":"@enum OWEState begin\n    OWE_EMPTY\n    OWE_WAITING\n    OWE_NOTIFYING\nend\n\nmutable struct OneWayEvent\n    @atomic state::OWEState\n    task::Task\n    OneWayEvent() = new(OWE_EMPTY)\nend\n\nfunction Base.notify(ev::OneWayEvent)\n    state = @atomic ev.state\n    while state !== OWE_NOTIFYING\n        # Spin until we successfully update the state to OWE_NOTIFYING:\n        state, ok = @atomicreplace(ev.state, state => OWE_NOTIFYING)\n        if ok\n            if state == OWE_WAITING\n                # OWE_WAITING -> OWE_NOTIFYING transition means that the waiter task is\n                # already waiting or about to call `wait`. The notifier task must wake up\n                # the waiter task.\n                schedule(ev.task)\n            else\n                @assert state == OWE_EMPTY\n                # Since we are assuming that there is only one notifier task (for\n                # simplicity), we know that the other possible case here is OWE_EMPTY.\n                # We do not need to do anything because we know that the waiter task has\n                # not called `wait(ev::OneWayEvent)` yet.\n            end\n            break\n        end\n    end\n    return\nend\n\nfunction Base.wait(ev::OneWayEvent)\n    ev.task = current_task()\n    state, ok = @atomicreplace(ev.state, OWE_EMPTY => OWE_WAITING)\n    if ok\n        # OWE_EMPTY -> OWE_WAITING transition means that the notifier task is guaranteed to\n        # invoke OWE_WAITING -> OWE_NOTIFYING transition. The waiter task must call\n        # `wait()` immediately. In particular, it MUST NOT invoke any function that may\n        # yield to the scheduler at this point in code.\n        wait()\n    else\n        @assert state == OWE_NOTIFYING\n        # Otherwise, the `state` must have already been moved to OWE_NOTIFYING by the\n        # notifier task.\n    end\n    return\nend\n\nev = OneWayEvent()\n@sync begin\n    Threads.@spawn begin\n        wait(ev)\n        println(\"done\")\n    end\n    println(\"notifying...\")\n    notify(ev)\nend\n\n# output\nnotifying...\ndone"},{"title":"Tasks","page":"Tasks","location":"base/parallel.html","category":"page","text":"OneWayEvent lets one task to wait for another task's notify. It is a limited communication interface since wait can only be used once from a single task (note the non-atomic assignment of ev.task)"},{"title":"Tasks","page":"Tasks","location":"base/parallel.html","category":"page","text":"In this example, notify(ev::OneWayEvent) is allowed to call schedule(ev.task) if and only if it modifies the state from OWE_WAITING to OWE_NOTIFYING. This lets us know that the task executing wait(ev::OneWayEvent) is now in the ok branch and that there cannot be other tasks that try to schedule(ev.task) since their @atomicreplace(ev.state, state => OWE_NOTIFYING) will fail."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html#scope-of-variables","category":"section","text":""},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"The scope of a variable is the region of code within which a variable is accessible. Variable scoping helps avoid variable naming conflicts. The concept is intuitive: two functions can both have arguments called x without the two x's referring to the same thing. Similarly, there are many other cases where different blocks of code can use the same name without referring to the same thing. The rules for when the same variable name does or doesn't refer to the same thing are called scope rules; this section spells them out in detail."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Certain constructs in the language introduce scope blocks, which are regions of code that are eligible to be the scope of some set of variables. The scope of a variable cannot be an arbitrary set of source lines; instead, it will always line up with one of these blocks. There are two main types of scopes in Julia, global scope and local scope. The latter can be nested. There is also a distinction in Julia between constructs which introduce a \"hard scope\" and those which only introduce a \"soft scope\", which affects whether shadowing a global variable by the same name is allowed or not."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"info: Summary\nVariables defined in global scope may be undefined in inner local scopes, depending on where the code is run, in order to balance safety and convenience. The hard and soft local scoping rules define the interplay between global and local variables.However, variables defined only in local scope behave consistently in all contexts. If the variable is already defined, it will be reused. If the variable is not defined, it will be made available to the current and inner scopes (but not outer scopes)."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"tip: A Common Confusion\nIf you run into an unexpectedly undefined variable,# Print the numbers 1 through 5\ni = 0\nwhile i < 5\n    i += 1     # ERROR: UndefVarError: `i` not defined\n    println(i)\nenda simple fix is to change all global variable definitions into local definitions by wrapping the code in a let block or function.# Print the numbers 1 through 5\nlet i = 0\n    while i < 5\n        i += 1     # Now outer `i` is defined in the inner scope of the while loop\n        println(i)\n    end\nendThis is a common source of confusion when writing procedural scripts, but it becomes a non-issue if code is moved inside functions or executed interactively in the REPL.See also the global and local keywords to explicitly achieve any desired scoping behavior."},{"title":"Scope Constructs","page":"Scope of Variables","location":"manual/variables-and-scoping.html#man-scope-table","category":"section","text":""},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"The constructs introducing scope blocks are:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Construct Scope Type Introduced Scope Types Able to Contain Construct\nmodule, baremodule global global\nstruct local (hard) global\nmacro local (hard) global\nfor, while, try local (soft) global, local\nfunction, do, let, comprehensions, generators local (hard) global, local"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Notably missing from this table are begin blocks and if blocks which do not introduce new scopes. The three types of scopes follow somewhat different rules which will be explained below."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Julia uses lexical scoping, meaning that a function's scope does not inherit from its caller's scope, but from the scope in which the function was defined. For example, in the following code the x inside foo refers to the x in the global scope of its module Bar:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> module Bar\n           x = 1\n           foo() = x\n       end;"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"and not a x in the scope where foo is used:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> import .Bar\n\njulia> x = -1;\n\njulia> Bar.foo()\n1"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Thus lexical scope means that what a variable in a particular piece of code refers to can be deduced from the code in which it appears alone and does not depend on how the program executes. A scope nested inside another scope can \"see\" variables in all the outer scopes in which it is contained. Outer scopes, on the other hand, cannot see variables in inner scopes."},{"title":"Global Scope","page":"Scope of Variables","location":"manual/variables-and-scoping.html#Global-Scope","category":"section","text":""},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Each module introduces a new global scope, separate from the global scope of all other modules—there is no all-encompassing global scope. Modules can introduce variables of other modules into their scope through the using or import statements or through qualified access using the dot-notation, i.e. each module is a so-called namespace as well as a first-class data structure associating names with values."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"If a top-level expression contains a variable declaration with keyword local, then that variable is not accessible outside that expression. The variable inside the expression does not affect global variables of the same name. An example is to declare local x in a begin or if block at the top-level:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> x = 1\n       begin\n           local x = 0\n           @show x\n       end\n       @show x;\nx = 0\nx = 1"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Note that the interactive prompt (aka REPL) is in the global scope of the module Main."},{"title":"Local Scope","page":"Scope of Variables","location":"manual/variables-and-scoping.html#local-scope","category":"section","text":""},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"A new local scope is introduced by most code blocks (see above table for a complete list). If such a block is syntactically nested inside of another local scope, the scope it creates is nested inside of all the local scopes that it appears within, which are all ultimately nested inside of the global scope of the module in which the code is evaluated. Variables in outer scopes are visible from any scope they contain — meaning that they can be read and written in inner scopes — unless there is a variable with the same name that \"shadows\" the outer variable of the same name. This is true even if the outer local is declared after (in the sense of textually below) an inner block. When we say that a variable \"exists\" in a given scope, this means that a variable by that name exists in any of the scopes that the current scope is nested inside of, including the current one. If a variable's value is used in a local scope, but nothing with its name exists in this scope, it is assumed to be a global."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Some programming languages require explicitly declaring new variables before using them. Explicit declaration works in Julia too: in any local scope, writing local x declares a new local variable in that scope, regardless of whether there is already a variable named x in an outer scope or not. Declaring each new variable like this is somewhat verbose and tedious, however, so Julia, like many other languages, considers assignment to a variable name that doesn't already exist to implicitly declare that variable. If the current scope is global, the new variable is global; if the current scope is local, the new variable is local to the innermost local scope and will be visible inside of that scope but not outside of it. If you assign to an existing local, it always updates that existing local: you can only shadow a local by explicitly declaring a new local in a nested scope with the local keyword. In particular, this applies to variables assigned in inner functions, which may surprise users coming from Python where assignment in an inner function creates a new local unless the variable is explicitly declared to be non-local."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Mostly this is pretty intuitive, but as with many things that behave intuitively, the details are more subtle than one might naïvely imagine."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"When x = <value> occurs in a local scope, Julia applies the following rules to decide what the expression means based on where the assignment expression occurs and what x already refers to at that location:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Existing local: If x is already a local variable, then the existing local x is assigned;\nHard scope: If x is not already a local variable and assignment occurs inside of any hard scope construct (i.e. within a let block, function, struct or macro body, comprehension, or generator), a new local named x is created in the scope of the assignment;\nSoft scope: If x is not already a local variable and all of the scope constructs containing the assignment are soft scopes (loops, try/catch blocks), the behavior depends on whether the global variable x is defined:\nif global x is undefined, a new local named x is created in the scope of the assignment;\nif global x is defined, the assignment is considered ambiguous:\nin non-interactive contexts (files, eval), an ambiguity warning is printed and a new local is created;\nin interactive contexts (REPL, notebooks), the global variable x is assigned."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"You may note that in non-interactive contexts the hard and soft scope behaviors are identical except that a warning is printed when an implicitly local variable (i.e. not declared with local x) shadows a global. In interactive contexts, the rules follow a more complex heuristic for the sake of convenience. This is covered in depth in examples that follow."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Now that you know the rules, let's look at some examples. Each example is assumed to be evaluated in a fresh REPL session so that the only globals in each snippet are the ones that are assigned in that block of code."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"We'll begin with a nice and clear-cut situation—assignment inside of a hard scope, in this case a function body, when no local variable by that name already exists:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> function greet()\n           x = \"hello\" # new local\n           println(x)\n       end\ngreet (generic function with 1 method)\n\njulia> greet()\nhello\n\njulia> x # global\nERROR: UndefVarError: `x` not defined in `Main`"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Inside of the greet function, the assignment x = \"hello\" causes x to be a new local variable in the function's scope. There are two relevant facts: the assignment occurs in local scope and there is no existing local x variable. Since x is local, it doesn't matter if there is a global named x or not. Here for example we define x = 123 before defining and calling greet:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> x = 123 # global\n123\n\njulia> function greet()\n           x = \"hello\" # new local\n           println(x)\n       end\ngreet (generic function with 1 method)\n\njulia> greet()\nhello\n\njulia> x # global\n123"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Since the x in greet is local, the value (or lack thereof) of the global x is unaffected by calling greet. The hard scope rule doesn't care whether a global named x exists or not: assignment to x in a hard scope is local (unless x is declared global)."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"The next clear cut situation we'll consider is when there is already a local variable named x, in which case x = <value> always assigns to this existing local x. This is true whether the assignment occurs in the same local scope, an inner local scope in the same function body, or in the body of a function nested inside of another function, also known as a closure."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"We'll use the sum_to function, which computes the sum of integers from one up to n, as an example:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"function sum_to(n)\n    s = 0 # new local\n    for i = 1:n\n        s = s + i # assign existing local\n    end\n    return s # same local\nend"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"As in the previous example, the first assignment to s at the top of sum_to causes s to be a new local variable in the body of the function. The for loop has its own inner local scope within the function scope. At the point where s = s + i occurs, s is already a local variable, so the assignment updates the existing s instead of creating a new local. We can test this out by calling sum_to in the REPL:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> function sum_to(n)\n           s = 0 # new local\n           for i = 1:n\n               s = s + i # assign existing local\n           end\n           return s # same local\n       end\nsum_to (generic function with 1 method)\n\njulia> sum_to(10)\n55\n\njulia> s # global\nERROR: UndefVarError: `s` not defined in `Main`"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Since s is local to the function sum_to, calling the function has no effect on the global variable s. We can also see that the update s = s + i in the for loop must have updated the same s created by the initialization s = 0 since we get the correct sum of 55 for the integers 1 through 10."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Let's dig into the fact that the for loop body has its own scope for a second by writing a slightly more verbose variation which we'll call sum_to_def, in which we save the sum s + i in a variable t before updating s:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> function sum_to_def(n)\n           s = 0 # new local\n           for i = 1:n\n               t = s + i # new local `t`\n               s = t # assign existing local `s`\n           end\n           return s, @isdefined(t)\n       end\nsum_to_def (generic function with 1 method)\n\njulia> sum_to_def(10)\n(55, false)"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"This version returns s as before but it also uses the @isdefined macro to return a boolean indicating whether there is a local variable named t defined in the function's outermost local scope. As you can see, there is no t defined outside of the for loop body. This is because of the hard scope rule again: since the assignment to t occurs inside of a function, which introduces a hard scope, the assignment causes t to become a new local variable in the local scope where it appears, i.e. inside of the loop body. Even if there were a global named t, it would make no difference—the hard scope rule isn't affected by anything in global scope."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Note that the local scope of a for loop body is no different from the local scope of an inner function. This means that we could rewrite this example so that the loop body is implemented as a call to an inner helper function and it behaves the same way:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> function sum_to_def_closure(n)\n           function loop_body(i)\n               t = s + i # new local `t`\n               s = t # assign same local `s` as below\n           end\n           s = 0 # new local\n           for i = 1:n\n               loop_body(i)\n           end\n           return s, @isdefined(t)\n       end\nsum_to_def_closure (generic function with 1 method)\n\njulia> sum_to_def_closure(10)\n(55, false)"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"This example illustrates a couple of key points:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Inner function scopes are just like any other nested local scope. In particular, if a variable is already a local outside of an inner function and you assign to it in the inner function, the outer local variable is updated.\nIt doesn't matter if the definition of an outer local happens below where it is updated, the rule remains the same. The entire enclosing local scope is parsed and its locals determined before inner local meanings are resolved."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"This design means that you can generally move code in or out of an inner function without changing its meaning, which facilitates a number of common idioms in the language using closures (see do blocks)."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Let's move onto some more ambiguous cases covered by the soft scope rule. We'll explore this by extracting the bodies of the greet and sum_to_def functions into soft scope contexts. First, let's put the body of greet in a for loop—which is soft, rather than hard—and evaluate it in the REPL:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> for i = 1:3\n           x = \"hello\" # new local\n           println(x)\n       end\nhello\nhello\nhello\n\njulia> x\nERROR: UndefVarError: `x` not defined in `Main`"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Since the global x is not defined when the for loop is evaluated, the first clause of the soft scope rule applies and x is created as local to the for loop and therefore global x remains undefined after the loop executes. Next, let's consider the body of sum_to_def extracted into global scope, fixing its argument to n = 10"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"s = 0\nfor i = 1:10\n    t = s + i\n    s = t\nend\ns\n@isdefined(t)"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"What does this code do? Hint: it's a trick question. The answer is \"it depends.\" If this code is entered interactively, it behaves the same way it does in a function body. But if the code appears in a file, it  prints an ambiguity warning and throws an undefined variable error. Let's see it working in the REPL first:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> s = 0 # global\n0\n\njulia> for i = 1:10\n           t = s + i # new local `t`\n           s = t # assign global `s`\n       end\n\njulia> s # global\n55\n\njulia> @isdefined(t) # global\nfalse"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"The REPL approximates being in the body of a function by deciding whether assignment inside the loop assigns to a global or creates new local based on whether a global variable by that name is defined or not. If a global by the name exists, then the assignment updates it. If no global exists, then the assignment creates a new local variable. In this example we see both cases in action:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"There is no global named t, so t = s + i creates a new t that is local to the for loop;\nThere is a global named s, so s = t assigns to it."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"The second fact is why execution of the loop changes the global value of s and the first fact is why t is still undefined after the loop executes. Now, let's try evaluating this same code as though it were in a file instead:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> code = \"\"\"\n       s = 0 # global\n       for i = 1:10\n           t = s + i # new local `t`\n           s = t # new local `s` with warning\n       end\n       s, # global\n       @isdefined(t) # global\n       \"\"\";\n\njulia> include_string(Main, code)\n┌ Warning: Assignment to `s` in soft scope is ambiguous because a global variable by the same name exists: `s` will be treated as a new local. Disambiguate by using `local s` to suppress this warning or `global s` to assign to the existing global variable.\n└ @ string:4\nERROR: LoadError: UndefVarError: `s` not defined in local scope"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Here we use include_string, to evaluate code as though it were the contents of a file. We could also save code to a file and then call include on that file—the result would be the same. As you can see, this behaves quite different from evaluating the same code in the REPL. Let's break down what's happening here:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"global s is defined with the value 0 before the loop is evaluated\nthe assignment s = t occurs in a soft scope—a for loop outside of any function body or other hard scope construct\ntherefore the second clause of the soft scope rule applies, and the assignment is ambiguous so a warning is emitted\nexecution continues, making s local to the for loop body\nsince s is local to the for loop, it is undefined when t = s + i is evaluated, causing an error\nevaluation stops there, but if it got to s and @isdefined(t), it would return 0 and false."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"This demonstrates some important aspects of scope: in a scope, each variable can only have one meaning, and that meaning is determined regardless of the order of expressions. The presence of the expression s = t in the loop causes s to be local to the loop, which means that it is also local when it appears on the right hand side of t = s + i, even though that expression appears first and is evaluated first. One might imagine that the s on the first line of the loop could be global while the s on the second line of the loop is local, but that's not possible since the two lines are in the same scope block and each variable can only mean one thing in a given scope."},{"title":"On Soft Scope","page":"Scope of Variables","location":"manual/variables-and-scoping.html#on-soft-scope","category":"section","text":""},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"We have now covered all the local scope rules, but before wrapping up this section, perhaps a few words should be said about why the ambiguous soft scope case is handled differently in interactive and non-interactive contexts. There are two obvious questions one could ask:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Why doesn't it just work like the REPL everywhere?\nWhy doesn't it just work like in files everywhere? And maybe skip the warning?"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"In Julia ≤ 0.6, all global scopes did work like the current REPL: when x = <value> occurred in a loop (or try/catch, or struct body) but outside of a function body (or let block or comprehension), it was decided based on whether a global named x was defined or not whether x should be local to the loop. This behavior has the advantage of being intuitive and convenient since it approximates the behavior inside of a function body as closely as possible. In particular, it makes it easy to move code back and forth between a function body and the REPL when trying to debug the behavior of a function. However, it has some downsides. First, it's quite a complex behavior: many people over the years were confused about this behavior and complained that it was complicated and hard both to explain and understand. Fair point. Second, and arguably worse, is that it's bad for programming \"at scale.\" When you see a small piece of code in one place like this, it's quite clear what's going on:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"s = 0\nfor i = 1:10\n    s += i\nend"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Obviously the intention is to modify the existing global variable s. What else could it mean? However, not all real world code is so short or so clear. We found that code like the following often occurs in the wild:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"x = 123\n\n# much later\n# maybe in a different file\n\nfor i = 1:10\n    x = \"hello\"\n    println(x)\nend\n\n# much later\n# maybe in yet another file\n# or maybe back in the first one where `x = 123`\n\ny = x + 234"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"It's far less clear what should happen here. Since x + \"hello\" is a method error, it seems probable that the intention is for x to be local to the for loop. But runtime values and what methods happen to exist cannot be used to determine the scopes of variables. With the Julia ≤ 0.6 behavior, it's especially concerning that someone might have written the for loop first, had it working just fine, but later when someone else adds a new global far away—possibly in a different file—the code suddenly changes meaning and either breaks noisily or, worse still, silently does the wrong thing. This kind of \"spooky action at a distance\" is something that good programming language designs should prevent."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"So in Julia 1.0, we simplified the rules for scope: in any local scope, assignment to a name that wasn't already a local variable created a new local variable. This eliminated the notion of soft scope entirely as well as removing the potential for spooky action. We uncovered and fixed a significant number of bugs due to the removal of soft scope, vindicating the choice to get rid of it. And there was much rejoicing! Well, no, not really. Because some people were angry that they now had to write:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"s = 0\nfor i = 1:10\n    global s += i\nend"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Do you see that global annotation in there? Hideous. Obviously this situation could not be tolerated. But seriously, there are two main issues with requiring global for this kind of top-level code:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"It's no longer convenient to copy and paste the code from inside a function body into the REPL to debug it—you have to add global annotations and then remove them again to go back;\nBeginners will write this kind of code without the global and have no idea why their code doesn't work—the error that they get is that s is undefined, which does not seem to enlighten anyone who happens to make this mistake."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"As of Julia 1.5, this code works without the global annotation in interactive contexts like the REPL or Jupyter notebooks (just like Julia 0.6) and in files and other non-interactive contexts, it prints this very direct warning:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Assignment to s in soft scope is ambiguous because a global variable by the same name exists: s will be treated as a new local. Disambiguate by using local s to suppress this warning or global s to assign to the existing global variable."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"This addresses both issues while preserving the \"programming at scale\" benefits of the 1.0 behavior: global variables have no spooky effect on the meaning of code that may be far away; in the REPL copy-and-paste debugging works and beginners don't have any issues; any time someone either forgets a global annotation or accidentally shadows an existing global with a local in a soft scope, which would be confusing anyway, they get a nice clear warning."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"An important property of this design is that any code that executes in a file without a warning will behave the same way in a fresh REPL. And on the flip side, if you take a REPL session and save it to file, if it behaves differently than it did in the REPL, then you will get a warning."},{"title":"Let Blocks","page":"Scope of Variables","location":"manual/variables-and-scoping.html#Let-Blocks","category":"section","text":""},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"let statements create a new hard scope block (see above) and introduce new variable bindings each time they run. The variable need not be immediately assigned:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> var1 = let x\n           for i in 1:5\n               (i == 4) && (x = i; break)\n           end\n           x\n       end\n4"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Whereas assignments might reassign a new value to an existing value location, let always creates a new location. This difference is usually not important, and is only detectable in the case of variables that outlive their scope via closures. The let syntax accepts a comma-separated series of assignments and variable names:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> x, y, z = -1, -1, -1;\n\njulia> let x = 1, z\n           println(\"x: $x, y: $y\") # x is local variable, y the global\n           println(\"z: $z\") # errors as z has not been assigned yet but is local\n       end\nx: 1, y: -1\nERROR: UndefVarError: `z` not defined in local scope"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"The assignments are evaluated in order, with each right-hand side evaluated in the scope before the new variable on the left-hand side has been introduced. Therefore it makes sense to write something like let x = x since the two x variables are distinct and have separate storage. Here is an example where the behavior of let is needed:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> Fs = Vector{Any}(undef, 2); i = 1;\n\njulia> while i <= 2\n           Fs[i] = ()->i\n           global i += 1\n       end\n\njulia> Fs[1]()\n3\n\njulia> Fs[2]()\n3"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Here we create and store two closures that return variable i. However, it is always the same variable i, so the two closures behave identically. We can use let to create a new binding for i:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> Fs = Vector{Any}(undef, 2); i = 1;\n\njulia> while i <= 2\n           let i = i\n               Fs[i] = ()->i\n           end\n           global i += 1\n       end\n\njulia> Fs[1]()\n1\n\njulia> Fs[2]()\n2"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Since the begin construct does not introduce a new scope, it can be useful to use a zero-argument let to just introduce a new scope block without creating any new bindings immediately:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> let\n           local x = 1\n           let\n               local x = 2\n           end\n           x\n       end\n1"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Since let introduces a new scope block, the inner local x is a different variable than the outer local x. This particular example is equivalent to:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> let x = 1\n           let x = 2\n           end\n           x\n       end\n1"},{"title":"Loops and Comprehensions","page":"Scope of Variables","location":"manual/variables-and-scoping.html#Loops-and-Comprehensions","category":"section","text":""},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"In loops and comprehensions, new variables introduced in their body scopes are freshly allocated for each loop iteration, as if the loop body were surrounded by a let block, as demonstrated by this example:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> Fs = Vector{Any}(undef, 2);\n\njulia> for j = 1:2\n           Fs[j] = ()->j\n       end\n\njulia> Fs[1]()\n1\n\njulia> Fs[2]()\n2"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"A for loop or comprehension iteration variable is always a new variable:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> function f()\n           i = 0\n           for i = 1:3\n               # empty\n           end\n           return i\n       end;\n\njulia> f()\n0"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"However, it is occasionally useful to reuse an existing local variable as the iteration variable. This can be done conveniently by adding the keyword outer:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> function f()\n           i = 0\n           for outer i = 1:3\n               # empty\n           end\n           return i\n       end;\n\njulia> f()\n3"},{"title":"Constants","page":"Scope of Variables","location":"manual/variables-and-scoping.html#Constants","category":"section","text":""},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"A common use of variables is giving names to specific, unchanging values. Such variables are only assigned once. This intent can be conveyed to the compiler using the const keyword:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> const e  = 2.71828182845904523536;\n\njulia> const pi = 3.14159265358979323846;"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Multiple variables can be declared in a single const statement:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> const a, b = 1, 2\n(1, 2)"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"The const declaration should only be used in global scope on globals. It is difficult for the compiler to optimize code involving global variables, since their values (or even their types) might change at almost any time. If a global variable will not change, adding a const declaration solves this performance problem."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Local constants are quite different. The compiler is able to determine automatically when a local variable is constant, so local constant declarations are not necessary, and in fact are currently not supported."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Special top-level assignments, such as those performed by the function and struct keywords, are constant by default."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Note that const only affects the variable binding; the variable may be bound to a mutable object (such as an array), and that object may still be modified. Additionally when one tries to assign a value to a variable that is declared constant the following scenarios are possible:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Attempting to replace a constant without the const keyword is disallowed:\njulia> const x = 1.0\n1.0\n\njulia> x = 1\nERROR: invalid assignment to constant x. This redefinition may be permitted using the `const` keyword.\nAll other definitions of constants are permitted, but may cause significant re-compilation:\njulia> const y = 1.0\n1.0\n\njulia> const y = 2.0\n2.0"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"compat: Julia 1.12\nPrior to julia 1.12, redefinition of constants was poorly supported. It was restricted to redefinition of constants of the same type and could lead to observably incorrect behavior or crashes. Constant redefinition is highly discouraged in versions of julia prior to 1.12. See the manual for prior julia versions for further information."},{"title":"Typed Globals","page":"Scope of Variables","location":"manual/variables-and-scoping.html#man-typed-globals","category":"section","text":""},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"compat: Julia 1.8\nSupport for typed globals was added in Julia 1.8"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Similar to being declared as constants, global bindings can also be declared to always be of a constant type. This can either be done without assigning an actual value using the syntax global x::T or upon assignment as x::T = 123."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> x::Float64 = 2.718\n2.718\n\njulia> f() = x\nf (generic function with 1 method)\n\njulia> Base.return_types(f)\n1-element Vector{Any}:\n Float64"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"For any assignment to a global, Julia will first try to convert it to the appropriate type using convert:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> global y::Int\n\njulia> y = 1.0\n1.0\n\njulia> y\n1\n\njulia> y = 3.14\nERROR: InexactError: Int64(3.14)\nStacktrace:\n[...]"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"The type does not need to be concrete, but annotations with abstract types typically have little performance benefit."},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"Once a global has either been assigned to or its type has been set, the binding type is not allowed to change:"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html","category":"page","text":"julia> x = 1\n1\n\njulia> global x::Int\nERROR: cannot set type for global x. It already has a value or is already set to a different type.\nStacktrace:\n[...]"},{"title":"Downloads","page":"Downloads","location":"stdlib/Downloads.html#Downloads","category":"section","text":""},{"title":"Downloads.download","page":"Downloads","location":"stdlib/Downloads.html#Downloads.download","category":"function","text":"download(url, [ output = tempname() ];\n    [ method = \"GET\", ]\n    [ headers = <none>, ]\n    [ timeout = <none>, ]\n    [ progress = <none>, ]\n    [ verbose = false, ]\n    [ debug = <none>, ]\n    [ downloader = <default>, ]\n) -> output\n\n    url        :: AbstractString\n    output     :: Union{AbstractString, AbstractCmd, IO}\n    method     :: AbstractString\n    headers    :: Union{AbstractVector, AbstractDict}\n    timeout    :: Real\n    progress   :: (total::Integer, now::Integer) --> Any\n    verbose    :: Bool\n    debug      :: (type, message) --> Any\n    downloader :: Downloader\n\nDownload a file from the given url, saving it to output or if not specified, a temporary path. The output can also be an IO handle, in which case the body of the response is streamed to that handle and the handle is returned. If output is a command, the command is run and output is sent to it on stdin.\n\nIf the downloader keyword argument is provided, it must be a Downloader object. Resources and connections will be shared between downloads performed by the same Downloader and cleaned up automatically when the object is garbage collected or there have been no downloads performed with it for a grace period. See Downloader for more info about configuration and usage.\n\nIf the headers keyword argument is provided, it must be a vector or dictionary whose elements are all pairs of strings. These pairs are passed as headers when downloading URLs with protocols that supports them, such as HTTP/S.\n\nThe timeout keyword argument specifies a timeout for the download to complete in seconds, with a resolution of milliseconds. By default no timeout is set, but this can also be explicitly requested by passing a timeout value of Inf. Separately, if 20 seconds elapse without receiving any data, the download will timeout. See extended help for how to disable this timeout.\n\nIf the progress keyword argument is provided, it must be a callback function which will be called whenever there are updates about the size and status of the ongoing download. The callback must take two integer arguments: total and now which are the total size of the download in bytes, and the number of bytes which have been downloaded so far. Note that total starts out as zero and remains zero until the server gives an indication of the total size of the download (e.g. with a Content-Length header), which may never happen. So a well-behaved progress callback should handle a total size of zero gracefully.\n\nIf the verbose option is set to true, libcurl, which is used to implement the download functionality will print debugging information to stderr. If the debug option is set to a function accepting two String arguments, then the verbose option is ignored and instead the data that would have been printed to stderr is passed to the debug callback with type and message arguments. The type argument indicates what kind of event has occurred, and is one of: TEXT, HEADER IN, HEADER OUT, DATA IN, DATA OUT, SSL DATA IN or SSL DATA OUT. The message argument is the description of the debug event.\n\nExtended Help\n\nFor further customization, use a Downloader and easy_hooks. For example, to disable the 20 second timeout when no data is received, you may use the following:\n\ndownloader = Downloads.Downloader()\ndownloader.easy_hook = (easy, info) -> Downloads.Curl.setopt(easy, Downloads.Curl.CURLOPT_LOW_SPEED_TIME, 0)\n\nDownloads.download(\"https://httpbingo.julialang.org/delay/30\"; downloader)\n\n\n\n\n\n"},{"title":"Downloads.request","page":"Downloads","location":"stdlib/Downloads.html#Downloads.request","category":"function","text":"request(url;\n    [ input = <none>, ]\n    [ output = <none>, ]\n    [ method = input ? \"PUT\" : output ? \"GET\" : \"HEAD\", ]\n    [ headers = <none>, ]\n    [ timeout = <none>, ]\n    [ progress = <none>, ]\n    [ verbose = false, ]\n    [ debug = <none>, ]\n    [ throw = true, ]\n    [ downloader = <default>, ]\n    [ interrupt = <none>, ]\n) -> Union{Response, RequestError}\n\n    url        :: AbstractString\n    input      :: Union{AbstractString, AbstractCmd, IO}\n    output     :: Union{AbstractString, AbstractCmd, IO}\n    method     :: AbstractString\n    headers    :: Union{AbstractVector, AbstractDict}\n    timeout    :: Real\n    progress   :: (dl_total, dl_now, ul_total, ul_now) --> Any\n    verbose    :: Bool\n    debug      :: (type, message) --> Any\n    throw      :: Bool\n    downloader :: Downloader\n    interrupt  :: Base.Event\n\nMake a request to the given url, returning a Response object capturing the status, headers and other information about the response. The body of the response is written to output if specified and discarded otherwise. For HTTP/S requests, if an input stream is given, a PUT request is made; otherwise if an output stream is given, a GET request is made; if neither is given a HEAD request is made. For other protocols, appropriate default methods are used based on what combination of input and output are requested. The following options differ from the download function:\n\ninput allows providing a request body; if provided default to PUT request\nprogress is a callback taking four integers for upload and download progress\nthrow controls whether to throw or return a RequestError on request error\n\nNote that unlike download which throws an error if the requested URL could not be downloaded (indicated by non-2xx status code), request returns a Response object no matter what the status code of the response is. If there is an error with getting a response at all, then a RequestError is thrown or returned.\n\nIf the interrupt keyword argument is provided, it must be a Base.Event object. If the event is triggered while the request is in progress, the request will be cancelled and an error will be thrown. This can be used to interrupt a long running request, for example if the user wants to cancel a download.\n\n\n\n\n\n"},{"title":"Downloads.Response","page":"Downloads","location":"stdlib/Downloads.html#Downloads.Response","category":"type","text":"struct Response\n    proto   :: String\n    url     :: String\n    status  :: Int\n    message :: String\n    headers :: Vector{Pair{String,String}}\nend\n\nResponse is a type capturing the properties of a successful response to a request as an object. It has the following fields:\n\nproto: the protocol that was used to get the response\nurl: the URL that was ultimately requested after following redirects\nstatus: the status code of the response, indicating success, failure, etc.\nmessage: a textual message describing the nature of the response\nheaders: any headers that were returned with the response\n\nThe meaning and availability of some of these responses depends on the protocol used for the request. For many protocols, including HTTP/S and S/FTP, a 2xx status code indicates a successful response. For responses in protocols that do not support headers, the headers vector will be empty. HTTP/2 does not include a status message, only a status code, so the message will be empty.\n\n\n\n\n\n"},{"title":"Downloads.RequestError","page":"Downloads","location":"stdlib/Downloads.html#Downloads.RequestError","category":"type","text":"struct RequestError <: ErrorException\n    url      :: String\n    code     :: Int\n    message  :: String\n    response :: Response\nend\n\nRequestError is a type capturing the properties of a failed response to a request as an exception object:\n\nurl: the original URL that was requested without any redirects\ncode: the libcurl error code; 0 if a protocol-only error occurred\nmessage: the libcurl error message indicating what went wrong\nresponse: response object capturing what response info is available\n\nThe same RequestError type is thrown by download if the request was successful but there was a protocol-level error indicated by a status code that is not in the 2xx range, in which case code will be zero and the message field will be the empty string. The request API only throws a RequestError if the libcurl error code is non-zero, in which case the included response object is likely to have a status of zero and an empty message. There are, however, situations where a curl-level error is thrown due to a protocol error, in which case both the inner and outer code and message may be of interest.\n\n\n\n\n\n"},{"title":"Downloads.Downloader","page":"Downloads","location":"stdlib/Downloads.html#Downloads.Downloader","category":"type","text":"Downloader(; [ grace::Real = 30 ])\n\nDownloader objects are used to perform individual download operations. Connections, name lookups and other resources are shared within a Downloader. These connections and resources are cleaned up after a configurable grace period (default: 30 seconds) since anything was downloaded with it, or when it is garbage collected, whichever comes first. If the grace period is set to zero, all resources will be cleaned up immediately as soon as there are no more ongoing downloads in progress. If the grace period is set to Inf then resources are not cleaned up until Downloader is garbage collected.\n\n\n\n\n\n"},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html#pkgimages","category":"section","text":""},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html","category":"page","text":"Julia package images provide object (native code) caches for Julia packages. They are similar to Julia's system image and support many of the same features. In fact the underlying serialization format is the same, and the system image is the base image that the package images are build against."},{"title":"High-level overview","page":"Package Images","location":"devdocs/pkgimg.html#High-level-overview","category":"section","text":""},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html","category":"page","text":"Package images are shared libraries that contain both code and data. Like .ji cache files, they are generated per package. The data section contains both global data (global variables in the package) as well as the necessary metadata about what methods and types are defined by the package. The code section contains native objects that cache the final output of Julia's LLVM-based compiler."},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html","category":"page","text":"The command line option --pkgimages=no can be used to turn off object caching for this session. Note that this means that cache files have to likely be regenerated. See JULIA_MAX_NUM_PRECOMPILE_FILES for the upper limit of variants Julia caches per default."},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html","category":"page","text":"note: Note\nWhile the package images present themselves as native shared libraries, they are only an approximation thereof. You will not be able to link against them from a native program and they must be loaded from Julia."},{"title":"Linking","page":"Package Images","location":"devdocs/pkgimg.html#Linking","category":"section","text":""},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html","category":"page","text":"Since the package images contain native code, we must run a linker over them before we can use them. You can set the environment variable JULIA_VERBOSE_LINKING to true to make the package image linking process verbose."},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html","category":"page","text":"Furthermore, we cannot assume that the user has a working system linker installed. Therefore, Julia ships with LLD, the LLVM linker, to provide a working out of the box experience. In base/linking.jl, we implement a limited interface to be able to link package images on all supported platforms."},{"title":"Quirks","page":"Package Images","location":"devdocs/pkgimg.html#Quirks","category":"section","text":""},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html","category":"page","text":"Despite LLD being a multi-platform linker, it does not provide a consistent interface across platforms. Furthermore, it is meant to be used from clang or another compiler driver, we therefore reimplement some of the logic from llvm-project/clang/lib/Driver/ToolChains. Thankfully one can use lld -flavor to set lld to the right platform"},{"title":"Windows","page":"Package Images","location":"devdocs/pkgimg.html#Windows","category":"section","text":""},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html","category":"page","text":"To avoid having to deal with link.exe we use -flavor gnu, effectively turning lld into a cross-linker from a mingw32 environment. Windows DLLs are required to contain a _DllMainCRTStartup function and to minimize our dependence on mingw32 libraries, we inject a stub definition ourselves."},{"title":"MacOS","page":"Package Images","location":"devdocs/pkgimg.html#MacOS","category":"section","text":""},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html","category":"page","text":"Dynamic libraries on macOS need to link against -lSystem. On recent macOS versions, -lSystem is only available for linking when Xcode is available. To that effect we link with -undefined dynamic_lookup."},{"title":"Package images optimized for multiple microarchitectures","page":"Package Images","location":"devdocs/pkgimg.html#pkgimgs-multi-versioning","category":"section","text":""},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html","category":"page","text":"Similar to multi-versioning for system images, package images support multi-versioning. This allows creating package caches that can run efficiently on different CPU architectures within the same environment."},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html","category":"page","text":"See the JULIA_CPU_TARGET environment variable for more information on how to set the CPU target for package images."},{"title":"Flags that impact package image creation and selection","page":"Package Images","location":"devdocs/pkgimg.html#Flags-that-impact-package-image-creation-and-selection","category":"section","text":""},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html","category":"page","text":"These are the Julia command line flags that impact cache selection. Package images that were created with different flags will be rejected."},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html","category":"page","text":"-g, --debug-info: Exact match required since it changes code generation.\n--check-bounds: Exact match required since it changes code generation.\n--inline: Exact match required since it changes code generation.\n--pkgimages: To allow running without object caching enabled.\n-O, --optimize: Reject package images generated for a lower optimization level, but allow for higher optimization levels to be loaded."},{"title":"Libdl","page":"Dynamic Linker","location":"stdlib/Libdl.html#Libdl","category":"module","text":"The Libdl module in Julia provides specialized and lower-level facilities for dynamic linking with shared libraries. While Julia inherently supports linking to runtime shared libraries through the ccall intrinsic, Libdl extends this capability by offering additional, more granular control. It enables users to search for shared libraries both in memory and the filesystem, manually load them with specific runtime linker options, and look up library symbols as low-level pointers.\n\n\n\n\n\n"},{"title":"Dynamic Linker","page":"Dynamic Linker","location":"stdlib/Libdl.html#Dynamic-Linker","category":"section","text":""},{"title":"Base.Libc.Libdl.dlopen","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dlopen","category":"function","text":"dlopen(libfile::AbstractString [, flags::Integer]; throw_error:Bool = true)\n\nLoad a shared library, returning an opaque handle.\n\nThe extension given by the constant dlext (.so, .dll, or .dylib) can be omitted from the libfile string, as it is automatically appended if needed.   If libfile is not an absolute path name, then the paths in the array DL_LOAD_PATH are searched for libfile, followed by the system load path.\n\nThe optional flags argument is a bitwise-or of zero or more of RTLD_LOCAL, RTLD_GLOBAL, RTLD_LAZY, RTLD_NOW, RTLD_NODELETE, RTLD_NOLOAD, RTLD_DEEPBIND, and RTLD_FIRST. These are converted to the corresponding flags of the POSIX (and/or GNU libc and/or MacOS) dlopen command, if possible, or are ignored if the specified functionality is not available on the current platform. The default flags are platform specific. On MacOS the default dlopen flags are RTLD_LAZY|RTLD_DEEPBIND|RTLD_GLOBAL while on other platforms the defaults are RTLD_LAZY|RTLD_DEEPBIND|RTLD_LOCAL. An important usage of these flags is to specify non default behavior for when the dynamic library loader binds library references to exported symbols and if the bound references are put into process local or global scope. For instance RTLD_LAZY|RTLD_DEEPBIND|RTLD_GLOBAL allows the library's symbols to be available for usage in other shared libraries, addressing situations where there are dependencies between shared libraries.\n\nIf the library cannot be found, this method throws an error, unless the keyword argument throw_error is set to false, in which case this method returns nothing.\n\nnote: Note\nFrom Julia 1.6 on, this method replaces paths starting with @executable_path/ with  the path to the Julia executable, allowing for relocatable relative-path loads. In  Julia 1.5 and earlier, this only worked on macOS.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dlopen_e","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dlopen_e","category":"function","text":"dlopen_e(libfile::AbstractString [, flags::Integer])\n\nSimilar to dlopen, except returns C_NULL instead of raising errors. This method is now deprecated in favor of dlopen(libfile::AbstractString [, flags::Integer]; throw_error=false).\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.RTLD_NOW","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.RTLD_NOW","category":"constant","text":"RTLD_DEEPBIND\nRTLD_FIRST\nRTLD_GLOBAL\nRTLD_LAZY\nRTLD_LOCAL\nRTLD_NODELETE\nRTLD_NOLOAD\nRTLD_NOW\n\nEnum constant for dlopen. See your platform man page for details, if applicable.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dlsym","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dlsym","category":"function","text":"dlsym(handle, sym; throw_error::Bool = true)\n\nLook up a symbol from a shared library handle, return callable function pointer on success.\n\nIf the symbol cannot be found, this method throws an error, unless the keyword argument throw_error is set to false, in which case this method returns nothing.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dlsym_e","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dlsym_e","category":"function","text":"dlsym_e(handle, sym)\n\nLook up a symbol from a shared library handle, silently return C_NULL on lookup failure. This method is now deprecated in favor of dlsym(handle, sym; throw_error=false).\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dlclose","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dlclose","category":"function","text":"dlclose(::Nothing)\n\nFor the very common pattern usage pattern of\n\ntry\n    hdl = dlopen(library_name)\n    ... do something\nfinally\n    dlclose(hdl)\nend\n\nWe define a dlclose() method that accepts a parameter of type Nothing, so that user code does not have to change its behavior for the case that library_name was not found.\n\n\n\n\n\ndlclose(handle)\n\nClose shared library referenced by handle.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dlext","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dlext","category":"constant","text":"dlext\n\nFile extension for dynamic libraries (e.g. dll, dylib, so) on the current platform.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dllist","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dllist","category":"function","text":"dllist()\n\nReturn the paths of dynamic libraries currently loaded in a Vector{String}.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dlpath","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dlpath","category":"function","text":"dlpath(libname::Union{AbstractString, Symbol})\n\nGet the full path of the library libname.\n\nExamples\n\njulia> dlpath(\"libjulia\")\n\n\n\n\n\ndlpath(handle::Ptr{Cvoid})\n\nGiven a library handle from dlopen, return the full path.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.find_library","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.find_library","category":"function","text":"find_library(names [, locations])\n\nSearches for the first library in names in the paths in the locations list, DL_LOAD_PATH, or system library paths (in that order) which can successfully be dlopen'd. On success, the return value will be one of the names (potentially prefixed by one of the paths in locations). This string can be assigned to a global const and used as the library name in future ccall's. On failure, it returns the empty string.\n\n\n\n\n\n"},{"title":"Base.DL_LOAD_PATH","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.DL_LOAD_PATH","category":"constant","text":"DL_LOAD_PATH\n\nWhen calling dlopen, the paths in this list will be searched first, in order, before searching the system locations for a valid library handle.\n\n\n\n\n\n"},{"title":"Lazy Library Loading","page":"Dynamic Linker","location":"stdlib/Libdl.html#Lazy-Library-Loading","category":"section","text":""},{"title":"Base.Libc.Libdl.LazyLibrary","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.LazyLibrary","category":"type","text":"LazyLibrary(name; flags = <default dlopen flags>,\n            dependencies = LazyLibrary[], on_load_callback = nothing)\n\nRepresents a lazily-loaded shared library that delays loading itself and its dependencies until first use in a ccall(), @ccall, dlopen(), dlsym(), dlpath(), or cglobal(). This is a thread-safe mechanism for on-demand library initialization.\n\nArguments\n\nname: Library name (or lazy path computation) as a String, LazyLibraryPath, or BundledLazyLibraryPath.\nflags: Optional dlopen flags (default: RTLD_LAZY | RTLD_DEEPBIND). See dlopen.\ndependencies: Vector of LazyLibrary object references to load before this one.\non_load_callback: Optional function to run arbitrary code on first load (use sparingly, as it is not expected that ccall() should result in large amounts of Julia code being run. You may call ccall() from within the on_load_callback but only for the current library and its dependencies, and user should not call wait() on any tasks within the on load callback as they may deadlock).\n\nThe dlopen operation is thread-safe: only one thread loads the library, acquired after the release store of the reference to each dependency from loading of each dependency. Other tasks block until loading completes. The handle is then cached and reused for all subsequent calls (there is no dlclose for lazy library and dlclose should not be called on the returned handled).\n\nExamples\n\n# Basic usage\nconst mylib = LazyLibrary(\"libmylib\")\n@ccall mylib.myfunc(42::Cint)::Cint\n\n# With dependencies\nconst libfoo = LazyLibrary(\"libfoo\")\nconst libbar = LazyLibrary(\"libbar\"; dependencies=[libfoo])\n\nFor more examples including platform-specific libraries, lazy path construction, and migration from __init__() patterns, see the manual section on Using LazyLibrary for Lazy Loading.\n\ncompat: Julia 1.11\nLazyLibrary was added in Julia 1.11.\n\nSee also LazyLibraryPath, BundledLazyLibraryPath, dlopen, dlsym, add_dependency!.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.LazyLibraryPath","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.LazyLibraryPath","category":"type","text":"LazyLibraryPath(path_pieces...)\n\nHelper type for lazily constructed library paths for use with LazyLibrary. Path pieces are stored unevaluated and joined with joinpath() when the library is first accessed. Arguments must be able to have string() called on them.\n\nExample\n\nconst mylib = LazyLibrary(LazyLibraryPath(artifact_dir, \"lib\", \"libmylib.so.1.2.3\"))\n\ncompat: Julia 1.11\nLazyLibraryPath was added in Julia 1.11.\n\nSee also LazyLibrary, BundledLazyLibraryPath.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.BundledLazyLibraryPath","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.BundledLazyLibraryPath","category":"function","text":"BundledLazyLibraryPath(subpath)\n\nHelper type for lazily constructed library paths within the Julia distribution. Constructs paths relative to Julia's private shared library directory.\n\nPrimarily used by Julia's standard library. For example:\n\nconst libgmp = LazyLibrary(BundledLazyLibraryPath(\"libgmp.so.10\"))\n\ncompat: Julia 1.11\nBundledLazyLibraryPath was added in Julia 1.11.\n\nSee also LazyLibrary, LazyLibraryPath.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.add_dependency!","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.add_dependency!","category":"function","text":"add_dependency!(library::LazyLibrary, dependency::LazyLibrary)\n\nDynamically add a dependency that must be loaded before library. Only needed when dependencies cannot be determined at construction time.\n\nwarning: Warning\nDependencies added with this function are ephemeral and only persist within the current process. They will not persist across precompilation boundaries.\n\nPrefer specifying dependencies in the LazyLibrary constructor when possible.\n\ncompat: Julia 1.11\nadd_dependency! was added in Julia 1.11.\n\nSee also LazyLibrary.\n\n\n\n\n\n"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html#man-logging","category":"section","text":""},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"The Logging module provides a way to record the history and progress of a computation as a log of events. Events are created by inserting a logging statement into the source code, for example:"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"@warn \"Abandon printf debugging, all ye who enter here!\"\n┌ Warning: Abandon printf debugging, all ye who enter here!\n└ @ Main REPL[1]:1"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"The system provides several advantages over peppering your source code with calls to println(). First, it allows you to control the visibility and presentation of messages without editing the source code. For example, in contrast to the @warn above"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"@debug \"The sum of some values $(sum(rand(100)))\""},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"will produce no output by default. Furthermore, it's very cheap to leave debug statements like this in the source code because the system avoids evaluating the message if it would later be ignored. In this case sum(rand(100)) and the associated string processing will never be executed unless debug logging is enabled."},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Second, the logging tools allow you to attach arbitrary data to each event as a set of key–value pairs. This allows you to capture local variables and other program state for later analysis. For example, to attach the local array variable A and the sum of a vector v as the key s you can use"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"A = ones(Int, 4, 4)\nv = ones(100)\n@info \"Some variables\"  A  s=sum(v)\n\n# output\n┌ Info: Some variables\n│   A =\n│    4×4 Matrix{Int64}:\n│     1  1  1  1\n│     1  1  1  1\n│     1  1  1  1\n│     1  1  1  1\n└   s = 100.0"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"All of the logging macros @debug, @info, @warn and @error share common features that are described in detail in the documentation for the more general macro @logmsg."},{"title":"Log event structure","page":"Logging","location":"stdlib/Logging.html#Log-event-structure","category":"section","text":""},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Each event generates several pieces of data, some provided by the user and some automatically extracted. Let's examine the user-defined data first:"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"The log level is a broad category for the message that is used for early filtering. There are several standard levels of type LogLevel; user-defined levels are also possible. Each is distinct in purpose:\nLogging.Debug (log level -1000) is information intended for the developer of the program. These events are disabled by default.\nLogging.Info (log level 0) is for general information to the user. Think of it as an alternative to using println directly.\nLogging.Warn (log level 1000) means something is wrong and action is likely required but that for now the program is still working.\nLogging.Error (log level 2000) means something is wrong and it is unlikely to be recovered, at least by this part of the code. Often this log-level is unneeded as throwing an exception can convey all the required information.\nThe message  is an object describing the event. By convention AbstractStrings passed as messages are assumed to be in markdown format. Other types will be displayed using print(io, obj) or string(obj) for text-based output and possibly show(io,mime,obj) for other multimedia displays used in the installed logger.\nOptional key–value pairs allow arbitrary data to be attached to each event. Some keys have conventional meaning that can affect the way an event is interpreted (see @logmsg)."},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"The system also generates some standard information for each event:"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"The module in which the logging macro was expanded.\nThe file and line where the logging macro occurs in the source code.\nA message id that is a unique, fixed identifier for the source code statement where the logging macro appears. This identifier is designed to be fairly stable even if the source code of the file changes, as long as the logging statement itself remains the same.\nA group for the event, which is set to the base name of the file by default, without extension. This can be used to group messages into categories more finely than the log level (for example, all deprecation warnings have group :depwarn), or into logical groupings across or within modules."},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Notice that some useful information such as the event time is not included by default. This is because such information can be expensive to extract and is also dynamically available to the current logger. It's simple to define a custom logger to augment event data with the time, backtrace, values of global variables and other useful information as required."},{"title":"Processing log events","page":"Logging","location":"stdlib/Logging.html#Processing-log-events","category":"section","text":""},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"As you can see in the examples, logging statements make no mention of where log events go or how they are processed. This is a key design feature that makes the system composable and natural for concurrent use. It does this by separating two different concerns:"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Creating log events is the concern of the module author who needs to decide where events are triggered and which information to include.\nProcessing of log events — that is, display, filtering, aggregation and recording — is the concern of the application author who needs to bring multiple modules together into a cooperating application."},{"title":"Loggers","page":"Logging","location":"stdlib/Logging.html#Loggers","category":"section","text":""},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Processing of events is performed by a logger, which is the first piece of user configurable code to see the event. All loggers must be subtypes of AbstractLogger."},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"When an event is triggered, the appropriate logger is found by looking for a task-local logger with the global logger as fallback. The idea here is that the application code knows how log events should be processed and exists somewhere at the top of the call stack. So we should look up through the call stack to discover the logger — that is, the logger should be dynamically scoped. (This is a point of contrast with logging frameworks where the logger is lexically scoped; provided explicitly by the module author or as a simple global variable. In such a system it's awkward to control logging while composing functionality from multiple modules.)"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"The global logger may be set with global_logger, and task-local loggers controlled using with_logger. Newly spawned tasks inherit the logger of the parent task."},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"There are three logger types provided by the library.  ConsoleLogger is the default logger you see when starting the REPL. It displays events in a readable text format and tries to give simple but user friendly control over formatting and filtering.  NullLogger is a convenient way to drop all messages where necessary; it is the logging equivalent of the devnull stream.  SimpleLogger is a very simplistic text formatting logger, mainly useful for debugging the logging system itself."},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Custom loggers should come with overloads for the functions described in the reference section."},{"title":"Early filtering and message handling","page":"Logging","location":"stdlib/Logging.html#Early-filtering-and-message-handling","category":"section","text":""},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"When an event occurs, a few steps of early filtering occur to avoid generating messages that will be discarded:"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"The message log level is checked against a global minimum level (set via disable_logging). This is a crude but extremely cheap global setting.\nThe current logger state is looked up and the message level checked against the logger's cached minimum level, as found by calling Logging.min_enabled_level. This behavior can be overridden via environment variables (more on this later).\nThe Logging.shouldlog function is called with the current logger, taking some minimal information (level, module, group, id) which can be computed statically. Most usefully, shouldlog is passed an event id which can be used to discard events early based on a cached predicate."},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"If all these checks pass, the message and key–value pairs are evaluated in full and passed to the current logger via the Logging.handle_message function. handle_message() may perform additional filtering as required and display the event to the screen, save it to a file, etc."},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Exceptions that occur while generating the log event are captured and logged by default. This prevents individual broken events from crashing the application, which is helpful when enabling little-used debug events in a production system. This behavior can be customized per logger type by extending Logging.catch_exceptions."},{"title":"Testing log events","page":"Logging","location":"stdlib/Logging.html#Testing-log-events","category":"section","text":""},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Log events are a side effect of running normal code, but you might find yourself wanting to test particular informational messages and warnings. The Test module provides a @test_logs macro that can be used to pattern match against the log event stream."},{"title":"Environment variables","page":"Logging","location":"stdlib/Logging.html#Environment-variables","category":"section","text":""},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Message filtering can be influenced through the JULIA_DEBUG environment variable, and serves as an easy way to enable debug logging for a file or module. Loading julia with JULIA_DEBUG=loading will activate @debug log messages in loading.jl. For example, in Linux shells:"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"$ JULIA_DEBUG=loading julia -e 'using OhMyREPL'\n┌ Debug: Rejecting cache file /home/user/.julia/compiled/v0.7/OhMyREPL.ji due to it containing an incompatible cache header\n└ @ Base loading.jl:1328\n[ Info: Recompiling stale cache file /home/user/.julia/compiled/v0.7/OhMyREPL.ji for module OhMyREPL\n┌ Debug: Rejecting cache file /home/user/.julia/compiled/v0.7/Tokenize.ji due to it containing an incompatible cache header\n└ @ Base loading.jl:1328\n..."},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"On windows, the same can be achieved in CMD via first running set JULIA_DEBUG=\"loading\" and in Powershell via $env:JULIA_DEBUG=\"loading\"."},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Similarly, the environment variable can be used to enable debug logging of modules, such as Pkg, or module roots (see Base.moduleroot). To enable all debug logging, use the special value all."},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"To turn debug logging on from the REPL, set ENV[\"JULIA_DEBUG\"] to the name of the module of interest. Functions defined in the REPL belong to module Main; logging for them can be enabled like this:"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"julia> foo() = @debug \"foo\"\nfoo (generic function with 1 method)\n\njulia> foo()\n\njulia> ENV[\"JULIA_DEBUG\"] = Main\nMain\n\njulia> foo()\n┌ Debug: foo\n└ @ Main REPL[1]:1\n"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Use a comma separator to enable debug for multiple modules: JULIA_DEBUG=loading,Main."},{"title":"Examples","page":"Logging","location":"stdlib/Logging.html#Examples","category":"section","text":""},{"title":"Example: Writing log events to a file","page":"Logging","location":"stdlib/Logging.html#Example:-Writing-log-events-to-a-file","category":"section","text":""},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Sometimes it can be useful to write log events to a file. Here is an example of how to use a task-local and global logger to write information to a text file:"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"# Load the logging module\njulia> using Logging\n\n# Open a textfile for writing\njulia> io = open(\"log.txt\", \"w+\")\nIOStream(<file log.txt>)\n\n# Create a simple logger\njulia> logger = SimpleLogger(io)\nSimpleLogger(IOStream(<file log.txt>), Info, Dict{Any,Int64}())\n\n# Log a task-specific message\njulia> with_logger(logger) do\n           @info(\"a context specific log message\")\n       end\n\n# Write all buffered messages to the file\njulia> flush(io)\n\n# Set the global logger to logger\njulia> global_logger(logger)\nSimpleLogger(IOStream(<file log.txt>), Info, Dict{Any,Int64}())\n\n# This message will now also be written to the file\njulia> @info(\"a global log message\")\n\n# Close the file\njulia> close(io)"},{"title":"Example: Enable debug-level messages","page":"Logging","location":"stdlib/Logging.html#Example:-Enable-debug-level-messages","category":"section","text":""},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Here is an example of creating a ConsoleLogger that lets through any messages with log level higher than, or equal, to Logging.Debug."},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"julia> using Logging\n\n# Create a ConsoleLogger that prints any log messages with level >= Debug to stderr\njulia> debuglogger = ConsoleLogger(stderr, Logging.Debug)\n\n# Enable debuglogger for a task\njulia> with_logger(debuglogger) do\n           @debug \"a context specific log message\"\n       end\n\n# Set the global logger\njulia> global_logger(debuglogger)"},{"title":"Reference","page":"Logging","location":"stdlib/Logging.html#Reference","category":"section","text":""},{"title":"Logging module","page":"Logging","location":"stdlib/Logging.html#Logging-module","category":"section","text":""},{"title":"Logging.Logging","page":"Logging","location":"stdlib/Logging.html#Logging.Logging","category":"module","text":"Utilities for capturing, filtering and presenting streams of log events. Normally you don't need to import Logging to create log events; for this the standard logging macros such as @info are already exported by Base and available by default.\n\n\n\n\n\n"},{"title":"Creating events","page":"Logging","location":"stdlib/Logging.html#Creating-events","category":"section","text":""},{"title":"Base.CoreLogging.@logmsg","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.@logmsg","category":"macro","text":"@debug message  [key=value | value ...]\n@info  message  [key=value | value ...]\n@warn  message  [key=value | value ...]\n@error message  [key=value | value ...]\n\n@logmsg level message [key=value | value ...]\n\nCreate a log record with an informational message.  For convenience, four logging macros @debug, @info, @warn and @error are defined which log at the standard severity levels Debug, Info, Warn and Error.  @logmsg allows level to be set programmatically to any LogLevel or custom log level types.\n\nmessage should be an expression which evaluates to a string which is a human readable description of the log event.  By convention, this string will be formatted as markdown when presented.\n\nThe optional list of key=value pairs supports arbitrary user defined metadata which will be passed through to the logging backend as part of the log record.  If only a value expression is supplied, a key representing the expression will be generated using Symbol. For example, x becomes x=x, and foo(10) becomes Symbol(\"foo(10)\")=foo(10).  For splatting a list of key value pairs, use the normal splatting syntax, @info \"blah\" kws....\n\nThere are some keys which allow automatically generated log data to be overridden:\n\n_module=mod can be used to specify a different originating module from the source location of the message.\n_group=symbol can be used to override the message group (this is normally derived from the base name of the source file).\n_id=symbol can be used to override the automatically generated unique message identifier.  This is useful if you need to very closely associate messages generated on different source lines.\n_file=string and _line=integer can be used to override the apparent source location of a log message.\n\nThere's also some key value pairs which have conventional meaning:\n\nmaxlog=integer should be used as a hint to the backend that the message should be displayed no more than maxlog times.\nexception=ex should be used to transport an exception with a log message, often used with @error. An associated backtrace bt may be attached using the tuple exception=(ex,bt).\n\nExamples\n\n@debug \"Verbose debugging information.  Invisible by default\"\n@info  \"An informational message\"\n@warn  \"Something was odd.  You should pay attention\"\n@error \"A non fatal error occurred\"\n\nx = 10\n@info \"Some variables attached to the message\" x a=42.0\n\n@debug begin\n    sA = sum(A)\n    \"sum(A) = $sA is an expensive operation, evaluated only when `shouldlog` returns true\"\nend\n\nfor i=1:10000\n    @info \"With the default backend, you will only see (i = $i) ten times\"  maxlog=10\n    @debug \"Algorithm1\" i progress=i/10000\nend\n\n\n\n\n\n"},{"title":"Base.CoreLogging.LogLevel","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.LogLevel","category":"type","text":"LogLevel(level)\n\nSeverity/verbosity of a log record.\n\nThe log level provides a key against which potential log records may be filtered, before any other work is done to construct the log record data structure itself.\n\nExamples\n\njulia> Logging.LogLevel(0) == Logging.Info\ntrue\n\n\n\n\n\n"},{"title":"Base.CoreLogging.Debug","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.Debug","category":"constant","text":"Debug\n\nAlias for LogLevel(-1000).\n\n\n\n\n\n"},{"title":"Base.CoreLogging.Info","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.Info","category":"constant","text":"Info\n\nAlias for LogLevel(0).\n\n\n\n\n\n"},{"title":"Base.CoreLogging.Warn","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.Warn","category":"constant","text":"Warn\n\nAlias for LogLevel(1000).\n\n\n\n\n\n"},{"title":"Base.CoreLogging.Error","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.Error","category":"constant","text":"Error\n\nAlias for LogLevel(2000).\n\n\n\n\n\n"},{"title":"Base.CoreLogging.BelowMinLevel","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.BelowMinLevel","category":"constant","text":"BelowMinLevel\n\nAlias for LogLevel(-1_000_001).\n\n\n\n\n\n"},{"title":"Base.CoreLogging.AboveMaxLevel","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.AboveMaxLevel","category":"constant","text":"AboveMaxLevel\n\nAlias for LogLevel(1_000_001).\n\n\n\n\n\n"},{"title":"Processing events with AbstractLogger","page":"Logging","location":"stdlib/Logging.html#AbstractLogger-interface","category":"section","text":""},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Event processing is controlled by overriding functions associated with AbstractLogger:"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Methods to implement  Brief description\nLogging.handle_message  Handle a log event\nLogging.shouldlog  Early filtering of events\nLogging.min_enabled_level  Lower bound for log level of accepted events\nOptional methods Default definition Brief description\nLogging.catch_exceptions true Catch exceptions during event evaluation"},{"title":"Base.CoreLogging.AbstractLogger","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.AbstractLogger","category":"type","text":"A logger controls how log records are filtered and dispatched.  When a log record is generated, the logger is the first piece of user configurable code which gets to inspect the record and decide what to do with it.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.handle_message","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.handle_message","category":"function","text":"handle_message(logger, level, message, _module, group, id, file, line; key1=val1, ...)\n\nLog a message to logger at level.  The logical location at which the message was generated is given by module _module and group; the source location by file and line. id is an arbitrary unique value (typically a Symbol) to be used as a key to identify the log statement when filtering.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.shouldlog","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.shouldlog","category":"function","text":"shouldlog(logger, level, _module, group, id)\n\nReturn true when logger accepts a message at level, generated for _module, group and with unique log identifier id.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.min_enabled_level","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.min_enabled_level","category":"function","text":"min_enabled_level(logger)\n\nReturn the minimum enabled level for logger for early filtering.  That is, the log level below or equal to which all messages are filtered.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.catch_exceptions","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.catch_exceptions","category":"function","text":"catch_exceptions(logger)\n\nReturn true if the logger should catch exceptions which happen during log record construction.  By default, messages are caught.\n\nBy default all exceptions are caught to prevent log message generation from crashing the program.  This lets users confidently toggle little-used functionality - such as debug logging - in a production system.\n\nIf you want to use logging as an audit trail you should disable this for your logger type.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.disable_logging","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.disable_logging","category":"function","text":"disable_logging(level)\n\nDisable all log messages at log levels equal to or less than level.  This is a global setting, intended to make debug logging extremely cheap when disabled. Note that this cannot be used to enable logging that is currently disabled by other mechanisms.\n\nExamples\n\nLogging.disable_logging(Logging.Info) # Disable debug and info\n\n\n\n\n\n"},{"title":"Using Loggers","page":"Logging","location":"stdlib/Logging.html#Using-Loggers","category":"section","text":""},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Logger installation and inspection:"},{"title":"Base.CoreLogging.global_logger","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.global_logger","category":"function","text":"global_logger()\n\nReturn the global logger, used to receive messages when no specific logger exists for the current task.\n\nglobal_logger(logger)\n\nSet the global logger to logger, and return the previous global logger.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.with_logger","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.with_logger","category":"function","text":"with_logger(function, logger)\n\nExecute function, directing all log messages to logger.\n\nExamples\n\nfunction test(x)\n    @info \"x = $x\"\nend\n\nwith_logger(logger) do\n    test(1)\n    test([1,2])\nend\n\n\n\n\n\n"},{"title":"Base.CoreLogging.current_logger","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.current_logger","category":"function","text":"current_logger()\n\nReturn the logger for the current task, or the global logger if none is attached to the task.\n\n\n\n\n\n"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html","category":"page","text":"Loggers that are supplied with the system:"},{"title":"Base.CoreLogging.NullLogger","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.NullLogger","category":"type","text":"NullLogger()\n\nLogger which disables all messages and produces no output - the logger equivalent of /dev/null.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.ConsoleLogger","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.ConsoleLogger","category":"type","text":"ConsoleLogger([stream,] min_level=Info; meta_formatter=default_metafmt,\n              show_limited=true, right_justify=0)\n\nLogger with formatting optimized for readability in a text console, for example interactive work with the Julia REPL.\n\nLog levels less than min_level are filtered out.\n\nThis Logger is thread-safe, with locks for both orchestration of message limits i.e. maxlog, and writes to the stream.\n\nMessage formatting can be controlled by setting keyword arguments:\n\nmeta_formatter is a function which takes the log event metadata (level, _module, group, id, file, line) and returns a color (as would be passed to printstyled), prefix and suffix for the log message.  The default is to prefix with the log level and a suffix containing the module, file and line location.\nshow_limited limits the printing of large data structures to something which can fit on the screen by setting the :limit IOContext key during formatting.\nright_justify is the integer column which log metadata is right justified at. The default is zero (metadata goes on its own line).\n\n\n\n\n\n"},{"title":"Base.CoreLogging.SimpleLogger","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.SimpleLogger","category":"type","text":"SimpleLogger([stream,] min_level=Info)\n\nSimplistic logger for logging all messages with level greater than or equal to min_level to stream. If stream is closed then messages with log level greater or equal to Warn will be logged to stderr and below to stdout.\n\nThis Logger is thread-safe, with a lock taken around orchestration of message limits i.e. maxlog, and writes to the stream.\n\n\n\n\n\n"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html#Fixing-precompilation-hangs-due-to-open-tasks-or-IO","category":"section","text":""},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"On Julia 1.10 or higher, you might see the following message:"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"(Image: Screenshot of precompilation hang)"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"This may repeat. If it continues to repeat with no hints that it will resolve itself, you may have a \"precompilation hang\" that requires fixing. Even if it's transient, you might prefer to resolve it so that users will not be bothered by this warning. This page walks you through how to analyze and fix such issues."},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"If you follow the advice and hit Ctrl-C, you might see"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"^C Interrupted: Exiting precompilation...\n\n  1 dependency had warnings during precompilation:\n┌ Test1 [ac89d554-e2ba-40bc-bc5c-de68b658c982]\n│  [pid 2745] waiting for IO to finish:\n│   Handle type        uv_handle_t->data\n│   timer              0x55580decd1e0->0x7f94c3a4c340"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"This message conveys two key pieces of information:"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"the hang is occurring during precompilation of Test1, a dependency of Test2 (the package we were trying to load with using Test2)\nduring precompilation of Test1, Julia created a Timer object (use ?Timer if you're unfamiliar with Timers) which is still open; until that closes, the process is hung"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"If this is enough of a hint for you to figure out how timer = Timer(args...) is being created, one good solution is to add wait(timer) if timer eventually finishes on its own, or close(timer) if you need to force-close it, before the final end of the module."},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"However, there are cases that may not be that straightforward. Usually the best option is to start by determining whether the hang is due to code in Test1 or whether it is due to one of Test1's dependencies:"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"Option 1: Pkg.add(\"Aqua\") and use Aqua.test_persistent_tasks. This should help you identify which package is causing the problem, after which the instructions below should be followed. If needed, you can create a PkgId as Base.PkgId(UUID(\"...\"), \"Test1\"), where ... comes from the uuid entry in Test1/Project.toml.\nOption 2: manually diagnose the source of the hang."},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"To manually diagnose:"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"Pkg.develop(\"Test1\")\nComment out all the code included or defined in Test1, except the using/import statements.\nTry using Test2 (or even using Test1 assuming that hangs too) again"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"Now we arrive at a fork in the road: either"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"the hang persists, indicating it is due to one of your dependencies\nthe hang disappears, indicating that it is due to something in your code."},{"title":"Diagnosing and fixing hangs due to a package dependency","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html#pchang_deps","category":"section","text":""},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"Use a binary search to identify the problematic dependency: start by commenting out half your dependencies, then when you isolate which half is responsible comment out half of that half, etc. (You don't have to remove them from the project, just comment out the using/import statements.)"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"Once you've identified a suspect (here we'll call it ThePackageYouThinkIsCausingTheProblem), first try precompiling that package. If it also hangs during precompilation, continue chasing the problem backwards."},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"However, most likely ThePackageYouThinkIsCausingTheProblem will precompile fine. This suggests it's in the function ThePackageYouThinkIsCausingTheProblem.__init__, which does not run during precompilation of ThePackageYouThinkIsCausingTheProblem but does in any package that loads ThePackageYouThinkIsCausingTheProblem. To test this theory, set up a minimal working example (MWE), something like"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"(@v1.10) pkg> generate MWE\n  Generating  project MWE:\n    MWE\\Project.toml\n    MWE\\src\\MWE.jl"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"where the source code of MWE.jl is"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"module MWE\nusing ThePackageYouThinkIsCausingTheProblem\nend"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"and you've added ThePackageYouThinkIsCausingTheProblem to MWE's dependencies."},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"If that MWE reproduces the hang, you've found your culprit: ThePackageYouThinkIsCausingTheProblem.__init__ must be creating the Timer object. If the timer object can be safely closed, that's a good option. Otherwise, the most common solution is to avoid creating the timer while any package is being precompiled: add"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"ccall(:jl_generating_output, Cint, ()) == 1 && return nothing"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"as the first line of ThePackageYouThinkIsCausingTheProblem.__init__, and it will avoid doing any initialization in any Julia process whose purpose is to precompile packages."},{"title":"Fixing package code to avoid hangs","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html#pchang_fix","category":"section","text":""},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"Search your package for suggestive words (here like \"Timer\") and see if you can identify where the problem is being created. Note that a method definition like"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"maketimer() = Timer(timer -> println(\"hi\"), 0; interval=1)"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"is not problematic in and of itself: it can cause this problem only if maketimer gets called while the module is being defined. This might be happening from a top-level statement such as"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"const GLOBAL_TIMER = maketimer()"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"or it might conceivably occur in a precompile workload."},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html","category":"page","text":"If you struggle to identify the causative lines, then consider doing a binary search: comment out sections of your package (or include lines to omit entire files) until you've reduced the problem in scope."},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html#Using-Valgrind-with-Julia","category":"section","text":""},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"Valgrind is a tool for memory debugging, memory leak detection, and profiling.  This section describes things to keep in mind when using Valgrind to debug memory issues with Julia."},{"title":"General considerations","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html#General-considerations","category":"section","text":""},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"By default, Valgrind assumes that there is no self modifying code in the programs it runs. This assumption works fine in most instances but fails miserably for a just-in-time compiler like julia. For this reason it is crucial to pass --smc-check=all-non-file to valgrind, else code may crash or behave unexpectedly (often in subtle ways)."},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"In some cases, to better detect memory errors using Valgrind, it can help to compile julia with memory pools disabled. The compile-time flag MEMDEBUG disables memory pools in Julia, and MEMDEBUG2 disables memory pools in FemtoLisp. To build julia with both flags, add the following line to Make.user:"},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"CFLAGS = -DMEMDEBUG -DMEMDEBUG2"},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"Another thing to note: if your program uses multiple worker processes, it is likely that you want all such worker processes to run under Valgrind, not just the parent process. To do this, pass --trace-children=yes to valgrind."},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"Yet another thing to note: if using valgrind errors with Unable to find compatible target in system image, try rebuilding the sysimage with target generic or julia with JULIA_CPU_TARGET=generic."},{"title":"Suppressions","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html#Suppressions","category":"section","text":""},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"Valgrind will typically display spurious warnings as it runs. To reduce the number of such warnings, it helps to provide a suppressions file to Valgrind. A sample suppressions file is included in the Julia source distribution at contrib/valgrind-julia.supp."},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"The suppressions file can be used from the julia/ source directory as follows:"},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"$ valgrind --smc-check=all-non-file --suppressions=contrib/valgrind-julia.supp ./julia progname.jl"},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"Any memory errors that are displayed should either be reported as bugs or contributed as additional suppressions. Note that some versions of Valgrind are shipped with insufficient default suppressions, so that may be one thing to consider before submitting any bugs."},{"title":"Running the Julia test suite under Valgrind","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html#Running-the-Julia-test-suite-under-Valgrind","category":"section","text":""},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"It is possible to run the entire Julia test suite under Valgrind, but it does take quite some time (typically several hours). To do so, run the following command from the julia/test/ directory:"},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"valgrind --smc-check=all-non-file --trace-children=yes --suppressions=$PWD/../contrib/valgrind-julia.supp ../julia runtests.jl all"},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"If you would like to see a report of \"definite\" memory leaks, pass the flags --leak-check=full --show-leak-kinds=definite to valgrind as well."},{"title":"Additional spurious warnings","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html#Additional-spurious-warnings","category":"section","text":""},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"This section covers Valgrind warnings that cannot be added to the suppressions file yet are nonetheless safe to ignore."},{"title":"Unhandled rr system calls","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html#Unhandled-rr-system-calls","category":"section","text":""},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"Valgrind will emit a warning if it encounters any of the system calls that are specific to rr, the Record and Replay Framework. In particular, a warning about an unhandled 1008 syscall will be shown when julia tries to detect whether it is running under rr:"},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"--xxxxxx-- WARNING: unhandled amd64-linux syscall: 1008\n--xxxxxx-- You may be able to write your own handler.\n--xxxxxx-- Read the file README_MISSING_SYSCALL_OR_IOCTL.\n--xxxxxx-- Nevertheless we consider this a bug.  Please report\n--xxxxxx-- it at http://valgrind.org/support/bug_reports.html."},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"This issue has been reported to the Valgrind developers as they have requested."},{"title":"Caveats","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html#Caveats","category":"section","text":""},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"Valgrind currently does not support multiple rounding modes, so code that adjusts the rounding mode will behave differently when run under Valgrind."},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html","category":"page","text":"In general, if after setting --smc-check=all-non-file you find that your program behaves differently when run under Valgrind, it may help to pass --tool=none to valgrind as you investigate further.  This will enable the minimal Valgrind machinery but will also run much faster than when the full memory checker is enabled."},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html#man-workflow-tips","category":"section","text":""},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"Here are some tips for working with Julia efficiently."},{"title":"REPL-based workflow","page":"Workflow Tips","location":"manual/workflow-tips.html#REPL-based-workflow","category":"section","text":""},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"As already elaborated in The Julia REPL, Julia's REPL provides rich functionality that facilitates an efficient interactive workflow. Here are some tips that might further enhance your experience at the command line."},{"title":"A basic editor/REPL workflow","page":"Workflow Tips","location":"manual/workflow-tips.html#A-basic-editor/REPL-workflow","category":"section","text":""},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"The most basic Julia workflows involve using a text editor in conjunction with the julia command line."},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"Create a file, say Tmp.jl, and include within it"},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"module Tmp\n\nsay_hello() = println(\"Hello!\")\n\n# Your other definitions here\n\nend # module\n\nusing .Tmp"},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"Then, in the same directory, start the Julia REPL (using the julia command). Run the new file as follows:"},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"julia> include(\"Tmp.jl\")\n\njulia> Tmp.say_hello()\nHello!"},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"Explore ideas in the REPL. Save good ideas in Tmp.jl. To reload the file after it has been changed, just include it again."},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"The key in the above is that your code is encapsulated in a module. That allows you to edit struct definitions and remove methods, without restarting Julia."},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"(Explanation: structs cannot be edited after definition, nor can methods be deleted. But you can overwrite the definition of a module, which is what we do when we re-include(\"Tmp.jl\"))."},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"In addition, the encapsulation of code in a module protects it from being influenced by previous state in the REPL, protecting you from hard-to-detect errors."},{"title":"Browser-based workflow","page":"Workflow Tips","location":"manual/workflow-tips.html#Browser-based-workflow","category":"section","text":""},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"There are a few ways to interact with Julia in a browser:"},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"Using Pluto notebooks through Pluto.jl\nUsing Jupyter notebooks through IJulia.jl"},{"title":"Revise-based workflows","page":"Workflow Tips","location":"manual/workflow-tips.html#Revise-based-workflows","category":"section","text":""},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"Whether you're at the REPL or in IJulia, you can typically improve your development experience with Revise. It is common to configure Revise to start whenever julia is started, as per the instructions in the Revise documentation. Once configured, Revise will track changes to files in any loaded modules, and to any files loaded in to the REPL with includet (but not with plain include); you can then edit the files and the changes take effect without restarting your julia session. A standard workflow is similar to the REPL-based workflow above, with the following modifications:"},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html","category":"page","text":"Put your code in a module somewhere on your load path. There are several options for achieving this, of which two recommended choices are:\nFor long-term projects, use PkgTemplates:\nusing PkgTemplates\nt = Template()\nt(\"MyPkg\")\nThis will create a blank package, \"MyPkg\", in your .julia/dev directory. Note that PkgTemplates allows you to control many different options through its Template constructor.\nIn step 2 below, edit MyPkg/src/MyPkg.jl to change the source code, and MyPkg/test/runtests.jl for the tests.\nFor \"throw-away\" projects, you can avoid any need for cleanup by doing your work in your temporary directory (e.g., /tmp).\nNavigate to your temporary directory and launch Julia, then do the following:\npkg> generate MyPkg            # type ] to enter pkg mode\njulia> push!(LOAD_PATH, pwd())   # hit backspace to exit pkg mode\nIf you restart your Julia session you'll have to re-issue that command modifying LOAD_PATH.\nIn step 2 below, edit MyPkg/src/MyPkg.jl to change the source code, and create any test file of your choosing.\nDevelop your package\nBefore loading any code, make sure you're running Revise: say using Revise or follow its documentation on configuring it to run automatically.\nThen navigate to the directory containing your test file (here assumed to be \"runtests.jl\") and do the following:\njulia> using MyPkg\n\njulia> include(\"runtests.jl\")\nYou can iteratively modify the code in MyPkg in your editor and re-run the tests with include(\"runtests.jl\"). You generally should not need to restart your Julia session to see the changes take effect (subject to a few limitations)."},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html#Complex-and-Rational-Numbers","category":"section","text":""},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Julia includes predefined types for both complex and rational numbers, and supports all the standard Mathematical Operations and Elementary Functions on them. Conversion and Promotion are defined so that operations on any combination of predefined numeric types, whether primitive or composite, behave as expected."},{"title":"Complex Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html#Complex-Numbers","category":"section","text":""},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"The global constant im is bound to the complex number i, representing the principal square root of -1. (Using mathematicians' i or engineers' j for this global constant was rejected since they are such popular index variable names.) Since Julia allows numeric literals to be juxtaposed with identifiers as coefficients, this binding suffices to provide convenient syntax for complex numbers, similar to the traditional mathematical notation:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 1+2im\n1 + 2im"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"You can perform all the standard arithmetic operations with complex numbers:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> (1 + 2im)*(2 - 3im)\n8 + 1im\n\njulia> (1 + 2im)/(1 - 2im)\n-0.6 + 0.8im\n\njulia> (1 + 2im) + (1 - 2im)\n2 + 0im\n\njulia> (-3 + 2im) - (5 - 1im)\n-8 + 3im\n\njulia> (-1 + 2im)^2\n-3 - 4im\n\njulia> (-1 + 2im)^2.5\n2.729624464784009 - 6.9606644595719im\n\njulia> (-1 + 2im)^(1 + 1im)\n-0.27910381075826657 + 0.08708053414102428im\n\njulia> 3(2 - 5im)\n6 - 15im\n\njulia> 3(2 - 5im)^2\n-63 - 60im\n\njulia> 3(2 - 5im)^-1.0\n0.20689655172413793 + 0.5172413793103449im"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"The promotion mechanism ensures that combinations of operands of different types just work:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 2(1 - 1im)\n2 - 2im\n\njulia> (2 + 3im) - 1\n1 + 3im\n\njulia> (1 + 2im) + 0.5\n1.5 + 2.0im\n\njulia> (2 + 3im) - 0.5im\n2.0 + 2.5im\n\njulia> 0.75(1 + 2im)\n0.75 + 1.5im\n\njulia> (2 + 3im) / 2\n1.0 + 1.5im\n\njulia> (1 - 3im) / (2 + 2im)\n-0.5 - 1.0im\n\njulia> 2im^2\n-2 + 0im\n\njulia> 1 + 3/4im\n1.0 - 0.75im"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Note that 3/4im == 3/(4*im) == -(3/4*im), since a literal coefficient binds more tightly than division."},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Standard functions to manipulate complex values are provided:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> z = 1 + 2im\n1 + 2im\n\njulia> real(1 + 2im) # real part of z\n1\n\njulia> imag(1 + 2im) # imaginary part of z\n2\n\njulia> conj(1 + 2im) # complex conjugate of z\n1 - 2im\n\njulia> abs(1 + 2im) # absolute value of z\n2.23606797749979\n\njulia> abs2(1 + 2im) # squared absolute value\n5\n\njulia> angle(1 + 2im) # phase angle in radians\n1.1071487177940904"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"As usual, the absolute value (abs) of a complex number is its distance from zero. abs2 gives the square of the absolute value, and is of particular use for complex numbers since it avoids taking a square root. angle returns the phase angle in radians (also known as the argument or arg function). The full gamut of other Elementary Functions is also defined for complex numbers:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> sqrt(1im)\n0.7071067811865476 + 0.7071067811865475im\n\njulia> sqrt(1 + 2im)\n1.272019649514069 + 0.7861513777574233im\n\njulia> cos(1 + 2im)\n2.0327230070196656 - 3.0518977991517997im\n\njulia> exp(1 + 2im)\n-1.1312043837568135 + 2.4717266720048188im\n\njulia> sinh(1 + 2im)\n-0.4890562590412937 + 1.4031192506220405im"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Note that mathematical functions typically return real values when applied to real numbers and complex values when applied to complex numbers. For example, sqrt behaves differently when applied to -1 versus -1 + 0im even though -1 == -1 + 0im:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> sqrt(-1)\nERROR: DomainError with -1.0:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\njulia> sqrt(-1 + 0im)\n0.0 + 1.0im"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"The literal numeric coefficient notation does not work when constructing a complex number from variables. Instead, the multiplication must be explicitly written out:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> a = 1; b = 2; a + b*im\n1 + 2im"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"However, this is not recommended. Instead, use the more efficient complex function to construct a complex value directly from its real and imaginary parts:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> a = 1; b = 2; complex(a, b)\n1 + 2im"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"This construction avoids the multiplication and addition operations."},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Inf and NaN propagate through complex numbers in the real and imaginary parts of a complex number as described in the Special floating-point values section:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 1 + Inf*im\n1.0 + Inf*im\n\njulia> 1 + NaN*im\n1.0 + NaN*im"},{"title":"Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html#Rational-Numbers","category":"section","text":""},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Julia has a rational number type to represent exact ratios of integers. Rationals are constructed using the // operator:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 2//3\n2//3"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"If the numerator and denominator of a rational have common factors, they are reduced to lowest terms such that the denominator is non-negative:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 6//9\n2//3\n\njulia> -4//8\n-1//2\n\njulia> 5//-15\n-1//3\n\njulia> -4//-12\n1//3"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"This normalized form for a ratio of integers is unique, so equality of rational values can be tested by checking for equality of the numerator and denominator. The standardized numerator and denominator of a rational value can be extracted using the numerator and denominator functions:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> numerator(2//3)\n2\n\njulia> denominator(2//3)\n3"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Direct comparison of the numerator and denominator is generally not necessary, since the standard arithmetic and comparison operations are defined for rational values:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 2//3 == 6//9\ntrue\n\njulia> 2//3 == 9//27\nfalse\n\njulia> 3//7 < 1//2\ntrue\n\njulia> 3//4 > 2//3\ntrue\n\njulia> 2//4 + 1//6\n2//3\n\njulia> 5//12 - 1//4\n1//6\n\njulia> 5//8 * 3//12\n5//32\n\njulia> 6//5 / 10//7\n21//25"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Rationals can easily be converted to floating-point numbers:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> float(3//4)\n0.75"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Conversion from rational to floating-point respects the following identity for any integral values of a and b, except when a==0 && b <= 0:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> a = 1; b = 2;\n\njulia> isequal(float(a//b), a/b)\ntrue\n\njulia> a, b = 0, 0\n(0, 0)\n\njulia> float(a//b)\nERROR: ArgumentError: invalid rational: zero(Int64)//zero(Int64)\nStacktrace:\n[...]\n\njulia> a/b\nNaN\n\njulia> a, b = 0, -1\n(0, -1)\n\njulia> float(a//b), a/b\n(0.0, -0.0)"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Constructing infinite rational values is acceptable:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 5//0\n1//0\n\njulia> x = -3//0\n-1//0\n\njulia> typeof(x)\nRational{Int64}"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"Trying to construct a NaN rational value, however, is invalid:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 0//0\nERROR: ArgumentError: invalid rational: zero(Int64)//zero(Int64)\nStacktrace:\n[...]"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"As usual, the promotion system makes interactions with other numeric types effortless:"},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html","category":"page","text":"julia> 3//5 + 1\n8//5\n\njulia> 3//5 - 0.5\n0.09999999999999998\n\njulia> 2//7 * (1 + 2im)\n2//7 + 4//7*im\n\njulia> 2//7 * (1.5 + 2im)\n0.42857142857142855 + 0.5714285714285714im\n\njulia> 3//2 / (1 + 2im)\n3//10 - 3//5*im\n\njulia> 1//2 + 2im\n1//2 + 2//1*im\n\njulia> 1 + 2//3im\n1//1 - 2//3*im\n\njulia> 0.5 == 1//2\ntrue\n\njulia> 0.33 == 1//3\nfalse\n\njulia> 0.33 < 1//3\ntrue\n\njulia> 1//3 - 0.33\n0.0033333333333332993"},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html#SHA","category":"section","text":""},{"title":"SHA functions","page":"SHA","location":"stdlib/SHA.html#SHA-functions","category":"section","text":""},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"Usage is very straightforward:"},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"julia> using SHA\n\njulia> bytes2hex(sha256(\"test\"))\n\"9f86d081884c7d659a2feaa0c55ad015a3bf4f1b2b0b822cd15d6c15b0f00a08\""},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"Each exported function (at the time of this writing, SHA-1, SHA-2 224, 256, 384 and 512, and SHA-3 224, 256, 384 and 512 functions are implemented) takes in either an AbstractVector{UInt8}, an AbstractString or an IO object.  This makes it trivial to checksum a file:"},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"shell> cat /tmp/test.txt\ntest\njulia> using SHA\n\njulia> open(\"/tmp/test.txt\") do f\n           sha2_256(f)\n       end\n32-element Vector{UInt8}:\n 0x9f\n 0x86\n 0xd0\n 0x81\n 0x88\n 0x4c\n 0x7d\n 0x65\n    ⋮\n 0x5d\n 0x6c\n 0x15\n 0xb0\n 0xf0\n 0x0a\n 0x08"},{"title":"All SHA functions","page":"SHA","location":"stdlib/SHA.html#All-SHA-functions","category":"section","text":""},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"Due to the colloquial usage of sha256 to refer to sha2_256, convenience functions are provided, mapping shaxxx() function calls to sha2_xxx(). For SHA-3, no such colloquialisms exist and the user must use the full sha3_xxx() names."},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"shaxxx() takes AbstractString and array-like objects (NTuple and Vector) with elements of type UInt8."},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-1"},{"title":"SHA.sha1","page":"SHA","location":"stdlib/SHA.html#SHA.sha1","category":"function","text":"sha1(data)\n\nHash data using the sha1 algorithm and return the resulting digest. See also SHA1_CTX.\n\n\n\n\n\nsha1(io::IO)\n\nHash data from io using sha1 algorithm.\n\n\n\n\n\n"},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-2"},{"title":"SHA.sha224","page":"SHA","location":"stdlib/SHA.html#SHA.sha224","category":"function","text":"sha224(data)\n\nHash data using the sha224 algorithm and return the resulting digest. See also SHA2_224_CTX.\n\n\n\n\n\nsha224(io::IO)\n\nHash data from io using sha224 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha256","page":"SHA","location":"stdlib/SHA.html#SHA.sha256","category":"function","text":"sha256(data)\n\nHash data using the sha256 algorithm and return the resulting digest. See also SHA2_256_CTX.\n\n\n\n\n\nsha256(io::IO)\n\nHash data from io using sha256 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha384","page":"SHA","location":"stdlib/SHA.html#SHA.sha384","category":"function","text":"sha384(data)\n\nHash data using the sha384 algorithm and return the resulting digest. See also SHA2_384_CTX.\n\n\n\n\n\nsha384(io::IO)\n\nHash data from io using sha384 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha512","page":"SHA","location":"stdlib/SHA.html#SHA.sha512","category":"function","text":"sha512(data)\n\nHash data using the sha512 algorithm and return the resulting digest. See also SHA2_512_CTX.\n\n\n\n\n\nsha512(io::IO)\n\nHash data from io using sha512 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha2_224","page":"SHA","location":"stdlib/SHA.html#SHA.sha2_224","category":"function","text":"sha2_224(data)\n\nHash data using the sha2_224 algorithm and return the resulting digest. See also SHA2_224_CTX.\n\n\n\n\n\nsha2_224(io::IO)\n\nHash data from io using sha2_224 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha2_256","page":"SHA","location":"stdlib/SHA.html#SHA.sha2_256","category":"function","text":"sha2_256(data)\n\nHash data using the sha2_256 algorithm and return the resulting digest. See also SHA2_256_CTX.\n\n\n\n\n\nsha2_256(io::IO)\n\nHash data from io using sha2_256 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha2_384","page":"SHA","location":"stdlib/SHA.html#SHA.sha2_384","category":"function","text":"sha2_384(data)\n\nHash data using the sha2_384 algorithm and return the resulting digest. See also SHA2_384_CTX.\n\n\n\n\n\nsha2_384(io::IO)\n\nHash data from io using sha2_384 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha2_512","page":"SHA","location":"stdlib/SHA.html#SHA.sha2_512","category":"function","text":"sha2_512(data)\n\nHash data using the sha2_512 algorithm and return the resulting digest. See also SHA2_512_CTX.\n\n\n\n\n\nsha2_512(io::IO)\n\nHash data from io using sha2_512 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha2_512_224","page":"SHA","location":"stdlib/SHA.html#SHA.sha2_512_224","category":"function","text":"sha2_512_224(data)\n\nHash data using the sha2_512_224 algorithm and return the resulting digest. See also SHA2_512_224_CTX.\n\n\n\n\n\nsha2_512_224(io::IO)\n\nHash data from io using sha2_512_224 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha2_512_256","page":"SHA","location":"stdlib/SHA.html#SHA.sha2_512_256","category":"function","text":"sha2_512_256(data)\n\nHash data using the sha2_512_256 algorithm and return the resulting digest. See also SHA2_512_256_CTX.\n\n\n\n\n\nsha2_512_256(io::IO)\n\nHash data from io using sha2_512_256 algorithm.\n\n\n\n\n\n"},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-3"},{"title":"SHA.sha3_224","page":"SHA","location":"stdlib/SHA.html#SHA.sha3_224","category":"function","text":"sha3_224(data)\n\nHash data using the sha3_224 algorithm and return the resulting digest. See also SHA3_224_CTX.\n\n\n\n\n\nsha3_224(io::IO)\n\nHash data from io using sha3_224 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha3_256","page":"SHA","location":"stdlib/SHA.html#SHA.sha3_256","category":"function","text":"sha3_256(data)\n\nHash data using the sha3_256 algorithm and return the resulting digest. See also SHA3_256_CTX.\n\n\n\n\n\nsha3_256(io::IO)\n\nHash data from io using sha3_256 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha3_384","page":"SHA","location":"stdlib/SHA.html#SHA.sha3_384","category":"function","text":"sha3_384(data)\n\nHash data using the sha3_384 algorithm and return the resulting digest. See also SHA3_384_CTX.\n\n\n\n\n\nsha3_384(io::IO)\n\nHash data from io using sha3_384 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha3_512","page":"SHA","location":"stdlib/SHA.html#SHA.sha3_512","category":"function","text":"sha3_512(data)\n\nHash data using the sha3_512 algorithm and return the resulting digest. See also SHA3_512_CTX.\n\n\n\n\n\nsha3_512(io::IO)\n\nHash data from io using sha3_512 algorithm.\n\n\n\n\n\n"},{"title":"Working with context","page":"SHA","location":"stdlib/SHA.html#Working-with-context","category":"section","text":""},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"To create a hash from multiple items the SHAX_XXX_CTX() types can be used to create a stateful hash object that is updated with update! and finalized with digest!"},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"julia> using SHA\n\njulia> ctx = SHA2_256_CTX()\nSHA2 256-bit hash state\n\njulia> update!(ctx, b\"some data\")\n0x0000000000000009\n\njulia> update!(ctx, b\"some more data\")\n0x0000000000000017\n\njulia> digest!(ctx)\n32-element Vector{UInt8}:\n 0xbe\n 0xcf\n 0x23\n 0xda\n 0xaf\n 0x02\n 0xf7\n 0xa3\n 0x57\n 0x92\n    ⋮\n 0x89\n 0x4f\n 0x59\n 0xd8\n 0xb3\n 0xb4\n 0x81\n 0x8b\n 0xc5"},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"Note that, at the time of this writing, the SHA3 code is not optimized, and as such is roughly an order of magnitude slower than SHA2."},{"title":"SHA.update!","page":"SHA","location":"stdlib/SHA.html#SHA.update!","category":"function","text":"update!(context, data[, datalen])\n\nUpdate the SHA context with the bytes in data. See also digest! for finalizing the hash.\n\nExamples\n\njulia> ctx = SHA1_CTX()\nSHA1 hash state\n\njulia> update!(ctx, b\"data to to be hashed\")\n\n\n\n\n\n"},{"title":"SHA.digest!","page":"SHA","location":"stdlib/SHA.html#SHA.digest!","category":"function","text":"digest!(context)\n\nFinalize the SHA context and return the hash as array of bytes (Vector{Uint8}). Updating the context after calling digest! on it will error.\n\nExamples\n\njulia> ctx = SHA1_CTX()\nSHA1 hash state\n\njulia> update!(ctx, b\"data to to be hashed\")\n\njulia> digest!(ctx)\n20-element Vector{UInt8}:\n 0x83\n 0xe4\n ⋮\n 0x89\n 0xf5\n\njulia> update!(ctx, b\"more data\")\nERROR: Cannot update CTX after `digest!` has been called on it\n[...]\n\n\n\n\n\n"},{"title":"All SHA context types","page":"SHA","location":"stdlib/SHA.html#All-SHA-context-types","category":"section","text":""},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-1"},{"title":"SHA.SHA1_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA1_CTX","category":"type","text":"SHA1_CTX()\n\nConstruct an empty SHA1 context.\n\n\n\n\n\n"},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-2"},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"Convenience types are also provided, where SHAXXX_CTX is a type alias for SHA2_XXX_CTX."},{"title":"SHA.SHA224_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA224_CTX","category":"type","text":"SHA2_224_CTX()\n\nConstruct an empty SHA2_224 context.\n\n\n\n\n\n"},{"title":"SHA.SHA256_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA256_CTX","category":"type","text":"SHA2_256_CTX()\n\nConstruct an empty SHA2_256 context.\n\n\n\n\n\n"},{"title":"SHA.SHA384_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA384_CTX","category":"type","text":"SHA2_384()\n\nConstruct an empty SHA2_384 context.\n\n\n\n\n\n"},{"title":"SHA.SHA512_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA512_CTX","category":"type","text":"SHA2_512_CTX()\n\nConstruct an empty SHA2_512 context.\n\n\n\n\n\n"},{"title":"SHA.SHA2_224_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA2_224_CTX","category":"type","text":"SHA2_224_CTX()\n\nConstruct an empty SHA2_224 context.\n\n\n\n\n\n"},{"title":"SHA.SHA2_256_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA2_256_CTX","category":"type","text":"SHA2_256_CTX()\n\nConstruct an empty SHA2_256 context.\n\n\n\n\n\n"},{"title":"SHA.SHA2_384_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA2_384_CTX","category":"type","text":"SHA2_384()\n\nConstruct an empty SHA2_384 context.\n\n\n\n\n\n"},{"title":"SHA.SHA2_512_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA2_512_CTX","category":"type","text":"SHA2_512_CTX()\n\nConstruct an empty SHA2_512 context.\n\n\n\n\n\n"},{"title":"SHA.SHA2_512_224_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA2_512_224_CTX","category":"type","text":"SHA2_512_224_CTX()\n\nConstruct an empty SHA2_512/224 context and set the initial hash value.\n\nFor the source of the initial value, refer to FIPS 180-4, 5.3.6.1 SHA-512/224\n\n\n\n\n\n"},{"title":"SHA.SHA2_512_256_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA2_512_256_CTX","category":"type","text":"SHA2_512_256_CTX()\n\nConstruct an empty SHA2_512/256 context and set the initial hash value.\n\nFor the source of the initial value, refer to FIPS 180-4, 5.3.6.2 SHA-512/256\n\n\n\n\n\n"},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-3"},{"title":"SHA.SHA3_224_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA3_224_CTX","category":"type","text":"SHA3_224_CTX()\n\nConstruct an empty SHA3_224 context.\n\n\n\n\n\n"},{"title":"SHA.SHA3_256_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA3_256_CTX","category":"type","text":"SHA3_256_CTX()\n\nConstruct an empty SHA3_256 context.\n\n\n\n\n\n"},{"title":"SHA.SHA3_384_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA3_384_CTX","category":"type","text":"SHA3_384_CTX()\n\nConstruct an empty SHA3_384 context.\n\n\n\n\n\n"},{"title":"SHA.SHA3_512_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA3_512_CTX","category":"type","text":"SHA3_512_CTX()\n\nConstruct an empty SHA3_512 context.\n\n\n\n\n\n"},{"title":"HMAC functions","page":"SHA","location":"stdlib/SHA.html#HMAC-functions","category":"section","text":""},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"julia> using SHA\n\njulia> key = collect(codeunits(\"key_string\"))\n10-element Vector{UInt8}:\n 0x6b\n 0x65\n 0x79\n 0x5f\n 0x73\n 0x74\n 0x72\n 0x69\n 0x6e\n 0x67\n\njulia> bytes2hex(hmac_sha3_256(key, \"test-message\"))\n\"bc49a6f2aa29b27ee5ed1e944edd7f3d153e8a01535d98b5e24dac9a589a6248\""},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"To create a hash from multiple items, the HMAC_CTX() types can be used to create a stateful hash object that is updated with update! and finalized with digest!."},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"julia> using SHA\n\njulia> key = collect(codeunits(\"key_string\"))\n10-element Vector{UInt8}:\n 0x6b\n 0x65\n 0x79\n 0x5f\n 0x73\n 0x74\n 0x72\n 0x69\n 0x6e\n 0x67\n\njulia> ctx = HMAC_CTX(SHA3_256_CTX(), key);\n\njulia> update!(ctx, b\"test-\")\n0x0000000000000000000000000000008d\n\njulia> update!(ctx, b\"message\")\n0x00000000000000000000000000000094\n\njulia> bytes2hex(digest!(ctx))\n\"bc49a6f2aa29b27ee5ed1e944edd7f3d153e8a01535d98b5e24dac9a589a6248\""},{"title":"All HMAC functions","page":"SHA","location":"stdlib/SHA.html#All-HMAC-functions","category":"section","text":""},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"HMAC context type"},{"title":"SHA.HMAC_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.HMAC_CTX","category":"type","text":"HMAC_CTX(ctx::CTX, key::Vector{UInt8}) where {CTX<:SHA_CTX}\n\nConstruct an empty HMAC_CTX context.\n\n\n\n\n\n"},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-1"},{"title":"SHA.hmac_sha1","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha1","category":"function","text":"hmac_sha1(key, data)\n\nHash data using the sha1 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha1(key, io::IO)\n\nHash data from io with the passed key using sha1 algorithm.\n\n\n\n\n\n"},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-2"},{"title":"SHA.hmac_sha224","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha224","category":"function","text":"hmac_sha224(key, data)\n\nHash data using the sha224 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha224(key, io::IO)\n\nHash data from io with the passed key using sha224 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha256","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha256","category":"function","text":"hmac_sha256(key, data)\n\nHash data using the sha256 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha256(key, io::IO)\n\nHash data from io with the passed key using sha256 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha384","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha384","category":"function","text":"hmac_sha384(key, data)\n\nHash data using the sha384 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha384(key, io::IO)\n\nHash data from io with the passed key using sha384 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha512","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha512","category":"function","text":"hmac_sha512(key, data)\n\nHash data using the sha512 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha512(key, io::IO)\n\nHash data from io with the passed key using sha512 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha2_224","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha2_224","category":"function","text":"hmac_sha2_224(key, data)\n\nHash data using the sha2_224 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_224(key, io::IO)\n\nHash data from io with the passed key using sha2_224 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha2_256","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha2_256","category":"function","text":"hmac_sha2_256(key, data)\n\nHash data using the sha2_256 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_256(key, io::IO)\n\nHash data from io with the passed key using sha2_256 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha2_384","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha2_384","category":"function","text":"hmac_sha2_384(key, data)\n\nHash data using the sha2_384 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_384(key, io::IO)\n\nHash data from io with the passed key using sha2_384 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha2_512","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha2_512","category":"function","text":"hmac_sha2_512(key, data)\n\nHash data using the sha2_512 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_512(key, io::IO)\n\nHash data from io with the passed key using sha2_512 algorithm.\n\n\n\n\n\n"},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html","category":"page","text":"SHA-3"},{"title":"SHA.hmac_sha3_224","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha3_224","category":"function","text":"hmac_sha3_224(key, data)\n\nHash data using the sha3_224 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_224(key, io::IO)\n\nHash data from io with the passed key using sha3_224 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha3_256","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha3_256","category":"function","text":"hmac_sha3_256(key, data)\n\nHash data using the sha3_256 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_256(key, io::IO)\n\nHash data from io with the passed key using sha3_256 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha3_384","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha3_384","category":"function","text":"hmac_sha3_384(key, data)\n\nHash data using the sha3_384 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_384(key, io::IO)\n\nHash data from io with the passed key using sha3_384 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha3_512","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha3_512","category":"function","text":"hmac_sha3_512(key, data)\n\nHash data using the sha3_512 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_512(key, io::IO)\n\nHash data from io with the passed key using sha3_512 algorithm.\n\n\n\n\n\n"},{"title":"LibCURL","page":"LibCURL","location":"stdlib/LibCURL.html#LibCURL","category":"section","text":""},{"title":"LibCURL","page":"LibCURL","location":"stdlib/LibCURL.html","category":"page","text":"This is a simple Julia wrapper around http://curl.haxx.se/libcurl/ generated using Clang.jl.  Please see the libcurl API documentation for help on how to use this package."},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html#JIT-Design-and-Implementation","category":"section","text":""},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"This document explains the design and implementation of Julia's JIT, after codegen has finished and unoptimized LLVM IR has been produced. The JIT is responsible for optimizing and compiling this IR to machine code, and for linking it into the current process and making the code available for execution."},{"title":"Introduction","page":"JIT Design and Implementation","location":"devdocs/jit.html#Introduction","category":"section","text":""},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"The JIT is responsible for managing compilation resources, looking up previously compiled code, and compiling new code. It is primarily built on LLVM's On-Request-Compilation (ORCv2) technology, which provides support for a number of useful features such as concurrent compilation, lazy compilation, and the ability to compile code in a separate process. Though LLVM provides a basic JIT compiler in the form of LLJIT, Julia uses many ORCv2 APIs directly to create its own custom JIT compiler."},{"title":"Overview","page":"JIT Design and Implementation","location":"devdocs/jit.html#Overview","category":"section","text":""},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"(Image: Diagram of the compiler flow)"},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"Codegen produces an LLVM module containing IR for one or more Julia functions from the original Julia SSA IR produced by type inference (labeled as translate on the compiler diagram above). It also produces a mapping of code-instance to LLVM function name. However, though some optimizations have been applied by the Julia-based compiler on Julia IR, the LLVM IR produced by codegen still contains many opportunities for optimization. Thus, the first step the JIT takes is to run a target-independent optimization pipeline[tdp] on the LLVM module. Then, the JIT runs a target-dependent optimization pipeline, which includes target-specific optimizations and code generation, and outputs an object file. Finally, the JIT links the resulting object file into the current process and makes the code available for execution. All of this is controlled by code in src/jitlayers.cpp."},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"[tdp]: This is not a totally-target independent pipeline, as transformations such as vectorization rely upon target information such as vector register width and cost modeling. Additionally, codegen itself makes a few target-dependent assumptions, and the optimization pipeline will take advantage of that knowledge."},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"Currently, only one thread at a time is permitted to enter the optimize-compile-link pipeline at a time, due to restrictions imposed by one of our linkers (RuntimeDyld). However, the JIT is designed to support concurrent optimization and compilation, and the linker restriction is expected to be lifted in the future when RuntimeDyld has been fully superseded on all platforms."},{"title":"Optimization Pipeline","page":"JIT Design and Implementation","location":"devdocs/jit.html#Optimization-Pipeline","category":"section","text":""},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"The optimization pipeline is based off LLVM's new pass manager, but the pipeline is customized for Julia's needs. The pipeline is defined in src/pipeline.cpp, and broadly proceeds through a number of stages as detailed below."},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"Early Simplification\nThese passes are mainly used to simplify the IR and canonicalize patterns so that later passes can identify those patterns more easily. Additionally, various intrinsic calls such as branch prediction hints and annotations are lowered into other metadata or other IR features. SimplifyCFG (simplify control flow graph), DCE (dead code elimination), and SROA (scalar replacement of aggregates) are some of the key players here.\nEarly Optimization\nThese passes are typically cheap and are primarily focused around reducing the number of instructions in the IR and propagating knowledge to other instructions. For example, EarlyCSE is used to perform common subexpression elimination, and InstCombine and InstSimplify perform a number of small peephole optimizations to make operations less expensive.\nLoop Optimization\nThese passes canonicalize and simplify loops. Loops are often hot code, which makes loop optimization extremely important for performance. Key players here include LoopRotate, LICM, and LoopFullUnroll. Some bounds check elimination also happens here, as a result of the IRCE pass which can prove certain bounds are never exceeded.\nScalar Optimization\nThe scalar optimization pipeline contains a number of more expensive, but more powerful passes such as GVN (global value numbering), SCCP (sparse conditional constant propagation), and another round of bounds check elimination. These passes are expensive, but they can often remove large amounts of code and make vectorization much more successful and effective. Several other simplification and optimization passes intersperse the more expensive ones to reduce the amount of work they have to do.\nVectorization\nAutomatic vectorization is an extremely powerful transformation for CPU-intensive code. Briefly, vectorization allows execution of a single instruction on multiple data (SIMD), e.g. performing 8 addition operations at the same time. However, proving code to be both capable of vectorization and profitable to vectorize is difficult, and this relies heavily on the prior optimization passes to massage the IR into a state where vectorization is worth it.\nIntrinsic Lowering\nJulia inserts a number of custom intrinsics, for reasons such as object allocation, garbage collection, and exception handling. These intrinsics were originally placed to make optimization opportunities more obvious, but they are now lowered into LLVM IR to enable the IR to be emitted as machine code.\nCleanup\nThese passes are last-chance optimizations, and perform small optimizations such as fused multiply-add propagation and division-remainder simplification. Additionally, targets that do not support half-precision floating point numbers will have their half-precision instructions lowered into single-precision instructions here, and passes are added to provide sanitizer support."},{"title":"Target-Dependent Optimization and Code Generation","page":"JIT Design and Implementation","location":"devdocs/jit.html#Target-Dependent-Optimization-and-Code-Generation","category":"section","text":""},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"LLVM provides target-dependent optimization and machine code generation in the same pipeline, located in the TargetMachine for a given platform. These passes include instruction selection, instruction scheduling, register allocation, and machine code emission. The LLVM documentation provides a good overview of the process, and the LLVM source code is the best place to look for details on the pipeline and passes."},{"title":"Linking","page":"JIT Design and Implementation","location":"devdocs/jit.html#Linking","category":"section","text":""},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"Currently, Julia is transitioning between two linkers: the older RuntimeDyld linker, and the newer JITLink linker. JITLink contains a number of features that RuntimeDyld does not have, such as concurrent and reentrant linking, but currently lacks good support for profiling integrations and does not yet support all of the platforms that RuntimeDyld supports. Over time, JITLink is expected to replace RuntimeDyld entirely. Further details on JITLink can be found in the LLVM documentation."},{"title":"Execution","page":"JIT Design and Implementation","location":"devdocs/jit.html#Execution","category":"section","text":""},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"Once the code has been linked into the current process, it is available for execution. This fact is made known to the generating codeinst by updating the invoke, specsigflags, and specptr fields appropriately. Codeinsts support upgrading invoke, specsigflags, and specptr fields, so long as every combination of these fields that exists at any given point in time is valid to be called. This allows the JIT to update these fields without invalidating existing codeinsts, supporting a potential future concurrent JIT. Specifically, the following states may be valid:"},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"invoke is NULL, specsigflags is 0b00, specptr is NULL\nThis is the initial state of a codeinst, and indicates that the codeinst has not yet been compiled.\ninvoke is non-null, specsigflags is 0b00, specptr is NULL\nThis indicates that the codeinst was not compiled with any specialization, and that the codeinst should be invoked directly. Note that in this instance, invoke does not read either the specsigflags or specptr fields, and therefore they may be modified without invalidating the invoke pointer.\ninvoke is non-null, specsigflags is 0b10, specptr is non-null\nThis indicates that the codeinst was compiled, but a specialized function signature was deemed unnecessary by codegen.\ninvoke is non-null, specsigflags is 0b11, specptr is non-null\nThis indicates that the codeinst was compiled, and a specialized function signature was deemed necessary by codegen. The specptr field contains a pointer to the specialized function signature. The invoke pointer is permitted to read both specsigflags and specptr fields."},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"In addition, there are a number of different transitional states that occur during the update process. To account for these potential situations, the following write and read patterns should be used when dealing with these codeinst fields."},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"When writing invoke, specsigflags, and specptr:\nPerform an atomic compare-exchange operation of specptr assuming the old value was NULL. This compare-exchange operation should have at least acquire-release ordering, to provide ordering guarantees of the remaining memory operations in the write.\nIf specptr was non-null, cease the write operation and wait for bit 0b10 of specsigflags to be written, then restart from step 1 if desired.\nWrite the new low bit of specsigflags to its final value. This may be a relaxed write.\nWrite the new invoke pointer to its final value. This must have at least a release memory ordering to synchronize with reads of invoke.\nSet the second bit of specsigflags to 1. This must be at least a release memory ordering to synchronize with reads of specsigflags. This step completes the write operation and announces to all other threads that all fields have been set.\nWhen reading all of invoke, specsigflags, and specptr:\nRead the specptr field with any memory ordering.\nRead the invoke field with at least an acquire memory ordering. This load will be referred to as initial_invoke.\nIf initial_invoke is NULL, the codeinst is not yet executable. invoke is NULL, specsigflags may be treated as 0b00, specptr may be treated as NULL.\nIf specptr is NULL, then the initial_invoke pointer must not be relying on specptr to guarantee correct execution. Therefore, invoke is non-null, specsigflags may be treated as 0b00, specptr may be treated as NULL.\nIf specptr is non-null, then initial_invoke might not be the final invoke field that uses specptr. This can occur if specptr has been written, but invoke has not yet been written. Therefore, spin on the second bit of specsigflags until it is set to 1 with at least acquire memory ordering.\nRe-read the invoke field with any memory ordering. This load will be referred to as final_invoke.\nRead the specsigflags field with any memory ordering.\ninvoke is final_invoke, specsigflags is the value read in step 7, specptr is the value read in step 3.\nWhen updating a specptr to a different but equivalent function pointer:\nPerform a release store of the new function pointer to specptr. Races here must be benign, as the old function pointer is required to still be valid, and any new ones are also required to be valid as well. Once a pointer has been written to specptr, it must always be callable whether or not it is later overwritten."},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"Correctly reading these fields is implemented in jl_read_codeinst_invoke."},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html","category":"page","text":"Although these write, read, and update steps are complicated, they ensure that the JIT can update codeinsts without invalidating existing codeinsts, and that the JIT can update codeinsts without invalidating existing invoke pointers. This allows the JIT to potentially reoptimize functions at higher optimization levels in the future, and also will allow the JIT to support concurrent compilation of functions in the future."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html#Working-with-LLVM","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"This is not a replacement for the LLVM documentation, but a collection of tips for working on LLVM for Julia."},{"title":"Overview of Julia to LLVM Interface","page":"Working with LLVM","location":"devdocs/llvm.html#Overview-of-Julia-to-LLVM-Interface","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"Julia dynamically links against LLVM by default. Build with USE_LLVM_SHLIB=0 to link statically."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"The code for lowering Julia AST to LLVM IR or interpreting it directly is in directory src/."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"File Description\naotcompile.cpp Compiler C-interface entry and object file emission\nbuiltins.c Builtin functions\nccall.cpp Lowering ccall\ncgutils.cpp Lowering utilities, notably for array and tuple accesses\ncodegen.cpp Top-level of code generation, pass list, lowering builtins\ndebuginfo.cpp Tracks debug information for JIT code\ndisasm.cpp Handles native object file and JIT code disassembly\ngf.c Generic functions\nintrinsics.cpp Lowering intrinsics\njitlayers.cpp JIT-specific code, ORC compilation layers/utilities\nllvm-alloc-helpers.cpp Julia-specific escape analysis\nllvm-alloc-opt.cpp Custom LLVM pass to demote heap allocations to the stack\nllvm-cpufeatures.cpp Custom LLVM pass to lower CPU-based functions (e.g. haveFMA)\nllvm-demote-float16.cpp Custom LLVM pass to lower 16b float ops to 32b float ops\nllvm-final-gc-lowering.cpp Custom LLVM pass to lower GC calls to their final form\nllvm-gc-invariant-verifier.cpp Custom LLVM pass to verify Julia GC invariants\nllvm-julia-licm.cpp Custom LLVM pass to hoist/sink Julia-specific intrinsics\nllvm-late-gc-lowering.cpp Custom LLVM pass to root GC-tracked values\nllvm-multiversioning.cpp Custom LLVM pass to generate sysimg code on multiple architectures\nllvm-propagate-addrspaces.cpp Custom LLVM pass to canonicalize addrspaces\nllvm-ptls.cpp Custom LLVM pass to lower TLS operations\nllvm-remove-addrspaces.cpp Custom LLVM pass to remove Julia addrspaces\nllvm-remove-ni.cpp Custom LLVM pass to remove Julia non-integral addrspaces\nllvm-simdloop.cpp Custom LLVM pass for @simd\npipeline.cpp New pass manager pipeline, pass pipeline parsing\nsys.c I/O and operating system utility functions"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"Some of the .cpp files form a group that compile to a single object."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"The difference between an intrinsic and a builtin is that a builtin is a first class function that can be used like any other Julia function. An intrinsic can operate only on unboxed data, and therefore its arguments must be statically typed."},{"title":"Alias Analysis","page":"Working with LLVM","location":"devdocs/llvm.html#LLVM-Alias-Analysis","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"Julia currently uses LLVM's Type Based Alias Analysis. To find the comments that document the inclusion relationships, look for static MDNode* in src/codegen.cpp."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"The -O option enables LLVM's Basic Alias Analysis."},{"title":"Building Julia with a different version of LLVM","page":"Working with LLVM","location":"devdocs/llvm.html#Building-Julia-with-a-different-version-of-LLVM","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"The default version of LLVM is specified in deps/llvm.version. You can override it by creating a file called Make.user in the top-level directory and adding a line to it such as:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"LLVM_VER = 13.0.0"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"Besides the LLVM release numerals, you can also use DEPS_GIT = llvm in combination with USE_BINARYBUILDER_LLVM = 0 to build against the latest development version of LLVM."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"You can also specify to build a debug version of LLVM, by setting either LLVM_DEBUG = 1 or LLVM_DEBUG = Release in your Make.user file. The former will be a fully unoptimized build of LLVM and the latter will produce an optimized build of LLVM. Depending on your needs the latter will suffice and it quite a bit faster. If you use LLVM_DEBUG = Release you will also want to set LLVM_ASSERTIONS = 1 to enable diagnostics for different passes. Only LLVM_DEBUG = 1 implies that option by default."},{"title":"Passing options to LLVM","page":"Working with LLVM","location":"devdocs/llvm.html#Passing-options-to-LLVM","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"You can pass options to LLVM via the environment variable JULIA_LLVM_ARGS. Here are example settings using bash syntax:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"export JULIA_LLVM_ARGS=-print-after-all dumps IR after each pass.\nexport JULIA_LLVM_ARGS=-debug-only=loop-vectorize dumps LLVM DEBUG(...) diagnostics for loop vectorizer. If you get warnings about \"Unknown command line argument\", rebuild LLVM with LLVM_ASSERTIONS = 1.\nexport JULIA_LLVM_ARGS=-help shows a list of available options. export JULIA_LLVM_ARGS=-help-hidden shows even more.\nexport JULIA_LLVM_ARGS=\"-fatal-warnings -print-options\" is an example how to use multiple options."},{"title":"Useful JULIA_LLVM_ARGS parameters","page":"Working with LLVM","location":"devdocs/llvm.html#Useful-JULIA_LLVM_ARGS-parameters","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"-print-after=PASS: prints the IR after any execution of PASS, useful for checking changes done by a pass.\n-print-before=PASS: prints the IR before any execution of PASS, useful for checking the input to a pass.\n-print-changed: prints the IR whenever a pass changes the IR, useful for narrowing down which passes are causing problems.\n-print-(before|after)=MARKER-PASS: the Julia pipeline ships with a number of marker passes in the pipeline, which can be used to identify where problems or optimizations are occurring. A marker pass is defined as a pass which appears once in the pipeline and performs no transformations on the IR, and is only useful for targeting print-before/print-after. Currently, the following marker passes exist in the pipeline:\nBeforeOptimization\nBeforeEarlySimplification\nAfterEarlySimplification\nBeforeEarlyOptimization\nAfterEarlyOptimization\nBeforeLoopOptimization\nBeforeLICM\nAfterLICM\nBeforeLoopSimplification\nAfterLoopSimplification\nAfterLoopOptimization\nBeforeScalarOptimization\nAfterScalarOptimization\nBeforeVectorization\nAfterVectorization\nBeforeIntrinsicLowering\nAfterIntrinsicLowering\nBeforeCleanup\nAfterCleanup\nAfterOptimization\n-time-passes: prints the time spent in each pass, useful for identifying which passes are taking a long time.\n-print-module-scope: used in conjunction with -print-(before|after), gets the entire module rather than the IR unit received by the pass\n-debug: prints out a lot of debugging information throughout LLVM\n-debug-only=NAME, prints out debugging statements from files with DEBUG_TYPE defined to NAME, useful for getting additional context about a problem"},{"title":"Debugging LLVM transformations in isolation","page":"Working with LLVM","location":"devdocs/llvm.html#Debugging-LLVM-transformations-in-isolation","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"On occasion, it can be useful to debug LLVM's transformations in isolation from the rest of the Julia system, e.g. because reproducing the issue inside julia would take too long, or because one wants to take advantage of LLVM's tooling (e.g. bugpoint)."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"To start with, you can install the developer tools to work with LLVM via:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"make -C deps install-llvm-tools"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"To get unoptimized IR for the entire system image, pass the --output-unopt-bc unopt.bc option to the system image build process, which will output the unoptimized IR to an unopt.bc file. This file can then be passed to LLVM tools as usual. libjulia can function as an LLVM pass plugin and can be loaded into LLVM tools, to make julia-specific passes available in this environment. In addition, it exposes the -julia meta-pass, which runs the entire Julia pass-pipeline over the IR. As an example, to generate a system image with the old pass manager, one could do:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"\nllc -o sys.o opt.bc\ncc -shared -o sys.so sys.o"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"To generate a system image with the new pass manager, one could do:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"./usr/tools/opt -load-pass-plugin=libjulia-codegen.so --passes='julia' -o opt.bc unopt.bc\n./usr/tools/llc -o sys.o opt.bc\n./usr/tools/cc -shared -o sys.so sys.o"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"This system image can then be loaded by julia as usual."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"It is also possible to dump an LLVM IR module for just one Julia function, using:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"fun, T = +, Tuple{Int,Int} # Substitute your function of interest here\noptimize = false\nopen(\"plus.ll\", \"w\") do file\n    code_llvm(file, fun, T; raw=true, dump_module=true, optimize)\nend"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"These files can be processed the same way as the unoptimized sysimg IR shown above, or if you want to see the LLVM IR yourself and get extra verification run, you can use"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"./usr/tools/opt -load-pass-plugin=libjulia-codegen.so --passes='julia' -S -verify-each plus.ll"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"(note on MacOS this would be libjulia-codegen.dylib and on Windows libjulia-codegen.dll)"},{"title":"Running the LLVM test suite","page":"Working with LLVM","location":"devdocs/llvm.html#Running-the-LLVM-test-suite","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"To run the llvm tests locally, you need to first install the tools, build julia, then you can run the tests:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"make -C deps install-llvm-tools\nmake -j julia-src-release\nmake -C test/llvmpasses"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"If you want to run the individual test files directly, via the commands at the top of each test file, the first step here will have installed the tools into ./usr/tools/opt. Then you'll want to manually replace %s with the name of the test file."},{"title":"Improving LLVM optimizations for Julia","page":"Working with LLVM","location":"devdocs/llvm.html#Improving-LLVM-optimizations-for-Julia","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"Improving LLVM code generation usually involves either changing Julia lowering to be more friendly to LLVM's passes, or improving a pass."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"If you are planning to improve a pass, be sure to read the LLVM developer policy. The best strategy is to create a code example in a form where you can use LLVM's opt tool to study it and the pass of interest in isolation."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"Create an example Julia code of interest.\nUse JULIA_LLVM_ARGS=-print-after-all to dump the IR.\nPick out the IR at the point just before the pass of interest runs.\nStrip the debug metadata and fix up the TBAA metadata by hand."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"The last step is labor intensive. Suggestions on a better way would be appreciated."},{"title":"The jlcall calling convention","page":"Working with LLVM","location":"devdocs/llvm.html#The-jlcall-calling-convention","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"Julia has a generic calling convention for unoptimized code, which looks somewhat as follows:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"jl_value_t *any_unoptimized_call(jl_value_t *, jl_value_t **, int);"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"where the first argument is the boxed function object, the second argument is an on-stack array of arguments and the third is the number of arguments. Now, we could perform a straightforward lowering and emit an alloca for the argument array. However, this would betray the SSA nature of the uses at the call site, making optimizations (including GC root placement), significantly harder. Instead, we emit it as follows:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"call %jl_value_t *@julia.call(jl_value_t *(*)(...) @any_unoptimized_call, %jl_value_t *%arg1, %jl_value_t *%arg2)"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"This allows us to retain the SSA-ness of the uses throughout the optimizer. GC root placement will later lower this call to the original C ABI."},{"title":"GC root placement","page":"Working with LLVM","location":"devdocs/llvm.html#GC-root-placement","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"GC root placement is done by an LLVM pass late in the pass pipeline. Doing GC root placement this late enables LLVM to make more aggressive optimizations around code that requires GC roots, as well as allowing us to reduce the number of required GC roots and GC root store operations (since LLVM doesn't understand our GC, it wouldn't otherwise know what it is and is not allowed to do with values stored to the GC frame, so it'll conservatively do very little). As an example, consider an error path"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"if some_condition()\n    #= Use some variables maybe =#\n    error(\"An error occurred\")\nend"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"During constant folding, LLVM may discover that the condition is always false, and can remove the basic block. However, if GC root lowering is done early, the GC root slots used in the deleted block, as well as any values kept alive in those slots only because they were used in the error path, would be kept alive by LLVM. By doing GC root lowering late, we give LLVM the license to do any of its usual optimizations (constant folding, dead code elimination, etc.), without having to worry (too much) about which values may or may not be GC tracked."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"However, in order to be able to do late GC root placement, we need to be able to identify a) which pointers are GC tracked and b) all uses of such pointers. The goal of the GC placement pass is thus simple:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"Minimize the number of needed GC roots/stores to them subject to the constraint that at every safepoint, any live GC-tracked pointer (i.e. for which there is a path after this point that contains a use of this pointer) is in some GC slot."},{"title":"Representation","page":"Working with LLVM","location":"devdocs/llvm.html#Representation","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"The primary difficulty is thus choosing an IR representation that allows us to identify GC-tracked pointers and their uses, even after the program has been run through the optimizer. Our design makes use of three LLVM features to achieve this:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"Custom address spaces\nOperand Bundles\nNon-integral pointers"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"Custom address spaces allow us to tag every point with an integer that needs to be preserved through optimizations. The compiler may not insert casts between address spaces that did not exist in the original program and it must never change the address space of a pointer on a load/store/etc operation. This allows us to annotate which pointers are GC-tracked in an optimizer-resistant way. Note that metadata would not be able to achieve the same purpose. Metadata is supposed to always be discardable without altering the semantics of the program. However, failing to identify a GC-tracked pointer alters the resulting program behavior dramatically - it'll probably crash or return wrong results. We currently use three different address spaces (their numbers are defined in src/codegen_shared.cpp):"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"GC Tracked Pointers (currently 10): These are pointers to boxed values that may be put into a GC frame. It is loosely equivalent to a jl_value_t* pointer on the C side. N.B. It is illegal to ever have a pointer in this address space that may not be stored to a GC slot.\nDerived Pointers (currently 11): These are pointers that are derived from some GC tracked pointer. Uses of these pointers generate uses of the original pointer. However, they need not themselves be known to the GC. The GC root placement pass MUST always find the GC tracked pointer from which this pointer is derived and use that as the pointer to root.\nCallee Rooted Pointers (currently 12): This is a utility address space to express the notion of a callee rooted value. All values of this address space MUST be storable to a GC root (though it is possible to relax this condition in the future), but unlike the other pointers need not be rooted if passed to a call (they do still need to be rooted if they are live across another safepoint between the definition and the call).\nPointers loaded from tracked object (currently 13): This is used by arrays, which themselves contain a pointer to the managed data. This data area is owned by the array, but is not a GC-tracked object by itself. The compiler guarantees that as long as this pointer is live, the object that this pointer was loaded from will keep being live."},{"title":"Invariants","page":"Working with LLVM","location":"devdocs/llvm.html#Invariants","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"The GC root placement pass makes use of several invariants, which need to be observed by the frontend and are preserved by the optimizer."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"First, only the following address space casts are allowed:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"0->{Tracked,Derived,CalleeRooted}: It is allowable to decay an untracked pointer to any of the others. However, do note that the optimizer has broad license to not root such a value. It is never safe to have a value in address space 0 in any part of the program if it is (or is derived from) a value that requires a GC root.\nTracked->Derived: This is the standard decay route for interior values. The placement pass will look for these to identify the base pointer for any use.\nTracked->CalleeRooted: Addrspace CalleeRooted serves merely as a hint that a GC root is not required. However, do note that the Derived->CalleeRooted decay is prohibited, since pointers should generally be storable to a GC slot, even in this address space."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"Now let us consider what constitutes a use:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"Loads whose loaded values is in one of the address spaces\nStores of a value in one of the address spaces to a location\nStores to a pointer in one of the address spaces\nCalls for which a value in one of the address spaces is an operand\nCalls in jlcall ABI, for which the argument array contains a value\nReturn instructions."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"We explicitly allow load/stores and simple calls in address spaces Tracked/Derived. Elements of jlcall argument arrays must always be in address space Tracked (it is required by the ABI that they are valid jl_value_t* pointers). The same is true for return instructions (though note that struct return arguments are allowed to have any of the address spaces). The only allowable use of an address space CalleeRooted pointer is to pass it to a call (which must have an appropriately typed operand)."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"Further, we disallow getelementptr in addrspace Tracked. This is because unless the operation is a noop, the resulting pointer will not be validly storable to a GC slot and may thus not be in this address space. If such a pointer is required, it should be decayed to addrspace Derived first."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"Lastly, we disallow inttoptr/ptrtoint instructions in these address spaces. Having these instructions would mean that some i64 values are really GC tracked. This is problematic, because it breaks that stated requirement that we're able to identify GC-relevant pointers. This invariant is accomplished using the LLVM \"non-integral pointers\" feature, which is new in LLVM 5.0. It prohibits the optimizer from making optimizations that would introduce these operations. Note we can still insert static constants at JIT time by using inttoptr in address space 0 and then decaying to the appropriate address space afterwards."},{"title":"Supporting ccall","page":"Working with LLVM","location":"devdocs/llvm.html#Supporting-[ccall](@ref)","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"One important aspect missing from the discussion so far is the handling of ccall. ccall has the peculiar feature that the location and scope of a use do not coincide. As an example consider:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"A = randn(1024)\nccall(:foo, Cvoid, (Ptr{Float64},), A)"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"In lowering, the compiler will insert a conversion from the array to the pointer which drops the reference to the array value. However, we of course need to make sure that the array does stay alive while we're doing the ccall. To understand how this is done, lets look at a hypothetical approximate possible lowering of the above code:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"return $(Expr(:foreigncall, Expr(:tuple, :(:foo)), Cvoid, svec(Ptr{Float64}), 0, :(:ccall), Expr(:foreigncall, Expr(:tuple, :(:jl_array_ptr)), Ptr{Float64}, svec(Any), 0, :(:ccall), :(A)), :(A)))"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"The last :(A), is an extra argument list inserted during lowering that informs the code generator which Julia level values need to be kept alive for the duration of this ccall. We then take this information and represent it in an \"operand bundle\" at the IR level. An operand bundle is essentially a fake use that is attached to the call site. At the IR level, this looks like so:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"call void inttoptr (i64 ... to void (double*)*)(double* %5) [ \"jl_roots\"(%jl_value_t addrspace(10)* %A) ]"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"The GC root placement pass will treat the jl_roots operand bundle as if it were a regular operand. However, as a final step, after the GC roots are inserted, it will drop the operand bundle to avoid confusing instruction selection."},{"title":"Supporting pointer_from_objref","page":"Working with LLVM","location":"devdocs/llvm.html#Supporting-[pointer_from_objref](@ref)","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"pointer_from_objref is special because it requires the user to take explicit control of GC rooting. By our above invariants, this function is illegal, because it performs an address space cast from 10 to 0. However, it can be useful, in certain situations, so we provide a special intrinsic:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"declared %jl_value_t *julia.pointer_from_objref(%jl_value_t addrspace(10)*)"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"which is lowered to the corresponding address space cast after GC root lowering. Do note however that by using this intrinsic, the caller assumes all responsibility for making sure that the value in question is rooted. Further this intrinsic is not considered a use, so the GC root placement pass will not provide a GC root for the function. As a result, the external rooting must be arranged while the value is still tracked by the system. I.e. it is not valid to attempt to use the result of this operation to establish a global root - the optimizer may have already dropped the value."},{"title":"Keeping values alive in the absence of uses","page":"Working with LLVM","location":"devdocs/llvm.html#Keeping-values-alive-in-the-absence-of-uses","category":"section","text":""},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"In certain cases it is necessary to keep an object alive, even though there is no compiler-visible use of said object. This may be case for low level code that operates on the memory-representation of an object directly or code that needs to interface with C code. In order to allow this, we provide the following intrinsics at the LLVM level:"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"token @llvm.julia.gc_preserve_begin(...)\nvoid @llvm.julia.gc_preserve_end(token)"},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html","category":"page","text":"(The llvm. in the name is required in order to be able to use the token type). The semantics of these intrinsics are as follows: At any safepoint that is dominated by a gc_preserve_begin call, but that is not not dominated by a corresponding gc_preserve_end call (i.e. a call whose argument is the token returned by a gc_preserve_begin call), the values passed as arguments to that gc_preserve_begin will be kept live. Note that the gc_preserve_begin still counts as a regular use of those values, so the standard lifetime semantics will ensure that the values will be kept alive before entering the preserve region."},{"title":"Printf","page":"Printf","location":"stdlib/Printf.html#man-printf","category":"section","text":""},{"title":"Printf","page":"Printf","location":"stdlib/Printf.html","category":"page","text":"The Printf module provides formatted output functions similar to the C standard library's printf. It allows formatted printing to an output stream or to a string."},{"title":"Printf.@printf","page":"Printf","location":"stdlib/Printf.html#Printf.@printf","category":"macro","text":"@printf([io::IO], \"%Fmt\", args...)\n\nPrint args using C printf style format specification string. Optionally, an IO may be passed as the first argument to redirect output.\n\nExamples\n\njulia> @printf \"Hello %s\" \"world\"\nHello world\n\njulia> @printf \"Scientific notation %e\" 1.234\nScientific notation 1.234000e+00\n\njulia> @printf \"Scientific notation three digits %.3e\" 1.23456\nScientific notation three digits 1.235e+00\n\njulia> @printf \"Decimal two digits %.2f\" 1.23456\nDecimal two digits 1.23\n\njulia> @printf \"Padded to length 5 %5i\" 123\nPadded to length 5   123\n\njulia> @printf \"Padded with zeros to length 6 %06i\" 123\nPadded with zeros to length 6 000123\n\njulia> @printf \"Use shorter of decimal or scientific %g %g\" 1.23 12300000.0\nUse shorter of decimal or scientific 1.23 1.23e+07\n\njulia> @printf \"Use dynamic width and precision  %*.*f\" 10 2 0.12345\nUse dynamic width and precision        0.12\n\nFor a systematic specification of the format, see here. See also @sprintf to get the result as a String instead of it being printed.\n\nCaveats\n\nInf and NaN are printed consistently as Inf and NaN for flags %a, %A, %e, %E, %f, %F, %g, and %G. Furthermore, if a floating point number is equally close to the numeric values of two possible output strings, the output string further away from zero is chosen.\n\nExamples\n\njulia> @printf(\"%f %F %f %F\", Inf, Inf, NaN, NaN)\nInf Inf NaN NaN\n\njulia> @printf \"%.0f %.1f %f\" 0.5 0.025 -0.0078125\n0 0.0 -0.007812\n\ncompat: Julia 1.8\nStarting in Julia 1.8, %s (string) and %c (character) widths are computed using textwidth, which e.g. ignores zero-width characters (such as combining characters for diacritical marks) and treats certain \"wide\" characters (e.g. emoji) as width 2.\n\ncompat: Julia 1.10\nDynamic width specifiers like %*s and %0*.*f require Julia 1.10.\n\n\n\n\n\n"},{"title":"Printf.@sprintf","page":"Printf","location":"stdlib/Printf.html#Printf.@sprintf","category":"macro","text":"@sprintf(\"%Fmt\", args...)\n\nReturn @printf formatted output as string.\n\nExamples\n\njulia> @sprintf \"this is a %s %15.1f\" \"test\" 34.567\n\"this is a test            34.6\"\n\n\n\n\n\n"},{"title":"Printf.Format","page":"Printf","location":"stdlib/Printf.html#Printf.Format","category":"type","text":"Printf.Format(format_str)\n\nCreate a C printf-compatible format object that can be used for formatting values.\n\nThe input format_str can include any valid format specifier character and modifiers.\n\nA Format object can be passed to Printf.format(f::Format, args...) to produce a formatted string, or Printf.format(io::IO, f::Format, args...) to print the formatted string directly to io.\n\nFor convenience, the Printf.format\"...\" string macro form can be used for building a Printf.Format object at macro-expansion-time.\n\ncompat: Julia 1.6\nPrintf.Format requires Julia 1.6 or later.\n\n\n\n\n\n"},{"title":"Printf.format","page":"Printf","location":"stdlib/Printf.html#Printf.format","category":"function","text":"Printf.format(f::Printf.Format, args...) => String\nPrintf.format(io::IO, f::Printf.Format, args...)\n\nApply a printf format object f to provided args and return the formatted string (1st method), or print directly to an io object (2nd method). See @printf for more details on C printf support.\n\n\n\n\n\n"},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#Sparse-Arrays","category":"section","text":""},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"Julia has support for sparse vectors and sparse matrices in the SparseArrays stdlib module. Sparse arrays are arrays that contain enough zeros that storing them in a special data structure leads to savings in space and execution time, compared to dense arrays."},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"External packages which implement different sparse storage types, multidimensional sparse arrays, and more can be found in Noteworthy External Sparse Packages"},{"title":"Compressed Sparse Column (CSC) Sparse Matrix Storage","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#man-csc","category":"section","text":""},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"In Julia, sparse matrices are stored in the Compressed Sparse Column (CSC) format. Julia sparse matrices have the type SparseMatrixCSC{Tv,Ti}, where Tv is the type of the stored values, and Ti is the integer type for storing column pointers and row indices. The internal representation of SparseMatrixCSC is as follows:"},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"struct SparseMatrixCSC{Tv,Ti<:Integer} <: AbstractSparseMatrixCSC{Tv,Ti}\n    m::Int                  # Number of rows\n    n::Int                  # Number of columns\n    colptr::Vector{Ti}      # Column j is in colptr[j]:(colptr[j+1]-1)\n    rowval::Vector{Ti}      # Row indices of stored values\n    nzval::Vector{Tv}       # Stored values, typically nonzeros\nend"},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"The compressed sparse column storage makes it easy and quick to access the elements in the column of a sparse matrix, whereas accessing the sparse matrix by rows is considerably slower. Operations such as insertion of previously unstored entries one at a time in the CSC structure tend to be slow. This is because all elements of the sparse matrix that are beyond the point of insertion have to be moved one place over."},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"All operations on sparse matrices are carefully implemented to exploit the CSC data structure for performance, and to avoid expensive operations."},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"If you have data in CSC format from a different application or library, and wish to import it in Julia, make sure that you use 1-based indexing. The row indices in every column need to be sorted, and if they are not, the matrix will display incorrectly.  If your SparseMatrixCSC object contains unsorted row indices, one quick way to sort them is by doing a double transpose. Since the transpose operation is lazy, make a copy to materialize each transpose."},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"In some applications, it is convenient to store explicit zero values in a SparseMatrixCSC. These are accepted by functions in Base (but there is no guarantee that they will be preserved in mutating operations). Such explicitly stored zeros are treated as structural nonzeros by many routines. The nnz function returns the number of elements explicitly stored in the sparse data structure, including non-structural zeros. In order to count the exact number of numerical nonzeros, use count(!iszero, x), which inspects every stored element of a sparse matrix. dropzeros, and the in-place dropzeros!, can be used to remove stored zeros from the sparse matrix."},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"julia> A = sparse([1, 1, 2, 3], [1, 3, 2, 3], [0, 1, 2, 0])\n3×3 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n 0  ⋅  1\n ⋅  2  ⋅\n ⋅  ⋅  0\n\njulia> dropzeros(A)\n3×3 SparseMatrixCSC{Int64, Int64} with 2 stored entries:\n ⋅  ⋅  1\n ⋅  2  ⋅\n ⋅  ⋅  ⋅"},{"title":"Sparse Vector Storage","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#Sparse-Vector-Storage","category":"section","text":""},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"Sparse vectors are stored in a close analog to compressed sparse column format for sparse matrices. In Julia, sparse vectors have the type SparseVector{Tv,Ti} where Tv is the type of the stored values and Ti the integer type for the indices. The internal representation is as follows:"},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"struct SparseVector{Tv,Ti<:Integer} <: AbstractSparseVector{Tv,Ti}\n    n::Int              # Length of the sparse vector\n    nzind::Vector{Ti}   # Indices of stored values\n    nzval::Vector{Tv}   # Stored values, typically nonzeros\nend"},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"Like SparseMatrixCSC, the SparseVector type can also contain explicitly stored zeros. (See Sparse Matrix Storage.)."},{"title":"Sparse Vector and Matrix Constructors","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#Sparse-Vector-and-Matrix-Constructors","category":"section","text":""},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"The simplest way to create a sparse array is to use a function equivalent to the zeros function that Julia provides for working with dense arrays. To produce a sparse array instead, you can use the same name with an sp prefix:"},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"julia> spzeros(3)\n3-element SparseVector{Float64, Int64} with 0 stored entries"},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"The sparse function is often a handy way to construct sparse arrays. For example, to construct a sparse matrix we can input a vector I of row indices, a vector J of column indices, and a vector V of stored values (this is also known as the COO (coordinate) format). sparse(I,J,V) then constructs a sparse matrix such that S[I[k], J[k]] = V[k]. The equivalent sparse vector constructor is sparsevec, which takes the (row) index vector I and the vector V with the stored values and constructs a sparse vector R such that R[I[k]] = V[k]."},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"julia> I = [1, 4, 3, 5]; J = [4, 7, 18, 9]; V = [1, 2, -5, 3];\n\njulia> S = sparse(I,J,V)\n5×18 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n⎡⠀⠈⠀⠀⠀⠀⠀⠀⢀⎤\n⎣⠀⠀⠀⠂⡀⠀⠀⠀⠀⎦\n\njulia> R = sparsevec(I,V)\n5-element SparseVector{Int64, Int64} with 4 stored entries:\n  [1]  =  1\n  [3]  =  -5\n  [4]  =  2\n  [5]  =  3"},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"The inverse of the sparse and sparsevec functions is findnz, which retrieves the inputs used to create the sparse array (including stored entries equal to zero). findall(!iszero, x) returns the Cartesian indices of non-zero entries in x (not including stored entries equal to zero)."},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"julia> findnz(S)\n([1, 4, 5, 3], [4, 7, 9, 18], [1, 2, 3, -5])\n\njulia> findall(!iszero, S)\n4-element Vector{CartesianIndex{2}}:\n CartesianIndex(1, 4)\n CartesianIndex(4, 7)\n CartesianIndex(5, 9)\n CartesianIndex(3, 18)\n\njulia> findnz(R)\n([1, 3, 4, 5], [1, -5, 2, 3])\n\njulia> findall(!iszero, R)\n4-element Vector{Int64}:\n 1\n 3\n 4\n 5"},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"Another way to create a sparse array is to convert a dense array into a sparse array using the sparse function:"},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"julia> sparse(Matrix(1.0I, 5, 5))\n5×5 SparseMatrixCSC{Float64, Int64} with 5 stored entries:\n 1.0   ⋅    ⋅    ⋅    ⋅\n  ⋅   1.0   ⋅    ⋅    ⋅\n  ⋅    ⋅   1.0   ⋅    ⋅\n  ⋅    ⋅    ⋅   1.0   ⋅\n  ⋅    ⋅    ⋅    ⋅   1.0\n\njulia> sparse([1.0, 0.0, 1.0])\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  1.0\n  [3]  =  1.0"},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"You can go in the other direction using the Array constructor. The issparse function can be used to query if a matrix is sparse."},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"julia> issparse(spzeros(5))\ntrue"},{"title":"Sparse matrix operations","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#Sparse-matrix-operations","category":"section","text":""},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"Arithmetic operations on sparse matrices also work as they do on dense matrices. Indexing of, assignment into, and concatenation of sparse matrices work in the same way as dense matrices. Indexing operations, especially assignment, are expensive, when carried out one element at a time. In many cases it may be better to convert the sparse matrix into (I,J,V) format using findnz, manipulate the values or the structure in the dense vectors (I,J,V), and then reconstruct the sparse matrix."},{"title":"Correspondence of dense and sparse methods","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#Correspondence-of-dense-and-sparse-methods","category":"section","text":""},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"The following table gives a correspondence between built-in methods on sparse matrices and their corresponding methods on dense matrix types. In general, methods that generate sparse matrices differ from their dense counterparts in that the resulting matrix follows the same sparsity pattern as a given sparse matrix S, or that the resulting sparse matrix has density d, i.e. each matrix element has a probability d of being non-zero."},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"Details can be found in the Sparse Vectors and Matrices section of the standard library reference."},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"Sparse Dense Description\nspzeros(m,n) zeros(m,n) Creates a m-by-n matrix of zeros. (spzeros(m,n) is empty.)\nsparse(I,n,n) Matrix(I,n,n) Creates a n-by-n identity matrix.\nsparse(A) Array(S) Interconverts between dense and sparse formats.\nsprand(m,n,d) rand(m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements distributed uniformly on the half-open interval 0 1).\nsprandn(m,n,d) randn(m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements distributed according to the standard normal (Gaussian) distribution.\nsprandn(rng,m,n,d) randn(rng,m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements generated with the rng random number generator"},{"title":"SparseArrays API","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#stdlib-sparse-arrays","category":"section","text":""},{"title":"SparseArrays.AbstractSparseArray","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseArray","category":"type","text":"AbstractSparseArray{Tv,Ti,N}\n\nSupertype for N-dimensional sparse arrays (or array-like types) with elements of type Tv and index type Ti. SparseMatrixCSC, SparseVector and SuiteSparse.CHOLMOD.Sparse are subtypes of this.\n\n\n\n\n\n"},{"title":"SparseArrays.AbstractSparseVector","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseVector","category":"type","text":"AbstractSparseVector{Tv,Ti}\n\nSupertype for one-dimensional sparse arrays (or array-like types) with elements of type Tv and index type Ti. Alias for AbstractSparseArray{Tv,Ti,1}.\n\n\n\n\n\n"},{"title":"SparseArrays.AbstractSparseMatrix","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseMatrix","category":"type","text":"AbstractSparseMatrix{Tv,Ti}\n\nSupertype for two-dimensional sparse arrays (or array-like types) with elements of type Tv and index type Ti. Alias for AbstractSparseArray{Tv,Ti,2}.\n\n\n\n\n\n"},{"title":"SparseArrays.SparseVector","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.SparseVector","category":"type","text":"SparseVector{Tv,Ti<:Integer} <: AbstractSparseVector{Tv,Ti}\n\nVector type for storing sparse vectors. Can be created by passing the length of the vector, a sorted vector of non-zero indices, and a vector of non-zero values.\n\nFor instance, the vector [5, 6, 0, 7] can be represented as\n\nSparseVector(4, [1, 2, 4], [5, 6, 7])\n\nThis indicates that the element at index 1 is 5, at index 2 is 6, at index 3 is zero(Int), and at index 4 is 7.\n\nIt may be more convenient to create sparse vectors directly from dense vectors using sparse as\n\nsparse([5, 6, 0, 7])\n\nyields the same sparse vector.\n\n\n\n\n\n"},{"title":"SparseArrays.SparseMatrixCSC","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.SparseMatrixCSC","category":"type","text":"SparseMatrixCSC{Tv,Ti<:Integer} <: AbstractSparseMatrixCSC{Tv,Ti}\n\nMatrix type for storing sparse matrices in the Compressed Sparse Column format. The standard way of constructing SparseMatrixCSC is through the sparse function. See also spzeros, spdiagm and sprand.\n\n\n\n\n\n"},{"title":"SparseArrays.sparse","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sparse","category":"function","text":"sparse(A::Union{AbstractVector, AbstractMatrix})\n\nConvert a vector or matrix A into a sparse array. Numerical zeros in A are turned into structural zeros.\n\nExamples\n\njulia> A = Matrix(1.0I, 3, 3)\n3×3 Matrix{Float64}:\n 1.0  0.0  0.0\n 0.0  1.0  0.0\n 0.0  0.0  1.0\n\njulia> sparse(A)\n3×3 SparseMatrixCSC{Float64, Int64} with 3 stored entries:\n 1.0   ⋅    ⋅\n  ⋅   1.0   ⋅\n  ⋅    ⋅   1.0\n\njulia> [1.0, 0.0, 1.0]\n3-element Vector{Float64}:\n 1.0\n 0.0\n 1.0\n\njulia> sparse([1.0, 0.0, 1.0])\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  1.0\n  [3]  =  1.0\n\n\n\n\n\nsparse(I, J, V,[ m, n, combine])\n\nCreate a sparse matrix S of dimensions m x n such that S[I[k], J[k]] = V[k]. The combine function is used to combine duplicates. If m and n are not specified, they are set to maximum(I) and maximum(J) respectively. If the combine function is not supplied, combine defaults to + unless the elements of V are Booleans in which case combine defaults to |. All elements of I must satisfy 1 <= I[k] <= m, and all elements of J must satisfy 1 <= J[k] <= n. Numerical zeros in (I, J, V) are retained as structural nonzeros; to drop numerical zeros, use dropzeros!.\n\nFor additional documentation and an expert driver, see SparseArrays.sparse!.\n\nExamples\n\njulia> Is = [1; 2; 3];\n\njulia> Js = [1; 2; 3];\n\njulia> Vs = [1; 2; 3];\n\njulia> sparse(Is, Js, Vs)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 1  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  3\n\n\n\n\n\n"},{"title":"SparseArrays.sparse!","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sparse!","category":"function","text":"sparse!(I::AbstractVector{Ti}, J::AbstractVector{Ti}, V::AbstractVector{Tv},\n        m::Integer, n::Integer, combine, klasttouch::Vector{Ti},\n        csrrowptr::Vector{Ti}, csrcolval::Vector{Ti}, csrnzval::Vector{Tv},\n        [csccolptr::Vector{Ti}], [cscrowval::Vector{Ti}, cscnzval::Vector{Tv}] ) where {Tv,Ti<:Integer}\n\nParent of and expert driver for sparse; see sparse for basic usage. This method allows the user to provide preallocated storage for sparse's intermediate objects and result as described below. This capability enables more efficient successive construction of SparseMatrixCSCs from coordinate representations, and also enables extraction of an unsorted-column representation of the result's transpose at no additional cost.\n\nThis method consists of three major steps: (1) Counting-sort the provided coordinate representation into an unsorted-row CSR form including repeated entries. (2) Sweep through the CSR form, simultaneously calculating the desired CSC form's column-pointer array, detecting repeated entries, and repacking the CSR form with repeated entries combined; this stage yields an unsorted-row CSR form with no repeated entries. (3) Counting-sort the preceding CSR form into a fully-sorted CSC form with no repeated entries.\n\nInput arrays csrrowptr, csrcolval, and csrnzval constitute storage for the intermediate CSR forms and require length(csrrowptr) >= m + 1, length(csrcolval) >= length(I), and length(csrnzval >= length(I)). Input array klasttouch, workspace for the second stage, requires length(klasttouch) >= n. Optional input arrays csccolptr, cscrowval, and cscnzval constitute storage for the returned CSC form S. If necessary, these are resized automatically to satisfy length(csccolptr) = n + 1, length(cscrowval) = nnz(S) and length(cscnzval) = nnz(S); hence, if nnz(S) is unknown at the outset, passing in empty vectors of the appropriate type (Vector{Ti}() and Vector{Tv}() respectively) suffices, or calling the sparse! method neglecting cscrowval and cscnzval.\n\nOn return, csrrowptr, csrcolval, and csrnzval contain an unsorted-column representation of the result's transpose.\n\nYou may reuse the input arrays' storage (I, J, V) for the output arrays (csccolptr, cscrowval, cscnzval). For example, you may call sparse!(I, J, V, csrrowptr, csrcolval, csrnzval, I, J, V). Note that they will be resized to satisfy the conditions above.\n\nFor the sake of efficiency, this method performs no argument checking beyond 1 <= I[k] <= m and 1 <= J[k] <= n. Use with care. Testing with --check-bounds=yes is wise.\n\nThis method runs in O(m, n, length(I)) time. The HALFPERM algorithm described in F. Gustavson, \"Two fast algorithms for sparse matrices: multiplication and permuted transposition,\" ACM TOMS 4(3), 250-269 (1978) inspired this method's use of a pair of counting sorts.\n\n\n\n\n\nSparseArrays.sparse!(I, J, V, [m, n, combine]) -> SparseMatrixCSC\n\nVariant of sparse! that re-uses the input vectors (I, J, V) for the final matrix storage. After construction the input vectors will alias the matrix buffers; S.colptr === I, S.rowval === J, and S.nzval === V holds, and they will be resize!d as necessary.\n\nNote that some work buffers will still be allocated. Specifically, this method is a convenience wrapper around sparse!(I, J, V, m, n, combine, klasttouch, csrrowptr, csrcolval, csrnzval, csccolptr, cscrowval, cscnzval) where this method allocates klasttouch, csrrowptr, csrcolval, and csrnzval of appropriate size, but reuses I, J, and V for csccolptr, cscrowval, and cscnzval.\n\nArguments m, n, and combine defaults to maximum(I), maximum(J), and +, respectively.\n\ncompat: Julia 1.10\nThis method requires Julia version 1.10 or later.\n\n\n\n\n\n"},{"title":"SparseArrays.sparsevec","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sparsevec","category":"function","text":"sparsevec(I, V, [m, combine])\n\nCreate a sparse vector S of length m such that S[I[k]] = V[k]. Duplicates are combined using the combine function, which defaults to + if no combine argument is provided, unless the elements of V are Booleans in which case combine defaults to |.\n\nExamples\n\njulia> II = [1, 3, 3, 5]; V = [0.1, 0.2, 0.3, 0.2];\n\njulia> sparsevec(II, V)\n5-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  0.1\n  [3]  =  0.5\n  [5]  =  0.2\n\njulia> sparsevec(II, V, 8, -)\n8-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  0.1\n  [3]  =  -0.1\n  [5]  =  0.2\n\njulia> sparsevec([1, 3, 1, 2, 2], [true, true, false, false, false])\n3-element SparseVector{Bool, Int64} with 3 stored entries:\n  [1]  =  1\n  [2]  =  0\n  [3]  =  1\n\n\n\n\n\nsparsevec(d::Dict, [m])\n\nCreate a sparse vector of length m where the nonzero indices are keys from the dictionary, and the nonzero values are the values from the dictionary.\n\nExamples\n\njulia> sparsevec(Dict(1 => 3, 2 => 2))\n2-element SparseVector{Int64, Int64} with 2 stored entries:\n  [1]  =  3\n  [2]  =  2\n\n\n\n\n\nsparsevec(A)\n\nConvert a vector A into a sparse vector of length m. Numerical zeros in A are turned into structural zeros.\n\nExamples\n\njulia> sparsevec([1.0, 2.0, 0.0, 0.0, 3.0, 0.0])\n6-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  1.0\n  [2]  =  2.0\n  [5]  =  3.0\n\n\n\n\n\n"},{"title":"Base.similar","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#Base.similar-Tuple{SparseArrays.AbstractSparseMatrixCSC, Type}","category":"method","text":"similar(A::AbstractSparseMatrixCSC{Tv,Ti}, [::Type{TvNew}, ::Type{TiNew}, m::Integer, n::Integer]) where {Tv,Ti}\n\nCreate an uninitialized mutable array with the given element type, index type, and size, based upon the given source SparseMatrixCSC. The new sparse matrix maintains the structure of the original sparse matrix, except in the case where dimensions of the output matrix are different from the output.\n\nThe output matrix has zeros in the same locations as the input, but uninitialized values for the nonzero locations.\n\n\n\n\n\n"},{"title":"SparseArrays.issparse","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.issparse","category":"function","text":"issparse(S)\n\nReturns true if S is sparse, and false otherwise.\n\nExamples\n\njulia> sv = sparsevec([1, 4], [2.3, 2.2], 10)\n10-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  2.3\n  [4]  =  2.2\n\njulia> issparse(sv)\ntrue\n\njulia> issparse(Array(sv))\nfalse\n\n\n\n\n\n"},{"title":"SparseArrays.nnz","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.nnz","category":"function","text":"nnz(A)\n\nReturns the number of stored (filled) elements in a sparse array.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> nnz(A)\n3\n\n\n\n\n\n"},{"title":"SparseArrays.findnz","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.findnz","category":"function","text":"findnz(A::SparseMatrixCSC)\n\nReturn a tuple (I, J, V) where I and J are the row and column indices of the stored (\"structurally non-zero\") values in sparse matrix A, and V is a vector of the values.\n\nExamples\n\njulia> A = sparse([1 2 0; 0 0 3; 0 4 0])\n3×3 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n 1  2  ⋅\n ⋅  ⋅  3\n ⋅  4  ⋅\n\njulia> findnz(A)\n([1, 1, 3, 2], [1, 2, 2, 3], [1, 2, 4, 3])\n\n\n\n\n\n"},{"title":"SparseArrays.spzeros","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.spzeros","category":"function","text":"spzeros([type,]m[,n])\n\nCreate a sparse vector of length m or sparse matrix of size m x n. This sparse array will not contain any nonzero values. No storage will be allocated for nonzero values during construction. The type defaults to Float64 if not specified.\n\nExamples\n\njulia> spzeros(3, 3)\n3×3 SparseMatrixCSC{Float64, Int64} with 0 stored entries:\n  ⋅    ⋅    ⋅\n  ⋅    ⋅    ⋅\n  ⋅    ⋅    ⋅\n\njulia> spzeros(Float32, 4)\n4-element SparseVector{Float32, Int64} with 0 stored entries\n\n\n\n\n\nspzeros([type], I::AbstractVector, J::AbstractVector, [m, n])\n\nCreate a sparse matrix S of dimensions m x n with structural zeros at S[I[k], J[k]].\n\nThis method can be used to construct the sparsity pattern of the matrix, and is more efficient than using e.g. sparse(I, J, zeros(length(I))).\n\nFor additional documentation and an expert driver, see SparseArrays.spzeros!.\n\ncompat: Julia 1.10\nThis methods requires Julia version 1.10 or later.\n\n\n\n\n\n"},{"title":"SparseArrays.spzeros!","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.spzeros!","category":"function","text":"spzeros!(::Type{Tv}, I::AbstractVector{Ti}, J::AbstractVector{Ti}, m::Integer, n::Integer,\n         klasttouch::Vector{Ti}, csrrowptr::Vector{Ti}, csrcolval::Vector{Ti},\n         [csccolptr::Vector{Ti}], [cscrowval::Vector{Ti}, cscnzval::Vector{Tv}]) where {Tv,Ti<:Integer}\n\nParent of and expert driver for spzeros(I, J) allowing user to provide preallocated storage for intermediate objects. This method is to spzeros what SparseArrays.sparse! is to sparse. See documentation for SparseArrays.sparse! for details and required buffer lengths.\n\ncompat: Julia 1.10\nThis methods requires Julia version 1.10 or later.\n\n\n\n\n\nSparseArrays.spzeros!(::Type{Tv}, I, J, [m, n]) -> SparseMatrixCSC{Tv}\n\nVariant of spzeros! that re-uses the input vectors I and J for the final matrix storage. After construction the input vectors will alias the matrix buffers; S.colptr === I and S.rowval === J holds, and they will be resize!d as necessary.\n\nNote that some work buffers will still be allocated. Specifically, this method is a convenience wrapper around spzeros!(Tv, I, J, m, n, klasttouch, csrrowptr, csrcolval, csccolptr, cscrowval) where this method allocates klasttouch, csrrowptr, and csrcolval of appropriate size, but reuses I and J for csccolptr and cscrowval.\n\nArguments m and n defaults to maximum(I) and maximum(J).\n\ncompat: Julia 1.10\nThis method requires Julia version 1.10 or later.\n\n\n\n\n\n"},{"title":"SparseArrays.spdiagm","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.spdiagm","category":"function","text":"spdiagm(kv::Pair{<:Integer,<:AbstractVector}...)\nspdiagm(m::Integer, n::Integer, kv::Pair{<:Integer,<:AbstractVector}...)\n\nConstruct a sparse diagonal matrix from Pairs of vectors and diagonals. Each vector kv.second will be placed on the kv.first diagonal.  By default, the matrix is square and its size is inferred from kv, but a non-square size m×n (padded with zeros as needed) can be specified by passing m,n as the first arguments.\n\nExamples\n\njulia> spdiagm(-1 => [1,2,3,4], 1 => [4,3,2,1])\n5×5 SparseMatrixCSC{Int64, Int64} with 8 stored entries:\n ⋅  4  ⋅  ⋅  ⋅\n 1  ⋅  3  ⋅  ⋅\n ⋅  2  ⋅  2  ⋅\n ⋅  ⋅  3  ⋅  1\n ⋅  ⋅  ⋅  4  ⋅\n\n\n\n\n\nspdiagm(v::AbstractVector)\nspdiagm(m::Integer, n::Integer, v::AbstractVector)\n\nConstruct a sparse matrix with elements of the vector as diagonal elements. By default (no given m and n), the matrix is square and its size is given by length(v), but a non-square size m×n can be specified by passing m and n as the first arguments.\n\ncompat: Julia 1.6\nThese functions require at least Julia 1.6.\n\nExamples\n\njulia> spdiagm([1,2,3])\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 1  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  3\n\njulia> spdiagm(sparse([1,0,3]))\n3×3 SparseMatrixCSC{Int64, Int64} with 2 stored entries:\n 1  ⋅  ⋅\n ⋅  ⋅  ⋅\n ⋅  ⋅  3\n\n\n\n\n\n"},{"title":"SparseArrays.sparse_hcat","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sparse_hcat","category":"function","text":"sparse_hcat(A...)\n\nConcatenate along dimension 2. Return a SparseMatrixCSC object.\n\ncompat: Julia 1.8\nThis method was added in Julia 1.8. It mimics previous concatenation behavior, where the concatenation with specialized \"sparse\" matrix types from LinearAlgebra.jl automatically yielded sparse output even in the absence of any SparseArray argument.\n\n\n\n\n\n"},{"title":"SparseArrays.sparse_vcat","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sparse_vcat","category":"function","text":"sparse_vcat(A...)\n\nConcatenate along dimension 1. Return a SparseMatrixCSC object.\n\ncompat: Julia 1.8\nThis method was added in Julia 1.8. It mimics previous concatenation behavior, where the concatenation with specialized \"sparse\" matrix types from LinearAlgebra.jl automatically yielded sparse output even in the absence of any SparseArray argument.\n\n\n\n\n\n"},{"title":"SparseArrays.sparse_hvcat","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sparse_hvcat","category":"function","text":"sparse_hvcat(rows::Tuple{Vararg{Int}}, values...)\n\nSparse horizontal and vertical concatenation in one call. This function is called for block matrix syntax. The first argument specifies the number of arguments to concatenate in each block row.\n\ncompat: Julia 1.8\nThis method was added in Julia 1.8. It mimics previous concatenation behavior, where the concatenation with specialized \"sparse\" matrix types from LinearAlgebra.jl automatically yielded sparse output even in the absence of any SparseArray argument.\n\n\n\n\n\n"},{"title":"SparseArrays.blockdiag","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.blockdiag","category":"function","text":"blockdiag(A...)\n\nConcatenate matrices block-diagonally. Currently only implemented for sparse matrices.\n\nExamples\n\njulia> blockdiag(sparse(2I, 3, 3), sparse(4I, 2, 2))\n5×5 SparseMatrixCSC{Int64, Int64} with 5 stored entries:\n 2  ⋅  ⋅  ⋅  ⋅\n ⋅  2  ⋅  ⋅  ⋅\n ⋅  ⋅  2  ⋅  ⋅\n ⋅  ⋅  ⋅  4  ⋅\n ⋅  ⋅  ⋅  ⋅  4\n\n\n\n\n\n"},{"title":"SparseArrays.sprand","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sprand","category":"function","text":"sprand([rng],[T::Type],m,[n],p::AbstractFloat)\nsprand([rng],m,[n],p::AbstractFloat,[rfn=rand])\n\nCreate a random length m sparse vector or m by n sparse matrix, in which the probability of any element being nonzero is independently given by p (and hence the mean density of nonzeros is also exactly p). The optional rng argument specifies a random number generator, see Random Numbers. The optional T argument specifies the element type, which defaults to Float64.\n\nBy default, nonzero values are sampled from a uniform distribution using the rand function, i.e. by rand(T), or rand(rng, T) if rng is supplied; for the default T=Float64, this corresponds to nonzero values sampled uniformly in [0,1).\n\nYou can sample nonzero values from a different distribution by passing a custom rfn function instead of rand.   This should be a function rfn(k) that returns an array of k random numbers sampled from the desired distribution; alternatively, if rng is supplied, it should instead be a function rfn(rng, k).\n\nExamples\n\njulia> sprand(Bool, 2, 2, 0.5)\n2×2 SparseMatrixCSC{Bool, Int64} with 2 stored entries:\n 1  1\n ⋅  ⋅\n\njulia> sprand(Float64, 3, 0.75)\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  0.795547\n  [2]  =  0.49425\n\n\n\n\n\n"},{"title":"SparseArrays.sprandn","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sprandn","category":"function","text":"sprandn([rng][,Type],m[,n],p::AbstractFloat)\n\nCreate a random sparse vector of length m or sparse matrix of size m by n with the specified (independent) probability p of any entry being nonzero, where nonzero values are sampled from the normal distribution. The optional rng argument specifies a random number generator, see Random Numbers.\n\ncompat: Julia 1.1\nSpecifying the output element type Type requires at least Julia 1.1.\n\nExamples\n\njulia> sprandn(2, 2, 0.75)\n2×2 SparseMatrixCSC{Float64, Int64} with 3 stored entries:\n -1.20577     ⋅\n  0.311817  -0.234641\n\n\n\n\n\n"},{"title":"SparseArrays.nonzeros","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.nonzeros","category":"function","text":"nonzeros(A)\n\nReturn a vector of the structural nonzero values in sparse array A. This includes zeros that are explicitly stored in the sparse array. The returned vector points directly to the internal nonzero storage of A, and any modifications to the returned vector will mutate A as well. See rowvals and nzrange.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> nonzeros(A)\n3-element Vector{Int64}:\n 2\n 2\n 2\n\n\n\n\n\n"},{"title":"SparseArrays.rowvals","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.rowvals","category":"function","text":"rowvals(A)\n\nReturn a vector of the row indices of sparse array A. Any modifications to the returned vector will mutate A as well. Providing access to how the row indices are stored internally can be useful in conjunction with iterating over structural nonzero values. See also nonzeros and nzrange.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> rowvals(A)\n3-element Vector{Int64}:\n 1\n 2\n 3\n\n\n\n\n\n"},{"title":"SparseArrays.nzrange","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.nzrange","category":"function","text":"nzrange(A, col::Integer)\n\nReturn the range of indices to the structural nonzero values of column col of sparse array A. In conjunction with nonzeros and rowvals, this allows for convenient iterating over a sparse matrix :\n\nA = sparse(I,J,V)\nrows = rowvals(A)\nvals = nonzeros(A)\nm, n = size(A)\nfor j = 1:n\n   for i in nzrange(A, j)\n      row = rows[i]\n      val = vals[i]\n      # perform sparse wizardry...\n   end\nend\n\nwarning: Warning\nAdding or removing nonzero elements to the ma