# LaTeX Backend Examples

```@meta
CurrentModule = PrettyTables
```

```@setup latex_examples
using PrettyTables

function create_latex_example(table, filename)
    mkdir("tmp")
    cd("tmp")

    write(
      "table.tex",
      """
      \\documentclass[a4paper, 12pt]{article}
      \\pagestyle{empty}
      \\usepackage{color}
      \\usepackage{booktabs}
      \\usepackage{xcolor}
      \\begin{document}
      $table
      \\end{document}
      """
    )

    run(`lualatex table.tex`)
    run(`lualatex table.tex`)
    run(`convert -density 600 table.pdf -flatten -trim $filename`)
    run(`mv $filename ..`)
    cd("..")
    rm("tmp"; recursive = true)
end
```

Here, we can see some examples of LaTeX tables generated by PrettyTables.jl. Notice that the
output has been rendered to PDF using LuaTeX and converted to PNG using ImageMagick.

---

```julia-repl
julia> data = Any[
    1    false      1.0     0x01
    2     true      2.0     0x02
    3    false      3.0     0x03
    4     true      4.0     0x04
    5    false      5.0     0x05
    6     true      6.0     0x06
]

julia> pretty_table(data; backend = :latex)
```

```@setup latex_examples
data = Any[
    1    false      1.0     0x01
    2     true      2.0     0x02
    3    false      3.0     0x03
    4     true      4.0     0x04
    5    false      5.0     0x05
    6     true      6.0     0x06
]

table = pretty_table(
    String,
    data,
    backend = :latex
)

create_latex_example(table, "latex_example_01.png")
```

```@raw html
<img src="../latex_example_01.png" alt="LaTeX Example 01" width="60%">
```

---

This example uses the predefined format for the LaTeX backend
`latex_table_format__booktabs`.

```julia-repl
julia> data = [
    10.0 6.5
     3.0 3.0
     0.1 1.0
]

julia> row_labels = [
    "Atmospheric drag"
    "Gravity gradient"
    "Solar radiation pressure"
]

julia> column_labels = [
    [MultiColumn(2, "Value", :c)],
    [
        latex_cell"\textbf{Torque} [10$^{-5}$ Nm]",
        latex_cell"\textbf{Angular Momentum} [10$^{-3}$ Nms]"
    ]
],

julia> pretty_table(
    data;
    backend = :latex,
    column_labels,
    merge_column_label_cells = :auto,
    row_labels,
    stubhead_label = "Effect",
    style = LatexTableStyle(column_label = String[]),
    summary_row_labels = ["Total"],
    summary_rows = [(data, i) -> sum(data[:, i])],
    table_format = latex_table_format__booktabs
)
```

```@setup latex_examples
data = [
    10.0 6.5
     3.0 3.0
     0.1 1.0
]

row_labels = [
    "Atmospheric drag"
    "Gravity gradient"
    "Solar radiation pressure"
]

column_labels = [
    [MultiColumn(2, "Value", :c)],
    [
        latex_cell"\textbf{Torque} [10$^{-5}$ Nm]",
        latex_cell"\textbf{Angular Momentum} [10$^{-3}$ Nms]"
    ]
]

table = pretty_table(
    String,
    data;
    backend                  = :latex,
    column_labels,
    merge_column_label_cells = :auto,
    row_labels,
    stubhead_label           = "Effect",
    style                    = LatexTableStyle(column_label = String[]),
    summary_row_labels       = ["Total"],
    summary_rows             = [(data, i) -> sum(data[:, i])],
    table_format             = latex_table_format__booktabs
)

create_latex_example(table, "latex_example_02.png")
```

![LaTeX Example 02](./latex_example_02.png)

---

```julia-repl
julia> t = 0:1:20

julia> data = hcat(t, ones(length(t) ), t, 0.5.*t.^2);

julia> column_labels = [
    [            "Time",           "Acceleration", "Velocity", "Distance"],
    [ latex_cell"{[s]}",  latex_cell"[m / s$^2$]",  "[m / s]",      "[m]"]
]

julia> hl_p = LatexHighlighter(
    (data, i, j) -> (j == 4) && (data[i, j] > 9),
    ["textcolor{blue!60}"]
)

julia> hl_v = LatexHighlighter(
    (data, i, j) -> (j == 3) && (data[i, j] > 9),
    ["textcolor{red!60}"]
)

julia> pretty_table(
    data;
    backend       = :latex,
    column_labels = column_labels,
    highlighters  = [hl_p, hl_v],
    style         = LatexTableStyle(column_label = String[]),
    table_format  = latex_table_format__booktabs
)
```

```@setup latex_examples
t = 0:1:20

data = hcat(t, ones(length(t) ), t, 0.5.*t.^2);

column_labels = [
    [            "Time",           "Acceleration", "Velocity", "Distance"],
    [ latex_cell"{[s]}",  latex_cell"[m / s$^2$]",  "[m / s]",      "[m]"]
]

hl_p = LatexHighlighter(
    (data, i, j) -> (j == 4) && (data[i, j] > 9),
    ["textcolor{blue!60}"]
);

hl_v = LatexHighlighter(
    (data, i, j) -> (j == 3) && (data[i, j] > 9),
    ["textcolor{red!60}"]
);

table = pretty_table(
    String,
    data;
    backend       = :latex,
    column_labels = column_labels,
    highlighters  = [hl_p, hl_v],
    style         = LatexTableStyle(column_label = String[]),
    table_format  = latex_table_format__booktabs
)

create_latex_example(table, "latex_example_03.png")
```

```@raw html
<img src="../latex_example_03.png" alt="LaTeX Example 01" width="70%">
```
