Advanced Expression Manipulation
==================================

```@setup Julia
using SymPyPythonCall
```

In this section, we discuss some ways that we can perform advanced
manipulation of expressions.


Understanding Expression Trees
==============================

Before we can do this, we need to understand how expressions are represented
in SymPy.  A mathematical expression is represented as a tree.  Let us take
the expression `x^2 + xy`, i.e., `x**2 + x*y`.  We can see what this
expression looks like internally by using `srepr`

!!! tip "Julia differences"

    The `srepr` function needs qualification

```@repl Julia
@syms x, y, z
expr = x^2 + x*y
sympy.srepr(expr)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> from sympy import *
    >>> x, y, z = symbols('x y z')

    >>> expr = x**2 + x*y
    >>> srepr(expr)
    "Add(Pow(Symbol('x'), Integer(2)), Mul(Symbol('x'), Symbol('y')))"
```
```@raw html
</details>
```
----


The easiest way to tear this apart is to look at a diagram of the expression
tree:

!!! tip "Julia differences"

    The diagram is not presented here

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
.. This comes from dotprint(x**2 + x*y, labelfunc=srepr)

.. graphviz::

    digraph{

    # Graph style
    "ordering"="out"
    "rankdir"="TD"

    #########
    # Nodes #
    #########

    "Add(Pow(Symbol('x'), Integer(2)), Mul(Symbol('x'), Symbol('y')))_()" ["color"="black", "label"="Add", "shape"="ellipse"];
    "Pow(Symbol('x'), Integer(2))_(0,)" ["color"="black", "label"="Pow", "shape"="ellipse"];
    "Symbol('x')_(0, 0)" ["color"="black", "label"="Symbol('x')", "shape"="ellipse"];
    "Integer(2)_(0, 1)" ["color"="black", "label"="Integer(2)", "shape"="ellipse"];
    "Mul(Symbol('x'), Symbol('y'))_(1,)" ["color"="black", "label"="Mul", "shape"="ellipse"];
    "Symbol('x')_(1, 0)" ["color"="black", "label"="Symbol('x')", "shape"="ellipse"];
    "Symbol('y')_(1, 1)" ["color"="black", "label"="Symbol('y')", "shape"="ellipse"];

    #########
    # Edges #
    #########

    "Add(Pow(Symbol('x'), Integer(2)), Mul(Symbol('x'), Symbol('y')))_()" -> "Pow(Symbol('x'), Integer(2))_(0,)";
    "Add(Pow(Symbol('x'), Integer(2)), Mul(Symbol('x'), Symbol('y')))_()" -> "Mul(Symbol('x'), Symbol('y'))_(1,)";
    "Pow(Symbol('x'), Integer(2))_(0,)" -> "Symbol('x')_(0, 0)";
    "Pow(Symbol('x'), Integer(2))_(0,)" -> "Integer(2)_(0, 1)";
    "Mul(Symbol('x'), Symbol('y'))_(1,)" -> "Symbol('x')_(1, 0)";
    "Mul(Symbol('x'), Symbol('y'))_(1,)" -> "Symbol('y')_(1, 1)";
    }
```
```@raw html
</details>
```
----


!!! note

    The above diagram was made using [Graphviz](https://www.graphviz.org/) and
    the [dotprint](sympy.printing.dot.dotprint) function.

First, let's look at the leaves of this tree.  Symbols are instances of the
class Symbol.  While we have been doing

```@repl Julia
@syms x
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> x = symbols('x')
```
```@raw html
</details>
```
----


we could have also done


```@repl Julia
x = symbols("x")
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> x = Symbol('x')
```
```@raw html
</details>
```
----


Either way, we get a Symbol with the name "`x`".  For the number in the
expression, 2, we got `Integer(2)`.  `Integer` is the SymPy class for
integers.  It is similar to the Python built-in type `int`, except that
`Integer` plays nicely with other SymPy types.

When we write `x**2`, this creates a `Pow` object.  `Pow` is short for
"power".


```@repl Julia
sympy.srepr(x^2)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> srepr(x**2)
    "Pow(Symbol('x'), Integer(2))"
```
```@raw html
</details>
```
----


We could have created the same object by calling `Pow(x, 2)`


```@repl Julia
sympy.Pow(x, 2)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> Pow(x, 2)
    x**2
```
```@raw html
</details>
```
----


Note that in the `srepr` output, we see `Integer(2)`, the SymPy version of
integers, even though technically, we input `2`, a Python int.  In general,
whenever you combine a SymPy object with a non-SymPy object via some function
or operation, the non-SymPy object will be converted into a SymPy object.  The
function that does this is `sympify`.

!!! tip "Julia differences"

    Using `Sym(2)` is more idiomatic. It uses `sympify`, though a more performant means would be to create a Python object (which differs between `PyCall` (`PyObject(2)`) and `PythonCall` (`Py(2)`)) and call `Sym` on that.


```@repl Julia
sympify(2)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> type(2)
    <... 'int'>
    >>> type(sympify(2))
    <class 'sympy.core.numbers.Integer'>
```
```@raw html
</details>
```
----


We have seen that `x**2` is represented as `Pow(x, 2)`.  What about
`x*y`?  As we might expect, this is the multiplication of `x` and `y`.
The SymPy class for multiplication is `Mul`.


```@repl Julia
sympy.srepr(x * y)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> srepr(x*y)
    "Mul(Symbol('x'), Symbol('y'))"
```
```@raw html
</details>
```
----


Thus, we could have created the same object by writing `Mul(x, y)`.


```@repl Julia
sympy.Mul(x, y)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> Mul(x, y)
    x*y
```
```@raw html
</details>
```
----


Now we get to our final expression, `x**2 + x*y`.  This is the addition of
our last two objects, `Pow(x, 2)`, and `Mul(x, y)`.  The SymPy class for
addition is `Add`, so, as you might expect, to create this object, we use
`Add(Pow(x, 2), Mul(x, y))`.


```@repl Julia
sympy.Add(sympy.Pow(x, 2), sympy.Mul(x, y))
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> Add(Pow(x, 2), Mul(x, y))
    x**2 + x*y
```
```@raw html
</details>
```
----


SymPy expression trees can have many branches, and can be quite deep or quite
broad.  Here is a more complicated example


```@repl Julia
expr = sin(x*2)/2 - x^2 + 1/y
sympy.srepr(expr)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> expr = sin(x*y)/2 - x**2 + 1/y
    >>> srepr(expr)
    "Add(Mul(Integer(-1), Pow(Symbol('x'), Integer(2))), Mul(Rational(1, 2),
    sin(Mul(Symbol('x'), Symbol('y')))), Pow(Symbol('y'), Integer(-1)))"
```
```@raw html
</details>
```
----


Here is a diagram

!!! tip "Julia differences"

    This is not present in this translation

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
.. dotprint(sin(x*y)/2 - x**2 + 1/y, labelfunc=srepr)

.. graphviz::

    digraph{

    # Graph style
    "rankdir"="TD"

    #########
    # Nodes #
    #########

    "Half()_(0, 0)" ["color"="black", "label"="Rational(1, 2)", "shape"="ellipse"];
    "Symbol(y)_(2, 0)" ["color"="black", "label"="Symbol('y')", "shape"="ellipse"];
    "Symbol(x)_(1, 1, 0)" ["color"="black", "label"="Symbol('x')", "shape"="ellipse"];
    "Integer(2)_(1, 1, 1)" ["color"="black", "label"="Integer(2)", "shape"="ellipse"];
    "NegativeOne()_(2, 1)" ["color"="black", "label"="Integer(-1)", "shape"="ellipse"];
    "NegativeOne()_(1, 0)" ["color"="black", "label"="Integer(-1)", "shape"="ellipse"];
    "Symbol(y)_(0, 1, 0, 1)" ["color"="black", "label"="Symbol('y')", "shape"="ellipse"];
    "Symbol(x)_(0, 1, 0, 0)" ["color"="black", "label"="Symbol('x')", "shape"="ellipse"];
    "Pow(Symbol(x), Integer(2))_(1, 1)" ["color"="black", "label"="Pow", "shape"="ellipse"];
    "Pow(Symbol(y), NegativeOne())_(2,)" ["color"="black", "label"="Pow", "shape"="ellipse"];
    "Mul(Symbol(x), Symbol(y))_(0, 1, 0)" ["color"="black", "label"="Mul", "shape"="ellipse"];
    "sin(Mul(Symbol(x), Symbol(y)))_(0, 1)" ["color"="black", "label"="sin", "shape"="ellipse"];
    "Mul(Half(), sin(Mul(Symbol(x), Symbol(y))))_(0,)" ["color"="black", "label"="Mul", "shape"="ellipse"];
    "Mul(NegativeOne(), Pow(Symbol(x), Integer(2)))_(1,)" ["color"="black", "label"="Mul", "shape"="ellipse"];
    "Add(Mul(Half(), sin(Mul(Symbol(x), Symbol(y)))), Mul(NegativeOne(), Pow(Symbol(x), Integer(2))), Pow(Symbol(y), NegativeOne()))_()" ["color"="black", "label"="Add", "shape"="ellipse"];

    #########
    # Edges #
    #########

    "Pow(Symbol(y), NegativeOne())_(2,)" -> "Symbol(y)_(2, 0)";
    "Pow(Symbol(x), Integer(2))_(1, 1)" -> "Symbol(x)_(1, 1, 0)";
    "Pow(Symbol(x), Integer(2))_(1, 1)" -> "Integer(2)_(1, 1, 1)";
    "Pow(Symbol(y), NegativeOne())_(2,)" -> "NegativeOne()_(2, 1)";
    "Mul(Symbol(x), Symbol(y))_(0, 1, 0)" -> "Symbol(x)_(0, 1, 0, 0)";
    "Mul(Symbol(x), Symbol(y))_(0, 1, 0)" -> "Symbol(y)_(0, 1, 0, 1)";
    "Mul(Half(), sin(Mul(Symbol(x), Symbol(y))))_(0,)" -> "Half()_(0, 0)";
    "Mul(NegativeOne(), Pow(Symbol(x), Integer(2)))_(1,)" -> "NegativeOne()_(1, 0)";
    "sin(Mul(Symbol(x), Symbol(y)))_(0, 1)" -> "Mul(Symbol(x), Symbol(y))_(0, 1, 0)";
    "Mul(NegativeOne(), Pow(Symbol(x), Integer(2)))_(1,)" -> "Pow(Symbol(x), Integer(2))_(1, 1)";
    "Mul(Half(), sin(Mul(Symbol(x), Symbol(y))))_(0,)" -> "sin(Mul(Symbol(x), Symbol(y)))_(0, 1)";
    "Add(Mul(Half(), sin(Mul(Symbol(x), Symbol(y)))), Mul(NegativeOne(), Pow(Symbol(x), Integer(2))), Pow(Symbol(y), NegativeOne()))_()" -> "Pow(Symbol(y), NegativeOne())_(2,)";
    "Add(Mul(Half(), sin(Mul(Symbol(x), Symbol(y)))), Mul(NegativeOne(), Pow(Symbol(x), Integer(2))), Pow(Symbol(y), NegativeOne()))_()" -> "Mul(Half(), sin(Mul(Symbol(x), Symbol(y))))_(0,)";
    "Add(Mul(Half(), sin(Mul(Symbol(x), Symbol(y)))), Mul(NegativeOne(), Pow(Symbol(x), Integer(2))), Pow(Symbol(y), NegativeOne()))_()" -> "Mul(NegativeOne(), Pow(Symbol(x), Integer(2)))_(1,)";
    }
```
```@raw html
</details>
```
----


This expression reveals some interesting things about SymPy expression
trees. Let's go through them one by one.

Let's first look at the term `x**2`.  As we expected, we see `Pow(x, 2)`.
One level up, we see we have `Mul(-1, Pow(x, 2))`.  There is no subtraction
class in SymPy.  `x - y` is represented as `x + -y`, or, more completely,
`x + -1*y`, i.e., `Add(x, Mul(-1, y))`.


```@repl Julia
sympy.srepr(x - y)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> srepr(x - y)
    "Add(Symbol('x'), Mul(Integer(-1), Symbol('y')))"
```
```@raw html
</details>
```
----



----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
.. dotprint(x - y, labelfunc=srepr)

.. graphviz::

    digraph{

    # Graph style
    "rankdir"="TD"

    #########
    # Nodes #
    #########

    "Symbol(x)_(1,)" ["color"="black", "label"="Symbol('x')", "shape"="ellipse"];
    "Symbol(y)_(0, 1)" ["color"="black", "label"="Symbol('y')", "shape"="ellipse"];
    "NegativeOne()_(0, 0)" ["color"="black", "label"="Integer(-1)", "shape"="ellipse"];
    "Mul(NegativeOne(), Symbol(y))_(0,)" ["color"="black", "label"="Mul", "shape"="ellipse"];
    "Add(Mul(NegativeOne(), Symbol(y)), Symbol(x))_()" ["color"="black", "label"="Add", "shape"="ellipse"];

    #########
    # Edges #
    #########

    "Mul(NegativeOne(), Symbol(y))_(0,)" -> "Symbol(y)_(0, 1)";
    "Mul(NegativeOne(), Symbol(y))_(0,)" -> "NegativeOne()_(0, 0)";
    "Add(Mul(NegativeOne(), Symbol(y)), Symbol(x))_()" -> "Symbol(x)_(1,)";
    "Add(Mul(NegativeOne(), Symbol(y)), Symbol(x))_()" -> "Mul(NegativeOne(), Symbol(y))_(0,)";
    }
```
```@raw html
</details>
```
----


Next, look at `1/y`.  We might expect to see something like `Div(1, y)`,
but similar to subtraction, there is no class in SymPy for division.  Rather,
division is represented by a power of -1.  Hence, we have `Pow(y, -1)`.
What if we had divided something other than 1 by `y`, like `x/y`?  Let's
see.


```@repl Julia
expr = x / y
sympy.srepr(expr)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> expr = x/y
    >>> srepr(expr)
    "Mul(Symbol('x'), Pow(Symbol('y'), Integer(-1)))"
```
```@raw html
</details>
```
----



----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
.. dotprint(x/y, labelfunc=srepr)

.. graphviz::

    digraph{

    # Graph style
    "rankdir"="TD"

    #########
    # Nodes #
    #########

    "Symbol(x)_(0,)" ["color"="black", "label"="Symbol('x')", "shape"="ellipse"];
    "Symbol(y)_(1, 0)" ["color"="black", "label"="Symbol('y')", "shape"="ellipse"];
    "NegativeOne()_(1, 1)" ["color"="black", "label"="Integer(-1)", "shape"="ellipse"];
    "Pow(Symbol(y), NegativeOne())_(1,)" ["color"="black", "label"="Pow", "shape"="ellipse"];
    "Mul(Symbol(x), Pow(Symbol(y), NegativeOne()))_()" ["color"="black", "label"="Mul", "shape"="ellipse"];

    #########
    # Edges #
    #########

    "Pow(Symbol(y), NegativeOne())_(1,)" -> "Symbol(y)_(1, 0)";
    "Pow(Symbol(y), NegativeOne())_(1,)" -> "NegativeOne()_(1, 1)";
    "Mul(Symbol(x), Pow(Symbol(y), NegativeOne()))_()" -> "Symbol(x)_(0,)";
    "Mul(Symbol(x), Pow(Symbol(y), NegativeOne()))_()" -> "Pow(Symbol(y), NegativeOne())_(1,)";
    }
```
```@raw html
</details>
```
----


We see that `x/y` is represented as `x*y**-1`, i.e., `Mul(x, Pow(y,
-1))`.

Finally, let's look at the `sin(x*y)/2` term.  Following the pattern of the
previous example, we might expect to see `Mul(sin(x*y), Pow(Integer(2),
-1))`.  But instead, we have `Mul(Rational(1, 2), sin(x*y))`.  Rational
numbers are always combined into a single term in a multiplication, so that
when we divide by 2, it is represented as multiplying by 1/2.

Finally, one last note.  You may have noticed that the order we entered our
expression and the order that it came out from `srepr` or in the graph were
different.  You may have also noticed this phenomenon earlier in the
tutorial.  For example

```@repl Julia
1 + x
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
     >>> 1 + x
     x + 1
```
```@raw html
</details>
```
----


This because in SymPy, the arguments of the commutative operations `Add` and
`Mul` are stored in an arbitrary (but consistent!) order, which is
independent of the order inputted (if you're worried about noncommutative
multiplication, don't be.  In SymPy, you can create noncommutative Symbols
using `Symbol('A', commutative=False)`, and the order of multiplication for
noncommutative Symbols is kept the same as the input).  Furthermore, as we
shall see in the next section, the printing order and the order in which
things are stored internally need not be the same either.

!!! note "Quick Tip"

    The way an expression is represented internally and the way it is printed
    are often not the same.

In general, an important thing to keep in mind when working with SymPy expression
trees is this:  the internal representation of an expression and the way it is
printed need not be the same.  The same is true for the input form.   If some
expression manipulation algorithm is not working in the way you expected it
to, chances are, the internal representation of the object is different from
what you thought it was.

Recursing through an Expression Tree
====================================

Now that you know how expression trees work in SymPy, let's look at how to dig
our way through an expression tree.  Every object in SymPy has two very
important attributes, `func`, and `args`.


func
----

`func` is the head of the object. For example, `(x*y).func` is `Mul`.
Usually it is the same as the class of the object (though there are exceptions
to this rule).

Two notes about `func`.  First, the class of an object need not be the same
as the one used to create it.  For example

!!! tip "Julia differences"

    The `Introspection.func` methods (and others) allows a more `Julia`n calling style

```@repl Julia
expr = sympy.Add(x,x)
Introspection.func(expr)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> expr = Add(x, x)
    >>> expr.func
    <class 'sympy.core.mul.Mul'>
```
```@raw html
</details>
```
----


We created `Add(x, x)`, so we might expect `expr.func` to be `Add`, but
instead we got `Mul`.  Why is that?  Let's take a closer look at `expr`.


```@repl Julia
expr
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> expr
    2*x
```
```@raw html
</details>
```
----


`Add(x, x)`, i.e., `x + x`, was automatically converted into `Mul(2,
x)`, i.e., `2*x`, which is a `Mul`.   SymPy classes make heavy use of the
`__new__` class constructor, which, unlike `__init__`, allows a different
class to be returned from the constructor.

Second, some classes are special-cased, usually for efficiency reasons.

```@repl Julia
Introspection.func(Sym(2))
Introspection.func(Sym(0))
Introspection.func(Sym(-1))
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> Integer(2).func
    <class 'sympy.core.numbers.Integer'>
    >>> Integer(0).func
    <class 'sympy.core.numbers.Zero'>
    >>> Integer(-1).func
    <class 'sympy.core.numbers.NegativeOne'>
```
```@raw html
</details>
```
----


For the most part, these issues will not bother us.  The special classes
`Zero`, `One`, `NegativeOne`, and so on are subclasses of `Integer`,
so as long as you use `isinstance`, it will not be an issue.

args
----

`args` are the top-level arguments of the object.  `(x*y).args` would be
`(x, y)`.  Let's look at some examples

!!! tip "Julia differences"

    Again, `Introspection` allows a more `Julia`n means to call this object method. Unlike `func`, `args` returns `Sym` values. In the example below, they are converted to underling Python values.

```@repl Julia
expr = 3 * y^2* x
Introspection.func(expr)
Introspection.args(expr)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> expr = 3*y**2*x
    >>> expr.func
    <class 'sympy.core.mul.Mul'>
    >>> expr.args
    (3, x, y**2)
```
```@raw html
</details>
```
----


From this, we can see that `expr == Mul(3, y**2, x)`.  In fact, we can see
that we can completely reconstruct `expr` from its `func` and its
`args`.

!!! tip "Julia differences"

    Actually, what is created is the Python value, which if swrapped in `Sym` becoes the original value. Note the use of `\downarrow[tab]` to pass the Python values to the output of `func`.

```@repl Julia
u = Introspection.func(expr)(↓(Introspection.args(expr))...)
↓(expr) == u
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> expr.func(*expr.args)
    3*x*y**2
    >>> expr == expr.func(*expr.args)
    True
```
```@raw html
</details>
```
----


Note that although we entered `3*y**2*x`, the `args` are `(3, x, y**2)`.
In a `Mul`, the Rational coefficient will come first in the `args`, but
other than that, the order of everything else follows no special pattern.  To
be sure, though, there is an order.


```@repl Julia
expr = y^2 * 3*x
Introspection.args(expr)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> expr = y**2*3*x
    >>> expr.args
    (3, x, y**2)
```
```@raw html
</details>
```
----


Mul's `args` are sorted, so that the same `Mul` will have the same
`args`.  But the sorting is based on some criteria designed to make the
sorting unique and efficient that has no mathematical significance.

The `srepr` form of our `expr` is `Mul(3, x, Pow(y, 2))`.  What if we
want to get at the `args` of `Pow(y, 2)`.  Notice that the `y**2` is in
the third slot of `expr.args`, i.e., `expr.args[2]`.

!!! tip "Julia differences"

    `Julia` is `1` based, not `0` based like `Python`, so the index is `3`

```@repl Julia
Introspection.args(expr)[3]
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> expr.args[2]
    y**2
```
```@raw html
</details>
```
----


So to get the `args` of this, we call `expr.args[2].args`.


```@repl Julia
Introspection.args(Introspection.args(expr)[3])
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> expr.args[2].args
    (y, 2)
```
```@raw html
</details>
```
----


Now what if we try to go deeper.  What are the args of `y`.  Or `2`.
Let's see.


```@repl Julia
Introspection.args(y)
Introspection.args(Sym(2))
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> y.args
    ()
    >>> Integer(2).args
    ()
```
```@raw html
</details>
```
----


They both have empty `args`.  In SymPy, empty `args` signal that we have
hit a leaf of the expression tree.

So there are two possibilities for a SymPy expression. Either it has empty
`args`, in which case it is a leaf node in any expression tree, or it has
`args`, in which case, it is a branch node of any expression tree.  When it
has `args`, it can be completely rebuilt from its `func` and its `args`.
This is expressed in the key invariant.

!!! note "Key Invariant"

    Every well-formed SymPy expression must either have empty `args` or
    satisfy `expr == expr.func(*expr.args)`.

(Recall that in Python if `a` is a tuple, then `f(*a)` means to call `f`
with arguments from the elements of `a`, e.g., `f(*(1, 2, 3))` is the same
as `f(1, 2, 3)`.)

This key invariant allows us to write simple algorithms that walk expression
trees, change them, and rebuild them into new expressions.

Walking the Tree
----------------

With this knowledge, let's look at how we can recurse through an expression
tree.  The nested nature of `args` is a perfect fit for recursive functions.
The base case will be empty `args`.  Let's write a simple function that goes
through an expression and prints all the `args` at each level.


```@repl Julia
function pre(expr)
   println(expr)
   for arg in Introspection.args(expr)
       pre(arg)
   end
end
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> def pre(expr):
    ...     print(expr)
    ...     for arg in expr.args:
    ...         pre(arg)
```
```@raw html
</details>
```
----


See how nice it is that `()` signals leaves in the expression tree.  We
don't even have to write a base case for our recursion; it is handled
automatically by the for loop.

Let's test our function.


```@repl Julia
expr = x*y + 1
pre(expr)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> expr = x*y + 1
    >>> pre(expr)
    x*y + 1
    1
    x*y
    x
    y
```
```@raw html
</details>
```
----


Can you guess why we called our function `pre`?  We just wrote a pre-order
traversal function for our expression tree.   See if you can write a
post-order traversal function.

Such traversals are so common in SymPy that the generator functions
`preorder_traversal` and `postorder_traversal` are provided to make such
traversals easy.  We could have also written our algorithm as


```@repl Julia
collect(sympy.preorder_traversal(expr))
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> for arg in preorder_traversal(expr):
    ...     print(arg)
    x*y + 1
    1
    x*y
    x
    y
```
```@raw html
</details>
```
----



Prevent expression evaluation
=============================

There are generally two ways to prevent the evaluation, either pass an
`evaluate=False` parameter while constructing the expression, or create
an evaluation stopper by wrapping the expression with `UnevaluatedExpr`.

For example:


```@repl Julia
x + x
sympy.Add(x, x)
sympy.Add(x, x, evaluate=false)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> from sympy import Add
    >>> from sympy.abc import x, y, z
    >>> x + x
    2*x
    >>> Add(x, x)
    2*x
    >>> Add(x, x, evaluate=False)
    x + x
```
```@raw html
</details>
```
----


If you don't remember the class corresponding to the expression you
want to build (operator overloading usually assumes `evaluate=True`),
just use `sympify` and pass a string:


```@repl Julia
sympify("x + x", evaluate = false)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> from sympy import sympify
    >>> sympify("x + x", evaluate=False)
    x + x
```
```@raw html
</details>
```
----


Note that `evaluate=False` won't prevent future evaluation in later
usages of the expression:

```@repl Julia
expr = sympy.Add(x, x, evaluate = false)
expr + x
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> expr = Add(x, x, evaluate=False)
    >>> expr
    x + x
    >>> expr + x
    3*x
```
```@raw html
</details>
```
----


That's why the class `UnevaluatedExpr` comes handy.
`UnevaluatedExpr` is a method provided by SymPy which lets the user keep
an expression unevaluated. By *unevaluated* it is meant that the value
inside of it will not interact with the expressions outside of it to give
simplified outputs. For example:

```@repl Julia
expr = x + sympy.UnevaluatedExpr(x)
x + expr
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> from sympy import UnevaluatedExpr
    >>> expr = x + UnevaluatedExpr(x)
    >>> expr
    x + x
    >>> x + expr
    2*x + x
```
```@raw html
</details>
```
----


The `x` remaining alone is the `x` wrapped by `UnevaluatedExpr`.
To release it:

!!! tip "Julia differences"

    the `doit` object method finds many other uses.

```@repl Julia
 (x + expr).doit()
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> (x + expr).doit()
    3*x
```
```@raw html
</details>
```
----


Other examples:

!!! tip "Julia differences"

    The `S` module of Python, is not available as `sympy.S`. Rather, we export it through `𝑆`

```@repl Julia
sympy.UnevaluatedExpr(𝑆.One * 5 /7) * sympy.UnevaluatedExpr(𝑆.One * 3 / 4)
x * sympy.UnevaluatedExpr(1/x)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> from sympy import *
    >>> from sympy.abc import x, y, z
    >>> uexpr = UnevaluatedExpr(S.One*5/7)*UnevaluatedExpr(S.One*3/4)
    >>> uexpr
    (5/7)*(3/4)
    >>> x*UnevaluatedExpr(1/x)
    x*1/x
```
```@raw html
</details>
```
----


A point to be noted is that  `UnevaluatedExpr` cannot prevent the
evaluation of an expression which is given as argument. For example:


```@repl Julia
expr1 = sympy.UnevaluatedExpr(x + x)
expr2 = sympify("x + x", evaluate=false)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> expr1 = UnevaluatedExpr(x + x)
    >>> expr1
    2*x
    >>> expr2 = sympify('x + x', evaluate=False)
    >>> expr2
    x + x
```
```@raw html
</details>
```
----


Remember that `expr2` will be evaluated if included into another
expression. Combine both of the methods to prevent both inside and outside
evaluations:


```@repl Julia
sympy.UnevaluatedExpr(sympify("x + x", evaluate=false)) + y
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> UnevaluatedExpr(sympify("x + x", evaluate=False)) + y
    y + (x + x)
```
```@raw html
</details>
```
----


`UnevaluatedExpr` is supported by SymPy printers and can be used to print the
result in different output forms. For example

```@repl Julia
using Latexify
uexpr = sympy.UnevaluatedExpr(𝑆.One * 5 /7) * sympy.UnevaluatedExpr(𝑆.One * 3 / 4)
latexify(uexpr)
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> from sympy import latex
    >>> uexpr = UnevaluatedExpr(S.One*5/7)*UnevaluatedExpr(S.One*3/4)
    >>> print(latex(uexpr))
    \frac{5}{7} \cdot \frac{3}{4}
```
```@raw html
</details>
```
----


In order to release the expression and get the evaluated LaTeX form,
just use `.doit()`:


```@repl Julia
latexify(uexpr.doit())
```

----

```@raw html
<details><summary>Expand for Python example</summary>
```

```python
    >>> print(latex(uexpr.doit()))
    \frac{15}{28}
```
```@raw html
</details>
```
----



!!! note "Footnotes"

    * We have been using `symbols` instead of `Symbol` because it
    automatically splits apart strings into multiple `Symbol`s.
  ` * symbols('x y z')` returns a tuple of three `Symbol` s.  `Symbol('x y z')`
    returns a single `Symbol` called `x y z`.
    * Technically, it is an internal function called `_sympify`,  which differs from `sympify`
	in that it does not convert strings.  `x +  '2'` is not allowed.
    * Classes like `One` and `Zero` are singletonized, meaning that only one object is ever created,
	no matter how many times the class is called.  This is done for space efficiency,
	as these classes are very  common.  For example, `Zero` might occur very often in a sparse matrix
    represented densely.  As we have seen, `NegativeOne` occurs any time we
    have `-x` or `1/x`.  It is also done for speed efficiency because
    singletonized objects can be compared by `is`.  The unique objects for
    each singletonized class can be accessed from the `S` object.
