/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.data.nc;

import gov.nasa.giss.data.nc.NcDataset;
import gov.nasa.giss.data.nc.NcException;
import gov.nasa.giss.data.nc.NcUtils;
import gov.nasa.giss.text.PrintfFormat;
import gov.nasa.giss.time.JulianTime;
import gov.nasa.giss.time.JulianType;
import gov.nasa.giss.time.RelativeTime;
import gov.nasa.giss.time.TimeFormatter;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.ma2.Array;
import ucar.ma2.DataType;
import ucar.ma2.Index;
import ucar.nc2.Attribute;
import ucar.nc2.Variable;

class NcDimensionVariable {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final PrintfFormat SFORMAT = new PrintfFormat("%G");
    private final NcDataset dataset_;
    private final Variable njvar_;
    private final String sname_;
    private final String longName_;
    private String unitsStr_;
    private TimeFormatter timeParser_;
    private int length_;
    private Object[] values_;
    private String[] formattedValues_;
    private Variable boundsVar_;
    private String boundsName_;
    private Object[][] bounds_;
    private String[] formattedBounds_;
    private boolean initted_;

    NcDimensionVariable(NcDataset ncd, Variable njvar) {
        Objects.requireNonNull(ncd, "Dataset cannot be null.");
        Objects.requireNonNull(njvar, "NJ variable cannot be null.");
        this.dataset_ = ncd;
        this.njvar_ = njvar;
        this.sname_ = njvar.getShortName();
        this.longName_ = NcUtils.getLongName(this.njvar_);
        this.initMe();
    }

    private void initMe() {
        if (this.njvar_.getRank() < 1) {
            throw new NcException("Cannot create dimension variable from a scalar.");
        }
        this.length_ = this.njvar_.getShape()[0];
        this.values_ = new Object[this.length_];
        this.formattedValues_ = new String[this.length_];
        this.unitsStr_ = NcUtils.getUnitsStr(this.njvar_);
        if (this.unitsStr_.length() > 0) {
            String lcunits = this.unitsStr_.toLowerCase();
            if (lcunits.startsWith("degree")) {
                switch (lcunits) {
                    case "degree_east": 
                    case "degrees_east": {
                        this.unitsStr_ = "\u00b0E";
                        break;
                    }
                    case "degree_north": 
                    case "degrees_north": {
                        this.unitsStr_ = "\u00b0N";
                        break;
                    }
                    case "degree_west": 
                    case "degrees_west": {
                        this.unitsStr_ = "\u00b0W";
                        break;
                    }
                    case "degree_south": 
                    case "degrees_south": {
                        this.unitsStr_ = "\u00b0S";
                        break;
                    }
                }
            } else if (this.unitsStr_.contains(" since ")) {
                try {
                    this.timeParser_ = NcUtils.getRelativeTimeHandler(this.dataset_, this.njvar_);
                    this.unitsStr_ = "";
                }
                catch (Exception exc) {
                    this.timeParser_ = null;
                    LOGGER.warn("Exc creating relative time handler for '{}': {}", (Object)this.unitsStr_, (Object)exc);
                }
                LOGGER.trace("Looks like Relative Time, {}", (Object)(this.timeParser_ != null ? 1 : 0));
            } else if (NcUtils.isTrueJulianTimeUnitsStr(this.unitsStr_)) {
                try {
                    this.timeParser_ = new JulianTime();
                }
                catch (Exception exc) {
                    this.timeParser_ = null;
                    LOGGER.warn("Exc creating JD time handler for '{}': {}", (Object)this.unitsStr_, (Object)exc);
                }
                LOGGER.trace("Looks like True Julian, {}", (Object)(this.timeParser_ != null ? 1 : 0));
            } else if (NcUtils.isModifiedJulianTimeUnitsStr(this.unitsStr_)) {
                try {
                    this.timeParser_ = new JulianTime(JulianType.MODIFIED);
                }
                catch (Exception exc) {
                    this.timeParser_ = null;
                    LOGGER.warn("Exc creating MJD time handler for '{}': {}", (Object)this.unitsStr_, (Object)exc);
                }
                LOGGER.trace("Looks like Modified Julian, {}", (Object)(this.timeParser_ != null ? 1 : 0));
            }
        }
    }

    public String getLongName() {
        return this.longName_;
    }

    public String getUnitsStr() {
        return this.unitsStr_;
    }

    public int getLength() {
        return this.length_;
    }

    public Object valueAt(int index) {
        if (!this.initted_) {
            this.initValues();
        }
        return this.values_[index];
    }

    public Object[] getValues() {
        if (!this.initted_) {
            this.initValues();
        }
        return this.values_;
    }

    public String formattedValueAt(int index) {
        if (!this.initted_) {
            this.initValues();
        }
        return this.formattedValues_[index];
    }

    public boolean hasBounds() {
        if (!this.initted_) {
            this.initValues();
        }
        return this.bounds_ != null;
    }

    public String formattedBoundsAt(int index) {
        if (!this.initted_) {
            this.initValues();
        }
        if (this.bounds_ == null) {
            return null;
        }
        if (this.formattedBounds_ == null) {
            this.formattedBounds_ = new String[this.length_];
        }
        if (this.formattedBounds_[index] == null) {
            this.formattedBounds_[index] = this.bounds_[0][0] instanceof Double ? SFORMAT.sprintfx(this.bounds_[index][0]) + " \u2014 " + SFORMAT.sprintfx(this.bounds_[index][1]) : this.bounds_[index][0] + " \u2014 " + this.bounds_[index][1];
        }
        return this.formattedBounds_[index];
    }

    private void initValues() {
        block13: {
            DataType dtype = this.njvar_.getDataType();
            LOGGER.trace("dtype {}", (Object)dtype);
            try {
                if (this.timeParser_ != null) {
                    LOGGER.trace("timeParser {}", (Object)this.timeParser_.getClass().getSimpleName());
                    if (this.timeParser_ instanceof RelativeTime) {
                        this.initRelativeTimeValues();
                        break block13;
                    }
                    if (this.timeParser_ instanceof JulianTime) {
                        this.initJulianTimeValues();
                        break block13;
                    }
                    throw new NcException("Dimension has unknown time type in dimension: " + this.timeParser_.getClass().getSimpleName());
                }
                if (dtype == DataType.INT || dtype == DataType.UINT || dtype == DataType.SHORT || dtype == DataType.USHORT || dtype == DataType.BYTE || dtype == DataType.UBYTE) {
                    this.initIntegerValues();
                    break block13;
                }
                if (dtype == DataType.LONG || dtype == DataType.ULONG) {
                    this.initLongValues();
                    break block13;
                }
                if (dtype == DataType.DOUBLE || dtype == DataType.FLOAT) {
                    this.initDoubleValues();
                    break block13;
                }
                if (dtype == DataType.CHAR) {
                    this.initCharValues();
                    break block13;
                }
                if (dtype == DataType.STRING) {
                    this.initObjectValues();
                    break block13;
                }
                if (dtype == DataType.BOOLEAN) {
                    this.initBooleanValues();
                    break block13;
                }
                throw new NcException("Unknown data type in dimension: " + dtype);
            }
            catch (NcException ncx) {
                throw ncx;
            }
            catch (Exception exc) {
                LOGGER.error("Could not init dimension variable {}: {}", (Object)this.sname_, (Object)exc.toString());
                if (LOGGER.isTraceEnabled()) {
                    exc.printStackTrace();
                }
                throw new NcException("Could not initialize dimension variable " + this.sname_);
            }
        }
        this.initted_ = true;
    }

    private void initCharValues() throws IOException {
        Array dimArray = this.njvar_.read();
        Index dimIndex = dimArray.getIndex();
        int[] shape = this.njvar_.getShape();
        for (int i = 0; i < this.length_; ++i) {
            char c;
            StringBuilder sb = new StringBuilder("");
            for (int ii = 0; ii < shape[1] && (c = dimArray.getChar(dimIndex.set(i, ii))) != '\u0000'; ++ii) {
                sb.append(c);
            }
            this.values_[i] = sb.toString();
            this.formattedValues_[i] = (String)this.values_[i];
        }
    }

    private void initBooleanValues() throws IOException {
        Array dimArray = this.njvar_.read();
        for (int i = 0; i < this.length_; ++i) {
            boolean value = dimArray.getBoolean(i);
            this.values_[i] = value;
            this.formattedValues_[i] = value ? "true" : "false";
        }
    }

    private void initDoubleValues() throws IOException {
        Array dimArray = this.njvar_.read();
        double[] vv = (double[])dimArray.get1DJavaArray(DataType.DOUBLE);
        for (int i = 0; i < this.length_; ++i) {
            this.values_[i] = vv[i];
            this.formattedValues_[i] = SFORMAT.sprintfx(vv[i]);
        }
        this.initDoubleBounds();
    }

    private void initDoubleBounds() throws IOException {
        this.findBoundsVariable();
        if (this.boundsVar_ == null) {
            return;
        }
        double modulo = this.findModulo();
        Array boundsArray = this.boundsVar_.read();
        Index boundsIndex = boundsArray.getIndex();
        for (int i = 0; i < this.length_; ++i) {
            double v1 = boundsArray.getDouble(boundsIndex.set(i, 0));
            double v2 = boundsArray.getDouble(boundsIndex.set(i, 1));
            if (modulo > 0.0) {
                v1 %= modulo;
                v2 %= modulo;
            }
            this.bounds_[i][0] = v1;
            this.bounds_[i][1] = v2;
        }
    }

    private void initIntegerValues() throws IOException {
        Array dimArray = this.njvar_.read();
        for (int i = 0; i < this.length_; ++i) {
            int value = dimArray.getInt(i);
            this.values_[i] = value;
            this.formattedValues_[i] = this.values_[i].toString();
        }
        this.initIntegerBounds();
    }

    private void initIntegerBounds() throws IOException {
        this.findBoundsVariable();
        if (this.boundsVar_ == null) {
            return;
        }
        double modulo = this.findModulo();
        Array boundsArray = this.boundsVar_.read();
        Index boundsIndex = boundsArray.getIndex();
        for (int i = 0; i < this.length_; ++i) {
            int v1 = boundsArray.getInt(boundsIndex.set(i, 0));
            int v2 = boundsArray.getInt(boundsIndex.set(i, 1));
            if (modulo > 0.0) {
                v1 %= (int)modulo;
                v2 %= (int)modulo;
            }
            this.bounds_[i][0] = v1;
            this.bounds_[i][1] = v2;
        }
    }

    private void initLongValues() throws IOException {
        Array dimArray = this.njvar_.read();
        for (int i = 0; i < this.length_; ++i) {
            long value = dimArray.getLong(i);
            this.values_[i] = value;
            this.formattedValues_[i] = this.values_[i].toString();
        }
        this.initLongBounds();
    }

    private void initLongBounds() throws IOException {
        this.findBoundsVariable();
        if (this.boundsVar_ == null) {
            return;
        }
        double modulo = this.findModulo();
        Array boundsArray = this.boundsVar_.read();
        Index boundsIndex = boundsArray.getIndex();
        for (int i = 0; i < this.length_; ++i) {
            long v1 = boundsArray.getLong(boundsIndex.set(i, 0));
            long v2 = boundsArray.getLong(boundsIndex.set(i, 1));
            if (modulo > 0.0) {
                v1 %= (long)modulo;
                v2 %= (long)modulo;
            }
            this.bounds_[i][0] = v1;
            this.bounds_[i][1] = v2;
        }
    }

    private void initObjectValues() throws IOException {
        Array dimArray = this.njvar_.read();
        Index dimIndex = dimArray.getIndex();
        int[] shape = this.njvar_.getShape();
        for (int i = 0; i < this.length_; ++i) {
            Object o = dimArray.getObject(dimIndex.set(i));
            if (o != null && !o.toString().isEmpty()) {
                this.formattedValues_[i] = o.toString();
                this.values_[i] = o;
                continue;
            }
            this.formattedValues_[i] = Integer.toString(i);
            this.values_[i] = this.formattedValues_[i];
        }
    }

    private void initRelativeTimeValues() throws IOException {
        Array dimArray = this.njvar_.read();
        DataType dtype = this.njvar_.getDataType();
        switch (dtype) {
            case INT: 
            case SHORT: {
                for (int i = 0; i < this.length_; ++i) {
                    this.values_[i] = dimArray.getInt(i);
                }
                break;
            }
            case LONG: {
                for (int i = 0; i < this.length_; ++i) {
                    this.values_[i] = dimArray.getLong(i);
                }
                break;
            }
            default: {
                double[] vv = (double[])dimArray.get1DJavaArray(DataType.DOUBLE);
                for (int i = 0; i < this.length_; ++i) {
                    if (Double.isNaN(vv[i])) {
                        throw new NcException("Time variable includes NaN value(s).");
                    }
                    this.values_[i] = vv[i];
                }
            }
        }
        for (int i = 0; i < this.length_; ++i) {
            this.formattedValues_[i] = this.timeParser_.format((Number)this.values_[i]);
        }
        this.initRelativeTimeBounds();
    }

    private void initRelativeTimeBounds() throws IOException {
        this.findBoundsVariable();
        if (this.boundsVar_ == null) {
            return;
        }
        Array boundsArray = this.boundsVar_.read();
        Index boundsIndex = boundsArray.getIndex();
        if (this.njvar_.getDataType() == DataType.INT) {
            for (int i = 0; i < this.length_; ++i) {
                int v1 = boundsArray.getInt(boundsIndex.set(i, 0));
                int v2 = boundsArray.getInt(boundsIndex.set(i, 1));
                this.bounds_[i][0] = this.timeParser_.format(v1);
                this.bounds_[i][1] = this.timeParser_.format(v2);
            }
        } else {
            for (int i = 0; i < this.length_; ++i) {
                double v1 = boundsArray.getDouble(boundsIndex.set(i, 0));
                double v2 = boundsArray.getDouble(boundsIndex.set(i, 1));
                if (Double.isNaN(v1) || Double.isNaN(v2)) {
                    throw new NcException("Time bounds variable includes NaN value(s).");
                }
                this.bounds_[i][0] = this.timeParser_.format(v1);
                this.bounds_[i][1] = this.timeParser_.format(v2);
            }
        }
    }

    private void initJulianTimeValues() throws IOException {
        Array dimArray = this.njvar_.read();
        if (this.njvar_.getDataType() == DataType.INT || this.njvar_.getDataType() == DataType.SHORT) {
            for (int i = 0; i < this.length_; ++i) {
                int value = dimArray.getInt(i);
                this.values_[i] = value;
                this.formattedValues_[i] = this.timeParser_.format(value);
            }
        } else {
            for (int i = 0; i < this.length_; ++i) {
                double value = dimArray.getDouble(i);
                if (Double.isNaN(value)) {
                    throw new NcException("Time variable includes NaN value(s).");
                }
                this.values_[i] = value;
                this.formattedValues_[i] = this.timeParser_.format(value);
            }
        }
        this.initJulianTimeBounds();
    }

    private void initJulianTimeBounds() throws IOException {
        this.findBoundsVariable();
        if (this.boundsVar_ == null) {
            return;
        }
        Array boundsArray = this.boundsVar_.read();
        Index boundsIndex = boundsArray.getIndex();
        if (this.njvar_.getDataType() == DataType.INT || this.njvar_.getDataType() == DataType.SHORT) {
            for (int i = 0; i < this.length_; ++i) {
                int v1 = boundsArray.getInt(boundsIndex.set(i, 0));
                int v2 = boundsArray.getInt(boundsIndex.set(i, 1));
                this.bounds_[i][0] = this.timeParser_.format(v1);
                this.bounds_[i][1] = this.timeParser_.format(v2);
            }
        } else {
            for (int i = 0; i < this.length_; ++i) {
                double v1 = boundsArray.getDouble(boundsIndex.set(i, 0));
                double v2 = boundsArray.getDouble(boundsIndex.set(i, 1));
                if (Double.isNaN(v1) || Double.isNaN(v2)) {
                    throw new NcException("Time bounds variable includes NaN value(s).");
                }
                this.bounds_[i][0] = this.timeParser_.format(v1);
                this.bounds_[i][1] = this.timeParser_.format(v2);
            }
        }
    }

    private double findModulo() {
        double modulo = 0.0;
        if (this.njvar_.findAttribute("topology") != null && this.njvar_.findAttribute("topology").getStringValue().equals("circular") && this.njvar_.findAttribute("modulo") != null) {
            modulo = this.njvar_.findAttribute("modulo").getNumericValue().doubleValue();
        }
        return modulo;
    }

    private void findBoundsVariable() {
        Attribute ba;
        if (this.boundsName_ == null && (ba = this.njvar_.findAttribute("bounds")) != null) {
            this.boundsName_ = ba.getStringValue();
        }
        if (this.boundsName_ == null && (ba = this.njvar_.findAttribute("climatology")) != null) {
            this.boundsName_ = ba.getStringValue();
        }
        if (this.boundsName_ != null) {
            this.boundsVar_ = this.dataset_.getVariableDS(this.boundsName_);
        }
        if (this.boundsVar_ != null) {
            int rank = this.boundsVar_.getRank();
            int[] shape = this.boundsVar_.getShape();
            if (rank == 2 && shape[0] == this.length_ && shape[1] == 2) {
                this.bounds_ = new Object[this.length_][2];
            } else {
                LOGGER.warn("Dimension variable {} has unintelligible bounds.", (Object)this.njvar_.getFullName());
                this.boundsVar_ = null;
            }
        }
    }
}

