/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.data.nc;

import gov.nasa.giss.data.nc.NcAxisType;
import gov.nasa.giss.data.nc.NcDataset;
import gov.nasa.giss.data.nc.NcDimensionVariable;
import gov.nasa.giss.data.nc.NcException;
import gov.nasa.giss.data.nc.NcUtils;
import java.lang.invoke.MethodHandles;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.ma2.Array;
import ucar.nc2.Attribute;
import ucar.nc2.Variable;
import ucar.nc2.constants.AxisType;
import ucar.nc2.dataset.CoordinateAxis;
import ucar.nc2.dataset.VariableDS;
import ucar.units.DerivedUnit;
import ucar.units.Unit;

public class NcAxis {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private AxisType njType_;
    protected NcDataset dataset_;
    protected Variable njvar_;
    private final String fullName_;
    private final String sname_;
    private NcDimensionVariable dimvar_;
    private int length_;
    private double[] values_;
    private double[][] bounds_;
    private boolean wraparound_;
    private int redundant_;
    private String unitsStr_;

    public NcAxis(NcAxisType type, String name, int length) {
        LOGGER.trace("Constructor using name {} and length", (Object)name);
        this.sname_ = name;
        this.fullName_ = name;
        this.length_ = length;
        this.njType_ = NcAxis.getNjAxisType(type, null);
        this.values_ = new double[this.length_];
        this.bounds_ = new double[this.length_][2];
        for (int i = 0; i < this.length_; ++i) {
            double val;
            this.values_[i] = val = (double)i;
            this.bounds_[i][0] = val - 0.5;
            this.bounds_[i][1] = val + 0.5;
        }
    }

    public NcAxis(NcAxisType type, NcDataset dataset, Variable njvar) {
        LOGGER.trace("Constructor using Variable, {}", (Object)njvar.getFullName());
        Objects.requireNonNull(type, "Axis type cannot be null.");
        Objects.requireNonNull(dataset, "Dataset cannot be null.");
        Objects.requireNonNull(njvar, "NJ variable cannot be null.");
        this.sname_ = njvar.getShortName();
        LOGGER.trace(":: {}, axis type {}, data type {}", new Object[]{this.sname_, type, njvar.getDataType()});
        int rank = njvar.getRank();
        if (rank == 0) {
            throw new IllegalArgumentException(String.format("Axis var '%s' is a scalar.", this.sname_));
        }
        if (rank > 1) {
            throw new IllegalArgumentException(String.format("Axis var '%s' is > 1D.", this.sname_));
        }
        this.dimvar_ = new NcDimensionVariable(dataset, njvar);
        this.dataset_ = dataset;
        this.njvar_ = njvar;
        this.fullName_ = njvar.getFullName();
        if (njvar instanceof CoordinateAxis) {
            this.njType_ = ((CoordinateAxis)njvar).getAxisType();
        }
        if (this.njType_ == null) {
            this.njType_ = NcAxis.getNjAxisType(type, njvar);
        }
        this.length_ = njvar.getShape()[0];
        this.values_ = new double[this.length_];
        try {
            Array aa = njvar.read();
            this.initValues(aa);
        }
        catch (NcException exc) {
            LOGGER.error("Could not read var {} -- {}", (Object)this.sname_, (Object)exc.getMessage());
            throw exc;
        }
        catch (Exception exc) {
            String excStr = exc.getMessage();
            LOGGER.error("Could not read var {} -- {}", (Object)this.sname_, (Object)excStr);
            if (LOGGER.isTraceEnabled()) {
                exc.printStackTrace();
            }
            throw new NcException(String.format("Could not read var %s: %s ", this.sname_, excStr));
        }
        catch (Error err) {
            LOGGER.error("Major error, {}", (Object)err.getMessage());
            throw err;
        }
        LOGGER.trace("read");
    }

    public NcAxis(NcAxisType type, String name, double[] values) {
        LOGGER.trace("Constructor using double[] array.");
        this.fullName_ = name;
        this.sname_ = name;
        this.njType_ = NcAxis.getNjAxisType(type, null);
        this.length_ = values.length;
        this.values_ = new double[this.length_];
        this.bounds_ = new double[this.length_][2];
        if (this.length_ == 1) {
            this.values_[0] = values[0];
            if (this.njType_ == AxisType.Lon) {
                this.bounds_[0][0] = values[0] - 180.0;
                this.bounds_[0][1] = values[0] + 180.0;
            } else {
                this.bounds_[0][0] = values[0];
                this.bounds_[0][1] = values[0];
            }
        } else {
            int last = this.length_ - 1;
            for (int i = 0; i < this.length_; ++i) {
                this.values_[i] = values[i];
                if (i == 0) {
                    this.bounds_[0][0] = values[0];
                    this.bounds_[i][1] = 0.5 * (values[0] + values[1]);
                    continue;
                }
                if (i == last) {
                    this.bounds_[i][0] = 0.5 * (values[last - 1] + values[last]);
                    this.bounds_[i][1] = values[last];
                    continue;
                }
                this.bounds_[i][0] = 0.5 * (values[i - 1] + values[i]);
                this.bounds_[i][1] = 0.5 * (values[i] + values[i + 1]);
            }
            if (this.njType_ == AxisType.Lon) {
                double gap1 = values[1] - values[0];
                double gapX = values[last] - values[last - 1];
                this.bounds_[0][0] = values[0] - gap1;
                this.bounds_[last][1] = values[last] + gapX;
            }
        }
    }

    private void initValues(Array vArray) {
        int j;
        LOGGER.trace("getting vals");
        if (this.njvar_ instanceof CoordinateAxis) {
            LOGGER.trace("valid min {}, max {}", (Object)((CoordinateAxis)this.njvar_).getValidMin(), (Object)((CoordinateAxis)this.njvar_).getValidMax());
        }
        for (j = 0; j < this.length_; ++j) {
            double value = vArray.getDouble(vArray.getIndex().set(j));
            if (Double.isNaN(value)) {
                LOGGER.error("NaN found :: {}, {}", (Object)j, (Object)value);
                throw new NcException("Axis " + this.sname_ + " includes NaN value(s).");
            }
            if (j > 0 && value == this.values_[j - 1]) {
                LOGGER.error("Not unique :: {}, {} :: {}, {}", j - 1, this.values_[j - 1], j, value);
                throw new NcException("Axis variable " + this.sname_ + " does not have unique values.");
            }
            this.values_[j] = value;
        }
        if (this.njType_ == AxisType.Lon) {
            LOGGER.trace("testing that lon vals are monotonic");
            for (j = 1; j < this.length_; ++j) {
                if (!(Math.abs(this.values_[j] - this.values_[j - 1]) > 180.0)) continue;
                if (this.values_[j] < this.values_[j - 1]) {
                    int n = j;
                    this.values_[n] = this.values_[n] + 360.0;
                    continue;
                }
                if (!(this.values_[j] > this.values_[j - 1])) continue;
                int n = j;
                this.values_[n] = this.values_[n] - 360.0;
            }
        }
        LOGGER.trace("looking for bounds");
        this.bounds_ = new double[this.length_][2];
        boolean gotBounds = this.readBounds();
        if (!gotBounds) {
            gotBounds = this.readEdges();
        }
        if (!gotBounds) {
            gotBounds = this.calculateBounds();
        }
        if (!gotBounds) {
            LOGGER.trace("no bounds found");
            throw new NcException("Unable to extract axis value bounds.");
        }
        LOGGER.trace("bounds found -- validating");
        this.fixBounds();
    }

    private boolean readBounds() {
        Array bArray;
        Attribute ba;
        if (this.njvar_ == null) {
            return false;
        }
        LOGGER.trace("");
        String boundsName = null;
        if (this.njvar_ instanceof CoordinateAxis) {
            LOGGER.trace("trying getBoundaryRef");
            boundsName = ((CoordinateAxis)this.njvar_).getBoundaryRef();
        }
        if (boundsName == null) {
            LOGGER.trace("trying bounds attribute");
            ba = this.njvar_.findAttribute("bounds");
            if (ba != null) {
                boundsName = ba.getStringValue();
            }
        }
        if (boundsName == null) {
            LOGGER.trace("trying climatology attribute");
            ba = this.njvar_.findAttribute("climatology");
            if (ba != null) {
                boundsName = ba.getStringValue();
            }
        }
        if (boundsName == null) {
            LOGGER.trace("no bounds name");
            return false;
        }
        LOGGER.trace("looking for bounds var '{}'", (Object)boundsName);
        VariableDS boundsVar = this.dataset_.getVariableDS(boundsName);
        if (boundsVar == null) {
            LOGGER.trace("not found");
            return false;
        }
        try {
            bArray = boundsVar.read();
        }
        catch (Exception exc) {
            LOGGER.warn("Exception reading axis bounds {}", (Object)exc.getMessage());
            throw new NcException(String.format("Could not read axis bounds $s", boundsName));
        }
        LOGGER.trace("validate bounds indices order");
        int[] bshape = boundsVar.getShape();
        if (bshape[0] == 2 && bshape[1] > 2) {
            bArray = bArray.transpose(0, 1);
        }
        try {
            if (this.length_ > 1) {
                double val0 = this.values_[0];
                double val1 = this.values_[1];
                double b10 = bArray.getDouble(bArray.getIndex().set(1, 0));
                double b11 = bArray.getDouble(bArray.getIndex().set(1, 1));
                if (Double.isNaN(b10)) {
                    b10 = 0.0;
                }
                if (Double.isNaN(b11)) {
                    b11 = 0.0;
                }
                if (b10 < val0 && b10 < val1 || b10 > val0 && b10 > val1) {
                    bArray = bArray.flip(1);
                }
            }
            for (int i = 0; i < this.length_; ++i) {
                double b0 = bArray.getDouble(bArray.getIndex().set(i, 0));
                double b1 = bArray.getDouble(bArray.getIndex().set(i, 1));
                if (Double.isNaN(b0)) {
                    b0 = 0.0;
                }
                if (Double.isNaN(b1)) {
                    b1 = 0.0;
                }
                if (this.njType_ == AxisType.Lon) {
                    if (Math.abs(b0) > 1080.0 || Math.abs(b1) > 1080.0) {
                        throw new NcException("Values in longitude bounds variable look unreasonable");
                    }
                    while (b1 < this.values_[i]) {
                        b1 += 360.0;
                    }
                    while (b0 > this.values_[i]) {
                        b0 -= 360.0;
                    }
                    while (b1 < this.values_[i]) {
                        b1 += 360.0;
                    }
                    while (this.values_[i] - b0 > 360.0) {
                        b0 += 360.0;
                    }
                    while (b1 - this.values_[i] > 360.0) {
                        b1 -= 360.0;
                    }
                } else if (this.njType_ == AxisType.Lat && (Math.abs(b0) > 91.0 || Math.abs(b1) > 91.0)) {
                    throw new NcException("Values in latitude bounds variable look unreasonable");
                }
                if (b0 == b1) {
                    // empty if block
                }
                this.bounds_[i][0] = b0;
                this.bounds_[i][1] = b1;
            }
            LOGGER.trace("apparently good");
            return true;
        }
        catch (NcException ncx) {
            LOGGER.warn("NcException reported: {}", (Object)ncx.getMessage());
            throw ncx;
        }
        catch (Exception exc) {
            LOGGER.warn("Failed to read axis bound values from dataset.");
            LOGGER.warn("Exception reported: {}", (Object)exc.toString());
            LOGGER.trace("not good");
            return false;
        }
    }

    private boolean readEdges() {
        Array eArray;
        if (this.njvar_ == null) {
            return false;
        }
        LOGGER.trace("");
        Attribute edgesA = this.njvar_.findAttribute("edges");
        if (edgesA == null) {
            return false;
        }
        String edgesName = edgesA.getStringValue();
        if (edgesName == null) {
            return false;
        }
        VariableDS edgesVar = this.dataset_.getVariableDS(edgesName);
        try {
            eArray = edgesVar.read();
        }
        catch (Exception exc) {
            throw new NcException(String.format("Could not read axis edges %s", edgesName));
        }
        for (int i = 0; i < this.length_; ++i) {
            double b0 = eArray.getDouble(eArray.getIndex().set(i));
            double b1 = eArray.getDouble(eArray.getIndex().set(i + 1));
            if (Double.isNaN(b0)) {
                b0 = 0.0;
            }
            if (Double.isNaN(b1)) {
                b1 = 0.0;
            }
            if (this.njType_ == AxisType.Lon) {
                while (b0 > this.values_[i]) {
                    b0 -= 360.0;
                }
                while (b1 < this.values_[i]) {
                    b1 += 360.0;
                }
                while (this.values_[i] - b0 > 360.0) {
                    b0 += 360.0;
                }
                while (b1 - this.values_[i] > 360.0) {
                    b1 -= 360.0;
                }
            }
            this.bounds_[i][0] = b0;
            this.bounds_[i][1] = b1;
        }
        return true;
    }

    private boolean calculateBounds() {
        LOGGER.trace("");
        if (this.length_ > 1) {
            int last = this.length_ - 1;
            for (int i = 0; i < last; ++i) {
                this.bounds_[i][1] = 0.5 * (this.values_[i] + this.values_[i + 1]);
                this.bounds_[i + 1][0] = this.bounds_[i][1];
            }
            this.bounds_[0][0] = 2.0 * this.values_[0] - this.bounds_[0][1];
            this.bounds_[last][1] = 2.0 * this.values_[last] - this.bounds_[last][0];
            if (this.isPressure()) {
                if (this.bounds_[0][0] <= 0.0) {
                    this.bounds_[0][0] = this.values_[0];
                }
                if (this.bounds_[last][1] <= 0.0) {
                    this.bounds_[last][1] = this.values_[last];
                }
            }
        } else if (this.njType_ == AxisType.Lon) {
            this.bounds_[0][0] = this.values_[0] - 5.0;
            this.bounds_[0][1] = this.values_[0] + 5.0;
        } else if (this.njType_ == AxisType.Lat) {
            this.bounds_[0][0] = this.values_[0] + 5.0;
            this.bounds_[0][1] = this.values_[0] - 5.0;
        } else {
            this.bounds_[0][0] = this.values_[0] - 5.0;
            this.bounds_[0][1] = this.values_[0] + 5.0;
        }
        return true;
    }

    private void fixBounds() {
        if (this.njType_ != AxisType.Lon) {
            LOGGER.trace("not lon -- no need to validate");
            return;
        }
        int last = this.length_ - 1;
        if (this.length_ > 2) {
            if (this.values_[last - 1] >= this.values_[0] + 360.0) {
                this.redundant_ = 2;
            } else if (this.values_[last] >= this.values_[0] + 360.0) {
                this.redundant_ = 1;
            }
            if (this.redundant_ > 0) {
                this.length_ -= this.redundant_;
                last = this.length_ - 1;
                double[] newvals = new double[this.length_];
                double[][] newbounds = new double[this.length_][2];
                for (int i = 0; i < this.length_; ++i) {
                    newvals[i] = this.values_[i];
                    newbounds[i][0] = this.bounds_[i][0];
                    newbounds[i][1] = this.bounds_[i][1];
                }
                this.values_ = newvals;
                this.bounds_ = newbounds;
            }
        }
        if (this.length_ > 1) {
            double delta = this.values_[last] - this.values_[last - 1];
            double overlap = this.bounds_[last][1] - (this.bounds_[0][0] + 360.0);
            boolean bl = this.wraparound_ = overlap >= 0.0 || Math.abs(overlap / delta) < 0.2;
            if (overlap > 0.0) {
                double[] dArray = this.bounds_[0];
                dArray[0] = dArray[0] + (overlap *= 0.5);
                double[] dArray2 = this.bounds_[last];
                dArray2[1] = dArray2[1] - overlap;
            }
        }
        for (int i = 0; i < this.length_; ++i) {
            while (this.bounds_[i][0] > this.values_[i]) {
                double[] dArray = this.bounds_[i];
                dArray[0] = dArray[0] - 360.0;
            }
            while (this.values_[i] - this.bounds_[i][0] >= 360.0) {
                double[] dArray = this.bounds_[i];
                dArray[0] = dArray[0] + 360.0;
            }
            while (this.bounds_[i][1] < this.values_[i]) {
                double[] dArray = this.bounds_[i];
                dArray[1] = dArray[1] + 360.0;
            }
            while (this.bounds_[i][1] - this.values_[i] >= 360.0) {
                double[] dArray = this.bounds_[i];
                dArray[1] = dArray[1] - 360.0;
            }
        }
    }

    public String getShortName() {
        return this.sname_;
    }

    public Variable getNjAxisVar() {
        return this.njvar_;
    }

    public boolean isLongitudeAxis() {
        return this.njType_ == AxisType.Lon;
    }

    public boolean isLatitudeAxis() {
        return this.njType_ == AxisType.Lat;
    }

    public boolean hasPositive() {
        if (this.njType_ == AxisType.Pressure) {
            return true;
        }
        if (this.njType_ == AxisType.GeoZ || this.njType_ == AxisType.Height || this.njType_ == AxisType.Pressure) {
            if (this.njvar_ instanceof CoordinateAxis) {
                String p = ((CoordinateAxis)this.njvar_).getPositive();
                return p != null;
            }
            if (this.njvar_ != null) {
                String p = NcUtils.getAttributeStringIgnoreCase(this.njvar_, "positive");
                return p != null;
            }
        }
        return false;
    }

    public boolean isPositiveUp() {
        if (this.njType_ == AxisType.Pressure) {
            return false;
        }
        if (this.njType_ == AxisType.GeoZ || this.njType_ == AxisType.Height) {
            if (this.njvar_ instanceof CoordinateAxis) {
                String p = ((CoordinateAxis)this.njvar_).getPositive();
                return p != null && p.equals("up");
            }
            if (this.njvar_ != null) {
                String p = NcUtils.getAttributeStringIgnoreCase(this.njvar_, "positive");
                return "up".equalsIgnoreCase(p);
            }
        }
        return false;
    }

    public boolean isPositiveDown() {
        if (this.njType_ == AxisType.Pressure) {
            LOGGER.trace("PRESSURE");
            return true;
        }
        if (this.njType_ == AxisType.GeoZ || this.njType_ == AxisType.Height) {
            LOGGER.trace("Z/HEIGHT");
            if (this.njvar_ instanceof CoordinateAxis) {
                String p = ((CoordinateAxis)this.njvar_).getPositive();
                LOGGER.trace("p? {}", (Object)p);
                return p != null && p.equals("down");
            }
            if (this.njvar_ != null) {
                String p = NcUtils.getAttributeStringIgnoreCase(this.njvar_, "positive");
                LOGGER.trace("a? {}", (Object)p);
                return p != null && "down".equals(p);
            }
        }
        return false;
    }

    public boolean isPressure() {
        if (this.njType_ == AxisType.Pressure) {
            return true;
        }
        Unit uu = this.getUdunits();
        if (uu == null) {
            return false;
        }
        DerivedUnit du = uu.getDerivedUnit();
        if (du == null) {
            return false;
        }
        return "Pa".equals(du.toString());
    }

    public boolean isHeight() {
        if (this.njType_ == AxisType.Height) {
            return true;
        }
        Unit uu = this.getUdunits();
        if (uu == null) {
            return false;
        }
        DerivedUnit du = uu.getDerivedUnit();
        if (du == null) {
            return false;
        }
        return "m".equals(du.toString());
    }

    public boolean isDepth() {
        Unit uu = this.getUdunits();
        if (uu == null) {
            return false;
        }
        DerivedUnit du = uu.getDerivedUnit();
        if (du == null) {
            return false;
        }
        boolean isDist = du.toString().equals("m");
        if (!isDist) {
            return false;
        }
        return this.isPositiveDown();
    }

    public int getLength() {
        return this.values_.length;
    }

    public double[] getValues() {
        return this.values_;
    }

    public double[][] getBounds() {
        return this.bounds_;
    }

    public String getLongName() {
        if (this.njvar_ != null) {
            return NcUtils.getLongName(this.njvar_);
        }
        if (this.fullName_ != null && !this.fullName_.isEmpty()) {
            return this.fullName_;
        }
        if (this.njType_ == AxisType.GeoX || this.njType_ == AxisType.GeoY) {
            return "Axis " + this.getLength();
        }
        return "";
    }

    public String getUnitsStr() {
        if (this.unitsStr_ == null) {
            this.unitsStr_ = this.njvar_ == null ? "" : NcUtils.getUnitsStr(this.njvar_);
        }
        return this.unitsStr_;
    }

    public Unit getUdunits() {
        return NcUtils.getUdunits(this.njvar_);
    }

    public double valueAt(int index) {
        if (index < 0 || index > this.values_.length) {
            throw new IndexOutOfBoundsException("Axis index out of range.");
        }
        return this.values_[index];
    }

    public double[] boundsAt(int index) {
        if (index < 0 || index > this.values_.length) {
            throw new IndexOutOfBoundsException("Axis dimension index out of range.");
        }
        return new double[]{this.bounds_[index][0], this.bounds_[index][1]};
    }

    public int findNearestIndex(double value) {
        double xvalue;
        if (this.njType_ == AxisType.Lat && (value < -90.0 || value > 90.0)) {
            return -1;
        }
        if (this.njType_ == AxisType.Lon) {
            int last = this.length_ - 1;
            for (xvalue = value; xvalue < this.bounds_[0][0] && xvalue < this.bounds_[last][1]; xvalue += 360.0) {
            }
            while (xvalue > this.bounds_[0][0] && xvalue > this.bounds_[last][1]) {
                xvalue -= 360.0;
            }
        }
        for (int i = 0; i < this.length_; ++i) {
            if (xvalue < this.bounds_[i][0] && xvalue < this.bounds_[i][1] || xvalue > this.bounds_[i][0] && xvalue > this.bounds_[i][1]) continue;
            return i;
        }
        return -1;
    }

    public boolean isWraparound() {
        return this.wraparound_;
    }

    public boolean isCompatible(NcAxis pa) {
        Objects.requireNonNull(pa, "Null axis.");
        return this.isCompatible(pa.getUdunits());
    }

    public boolean isCompatible(Unit u2) {
        Unit u1 = this.getUdunits();
        if (u1 == null && u2 == null) {
            return true;
        }
        if (u1 == null || u2 == null) {
            return false;
        }
        return u2.isCompatible(u1);
    }

    public boolean equals(Object o) {
        if (o == null) {
            return false;
        }
        if (o == this) {
            return true;
        }
        if (!(o instanceof NcAxis)) {
            return false;
        }
        NcAxis a2 = (NcAxis)o;
        if (a2.getLength() != this.getLength()) {
            return false;
        }
        if (!a2.getUnitsStr().equals(this.getUnitsStr())) {
            return false;
        }
        double[] a2Values = a2.getValues();
        double[][] a2Bounds = a2.getBounds();
        for (int i = 0; i < this.values_.length; ++i) {
            if (a2Values[i] != this.values_[i]) {
                return false;
            }
            if (a2Bounds[i][0] != this.bounds_[i][0]) {
                return false;
            }
            if (a2Bounds[i][1] == this.bounds_[i][1]) continue;
            return false;
        }
        return true;
    }

    public int hashCode() {
        if (this.njvar_ != null) {
            return this.njvar_.hashCode();
        }
        return this.values_.hashCode();
    }

    private static AxisType getNjAxisType(NcAxisType type, Variable njvar) {
        switch (type) {
            case LON: {
                return AxisType.Lon;
            }
            case LAT: {
                return AxisType.Lat;
            }
            case VERT: {
                if (NcUtils.isPressure(njvar)) {
                    return AxisType.Pressure;
                }
                if (NcUtils.isHeight(njvar)) {
                    return AxisType.Height;
                }
                return AxisType.GeoZ;
            }
            case TIME: {
                return AxisType.Time;
            }
            case GEOX: 
            case X: {
                return AxisType.GeoX;
            }
            case GEOY: 
            case Y: {
                return AxisType.GeoY;
            }
            case SPECTRAL: {
                return AxisType.Spectral;
            }
        }
        throw new IllegalArgumentException("Unknown axis type " + type);
    }
}

