"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.MjolnirAppService = void 0;
const matrix_appservice_bridge_1 = require("matrix-appservice-bridge");
const PgDataStore_1 = require("./postgres/PgDataStore");
const Api_1 = require("./Api");
const AccessControl_1 = require("./AccessControl");
const AppserviceCommandHandler_1 = require("./bot/AppserviceCommandHandler");
const config_1 = require("../config");
const prom_client_1 = require("prom-client");
const matrix_protection_suite_for_matrix_bot_sdk_1 = require("matrix-protection-suite-for-matrix-bot-sdk");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const AppServiceDraupnirManager_1 = require("./AppServiceDraupnirManager");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const SqliteRoomStateBackingStore_1 = require("../backingstore/better-sqlite3/SqliteRoomStateBackingStore");
const utils_1 = require("../utils");
const log = new matrix_appservice_bridge_1.Logger("AppService");
/**
 * Responsible for setting up listeners and delegating functionality to a matrix-appservice-bridge `Bridge` for
 * the entrypoint of the application.
 */
class MjolnirAppService {
    /**
     * The constructor is private because we want to ensure intialization steps are followed,
     * use `makeMjolnirAppService`.
     */
    constructor(config, bridge, draupnirManager, accessControl, dataStore, eventDecoder, roomStateManagerFactory, clientCapabilityFactory, clientsInRoomMap, prometheusMetrics, accessControlRoomID, botUserID) {
        this.config = config;
        this.bridge = bridge;
        this.draupnirManager = draupnirManager;
        this.accessControl = accessControl;
        this.dataStore = dataStore;
        this.eventDecoder = eventDecoder;
        this.roomStateManagerFactory = roomStateManagerFactory;
        this.clientCapabilityFactory = clientCapabilityFactory;
        this.clientsInRoomMap = clientsInRoomMap;
        this.prometheusMetrics = prometheusMetrics;
        this.accessControlRoomID = accessControlRoomID;
        this.botUserID = botUserID;
        this.api = new Api_1.Api(config.homeserver.url, draupnirManager);
        const client = this.bridge.getBot().getClient();
        this.commands = new AppserviceCommandHandler_1.AppserviceCommandHandler(botUserID, client, accessControlRoomID, this.clientCapabilityFactory.makeClientPlatform(botUserID, client), this);
    }
    /**
     * Make and initialize the app service from the config, ready to be started.
     * @param config The appservice's config, not draupnirs's, see `src/appservice/config`.
     * @param dataStore A datastore to persist infomration about the draupnir to.
     * @param registrationFilePath A file path to the registration file to read the namespace and tokens from.
     * @returns A new `MjolnirAppService`.
     */
    static async makeMjolnirAppService(config, dataStore, eventDecoder, registrationFilePath, stores) {
        const bridge = new matrix_appservice_bridge_1.Bridge({
            homeserverUrl: config.homeserver.url,
            domain: config.homeserver.domain,
            registration: registrationFilePath,
            // We lazily initialize the controller to avoid null checks
            // It also allows us to combine constructor/initialize logic
            // to make the code base much simpler. A small hack to pay for an overall less hacky code base.
            controller: {
                onUserQuery: () => {
                    throw new Error("Draupnir uninitialized");
                },
                onEvent: () => {
                    throw new Error("Draupnir uninitialized");
                },
            },
            suppressEcho: false,
            disableStores: true,
        });
        await bridge.initialise();
        const clientsInRoomMap = new matrix_protection_suite_1.StandardClientsInRoomMap();
        const clientProvider = async (clientUserID) => bridge.getIntent(clientUserID).matrixClient;
        const roomStateManagerFactory = new matrix_protection_suite_for_matrix_bot_sdk_1.RoomStateManagerFactory(clientsInRoomMap, clientProvider, eventDecoder, stores.roomStateBackingStore, stores.hashStore);
        const clientCapabilityFactory = new matrix_protection_suite_for_matrix_bot_sdk_1.ClientCapabilityFactory(clientsInRoomMap, eventDecoder);
        const botUserID = bridge.getBot().getUserId();
        (await clientsInRoomMap.makeClientRooms(botUserID, async () => (0, matrix_protection_suite_for_matrix_bot_sdk_1.joinedRoomsSafe)(bridge.getBot().getClient()))).expect("Unable to initialize client rooms for the appservice bot user");
        const botRoomJoiner = clientCapabilityFactory
            .makeClientPlatform(botUserID, bridge.getBot().getClient())
            .toRoomJoiner();
        const adminRoom = (() => {
            if ((0, matrix_basic_types_1.isStringRoomID)(config.adminRoom)) {
                return matrix_basic_types_1.MatrixRoomReference.fromRoomID(config.adminRoom);
            }
            else if ((0, matrix_basic_types_1.isStringRoomAlias)(config.adminRoom)) {
                return matrix_basic_types_1.MatrixRoomReference.fromRoomIDOrAlias(config.adminRoom);
            }
            else {
                const parseResult = matrix_basic_types_1.MatrixRoomReference.fromPermalink(config.adminRoom);
                if ((0, matrix_protection_suite_1.isError)(parseResult)) {
                    throw new TypeError(`${config.adminRoom} needs to be a room id, alias or permalink`);
                }
                return parseResult.ok;
            }
        })();
        const accessControlRoom = (await botRoomJoiner.resolveRoom(adminRoom)).expect("Unable to resolve the admin room");
        const appserviceBotPolicyRoomManager = await roomStateManagerFactory.getPolicyRoomManager(botUserID);
        const accessControl = (await AccessControl_1.AccessControl.setupAccessControlForRoom(accessControlRoom, appserviceBotPolicyRoomManager, botRoomJoiner)).expect("Unable to setup access control for the appservice");
        // Activate /metrics endpoint for Prometheus
        // This should happen automatically but in testing this didn't happen in the docker image
        (0, matrix_appservice_bridge_1.setBridgeVersion)(config_1.SOFTWARE_VERSION);
        // Due to the way the tests and this prom library works we need to explicitly create a new one each time.
        const prometheus = bridge.getPrometheusMetrics(true, new prom_client_1.Registry());
        const instanceCountGauge = prometheus.addGauge({
            name: "draupnir_instances",
            help: "Count of Draupnir Instances",
            labels: ["status", "uuid"],
        });
        const serverName = config.homeserver.domain;
        const mjolnirManager = await AppServiceDraupnirManager_1.AppServiceDraupnirManager.makeDraupnirManager(serverName, dataStore, bridge, accessControl, roomStateManagerFactory, stores, clientCapabilityFactory, clientsInRoomMap, clientProvider, instanceCountGauge);
        const appService = new MjolnirAppService(config, bridge, mjolnirManager, accessControl, dataStore, eventDecoder, roomStateManagerFactory, clientCapabilityFactory, clientsInRoomMap, prometheus, accessControlRoom.toRoomIDOrAlias(), botUserID);
        bridge.opts.controller = {
            onUserQuery: appService.onUserQuery.bind(appService),
            onEvent: appService.onEvent.bind(appService),
        };
        return appService;
    }
    /**
     * Start the appservice for the end user with the appropriate settings from their config and registration file.
     * @param port The port to make the appservice listen for transactions from the homeserver on (usually sourced from the cli).
     * @param config The parsed configuration file.
     * @param registrationFilePath A path to their homeserver registration file.
     */
    static async run(port, config, registrationFilePath) {
        matrix_appservice_bridge_1.Logger.configure(config.logging ?? { console: "debug" });
        (0, utils_1.patchMatrixClient)();
        const dataStore = new PgDataStore_1.PgDataStore(config.db.connectionString);
        await dataStore.init();
        const eventDecoder = matrix_protection_suite_1.DefaultEventDecoder;
        const storagePath = (0, config_1.getStoragePath)(config.dataPath);
        const backingStore = config.roomStateBackingStore?.enabled
            ? SqliteRoomStateBackingStore_1.SqliteRoomStateBackingStore.create(storagePath, eventDecoder)
            : undefined;
        const service = await MjolnirAppService.makeMjolnirAppService(config, dataStore, matrix_protection_suite_1.DefaultEventDecoder, registrationFilePath, {
            roomStateBackingStore: backingStore,
            dispose() {
                backingStore?.destroy();
            },
        } // we don't support any stores in appservice atm except backing store.
        );
        // The call to `start` MUST happen last. As it needs the datastore, and the mjolnir manager to be initialized before it can process events from the homeserver.
        await service.start(port);
        return service;
    }
    onUserQuery(_queriedUser) {
        return {}; // auto-provision users with no additonal data
    }
    // Provision a new draupnir for the invitee when the appservice bot (designated by this.bridge.botUserId) is invited to a room.
    // Acts as an alternative to the web api provided for the widget.
    async handleProvisionInvite(mxEvent) {
        log.info(`${mxEvent.sender} has sent an invitation to the appservice bot ${this.bridge.botUserId}, attempting to provision them a draupnir`);
        // Join the room and try to send the welcome flow
        try {
            await this.bridge.getBot().getClient().joinRoom(mxEvent.room_id);
            await this.bridge
                .getBot()
                .getClient()
                .sendText(mxEvent.room_id, "Your Draupnir is currently being provisioned. Please wait while we set up the rooms.");
        }
        catch (e) {
            log.error(`Failed to join the room by ${mxEvent.sender} to display the welcome flow`);
        }
        try {
            const result = await this.draupnirManager.provisionNewDraupnir(mxEvent.sender);
            if ((0, matrix_protection_suite_1.isError)(result)) {
                log.error(`Failed to provision a draupnir for ${mxEvent.sender} after they invited ${this.bridge.botUserId}`, result.error);
            }
            // Send a notice that the invite must be accepted
            await this.bridge
                .getBot()
                .getClient()
                .sendText(mxEvent.room_id, "Please accept the inviations to the newly provisioned rooms. These will be the home of your Draupnir Instance. This room will not be used in the future.");
        }
        catch (e) {
            log.error(`Failed to provision a draupnir for ${mxEvent.sender} after they invited ${this.bridge.botUserId}:`, e);
            // continue, we still want to reject this invitation.
            // Send a notice that the provisioning failed
            await this.bridge
                .getBot()
                .getClient()
                .sendText(mxEvent.room_id, "Please make sure you are allowed to provision a bot. Otherwise please notify the admin. The provisioning request was rejected.");
        }
        try {
            // reject the invite to keep the room clean and make sure the invetee doesn't get confused and think this is their draupnir.
            await this.bridge.getBot().getClient().leaveRoom(mxEvent.room_id);
        }
        catch (e) {
            log.warn("Unable to reject an invite to a room", e);
        }
    }
    /**
     * Handle an individual event pushed by the homeserver to us.
     * This function is async (and anything downstream would be anyway), which does mean that events can be processed out of order.
     * Not a huge problem for us, but is something to be aware of.
     * @param request A matrix-appservice-bridge request encapsulating a Matrix event.
     * @param context Additional context for the Matrix event.
     */
    onEvent(request) {
        const mxEvent = request.getData();
        if ("m.room.member" === mxEvent.type) {
            if ("invite" === mxEvent.content["membership"]) {
                if (mxEvent.room_id === this.accessControlRoomID) {
                    // do nothing, setup code should handle this.
                }
                else if (mxEvent.state_key === this.bridge.botUserId) {
                    void (0, matrix_protection_suite_1.Task)(this.handleProvisionInvite(mxEvent));
                }
            }
        }
        this.commands.handleEvent(mxEvent);
        const decodeResult = this.eventDecoder.decodeEvent(mxEvent);
        if ((0, matrix_protection_suite_1.isError)(decodeResult)) {
            log.error(`Got an error when decoding an event for the appservice`, decodeResult.error.uuid, decodeResult.error);
            return;
        }
        const roomID = decodeResult.ok.room_id;
        this.roomStateManagerFactory.handleTimelineEvent(roomID, decodeResult.ok);
        this.clientsInRoomMap.handleTimelineEvent(roomID, decodeResult.ok);
    }
    /**
     * Start the appservice. See `run`.
     * @param port The port that the appservice should listen on to receive transactions from the homeserver.
     */
    async start(port) {
        log.info("Starting DraupnirAppService, Matrix-side to listen on port", port);
        this.api.start(this.config.webAPI.port);
        await this.bridge.listen(port);
        this.prometheusMetrics.addAppServicePath(this.bridge);
        this.bridge.addAppServicePath({
            method: "GET",
            path: "/healthz",
            authenticate: false,
            handler: (_req, res) => {
                res.status(200).send("ok");
            },
        });
        log.info("DraupnirAppService started successfully");
    }
    /**
     * Stop listening to requests from both the homeserver and web api and disconnect from the datastore.
     */
    async close() {
        await this.bridge.close();
        await this.dataStore.close();
        await this.api.close();
        this.draupnirManager.unregisterListeners();
    }
    /**
     * Generate a registration file for a fresh deployment of the appservice.
     * Included to satisfy `matrix-appservice-bridge`'s `Cli` utility which allows a registration file to be registered when setting up a deployment of an appservice.
     * @param reg Any existing parameters to be included in the registration, to be mutated by this method.
     * @param callback To call when the registration has been generated with the final registration.
     */
    static generateRegistration(reg, callback) {
        reg.setId(matrix_appservice_bridge_1.AppServiceRegistration.generateToken());
        reg.setHomeserverToken(matrix_appservice_bridge_1.AppServiceRegistration.generateToken());
        reg.setAppServiceToken(matrix_appservice_bridge_1.AppServiceRegistration.generateToken());
        reg.setSenderLocalpart("draupnir-moderation");
        // This is maintained for backwards compatibility with mjolnir4all.
        reg.addRegexPattern("users", "@mjolnir_.*", true);
        reg.addRegexPattern("users", "@draupnir_.*", true);
        reg.setRateLimited(false);
        callback(reg);
    }
}
exports.MjolnirAppService = MjolnirAppService;
//# sourceMappingURL=AppService.js.map