import { randomUUID } from 'crypto';
import { ActionError, ActionResult } from './Action';
/**
 * A way to catagorise different Exceptions.
 */
export declare enum ActionExceptionKind {
    /**
     * This kind is for exceptions that need to be reported to the user,
     * but are mostly irrelevant to the developers because the behaviour is well
     * understood and expected. These exceptions will never be logged to the error
     * level.
     */
    Known = "Known",
    /**
     * This kind is to be used for reporting unexpected or unknown exceptions
     * that the developers need to know about.
     */
    Unknown = "Unknown"
}
/**
 * `ActionExceptions` are used to convert throwables into `ActionError`s.
 * Each `ActionException` is given a unique identifier and is logged immediatley
 * (depending on {@link ActionExceptionKind}).
 *
 * You will want to create these using {@link ActionException.Result}.
 */
export declare class ActionException extends ActionError {
    readonly exceptionKind: ActionExceptionKind;
    readonly exception: unknown;
    readonly uuid: ReturnType<typeof randomUUID>;
    constructor(exceptionKind: ActionExceptionKind, exception: unknown, message: string, { uuid, suppressLog, elaborations, }?: {
        uuid?: ReturnType<typeof randomUUID>;
        suppressLog?: boolean;
        elaborations?: string[];
    });
    /**
     * Convienant factory method for `ActionException`s that will return an
     * `ActionResult`.
     * @param message The message for the `ActionError` that concisely describes the problem.
     * @param options.exception The `Error` that was thrown.
     * @param options.exceptionKind The `ActionExceptionKind` that catagorieses the exception.
     * @returns An `ActionResult` with the exception as the `Error` value.
     */
    static Result(message: string, options: {
        exception: unknown;
        exceptionKind: ActionExceptionKind;
    }): ActionResult<never, ActionException>;
    protected log(): void;
    toReadableString(): string;
}
//# sourceMappingURL=ActionException.d.ts.map