"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardProtectionsManager = void 0;
// Copyright 2023 - 2024 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
const typescript_result_1 = require("@gnuxie/typescript-result");
const CapabilitySet_1 = require("../Capability/CapabilitySet");
const Logger_1 = require("../../Logging/Logger");
const log = new Logger_1.Logger('StandardProtectionsManager');
// FIXME: Dialemma, if we want to be able to change protection settings
// or dry run protections with dummy capabilities, we need to know whether
// the protection has external resources that will conflict.
// So for example, a webserver or something like that, we need to make sure that
// both protections can run at the same time. This would mean duplicating
// the listeners for a webserver and we need to warn protections about this
// in the documentation.
class StandardProtectionsManager {
    constructor(enabledProtectionsConfig, capabilityProviderSetConfig, settingsConfig) {
        this.enabledProtectionsConfig = enabledProtectionsConfig;
        this.capabilityProviderSetConfig = capabilityProviderSetConfig;
        this.settingsConfig = settingsConfig;
        this.enabledProtections = new Map();
        // nothing to do mare.
    }
    get allProtections() {
        return [...this.enabledProtections.values()];
    }
    async startProtection(protectionDescription, protectedRoomsSet, context, { settings, capabilityProviderSet, }) {
        if (settings === undefined) {
            const settingsResult = await this.settingsConfig.getProtectionSettings(protectionDescription);
            if ((0, typescript_result_1.isError)(settingsResult)) {
                return settingsResult;
            }
            settings = settingsResult.ok;
        }
        if (capabilityProviderSet === undefined) {
            const capabilityProviders = await this.capabilityProviderSetConfig.getCapabilityProviderSet(protectionDescription);
            if ((0, typescript_result_1.isError)(capabilityProviders)) {
                return capabilityProviders.elaborate(`Couldn't find the capability provider set for ${protectionDescription.name}`);
            }
            capabilityProviderSet = capabilityProviders.ok;
        }
        const capabilities = (0, CapabilitySet_1.initializeCapabilitySet)(protectionDescription, capabilityProviderSet, context);
        const protectionResult = await protectionDescription.factory(protectionDescription, protectedRoomsSet, context, capabilities, settings);
        if ((0, typescript_result_1.isError)(protectionResult)) {
            return protectionResult;
        }
        const enabledProtection = this.enabledProtections.get(protectionDescription.name);
        if (enabledProtection !== undefined) {
            this.removeProtectionWithoutStore(protectionDescription);
        }
        this.enabledProtections.set(protectionDescription.name, protectionResult.ok);
        return protectionResult;
    }
    async addProtection(protectionDescription, protectedRoomsSet, context) {
        const startResult = await this.startProtection(protectionDescription, protectedRoomsSet, context, {});
        if ((0, typescript_result_1.isError)(startResult)) {
            return startResult;
        }
        const storeResult = await this.enabledProtectionsConfig.enableProtection(protectionDescription);
        return storeResult;
    }
    removeProtectionWithoutStore(protectionDescription) {
        var _a;
        const protection = this.enabledProtections.get(protectionDescription.name);
        this.enabledProtections.delete(protectionDescription.name);
        if (protection !== undefined) {
            try {
                (_a = protection.handleProtectionDisable) === null || _a === void 0 ? void 0 : _a.call(protection);
            }
            catch (ex) {
                log.error(`Caught unhandled exception while disabling ${protectionDescription.name}:`, ex);
            }
        }
    }
    async removeProtection(protection) {
        const storeResult = await this.enabledProtectionsConfig.disableProtection(protection.name);
        if ((0, typescript_result_1.isError)(storeResult)) {
            return storeResult;
        }
        this.removeProtectionWithoutStore(protection);
        return (0, typescript_result_1.Ok)(undefined);
    }
    async loadProtections(protectedRoomsSet, context, protectionFailedToStart) {
        if (this.allProtections.length > 0) {
            throw new TypeError('This can only be used at startup');
        }
        for (const protectionDescription of this.enabledProtectionsConfig.getKnownEnabledProtections()) {
            const startResult = await this.startProtection(protectionDescription, protectedRoomsSet, context, {});
            if ((0, typescript_result_1.isError)(startResult)) {
                await protectionFailedToStart(startResult.error, protectionDescription.name, protectionDescription);
                continue;
            }
        }
        return (0, typescript_result_1.Ok)(undefined);
    }
    async changeProtectionSettings(protectionDescription, protectedRoomsSet, context, settings) {
        // It is important that we check that storing the settings is successful
        // BEFORE enabling the new protection. This is to make sure that a cascade
        // failure cannot occur when a protection can modify its own settings at
        // startup.
        // The current settings are used to restore the previous state if the protection
        // cannot start through a config use error.
        const currentSettings = await this.settingsConfig.getProtectionSettings(protectionDescription);
        if ((0, typescript_result_1.isError)(currentSettings)) {
            return currentSettings.elaborate("Couldn't fetch the current settings so they cannot be changed");
        }
        const settingsResult = await this.settingsConfig.storeProtectionSettings(protectionDescription, settings);
        if ((0, typescript_result_1.isError)(settingsResult)) {
            return settingsResult.elaborate('Could not store the changed protection settings');
        }
        const protectionEnableResult = await this.startProtection(protectionDescription, protectedRoomsSet, context, { settings });
        if ((0, typescript_result_1.isError)(protectionEnableResult)) {
            const restoreResult = await this.settingsConfig.storeProtectionSettings(protectionDescription, currentSettings.ok);
            if ((0, typescript_result_1.isError)(restoreResult)) {
                log.error(`Unable to restore original settings for ${protectionDescription.name}:`, restoreResult);
            }
            else {
                log.info(`Restored the previous settings for ${protectionDescription.name} as the protection would not start`);
            }
            return protectionEnableResult.elaborate('Could not restart the protection with the new settings');
        }
        return protectionEnableResult;
    }
    async changeCapabilityProvider(context, protectedRoomsSet, protectionDescription, capabilityKey, capabilityProvider) {
        const currentCapabilityProviderSet = await this.capabilityProviderSetConfig.getCapabilityProviderSet(protectionDescription);
        if ((0, typescript_result_1.isError)(currentCapabilityProviderSet)) {
            return currentCapabilityProviderSet;
        }
        const capabilityInterface = protectionDescription.capabilities[capabilityKey];
        if (capabilityInterface === undefined) {
            return typescript_result_1.ResultError.Result(`Cannot find the capability interface ${capabilityKey} for ${protectionDescription.name}`);
        }
        if (capabilityProvider.interface !== capabilityInterface) {
            return typescript_result_1.ResultError.Result(`The capability provider ${capabilityProvider.name} does not implement the interface ${capabilityInterface.name}`);
        }
        const newCapabilityProviderSet = {
            ...currentCapabilityProviderSet.ok,
            [capabilityKey]: capabilityProvider,
        };
        currentCapabilityProviderSet.ok[capabilityKey] = capabilityProvider;
        return await this.changeCapabilityProviderSet(protectionDescription, protectedRoomsSet, context, newCapabilityProviderSet);
    }
    async changeCapabilityProviderSet(protectionDescription, protectedRoomsSet, context, capabilityProviderSet) {
        const result = await this.startProtection(protectionDescription, protectedRoomsSet, context, { capabilityProviderSet });
        if ((0, typescript_result_1.isError)(result)) {
            return result;
        }
        return await this.capabilityProviderSetConfig.storeActivateCapabilityProviderSet(protectionDescription, capabilityProviderSet);
    }
    async getCapabilityProviderSet(protectionDescription) {
        return await this.capabilityProviderSetConfig.getCapabilityProviderSet(protectionDescription);
    }
    async getProtectionSettings(protectionDescription) {
        return (await this.settingsConfig.getProtectionSettings(protectionDescription));
    }
    isEnabledProtection(protectionDescription) {
        return this.enabledProtections.has(protectionDescription.name);
    }
    findEnabledProtection(name) {
        return this.enabledProtections.get(name);
    }
    withEnabledProtection(name, cb) {
        const protection = this.findEnabledProtection(name);
        if (protection !== undefined) {
            cb(protection);
        }
    }
    unregisterListeners() {
        var _a;
        for (const protection of this.allProtections) {
            try {
                (_a = protection.handleProtectionDisable) === null || _a === void 0 ? void 0 : _a.call(protection);
            }
            catch (ex) {
                log.error(`Caught unhandled exception while unregistering listeners for ${protection.description.name}:`, ex);
            }
        }
        this.enabledProtections.clear();
    }
}
exports.StandardProtectionsManager = StandardProtectionsManager;
//# sourceMappingURL=StandardProtectionsManager.js.map