"use strict";
// Copyright 2022 - 2023 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirBotModeToggle = void 0;
exports.constructWebAPIs = constructWebAPIs;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_protection_suite_for_matrix_bot_sdk_1 = require("matrix-protection-suite-for-matrix-bot-sdk");
const DraupnirFactory_1 = require("./draupnirfactory/DraupnirFactory");
const WebAPIs_1 = require("./webapis/WebAPIs");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const typescript_result_1 = require("@gnuxie/typescript-result");
const SafeModeToggle_1 = require("./safemode/SafeModeToggle");
const typescript_result_2 = require("@gnuxie/typescript-result");
const SafeModeCause_1 = require("./safemode/SafeModeCause");
const BootOption_1 = require("./safemode/BootOption");
const utils_1 = require("./utils");
const log = new matrix_protection_suite_1.Logger("DraupnirBotMode");
function constructWebAPIs(draupnir) {
    return new WebAPIs_1.WebAPIs(draupnir.reportManager, draupnir.config, draupnir.synapseHTTPAntispam);
}
class DraupnirBotModeToggle {
    constructor(clientUserID, managementRoom, clientsInRoomMap, roomStateManagerFactory, draupnirFactory, matrixEmitter, config) {
        this.clientUserID = clientUserID;
        this.managementRoom = managementRoom;
        this.clientsInRoomMap = clientsInRoomMap;
        this.roomStateManagerFactory = roomStateManagerFactory;
        this.draupnirFactory = draupnirFactory;
        this.matrixEmitter = matrixEmitter;
        this.config = config;
        this.draupnir = null;
        this.safeModeDraupnir = null;
        this.webAPIs = null;
        this.matrixEmitter.on("room.invite", (roomID, event) => {
            this.clientsInRoomMap.handleTimelineEvent(roomID, event);
        });
        this.matrixEmitter.on("room.event", (roomID, event) => {
            this.roomStateManagerFactory.handleTimelineEvent(roomID, event);
            this.clientsInRoomMap.handleTimelineEvent(roomID, event);
        });
        this.matrixEmitter.on("room.leave", (roomID, event) => {
            this.roomStateManagerFactory.handleTimelineEvent(roomID, event);
            this.clientsInRoomMap.handleTimelineEvent(roomID, event);
        });
    }
    static async create(client, matrixEmitter, config, stores) {
        (0, utils_1.patchMatrixClient)();
        const clientUserID = await client.getUserId();
        if (!(0, matrix_basic_types_1.isStringUserID)(clientUserID)) {
            throw new TypeError(`${clientUserID} is not a valid mxid`);
        }
        if (!(0, matrix_basic_types_1.isStringRoomAlias)(config.managementRoom) &&
            !(0, matrix_basic_types_1.isStringRoomID)(config.managementRoom)) {
            throw new TypeError(`${config.managementRoom} is not a valid room id or alias`);
        }
        const configManagementRoomReference = matrix_basic_types_1.MatrixRoomReference.fromRoomIDOrAlias(config.managementRoom);
        const clientsInRoomMap = new matrix_protection_suite_1.StandardClientsInRoomMap();
        const clientCapabilityFactory = new matrix_protection_suite_for_matrix_bot_sdk_1.ClientCapabilityFactory(clientsInRoomMap, matrix_protection_suite_1.DefaultEventDecoder);
        // needed to have accurate join infomration.
        (await clientsInRoomMap.makeClientRooms(clientUserID, async () => (0, matrix_protection_suite_for_matrix_bot_sdk_1.joinedRoomsSafe)(client))).expect("Unable to create ClientRooms");
        const clientPlatform = clientCapabilityFactory.makeClientPlatform(clientUserID, client);
        const managementRoom = (await clientPlatform
            .toRoomResolver()
            .resolveRoom(configManagementRoomReference)).expect("Unable to resolve Draupnir's management room");
        (await clientPlatform.toRoomJoiner().joinRoom(managementRoom)).expect("Unable to join Draupnir's management room");
        const clientProvider = async (userID) => {
            if (userID !== clientUserID) {
                throw new TypeError(`Bot mode shouldn't be requesting any other mxids`);
            }
            return client;
        };
        const roomStateManagerFactory = new matrix_protection_suite_for_matrix_bot_sdk_1.RoomStateManagerFactory(clientsInRoomMap, clientProvider, matrix_protection_suite_1.DefaultEventDecoder, stores.roomStateBackingStore, stores.hashStore);
        const draupnirFactory = new DraupnirFactory_1.DraupnirFactory(clientsInRoomMap, clientCapabilityFactory, clientProvider, roomStateManagerFactory, stores);
        return new DraupnirBotModeToggle(clientUserID, managementRoom, clientsInRoomMap, roomStateManagerFactory, draupnirFactory, matrixEmitter, config);
    }
    async switchToDraupnir(options) {
        if (this.draupnir !== null) {
            return typescript_result_2.ResultError.Result(`There is a draupnir for ${this.clientUserID} already running`);
        }
        this.stopSafeModeDraupnir();
        const draupnirResult = await this.draupnirFactory.makeDraupnir(this.clientUserID, this.managementRoom, this.config, this);
        if ((0, typescript_result_1.isError)(draupnirResult)) {
            const safeModeResult = await this.maybeRecoverWithSafeMode(draupnirResult.error, options);
            if ((0, typescript_result_1.isError)(safeModeResult)) {
                return safeModeResult;
            }
            else {
                return SafeModeToggle_1.DraupnirRestartError.Result("Draupnir failed to start, switching to safe mode.", { safeModeDraupnir: safeModeResult.ok });
            }
        }
        this.draupnir = draupnirResult.ok;
        this.draupnir.start();
        if (options?.sendStatusOnStart) {
            void (0, matrix_protection_suite_1.Task)(this.draupnir.startupComplete());
            try {
                this.webAPIs = constructWebAPIs(this.draupnir);
                await this.webAPIs.start();
            }
            catch (e) {
                if (e instanceof Error) {
                    await this.stopDraupnir();
                    log.error("Failed to start webAPIs", e);
                    return matrix_protection_suite_1.ActionException.Result("Failed to start webAPIs", {
                        exceptionKind: matrix_protection_suite_1.ActionExceptionKind.Unknown,
                        exception: e,
                    });
                }
                else {
                    throw new TypeError("Someone is throwing garbage.");
                }
            }
        }
        return draupnirResult;
    }
    async switchToSafeMode(cause, options) {
        if (this.safeModeDraupnir !== null) {
            return typescript_result_2.ResultError.Result(`There is a safe mode draupnir for ${this.clientUserID} already running`);
        }
        const safeModeResult = await this.draupnirFactory.makeSafeModeDraupnir(this.clientUserID, this.managementRoom, this.config, cause, this);
        if ((0, typescript_result_1.isError)(safeModeResult)) {
            return safeModeResult;
        }
        await this.stopDraupnir();
        this.safeModeDraupnir = safeModeResult.ok;
        this.safeModeDraupnir.start();
        if (options?.sendStatusOnStart) {
            this.safeModeDraupnir.startupComplete();
        }
        return safeModeResult;
    }
    async startFromScratch(options) {
        const draupnirResult = await this.switchToDraupnir(options ?? {});
        if ((0, typescript_result_1.isError)(draupnirResult)) {
            if (draupnirResult.error instanceof SafeModeToggle_1.DraupnirRestartError) {
                return (0, typescript_result_1.Ok)(draupnirResult.error.safeModeDraupnir);
            }
            else {
                return draupnirResult;
            }
        }
        return draupnirResult;
    }
    async maybeRecoverWithSafeMode(error, options) {
        switch (this.config.safeMode?.bootOption) {
            case BootOption_1.SafeModeBootOption.Never:
                return (0, typescript_result_1.Err)(error);
            case BootOption_1.SafeModeBootOption.RecoveryOnly:
                if (!(error instanceof matrix_protection_suite_1.ConfigRecoverableError)) {
                    return (0, typescript_result_1.Err)(error);
                }
            // fallthrough
            default:
                log.error("Failed to start draupnir, switching to safe mode as configured", error);
                return await this.switchToSafeMode({
                    reason: SafeModeCause_1.SafeModeReason.InitializationError,
                    error: error,
                }, options ?? {});
        }
    }
    async encryptionInitialized() {
        if (this.draupnir !== null) {
            try {
                this.webAPIs = constructWebAPIs(this.draupnir);
                await this.webAPIs.start();
                await this.draupnir.startupComplete();
            }
            catch (e) {
                await this.stopEverything();
                throw e;
            }
        }
        else if (this.safeModeDraupnir !== null) {
            this.safeModeDraupnir.startupComplete();
        }
    }
    async stopDraupnir() {
        this.draupnir?.stop();
        this.draupnir = null;
        await this.webAPIs?.stop();
        this.webAPIs = null;
    }
    stopSafeModeDraupnir() {
        this.safeModeDraupnir?.stop();
        this.safeModeDraupnir = null;
    }
    async stopEverything() {
        await this.stopDraupnir();
        this.stopSafeModeDraupnir();
    }
}
exports.DraupnirBotModeToggle = DraupnirBotModeToggle;
//# sourceMappingURL=DraupnirBotMode.js.map