"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Api = void 0;
const request_1 = __importDefault(require("request"));
const express_1 = __importDefault(require("express"));
const bodyParser = __importStar(require("body-parser"));
const matrix_appservice_bridge_1 = require("matrix-appservice-bridge");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const log = new matrix_appservice_bridge_1.Logger("Api");
/**
 * This provides a web api that is designed to power the mjolnir widget https://github.com/matrix-org/mjolnir-widget.
 */
class Api {
    constructor(homeserver, mjolnirManager) {
        this.homeserver = homeserver;
        this.mjolnirManager = mjolnirManager;
        this.httpdConfig = (0, express_1.default)();
    }
    /**
     * Resolves an open id access token to find a matching user that the token is valid for.
     * @param accessToken An openID token.
     * @returns The mxid of the user that this token belongs to or null if the token could not be authenticated.
     */
    resolveAccessToken(accessToken) {
        return new Promise((resolve, reject) => {
            (0, request_1.default)({
                url: `${this.homeserver}/_matrix/federation/v1/openid/userinfo`,
                qs: { access_token: accessToken },
            }, (err, homeserver_response, body) => {
                if (err) {
                    log.error(`Error resolving openID token from ${this.homeserver}`, err);
                    if (err instanceof Error) {
                        reject(err);
                    }
                    else {
                        reject(new Error(`There was an error when resolving openID token from ${this.homeserver}`));
                    }
                }
                let response;
                try {
                    response = JSON.parse(body);
                }
                catch (e) {
                    log.error(`Received ill formed response from ${this.homeserver} when resolving an openID token`, e);
                    if (err instanceof Error) {
                        reject(err);
                    }
                    reject(new Error(`Received ill formed response from ${this.homeserver} when resolving an openID token ${e}`));
                    return;
                }
                resolve(response.sub);
            });
        });
    }
    async close() {
        await new Promise((resolve, reject) => {
            if (!this.httpServer) {
                throw new TypeError("Server was never started");
            }
            this.httpServer.close((error) => {
                error ? reject(error) : resolve(undefined);
            });
        });
    }
    start(port) {
        if (this.httpServer) {
            throw new TypeError("server already started");
        }
        this.httpdConfig.use(bodyParser.json());
        this.httpdConfig.get("/get", this.pathGet.bind(this));
        this.httpdConfig.get("/list", this.pathList.bind(this));
        this.httpdConfig.post("/create", this.pathCreate.bind(this));
        this.httpdConfig.post("/join", this.pathJoin.bind(this));
        this.httpServer = this.httpdConfig.listen(port);
    }
    /**
     * Finds the management room for a draupnir.
     * @param req.body.openId An OpenID token to verify that the sender of the request owns the draupnir described in `req.body.mxid`.
     * @param req.body.mxid   The mxid of the draupnir we want to find the management room for.
     */
    async pathGet(req, response) {
        const accessToken = req.body["openId"];
        if (accessToken === undefined) {
            response.status(401).send("unauthorised");
            return;
        }
        const userId = await this.resolveAccessToken(accessToken);
        if (userId === null) {
            response.status(401).send("unauthorised");
            return;
        }
        if (!(0, matrix_basic_types_1.isStringUserID)(userId)) {
            response.status(400).send("invalid user mxid");
            return;
        }
        const mjolnirId = req.body["mxid"];
        if (mjolnirId === undefined || !(0, matrix_basic_types_1.isStringUserID)(mjolnirId)) {
            response.status(400).send("invalid request");
            return;
        }
        const mjolnir = await this.mjolnirManager.getRunningDraupnir(mjolnirId, userId);
        if (mjolnir === undefined) {
            response.status(400).send("unknown draupnir mxid");
            return;
        }
        response.status(200).json({ managementRoom: mjolnir.managementRoomID });
    }
    /**
     * Return the mxids of draupnirs that this user has provisioned.
     * @param req.body.openId An OpenID token to find the sender of the request with and find their provisioned draupnirs.
     */
    async pathList(req, response) {
        const accessToken = req.body["openId"];
        if (accessToken === undefined) {
            response.status(401).send("unauthorised");
            return;
        }
        const userId = await this.resolveAccessToken(accessToken);
        if (userId === null) {
            response.status(401).send("unauthorised");
            return;
        }
        if (!(0, matrix_basic_types_1.isStringUserID)(userId)) {
            response.status(400).send("invalid user mxid");
            return;
        }
        const existing = this.mjolnirManager.getOwnedDraupnir(userId);
        response.status(200).json(existing);
    }
    /**
     * Creates a new draupnir for the requesting user and protects their first room.
     * @param req.body.roomId The room id that the request to create a draupnir originates from.
     * This is so that draupnir can protect the room once the authenticity of the request has been verified.
     * @param req.body.openId An OpenID token to find the sender of the request with.
     */
    async pathCreate(req, response) {
        const accessToken = req.body["openId"];
        if (accessToken === undefined) {
            response.status(401).send("unauthorised");
            return;
        }
        const roomId = req.body["roomId"];
        if (roomId === undefined) {
            response.status(400).send("invalid request");
            return;
        }
        const userId = await this.resolveAccessToken(accessToken);
        if (userId === null) {
            response.status(401).send("unauthorised");
            return;
        }
        if (!(0, matrix_basic_types_1.isStringUserID)(userId)) {
            response.status(400).send("invalid user mxid");
            return;
        }
        const record = await this.mjolnirManager.provisionNewDraupnir(userId);
        if ((0, matrix_protection_suite_1.isError)(record)) {
            response.status(500).send(record.error.message);
            return;
        }
        response.status(200).json({
            mxid: this.mjolnirManager.draupnirMXID(record.ok),
            roomId: record.ok.management_room,
        });
    }
    /**
     * Request a draupnir to join and protect a room.
     * @param req.body.openId An OpenID token to find the sender of the request with and that they own the draupnir described in `req.body.mxid`.
     * @param req.body.mxid   The mxid of the draupnir that should join the room.
     * @param req.body.roomId The room that this draupnir should join and protect.
     */
    async pathJoin(req, response) {
        const accessToken = req.body["openId"];
        if (accessToken === undefined) {
            response.status(401).send("unauthorised");
            return;
        }
        const userId = await this.resolveAccessToken(accessToken);
        if (userId === null) {
            response.status(401).send("unauthorised");
            return;
        }
        if (!(0, matrix_basic_types_1.isStringUserID)(userId)) {
            response.status(400).send("invalid user mxid");
            return;
        }
        const mjolnirId = req.body["mxid"];
        if (mjolnirId === undefined || !(0, matrix_basic_types_1.isStringUserID)(mjolnirId)) {
            response.status(400).send("invalid request");
            return;
        }
        const roomId = req.body["roomId"];
        if (roomId === undefined) {
            response.status(400).send("invalid request");
            return;
        }
        // TODO: getMjolnir can fail if the ownerId doesn't match the requesting userId.
        // https://github.com/matrix-org/mjolnir/issues/408
        const mjolnir = await this.mjolnirManager.getRunningDraupnir(mjolnirId, userId);
        if (mjolnir === undefined) {
            response.status(400).send("unknown draupnir mxid");
            return;
        }
        await mjolnir.client.joinRoom(roomId);
        await mjolnir.protectedRoomsSet.protectedRoomsManager.addRoom(roomId);
        response.status(200).json({});
    }
}
exports.Api = Api;
//# sourceMappingURL=Api.js.map