"use strict";
// Copyright 2022, 2025 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirUnwatchPolicyRoomCommand = exports.DraupnirWatchPolicyRoomCommand = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const typescript_result_1 = require("@gnuxie/typescript-result");
const DraupnirCommandPrerequisites_1 = require("./DraupnirCommandPrerequisites");
const WatchPreview_1 = require("./WatchPreview");
exports.DraupnirWatchPolicyRoomCommand = (0, interface_manager_1.describeCommand)({
    summary: "Watches a list and applies the list's assocated policies to draupnir's protected rooms.",
    parameters: (0, interface_manager_1.tuple)({
        name: "policy room",
        acceptor: interface_manager_1.MatrixRoomReferencePresentationSchema,
    }),
    keywords: {
        keywordDescriptions: {
            "no-confirm": {
                isFlag: true,
                description: "Runs the command without the preview.",
            },
        },
    },
    async executor({ watchedPolicyRooms, roomJoiner, policyRoomManager, protectedRoomsSet, }, _info, keywords, _rest, policyRoomReference) {
        const policyRoom = await roomJoiner.joinRoom(policyRoomReference);
        if ((0, matrix_protection_suite_1.isError)(policyRoom)) {
            return policyRoom.elaborate("Failed to resolve or join the room");
        }
        if (watchedPolicyRooms.allRooms.some((profile) => profile.room.toRoomIDOrAlias() === policyRoom.ok.toRoomIDOrAlias())) {
            return typescript_result_1.ResultError.Result("We are already watching this list.");
        }
        if (keywords.getKeywordValue("no-confirm", false)) {
            const watchResult = await watchedPolicyRooms.watchPolicyRoomDirectly(policyRoom.ok);
            if ((0, matrix_protection_suite_1.isError)(watchResult)) {
                return watchResult;
            }
            return (0, typescript_result_1.Ok)(undefined);
        }
        const revisionIssuer = await policyRoomManager.getPolicyRoomRevisionIssuer(policyRoom.ok);
        if ((0, matrix_protection_suite_1.isError)(revisionIssuer)) {
            return revisionIssuer.elaborate("Failed to fetch policy room revision issuer");
        }
        return (0, typescript_result_1.Ok)((0, WatchPreview_1.generateWatchPreview)(protectedRoomsSet, revisionIssuer.ok.currentRevision));
    },
});
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirWatchPolicyRoomCommand, {
    isAlwaysSupposedToUseDefaultRenderer: true,
    confirmationPromptJSXRenderer(commandResult) {
        if ((0, matrix_protection_suite_1.isError)(commandResult)) {
            return (0, typescript_result_1.Ok)(undefined);
        }
        else if (commandResult.ok === undefined) {
            return (0, typescript_result_1.Ok)(undefined);
        }
        else {
            return (0, typescript_result_1.Ok)((0, WatchPreview_1.renderWatchCommandPreview)(commandResult.ok));
        }
    },
});
DraupnirCommandPrerequisites_1.DraupnirContextToCommandContextTranslator.registerTranslation(exports.DraupnirWatchPolicyRoomCommand, buildWatchContext);
exports.DraupnirUnwatchPolicyRoomCommand = (0, interface_manager_1.describeCommand)({
    summary: "Unwatches a list and stops applying the list's assocated policies to draupnir's protected rooms.",
    parameters: (0, interface_manager_1.tuple)({
        name: "policy room",
        acceptor: interface_manager_1.MatrixRoomReferencePresentationSchema,
    }),
    async executor({ watchedPolicyRooms, roomJoiner }, _info, _keywords, _rest, policyRoomReference) {
        const policyRoom = await roomJoiner.resolveRoom(policyRoomReference);
        if ((0, matrix_protection_suite_1.isError)(policyRoom)) {
            return policyRoom;
        }
        return await watchedPolicyRooms.unwatchPolicyRoom(policyRoom.ok);
    },
});
function buildWatchContext(draupnir) {
    return {
        watchedPolicyRooms: draupnir.protectedRoomsSet.watchedPolicyRooms,
        roomJoiner: draupnir.clientPlatform.toRoomJoiner(),
        policyRoomManager: draupnir.policyRoomManager,
        protectedRoomsSet: draupnir.protectedRoomsSet,
    };
}
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirUnwatchPolicyRoomCommand, {
    isAlwaysSupposedToUseDefaultRenderer: true,
});
DraupnirCommandPrerequisites_1.DraupnirContextToCommandContextTranslator.registerTranslation(exports.DraupnirUnwatchPolicyRoomCommand, buildWatchContext);
//# sourceMappingURL=WatchUnwatchCommand.js.map