"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynapseAdminSuspendUserCommand = void 0;
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const typescript_result_1 = require("@gnuxie/typescript-result");
const DraupnirCommandPrerequisites_1 = require("../DraupnirCommandPrerequisites");
const mps_interface_adaptor_1 = require("@the-draupnir-project/mps-interface-adaptor");
const UserSuspensionCapability_1 = require("../../protections/HomeserverUserPolicyApplication/UserSuspensionCapability");
exports.SynapseAdminSuspendUserCommand = (0, interface_manager_1.describeCommand)({
    summary: "Suspend a user on the homeserver, this allows them to login but not interact with anything",
    parameters: (0, interface_manager_1.tuple)({
        name: "user",
        description: "The user to suspend",
        acceptor: interface_manager_1.MatrixUserIDPresentationType,
    }),
    keywords: {
        keywordDescriptions: {
            "no-confirm": {
                isFlag: true,
                description: "Runs the command without the preview of the unban and the confirmation prompt.",
            },
        },
    },
    async executor(draupnir, info, keywords, _rest, targetUser) {
        const isAdmin = await draupnir.synapseAdminClient?.isSynapseAdmin();
        if (draupnir.synapseAdminClient === undefined ||
            isAdmin === undefined ||
            (0, typescript_result_1.isError)(isAdmin) ||
            !isAdmin.ok ||
            draupnir.stores.userRestrictionAuditLog === undefined) {
            return typescript_result_1.ResultError.Result("I am not a Synapse administrator, or the endpoint to deactivate a user is blocked");
        }
        const preview = {
            userID: targetUser.toString(),
        };
        const isNoConfirm = keywords.getKeywordValue("no-confirm", false);
        if (!isNoConfirm) {
            return (0, typescript_result_1.Ok)(preview);
        }
        // we do this because it handles all the audit logging for us.
        const suspensionCapability = new UserSuspensionCapability_1.SynapseAdminUserSuspensionCapability(draupnir.synapseAdminClient, draupnir.stores.userRestrictionAuditLog);
        const suspensionResult = await suspensionCapability.restrictUser(targetUser.toString(), { rule: null, sender: info.commandSender });
        if ((0, typescript_result_1.isError)(suspensionResult)) {
            return suspensionResult;
        }
        else {
            return (0, typescript_result_1.Ok)(preview);
        }
    },
});
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.SynapseAdminSuspendUserCommand, {
    isAlwaysSupposedToUseDefaultRenderer: true,
    confirmationPromptJSXRenderer(commandResult) {
        if ((0, typescript_result_1.isError)(commandResult)) {
            return (0, typescript_result_1.Ok)(undefined);
        }
        return (0, typescript_result_1.Ok)(interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
            "You are about to suspend the user",
            " ",
            (0, mps_interface_adaptor_1.renderMentionPill)(commandResult.ok.userID, commandResult.ok.userID),
            " ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, commandResult.ok.userID),
            ". Doing so will prevent further activity from their account. However, they will still be able to login, manage their account, and redact messages. See https://spec.matrix.org/v1.14/client-server-api/#account-suspension for a complete explanation."));
    },
    JSXRenderer(commandResult) {
        if ((0, typescript_result_1.isError)(commandResult)) {
            return (0, typescript_result_1.Ok)(undefined);
        }
        return (0, typescript_result_1.Ok)(interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
            "The user ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, commandResult.ok.userID),
            " has been suspended."));
    },
});
//# sourceMappingURL=SuspendCommand.js.map