"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirTakedownCommand = void 0;
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const Ban_1 = require("../Ban");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const typescript_result_1 = require("@gnuxie/typescript-result");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const DraupnirCommandPrerequisites_1 = require("../DraupnirCommandPrerequisites");
const mps_interface_adaptor_1 = require("@the-draupnir-project/mps-interface-adaptor");
const SynapseAdminRoomTakedown_1 = require("../../capabilities/SynapseAdminRoomTakedown/SynapseAdminRoomTakedown");
const RoomTakedownProtection_1 = require("../../protections/RoomTakedown/RoomTakedownProtection");
const BlockInvitationsOnServerProtection_1 = require("../../protections/BlockInvitationsOnServerProtection");
const HomeserverUserPolicyProtection_1 = require("../../protections/HomeserverUserPolicyApplication/HomeserverUserPolicyProtection");
const log = new matrix_protection_suite_1.Logger("DraupnirTakedownCommand");
const DownstreamTakedownProtectionNames = [
    RoomTakedownProtection_1.RoomTakedownProtection.name,
    BlockInvitationsOnServerProtection_1.BlockInvitationsOnServerProtection.name,
    HomeserverUserPolicyProtection_1.HomeserverUserPolicyProtection.name,
];
/**
 * Make sure that the hash store is up to date with the entity
 */
async function handleRoomDiscovery(roomID, store, detailsProvider) {
    if (store === undefined) {
        log.warn("Unable to discover a room provided by the takedown command because the hash store has not been configured");
        return (0, matrix_protection_suite_1.Ok)(undefined);
    }
    if (detailsProvider === undefined) {
        log.warn("Unable to discover a room provided by the takedown command because the details provider has not been configured");
        return (0, matrix_protection_suite_1.Ok)(undefined);
    }
    const storeResult = await store.storeUndiscoveredRooms([roomID]);
    if ((0, matrix_protection_suite_1.isError)(storeResult)) {
        return storeResult.elaborate("Unable to store the room from takedown command into the hash store");
    }
    const detailsResult = await detailsProvider.getRoomDetails(roomID);
    if ((0, matrix_protection_suite_1.isError)(detailsResult)) {
        return detailsResult.elaborate("Failed to fetch details for a room discovered via the takedown command");
    }
    if (detailsResult.ok.creator === undefined) {
        log.warn("No creator was provided in the details for the room, so we cannot store them", roomID, detailsResult.ok);
        return (0, matrix_protection_suite_1.Ok)(undefined);
    }
    return await store.storeRoomIdentification({
        roomID,
        creator: detailsResult.ok.creator,
        server: (0, matrix_basic_types_1.userServerName)(detailsResult.ok.creator),
    });
}
exports.DraupnirTakedownCommand = (0, interface_manager_1.describeCommand)({
    summary: "Mark an entity for takedown. This command is used to mark illegal or intollerable content as takedown. This is the strongest of consequences and is usualy irreversable.\
    This can be used to block rooms on your homeserver and discard spam invitations.",
    parameters: (0, interface_manager_1.tuple)({
        name: "entity",
        description: "The entity to ban. This can be a user ID, room ID, or server name.",
        acceptor: (0, interface_manager_1.union)(interface_manager_1.MatrixUserIDPresentationType, interface_manager_1.MatrixRoomReferencePresentationSchema, interface_manager_1.StringPresentationType),
    }, {
        name: "list",
        acceptor: (0, interface_manager_1.union)(interface_manager_1.MatrixRoomReferencePresentationSchema, interface_manager_1.StringPresentationType),
        prompt: async function ({ policyRoomManager, clientUserID, }) {
            return (0, matrix_protection_suite_1.Ok)({
                suggestions: policyRoomManager
                    .getEditablePolicyRoomIDs(clientUserID, matrix_protection_suite_1.PolicyRuleType.User)
                    .map((room) => interface_manager_1.MatrixRoomIDPresentationType.wrap(room)),
            });
        },
    }),
    keywords: {
        keywordDescriptions: {
            "no-confirm": {
                isFlag: true,
                description: "Runs the command without the preview of the unban and the confirmation prompt.",
            },
            "plain-text": {
                isFlag: true,
                description: "Creates a plain-text version of the policy rather than masking the entity with SHA256. There are not many reason to do this other than compatibility with other tools.",
            },
        },
    },
    async executor({ watchedPolicyRooms, policyRoomManager, roomResolver, hashStore, detailsProvider, enabledProtections, }, _info, keywords, _rest, entity, policyRoomDesignator) {
        const enabledTakedownProtections = enabledProtections.filter((protection) => DownstreamTakedownProtectionNames.includes(protection.description.name));
        const takedownProtections = DownstreamTakedownProtectionNames.map((name) => ({
            name,
            isEnabled: enabledTakedownProtections.some((protection) => protection.description.name === name),
        }));
        const policyRoomReference = typeof policyRoomDesignator === "string"
            ? (0, matrix_protection_suite_1.Ok)(watchedPolicyRooms.findPolicyRoomFromShortcode(policyRoomDesignator)
                ?.room)
            : (0, matrix_protection_suite_1.Ok)(policyRoomDesignator);
        if ((0, matrix_protection_suite_1.isError)(policyRoomReference)) {
            return policyRoomReference;
        }
        if (policyRoomReference.ok === undefined) {
            return typescript_result_1.ResultError.Result(`Unable to find a policy room from the shortcode ${policyRoomDesignator.toString()}`);
        }
        const policyRoomEditorResult = await (0, Ban_1.findPolicyRoomEditorFromRoomReference)(roomResolver, policyRoomManager, policyRoomReference.ok);
        if ((0, matrix_protection_suite_1.isError)(policyRoomEditorResult)) {
            return policyRoomEditorResult;
        }
        const policyRoomEditor = policyRoomEditorResult.ok;
        const preview = await (async () => {
            if (entity instanceof matrix_basic_types_1.MatrixUserID) {
                return (0, matrix_protection_suite_1.Ok)({
                    ruleType: matrix_protection_suite_1.PolicyRuleType.User,
                    entity: entity.toString(),
                    policyRoom: policyRoomEditor.room,
                    takedownProtections: takedownProtections,
                });
            }
            else if (typeof entity === "string") {
                return (0, matrix_protection_suite_1.Ok)({
                    ruleType: matrix_protection_suite_1.PolicyRuleType.Server,
                    entity,
                    policyRoom: policyRoomEditor.room,
                    takedownProtections: takedownProtections,
                });
            }
            else {
                const resolvedRoomReference = await roomResolver.resolveRoom(entity);
                if ((0, matrix_protection_suite_1.isError)(resolvedRoomReference)) {
                    return resolvedRoomReference;
                }
                return (0, matrix_protection_suite_1.Ok)({
                    ruleType: matrix_protection_suite_1.PolicyRuleType.Room,
                    entity: resolvedRoomReference.ok.toRoomIDOrAlias(),
                    policyRoom: policyRoomEditor.room,
                    takedownProtections: takedownProtections,
                });
            }
        })();
        if ((0, matrix_protection_suite_1.isError)(preview)) {
            return preview;
        }
        if (!keywords.getKeywordValue("no-confirm", false)) {
            return preview;
        }
        const plainText = keywords.getKeywordValue("plain-text", false);
        const takedownResult = await policyRoomEditor.takedownEntity(preview.ok.ruleType, preview.ok.entity, { shouldHash: !plainText });
        if ((0, matrix_protection_suite_1.isError)(takedownResult) ||
            preview.ok.ruleType !== matrix_protection_suite_1.PolicyRuleType.Room) {
            return takedownResult;
        }
        const roomDiscoveryResult = await handleRoomDiscovery(preview.ok.entity, hashStore, detailsProvider);
        if ((0, matrix_protection_suite_1.isError)(roomDiscoveryResult)) {
            return roomDiscoveryResult.elaborate("Failed to inform room discovery of the room provided in this command");
        }
        return takedownResult;
    },
});
function renderTakedownPreview(preview) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        "You are about to mark an entity to be takendown by any means necessary:",
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
                "entity: ",
                interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, preview.entity)),
            interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
                "policy type: ",
                interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, preview.ruleType)),
            interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
                "policy room: ",
                (0, mps_interface_adaptor_1.renderRoomPill)(preview.policyRoom))),
        interface_manager_1.DeadDocumentJSX.JSXFactory("h5", null, "Please consider that doing so may have irreversable effects."),
        interface_manager_1.DeadDocumentJSX.JSXFactory("p", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, "You MUST only use takedown policies to mark spam, illegal, or otherwise intolerable content. DO NOT takedown users who have committed a code of conduct violation.")),
        interface_manager_1.DeadDocumentJSX.JSXFactory("h5", null, "The following protections consume takedown policies:"),
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, preview.takedownProtections.map((protection) => (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
            protection.isEnabled ? "🟢 (enabled)" : "🔴 (disabled)",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, protection.name)))))));
}
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirTakedownCommand, {
    isAlwaysSupposedToUseDefaultRenderer: true,
    confirmationPromptJSXRenderer(commandResult) {
        if ((0, matrix_protection_suite_1.isError)(commandResult)) {
            return (0, matrix_protection_suite_1.Ok)(undefined);
        }
        else if (typeof commandResult.ok === "string") {
            return (0, matrix_protection_suite_1.Ok)(undefined); // it's an event id that's got nothing going for it.
        }
        else {
            return (0, matrix_protection_suite_1.Ok)(interface_manager_1.DeadDocumentJSX.JSXFactory("root", null, renderTakedownPreview(commandResult.ok)));
        }
    },
});
DraupnirCommandPrerequisites_1.DraupnirContextToCommandContextTranslator.registerTranslation(exports.DraupnirTakedownCommand, function (draupnir) {
    return {
        policyRoomManager: draupnir.policyRoomManager,
        watchedPolicyRooms: draupnir.protectedRoomsSet.watchedPolicyRooms,
        defaultReasons: draupnir.config.commands.ban.defaultReasons,
        roomResolver: draupnir.clientPlatform.toRoomResolver(),
        clientUserID: draupnir.clientUserID,
        hashStore: draupnir.stores.hashStore,
        detailsProvider: draupnir.synapseAdminClient
            ? new SynapseAdminRoomTakedown_1.SynapseAdminRoomDetailsProvider(draupnir.synapseAdminClient)
            : undefined,
        enabledProtections: draupnir.protectedRoomsSet.protections.allProtections,
    };
});
//# sourceMappingURL=Takedown.js.map