"use strict";
// Copyright 2024 - 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from Draupnir
// https://github.com/the-draupnir-project/Draupnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.matrixEventsFromConfirmationPrompt = exports.COMMAND_CONFIRMATION_LISTENER = void 0;
exports.makeConfirmationPromptListener = makeConfirmationPromptListener;
exports.makeconfirmationPromptSender = makeconfirmationPromptSender;
exports.sendConfirmationPrompt = sendConfirmationPrompt;
const typescript_result_1 = require("@gnuxie/typescript-result");
const MPSMatrixInterfaceAdaptor_1 = require("./MPSMatrixInterfaceAdaptor");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const MatrixPromptForAccept_1 = require("./MatrixPromptForAccept");
const log = new matrix_protection_suite_1.Logger("MatrixPromptForConfirmation");
exports.COMMAND_CONFIRMATION_LISTENER = "me.marewolf.draupnir.command_confirmation";
function makeConfirmationPromptListener(commandRoomID, commandDispatcher, reactionHandler) {
    return (key, item, rawContext, _reactionMap, annotatedEvent) => {
        if (annotatedEvent.room_id !== commandRoomID) {
            return;
        }
        if (key === "Cancel") {
            void (0, matrix_protection_suite_1.Task)(reactionHandler.cancelPrompt(annotatedEvent));
            return;
        }
        if (key !== "OK") {
            return;
        }
        const promptContext = matrix_protection_suite_1.Value.Decode(MatrixPromptForAccept_1.CommandPromptContext, rawContext);
        if ((0, typescript_result_1.isError)(promptContext)) {
            log.error(`malformed event context when trying to accept a prompted argument`, rawContext, promptContext.error);
            return;
        }
        (0, MatrixPromptForAccept_1.continueCommandAcceptingPrompt)({ event: annotatedEvent, roomID: annotatedEvent.room_id }, promptContext.ok, "--no-confirm", commandDispatcher, reactionHandler);
    };
}
function makeconfirmationPromptSender(adaptorContext) {
    return async function ({ commandDesignator, readItems }, document, { roomID, event }) {
        return sendConfirmationPrompt(adaptorContext, { commandDesignator, readItems }, document, { roomID, event });
    };
}
/**
 * This utility allows protections to send confirmation prompts that appear like confirmation prompts
 * for commands that have been sent without the `--no-confirm` option, but require confirmation.
 */
async function sendConfirmationPrompt({ clientPlatform, reactionHandler }, { commandDesignator, readItems, }, document, { roomID, event, }) {
    const roomIDToUse = roomID !== null && roomID !== void 0 ? roomID : event === null || event === void 0 ? void 0 : event.room_id;
    if (roomIDToUse === undefined) {
        throw new TypeError(`You must provide either a room ID or an event`);
    }
    const reactionMap = new Map(Object.entries({ OK: "OK", Cancel: "Cancel" }));
    const sendResult = await (0, MPSMatrixInterfaceAdaptor_1.sendMatrixEventsFromDeadDocument)(clientPlatform.toRoomMessageSender(), roomIDToUse, document, {
        replyToEvent: event,
        additionalContent: reactionHandler.createAnnotation(exports.COMMAND_CONFIRMATION_LISTENER, reactionMap, {
            command_designator: commandDesignator,
            read_items: readItems,
        }),
    });
    if ((0, typescript_result_1.isError)(sendResult)) {
        return sendResult;
    }
    if (sendResult.ok[0] === undefined) {
        throw new TypeError(`We exepct to have sent at least one event`);
    }
    return await reactionHandler.addReactionsToEvent(roomIDToUse, sendResult.ok[0], reactionMap);
}
exports.matrixEventsFromConfirmationPrompt = async function (adaptorContext, { event }, command, document) {
    return await sendConfirmationPrompt(adaptorContext, {
        commandDesignator: command.designator,
        readItems: command
            .toPartialCommand()
            .stream.source.slice(command.designator.length)
            .map((p) => interface_manager_1.TextPresentationRenderer.render(p)),
    }, document, { event });
};
//# sourceMappingURL=MatrixPromptForConfirmation.js.map