"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardClientRooms = void 0;
// Copyright (C) 2023 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
const Action_1 = require("../Interface/Action");
const Value_1 = require("../Interface/Value");
const MembershipEvent_1 = require("../MatrixTypes/MembershipEvent");
const RoomPauser_1 = require("./RoomPauser");
const ClientRooms_1 = require("./ClientRooms");
const await_lock_1 = __importDefault(require("await-lock"));
const Logger_1 = require("../Logging/Logger");
const JoinedRoomsRevision_1 = require("./JoinedRoomsRevision");
const MembershipChange_1 = require("../Membership/MembershipChange");
const log = new Logger_1.Logger('StandardClientRooms');
/**
 * An implementation of `ClientRooms` that will work for both bots and appservice
 * intents.
 */
class StandardClientRooms extends ClientRooms_1.AbstractClientRooms {
    constructor(joinedRoomsThunk, ...rest) {
        super(...rest);
        this.joinedRoomsThunk = joinedRoomsThunk;
        this.roomPauser = new RoomPauser_1.StandardRoomPauser();
        this.preemptivelyJoinedRooms = new Set();
        this.joinedRoomsCallLock = new await_lock_1.default();
    }
    /**
     * Create a clientRooms, initializing the joinedRoomsSet.
     * @param clientUserID The Matrix UserID of the client.
     * @param joinedRoomsThunk A thunk that returns the rooms the user is joined to.
     * @returns A new ClientRooms instance.
     */
    static async makeClientRooms(clientUserID, joinedRoomsThunk) {
        const joinedRooms = await joinedRoomsThunk();
        if ((0, Action_1.isError)(joinedRooms)) {
            return joinedRooms;
        }
        const revision = JoinedRoomsRevision_1.StandardJoinedRoomsRevision.blankRevision(clientUserID).reviseFromJoinedRooms(joinedRooms.ok);
        return (0, Action_1.Ok)(new StandardClientRooms(joinedRoomsThunk, clientUserID, revision));
    }
    get allPreemptedRooms() {
        return [...this.preemptivelyJoinedRooms];
    }
    isPreemptivelyJoinedRoom(roomID) {
        return (this.joinedRoomsRevision.isJoinedRoom(roomID) ||
            this.preemptivelyJoinedRooms.has(roomID));
    }
    preemptTimelineJoin(roomID) {
        if (this.isPreemptivelyJoinedRoom(roomID)) {
            return;
        }
        this.preemptivelyJoinedRooms.add(roomID);
        const changes = {
            preemptivelyJoined: [roomID],
            failedPreemptiveJoins: [],
            joined: [],
            parted: [],
        };
        this.emit('revision', this.joinedRoomsRevision, changes, this.joinedRoomsRevision);
    }
    handleTimelineEvent(roomID, event) {
        if (Value_1.Value.Check(MembershipEvent_1.MembershipEvent, event) &&
            event.state_key === this.clientUserID) {
            switch (event.content.membership) {
                case MembershipChange_1.Membership.Invite:
                    // You might be wondering if we should show invitations some other way
                    // but this is how appservices also get their invitations, so it makes
                    // sense to do it this way for our clients too.
                    this.emit('timeline', roomID, event);
                    break;
                case MembershipChange_1.Membership.Join:
                    if (this.isJoinedRoom(roomID)) {
                        this.emit('timeline', roomID, event);
                    }
                    else {
                        this.handleRoomJoin(roomID, event);
                    }
                    break;
                case MembershipChange_1.Membership.Leave:
                    if (this.isJoinedRoom(roomID)) {
                        this.handleRoomLeave(roomID, event);
                    }
                    else {
                        this.emit('timeline', roomID, event);
                    }
                    break;
            }
            return;
        }
        else if (this.isJoinedRoom(roomID)) {
            this.emit('timeline', roomID, event);
        }
    }
    handleRoomJoin(roomID, event) {
        if (this.roomPauser.isRoomPaused(roomID)) {
            this.roomPauser.handleTimelineEventInPausedRoom(roomID, event);
        }
        else {
            this.handleRoomChange(roomID);
            this.roomPauser.handleTimelineEventInPausedRoom(roomID, event);
        }
    }
    handleRoomLeave(roomID, event) {
        if (this.roomPauser.isRoomPaused(roomID)) {
            this.roomPauser.handleTimelineEventInPausedRoom(roomID, event);
        }
        else {
            this.handleRoomChange(roomID);
            this.roomPauser.handleTimelineEventInPausedRoom(roomID, event);
        }
    }
    async checkRoomTask() {
        // we lock specifically so that we can be sure we have checked all the rooms currently marked as preemptively joined
        await this.joinedRoomsCallLock.acquireAsync();
        try {
            const preemptivelyJoinedRoomsToCheck = [...this.preemptivelyJoinedRooms];
            const joinedRoomsResult = await this.joinedRoomsThunk();
            if ((0, Action_1.isError)(joinedRoomsResult)) {
                log.error(`Unable to fetch joined_members when calculating joined rooms`, joinedRoomsResult.error);
                return;
            }
            const joinedRooms = joinedRoomsResult.ok;
            // We have to mark the room as joined before asking for the room state
            // otherwise appservices will not be able to find an intent to use
            // to fetch the sate with.
            const previousRevision = this.joinedRoomsRevision;
            this.joinedRoomsRevision =
                this.joinedRoomsRevision.reviseFromJoinedRooms(joinedRooms);
            const failedPreemptiveJoins = preemptivelyJoinedRoomsToCheck.filter((roomID) => !this.joinedRoomsRevision.isJoinedRoom(roomID));
            if (failedPreemptiveJoins.length > 0) {
                log.error(`A caller to ClientRooms preemptTimelineJoin is using the method inappropriately. You should alert the developers with logs and any other context for what you just did, the rooms that were added are:`, failedPreemptiveJoins);
            }
            for (const roomID of preemptivelyJoinedRoomsToCheck) {
                // all checked rooms need deleting, regardless of whether someone lied to us about joining a room.
                this.preemptivelyJoinedRooms.delete(roomID);
            }
            const changes = {
                ...previousRevision.changesFromJoinedRooms(joinedRooms),
                preemptivelyJoined: [],
                failedPreemptiveJoins,
            };
            // we have to emit before we preload room state so that the ClientsInRoomsMap can be updated.
            this.emit('revision', this.joinedRoomsRevision, changes, previousRevision);
        }
        finally {
            this.joinedRoomsCallLock.release();
        }
    }
    handleRoomChange(roomID) {
        this.roomPauser.pauseRoom(roomID, this.checkRoomTask.bind(this), this.handleTimelineEvent.bind(this));
    }
}
exports.StandardClientRooms = StandardClientRooms;
//# sourceMappingURL=StandardClientRooms.js.map