"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.TrustedReporters = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const typebox_1 = require("@sinclair/typebox");
const MAX_REPORTED_EVENT_BACKLOG = 20;
const TrustedReportersProtectionSettings = typebox_1.Type.Object({
    mxids: typebox_1.Type.Array(matrix_protection_suite_1.StringUserIDSchema, {
        default: [],
        uniqueItems: true,
        description: "The users who are marked as trusted reporters.",
    }),
    alertThreshold: typebox_1.Type.Integer({
        default: 3,
        description: "The number of reports from trusted reporters required before sending an alert to the management room.",
    }),
    redactThreshold: typebox_1.Type.Integer({
        default: -1,
        description: "The number of reports from trusted reporters required before the relevant event is redacted.",
    }),
    banThreshold: typebox_1.Type.Integer({
        default: -1,
        description: "The number of reports from trusted reporters required before the reported user banned.",
    }),
}, { title: "TrustedReportersProtectionSettings" });
(0, matrix_protection_suite_1.describeProtection)({
    name: "TrustedReporters",
    description: "Count reports from trusted reporters and take a configured action",
    capabilityInterfaces: {
        userConsequences: "UserConsequences",
        eventConsequences: "EventConsequences",
    },
    defaultCapabilities: {
        userConsequences: "StandardUserConsequences",
        eventConsequences: "StandardEventConsequences",
    },
    configSchema: TrustedReportersProtectionSettings,
    factory: async function (description, lifetime, protectedRoomsSet, draupnir, capabilities, rawSettings) {
        const parsedSettings = description.protectionSettings.parseConfig(rawSettings);
        if ((0, matrix_protection_suite_1.isError)(parsedSettings)) {
            return parsedSettings;
        }
        return (0, matrix_protection_suite_1.Ok)(new TrustedReporters(description, lifetime, capabilities, protectedRoomsSet, draupnir, parsedSettings.ok));
    },
});
/*
 * Hold a list of users trusted to make reports, and enact consequences on
 * events that surpass configured report count thresholds
 */
class TrustedReporters extends matrix_protection_suite_1.AbstractProtection {
    constructor(description, lifetime, capabilities, protectedRoomsSet, draupnir, settings) {
        super(description, lifetime, capabilities, protectedRoomsSet, {});
        this.draupnir = draupnir;
        this.settings = settings;
        this.recentReported = new Map();
        this.reporters = new Set();
        this.userConsequences = capabilities.userConsequences;
        this.eventConsequences = capabilities.eventConsequences;
        this.reporters = new Set(settings.mxids);
    }
    async handleEventReport(report) {
        if (!this.reporters.has(report.sender)) {
            // not a trusted user, we're not interested
            return (0, matrix_protection_suite_1.Ok)(undefined);
        }
        let reporters = this.recentReported.get(report.event_id);
        if (reporters === undefined) {
            // first report we've seen recently for this event
            reporters = new Set();
            this.recentReported.set(report.event_id, reporters);
            if (this.recentReported.size > MAX_REPORTED_EVENT_BACKLOG) {
                // queue too big. push the oldest reported event off the queue
                const oldest = Array.from(this.recentReported.keys())[0];
                if (oldest === undefined) {
                    throw new TypeError(`There should be a key in this queue if it's at max size`);
                }
                this.recentReported.delete(oldest);
            }
        }
        reporters.add(report.sender);
        const met = [];
        if (reporters.size === this.settings.alertThreshold) {
            met.push("alert");
            // do nothing. let the `sendMessage` call further down be the alert
        }
        if (reporters.size === this.settings.redactThreshold) {
            met.push("redact");
            await this.eventConsequences.consequenceForEvent(report.room_id, report.event_id, "abuse detected");
        }
        if (reporters.size === this.settings.banThreshold) {
            met.push("ban");
            await this.userConsequences.consequenceForUserInRoom(report.room_id, report.event.sender, "abuse detected");
        }
        if (met.length > 0) {
            await this.draupnir.client.sendMessage(this.draupnir.config.managementRoom, {
                msgtype: "m.notice",
                body: `message ${report.event_id} reported by ${[...reporters].join(", ")}. ` +
                    `actions: ${met.join(", ")}`,
            });
        }
        return (0, matrix_protection_suite_1.Ok)(undefined);
    }
}
exports.TrustedReporters = TrustedReporters;
//# sourceMappingURL=TrustedReporters.js.map