"use strict";
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardPersistentConfigRenderer = void 0;
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const mps_interface_adaptor_1 = require("@the-draupnir-project/mps-interface-adaptor");
const ConfigStatusIndicator = Object.freeze({
    Ok: "✔",
    UseError: "⚠",
    ParseError: "❌",
});
function findError(propertyKey, errors) {
    const path = `/${propertyKey}`;
    return errors.find((error) => error.path.startsWith(path));
}
function findItemError(propertyKey, index, errors) {
    const path = `/${propertyKey}/${index}`;
    return errors.find((error) => error.path === path);
}
function renderPrimitiveValue(value, type) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, value),
        " ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("span", null,
            "(",
            type,
            ")")));
}
function renderConfigPropertyValue(value) {
    if (typeof value === "object" && value !== null) {
        if (value instanceof matrix_basic_types_1.MatrixRoomAlias || value instanceof matrix_basic_types_1.MatrixRoomID) {
            return (0, mps_interface_adaptor_1.renderRoomPill)(value);
        }
        else if (value instanceof matrix_basic_types_1.MatrixUserID) {
            return (0, mps_interface_adaptor_1.renderMentionPill)(value.toString(), value.toString());
        }
        else if (Array.isArray(value)) {
            if (value.length === 0) {
                return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
                    interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, "[]"),
                    " (empty array)"));
            }
            return interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, value.map((value) => renderConfigPropertyValue(value)));
        }
        else {
            return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
                interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, String(value)),
                " ",
                interface_manager_1.DeadDocumentJSX.JSXFactory("span", { "data-mx-color": "#D2691E" }, "(object)")));
        }
    }
    else if (typeof value === "string") {
        return renderPrimitiveValue(value, "string");
    }
    else if (typeof value === "number") {
        return renderPrimitiveValue(String(value), "number");
    }
    else {
        return renderPrimitiveValue(String(value), "unknown");
    }
}
function renderConfigPropertyItem(propertyKey, index, value, errors) {
    const error = findItemError(propertyKey, index, errors);
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
        renderConfigPropertyError(error),
        " ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, index),
        ":",
        " ",
        renderConfigPropertyValue(value)));
}
function renderConfigPropertyError(error) {
    if (error === undefined) {
        return ConfigStatusIndicator.Ok;
    }
    else if (error instanceof matrix_protection_suite_1.ConfigPropertyUseError) {
        return ConfigStatusIndicator.UseError;
    }
    else if (error instanceof matrix_protection_suite_1.ConfigParseError) {
        if (error.errors.every((e) => e instanceof matrix_protection_suite_1.ConfigPropertyUseError)) {
            return ConfigStatusIndicator.UseError;
        }
        else {
            return ConfigStatusIndicator.ParseError;
        }
    }
    else {
        return ConfigStatusIndicator.ParseError;
    }
}
function renderConfigProperty(configProperty, data, errors) {
    const propertyValue = data[configProperty.name];
    const error = findError(configProperty.name, errors);
    if (Array.isArray(propertyValue)) {
        return (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
                interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
                    renderConfigPropertyError(error),
                    " ",
                    interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, configProperty.name),
                    ":"),
                configProperty.description ?? "No description provided.",
                " default value: ",
                renderConfigPropertyValue(configProperty.default),
                ".",
                interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, propertyValue.map((value, index) => renderConfigPropertyItem(configProperty.name, index, value, errors))))));
    }
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
        renderConfigPropertyError(error),
        " ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, configProperty.name),
        ":",
        " ",
        configProperty.description ?? "No description provided.",
        " default value:",
        " ",
        renderConfigPropertyValue(configProperty.default),
        ".",
        interface_manager_1.DeadDocumentJSX.JSXFactory("p", null, renderConfigPropertyValue(propertyValue))));
}
function renderConfigDetails(error, description, children) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        renderConfigPropertyError(error),
        " ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, description.schema.title ?? "Untitled Config"),
        " ",
        children));
}
function renderBodgedConfig(config) {
    if (config.data === undefined) {
        return renderConfigDetails(undefined, config.description, interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null, "No data is currently persisted for this config."));
    }
    return renderConfigDetails(config.error, config.description, interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        "The config seems to be entirely invalid:",
        " ",
        renderConfigPropertyValue(config.data)));
}
function renderConfigDocumentation(description) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("p", null, description.schema.description ?? "No description"),
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, description.properties().map((property) => (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, property.name),
            ":",
            " ",
            property.description ?? "No description",
            interface_manager_1.DeadDocumentJSX.JSXFactory("p", null,
                "default value: ",
                renderConfigPropertyValue(property.default))))))));
}
exports.StandardPersistentConfigRenderer = Object.freeze({
    renderConfigStatus(config) {
        if (typeof config.data !== "object" || config.data === null) {
            return renderBodgedConfig(config);
        }
        return renderConfigDetails(config.error, config.description, interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, config.description
            .properties()
            .map((property) => renderConfigProperty(property, config.data, config.error?.errors ?? []))));
    },
    renderAdaptorStatus(info) {
        return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
            "Persistent configuration status:",
            interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, info.map((config) => (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null, this.renderConfigStatus(config)))))));
    },
    renderConfigDocumentation,
});
//# sourceMappingURL=PersistentConfigRenderer.js.map