"use strict";
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.SafeModeStatusCommand = void 0;
exports.safeModeHeader = safeModeHeader;
exports.renderSafeModeStatusInfo = renderSafeModeStatusInfo;
exports.safeModeStatusInfo = safeModeStatusInfo;
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const SafeModeCause_1 = require("../SafeModeCause");
const config_1 = require("../../config");
const SafeModeAdaptor_1 = require("./SafeModeAdaptor");
const RecoveryOptions_1 = require("../RecoveryOptions");
const RecoverCommand_1 = require("./RecoverCommand");
const PersistentConfigEditor_1 = require("../PersistentConfigEditor");
const PersistentConfigRenderer_1 = require("../PersistentConfigRenderer");
function safeModeHeader() {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("span", null, "\u26A0\uFE0F Draupnir is in safe mode (see status command) \u26A0\uFE0F"),
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null)));
}
function renderSafeModeCauseError(error) {
    if (error instanceof matrix_protection_suite_1.ActionException) {
        return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("p", null, "Draupnir is in safe mode because Draupnir failed to start."),
            interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
                interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
                    interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, error.mostRelevantElaboration)),
                "Details can be found by providing the reference",
                " ",
                interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, error.uuid),
                "to an administrator.",
                interface_manager_1.DeadDocumentJSX.JSXFactory("pre", null, error.toReadableString()))));
    }
    else {
        return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
            "Draupnir is in safe mode because Draupnir failed to start.",
            interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
                interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
                    interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, error.mostRelevantElaboration)),
                interface_manager_1.DeadDocumentJSX.JSXFactory("pre", null, error.toReadableString()))));
    }
}
function renderSafeModeCause(safeModeCause) {
    if (safeModeCause.reason === SafeModeCause_1.SafeModeReason.ByRequest) {
        return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
            "Draupnir is in safe mode by request of ",
            safeModeCause.user,
            "."));
    }
    else {
        return renderSafeModeCauseError(safeModeCause.error);
    }
}
function renderSafeModeStatusInfo(info, { showDocumentationURL = true } = {}) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("h3", null,
            "\u26A0\uFE0F ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, "Draupnir is in safe mode"),
            " \u26A0\uFE0F"),
        renderSafeModeCause(info.safeModeCause),
        (0, RecoveryOptions_1.renderRecoveryOptions)(info.safeModeCause),
        interface_manager_1.DeadDocumentJSX.JSXFactory("hr", null),
        PersistentConfigRenderer_1.StandardPersistentConfigRenderer.renderAdaptorStatus(info.configStatus),
        interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, "Version: "),
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, info.version),
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, "Repository: "),
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, info.repository),
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        showDocumentationURL ? (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, "Documentation: "),
            " ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("a", { href: info.documentationURL }, info.documentationURL))) : (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null)),
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        "To attempt to restart, use ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, "!draupnir restart"),
        "."));
}
function safeModeStatusInfo(cause, configStatus) {
    return {
        safeModeCause: cause,
        configStatus,
        documentationURL: config_1.DOCUMENTATION_URL,
        version: config_1.SOFTWARE_VERSION,
        repository: config_1.PACKAGE_JSON["repository"] ?? "Unknown",
    };
}
exports.SafeModeStatusCommand = (0, interface_manager_1.describeCommand)({
    summary: "Display the status of safe mode, including the reason Draupnir started in safe mode.",
    parameters: [],
    async executor(safeModeDraupnir) {
        const editor = new PersistentConfigEditor_1.StandardPersistentConfigEditor(safeModeDraupnir.client);
        const configStatus = await editor.supplementStatusWithSafeModeCause(safeModeDraupnir.cause);
        if ((0, matrix_protection_suite_1.isError)(configStatus)) {
            return configStatus;
        }
        return (0, matrix_protection_suite_1.Ok)(safeModeStatusInfo(safeModeDraupnir.cause, configStatus.ok));
    },
});
SafeModeAdaptor_1.SafeModeInterfaceAdaptor.describeRenderer(exports.SafeModeStatusCommand, {
    async arbritraryRenderer(context, eventContext, commandResult) {
        if ((0, matrix_protection_suite_1.isError)(commandResult)) {
            return (0, matrix_protection_suite_1.Ok)(undefined);
        }
        return await (0, RecoverCommand_1.sendAndAnnotateWithRecoveryOptions)(context, interface_manager_1.DeadDocumentJSX.JSXFactory("root", null, renderSafeModeStatusInfo(commandResult.ok)), { replyToEvent: eventContext.event });
    },
});
//# sourceMappingURL=StatusCommand.js.map