const node = require('@sentry/node');
const electron = require('electron');
const context = require('./context.js');
const ipc = require('./ipc.js');
const stackParse = require('./stack-parse.js');
const electronOfflineNet = require('./transports/electron-offline-net.js');
const version = require('./version.js');
const process$1 = require('../common/process.js');
const index = require('./integrations/sentry-minidump/index.js');
const electronBreadcrumbs = require('./integrations/electron-breadcrumbs.js');
const netBreadcrumbs = require('./integrations/net-breadcrumbs.js');
const mainContext = require('./integrations/main-context.js');
const childProcess = require('./integrations/child-process.js');
const onuncaughtexception = require('./integrations/onuncaughtexception.js');
const preloadInjection = require('./integrations/preload-injection.js');
const additionalContext = require('./integrations/additional-context.js');
const screenshots = require('./integrations/screenshots.js');
const mainProcessSession = require('./integrations/main-process-session.js');
const mode = require('../common/mode.js');

process$1.ensureProcess('main');
const defaultIntegrations = [
    new index.SentryMinidump(),
    new electronBreadcrumbs.ElectronBreadcrumbs(),
    new netBreadcrumbs.Net(),
    new mainContext.MainContext(),
    new childProcess.ChildProcess(),
    new onuncaughtexception.OnUncaughtException(),
    new preloadInjection.PreloadInjection(),
    new additionalContext.AdditionalContext(),
    new screenshots.Screenshots(),
    ...node.defaultIntegrations.filter((integration) => integration.name !== 'OnUncaughtException' && integration.name !== 'Context'),
];
const defaultOptions = {
    _metadata: { sdk: { name: 'sentry.javascript.electron', version: version.SDK_VERSION } },
    ipcMode: mode.IPCMode.Both,
    getSessions: () => [electron.session.defaultSession],
};
/**
 * Initialize Sentry in the Electron main process
 */
function init(userOptions) {
    var _a;
    const options = Object.assign(defaultOptions, userOptions);
    const defaults = defaultIntegrations;
    if (process.env.SENTRY_ANR_CHILD_PROCESS) {
        (_a = electron.app.dock) === null || _a === void 0 ? void 0 : _a.hide();
        options.autoSessionTracking = false;
        options.tracesSampleRate = 0;
    }
    // If we don't set a release, @sentry/node will automatically fetch from environment variables
    if (options.release === undefined) {
        options.release = context.getDefaultReleaseName();
    }
    // If we don't set an environment, @sentry/core defaults to production
    if (options.environment === undefined) {
        options.environment = context.getDefaultEnvironment();
    }
    // Unless autoSessionTracking is specifically disabled, we track sessions as the
    // lifetime of the Electron main process
    if (options.autoSessionTracking !== false) {
        defaults.push(new mainProcessSession.MainProcessSession());
        // We don't want nodejs autoSessionTracking
        options.autoSessionTracking = false;
    }
    if (options.stackParser === undefined) {
        options.stackParser = stackParse.defaultStackParser;
    }
    setDefaultIntegrations(defaults, options);
    if (options.dsn && options.transport === undefined) {
        options.transport = electronOfflineNet.makeElectronOfflineTransport;
    }
    ipc.configureIPC(options);
    node.init(options);
}
/** A list of integrations which cause default integrations to be removed */
const INTEGRATION_OVERRIDES = [
    { override: 'ElectronMinidump', remove: 'SentryMinidump' },
    { override: 'BrowserWindowSession', remove: 'MainProcessSession' },
];
/** Sets the default integrations and ensures that multiple minidump or session integrations are not enabled */
function setDefaultIntegrations(defaults, options) {
    if (options.defaultIntegrations === undefined) {
        const removeDefaultsMatching = (user, defaults) => {
            const toRemove = INTEGRATION_OVERRIDES.filter(({ override }) => user.some((i) => i.name === override)).map(({ remove }) => remove);
            return defaults.filter((i) => !toRemove.includes(i.name));
        };
        if (Array.isArray(options.integrations)) {
            options.defaultIntegrations = removeDefaultsMatching(options.integrations, defaults);
            return;
        }
        else if (typeof options.integrations === 'function') {
            const originalFn = options.integrations;
            options.integrations = (integrations) => {
                const resultIntegrations = originalFn(integrations);
                return removeDefaultsMatching(resultIntegrations, resultIntegrations);
            };
        }
        options.defaultIntegrations = defaults;
    }
}

exports.defaultIntegrations = defaultIntegrations;
exports.init = init;
//# sourceMappingURL=sdk.js.map
