"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mobx_1 = require("mobx");
const lodash_1 = __importDefault(require("lodash"));
const utils_1 = require("./utils");
const OptionsModel_1 = require("./models/OptionsModel");
class Validator {
    constructor(obj) {
        Object.defineProperty(this, "promises", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(this, "form", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: null
        });
        Object.defineProperty(this, "drivers", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(this, "plugins", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {
                vjf: undefined,
                dvr: undefined,
                svk: undefined,
                yup: undefined,
                zod: undefined,
                joi: undefined,
            }
        });
        Object.defineProperty(this, "error", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: null
        });
        (0, mobx_1.makeObservable)(this, {
            error: mobx_1.observable,
            validate: mobx_1.action,
            validateField: mobx_1.action,
        });
        this.form = obj.form;
        Object.assign(this.plugins, obj.plugins);
        this.initDrivers();
    }
    initDrivers() {
        lodash_1.default.map(this.plugins, (plugin, key) => (this.drivers[key] = (plugin && plugin.class) &&
            new plugin.class({
                config: plugin.config,
                state: this.form.state,
                promises: this.promises,
            })));
    }
    validate(opt, obj) {
        const path = (0, utils_1.$try)(opt === null || opt === void 0 ? void 0 : opt.path, opt);
        const instance = (0, utils_1.$try)(opt === null || opt === void 0 ? void 0 : opt.field, this.form.select(path, null, false), this.form);
        const related = (0, utils_1.$try)(opt === null || opt === void 0 ? void 0 : opt.related, obj === null || obj === void 0 ? void 0 : obj.related, true);
        const showErrors = (0, utils_1.$try)(opt === null || opt === void 0 ? void 0 : opt.showErrors, obj === null || obj === void 0 ? void 0 : obj.showErrors, false);
        instance.$validating = true;
        instance.$validated += 1;
        this.error = null;
        return new Promise((resolve) => {
            // validate instance (form or filed)
            if (instance.path || lodash_1.default.isString(path)) {
                this.validateField({
                    field: instance,
                    showErrors,
                    related,
                    path,
                });
            }
            // validate nested fields
            instance.each((field) => this.validateField({
                path: field.path,
                field: field,
                showErrors,
                related,
            }));
            // wait all promises
            resolve(Promise.all(this.promises));
        })
            .then((0, mobx_1.action)(() => {
            instance.$validating = false;
            instance.$clearing = false;
            instance.$resetting = false;
        }))
            .catch((0, mobx_1.action)((err) => {
            instance.$validating = false;
            instance.$clearing = false;
            instance.$resetting = false;
            throw err;
        }))
            .then(() => instance);
    }
    validateField({ showErrors = false, related = false, field = null, path, }) {
        const instance = field || this.form.select(path);
        const { options } = this.form.state;
        // check if the field is a valid instance
        if (!instance.path)
            throw new Error("Validation Error: Invalid Field Instance");
        // do not validate soft deleted fields
        if (instance.deleted && !options.get(OptionsModel_1.OptionsEnum.validateDeletedFields, instance))
            return;
        // do not validate disabled fields
        if (instance.disabled && !options.get(OptionsModel_1.OptionsEnum.validateDisabledFields, instance))
            return;
        // do not validate pristine fields
        if (instance.isPristine && !options.get(OptionsModel_1.OptionsEnum.validatePristineFields, instance))
            return;
        // reset validation before validate
        if (options.get(OptionsModel_1.OptionsEnum.resetValidationBeforeValidate, instance))
            instance.resetValidation();
        // trim string value before validation
        if (options.get(OptionsModel_1.OptionsEnum.validateTrimmedValue, instance))
            instance.trim();
        // get stop on error
        const stopOnError = options.get(OptionsModel_1.OptionsEnum.stopValidationOnError, instance);
        // get validation plugin order
        const validationPluginOrder = options.get(OptionsModel_1.OptionsEnum.validationPluginsOrder, instance);
        const drivers = validationPluginOrder
            ? validationPluginOrder.map((n) => this.drivers[n])
            : this.drivers;
        // validate with all enabled drivers
        lodash_1.default.each(drivers, (driver) => {
            driver && driver.validate(instance);
            if (stopOnError && instance.hasError)
                return;
        });
        // send error to the view
        instance.showErrors(showErrors, false);
        // related validation
        if (related)
            this.validateRelatedFields(instance, showErrors);
    }
    /**
      Validate 'related' fields if specified
      and related validation allowed (recursive)
    */
    validateRelatedFields(field, showErrors) {
        if (!field.related || !field.related.length)
            return;
        field.related.map((path) => this.validateField({
            related: false,
            showErrors,
            path,
        }));
    }
}
exports.default = Validator;
//# sourceMappingURL=Validator.js.map