"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.pathToFieldsTree = exports.prepareFieldsData = exports.handleFieldsArrayOfStrings = exports.handleFieldsNested = exports.mergeSchemaDefaults = exports.parseCheckOutput = exports.parseCheckArray = exports.parseArrayProp = exports.parsePath = exports.parseInput = exports.defaultValue = void 0;
const lodash_1 = __importDefault(require("lodash"));
const FieldProps_1 = require("./models/FieldProps");
const utils_1 = require("./utils");
const defaultValue = ({ type = undefined, value = undefined, isEmptyArray = false, fallbackValueOption = "", }) => {
    if (Array.isArray(value) || isEmptyArray)
        return [];
    if (lodash_1.default.isDate(value) || type === "date" || type === "datetime-local")
        return null;
    if (lodash_1.default.isNumber(value) || type === "number")
        return 0;
    if (lodash_1.default.isBoolean(value) || type === "checkbox")
        return false;
    if (lodash_1.default.isString(value) || type === "file")
        return "";
    return fallbackValueOption;
};
exports.defaultValue = defaultValue;
const parsePath = (path) => {
    let $path = path;
    $path = lodash_1.default.replace($path, new RegExp("\\[", "g"), ".");
    $path = lodash_1.default.replace($path, new RegExp("\\]", "g"), "");
    return $path;
};
exports.parsePath = parsePath;
const parseInput = (input, { fallbackValueOption = "", type, isEmptyArray, separated, unified, fallback }) => input((0, utils_1.$try)(separated, unified, fallback, defaultValue({
    fallbackValueOption,
    type,
    isEmptyArray,
})));
exports.parseInput = parseInput;
const parseArrayProp = (val, prop, removeNullishValuesInArrays) => {
    const values = lodash_1.default.values(val);
    const isValProp = [
        FieldProps_1.FieldPropsEnum.value,
        FieldProps_1.FieldPropsEnum.initial,
        FieldProps_1.FieldPropsEnum.default,
    ].includes(prop);
    if (removeNullishValuesInArrays && isValProp) {
        return lodash_1.default.without(values, ...[null, undefined, ""]);
    }
    return values;
};
exports.parseArrayProp = parseArrayProp;
const parseCheckArray = (field, value, prop, removeNullishValuesInArrays) => {
    if (field.incremental && lodash_1.default.isObject(value) && lodash_1.default.isEmpty(value))
        return [];
    return field.hasIncrementalKeys ? parseArrayProp(value, prop, removeNullishValuesInArrays) : value;
};
exports.parseCheckArray = parseCheckArray;
const parseCheckOutput = (field, prop, retrieveNullifiedEmptyStrings = false) => {
    if (prop === FieldProps_1.FieldPropsEnum.value || prop.startsWith("value.")) {
        const base = field.$output ? field.$output(field[FieldProps_1.FieldPropsEnum.value]) : field[FieldProps_1.FieldPropsEnum.value];
        const value = prop.startsWith("value.") ? lodash_1.default.get(base, prop.substring(6)) : base;
        if (lodash_1.default.isString(value) && lodash_1.default.isEmpty(value) && retrieveNullifiedEmptyStrings)
            return null;
        return value;
    }
    return field[prop];
};
exports.parseCheckOutput = parseCheckOutput;
const defineFieldsFromStruct = (struct, add = false) => lodash_1.default.reduceRight(struct, ($, name) => {
    const obj = {};
    if (lodash_1.default.endsWith(name, "[]")) {
        const val = add ? [$] : [];
        obj[lodash_1.default.trimEnd(name, "[]")] = val;
        return obj;
    }
    // no brakets
    const prev = struct[struct.indexOf(name) - 1];
    const stop = lodash_1.default.endsWith(prev, "[]") && lodash_1.default.last(struct) === name;
    if (!add && stop)
        return obj;
    obj[name] = $;
    return obj;
}, {});
const handleFieldsArrayOfStrings = ($fields, add = false) => {
    let fields = $fields;
    // handle array with field struct (strings)
    if ((0, utils_1.isArrayOfStrings)(fields)) {
        fields = lodash_1.default.transform(fields, ($obj, $) => {
            const pathStruct = lodash_1.default.split($, ".");
            // as array of strings (with empty values)
            if (!pathStruct.length)
                return Object.assign($obj, { [$]: "" });
            // define flat or nested fields from pathStruct
            return lodash_1.default.merge($obj, defineFieldsFromStruct(pathStruct, add));
        }, {});
    }
    return fields;
};
exports.handleFieldsArrayOfStrings = handleFieldsArrayOfStrings;
const handleFieldsArrayOfObjects = ($fields) => {
    let fields = $fields;
    // handle array of objects (with unified props)
    if ((0, utils_1.isArrayOfObjects)(fields)) {
        fields = lodash_1.default.transform(fields, ($obj, field) => {
            if ((0, utils_1.hasUnifiedProps)({ fields: { field } }) && !lodash_1.default.has(field, FieldProps_1.FieldPropsEnum.name))
                return undefined;
            return Object.assign($obj, { [field.name]: field });
        }, {});
    }
    return fields;
};
const handleFieldsNested = (fields, strictProps = true) => lodash_1.default.transform(fields, (obj, field, key) => {
    if ((0, utils_1.allowNested)(field, strictProps)) {
        // define nested field
        return Object.assign(obj, {
            [key]: {
                fields: (0, utils_1.isEmptyArray)(field) ? [] : handleFieldsNested(field),
            },
        });
    }
    return Object.assign(obj, { [key]: field });
}, {});
exports.handleFieldsNested = handleFieldsNested;
/* mapNestedValuesToUnifiedValues

FROM:

{
  street: '123 Fake St.',
  zip: '12345',
}

TO:

[{
  name: 'street'
  value: '123 Fake St.',
}, {
  name: 'zip'
  value: '12345',
}]

*/
const mapNestedValuesToUnifiedValues = (data) => lodash_1.default.isPlainObject(data)
    ? lodash_1.default.map(data, (value, name) => ({ value, name }))
    : undefined;
/* reduceValuesToUnifiedFields

FROM:

{
  name: 'fatty',
  address: {
    street: '123 Fake St.',
    zip: '12345',
  },
};

TO:

{
  name: {
    value: 'fatty',
    fields: undefined
  },
  address: {
    value: {
      street: '123 Fake St.',
      zip: '12345'
    },
    fields: [ ... ]
  },
};

*/
const reduceValuesToUnifiedFields = (values) => lodash_1.default.transform(values, (obj, value, key) => Object.assign(obj, {
    [key]: {
        value,
        fields: mapNestedValuesToUnifiedValues(value),
    },
}), {});
/*
  Fallback Unified Props to Separated Mode
*/
const handleFieldsPropsFallback = (fields, initial, fallback) => {
    if (!lodash_1.default.has(initial, FieldProps_1.SeparatedPropsMode.values))
        return fields;
    // if the 'values' object is passed in constructor
    // then update the fields definitions
    let { values } = initial;
    if ((0, utils_1.hasUnifiedProps)({ fields: initial.fields })) {
        values = reduceValuesToUnifiedFields(values);
    }
    return lodash_1.default.merge(fields, lodash_1.default.transform(values, (result, v, k) => {
        if (Array.isArray(fields[k]))
            result[k] = v;
        if (!(k in fields) && (!isNaN(Number(k)) || fallback))
            result[k] = v;
    }, {}));
};
const mergeSchemaDefaults = (fields, validator) => {
    if (validator) {
        const schema = lodash_1.default.get(validator.plugins, "svk.config.schema");
        if (lodash_1.default.isEmpty(fields) && schema && !!schema.properties) {
            lodash_1.default.each(schema.properties, (prop, key) => {
                lodash_1.default.set(fields, key, {
                    value: prop.default,
                    label: prop.title,
                });
            });
        }
    }
    return fields;
};
exports.mergeSchemaDefaults = mergeSchemaDefaults;
const prepareFieldsData = (initial, strictProps = true, fallback = true) => {
    let fields = lodash_1.default.merge(handleFieldsArrayOfStrings(initial.fields, false), handleFieldsArrayOfStrings(initial.struct, false));
    fields = handleFieldsArrayOfObjects(fields);
    fields = handleFieldsPropsFallback(fields, initial, fallback);
    fields = handleFieldsNested(fields, strictProps);
    return fields;
};
exports.prepareFieldsData = prepareFieldsData;
const pathToFieldsTree = (struct, path, n = 0, add = false) => {
    const structPath = (0, utils_1.pathToStruct)(path);
    const structArray = lodash_1.default.filter(struct, (item) => lodash_1.default.startsWith(item, structPath));
    const $tree = handleFieldsArrayOfStrings(structArray, add);
    const $struct = lodash_1.default.replace(structPath, new RegExp("\\[]", "g"), `[${n}]`);
    const fields = handleFieldsNested(lodash_1.default.get($tree, $struct));
    // fix issues #614 & #615
    struct.length && struct
        .filter(s => s.startsWith(path + '[]'))
        .map(s => s.substring((path + '[].').length))
        .filter(s => s.endsWith('[]'))
        .map(s => s.substring(0, s.length - 2))
        .forEach(s => {
        var _a, _b;
        const ss = s.split('.');
        let t = (_a = fields[0]) === null || _a === void 0 ? void 0 : _a.fields;
        for (let i = 0; i < ss.length; i++) {
            t = (_b = t === null || t === void 0 ? void 0 : t[ss[i]]) === null || _b === void 0 ? void 0 : _b[FieldProps_1.FieldPropsEnum.fields];
            if (!t)
                break;
        }
        if (t)
            delete t[0];
    });
    return fields;
};
exports.pathToFieldsTree = pathToFieldsTree;
//# sourceMappingURL=parser.js.map