"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DVR = void 0;
const lodash_1 = __importDefault(require("lodash"));
/**
  Declarative Validation Rules

    const plugins = {
      dvr: dvr({
        package: validatorjs,
        extend: callback,
      }),
    };

*/
class DVR {
    constructor({ config, state = null, promises = [], }) {
        Object.defineProperty(this, "promises", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(this, "config", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: null
        });
        Object.defineProperty(this, "state", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: null
        });
        Object.defineProperty(this, "extend", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: null
        });
        Object.defineProperty(this, "validator", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: null
        });
        this.state = state;
        this.promises = promises;
        this.extend = config === null || config === void 0 ? void 0 : config.extend;
        this.validator = config.package;
        this.extendValidator();
    }
    extendValidator() {
        // extend using "extend" callback
        if (typeof this.extend === 'function') {
            this.extend({
                validator: this.validator,
                form: this.state.form,
            });
        }
    }
    validate(field) {
        // get form fields data
        const data = this.state.form.validatedValues;
        this.validateFieldAsync(field, data);
        this.validateFieldSync(field, data);
    }
    makeLabels(validation, field) {
        const labels = { [field.path]: field.label };
        lodash_1.default.forIn(validation.rules[field.path], (rule) => {
            if (typeof rule.value === "string" &&
                rule.name.match(/^(required_|same|different)/)) {
                lodash_1.default.forIn(rule.value.split(","), (p, i) => {
                    if (!rule.name.match(/^required_(if|unless)/) || i % 2 === 0) {
                        const f = this.state.form.$(p);
                        if (f && f.path && f.label) {
                            labels[f.path] = f.label;
                        }
                    }
                });
            }
            else if (typeof rule.value === "string" &&
                rule.name.match(/^(before|after)/)) {
                const f = this.state.form.$(rule.value);
                if (f && f.path && f.label) {
                    labels[f.path] = f.label;
                }
            }
        });
        validation.setAttributeNames(labels);
    }
    validateFieldSync(field, data) {
        const $rules = this.rules(field.rules, "sync");
        // exit if no rules found
        if (lodash_1.default.isEmpty($rules[0]))
            return;
        // get field rules
        const rules = { [field.path]: $rules };
        // create the validator instance
        const validation = new this.validator(data, rules);
        // set label into errors messages instead key
        this.makeLabels(validation, field);
        // check validation
        if (validation.passes())
            return;
        // the validation is failed, set the field error
        field.invalidate(lodash_1.default.head(validation.errors.get(field.path)), false);
    }
    validateFieldAsync(field, data) {
        const $rules = this.rules(field.rules, "async");
        // exit if no rules found
        if (lodash_1.default.isEmpty($rules[0]))
            return;
        // get field rules
        const rules = { [field.path]: $rules };
        // create the validator instance
        const validation = new this.validator(data, rules);
        // set label into errors messages instead key
        this.makeLabels(validation, field);
        const $p = new Promise((resolve) => validation.checkAsync(() => this.handleAsyncPasses(field, resolve), () => this.handleAsyncFails(field, validation, resolve)));
        this.promises.push($p);
    }
    handleAsyncPasses(field, resolve) {
        field.setValidationAsyncData(true);
        resolve();
    }
    handleAsyncFails(field, validation, resolve) {
        field.setValidationAsyncData(false, lodash_1.default.head(validation.errors.get(field.path)));
        this.executeAsyncValidation(field);
        resolve();
    }
    executeAsyncValidation(field) {
        if (field.validationAsyncData.valid === false) {
            field.invalidate(field.validationAsyncData.message, false, true);
        }
    }
    rules(rules, type) {
        const $rules = lodash_1.default.isString(rules) ? lodash_1.default.split(rules, "|") : rules;
        // eslint-disable-next-line new-cap
        const v = new this.validator();
        return lodash_1.default.filter($rules, ($rule) => type === "async"
            ? v.getRule(lodash_1.default.split($rule, ":")[0]).async
            : !v.getRule(lodash_1.default.split($rule, ":")[0]).async);
    }
}
exports.DVR = DVR;
exports.default = (config) => ({
    class: DVR,
    config,
});
//# sourceMappingURL=DVR.js.map