/*=========================================================================
 *
 *  Copyright NumFOCUS
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *         https://www.apache.org/licenses/LICENSE-2.0.txt
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *=========================================================================*/
#ifndef itkNeighborhoodBinaryThresholdImageFunction_hxx
#define itkNeighborhoodBinaryThresholdImageFunction_hxx

#include "itkNumericTraits.h"
#include "itkConstNeighborhoodIterator.h"

namespace itk
{

template <typename TInputImage, typename TCoordinate>
NeighborhoodBinaryThresholdImageFunction<TInputImage, TCoordinate>::NeighborhoodBinaryThresholdImageFunction()
{
  m_Radius.Fill(1);
}

template <typename TInputImage, typename TCoordinate>
void
NeighborhoodBinaryThresholdImageFunction<TInputImage, TCoordinate>::PrintSelf(std::ostream & os, Indent indent) const
{
  this->Superclass::PrintSelf(os, indent);

  os << indent << "Radius: " << m_Radius << std::endl;
}

template <typename TInputImage, typename TCoordinate>
bool
NeighborhoodBinaryThresholdImageFunction<TInputImage, TCoordinate>::EvaluateAtIndex(const IndexType & index) const
{
  if (!this->GetInputImage())
  {
    return false;
  }

  if (!this->IsInsideBuffer(index))
  {
    return false;
  }

  // Create an N-d neighborhood kernel, using a zeroflux boundary condition
  ConstNeighborhoodIterator<InputImageType> it(
    m_Radius, this->GetInputImage(), this->GetInputImage()->GetBufferedRegion());

  // Set the iterator at the desired location
  it.SetLocation(index);

  // Walk the neighborhood
  bool               allInside = true;
  const PixelType    lower = this->GetLower();
  const PixelType    upper = this->GetUpper();
  PixelType          value;
  const unsigned int size = it.Size();
  for (unsigned int i = 0; i < size; ++i)
  {
    value = it.GetPixel(i);
    if (lower > value || value > upper)
    {
      allInside = false;
      break;
    }
  }

  return allInside;
}
} // end namespace itk

#endif
