/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing#kwsys for details.  */
#ifndef itksys_SystemTools_hxx
#define itksys_SystemTools_hxx

#include <itksys/Configure.hxx>
#include <itksys/Status.hxx>

#include <iosfwd>
#include <map>
#include <string>
#include <vector>

#include <sys/types.h>
// include sys/stat.h after sys/types.h
#include <sys/stat.h>

#if !defined(_WIN32) || defined(__CYGWIN__)
#  include <unistd.h> // For access permissions for use with access()
#endif

// Required for va_list
#include <stdarg.h>
// Required for FILE*
#include <stdio.h>
#if !defined(va_list)
// Some compilers move va_list into the std namespace and there is no way to
// tell that this has been done. Playing with things being included before or
// after stdarg.h does not solve things because we do not have control over
// what the user does. This hack solves this problem by moving va_list to our
// own namespace that is local for kwsys.
namespace std {
} // Required for platforms that do not have std namespace
namespace itksys_VA_LIST {
using namespace std;
typedef va_list hack_va_list;
}
namespace itksys {
typedef itksys_VA_LIST::hack_va_list va_list;
}
#endif // va_list

namespace itksys {

class SystemToolsStatic;

/** \class SystemToolsManager
 * \brief Use to make sure SystemTools is initialized before it is used
 * and is the last static object destroyed
 */
class itksys_EXPORT SystemToolsManager
{
public:
  SystemToolsManager();
  ~SystemToolsManager();

  SystemToolsManager(SystemToolsManager const&) = delete;
  SystemToolsManager& operator=(SystemToolsManager const&) = delete;
};

// This instance will show up in any translation unit that uses
// SystemTools. It will make sure SystemTools is initialized
// before it is used and is the last static object destroyed.
static SystemToolsManager SystemToolsManagerInstance;

// Flags for use with TestFileAccess.  Use a typedef in case any operating
// system in the future needs a special type.  These are flags that may be
// combined using the | operator.
typedef int TestFilePermissions;
#if defined(_WIN32) && !defined(__CYGWIN__)
// On Windows (VC), no system header defines these constants...
static TestFilePermissions const TEST_FILE_OK = 0;
static TestFilePermissions const TEST_FILE_READ = 4;
static TestFilePermissions const TEST_FILE_WRITE = 2;
static TestFilePermissions const TEST_FILE_EXECUTE = 1;
#else
// Standard POSIX constants
static TestFilePermissions const TEST_FILE_OK = F_OK;
static TestFilePermissions const TEST_FILE_READ = R_OK;
static TestFilePermissions const TEST_FILE_WRITE = W_OK;
static TestFilePermissions const TEST_FILE_EXECUTE = X_OK;
#endif

/** \class SystemTools
 * \brief A collection of useful platform-independent system functions.
 */
class itksys_EXPORT SystemTools
{
public:
  /** -----------------------------------------------------------------
   *               String Manipulation Routines
   *  -----------------------------------------------------------------
   */

  /**
   * Replace symbols in str that are not valid in C identifiers as
   * defined by the 1999 standard, ie. anything except [A-Za-z0-9_].
   * They are replaced with `_' and if the first character is a digit
   * then an underscore is prepended.  Note that this can produce
   * identifiers that the standard reserves (_[A-Z].* and __.*).
   */
  static std::string MakeCidentifier(std::string const& s);

  static std::string MakeCindentifier(std::string const& s)
  {
    return MakeCidentifier(s);
  }

  /**
   * Replace replace all occurrences of the string in the source string.
   */
  static void ReplaceString(std::string& source, char const* replace,
                            char const* with);
  static void ReplaceString(std::string& source, std::string const& replace,
                            std::string const& with);

  /**
   * Return a capitalized string (i.e the first letter is uppercased,
   * all other are lowercased).
   */
  static std::string Capitalized(std::string const&);

  /**
   * Return a 'capitalized words' string (i.e the first letter of each word
   * is uppercased all other are left untouched though).
   */
  static std::string CapitalizedWords(std::string const&);

  /**
   * Return a 'uncapitalized words' string (i.e the first letter of each word
   * is lowercased all other are left untouched though).
   */
  static std::string UnCapitalizedWords(std::string const&);

  /**
   * Return a lower case string
   */
  static std::string LowerCase(std::string const&);

  /**
   * Return a lower case string
   */
  static std::string UpperCase(std::string const&);

  /**
   * Count char in string
   */
  static size_t CountChar(char const* str, char c);

  /**
   * Remove some characters from a string.
   * Return a pointer to the new resulting string (allocated with 'new')
   */
  static char* RemoveChars(char const* str, char const* toremove);

  /**
   * Remove remove all but 0->9, A->F characters from a string.
   * Return a pointer to the new resulting string (allocated with 'new')
   */
  static char* RemoveCharsButUpperHex(char const* str);

  /**
   * Replace some characters by another character in a string (in-place)
   * Return a pointer to string
   */
  static char* ReplaceChars(char* str, char const* toreplace,
                            char replacement);

  /**
   * Returns true if str1 starts (respectively ends) with str2
   */
  static bool StringStartsWith(char const* str1, char const* str2);
  static bool StringStartsWith(std::string const& str1, char const* str2);
  static bool StringEndsWith(char const* str1, char const* str2);
  static bool StringEndsWith(std::string const& str1, char const* str2);

  /**
   * Returns a pointer to the last occurrence of str2 in str1
   */
  static char const* FindLastString(char const* str1, char const* str2);

  /**
   * Make a duplicate of the string similar to the strdup C function
   * but use new to create the 'new' string, so one can use
   * 'delete' to remove it. Returns 0 if the input is empty.
   */
  static char* DuplicateString(char const* str);

  /**
   * Return the string cropped to a given length by removing chars in the
   * center of the string and replacing them with an ellipsis (...)
   */
  static std::string CropString(std::string const&, size_t max_len);

  /** split a path by separator into an array of strings, default is /.
      If isPath is true then the string is treated like a path and if
      s starts with a / then the first element of the returned array will
      be /, so /foo/bar will be [/, foo, bar]
  */
  static std::vector<std::string> SplitString(std::string const& s,
                                              char separator = '/',
                                              bool isPath = false);
  /**
   * Perform a case-independent string comparison
   */
  static int Strucmp(char const* s1, char const* s2);

  /**
   * Split a string on its newlines into multiple lines
   * Return false only if the last line stored had no newline
   */
  static bool Split(std::string const& s, std::vector<std::string>& l);
  static bool Split(std::string const& s, std::vector<std::string>& l,
                    char separator);

  /**
   * Joins a vector of strings into a single string, with separator in between
   * each string.
   */
  static std::string Join(std::vector<std::string> const& list,
                          std::string const& separator);

  /**
   * Return string with space added between capitalized words
   * (i.e. EatMyShorts becomes Eat My Shorts )
   * (note that IEatShorts becomes IEat Shorts)
   */
  static std::string AddSpaceBetweenCapitalizedWords(std::string const&);

  /**
   * Append two or more strings and produce new one.
   * Programmer must 'delete []' the resulting string, which was allocated
   * with 'new'.
   * Return 0 if inputs are empty or there was an error
   */
  static char* AppendStrings(char const* str1, char const* str2);
  static char* AppendStrings(char const* str1, char const* str2,
                             char const* str3);

  /**
   * Estimate the length of the string that will be produced
   * from printing the given format string and arguments.  The
   * returned length will always be at least as large as the string
   * that will result from printing.
   * WARNING: since va_arg is called to iterate of the argument list,
   * you will not be able to use this 'ap' anymore from the beginning.
   * It's up to you to call va_end though.
   */
  static int EstimateFormatLength(char const* format, va_list ap);

  /**
   * Escape specific characters in 'str'.
   */
  static std::string EscapeChars(char const* str, char const* chars_to_escape,
                                 char escape_char = '\\');

  /** -----------------------------------------------------------------
   *               Filename Manipulation Routines
   *  -----------------------------------------------------------------
   */

  /**
   * Replace Windows file system slashes with Unix-style slashes.
   */
  static void ConvertToUnixSlashes(std::string& path);

#ifdef _WIN32
  /** Calls Encoding::ToWindowsExtendedPath.  */
  static std::wstring ConvertToWindowsExtendedPath(std::string const&);
#endif

  /**
   * For windows this calls ConvertToWindowsOutputPath and for unix
   * it calls ConvertToUnixOutputPath
   */
  static std::string ConvertToOutputPath(std::string const&);

  /**
   * Convert the path to a string that can be used in a unix makefile.
   * double slashes are removed, and spaces are escaped.
   */
  static std::string ConvertToUnixOutputPath(std::string const&);

  /**
   * Convert the path to string that can be used in a windows project or
   * makefile.   Double slashes are removed if they are not at the start of
   * the string, the slashes are converted to windows style backslashes, and
   * if there are spaces in the string it is double quoted.
   */
  static std::string ConvertToWindowsOutputPath(std::string const&);

  /**
   * Return true if a path with the given name exists in the current directory.
   */
  static bool PathExists(std::string const& path);

  /**
   * Return true if a file exists in the current directory.
   * If isFile = true, then make sure the file is a file and
   * not a directory.  If isFile = false, then return true
   * if it is a file or a directory.  Note that the file will
   * also be checked for read access.  (Currently, this check
   * for read access is only done on POSIX systems.)
   */
  static bool FileExists(char const* filename, bool isFile);
  static bool FileExists(std::string const& filename, bool isFile);
  static bool FileExists(char const* filename);
  static bool FileExists(std::string const& filename);

  /**
   * Test if a file exists and can be accessed with the requested
   * permissions.  Symbolic links are followed.  Returns true if
   * the access test was successful.
   *
   * On POSIX systems (including Cygwin), this maps to the access
   * function.  On Windows systems, all existing files are
   * considered readable, and writable files are considered to
   * have the read-only file attribute cleared.
   */
  static bool TestFileAccess(char const* filename,
                             TestFilePermissions permissions);
  static bool TestFileAccess(std::string const& filename,
                             TestFilePermissions permissions);
/**
 * Cross platform wrapper for stat struct
 */
#if defined(_WIN32) && !defined(__CYGWIN__)
  typedef struct _stat64 Stat_t;
#else
  typedef struct stat Stat_t;
#endif

  /**
   * Cross platform wrapper for stat system call
   *
   * On Windows this may not work for paths longer than 250 characters
   * due to limitations of the underlying '_wstat64' call.
   */
  static int Stat(char const* path, Stat_t* buf);
  static int Stat(std::string const& path, Stat_t* buf);

  /**
   * Return file length
   */
  static unsigned long FileLength(std::string const& filename);

  /**
     Change the modification time or create a file
  */
  static Status Touch(std::string const& filename, bool create);

  /**
   *  Compare file modification times.
   *  Return true for successful comparison and false for error.
   *  When true is returned, result has -1, 0, +1 for
   *  f1 older, same, or newer than f2.
   */
  static Status FileTimeCompare(std::string const& f1, std::string const& f2,
                                int* result);

  /**
   *  Get the file extension (including ".") needed for an executable
   *  on the current platform ("" for unix, ".exe" for Windows).
   */
  static char const* GetExecutableExtension();

  /**
   * Given a path on a Windows machine, return the actual case of
   * the path as it exists on disk.  Path components that do not
   * exist on disk are returned unchanged.  Relative paths are always
   * returned unchanged.  Drive letters are always made upper case.
   * This does nothing on non-Windows systems but return the path.
   */
  static std::string GetActualCaseForPath(std::string const& path);

  /**
   * Given the path to a program executable, get the directory part of
   * the path with the file stripped off.  If there is no directory
   * part, the empty string is returned.
   */
  static std::string GetProgramPath(std::string const&);
  static bool SplitProgramPath(std::string const& in_name, std::string& dir,
                               std::string& file, bool errorReport = true);

  /**
   * Given a path to a file or directory, convert it to a full path.
   * This collapses away relative paths relative to the cwd argument
   * (which defaults to the current working directory).  The full path
   * is returned.
   */
  static std::string CollapseFullPath(std::string const& in_path);
  static std::string CollapseFullPath(std::string const& in_path,
                                      char const* in_base);
  static std::string CollapseFullPath(std::string const& in_path,
                                      std::string const& in_base);

  /**
   * Get the real path for a given path, removing all symlinks.  In
   * the event of an error (non-existent path, permissions issue,
   * etc.) the original path is returned if errorMessage pointer is
   * nullptr.  Otherwise empty string is returned and errorMessage
   * contains error description.
   */
  static std::string GetRealPath(std::string const& path,
                                 std::string* errorMessage = nullptr);

  /**
   * Split a path name into its root component and the rest of the
   * path.  The root component is one of the following:
   *    "/"   = UNIX full path
   *    "c:/" = Windows full path (can be any drive letter)
   *    "c:"  = Windows drive-letter relative path (can be any drive letter)
   *    "//"  = Network path
   *    "~/"  = Home path for current user
   *    "~u/" = Home path for user 'u'
   *    ""    = Relative path
   *
   * A pointer to the rest of the path after the root component is
   * returned.  The root component is stored in the "root" string if
   * given.
   */
  static char const* SplitPathRootComponent(std::string const& p,
                                            std::string* root = nullptr);

  /**
   * Split a path name into its basic components.  The first component
   * always exists and is the root returned by SplitPathRootComponent.
   * The remaining components form the path.  If there is a trailing
   * slash then the last component is the empty string.  The
   * components can be recombined as "c[0]c[1]/c[2]/.../c[n]" to
   * produce the original path.  Home directory references are
   * automatically expanded if expand_home_dir is true and this
   * platform supports them.
   *
   * This does *not* normalize the input path.  All components are
   * preserved, including empty ones.  Typically callers should use
   * this only on paths that have already been normalized.
   */
  static void SplitPath(std::string const& p,
                        std::vector<std::string>& components,
                        bool expand_home_dir = true);

  /**
   * Join components of a path name into a single string.  See
   * SplitPath for the format of the components.
   *
   * This does *not* normalize the input path.  All components are
   * preserved, including empty ones.  Typically callers should use
   * this only on paths that have already been normalized.
   */
  static std::string JoinPath(std::vector<std::string> const& components);
  static std::string JoinPath(std::vector<std::string>::const_iterator first,
                              std::vector<std::string>::const_iterator last);

  /**
   * Compare a path or components of a path.
   */
  static bool ComparePath(std::string const& c1, std::string const& c2);

  /**
   * Return path of a full filename (no trailing slashes)
   */
  static std::string GetFilenamePath(std::string const&);

  /**
   * Return file name of a full filename (i.e. file name without path)
   */
  static std::string GetFilenameName(std::string const&);

  /**
   * Return longest file extension of a full filename (dot included)
   */
  static std::string GetFilenameExtension(std::string const&);

  /**
   * Return shortest file extension of a full filename (dot included)
   */
  static std::string GetFilenameLastExtension(std::string const& filename);

  /**
   * Return file name without extension of a full filename
   */
  static std::string GetFilenameWithoutExtension(std::string const&);

  /**
   * Return file name without its last (shortest) extension
   */
  static std::string GetFilenameWithoutLastExtension(std::string const&);

  /**
   * Return whether the path represents a full path (not relative)
   */
  static bool FileIsFullPath(std::string const&);
  static bool FileIsFullPath(char const*);

  /**
   * For windows return the short path for the given path,
   * Unix just a pass through
   */
  static Status GetShortPath(std::string const& path, std::string& result);

  /**
   * Read line from file. Make sure to read a full line and truncates it if
   * requested via sizeLimit. Returns true if any data were read before the
   * end-of-file was reached. If the has_newline argument is specified, it will
   * be true when the line read had a newline character.
   */
  static bool GetLineFromStream(
    std::istream& istr, std::string& line, bool* has_newline = nullptr,
    std::string::size_type sizeLimit = std::string::npos);

  /**
   * Get the parent directory of the directory or file
   */
  static std::string GetParentDirectory(std::string const& fileOrDir);

  /**
   * Check if the given file or directory is in subdirectory of dir
   */
  static bool IsSubDirectory(std::string const& fileOrDir,
                             std::string const& dir);

  /** -----------------------------------------------------------------
   *               File Manipulation Routines
   *  -----------------------------------------------------------------
   */

  /**
   * Open a file considering unicode. On Windows, if 'e' is present in
   * mode it is first discarded.
   */
  static FILE* Fopen(std::string const& file, char const* mode);

/**
 * Visual C++ does not define mode_t.
 */
#if defined(_MSC_VER)
  typedef unsigned short mode_t;
#endif

  /**
   * Make a new directory if it is not there.  This function
   * can make a full path even if none of the directories existed
   * prior to calling this function.
   */
  static Status MakeDirectory(char const* path, mode_t const* mode = nullptr);
  static Status MakeDirectory(std::string const& path,
                              mode_t const* mode = nullptr);

  /**
   * Represent the result of a file copy operation.
   * This is the result 'Status' and, if the operation failed,
   * an indication of whether the error occurred on the source
   * or destination path.
   */
  struct CopyStatus : public Status
  {
    enum WhichPath
    {
      NoPath,
      SourcePath,
      DestPath,
    };
    CopyStatus() = default;
    CopyStatus(Status s, WhichPath p)
      : Status(s)
      , Path(p)
    {
    }
    WhichPath Path = NoPath;
  };

  /**
   * Copy the source file to the destination file only
   * if the two files differ.
   */
  static CopyStatus CopyFileIfDifferent(std::string const& source,
                                        std::string const& destination);

  /**
   * Copy the source file to the destination file only
   * if the source file is newer than the destination file.
   */
  static CopyStatus CopyFileIfNewer(std::string const& source,
                                    std::string const& destination);

  /**
   * Compare the contents of two files.  Return true if different
   */
  static bool FilesDiffer(std::string const& source,
                          std::string const& destination);

  /**
   * Compare the contents of two files, ignoring line ending differences.
   * Return true if different
   */
  static bool TextFilesDiffer(std::string const& path1,
                              std::string const& path2);

  /**
   * Blockwise copy source to destination file
   */
  static CopyStatus CopyFileContentBlockwise(std::string const& source,
                                             std::string const& destination);
  /**
   * Clone the source file to the destination file
   */
  static CopyStatus CloneFileContent(std::string const& source,
                                     std::string const& destination);

  /**
   * Object encapsulating a unique identifier for a file
   * or directory
   */
#ifdef _WIN32
  class WindowsFileId
  {
  public:
    WindowsFileId() = default;
    WindowsFileId(unsigned long volumeSerialNumber,
                  unsigned long fileIndexHigh, unsigned long fileIndexLow);

    bool operator==(WindowsFileId const& o) const;
    bool operator!=(WindowsFileId const& o) const;

  private:
    unsigned long m_volumeSerialNumber;
    unsigned long m_fileIndexHigh;
    unsigned long m_fileIndexLow;
  };
  using FileId = WindowsFileId;
#else
  class UnixFileId
  {
  public:
    UnixFileId() = default;
    UnixFileId(dev_t volumeSerialNumber, ino_t fileSerialNumber,
               off_t fileSize);

    bool operator==(UnixFileId const& o) const;
    bool operator!=(UnixFileId const& o) const;

  private:
    dev_t m_volumeSerialNumber;
    ino_t m_fileSerialNumber;
    off_t m_fileSize;
  };
  using FileId = UnixFileId;
#endif

  /**
   * Outputs a FileId for the given file or directory.
   * Returns true on success, false on failure
   */
  static bool GetFileId(std::string const& file, FileId& id);

  /**
   * Return true if the two files are the same file
   */
  static bool SameFile(std::string const& file1, std::string const& file2);

  /**
   * Copy a file.
   */
  static CopyStatus CopyFileAlways(std::string const& source,
                                   std::string const& destination);

  enum class CopyWhen
  {
    Always,
    OnlyIfDifferent,
    OnlyIfNewer,
  };

  /**
   * Copy a file with specified copy behavior.
   */
  static CopyStatus CopyAFile(std::string const& source,
                              std::string const& destination,
                              CopyWhen when = CopyWhen::Always);
  static CopyStatus CopyAFile(std::string const& source,
                              std::string const& destination, bool always);

  /**
   * Copy content directory to another directory with all files and
   * subdirectories.  The "when" argument controls when files are copied:
   * Always: all files are always copied.
   * OnlyIfDifferent: only files that have changed are copied.
   * OnlyIfNewer: only files that are newer than the destination are copied.
   */
  static Status CopyADirectory(std::string const& source,
                               std::string const& destination,
                               CopyWhen when = CopyWhen::Always);
  static Status CopyADirectory(std::string const& source,
                               std::string const& destination, bool always);

  /**
   * Remove a file
   */
  static Status RemoveFile(std::string const& source);

  /**
   * Remove a directory
   */
  static Status RemoveADirectory(std::string const& source);

  /**
   * Get the maximum full file path length
   */
  static size_t GetMaximumFilePathLength();

  /**
   * Find a file in the system PATH, with optional extra paths
   */
  static std::string FindFile(
    std::string const& name,
    std::vector<std::string> const& path = std::vector<std::string>(),
    bool no_system_path = false);

  /**
   * Find a directory in the system PATH, with optional extra paths
   */
  static std::string FindDirectory(
    std::string const& name,
    std::vector<std::string> const& path = std::vector<std::string>(),
    bool no_system_path = false);

  /**
   * Find an executable in the system PATH, with optional extra paths
   */
  static std::string FindProgram(
    std::string const& name,
    std::vector<std::string> const& path = std::vector<std::string>(),
    bool no_system_path = false);

  /**
   * Return true if the file is a directory
   */
  static bool FileIsDirectory(std::string const& name);

  /**
   * Return true if the file is an executable
   */
  static bool FileIsExecutable(std::string const& name);

#if defined(_WIN32)
  /**
   * Return true if the file with FileAttributes `attr` is a symlink
   * Only available on Windows. This avoids an expensive `GetFileAttributesW`
   * call.
   */
  static bool FileIsSymlinkWithAttr(std::wstring const& path,
                                    unsigned long attr);
#endif

  /**
   * Return true if the file is a symlink
   */
  static bool FileIsSymlink(std::string const& name);

  /**
   * Return true if the file is a FIFO
   */
  static bool FileIsFIFO(std::string const& name);

  /**
   * Return true if the file has a given signature (first set of bytes)
   */
  static bool FileHasSignature(char const* filename, char const* signature,
                               long offset = 0);

  /**
   * Attempt to detect and return the type of a file.
   * Up to 'length' bytes are read from the file, if more than 'percent_bin' %
   * of the bytes are non-textual elements, the file is considered binary,
   * otherwise textual. Textual elements are bytes in the ASCII [0x20, 0x7E]
   * range, but also \\n, \\r, \\t.
   * The algorithm is simplistic, and should probably check for usual file
   * extensions, 'magic' signature, unicode, etc.
   */
  enum FileTypeEnum
  {
    FileTypeUnknown,
    FileTypeBinary,
    FileTypeText
  };
  static SystemTools::FileTypeEnum DetectFileType(char const* filename,
                                                  unsigned long length = 256,
                                                  double percent_bin = 0.05);

  /**
   * Create a symbolic link if the platform supports it.  Returns whether
   * creation succeeded.
   */
  static Status CreateSymlink(std::string const& origName,
                              std::string const& newName);

  /**
   * Read the contents of a symbolic link.  Returns whether reading
   * succeeded.
   */
  static Status ReadSymlink(std::string const& newName, std::string& origName);

  /**
   * Try to locate the file 'filename' in the directory 'dir'.
   * If 'filename' is a fully qualified filename, the basename of the file is
   * used to check for its existence in 'dir'.
   * If 'dir' is not a directory, GetFilenamePath() is called on 'dir' to
   * get its directory first (thus, you can pass a filename as 'dir', as
   * a convenience).
   * 'filename_found' is assigned the fully qualified name/path of the file
   * if it is found (not touched otherwise).
   * If 'try_filename_dirs' is true, try to find the file using the
   * components of its path, i.e. if we are looking for c:/foo/bar/bill.txt,
   * first look for bill.txt in 'dir', then in 'dir'/bar, then in 'dir'/foo/bar
   * etc.
   * Return true if the file was found, false otherwise.
   */
  static bool LocateFileInDir(char const* filename, char const* dir,
                              std::string& filename_found,
                              int try_filename_dirs = 0);

  /** compute the relative path from local to remote.  local must
      be a directory.  remote can be a file or a directory.
      Both remote and local must be full paths.  Basically, if
      you are in directory local and you want to access the file in remote
      what is the relative path to do that.  For example:
      /a/b/c/d to /a/b/c1/d1 -> ../../c1/d1
      from /usr/src to /usr/src/test/blah/foo.cpp -> test/blah/foo.cpp
  */
  static std::string RelativePath(std::string const& local,
                                  std::string const& remote);

  /**
   * Return file's modified time
   */
  static long int ModifiedTime(std::string const& filename);

  /**
   * Return file's creation time (Win32: works only for NTFS, not FAT)
   */
  static long int CreationTime(std::string const& filename);

  /**
   * Get and set permissions of the file.  If honor_umask is set, the umask
   * is queried and applied to the given permissions.  Returns false if
   * failure.
   *
   * WARNING:  A non-thread-safe method is currently used to get the umask
   * if a honor_umask parameter is set to true.
   */
  static Status GetPermissions(char const* file, mode_t& mode);
  static Status GetPermissions(std::string const& file, mode_t& mode);
  static Status SetPermissions(char const* file, mode_t mode,
                               bool honor_umask = false);
  static Status SetPermissions(std::string const& file, mode_t mode,
                               bool honor_umask = false);

  /** -----------------------------------------------------------------
   *               Time Manipulation Routines
   *  -----------------------------------------------------------------
   */

  /** Get current time in seconds since Posix Epoch (Jan 1, 1970).  */
  static double GetTime();

  /**
   * Get current date/time
   */
  static std::string GetCurrentDateTime(char const* format);

  /** -----------------------------------------------------------------
   *               Registry Manipulation Routines
   *  -----------------------------------------------------------------
   */

  /**
   * Specify access to the 32-bit or 64-bit application view of
   * registry values.  The default is to match the currently running
   * binary type.
   */
  enum KeyWOW64
  {
    KeyWOW64_Default,
    KeyWOW64_32,
    KeyWOW64_64
  };

  /**
   * Get a list of subkeys.
   */
  static bool GetRegistrySubKeys(std::string const& key,
                                 std::vector<std::string>& subkeys,
                                 KeyWOW64 view = KeyWOW64_Default);

  /**
   * Read a registry value
   */
  static bool ReadRegistryValue(std::string const& key, std::string& value,
                                KeyWOW64 view = KeyWOW64_Default);

  /**
   * Write a registry value
   */
  static bool WriteRegistryValue(std::string const& key,
                                 std::string const& value,
                                 KeyWOW64 view = KeyWOW64_Default);

  /**
   * Delete a registry value
   */
  static bool DeleteRegistryValue(std::string const& key,
                                  KeyWOW64 view = KeyWOW64_Default);

  /** -----------------------------------------------------------------
   *               Environment Manipulation Routines
   *  -----------------------------------------------------------------
   */

  /**
   *  Add the paths from the environment variable PATH to the
   *  string vector passed in.  If env is set then the value
   *  of env will be used instead of PATH.
   */
  static void GetPath(std::vector<std::string>& path,
                      char const* env = nullptr);

  /**
   * Read an environment variable
   */
  static char const* GetEnv(char const* key);
  static char const* GetEnv(std::string const& key);
  static bool GetEnv(char const* key, std::string& result);
  static bool GetEnv(std::string const& key, std::string& result);
  static bool HasEnv(char const* key);
  static bool HasEnv(std::string const& key);

  /** Put a string into the environment
      of the form var=value */
  static bool PutEnv(std::string const& env);

  /** Remove a string from the environment.
      Input is of the form "var" or "var=value" (value is ignored). */
  static bool UnPutEnv(std::string const& env);

  /**
   * Get current working directory CWD
   */
  static std::string GetCurrentWorkingDirectory();

  /**
   * Change directory to the directory specified
   */
  static Status ChangeDirectory(std::string const& dir);

  /**
   * Get the result of strerror(errno)
   */
  static std::string GetLastSystemError();

  /**
   * When building DEBUG with MSVC, this enables a hook that prevents
   * error dialogs from popping up if the program is being run from
   * DART.
   */
  static void EnableMSVCDebugHook();

  /**
   * Get the width of the terminal window. The code may or may not work, so
   * make sure you have some reasonable defaults prepared if the code returns
   * some bogus size.
   */
  static int GetTerminalWidth();

  /**
   * Delay the execution for a specified amount of time specified
   * in milliseconds
   */
  static void Delay(unsigned int msec);

  /**
   * Get the operating system name and version
   * This is implemented for Win32 only for the moment
   */
  static std::string GetOperatingSystemNameAndVersion();

  /** -----------------------------------------------------------------
   *               URL Manipulation Routines
   *  -----------------------------------------------------------------
   */

  /**
   * Parse a character string :
   *       protocol://dataglom
   * and fill protocol as appropriate.
   * decode the dataglom using DecodeURL if set to true.
   * Return false if the URL does not have the required form, true otherwise.
   */
  static bool ParseURLProtocol(std::string const& URL, std::string& protocol,
                               std::string& dataglom, bool decode = false);

  /**
   * Parse a string (a URL without protocol prefix) with the form:
   *  protocol://[[username[':'password]'@']hostname[':'dataport]]'/'[datapath]
   * and fill protocol, username, password, hostname, dataport, and datapath
   * when values are found.
   * decode all string except the protocol using DecodeUrl if set to true.
   * Return true if the string matches the format; false otherwise.
   */
  static bool ParseURL(std::string const& URL, std::string& protocol,
                       std::string& username, std::string& password,
                       std::string& hostname, std::string& dataport,
                       std::string& datapath, bool decode = false);

  /**
   * Decode the percent-encoded string from an URL or an URI
   * into their correct char values.
   * Does not perform any other sort of validation.
   * Return the decoded string
   */
  static std::string DecodeURL(std::string const& url);

private:
  static void ClassInitialize();
  static void ClassFinalize();

  /**
   * This method prevents warning on SGI
   */
  SystemToolsManager* GetSystemToolsManager()
  {
    return &SystemToolsManagerInstance;
  }

  friend class SystemToolsStatic;
  friend class SystemToolsManager;
};

} // namespace itksys

#endif
