"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const BaseModel_1 = require("../../BaseModel");
const Folder_1 = require("../../models/Folder");
const Note_1 = require("../../models/Note");
const test_utils_1 = require("../../testing/test-utils");
const restoreItems_1 = require("./restoreItems");
describe('restoreItems', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    it('should restore notes', async () => {
        const folder = await Folder_1.default.save({});
        const note1 = await Note_1.default.save({ parent_id: folder.id });
        const note2 = await Note_1.default.save({ parent_id: folder.id });
        await Note_1.default.delete(note1.id, { toTrash: true });
        await Note_1.default.delete(note2.id, { toTrash: true });
        expect((await Folder_1.default.noteIds(folder.id)).length).toBe(0);
        await (0, restoreItems_1.default)(BaseModel_1.ModelType.Note, [await Note_1.default.load(note1.id), await Note_1.default.load(note2.id)]);
        expect((await Folder_1.default.noteIds(folder.id)).length).toBe(2);
    });
    it('should restore folders and included notes', async () => {
        const folder1 = await Folder_1.default.save({});
        const note1 = await Note_1.default.save({ parent_id: folder1.id });
        const note2 = await Note_1.default.save({ parent_id: folder1.id });
        await Folder_1.default.delete(folder1.id, { toTrash: true });
        await (0, restoreItems_1.default)(BaseModel_1.ModelType.Folder, [await Folder_1.default.load(folder1.id)]);
        expect((await Folder_1.default.load(folder1.id)).deleted_time).toBe(0);
        expect((await Note_1.default.load(note1.id)).deleted_time).toBe(0);
        expect((await Note_1.default.load(note2.id)).deleted_time).toBe(0);
    });
    it('should restore folders and sub-folders', async () => {
        const folder1 = await Folder_1.default.save({});
        const folder2 = await Folder_1.default.save({ parent_id: folder1.id });
        const note1 = await Note_1.default.save({ parent_id: folder2.id });
        const note2 = await Note_1.default.save({ parent_id: folder2.id });
        const beforeTime = Date.now();
        await Folder_1.default.delete(folder1.id, { toTrash: true, deleteChildren: true });
        expect((await Folder_1.default.load(folder1.id)).deleted_time).toBeGreaterThanOrEqual(beforeTime);
        expect((await Folder_1.default.load(folder2.id)).deleted_time).toBeGreaterThanOrEqual(beforeTime);
        expect((await Note_1.default.load(note1.id)).deleted_time).toBeGreaterThanOrEqual(beforeTime);
        expect((await Note_1.default.load(note2.id)).deleted_time).toBeGreaterThanOrEqual(beforeTime);
        await (0, restoreItems_1.default)(BaseModel_1.ModelType.Folder, [await Folder_1.default.load(folder1.id)]);
        expect((await Folder_1.default.load(folder1.id)).deleted_time).toBe(0);
        expect((await Folder_1.default.load(folder2.id)).deleted_time).toBe(0);
        expect((await Note_1.default.load(note1.id)).deleted_time).toBe(0);
        expect((await Note_1.default.load(note2.id)).deleted_time).toBe(0);
    });
    it('should restore a note, even if the parent folder no longer exists', async () => {
        const folder = await Folder_1.default.save({});
        const note = await Note_1.default.save({ parent_id: folder.id });
        await Folder_1.default.delete(folder.id, { toTrash: true });
        await (0, restoreItems_1.default)(BaseModel_1.ModelType.Note, [await Note_1.default.load(note.id)]);
        const noteReloaded = await Note_1.default.load(note.id);
        expect(noteReloaded.parent_id).toBe('');
    });
    it('should restore a folder, even if the parent folder no longer exists', async () => {
        const folder1 = await Folder_1.default.save({});
        const folder2 = await Folder_1.default.save({});
        const note = await Note_1.default.save({ parent_id: folder2.id });
        await Folder_1.default.delete(folder1.id, { toTrash: true });
        await (0, restoreItems_1.default)(BaseModel_1.ModelType.Note, [await Folder_1.default.load(folder2.id)]);
        const folderReloaded2 = await Folder_1.default.load(folder2.id);
        const noteReloaded = await Note_1.default.load(note.id);
        expect(folderReloaded2.parent_id).toBe('');
        expect(noteReloaded.parent_id).toBe(folderReloaded2.id);
    });
    it('should restore a conflict', async () => {
        const note = await Note_1.default.save({ is_conflict: 1, title: 'Test' });
        await Note_1.default.delete(note.id, { toTrash: true });
        await (0, restoreItems_1.default)(BaseModel_1.ModelType.Note, [await Note_1.default.load(note.id)]);
        const noteReloaded = await Note_1.default.load(note.id);
        expect(noteReloaded.title).toBe('Test');
        expect(noteReloaded.is_conflict).toBe(1);
        expect(noteReloaded.deleted_time).toBe(0);
    });
});
//# sourceMappingURL=restoreItems.test.js.map