"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.newFileApi = newFileApi;
exports.initFileApi = initFileApi;
const file_api_driver_joplinServer_1 = require("./file-api-driver-joplinServer");
const Setting_1 = require("./models/Setting");
const Synchronizer_1 = require("./Synchronizer");
const locale_1 = require("./locale");
const JoplinServerApi_1 = require("./JoplinServerApi");
const BaseSyncTarget_1 = require("./BaseSyncTarget");
const file_api_1 = require("./file-api");
const Logger_1 = require("@joplin/utils/Logger");
const staticLogger = Logger_1.default.create('SyncTargetJoplinServer');
async function newFileApi(id, options) {
    const apiOptions = {
        baseUrl: () => options.path(),
        userContentBaseUrl: () => options.userContentPath(),
        username: () => options.username(),
        password: () => options.password(),
        session: () => null,
        env: Setting_1.default.value('env'),
    };
    const api = new JoplinServerApi_1.default(apiOptions);
    const driver = new file_api_driver_joplinServer_1.default(api);
    const fileApi = new file_api_1.FileApi('', driver);
    fileApi.setSyncTargetId(id);
    await fileApi.initialize();
    return fileApi;
}
async function initFileApi(syncTargetId, logger, options) {
    const fileApi = await newFileApi(syncTargetId, options);
    fileApi.setLogger(logger);
    return fileApi;
}
class SyncTargetJoplinServer extends BaseSyncTarget_1.default {
    static id() {
        return 9;
    }
    static supportsConfigCheck() {
        return true;
    }
    static targetName() {
        return 'joplinServer';
    }
    static description() {
        return 'Besides synchronisation and improved performances, Joplin Server also gives access to Joplin-specific sharing features.';
    }
    static label() {
        return `${(0, locale_1._)('Joplin Server')} (Beta)`;
    }
    async isAuthenticated() {
        return true;
    }
    static requiresPassword() {
        return true;
    }
    static supportsShare() {
        return true;
    }
    async fileApi() {
        return super.fileApi();
    }
    static async checkConfig(options, syncTargetId = null) {
        const output = {
            ok: false,
            errorMessage: '',
        };
        syncTargetId = syncTargetId === null ? this.id() : syncTargetId;
        let fileApi = null;
        try {
            fileApi = await newFileApi(syncTargetId, options);
            fileApi.requestRepeatCount_ = 0;
        }
        catch (error) {
            // If there's an error it's probably an application error, but we
            // can't proceed anyway, so exit.
            output.errorMessage = error.message;
            if (error.code)
                output.errorMessage += ` (Code ${error.code})`;
            return output;
        }
        // First we try to fetch info.json. It may not be present if it's a new
        // sync target but otherwise, if it is, and it's valid, we know the
        // credentials are valid. We do this test first because it will work
        // even if account upload is disabled. And we need such account to
        // successfully login so that they can fix it by deleting extraneous
        // notes or resources.
        try {
            const r = await fileApi.get('info.json');
            if (r) {
                const parsed = JSON.parse(r);
                if (parsed) {
                    output.ok = true;
                    return output;
                }
            }
        }
        catch (error) {
            // Ignore because we'll use the next test to check for sure if it
            // works or not.
            staticLogger.warn('Could not fetch or parse info.json:', error);
        }
        // This is a more generic test, which writes a file and tries to read it
        // back.
        try {
            await fileApi.put('testing.txt', 'testing');
            const result = await fileApi.get('testing.txt');
            if (result !== 'testing')
                throw new Error(`Could not access data on server "${options.path()}"`);
            await fileApi.delete('testing.txt');
            output.ok = true;
        }
        catch (error) {
            output.errorMessage = error.message;
            if (error.code)
                output.errorMessage += ` (Code ${error.code})`;
        }
        return output;
    }
    async initFileApi() {
        return initFileApi(SyncTargetJoplinServer.id(), this.logger(), {
            path: () => Setting_1.default.value('sync.9.path'),
            userContentPath: () => Setting_1.default.value('sync.9.userContentPath'),
            username: () => Setting_1.default.value('sync.9.username'),
            password: () => Setting_1.default.value('sync.9.password'),
        });
    }
    async initSynchronizer() {
        return new Synchronizer_1.default(this.db(), await this.fileApi(), Setting_1.default.value('appType'));
    }
}
exports.default = SyncTargetJoplinServer;
//# sourceMappingURL=SyncTargetJoplinServer.js.map