"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const AdmZip = require("adm-zip");
const fs_driver_base_1 = require("./fs-driver-base");
const time_1 = require("./time");
const md5File = require('md5-file');
const fs = require('fs-extra');
class FsDriverNode extends fs_driver_base_1.default {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    fsErrorToJsError_(error, path = null) {
        let msg = error.toString();
        if (path !== null)
            msg += `. Path: ${path}`;
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        const output = new Error(msg);
        if (error.code)
            output.code = error.code;
        return output;
    }
    appendFileSync(path, string) {
        return fs.appendFileSync(path, string);
    }
    async appendFile(path, string, encoding = 'base64') {
        try {
            return await fs.appendFile(path, string, { encoding: encoding });
        }
        catch (error) {
            throw this.fsErrorToJsError_(error, path);
        }
    }
    async writeFile(path, string, encoding = 'base64') {
        try {
            if (encoding === 'buffer') {
                return await fs.writeFile(path, string);
            }
            else {
                return await fs.writeFile(path, string, { encoding: encoding });
            }
        }
        catch (error) {
            throw this.fsErrorToJsError_(error, path);
        }
    }
    // same as rm -rf
    async remove(path) {
        try {
            const r = await fs.remove(path);
            return r;
        }
        catch (error) {
            throw this.fsErrorToJsError_(error, path);
        }
    }
    async move(source, dest) {
        let lastError = null;
        for (let i = 0; i < 5; i++) {
            try {
                const output = await fs.move(source, dest, { overwrite: true });
                return output;
            }
            catch (error) {
                lastError = error;
                // Normally cannot happen with the `overwrite` flag but sometime it still does.
                // In this case, retry.
                if (error.code === 'EEXIST') {
                    await time_1.default.sleep(1);
                    continue;
                }
                throw this.fsErrorToJsError_(error);
            }
        }
        throw lastError;
    }
    exists(path) {
        return fs.pathExists(path);
    }
    async mkdir(path) {
        // Note that mkdirp() does not throw an error if the directory
        // could not be created. This would make the synchroniser to
        // incorrectly try to sync with a non-existing dir:
        // https://github.com/laurent22/joplin/issues/2117
        const r = await fs.mkdirp(path);
        if (!(await this.exists(path)))
            throw new Error(`Could not create directory: ${path}`);
        return r;
    }
    async stat(path) {
        try {
            const stat = await fs.stat(path);
            return {
                birthtime: stat.birthtime,
                mtime: stat.mtime,
                isDirectory: () => stat.isDirectory(),
                path: path,
                size: stat.size,
            };
        }
        catch (error) {
            if (error.code === 'ENOENT')
                return null;
            throw error;
        }
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async setTimestamp(path, timestampDate) {
        return fs.utimes(path, timestampDate, timestampDate);
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async readDirStats(path, options = null) {
        if (!options)
            options = {};
        if (!('recursive' in options))
            options.recursive = false;
        let items = [];
        try {
            items = await fs.readdir(path);
        }
        catch (error) {
            throw this.fsErrorToJsError_(error);
        }
        let output = [];
        for (let i = 0; i < items.length; i++) {
            const item = items[i];
            const stat = await this.stat(`${path}/${item}`);
            if (!stat)
                continue; // Has been deleted between the readdir() call and now
            stat.path = stat.path.substr(path.length + 1);
            output.push(stat);
            output = await this.readDirStatsHandleRecursion_(path, stat, output, options);
        }
        return output;
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async open(path, mode) {
        try {
            return await fs.open(path, mode);
        }
        catch (error) {
            throw this.fsErrorToJsError_(error, path);
        }
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async close(handle) {
        try {
            return await fs.close(handle);
        }
        catch (error) {
            throw this.fsErrorToJsError_(error, '');
        }
    }
    async readFile(path, encoding = 'utf8') {
        try {
            if (encoding === 'Buffer')
                return await fs.readFile(path); // Returns the raw buffer
            return await fs.readFile(path, encoding);
        }
        catch (error) {
            throw this.fsErrorToJsError_(error, path);
        }
    }
    // Always overwrite destination
    async copy(source, dest) {
        try {
            return await fs.copy(source, dest, { overwrite: true });
        }
        catch (error) {
            throw this.fsErrorToJsError_(error, source);
        }
    }
    async chmod(source, mode) {
        return fs.chmod(source, mode);
    }
    async unlink(path) {
        try {
            await fs.unlink(path);
        }
        catch (error) {
            if (error.code === 'ENOENT')
                return; // Don't throw if the file does not exist
            throw error;
        }
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async readFileChunk(handle, length, encoding = 'base64') {
        // let buffer = new Buffer(length);
        let buffer = Buffer.alloc(length);
        const result = await fs.read(handle, buffer, 0, length, null);
        if (!result.bytesRead)
            return null;
        buffer = buffer.slice(0, result.bytesRead);
        if (encoding === 'base64')
            return buffer.toString('base64');
        if (encoding === 'ascii')
            return buffer.toString('ascii');
        throw new Error(`Unsupported encoding: ${encoding}`);
    }
    resolve(...pathComponents) {
        return require('path').resolve(...pathComponents);
    }
    async md5File(path) {
        return md5File(path);
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async tarExtract(options) {
        await require('tar').extract(options);
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async tarCreate(options, filePaths) {
        await require('tar').create(options, filePaths);
    }
    async zipExtract(options) {
        const zip = new AdmZip(options.source);
        zip.extractAllTo(options.extractTo, false);
        return zip.getEntries();
    }
}
exports.default = FsDriverNode;
//# sourceMappingURL=fs-driver-node.js.map