"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Setting_1 = require("../../models/Setting");
const BaseService_1 = require("../BaseService");
const Logger_1 = require("@joplin/utils/Logger");
const logger = Logger_1.default.create('KeychainService');
class KeychainService extends BaseService_1.default {
    constructor() {
        super(...arguments);
        this.enabled_ = true;
        this.readOnly_ = false;
    }
    static instance() {
        if (!this.instance_)
            this.instance_ = new KeychainService();
        return this.instance_;
    }
    // The drivers list should be provided in order of preference, with the most preferred driver
    // first. If not present in the first supported driver, the keychain service will attempt to
    // migrate keys to it.
    async initialize(drivers) {
        if (drivers.some(driver => !driver.appId || !driver.clientId)) {
            throw new Error('appId and clientId must be set on the KeychainServiceDriver');
        }
        this.drivers_ = [];
        this.keysNeedingMigration_ = new Set();
        for (const driver of drivers) {
            if (await driver.supported()) {
                this.drivers_.push(driver);
            }
            else {
                logger.info(`Driver unsupported:${driver.driverId}`);
            }
        }
    }
    // This is to programatically disable the keychain service, whether keychain
    // is supported or not in the system (In other word, this be might "enabled"
    // but nothing will be saved to the keychain if there isn't one).
    get enabled() {
        if (!this.enabled_)
            return false;
        // Otherwise we assume it's enabled if "keychain.supported" is either -1
        // (undetermined) or 1 (working). We make it work for -1 too because the
        // setPassword() and password() functions need to work to test if the
        // keychain is supported (in detectIfKeychainSupported).
        return Setting_1.default.value('keychain.supported') !== 0;
    }
    set enabled(v) {
        this.enabled_ = v;
    }
    get readOnly() {
        return this.readOnly_;
    }
    set readOnly(v) {
        this.readOnly_ = v;
    }
    async setPassword(name, password) {
        if (!this.enabled)
            return false;
        if (this.readOnly_)
            return false;
        // Optimization: Handles the case where the password doesn't need to change.
        // TODO: Re-evaluate whether this optimization is necessary after refactoring the driver
        //       logic.
        if (!this.keysNeedingMigration_.has(name) && await this.password(name) === password) {
            return true;
        }
        // Due to a bug in macOS, this may throw an exception "The user name or passphrase you entered is not correct."
        // The fix is to open Keychain Access.app. Right-click on the login keychain and try locking it and then unlocking it again.
        // https://github.com/atom/node-keytar/issues/76
        let i = 0;
        let didSet = false;
        for (; i < this.drivers_.length && !didSet; i++) {
            didSet = await this.drivers_[i].setPassword(name, password);
        }
        if (didSet && this.keysNeedingMigration_.has(name)) {
            logger.info(`Marking key ${name} as copied to new keychain backend...`);
            // At this point, the key has been saved in drivers[i - 1].
            //
            // Deleting the key from the less-preferred drivers would complete the
            // migration. However, to allow users to roll back to a previous Joplin
            // version without data loss, avoid deleting old keys here.
            this.keysNeedingMigration_.delete(name);
        }
        return didSet;
    }
    async password(name) {
        if (!this.enabled)
            return null;
        let foundInPreferredDriver = true;
        let password = null;
        for (const driver of this.drivers_) {
            password = await driver.password(name);
            if (password) {
                break;
            }
            foundInPreferredDriver = false;
        }
        if (password && !foundInPreferredDriver) {
            this.keysNeedingMigration_.add(name);
        }
        return password;
    }
    async deletePassword(name) {
        if (!this.enabled)
            return;
        for (const driver of this.drivers_) {
            await driver.deletePassword(name);
        }
    }
    async detectIfKeychainSupported() {
        this.logger().info('KeychainService: checking if keychain supported');
        const lastAvailableDrivers = Setting_1.default.value('keychain.lastAvailableDrivers');
        const availableDriversChanged = (() => {
            if (lastAvailableDrivers.length !== this.drivers_.length)
                return true;
            return this.drivers_.some(driver => {
                return !lastAvailableDrivers.includes(driver.driverId);
            });
        })();
        const checkAlreadyDone = Setting_1.default.value('keychain.supported') >= 0;
        if (checkAlreadyDone && !availableDriversChanged) {
            this.logger().info('KeychainService: check was already done - skipping. Supported:', Setting_1.default.value('keychain.supported'));
            return;
        }
        if (availableDriversChanged) {
            // Reset supported -- this allows the test .setPassword to work.
            Setting_1.default.setValue('keychain.supported', -1);
        }
        if (!this.readOnly) {
            const passwordIsSet = await this.setPassword('zz_testingkeychain', 'mytest');
            if (!passwordIsSet) {
                this.logger().info('KeychainService: could not set test password - keychain support will be disabled');
                Setting_1.default.setValue('keychain.supported', 0);
            }
            else {
                const result = await this.password('zz_testingkeychain');
                await this.deletePassword('zz_testingkeychain');
                this.logger().info('KeychainService: tried to set and get password. Result was:', result);
                Setting_1.default.setValue('keychain.supported', result === 'mytest' ? 1 : 0);
            }
        }
        else {
            // The supported check requires write access to the keychain -- rely on the more
            // limited support checks done by each driver.
            const supported = this.drivers_.length > 0;
            Setting_1.default.setValue('keychain.supported', supported ? 1 : 0);
            if (supported) {
                logger.info('Starting KeychainService in read-only mode. Keys will be read, but not written.');
            }
            else {
                logger.info('Failed to start in read-only mode -- no supported drivers found.');
            }
        }
        Setting_1.default.setValue('keychain.lastAvailableDrivers', this.drivers_.map(driver => driver.driverId));
    }
}
exports.default = KeychainService;
//# sourceMappingURL=KeychainService.js.map