"use strict";
// The goal of this class is to simplify the integration of the `joplin.views.editor` plugin logic
// in the desktop and mobile app. See here for more information:
//
// packages/lib/services/plugins/api/JoplinViewsEditor.ts
Object.defineProperty(exports, "__esModule", { value: true });
const Logger_1 = require("@joplin/utils/Logger");
const AsyncActionQueue_1 = require("../../AsyncActionQueue");
const eventManager_1 = require("../../eventManager");
const logger = Logger_1.default.create('EditorPluginHandler');
const makeNoteUpdateAction = (pluginService, event, shownEditorViewIds) => {
    return async () => {
        for (const viewId of shownEditorViewIds) {
            const controller = pluginService.viewControllerByViewId(viewId);
            if (controller) {
                controller.emitUpdate({
                    noteId: event.noteId,
                    newBody: event.newBody,
                });
            }
        }
    };
};
class default_1 {
    constructor(pluginService_, onSaveNote_) {
        this.pluginService_ = pluginService_;
        this.onSaveNote_ = onSaveNote_;
        this.viewUpdateAsyncQueue_ = new AsyncActionQueue_1.default(100, AsyncActionQueue_1.IntervalType.Fixed);
        this.lastNoteState_ = null;
        this.lastShownEditorViewIds_ = '';
        this.lastEditorPluginShown_ = null;
    }
    emitUpdate(event, shownEditorViewIds) {
        if (shownEditorViewIds.length === 0)
            return;
        const isEventDifferentFrom = (other) => {
            if (!other)
                return true;
            return event.noteId !== other.noteId || event.newBody !== other.newBody;
        };
        const shownEditorViewIdsString = shownEditorViewIds.join(',');
        const differentEditorViewsShown = shownEditorViewIdsString !== this.lastShownEditorViewIds_;
        // lastNoteState_ often contains the last change saved by the editor. As a result,
        // if `event` matches `lastNoteState_`, the event was probably caused by the last save.
        // In this case, avoid sending an update event (which plugins often interpret as refreshing
        // the editor):
        const isDifferentFromSave = isEventDifferentFrom(this.lastNoteState_);
        if (isDifferentFromSave || differentEditorViewsShown) {
            logger.info('emitUpdate:', shownEditorViewIds);
            this.viewUpdateAsyncQueue_.push(makeNoteUpdateAction(this.pluginService_, event, shownEditorViewIds));
            this.lastNoteState_ = Object.assign({}, event);
            this.lastShownEditorViewIds_ = shownEditorViewIdsString;
        }
    }
    async emitActivationCheck({ noteId, parentWindowId }) {
        let filterObject = {
            activatedEditors: [],
            effectiveNoteId: noteId,
            windowId: parentWindowId,
        };
        filterObject = await eventManager_1.default.filterEmit('editorActivationCheck', filterObject);
        logger.info('emitActivationCheck: responses:', filterObject);
        for (const editor of filterObject.activatedEditors) {
            const controller = this.pluginService_.pluginById(editor.pluginId).viewController(editor.viewId);
            if (controller.parentWindowId === parentWindowId) {
                controller.setActive(editor.isActive);
            }
        }
    }
    onEditorPluginShown(editorViewId) {
        // Don't double-register callbacks
        if (editorViewId === this.lastEditorPluginShown_) {
            return;
        }
        this.lastEditorPluginShown_ = editorViewId;
        const controller = this.pluginService_.viewControllerByViewId(editorViewId);
        controller === null || controller === void 0 ? void 0 : controller.onNoteSaveRequested(event => {
            this.scheduleSaveNote_(event.noteId, event.body);
        });
    }
    scheduleSaveNote_(noteId, noteBody) {
        this.lastNoteState_ = {
            noteId,
            newBody: noteBody,
        };
        return this.onSaveNote_({
            id: noteId,
            body: noteBody,
        });
    }
}
exports.default = default_1;
//# sourceMappingURL=EditorPluginHandler.js.map