"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const time_1 = require("../../time");
const shim_1 = require("../../shim");
const Setting_1 = require("../../models/Setting");
const test_utils_1 = require("../../testing/test-utils");
const Folder_1 = require("../../models/Folder");
const Note_1 = require("../../models/Note");
const Resource_1 = require("../../models/Resource");
const ResourceFetcher_1 = require("../../services/ResourceFetcher");
const MasterKey_1 = require("../../models/MasterKey");
const BaseItem_1 = require("../../models/BaseItem");
const syncInfoUtils_1 = require("../synchronizer/syncInfoUtils");
const utils_1 = require("../e2ee/utils");
const test_utils_synchronizer_1 = require("../../testing/test-utils-synchronizer");
const EncryptionService_1 = require("../e2ee/EncryptionService");
let insideBeforeEach = false;
function newResourceFetcher(synchronizer) {
    return new ResourceFetcher_1.default(() => { return synchronizer.api(); });
}
describe('Synchronizer.e2ee', () => {
    beforeEach(async () => {
        insideBeforeEach = true;
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(2);
        await (0, test_utils_1.switchClient)(1);
        insideBeforeEach = false;
    });
    it.each([
        EncryptionService_1.EncryptionMethod.SJCL1a,
        EncryptionService_1.EncryptionMethod.StringV1,
    ])('notes and folders should get encrypted when encryption is enabled', (async (encryptionMethod) => {
        (0, syncInfoUtils_1.setEncryptionEnabled)(true);
        (0, test_utils_1.encryptionService)().defaultEncryptionMethod_ = encryptionMethod;
        const masterKey = await (0, test_utils_1.loadEncryptionMasterKey)();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        let note1 = await Note_1.default.save({ title: 'un', body: 'to be encrypted', parent_id: folder1.id });
        await (0, test_utils_1.synchronizerStart)();
        // After synchronisation, remote items should be encrypted but local ones remain plain text
        note1 = await Note_1.default.load(note1.id);
        expect(note1.title).toBe('un');
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        let folder1_2 = await Folder_1.default.load(folder1.id);
        let note1_2 = await Note_1.default.load(note1.id);
        const masterKey_2 = await MasterKey_1.default.load(masterKey.id);
        // On this side however it should be received encrypted
        expect(!note1_2.title).toBe(true);
        expect(!folder1_2.title).toBe(true);
        expect(!!note1_2.encryption_cipher_text).toBe(true);
        expect(!!folder1_2.encryption_cipher_text).toBe(true);
        // Master key is already encrypted so it does not get re-encrypted during sync
        expect(masterKey_2.content).toBe(masterKey.content);
        expect(masterKey_2.checksum).toBe(masterKey.checksum);
        // Now load the master key we got from client 1 and try to decrypt
        await (0, test_utils_1.encryptionService)().loadMasterKey(masterKey_2, '123456', true);
        // Get the decrypted items back
        await Folder_1.default.decrypt(folder1_2);
        await Note_1.default.decrypt(note1_2);
        folder1_2 = await Folder_1.default.load(folder1.id);
        note1_2 = await Note_1.default.load(note1.id);
        // Check that properties match the original items. Also check
        // the encryption did not affect the updated_time timestamp.
        expect(note1_2.title).toBe(note1.title);
        expect(note1_2.body).toBe(note1.body);
        expect(note1_2.updated_time).toBe(note1.updated_time);
        expect(!note1_2.encryption_cipher_text).toBe(true);
        expect(folder1_2.title).toBe(folder1.title);
        expect(folder1_2.updated_time).toBe(folder1.updated_time);
        expect(!folder1_2.encryption_cipher_text).toBe(true);
    }));
    it('should not encrypt structural properties', (async () => {
        (0, syncInfoUtils_1.setEncryptionEnabled)(true);
        await (0, test_utils_1.loadEncryptionMasterKey)();
        const folder1 = await Folder_1.default.save({});
        const folder2 = await Folder_1.default.save({});
        const note1 = await Note_1.default.save({ parent_id: folder1.id });
        const note2 = await Note_1.default.save({ parent_id: folder2.id });
        await Folder_1.default.delete(folder2.id, { toTrash: true, deleteChildren: true });
        await (0, test_utils_1.synchronizerStart)();
        const remoteItems = await (0, test_utils_synchronizer_1.remoteNotesAndFolders)();
        expect(remoteItems.find(i => i.id === folder1.id).deleted_time).toBe(0);
        expect(remoteItems.find(i => i.id === folder2.id).deleted_time).toBeGreaterThan(0);
        expect(remoteItems.find(i => i.id === note1.id).deleted_time).toBe(0);
        expect(remoteItems.find(i => i.id === note2.id).deleted_time).toBeGreaterThan(0);
    }));
    it('should mark the key has having been used when synchronising the first time', (async () => {
        (0, syncInfoUtils_1.setEncryptionEnabled)(true);
        await (0, test_utils_1.loadEncryptionMasterKey)();
        await Folder_1.default.save({ title: 'folder1' });
        await (0, test_utils_1.synchronizerStart)();
        const localInfo = (0, syncInfoUtils_1.localSyncInfo)();
        const remoteInfo = await (0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.fileApi)());
        expect(localInfo.masterKeys[0].hasBeenUsed).toBe(true);
        expect(remoteInfo.masterKeys[0].hasBeenUsed).toBe(true);
    }));
    it('should mark the key has having been used when synchronising after enabling encryption', (async () => {
        await Folder_1.default.save({ title: 'folder1' });
        await (0, test_utils_1.synchronizerStart)();
        (0, syncInfoUtils_1.setEncryptionEnabled)(true);
        await (0, test_utils_1.loadEncryptionMasterKey)();
        await (0, test_utils_1.synchronizerStart)();
        const localInfo = (0, syncInfoUtils_1.localSyncInfo)();
        const remoteInfo = await (0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.fileApi)());
        expect(localInfo.masterKeys[0].hasBeenUsed).toBe(true);
        expect(remoteInfo.masterKeys[0].hasBeenUsed).toBe(true);
    }));
    it('should enable encryption automatically when downloading new master key (and none was previously available)', (async () => {
        // Enable encryption on client 1 and sync an item
        (0, syncInfoUtils_1.setEncryptionEnabled)(true);
        await (0, test_utils_1.loadEncryptionMasterKey)();
        let folder1 = await Folder_1.default.save({ title: 'folder1' });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        // Synchronising should enable encryption since we're going to get a master key
        expect((0, syncInfoUtils_1.getEncryptionEnabled)()).toBe(false);
        await (0, test_utils_1.synchronizerStart)();
        expect((0, syncInfoUtils_1.getEncryptionEnabled)()).toBe(true);
        // Check that we got the master key from client 1
        const masterKey = (await MasterKey_1.default.all())[0];
        expect(!!masterKey).toBe(true);
        // Since client 2 hasn't supplied a password yet, no master key is currently loaded
        expect((0, test_utils_1.encryptionService)().loadedMasterKeyIds().length).toBe(0);
        // If we sync now, nothing should be sent to target since we don't have a password.
        // Technically it's incorrect to set the property of an encrypted variable but it allows confirming
        // that encryption doesn't work if user hasn't supplied a password.
        await BaseItem_1.default.forceSync(folder1.id);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(1);
        await (0, test_utils_1.synchronizerStart)();
        folder1 = await Folder_1.default.load(folder1.id);
        expect(folder1.title).toBe('folder1'); // Still at old value
        await (0, test_utils_1.switchClient)(2);
        // Now client 2 set the master key password
        Setting_1.default.setObjectValue('encryption.passwordCache', masterKey.id, '123456');
        await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
        // Now that master key should be loaded
        expect((0, test_utils_1.encryptionService)().loadedMasterKeyIds()[0]).toBe(masterKey.id);
        // Decrypt all the data. Now change the title and sync again - this time the changes should be transmitted
        await (0, test_utils_1.decryptionWorker)().start();
        await Folder_1.default.save({ id: folder1.id, title: 'change test' });
        // If we sync now, this time client 1 should get the changes we did earlier
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(1);
        await (0, test_utils_1.synchronizerStart)();
        // Decrypt the data we just got
        await (0, test_utils_1.decryptionWorker)().start();
        folder1 = await Folder_1.default.load(folder1.id);
        expect(folder1.title).toBe('change test'); // Got title from client 2
    }));
    it('should encrypt existing notes too when enabling E2EE', (async () => {
        // First create a folder, without encryption enabled, and sync it
        await Folder_1.default.save({ title: 'folder1' });
        await (0, test_utils_1.synchronizerStart)();
        let files = await (0, test_utils_1.fileApi)().list('', { includeDirs: false, syncItemsOnly: true });
        let content = await (0, test_utils_1.fileApi)().get(files.items[0].path);
        expect(content.indexOf('folder1') >= 0).toBe(true);
        // Then enable encryption and sync again
        let masterKey = await (0, test_utils_1.encryptionService)().generateMasterKey('123456');
        masterKey = await MasterKey_1.default.save(masterKey);
        await (0, utils_1.setupAndEnableEncryption)((0, test_utils_1.encryptionService)(), masterKey, '123456');
        await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
        await (0, test_utils_1.synchronizerStart)();
        // Even though the folder has not been changed it should have been synced again so that
        // an encrypted version of it replaces the decrypted version.
        files = await (0, test_utils_1.fileApi)().list('', { includeDirs: false, syncItemsOnly: true });
        expect(files.items.length).toBe(1);
        // By checking that the folder title is not present, we can confirm that the item has indeed been encrypted
        content = await (0, test_utils_1.fileApi)().get(files.items[0].path);
        expect(content.indexOf('folder1') < 0).toBe(true);
    }));
    it('should upload decrypted items to sync target after encryption disabled', (async () => {
        (0, syncInfoUtils_1.setEncryptionEnabled)(true);
        await (0, test_utils_1.loadEncryptionMasterKey)();
        await Folder_1.default.save({ title: 'folder1' });
        await (0, test_utils_1.synchronizerStart)();
        let allEncrypted = await (0, test_utils_1.allSyncTargetItemsEncrypted)();
        expect(allEncrypted).toBe(true);
        await (0, utils_1.setupAndDisableEncryption)((0, test_utils_1.encryptionService)());
        await (0, test_utils_1.synchronizerStart)();
        allEncrypted = await (0, test_utils_1.allSyncTargetItemsEncrypted)();
        expect(allEncrypted).toBe(false);
    }));
    it('should not upload any item if encryption was enabled, and items have not been decrypted, and then encryption disabled', (async () => {
        // For some reason I can't explain, this test is sometimes executed before beforeEach is finished
        // which means it's going to fail in unexpected way. So the loop below wait for beforeEach to be done.
        while (insideBeforeEach)
            await time_1.default.msleep(100);
        (0, syncInfoUtils_1.setEncryptionEnabled)(true);
        const masterKey = await (0, test_utils_1.loadEncryptionMasterKey)();
        await Folder_1.default.save({ title: 'folder1' });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        expect((0, syncInfoUtils_1.getEncryptionEnabled)()).toBe(true);
        // If we try to disable encryption now, it should throw an error because some items are
        // currently encrypted. They must be decrypted first so that they can be sent as
        // plain text to the sync target.
        // let hasThrown = await checkThrowAsync(async () => await setupAndDisableEncryption(encryptionService()));
        // expect(hasThrown).toBe(true);
        // Now supply the password, and decrypt the items
        Setting_1.default.setObjectValue('encryption.passwordCache', masterKey.id, '123456');
        await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
        await (0, test_utils_1.decryptionWorker)().start();
        // Try to disable encryption again
        const hasThrown = await (0, test_utils_1.checkThrowAsync)(async () => await (0, utils_1.setupAndDisableEncryption)((0, test_utils_1.encryptionService)()));
        expect(hasThrown).toBe(false);
        // If we sync now the target should receive the decrypted items
        await (0, test_utils_1.synchronizerStart)();
        const allEncrypted = await (0, test_utils_1.allSyncTargetItemsEncrypted)();
        expect(allEncrypted).toBe(false);
    }));
    it.each([
        [EncryptionService_1.EncryptionMethod.SJCL1a, EncryptionService_1.EncryptionMethod.SJCL1a],
        [EncryptionService_1.EncryptionMethod.StringV1, EncryptionService_1.EncryptionMethod.FileV1],
    ])('should set the resource file size after decryption', (async (stringEncryptionMethod, fileEncryptionMethod) => {
        (0, syncInfoUtils_1.setEncryptionEnabled)(true);
        (0, test_utils_1.encryptionService)().defaultEncryptionMethod_ = stringEncryptionMethod;
        (0, test_utils_1.encryptionService)().defaultFileEncryptionMethod_ = fileEncryptionMethod;
        const masterKey = await (0, test_utils_1.loadEncryptionMasterKey)();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        const resource1 = (await Resource_1.default.all())[0];
        await Resource_1.default.setFileSizeOnly(resource1.id, -1);
        Resource_1.default.fullPath(resource1);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        Setting_1.default.setObjectValue('encryption.passwordCache', masterKey.id, '123456');
        await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
        const fetcher = newResourceFetcher((0, test_utils_1.synchronizer)());
        fetcher.queueDownload_(resource1.id);
        await fetcher.waitForAllFinished();
        await (0, test_utils_1.decryptionWorker)().start();
        const resource1_2 = await Resource_1.default.load(resource1.id);
        expect(resource1_2.size).toBe(2720);
    }));
    it.each([
        [EncryptionService_1.EncryptionMethod.SJCL1a, EncryptionService_1.EncryptionMethod.SJCL1a],
        [EncryptionService_1.EncryptionMethod.StringV1, EncryptionService_1.EncryptionMethod.FileV1],
    ])('should encrypt remote resources after encryption has been enabled', (async (stringEncryptionMethod, fileEncryptionMethod) => {
        while (insideBeforeEach)
            await time_1.default.msleep(100);
        (0, test_utils_1.encryptionService)().defaultEncryptionMethod_ = stringEncryptionMethod;
        (0, test_utils_1.encryptionService)().defaultFileEncryptionMethod_ = fileEncryptionMethod;
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        await (0, test_utils_1.synchronizerStart)();
        expect(await (0, test_utils_1.allSyncTargetItemsEncrypted)()).toBe(false);
        const masterKey = await (0, test_utils_1.loadEncryptionMasterKey)();
        await (0, utils_1.setupAndEnableEncryption)((0, test_utils_1.encryptionService)(), masterKey, '123456');
        await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
        await (0, test_utils_1.synchronizerStart)();
        expect(await (0, test_utils_1.allSyncTargetItemsEncrypted)()).toBe(true);
    }));
    it('should upload encrypted resource, but it should not mark the blob as encrypted locally', (async () => {
        while (insideBeforeEach)
            await time_1.default.msleep(100);
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        const masterKey = await (0, test_utils_1.loadEncryptionMasterKey)();
        await (0, utils_1.setupAndEnableEncryption)((0, test_utils_1.encryptionService)(), masterKey, '123456');
        await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
        // await synchronizerStart();
        // const resource1 = (await Resource.all())[0];
        // expect(resource1.encryption_blob_encrypted).toBe(0);
    }));
    it('should decrypt the resource metadata, but not try to decrypt the file, if it is not present', (async () => {
        const note1 = await Note_1.default.save({ title: 'note' });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        const masterKey = await (0, test_utils_1.loadEncryptionMasterKey)();
        await (0, utils_1.setupAndEnableEncryption)((0, test_utils_1.encryptionService)(), masterKey, '123456');
        await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
        await (0, test_utils_1.synchronizerStart)();
        expect(await (0, test_utils_1.allSyncTargetItemsEncrypted)()).toBe(true);
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        Setting_1.default.setObjectValue('encryption.passwordCache', masterKey.id, '123456');
        await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
        await (0, test_utils_1.decryptionWorker)().start();
        let resource = (await Resource_1.default.all())[0];
        expect(!!resource.encryption_applied).toBe(false);
        expect(!!resource.encryption_blob_encrypted).toBe(true);
        const resourceFetcher = newResourceFetcher((0, test_utils_1.synchronizer)());
        await resourceFetcher.start();
        await resourceFetcher.waitForAllFinished();
        const ls = await Resource_1.default.localState(resource);
        expect(ls.fetch_status).toBe(Resource_1.default.FETCH_STATUS_DONE);
        await (0, test_utils_1.decryptionWorker)().start();
        resource = (await Resource_1.default.all())[0];
        expect(!!resource.encryption_blob_encrypted).toBe(false);
    }));
    it.each([
        EncryptionService_1.EncryptionMethod.SJCL1a,
        EncryptionService_1.EncryptionMethod.StringV1,
    ])('should stop trying to decrypt item after a few attempts', (async (encryptionMethod) => {
        let hasThrown;
        (0, test_utils_1.encryptionService)().defaultEncryptionMethod_ = encryptionMethod;
        const note = await Note_1.default.save({ title: 'ma note' });
        const masterKey = await (0, test_utils_1.loadEncryptionMasterKey)();
        await (0, utils_1.setupAndEnableEncryption)((0, test_utils_1.encryptionService)(), masterKey, '123456');
        await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        // First, simulate a broken note and check that the decryption worker
        // gives up decrypting after a number of tries. This is mainly relevant
        // for data that crashes the mobile application - we don't want to keep
        // decrypting these.
        const encryptedNote = await Note_1.default.load(note.id);
        const goodCipherText = encryptedNote.encryption_cipher_text;
        await Note_1.default.save({ id: note.id, encryption_cipher_text: 'doesntlookright' });
        Setting_1.default.setObjectValue('encryption.passwordCache', masterKey.id, '123456');
        await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
        hasThrown = await (0, test_utils_1.checkThrowAsync)(async () => await (0, test_utils_1.decryptionWorker)().start({ errorHandler: 'throw' }));
        expect(hasThrown).toBe(true);
        hasThrown = await (0, test_utils_1.checkThrowAsync)(async () => await (0, test_utils_1.decryptionWorker)().start({ errorHandler: 'throw' }));
        expect(hasThrown).toBe(true);
        // Third time, an error is logged and no error is thrown
        hasThrown = await (0, test_utils_1.checkThrowAsync)(async () => await (0, test_utils_1.decryptionWorker)().start({ errorHandler: 'throw' }));
        expect(hasThrown).toBe(false);
        const disabledItems = await (0, test_utils_1.decryptionWorker)().decryptionDisabledItems();
        expect(disabledItems.length).toBe(1);
        expect(disabledItems[0].id).toBe(note.id);
        expect((await (0, test_utils_1.kvStore)().searchByPrefix('decrypt:')).length).toBe(1);
        await (0, test_utils_1.kvStore)().clear();
        // Now check that if it fails once but succeed the second time, the note
        // is correctly decrypted and the counters are cleared.
        hasThrown = await (0, test_utils_1.checkThrowAsync)(async () => await (0, test_utils_1.decryptionWorker)().start({ errorHandler: 'throw' }));
        expect(hasThrown).toBe(true);
        await Note_1.default.save({ id: note.id, encryption_cipher_text: goodCipherText });
        hasThrown = await (0, test_utils_1.checkThrowAsync)(async () => await (0, test_utils_1.decryptionWorker)().start({ errorHandler: 'throw' }));
        expect(hasThrown).toBe(false);
        const decryptedNote = await Note_1.default.load(note.id);
        expect(decryptedNote.title).toBe('ma note');
        expect((await (0, test_utils_1.kvStore)().all()).length).toBe(0);
        expect((await (0, test_utils_1.decryptionWorker)().decryptionDisabledItems()).length).toBe(0);
    }));
});
//# sourceMappingURL=Synchronizer.e2ee.test.js.map