//! Types and functions related to file system path operations.

use std::{
    env,
    ffi::OsStr,
    fs,
    io::ErrorKind,
    path::{Path, PathBuf},
};

use log::{info, warn};

#[cfg(target_family = "windows")]
const DEFAULT_SYS_PATH: &str = "C:\\Program Files\\ChewingTextService\\Dictionary";
#[cfg(target_family = "unix")]
const DEFAULT_SYS_PATH: &str = "/usr/share/libchewing";
const SYS_PATH: Option<&str> = option_env!("CHEWING_DATADIR");

#[cfg(target_family = "windows")]
const SEARCH_PATH_SEP: char = ';';
#[cfg(target_family = "unix")]
const SEARCH_PATH_SEP: char = ':';

const DICT_FOLDER: &str = "dictionary.d";

// On Windows if a low integrity process tries to write to a higher integrity
// process, it fails with PermissionDenied error. Current `fs::exists()` in Rust
// happens to use CreateFile to check if a file exists that triggers this error.
fn file_exists(path: &Path) -> bool {
    match fs::exists(path) {
        Ok(true) => true,
        Ok(false) => false,
        Err(error) => matches!(error.kind(), ErrorKind::PermissionDenied),
    }
}

pub(crate) fn sys_path_from_env_var() -> String {
    let chewing_path = env::var("CHEWING_PATH");
    if let Ok(chewing_path) = chewing_path {
        info!("Using syspath from env CHEWING_PATH: {}", chewing_path);
        chewing_path
    } else {
        let user_datadir = data_dir();
        let sys_datadir = SYS_PATH.unwrap_or(DEFAULT_SYS_PATH);
        let chewing_path = if let Some(datadir) = user_datadir.as_ref().and_then(|p| p.to_str()) {
            format!("{datadir}{SEARCH_PATH_SEP}{sys_datadir}")
        } else {
            sys_datadir.into()
        };
        info!("Using default syspath: {}", chewing_path);
        chewing_path
    }
}

pub(crate) fn find_path_by_files(search_path: &str, files: &[&str]) -> Option<PathBuf> {
    for path in search_path.split(SEARCH_PATH_SEP) {
        let prefix = Path::new(path).to_path_buf();
        info!("Search files {:?} in {}", files, prefix.display());
        if files
            .iter()
            .map(|it| {
                let mut path = prefix.clone();
                path.push(it);
                path
            })
            .all(|it| file_exists(&it))
        {
            info!("Found {:?} in {}", files, prefix.display());
            return Some(prefix);
        }
    }
    None
}

pub(crate) fn find_drop_in_dat_by_path(search_path: &str) -> Vec<PathBuf> {
    let mut results = vec![];
    for path in search_path.split(SEARCH_PATH_SEP) {
        let prefix = Path::new(path).join(DICT_FOLDER);
        info!("Search dictionary files in {}", prefix.display());
        if let Ok(read_dir) = prefix.read_dir() {
            let mut files = vec![];
            for entry in read_dir.flatten() {
                let path = entry.path();
                let is_dat = path.extension().and_then(OsStr::to_str) == Some("dat");
                if path.is_file() && is_dat {
                    info!("Found {}", path.display());
                    files.push(path);
                }
            }
            files.sort();
            results.extend(files);
        }
    }
    results
}

/// Returns the path to the user's default chewing data directory.
///
/// The returned value depends on the operating system and is either a
/// Some, containing a value from the following table, or a None.
///
/// |Platform | Base                                     | Example                                                     |
/// | ------- | ---------------------------------------- | ------------------------------------------------------------|
/// | Linux   | `$XDG_DATA_HOME` or `$HOME`/.local/share | /home/alice/.local/share/chewing                            |
/// | macOS   | `$HOME`/Library/Application Support      | /Users/Alice/Library/Application Support/im.chewing.Chewing |
/// | Windows | `{FOLDERID_RoamingAppData}`              | C:\Users\Alice\AppData\Roaming\chewing\Chewing\data         |
///
/// Legacy path is automatically detected and used
///
/// |Platform | Base           | Example                            |
/// | ------- | -------------- | --------------------- ------------ |
/// | Linux   | `$HOME`        | /home/alice/.chewing               |
/// | macOS   | /Library       | /Library/ChewingOSX                |
/// | Windows | `$USERPROFILE` | C:\Users\Alice\ChewingTextService  |
///
/// Users can set the `CHEWING_USER_PATH` environment variable to
/// override the default path.
pub fn data_dir() -> Option<PathBuf> {
    if let Ok(path) = env::var("CHEWING_USER_PATH") {
        info!("Using userpath from env CHEWING_USER_PATH: {}", path);
        return Some(path.into());
    }
    if let Some(path) = legacy_data_dir() {
        if file_exists(&path) && path.is_dir() {
            info!("Using legacy userpath: {}", path.display());
            return Some(path);
        }
    }
    let data_dir = project_data_dir();
    if let Some(path) = &data_dir {
        info!("Using default userpath: {}", path.display());
    } else {
        warn!("No valid home directory path could be retrieved from the operating system.");
    }
    data_dir
}

fn project_data_dir() -> Option<PathBuf> {
    #[cfg(target_os = "windows")]
    {
        if let Ok(path) = env::var("AppData") {
            return Some(PathBuf::from(path).join("Chewing"));
        }
    }
    #[cfg(target_os = "macos")]
    {
        return env::home_dir().map(|path| {
            path.join("Library")
                .join("Application Support")
                .join("im.chewing.Chewing")
        });
    }
    #[cfg(not(target_family = "unix"))]
    {
        return None;
    }

    #[cfg(target_family = "unix")]
    {
        if let Ok(path) = env::var("XDG_DATA_HOME") {
            return Some(PathBuf::from(path).join("chewing"));
        }
        env::home_dir().map(|path| path.join(".local").join("share").join("chewing"))
    }
}

fn legacy_data_dir() -> Option<PathBuf> {
    if cfg!(target_os = "windows") {
        return env::home_dir().map(|path| path.join("ChewingTextService"));
    }

    if cfg!(any(target_os = "macos", target_os = "ios")) {
        return Some("/Library/ChewingOSX".into());
    }

    env::home_dir().map(|path| path.join(".chewing"))
}

/// Returns the path to the user's default userphrase database file.
///
/// This function uses the default path from the [`data_dir()`] method
/// and also respects the `CHEWING_USER_PATH` environment variable.
pub fn userphrase_path() -> Option<PathBuf> {
    data_dir().map(|path| path.join("chewing.dat"))
}

#[cfg(test)]
mod tests {
    use super::{data_dir, project_data_dir};

    #[test]
    fn support_project_data_dir() {
        assert!(project_data_dir().is_some());
    }

    #[test]
    fn resolve_data_dir() {
        if project_data_dir().is_some() {
            let data_dir = data_dir();
            assert!(data_dir.is_some());
        }
    }
}
