# electron_shell_dialog.py
#
# Copyright 2024 Lo
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later

import gi, math

from gi.repository import Adw, Gtk, Gio, GObject


@Gtk.Template(resource_path="/page/codeberg/lo_vely/Nucleus/ui/electron-shell-dialog.ui")
class ElectronShellDialog(Adw.Dialog):
    __gtype_name__ = "ElectronShellDialog"

    drawing_area = Gtk.Template.Child()
    electron_pet_shell_row = Gtk.Template.Child()
    element_name = GObject.Property(type=str)

    drawing_color_light = [0,0,0,1]
    drawing_color_dark = [1,1,1,1]
    current_drawing_color = drawing_color_light

    shells = None

    def __init__(self, element_name, shells, **kwargs):
        super().__init__(**kwargs)

        self.props.element_name = element_name
        self.shells = shells

        shell_letter = ['K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R']
        for idx, shell in enumerate(self.shells):
            self.electron_pet_shell_row.props.subtitle += f"{shell_letter[idx]}{str(shell)} "

        self.drawing_area.set_content_height(len(self.shells) * 40 + 45)

        style_manager = Adw.StyleManager().get_default()
        style_manager.connect("notify::dark", self._on_color_scheme_changed)

        if style_manager.props.dark:
            self.current_drawing_color = self.drawing_color_dark

        self.drawing_area.set_draw_func(self._draw)


    def _on_color_scheme_changed(self, manager, dark):
        if manager.props.dark:
            self.current_drawing_color = self.drawing_color_dark
        else:
            self.current_drawing_color = self.drawing_color_light

        self.drawing_area.queue_draw()


    def _draw(self, _self, cr, width, height) -> None:
        cr.translate(width / 2, height / 2)

        for idx, shell in enumerate(self.shells, start=1):
            # Orbit
            edge_pos = idx * 20
            cr.set_source_rgba(
                self.current_drawing_color[0],
                self.current_drawing_color[1],
                self.current_drawing_color[2],
                self.current_drawing_color[3] - 0.5
            )
            cr.arc(0, 0, edge_pos, math.radians(0), math.radians(360))
            cr.stroke()

            for electron in range(shell):
                cr.set_source_rgba(
                    self.current_drawing_color[0],
                    self.current_drawing_color[1],
                    self.current_drawing_color[2],
                    self.current_drawing_color[3]
                )
                cr.arc(0, -edge_pos, 5, 0, 100)
                cr.rotate(math.radians(360 / shell))
                cr.fill()
                cr.stroke()
        
