/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.data.nc;

import com.google.common.collect.ImmutableList;
import gov.nasa.giss.data.nc.NcAxisType;
import gov.nasa.giss.data.nc.NcDataset;
import gov.nasa.giss.data.nc.NcDiscreteVarType;
import gov.nasa.giss.data.nc.NcUnitUtils;
import gov.nasa.giss.data.nc.NcVarType;
import gov.nasa.giss.data.nc.NcVariable;
import gov.nasa.giss.data.nc.array.NcArrayFeatureTypeTrajectory;
import gov.nasa.giss.text.PrintfFormat;
import gov.nasa.giss.time.RelativeTime;
import java.lang.invoke.MethodHandles;
import java.text.ParseException;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.ma2.DataType;
import ucar.nc2.Attribute;
import ucar.nc2.AttributeContainer;
import ucar.nc2.Dimension;
import ucar.nc2.Group;
import ucar.nc2.Variable;
import ucar.nc2.constants.AxisType;
import ucar.nc2.dataset.CoordinateAxis;
import ucar.nc2.dataset.CoordinateAxis1D;
import ucar.nc2.dataset.CoordinateAxis2D;
import ucar.nc2.dataset.CoordinateSystem;
import ucar.nc2.dataset.CoordinateTransform;
import ucar.nc2.dataset.VariableDS;
import ucar.units.BaseUnit;
import ucar.units.SI;
import ucar.units.Unit;

public final class NcUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final PrintfFormat PFORMAT_7G = new PrintfFormat("%.7g");
    private static final PrintfFormat PFORMAT_10G = new PrintfFormat("%.10g");

    public static String getAttributeStringIgnoreCase(Variable njvar, String aname) {
        Objects.requireNonNull(njvar, "Variable cannot be null.");
        Attribute a = njvar.attributes().findAttributeIgnoreCase(aname);
        if (a == null || a.isArray() || !a.isString()) {
            return null;
        }
        return a.getStringValue();
    }

    public static String getAttributeStringIgnoreCase(Group njgroup, String aname) {
        Objects.requireNonNull(njgroup, "Group cannot be null.");
        Attribute a = njgroup.attributes().findAttributeIgnoreCase(aname);
        if (a == null || a.isArray() || !a.isString()) {
            return null;
        }
        return a.getStringValue();
    }

    public static Integer getAttributeIntegerIgnoreCase(Variable njvar, String aname) {
        Objects.requireNonNull(njvar, "Variable cannot be null.");
        Attribute a = njvar.attributes().findAttributeIgnoreCase(aname);
        if (a == null || a.isArray()) {
            return null;
        }
        Number nn = a.getNumericValue();
        if (nn == null) {
            return null;
        }
        Integer ii = nn.intValue();
        return ii;
    }

    public static String getLongName(Variable njvar) {
        Objects.requireNonNull(njvar, "Variable cannot be null.");
        String str = NcUtils.getAttributeStringIgnoreCase(njvar, "long_name");
        if (str == null || str.isEmpty()) {
            str = NcUtils.getAttributeStringIgnoreCase(njvar, "title");
        }
        if (str == null || str.isEmpty()) {
            str = NcUtils.getAttributeStringIgnoreCase(njvar, "standard_name");
        }
        if (str == null || str.isEmpty()) {
            str = njvar.getShortName();
        }
        if (str == null || str.isEmpty()) {
            return "";
        }
        return str.stripTrailing().replace('_', ' ');
    }

    public static String getLongName(Group njgroup) {
        Objects.requireNonNull(njgroup, "Group cannot be null.");
        String str = NcUtils.getAttributeStringIgnoreCase(njgroup, "long_name");
        if (str == null || str.isEmpty()) {
            str = NcUtils.getAttributeStringIgnoreCase(njgroup, "title");
        }
        if (str == null || str.isEmpty()) {
            str = njgroup.getShortName();
        }
        if (str == null || str.isEmpty()) {
            return "";
        }
        return str.stripTrailing();
    }

    public static String getUnitsStr(Variable njvar) {
        if (njvar == null) {
            return "";
        }
        Attribute a = njvar.attributes().findAttributeIgnoreCase("units");
        if (a == null) {
            a = njvar.attributes().findAttributeIgnoreCase("unit");
        }
        if (a == null) {
            return "";
        }
        String ss = a.getStringValue();
        if (ss == null) {
            LOGGER.debug("Variable has units attribute that is not a char string.");
            return "";
        }
        ss = ss.replaceAll("\\*\\*", "^");
        ss = ss.stripTrailing();
        return ss;
    }

    public static Unit getUdunits(Variable njvar) {
        if (njvar == null) {
            LOGGER.trace("Null var received; returning null units");
            return null;
        }
        String unitsStr = NcUtils.getUnitsStr(njvar);
        if (unitsStr.isEmpty()) {
            return null;
        }
        try {
            return NcUnitUtils.parse(unitsStr);
        }
        catch (Exception exc) {
            LOGGER.debug("Could not parse {} for UDUNITS", (Object)unitsStr);
            return null;
        }
    }

    public static boolean isLongitudeVar1D(Variable var) {
        if (var == null || var.getRank() != 1) {
            return false;
        }
        String ustr = NcUtils.getUnitsStr(var);
        if (ustr.isEmpty()) {
            return false;
        }
        return NcUnitUtils.isDegreesEast(ustr);
    }

    public static boolean isLatitudeVar1D(Variable var) {
        if (var == null || var.getRank() != 1) {
            return false;
        }
        String ustr = NcUtils.getUnitsStr(var);
        if (ustr.isEmpty()) {
            return false;
        }
        return NcUnitUtils.isDegreesNorth(ustr);
    }

    public static boolean isPressure(Variable njvar) {
        if (njvar == null) {
            LOGGER.trace("Null var received; returning false");
            return false;
        }
        Unit uu = NcUtils.getUdunits(njvar);
        if (uu == null) {
            return false;
        }
        Unit pressure = SI.PASCAL;
        return uu.isCompatible(pressure);
    }

    public static boolean isHeight(Variable njvar) {
        if (njvar == null) {
            LOGGER.trace("Null var received; returning false");
            return false;
        }
        Unit uu = NcUtils.getUdunits(njvar);
        if (uu == null) {
            return false;
        }
        BaseUnit distance = SI.METER;
        if (!uu.isCompatible(distance)) {
            return false;
        }
        String p = NcUtils.getAttributeStringIgnoreCase(njvar, "positive");
        return "up".equals(p);
    }

    public static boolean isDepth(Variable njvar) {
        if (njvar == null) {
            LOGGER.trace("Null var received; returning false");
            return false;
        }
        Unit uu = NcUtils.getUdunits(njvar);
        if (uu == null) {
            return false;
        }
        BaseUnit distance = SI.METER;
        if (!uu.isCompatible(distance)) {
            return false;
        }
        String p = NcUtils.getAttributeStringIgnoreCase(njvar, "positive");
        return "down".equals(p);
    }

    public static boolean isUnixTime(Variable njvar) {
        if (njvar == null) {
            LOGGER.trace("Null var received; returning false");
            return false;
        }
        String ustr = NcUtils.getUnitsStr(njvar);
        if (ustr.isEmpty()) {
            return false;
        }
        if (ustr.contains(" since ")) {
            try {
                NcUtils.getRelativeTimeHandler(null, njvar);
                return true;
            }
            catch (Exception exc) {
                LOGGER.debug("Could not parse {} as relative time", (Object)ustr);
            }
        } else {
            return NcUtils.isJulianTimeUnitsStr(ustr);
        }
        return false;
    }

    public static boolean isRelativeTimeVar(NcDataset ncd, Variable njvar) {
        String unitsStr = NcUtils.getUnitsStr(njvar);
        if (unitsStr.isEmpty()) {
            return false;
        }
        if (!unitsStr.toLowerCase().contains(" since ")) {
            return false;
        }
        Attribute calendarA = njvar.findAttribute("calendar");
        String calendar = null;
        if (calendarA != null) {
            calendar = calendarA.getStringValue();
        } else if (ncd != null) {
            calendar = ncd.getGlobalAttributeString("calendar");
        }
        if (calendar == null) {
            calendar = "gregorian";
        }
        try {
            new RelativeTime(unitsStr, calendar, njvar.getDataType().isFloatingPoint());
            return true;
        }
        catch (Exception exc) {
            return false;
        }
    }

    public static boolean isJulianTimeVar(Variable njvar) {
        String unitsStr = NcUtils.getUnitsStr(njvar);
        return NcUtils.isJulianTimeUnitsStr(unitsStr);
    }

    private static boolean isJulianTimeUnitsStr(String unitsStr) {
        if (unitsStr == null || unitsStr.isEmpty()) {
            return false;
        }
        return NcUtils.isTrueJulianTimeUnitsStr(unitsStr) || NcUtils.isModifiedJulianTimeUnitsStr(unitsStr);
    }

    public static boolean isTrueJulianTimeUnitsStr(String unitsStr) {
        if (unitsStr == null || unitsStr.isEmpty()) {
            return false;
        }
        String ustr = unitsStr.toLowerCase().replaceAll(" ", "_");
        return "julian_date".equals(ustr) || "julian_day".equals(ustr) || "true_julian_date".equals(ustr) || "true_julian_day".equals(ustr) || "jd".equals(ustr);
    }

    public static boolean isModifiedJulianTimeUnitsStr(String unitsStr) {
        if (unitsStr == null || unitsStr.isEmpty()) {
            return false;
        }
        String ustr = unitsStr.toLowerCase().replaceAll(" ", "_");
        return "modified_julian_date".equals(ustr) || "modified_julian_day".equals(ustr) || "mjd".equals(unitsStr);
    }

    public static RelativeTime getRelativeTimeHandler(NcDataset ncd, Variable dimVar) throws ParseException {
        String unitsStr = NcUtils.getUnitsStr(dimVar);
        String calendar = null;
        if (dimVar.findAttribute("calendar") != null) {
            calendar = dimVar.findAttribute("calendar").getStringValue();
        } else if (ncd != null) {
            calendar = ncd.getGlobalAttributeString("calendar");
        }
        if (calendar == null) {
            calendar = "gregorian";
        }
        return new RelativeTime(unitsStr, calendar, dimVar.getDataType().isFloatingPoint());
    }

    public static String[] parseCoordinatesAttribute(Variable njvar) {
        Objects.requireNonNull(njvar, "Variable cannot be null.");
        AttributeContainer attribs = njvar.attributes();
        if (attribs == null) {
            return null;
        }
        Attribute coordsA = attribs.findAttributeIgnoreCase("coordinates");
        if (coordsA == null) {
            return null;
        }
        String s = coordsA.getStringValue();
        return s.split(" ");
    }

    public static boolean isGriddable(VariableDS njvarDS) {
        Objects.requireNonNull(njvarDS, "VariableDS cannot be null.");
        return NcUtils.isGriddable1D(njvarDS) || NcUtils.isGriddable2D(njvarDS);
    }

    public static boolean isGriddable1D(VariableDS njvarDS) {
        Objects.requireNonNull(njvarDS, "VariableDS cannot be null.");
        DataType dt = njvarDS.getDataType();
        if (!dt.isNumeric() || njvarDS.isScalar()) {
            return false;
        }
        int rank = njvarDS.getRank();
        int[] shape = njvarDS.getShape();
        int count = 0;
        for (int i = 0; i < rank; ++i) {
            if (shape[i] == 0) {
                return false;
            }
            if (!NcUtils.isApparentNumericDimension(njvarDS, i) || shape[i] <= 1) continue;
            ++count;
        }
        return count > 0;
    }

    public static boolean isGriddable2D(VariableDS njvarDS) {
        Objects.requireNonNull(njvarDS, "VariableDS cannot be null.");
        DataType dt = njvarDS.getDataType();
        if (!dt.isNumeric() || njvarDS.isScalar()) {
            return false;
        }
        int rank = njvarDS.getRank();
        if (rank < 2) {
            return false;
        }
        int count = 0;
        int[] shape = njvarDS.getShape();
        for (int i = 0; i < rank; ++i) {
            if (shape[i] == 0) {
                return false;
            }
            if (!NcUtils.isApparentNumericDimension(njvarDS, i) || shape[i] <= 1) continue;
            ++count;
        }
        return count > 1;
    }

    public static boolean isApparentNumericDimension(VariableDS njvarDS, int index) {
        Objects.requireNonNull(njvarDS, "VariableDS cannot be null.");
        Dimension d = njvarDS.getDimension(index);
        String dsname = d.getShortName();
        if (dsname == null) {
            return true;
        }
        List csList = njvarDS.getCoordinateSystems();
        if (csList.isEmpty()) {
            return true;
        }
        CoordinateSystem cs = (CoordinateSystem)csList.get(0);
        ImmutableList<CoordinateAxis> axes = cs.getCoordinateAxes();
        for (CoordinateAxis ax : axes) {
            String asname = ax.getShortName();
            if (!asname.equals(dsname)) continue;
            return ax.isNumeric();
        }
        return true;
    }

    public static int findDimensionIndex(NcVariable ncvar, String dimName) {
        VariableDS varDS = (VariableDS)ncvar.getObject();
        return NcUtils.findDimensionIndex(varDS, dimName);
    }

    public static int findDimensionIndex(NcVariable ncvar, Dimension dd) {
        VariableDS varDS = (VariableDS)ncvar.getObject();
        return NcUtils.findDimensionIndex(varDS, dd);
    }

    public static int findDimensionIndex(VariableDS njvarDS, String dimName) {
        Objects.requireNonNull(njvarDS, "VariableDS cannot be null.");
        if (dimName == null || dimName.isEmpty()) {
            return -1;
        }
        int index = njvarDS.findDimensionIndex(dimName);
        if (index > -1) {
            return index;
        }
        List dimensions = njvarDS.getDimensions();
        int isize = dimensions.size();
        for (int i = 0; i < isize; ++i) {
            Dimension d = (Dimension)dimensions.get(i);
            String dfname = d.makeFullName();
            boolean matches = dimName.equals(dfname);
            if (!matches) continue;
            return i;
        }
        return -1;
    }

    public static int findDimensionIndex(VariableDS var, Dimension dd) {
        String ddFullName = dd.getFullName();
        String ddShortName = dd.getShortName();
        int ddLength = dd.getLength();
        LOGGER.trace("Searching for {}", (Object)ddFullName);
        List dimensions = var.getDimensions();
        int isize = dimensions.size();
        for (int i = 0; i < isize; ++i) {
            Dimension d = (Dimension)dimensions.get(i);
            String dFullName = d.makeFullName();
            LOGGER.trace("-- Compare to {}, {}", (Object)i, (Object)dFullName);
            if (d.equals(dd)) {
                LOGGER.trace("-- Objects matched for dim {}", (Object)i);
                return i;
            }
            if (d.getLength() != ddLength) {
                LOGGER.trace("-- Lengths do NOT match for dim {}", (Object)i);
                continue;
            }
            if (dFullName.equals(ddFullName)) {
                LOGGER.trace("-- Full names matched for dim {}", (Object)i);
                return i;
            }
            if (d.getShortName().equals(ddShortName) && ddShortName.matches("dimension\\d+")) {
                LOGGER.trace("-- HDF unnamed dimension names matched for dim {}", (Object)i);
                return i;
            }
            if (d.getShortName().equals(ddShortName)) {
                LOGGER.trace("-- Short names matched for dim {}", (Object)i);
                return i;
            }
            LOGGER.trace("-- No match {}", (Object)i);
        }
        LOGGER.trace("-- No match");
        return -1;
    }

    public static String getGridMappingName(VariableDS tvarDS) {
        Attribute a = tvarDS.attributes().findAttributeIgnoreCase("grid_mapping_name");
        if (a == null) {
            a = tvarDS.attributes().findAttributeIgnoreCase("Projection_Name");
        }
        if (a == null) {
            a = tvarDS.attributes().findAttributeIgnoreCase("transform_name");
        }
        if (a == null) {
            return null;
        }
        return a.getStringValue();
    }

    public static VariableDS findGridMappingVariable(NcDataset ncd, VariableDS varDS) {
        String tvarName = null;
        String possibleTvarName = null;
        List csList = varDS.getCoordinateSystems();
        if (csList == null || csList.isEmpty()) {
            LOGGER.trace("cs list null/empty");
        }
        if (csList != null && !csList.isEmpty()) {
            ImmutableList<CoordinateTransform> ctlist;
            CoordinateSystem cs = (CoordinateSystem)csList.get(0);
            ImmutableList<CoordinateTransform> immutableList = ctlist = cs == null ? null : cs.getCoordinateTransforms();
            if (ctlist != null && ctlist.size() > 0) {
                LOGGER.trace("ctlist size {}", (Object)ctlist.size());
                CoordinateTransform ct = (CoordinateTransform)ctlist.get(0);
                String ctname = ct.getName();
                LOGGER.trace("ct name? {}", (Object)ctname);
                VariableDS tvarDS = ncd.getVariableDS(ctname);
                if (tvarDS != null) {
                    LOGGER.trace("transform var found via CS");
                    return tvarDS;
                }
                if (ctlist.size() == 1) {
                    tvarName = ctname;
                } else {
                    possibleTvarName = ctname;
                }
            }
        }
        LOGGER.trace("tvarName {}, possible {}", (Object)tvarName, (Object)possibleTvarName);
        if (tvarName == null) {
            Attribute a = varDS.attributes().findAttributeIgnoreCase("grid_mapping");
            LOGGER.trace("grid_mapping att {}", (Object)a);
            if (a == null) {
                a = varDS.attributes().findAttributeIgnoreCase("_CoordinateSystems");
                LOGGER.trace("_CoordinateSystems {}", (Object)a);
                if (a == null) {
                    return null;
                }
                VariableDS pcsDS = ncd.getVariableDS(a.getStringValue());
                if (pcsDS == null) {
                    return null;
                }
                a = pcsDS.attributes().findAttributeIgnoreCase("_CoordinateTransforms");
                LOGGER.trace("_CoordinateTransforms {}", (Object)a);
            }
            if (a != null) {
                tvarName = a.getStringValue();
            }
        }
        LOGGER.trace("tvarName {}", (Object)tvarName);
        if (tvarName == null && possibleTvarName != null) {
            tvarName = possibleTvarName;
        }
        if (tvarName == null) {
            return null;
        }
        VariableDS mappingVarDS = null;
        Group njgroup = varDS.getParentGroup();
        if (njgroup != null) {
            mappingVarDS = (VariableDS)njgroup.findVariableLocal(tvarName);
            if (mappingVarDS != null) {
                return mappingVarDS;
            }
            Group njgroupUp1 = njgroup.getParentGroup();
            if (njgroupUp1 != null && (mappingVarDS = (VariableDS)njgroupUp1.findVariableLocal(tvarName)) != null) {
                return mappingVarDS;
            }
        }
        mappingVarDS = ncd.getVariableDS(tvarName);
        return mappingVarDS;
    }

    public static CoordinateAxis getCoordinateAxisOfType(NcVariable ncvar, NcAxisType axtype) {
        VariableDS njvarDS = (VariableDS)ncvar.getObject();
        List csList = njvarDS.getCoordinateSystems();
        int isize = csList.size();
        LOGGER.trace("cslist size {}", (Object)isize);
        if (isize > 0) {
            if (isize > 1) {
                LOGGER.debug("Variable has more than one coordinate system reported");
            }
            CoordinateSystem cs = (CoordinateSystem)csList.get(0);
            return NcUtils.getCoordinateAxisCS(ncvar, axtype, cs);
        }
        return NcUtils.getCoordinateAxisNoCS(ncvar, axtype);
    }

    private static CoordinateAxis getCoordinateAxisCS(NcVariable ncvar, NcAxisType axtype, CoordinateSystem cs) {
        NcDataset dataset = ncvar.getDataset();
        VariableDS njvarDS = (VariableDS)ncvar.getObject();
        int rank = njvarDS.getRank();
        switch (axtype) {
            case LAT: {
                CoordinateAxis lat = cs.getLatAxis();
                if (lat == null) {
                    for (int i = 0; i < rank; ++i) {
                        Dimension d = njvarDS.getDimension(i);
                        VariableDS cv = dataset.getCoordinateVariable(d);
                        if (cv == null || !NcUtils.isLatitudeVar1D(cv)) continue;
                        lat = NcUtils.getCoordinateAxisOfType(cv, AxisType.Lat);
                    }
                }
                return lat;
            }
            case LON: {
                CoordinateAxis lon = cs.getLonAxis();
                if (lon == null) {
                    for (int i = 0; i < rank; ++i) {
                        Dimension d = njvarDS.getDimension(i);
                        VariableDS cv = dataset.getCoordinateVariable(d);
                        if (!NcUtils.isLongitudeVar1D(cv)) continue;
                        lon = NcUtils.getCoordinateAxisOfType(cv, AxisType.Lon);
                    }
                }
                return lon;
            }
            case VERT: {
                if (!cs.hasVerticalAxis()) {
                    return null;
                }
                CoordinateAxis vertAxis = cs.getPressureAxis();
                if (vertAxis == null) {
                    vertAxis = cs.getHeightAxis();
                }
                if (vertAxis == null) {
                    vertAxis = cs.getZaxis();
                }
                return vertAxis;
            }
            case TIME: {
                CoordinateAxis tAxis = cs.getTaxis();
                return tAxis;
            }
            case GEOX: {
                return cs.getXaxis();
            }
            case GEOY: {
                return cs.getYaxis();
            }
        }
        return null;
    }

    private static CoordinateAxis getCoordinateAxisNoCS(NcVariable ncvar, NcAxisType axtype) {
        String[] tokens;
        NcDataset dataset = ncvar.getDataset();
        VariableDS njvarDS = (VariableDS)ncvar.getObject();
        Attribute coordsA = njvarDS.attributes().findAttributeIgnoreCase("coordinates");
        VariableDS lonVar = null;
        VariableDS latVar = null;
        String cstring = coordsA == null ? "" : coordsA.getStringValue();
        for (String token : tokens = cstring.split(" ")) {
            VariableDS tokenVar = dataset.getVariableDS(token);
            if (tokenVar == null) continue;
            String unitsStr = NcUtils.getUnitsStr(tokenVar);
            if (NcUnitUtils.isDegreesEast(unitsStr)) {
                lonVar = tokenVar;
                continue;
            }
            if (!NcUnitUtils.isDegreesNorth(unitsStr)) continue;
            latVar = tokenVar;
        }
        if (axtype == NcAxisType.LAT) {
            if (latVar == null) {
                return null;
            }
            return NcUtils.getCoordinateAxisOfType(latVar, AxisType.Lat);
        }
        if (axtype == NcAxisType.LON) {
            if (lonVar == null) {
                return null;
            }
            return NcUtils.getCoordinateAxisOfType(lonVar, AxisType.Lon);
        }
        return null;
    }

    public static CoordinateAxis getCoordinateAxisWithStdName(VariableDS njvarDS, String stdname) {
        Objects.requireNonNull(stdname, "standard_name cannot be null.");
        LOGGER.trace("Searching for CoordinateAxes by standard_name");
        List csList = njvarDS.getCoordinateSystems();
        int isize = csList.size();
        if (isize > 0) {
            if (isize > 1) {
                LOGGER.debug("Variable has more than one coordinate system reported");
            }
            CoordinateSystem cs = (CoordinateSystem)csList.get(0);
            ImmutableList<CoordinateAxis> axes = cs.getCoordinateAxes();
            for (CoordinateAxis ax : axes) {
                if (ax.getRank() != 1) continue;
                String axname = ax.getShortName();
                String sname = NcUtils.getAttributeStringIgnoreCase(ax, "standard_name");
                if (!"std_name".equals(sname)) continue;
                LOGGER.trace("Axis {} matches standad_name {}", (Object)axname, (Object)sname);
                return ax;
            }
        }
        return null;
    }

    public static CoordinateAxis getCoordinateAxisMatchingShortName(VariableDS njvarDS, String regex) {
        Objects.requireNonNull(regex, "regex cannot be null.");
        LOGGER.trace("Searching for CoordinateAxes by standard_name");
        List csList = njvarDS.getCoordinateSystems();
        int isize = csList.size();
        if (isize > 0) {
            if (isize > 1) {
                LOGGER.debug("Variable has more than one coordinate system reported");
            }
            CoordinateSystem cs = (CoordinateSystem)csList.get(0);
            ImmutableList<CoordinateAxis> axes = cs.getCoordinateAxes();
            for (CoordinateAxis ax : axes) {
                String axname;
                if (ax.getRank() != 1 || !(axname = ax.getShortName()).matches(regex)) continue;
                LOGGER.trace("Axis {} matches regex {}", (Object)axname, (Object)regex);
                return ax;
            }
        }
        return null;
    }

    public static CoordinateAxis getCoordinateAxisOfType(VariableDS cv, AxisType axtype) {
        LOGGER.trace("{}, {}", (Object)cv, (Object)axtype);
        Objects.requireNonNull(cv, "Coordinate variable cannot be null.");
        Objects.requireNonNull(axtype, "Axis type cannot be null.");
        if (cv instanceof CoordinateAxis) {
            LOGGER.trace("CV is a CoordinateAxis. Returning cast and ignoring axtype.");
            return (CoordinateAxis)cv;
        }
        LOGGER.trace("CV is not a CoordinateAxis. Attempting to build one.");
        CoordinateAxis ca = null;
        int rank = cv.getRank();
        if (rank == 1) {
            Group grp = cv.getParentGroup();
            ca = ((CoordinateAxis1D.Builder)((CoordinateAxis1D.Builder)CoordinateAxis1D.builder().setOriginalVariable(cv)).setAxisType(axtype)).build(grp);
        } else if (rank == 2) {
            Group grp = cv.getParentGroup();
            ca = ((CoordinateAxis2D.Builder)((CoordinateAxis2D.Builder)CoordinateAxis2D.builder().setOriginalVariable(cv)).setAxisType(axtype)).build(grp);
        } else {
            LOGGER.trace("VariableDS is not 1D or 2D. Returning null CoordinateAxis.");
        }
        return ca;
    }

    public static VariableDS getCoordinateVariable(NcVariable ncvar, NcAxisType axtype) {
        NcDataset dataset = ncvar.getDataset();
        VariableDS njvarDS = (VariableDS)ncvar.getObject();
        int rank = ncvar.getRank();
        int[] shape = ncvar.getShape();
        for (int i = 0; i < rank; ++i) {
            Dimension d;
            VariableDS cv;
            if (shape[i] < 2 || (cv = dataset.getCoordinateVariable(d = njvarDS.getDimension(i))) == null) continue;
            if (axtype == NcAxisType.LON && NcUtils.isLongitudeVar1D(cv)) {
                return cv;
            }
            if (axtype == NcAxisType.LAT && NcUtils.isLatitudeVar1D(cv)) {
                return cv;
            }
            if (axtype != NcAxisType.TIME || !NcUtils.isRelativeTimeVar(dataset, cv) && !NcUtils.isJulianTimeVar(cv)) continue;
            return cv;
        }
        return null;
    }

    public static VariableDS getTrajectoryTimeVar(NcVariable ncvar) {
        NcVarType vtype = ncvar.getVarType();
        if (vtype instanceof NcDiscreteVarType && ((NcDiscreteVarType)vtype).isTrajectory()) {
            NcDataset ncd = ncvar.getDataset();
            VariableDS varDS = (VariableDS)ncvar.getObject();
            return NcArrayFeatureTypeTrajectory.getTimeCoordinateVar(ncd, varDS);
        }
        return null;
    }

    public static String formatLongitude(double lon) {
        return PFORMAT_7G.sprintfx(lon) + "\u00b0E";
    }

    public static String formatLatitude(double lat) {
        return PFORMAT_7G.sprintfx(lat) + "\u00b0N";
    }

    public static double objectValueToDouble(Object oo) {
        if (oo == null) {
            LOGGER.trace("Object is null");
            return Double.NaN;
        }
        if (oo instanceof Double || oo instanceof Float) {
            return (Double)oo;
        }
        if (oo instanceof Long || oo instanceof Integer || oo instanceof Short || oo instanceof Byte) {
            return (Double)oo;
        }
        if (oo instanceof Character) {
            Character cc = (Character)oo;
            return cc.charValue();
        }
        LOGGER.trace("Not a type we know how to convert to a double.");
        return Double.NaN;
    }

    public static void appendScalarObjectValue(StringBuilder sb, Object oo) {
        if (oo instanceof Double || oo instanceof Float) {
            sb.append(PFORMAT_10G.sprintfx(oo));
        } else if (oo instanceof Long || oo instanceof Integer || oo instanceof Short || oo instanceof Byte) {
            sb.append(oo.toString());
        } else if (oo instanceof Character) {
            Character cc = (Character)oo;
            sb.append((int)cc.charValue());
            if (Character.isAlphabetic(cc.charValue())) {
                sb.append(" = '").append(cc).append("'");
            }
        } else {
            sb.append("'").append(oo.toString()).append("'");
        }
    }

    private NcUtils() {
    }
}

