'use strict';

var router = module.parent.router;
var testU = require('../utils/test_utils.js');

var deepEqual = testU.deepEqual;
var roundDecimal = testU.roundDecimal;

describe('Types', function () {

    before(function () { return router.setup(); }); // eslint-disable-line

    var typesTableSpec = {
        domain: 'restbase.cassandra.test.local',
        table: 'typeTable',
        options: { durability: 'low' },
        attributes: {
            string: 'string',
            blob: 'blob',
            set: 'set<string>',
            int: 'int',
            varint: 'varint',
            decimal: 'decimal',
            float: 'float',
            double: 'double',
            boolean: 'boolean',
            timeuuid: 'timeuuid',
            uuid: 'uuid',
            timestamp: 'timestamp',
            json: 'json',
            num_string: 'string', // String containing a `numeric` value to check it's not converted to number
            long: 'long'
        },
        index: [
            { attribute: 'string', type: 'hash' }
        ]
    };
    var typeSetsTableSpec = {
        domain: 'restbase.cassandra.test.local',
        table: 'typeSetsTable',
        options: { durability: 'low' },
        attributes: {
            string: 'string',
            set: 'set<string>',
            blob: 'set<blob>',
            int: 'set<int>',
            varint: 'set<varint>',
            decimal: 'set<decimal>',
            float: 'set<float>',
            double: 'set<double>',
            boolean: 'set<boolean>',
            timeuuid: 'set<timeuuid>',
            uuid: 'set<uuid>',
            timestamp: 'set<timestamp>',
            json: 'set<json>'
        },
        index: [
            { attribute: 'string', type: 'hash' }
        ]
    };
    var longRangeKeyTableSpec = {
        domain: 'restbase.cassandra.test.local',
        table: 'longRangeKeyTable',
        options: { durability: 'low' },
        attributes: {
            string: 'string',
            long: 'long',
            value: 'string'
        },
        index: [
            { attribute: 'string', type: 'hash' },
            { attribute: 'long', type: 'range', order: 'desc' }
        ]
    };

    context('Standard', function () {
        this.timeout(5000);
        it('creates table with various types', function () {
            return router.request({
                uri: '/restbase.cassandra.test.local/sys/table/typeTable',
                method: 'put',
                body: typesTableSpec
            })
            .then(function (response) {
                deepEqual(response.status, 201);
                return router.request({
                    uri: '/restbase.cassandra.test.local/sys/table/typeTable',
                    method: 'get',
                    body: {}
                });
            })
            .then(function (response) {
                deepEqual(response.status, 200);
                deepEqual(response.body, typesTableSpec);
            });
        });
        it('inserts row with various types', function () {
            return router.request({
                uri: '/restbase.cassandra.test.local/sys/table/typeTable/',
                method: 'put',
                body: {
                    table: 'typeTable',
                    attributes: {
                        string: 'string',
                        blob: new Buffer('blob'),
                        set: ['bar', 'baz', 'foo'],
                        int: -1,
                        varint: -4503599627370496,
                        decimal: '1.2',
                        float: -1.1,
                        double: 1.2,
                        boolean: true,
                        timeuuid: 'c931ec94-6c31-11e4-b6d0-0f67e29867e0',
                        uuid: 'd6938370-c996-4def-96fb-6af7ba9b6f72',
                        timestamp: '2014-11-14T19:10:40.912Z',
                        json: {
                            foo: 'bar'
                        },
                        long: '9223372036854775807'
                    }
                }
            })
            .then(function (response) {
                deepEqual(response, { status: 201 });
            });
        });
        it('insert additional rows with various types', function () {
            return router.request({
                uri: '/restbase.cassandra.test.local/sys/table/typeTable/',
                method: 'put',
                body: {
                    table: 'typeTable',
                    attributes: {
                        string: 'string',
                        blob: new Buffer('blob'),
                        set: ['bar', 'baz', 'foo'],
                        int: 1,
                        varint: 1,
                        decimal: '1.4',
                        float: -3.434,
                        double: 1.2,
                        boolean: true,
                        timeuuid: 'c931ec94-6c31-11e4-b6d0-0f67e29867e0',
                        uuid: 'd6938370-c996-4def-96fb-6af7ba9b6f72',
                        timestamp: '2014-11-14T19:10:40.912Z',
                        json: {
                            foo: 'bar'
                        },
                        long: '9223372036854775806'
                    }
                }
            })
            .then(function (response) {
                deepEqual(response, { status: 201 });
            });
        });
        it('retrieves rows of matching types', function () {
            return router.request({
                uri: '/restbase.cassandra.test.local/sys/table/typeTable/',
                method: 'get',
                body: {
                    table: 'typeTable',
                    attributes: {
                        string: 'string'
                    },
                    proj: ['string', 'blob', 'set', 'int', 'varint', 'decimal',
                        'float', 'double', 'boolean', 'timeuuid', 'uuid',
                        'timestamp', 'json', 'long']
                }
            })
            .then(function (response) {
                response.body.items[0].float = roundDecimal(response.body.items[0].float);
                deepEqual(response.body.items, [{
                    string: 'string',
                    blob: new Buffer('blob'),
                    set: ['bar', 'baz', 'foo'],
                    int: 1,
                    varint: 1,
                    decimal: '1.4',
                    float: -3.43,
                    double: 1.2,
                    boolean: true,
                    timeuuid: 'c931ec94-6c31-11e4-b6d0-0f67e29867e0',
                    uuid: 'd6938370-c996-4def-96fb-6af7ba9b6f72',
                    timestamp: '2014-11-14T19:10:40.912Z',
                    json: {
                        foo: 'bar'
                    },
                    long: '9223372036854775806'
                }]);
            });
        });
        it('return string if numeric data saved in string typed field', function () {
            return router.request({
                uri: '/restbase.cassandra.test.local/sys/table/typeTable/',
                method: 'put',
                body: {
                    table: 'typeTable',
                    attributes: {
                        string: 'num_string_test',
                        int: 10,
                        boolean: false,
                        num_string: '0'
                    }
                }
            })
            .then(function (response) {
                deepEqual(response, { status: 201 });
                return router.request({
                    uri: '/restbase.cassandra.test.local/sys/table/typeTable/',
                    method: 'get',
                    body: {
                        table: 'typeTable',
                        attributes: {
                            string: 'num_string_test'
                        },
                        proj: ['num_string']
                    }
                });
            })
            .then(function (response) {
                deepEqual(response.status, 200);
                deepEqual(response.body.items.length, 1);
                deepEqual(typeof response.body.items[0].num_string, 'string');
            });
        });
        it('supports comparing longs', function () {
            return router.request({
                uri: '/restbase.cassandra.test.local/sys/table/' + longRangeKeyTableSpec.table,
                method: 'put',
                body: longRangeKeyTableSpec
            })
            .then(function (res) {
                deepEqual(res.status, 201);
                return router.request({
                    uri: '/restbase.cassandra.test.local/sys/table/' + longRangeKeyTableSpec.table + '/',
                    method: 'put',
                    body: {
                        table: longRangeKeyTableSpec.table,
                        attributes: {
                            string: 'key',
                            long: '9223372036854775807',
                            value: 'value1'
                        }
                    }
                });
            })
            .then(function (res) {
                deepEqual(res.status, 201);
                return router.request({
                    uri: '/restbase.cassandra.test.local/sys/table/' + longRangeKeyTableSpec.table + '/',
                    method: 'put',
                    body: {
                        table: longRangeKeyTableSpec.table,
                        attributes: {
                            string: 'key',
                            long: '10',
                            value: 'value2'
                        }
                    }
                });
            })
            .then(function (res) {
                deepEqual(res.status, 201);
                return router.request({
                    uri: '/restbase.cassandra.test.local/sys/table/' + longRangeKeyTableSpec.table + '/',
                    method: 'get',
                    body: {
                        table: longRangeKeyTableSpec.table,
                        attributes: {
                            string: 'key',
                            long: { ge: '20' }
                        }
                    }
                });
            })
            .then(function (res) {
                deepEqual(res.status, 200);
                deepEqual(res.body.items.length, 1);
                deepEqual(res.body.items[0].value, 'value1');
            });
        });
        it('drops table', function () {
            this.timeout(15000);
            return router.request({
                uri: '/restbase.cassandra.test.local/sys/table/typeTable',
                method: 'delete',
                body: {}
            })
            .then(function (res) {
                deepEqual(res.status, 204);
                return router.request({
                    uri: '/restbase.cassandra.test.local/sys/table/typeTable',
                    method: 'get',
                    body: {}
                });
            })
            .then(function (res) {
                deepEqual(res.status, 500);
                return router.request({
                    uri: '/restbase.cassandra.test.local/sys/table/' + longRangeKeyTableSpec.table,
                    method: 'delete',
                    body: {}
                });
            });
        });
    });

    context('Sets', function () {
        this.timeout(5000);
        it('creates table with various sets types', function () {
            return router.request({
                uri: '/restbase.cassandra.test.local/sys/table/typeSetsTable',
                method: 'put',
                body: typeSetsTableSpec
            })
            .then(function (response) {
                deepEqual(response.status, 201);
                return router.request({
                    uri: '/restbase.cassandra.test.local/sys/table/typeSetsTable',
                    method: 'get',
                    body: {}
                });
            })
            .then(function (response) {
                deepEqual(response.status, 200);
                deepEqual(response.body, typeSetsTableSpec);
            });
        });
        it('inserts nulls and equivalents', function () {
            return router.request({
                uri: '/restbase.cassandra.test.local/sys/table/typeSetsTable/',
                method: 'put',
                body: {
                    table: 'typeSetsTable',
                    attributes: {
                        string: 'nulls',
                        set: [],
                        blob: [],
                        int: [],
                        varint: null
                    }
                }
            })
            .then(function (response) {
                deepEqual(response, { status: 201 });
            });
        });
        it('inserts sets', function () {
            return router.request({
                uri: '/restbase.cassandra.test.local/sys/table/typeSetsTable/',
                method: 'put',
                body: {
                    table: 'typeSetsTable',
                    attributes: {
                        string: 'string',
                        blob: [new Buffer('blob')],
                        set: ['bar', 'baz', 'foo'],
                        varint: [-4503599627370496, 12233232],
                        decimal: ['1.2', '1.6'],
                        float: [1.3, 1.1],
                        double: [1.2, 1.567],
                        boolean: [true, false],
                        timeuuid: ['c931ec94-6c31-11e4-b6d0-0f67e29867e0'],
                        uuid: ['d6938370-c996-4def-96fb-6af7ba9b6f72'],
                        timestamp: ['2014-11-14T19:10:40.912Z', '2014-12-14T19:10:40.912Z'],
                        int: [123456, 2567, 598765],
                        json: [
                            { one: 1, two: 'two' },
                            { foo: 'bar' },
                            { test: [{ a: 'b' }, 3] }
                        ]
                    }
                }
            })
            .then(function (response) {
                deepEqual(response, { status: 201 });
            });
        });
        it('removes duplicates from sets', function () {
            return router.request({
                uri: '/restbase.cassandra.test.local/sys/table/typeSetsTable/',
                method: 'put',
                body: {
                    table: 'typeSetsTable',
                    attributes: {
                        string: 'duplicates',
                        set: ['bar', 'baz', 'foo', 'bar', 'baz']
                    }
                }
            })
            .then(function (response) {
                deepEqual(response, { status: 201 });
                return router.request({
                    uri: '/restbase.cassandra.test.local/sys/table/typeSetsTable/',
                    method: 'get',
                    body: {
                        table: 'typeSetsTable',
                        attributes: {
                            string: 'duplicates'
                        }
                    }
                });
            })
            .then(function (response) {
                deepEqual(response.status, 200);
                deepEqual(response.body.items.length, 1);
                deepEqual(response.body.items[0].set, ['bar', 'baz', 'foo']);
            });
        });
        it('retrieves nulls and equivalents', function () {
            return router.request({
                uri: '/restbase.cassandra.test.local/sys/table/typeSetsTable/',
                method: 'get',
                body: {
                    table: 'typeSetsTable',
                    attributes: {
                        string: 'nulls'
                    }
                }
            })
            .then(function (res) {
                deepEqual(res.body.items[0].string, 'nulls');
                deepEqual(res.body.items[0].blob, null);
            });
        });
        it('retrieves sets', function () {
            return router.request({
                uri: '/restbase.cassandra.test.local/sys/table/typeSetsTable/',
                method: 'get',
                body: {
                    table: 'typeSetsTable',
                    attributes: {
                        string: 'string'
                    },
                    proj: ['string', 'blob', 'set', 'int', 'varint', 'decimal',
                        'double', 'boolean', 'timeuuid', 'uuid', 'float',
                        'timestamp', 'json']
                }
            })
            .then(function (response) {
                // note: Cassandra orders sets, so the expected rows are
                // slightly different than the original, supplied ones
                response.body.items[0].float = [roundDecimal(response.body.items[0].float[0]),
                    roundDecimal(response.body.items[0].float[1])];
                deepEqual(response.body.items[0], {
                    string: 'string',
                    blob: [new Buffer('blob')],
                    set: ['bar', 'baz', 'foo'],
                    int: [2567, 123456, 598765],
                    varint: [
                        -4503599627370496,
                        12233232
                    ],
                    decimal: [
                        '1.2',
                        '1.6'
                    ],
                    double: [1.2, 1.567],
                    boolean: [false, true],
                    timeuuid: ['c931ec94-6c31-11e4-b6d0-0f67e29867e0'],
                    uuid: ['d6938370-c996-4def-96fb-6af7ba9b6f72'],
                    float: [1.1, 1.3],
                    timestamp: ['2014-11-14T19:10:40.912Z', '2014-12-14T19:10:40.912Z'],
                    json: [
                        { foo: 'bar' },
                        { one: 1, two: 'two' },
                        { test: [{ a: 'b' }, 3] }
                    ]
                });
            });
        });
        it('drops table', function () {
            this.timeout(15000);
            return router.request({
                uri: '/restbase.cassandra.test.local/sys/table/typeSetsTable',
                method: 'delete',
                body: {}
            })
            .then(function (res) {
                deepEqual(res.status, 204);
                return router.request({
                    uri: '/restbase.cassandra.test.local/sys/table/typeSetsTable',
                    method: 'get',
                    body: {}
                });
            })
            .then(function (res) {
                deepEqual(res.status, 500);
            });
        });
    });
});
