import { z } from "zod";
const DescriptionSchema = z.object({
  /**
   * A description for security scheme. CommonMark syntax MAY be used for rich text representation.
   */
  description: z.string().optional()
});
const ApiKeyInValues = ["query", "header", "cookie"];
const ApiKeySchema = DescriptionSchema.extend({
  /**
   * REQUIRED. The type of the security scheme. Valid values are "apiKey", "http", "mutualTLS", "oauth2",
   * "openIdConnect".
   */
  type: z.literal("apiKey"),
  /**
   * REQUIRED. The name of the header, query or cookie parameter to be used.
   *
   * TODO: Why do we use an empty string as the default?
   */
  name: z.string().optional().default(""),
  /**
   * REQUIRED. The location of the API key. Valid values are "query", "header", or "cookie".
   */
  in: z.enum(ApiKeyInValues).optional().default("header").catch("header")
});
const HttpSchema = DescriptionSchema.extend({
  /**
   * REQUIRED. The type of the security scheme. Valid values are "apiKey", "http", "mutualTLS", "oauth2",
   * "openIdConnect".
   */
  type: z.literal("http"),
  /**
   * REQUIRED. The name of the HTTP Authentication scheme to be used in the Authorization header as defined in RFC7235.
   * The values used SHOULD be registered in the IANA Authentication Scheme registry. The value is case-insensitive,
   * as defined in RFC7235.
   */
  scheme: z.string().toLowerCase().pipe(z.enum(["basic", "bearer"])).optional().default("basic"),
  /**
   * A hint to the client to identify how the bearer token is formatted. Bearer tokens are usually generated by an
   * authorization server, so this information is primarily for documentation purposes.
   */
  bearerFormat: z.union([z.literal("JWT"), z.literal("bearer"), z.string()]).optional()
});
const OpenIdConnectSchema = DescriptionSchema.extend({
  /**
   * REQUIRED. The type of the security scheme. Valid values are "apiKey", "http", "mutualTLS", "oauth2",
   * "openIdConnect".
   */
  type: z.literal("openIdConnect"),
  /**
   * REQUIRED. Well-known URL to discover the [[OpenID-Connect-Discovery]] provider metadata.
   */
  openIdConnectUrl: z.string().optional().default("")
});
const authorizationUrl = z.string().default("");
const tokenUrl = z.string().default("");
const OAuthFlowObjectSchema = z.object({
  /**
   * The URL to be used for obtaining refresh tokens. This MUST be in the form of a URL. The OAuth2 standard requires
   * the use of TLS.
   */
  refreshUrl: z.string().optional(),
  /**
   * REQUIRED. The available scopes for the OAuth2 security scheme. A map
   * between the scope name and a short description for it. The map MAY be empty.
   */
  scopes: z.record(z.string(), z.string().optional()).optional().default({}).catch({})
});
const ImplicitFlowSchema = OAuthFlowObjectSchema.extend({
  type: z.literal("implicit").optional(),
  authorizationUrl
});
const PasswordFlowSchema = OAuthFlowObjectSchema.extend({
  type: z.literal("password").optional(),
  tokenUrl
});
const ClientCredentialsFlowSchema = OAuthFlowObjectSchema.extend({
  type: z.literal("clientCredentials").optional(),
  tokenUrl
});
const AuthorizationCodeFlowSchema = OAuthFlowObjectSchema.extend({
  type: z.literal("authorizationCode").optional(),
  authorizationUrl,
  tokenUrl
});
const OAuthFlowsObjectSchema = DescriptionSchema.extend({
  /**
   * REQUIRED. The type of the security scheme. Valid values are "apiKey", "http", "mutualTLS", "oauth2",
   * "openIdConnect".
   */
  type: z.literal("oauth2"),
  /**
   * REQUIRED. An object containing configuration information for the flow types supported.
   */
  flows: z.object({
    /**
     * Configuration for the OAuth Implicit flow
     */
    implicit: ImplicitFlowSchema.optional(),
    /**
     * Configuration for the OAuth Resource Owner Password flow
     */
    password: PasswordFlowSchema.optional(),
    /**
     * Configuration for the OAuth Client Credentials flow. Previously called application in OpenAPI 2.0.
     */
    clientCredentials: ClientCredentialsFlowSchema.optional(),
    /**
     * Configuration for the OAuth Authorization Code flow. Previously called accessCode in OpenAPI 2.0.
     */
    authorizationCode: AuthorizationCodeFlowSchema.optional()
  }).partial()
});
const MutualTlsSchema = DescriptionSchema.extend({
  /**
   * REQUIRED. The type of the security scheme. Valid values are "apiKey", "http", "mutualTLS", "oauth2",
   * "openIdConnect".
   */
  type: z.literal("mutualTLS")
});
const SecuritySchemeObjectSchema = z.union([
  ApiKeySchema,
  HttpSchema,
  MutualTlsSchema,
  OAuthFlowsObjectSchema,
  OpenIdConnectSchema
]);
export {
  ApiKeyInValues,
  ApiKeySchema,
  AuthorizationCodeFlowSchema,
  ClientCredentialsFlowSchema,
  HttpSchema,
  ImplicitFlowSchema,
  MutualTlsSchema,
  OAuthFlowObjectSchema,
  OAuthFlowsObjectSchema,
  OpenIdConnectSchema,
  PasswordFlowSchema,
  SecuritySchemeObjectSchema
};
//# sourceMappingURL=security-scheme-object.js.map
