import { N as NarouNovel, b as NarouRankingResult, R as RankingResult, D as DefaultSearchResultFields } from './narou-DCp4aGfA.cjs';
import { b as RankingParams, R as RankingType, q as GzipLevel, F as Fields, u as SearchResultFields, O as OptionalFields } from './narou-search-results-C21hWrnL.cjs';
import './ranking-history.cjs';
import './util/type.cjs';

/**
 * なろう小説ランキングAPIのヘルパークラス。
 *
 * ランキング種別や日付を指定してランキングデータを取得します。
 * また、取得したランキングデータに含まれるNコードを元に、
 * なろう小説APIを利用して詳細な小説情報を取得することも可能です。
 *
 * @class RankingBuilder
 * @see https://dev.syosetu.com/man/rankapi/ なろう小説ランキングAPI仕様
 */
declare class RankingBuilder {
    protected params: Partial<RankingParams>;
    protected api: NarouNovel;
    /**
     * ランキング集計対象の日付
     * @protected
     */
    protected date$: Date;
    /**
     * ランキング種別
     * @protected
     */
    protected type$: RankingType;
    /**
     * constructor
     * @param params - 初期クエリパラメータ
     * @param api - API実行クラスのインスタンス
     * @private
     */
    constructor(params: Partial<RankingParams> | undefined, api: NarouNovel);
    /**
     * ランキング集計対象の日付を指定します。
     *
     * - 日間: 任意の日付
     * - 週間: 火曜日の日付
     * - 月間・四半期: 1日の日付
     *
     * @param date 集計対象の日付
     * @returns {RankingBuilder} this
     * @see https://dev.syosetu.com/man/rankapi/
     */
    date(date: Date): this;
    /**
     * ランキング種別を指定します。
     * @param type ランキング種別
     * @returns {RankingBuilder} this
     * @see https://dev.syosetu.com/man/rankapi/
     */
    type(type: RankingType): this;
    /**
     * gzip圧縮する。
     *
     * 転送量上限を減らすためにも推奨
     * @param {GzipLevel} level gzip圧縮レベル(1～5)
     * @return {RankingBuilder} this
     */
    gzip(level: GzipLevel): this;
    /**
     * クエリパラメータを内部的にセットします。
     * @param obj - セットするパラメータオブジェクト
     * @returns {RankingBuilder} this
     * @private
     */
    protected set(obj: Partial<RankingParams>): this;
    /**
     * 設定されたパラメータに基づき、なろう小説ランキングAPIへのリクエストを実行します。
     *
     * 返される結果には、Nコード、ポイント、順位が含まれます。
     * @returns {Promise<NarouRankingResult[]>} ランキング結果の配列
     * @see https://dev.syosetu.com/man/rankapi/#output
     */
    execute(): Promise<NarouRankingResult[]>;
    /**
     * ランキングAPIを実行し、取得したNコードを元になろう小説APIで詳細情報を取得して結合します。
     */
    executeWithFields(): Promise<RankingResult<DefaultSearchResultFields>[]>;
    /**
     * ランキングAPIを実行し、取得したNコードを元になろう小説APIで詳細情報を取得して結合します。
     *
     * @template TFields - 取得する小説情報のフィールド型
     * @param fields - 取得するフィールドの配列
     * @returns {Promise<RankingResult<SearchResultFields<TFields>>[]>} 詳細情報を含むランキング結果の配列
     */
    executeWithFields<TFields extends Fields>(fields: TFields | TFields[]): Promise<RankingResult<SearchResultFields<TFields>>[]>;
    /**
     * ランキングAPIを実行し、取得したNコードを元になろう小説APIで詳細情報を取得して結合します。
     *
     * @param opt - オプショナルな取得フィールド (`weekly` など)
     * @returns {Promise<RankingResult<DefaultSearchResultFields | "weekly_unique">[]>} 詳細情報を含むランキング結果の配列
     */
    executeWithFields(fields: never[], opt: OptionalFields | OptionalFields[]): Promise<RankingResult<DefaultSearchResultFields | "weekly_unique">[]>;
    /**
     * ランキングAPIを実行し、取得したNコードを元になろう小説APIで詳細情報を取得して結合します。
     *
     * @template TFields - 取得する小説情報のフィールド型
     * @param fields - 取得するフィールドの配列
     * @param opt - オプショナルな取得フィールド (`weekly` など)
     * @returns {Promise<RankingResult<SearchResultFields<TFields> | "weekly_unique">[]>} 詳細情報を含むランキング結果の配列
     */
    executeWithFields<TFields extends Fields>(fields: TFields | TFields[], opt: OptionalFields | OptionalFields[]): Promise<RankingResult<SearchResultFields<TFields> | "weekly_unique">[]>;
}

export { RankingBuilder as default };
