(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.BbobPresetHTML5 = {}));
})(this, (function (exports) { 'use strict';

    const OPEN_BRAKET = '[';
    const CLOSE_BRAKET = ']';
    const SLASH = '/';

    function isTagNode(el) {
        return typeof el === 'object' && el !== null && 'tag' in el;
    }
    function isStringNode(el) {
        return typeof el === 'string';
    }
    function keysReduce(obj, reduce, def) {
        const keys = Object.keys(obj);
        return keys.reduce((acc, key)=>reduce(acc, key, obj), def);
    }
    function getNodeLength(node) {
        if (isTagNode(node) && Array.isArray(node.content)) {
            return node.content.reduce((count, contentNode)=>{
                return count + getNodeLength(contentNode);
            }, 0);
        }
        if (isStringNode(node)) {
            return String(node).length;
        }
        return 0;
    }
    function appendToNode(node, value) {
        if (Array.isArray(node.content)) {
            node.content.push(value);
        }
    }
    /**
     * Replaces " to &qquot;
     * @param {string} value
     */ function escapeAttrValue(value) {
        return value.replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#039;')// eslint-disable-next-line no-script-url
        .replace(/(javascript|data|vbscript|file):/gi, '$1%3A');
    }
    /**
     * Accept name and value and return valid html5 attribute string
     */ function attrValue(name, value) {
        // in case of performance
        switch(typeof value){
            case 'boolean':
                return value ? `${name}` : '';
            case 'number':
                return `${name}="${value}"`;
            case 'string':
                return `${name}="${escapeAttrValue(value)}"`;
            case 'object':
                return `${name}="${escapeAttrValue(JSON.stringify(value))}"`;
            default:
                return '';
        }
    }
    /**
     * Transforms attrs to html params string
     * @example
     * attrsToString({ 'foo': true, 'bar': bar' }) => 'foo="true" bar="bar"'
     */ function attrsToString(values) {
        // To avoid some malformed attributes
        if (values == null) {
            return '';
        }
        return keysReduce(values, (arr, key, obj)=>[
                ...arr,
                attrValue(key, obj[key])
            ], [
            ''
        ]).join(' ');
    }
    /**
     * Gets value from
     * @example
     * getUniqAttr({ 'foo': true, 'bar': bar' }) => 'bar'
     */ function getUniqAttr(attrs) {
        return keysReduce(attrs || {}, (res, key, obj)=>obj[key] === key ? obj[key] : null, null);
    }

    const getTagAttrs = (tag, params)=>{
        const uniqAttr = getUniqAttr(params);
        if (uniqAttr) {
            const tagAttr = attrValue(tag, uniqAttr);
            const attrs = {
                ...params
            };
            delete attrs[String(uniqAttr)];
            const attrsStr = attrsToString(attrs);
            return `${tagAttr}${attrsStr}`;
        }
        return `${tag}${attrsToString(params)}`;
    };
    const toString = (node, openTag, closeTag)=>{
        if (isTagNode(node)) {
            return node.toString({
                openTag,
                closeTag
            });
        }
        return String(node);
    };
    const nodeTreeToString = (content, openTag, closeTag)=>{
        if (Array.isArray(content)) {
            return content.reduce((r, node)=>{
                if (node !== null) {
                    return r + toString(node, openTag, closeTag);
                }
                return r;
            }, '');
        }
        if (content) {
            return toString(content, openTag, closeTag);
        }
        return null;
    };
    class TagNode {
        get length() {
            return getNodeLength(this);
        }
        attr(name, value) {
            if (typeof value !== 'undefined') {
                this.attrs[name] = value;
            }
            return this.attrs[name];
        }
        append(value) {
            return appendToNode(this, value);
        }
        setStart(value) {
            this.start = value;
        }
        setEnd(value) {
            this.end = value;
        }
        toTagStart({ openTag = OPEN_BRAKET, closeTag = CLOSE_BRAKET } = {}) {
            const tagAttrs = getTagAttrs(String(this.tag), this.attrs);
            return `${openTag}${tagAttrs}${closeTag}`;
        }
        toTagEnd({ openTag = OPEN_BRAKET, closeTag = CLOSE_BRAKET } = {}) {
            return `${openTag}${SLASH}${this.tag}${closeTag}`;
        }
        toTagNode() {
            return new TagNode(this.tag, this.attrs, this.content, this.start, this.end);
        }
        toString({ openTag = OPEN_BRAKET, closeTag = CLOSE_BRAKET } = {}) {
            const content = this.content ? nodeTreeToString(this.content, openTag, closeTag) : '';
            const tagStart = this.toTagStart({
                openTag,
                closeTag
            });
            if (this.content === null || Array.isArray(this.content) && this.content.length === 0) {
                return tagStart;
            }
            return `${tagStart}${content}${this.toTagEnd({
            openTag,
            closeTag
        })}`;
        }
        toJSON() {
            return {
                tag: this.tag,
                attrs: this.attrs,
                content: this.content,
                start: this.start,
                end: this.end
            };
        }
        static create(tag, attrs = {}, content = null, start) {
            return new TagNode(tag, attrs, content, start);
        }
        static isOf(node, type) {
            return node.tag === type;
        }
        constructor(tag, attrs, content, start, end){
            this.tag = tag;
            this.attrs = attrs;
            this.content = content;
            this.start = start;
            this.end = end;
        }
    }

    function process(tags, tree, core, options) {
        return tree.walk((node)=>{
            if (isTagNode(node)) {
                const tag = node.tag;
                const tagCallback = tags[tag];
                if (typeof tagCallback === "function") {
                    return tagCallback(node, core, options);
                }
            }
            return node;
        });
    }
    /**
     * Create a preset plugin for @bbob/core
     */ function createPreset(defTags, processor = process) {
        const presetFactory = (opts)=>{
            presetFactory.options = Object.assign(presetFactory.options || {}, opts);
            function presetExecutor(tree, core) {
                return processor(defTags, tree, core, presetFactory.options || {});
            }
            presetExecutor.options = presetFactory.options;
            return presetExecutor;
        };
        presetFactory.extend = function presetExtend(callback) {
            const newTags = callback(defTags, presetFactory.options);
            return createPreset(newTags, processor);
        };
        return presetFactory;
    }

    const isStartsWith = (node, type)=>node[0] === type;
    const styleAttrs = (attrs)=>{
        const values = attrs || {};
        return Object.keys(values).reduce((acc, key)=>{
            const value = values[key];
            if (typeof value === "string") {
                if (key === "color") {
                    return acc.concat(`color:${value};`);
                }
                if (key === "size") {
                    return acc.concat(`font-size:${value};`);
                }
            }
            return acc;
        }, []).join(" ");
    };
    const toListNodes = (content)=>{
        if (content && Array.isArray(content)) {
            return content.reduce((acc, node)=>{
                const listItem = acc[acc.length - 1];
                // *Entry
                if (isStringNode(node) && isStartsWith(String(node), "*")) {
                    // from '*Entry' to 'Entry'
                    const content = String(node).slice(1);
                    acc.push(TagNode.create("li", {}, [
                        content
                    ]));
                    return acc;
                }
                // { tag: '*', attrs: {}, content: [] }
                if (isTagNode(node) && TagNode.isOf(node, "*")) {
                    acc.push(TagNode.create("li", {}, []));
                    return acc;
                }
                if (!isTagNode(listItem)) {
                    acc.push(node);
                    return acc;
                }
                if (listItem && isTagNode(listItem) && Array.isArray(listItem.content)) {
                    listItem.content = listItem.content.concat(node);
                    return acc;
                }
                acc.push(node);
                return acc;
            }, []);
        }
        return content;
    };
    const renderUrl = (node, render)=>getUniqAttr(node.attrs) ? getUniqAttr(node.attrs) : render(node.content || []);
    const toNode = (tag, attrs, content)=>TagNode.create(tag, attrs, content);
    const toStyle = (style)=>({
            style
        });
    const defineStyleNode = (tag, style)=>(node)=>toNode(tag, toStyle(style), node.content);
    const defaultTags = function createTags() {
        const tags = {
            b: defineStyleNode("span", "font-weight: bold;"),
            i: defineStyleNode("span", "font-style: italic;"),
            u: defineStyleNode("span", "text-decoration: underline;"),
            s: defineStyleNode("span", "text-decoration: line-through;"),
            url: (node, { render })=>toNode("a", {
                    href: renderUrl(node, render)
                }, node.content),
            img: (node, { render })=>toNode("img", {
                    ...node.attrs,
                    src: render(node.content)
                }, null),
            quote: (node)=>toNode("blockquote", {}, [
                    toNode("p", {}, node.content)
                ]),
            code: (node)=>toNode("pre", {}, node.content),
            style: (node)=>toNode("span", toStyle(styleAttrs(node.attrs)), node.content),
            list: (node)=>{
                const type = getUniqAttr(node.attrs);
                return toNode(type ? "ol" : "ul", type ? {
                    type
                } : {}, toListNodes(node.content));
            },
            color: (node)=>toNode("span", toStyle(`color: ${getUniqAttr(node.attrs)};`), node.content)
        };
        return tags;
    }();

    var index = createPreset(defaultTags);

    exports.default = index;

    Object.defineProperty(exports, '__esModule', { value: true });

}));
