"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getSubtitles = exports.getVideoDetails = void 0;
const he_1 = __importDefault(require("he"));
const striptags_1 = __importDefault(require("striptags"));
// Universal logger that works in all environments (Node.js, Cloudflare Workers, etc.)
const createLogger = (namespace) => {
    const isDebugEnabled = () => {
        try {
            // Try to access environment variables in a safe way
            const env = typeof process !== 'undefined' && process.env ? process.env : {};
            const debugEnv = env.DEBUG || '';
            return debugEnv === '*' || debugEnv.includes(namespace);
        }
        catch (_a) {
            return false;
        }
    };
    return (message, ...args) => {
        if (isDebugEnabled()) {
            const timestamp = new Date().toISOString();
            const logMessage = `${timestamp} ${namespace} ${message}`;
            // Use console.log safely - available in all environments
            if (args.length > 0) {
                console.log(logMessage, ...args);
            }
            else {
                console.log(logMessage);
            }
        }
    };
};
const debug = createLogger('youtube-caption-extractor');
// YouTube InnerTube API configuration based on YouTube.js
const INNERTUBE_CONFIG = {
    API_BASE: 'https://www.youtube.com/youtubei/v1',
    API_KEY: 'AIzaSyAO_FJ2SlqU8Q4STEHLGCilw_Y9_11qcW8',
    CLIENT: {
        WEB: {
            NAME: 'WEB',
            VERSION: '2.20250222.10.00',
        },
        ANDROID: {
            NAME: 'ANDROID',
            VERSION: '19.35.36',
        },
    },
};
// Detect serverless environment
const isServerless = !!(process.env.VERCEL ||
    process.env.AWS_LAMBDA_FUNCTION_NAME ||
    process.env.NETLIFY ||
    process.env.CF_WORKER);
// Generate proper session data for serverless environments
function generateSessionData() {
    const visitorData = generateVisitorData();
    return {
        context: {
            client: {
                hl: 'en',
                gl: 'US',
                clientName: INNERTUBE_CONFIG.CLIENT.WEB.NAME,
                clientVersion: INNERTUBE_CONFIG.CLIENT.WEB.VERSION,
                visitorData,
            },
            user: {
                enableSafetyMode: false,
            },
            request: {
                useSsl: true,
            },
        },
        visitorData,
    };
}
// Generate visitor data (simplified version of YouTube.js approach)
function generateVisitorData() {
    // This is a simplified version - YouTube.js uses more complex protobuf encoding
    const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_';
    let result = '';
    for (let i = 0; i < 11; i++) {
        result += chars.charAt(Math.floor(Math.random() * chars.length));
    }
    return result;
}
// Enhanced fetch with proper InnerTube headers
async function fetchInnerTube(endpoint, data) {
    const headers = {
        'Content-Type': 'application/json',
        Accept: '*/*',
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/125.0.0.0 Safari/537.36',
        'X-Youtube-Client-Version': INNERTUBE_CONFIG.CLIENT.WEB.VERSION,
        'X-Youtube-Client-Name': '1', // WEB client ID
        'X-Goog-Visitor-Id': data.visitorData,
        Origin: 'https://www.youtube.com',
        Referer: 'https://www.youtube.com/',
    };
    const url = `${INNERTUBE_CONFIG.API_BASE}${endpoint}?key=${INNERTUBE_CONFIG.API_KEY}`;
    debug(`Calling InnerTube endpoint: ${endpoint}`);
    return await fetch(url, {
        method: 'POST',
        headers,
        body: JSON.stringify(data),
    });
}
// Get video info using proper InnerTube API
async function getVideoInfo(videoID) {
    var _a, _b;
    const sessionData = generateSessionData();
    const payload = {
        ...sessionData,
        videoId: videoID,
        playbackContext: {
            contentPlaybackContext: {
                vis: 0,
                splay: false,
                lactMilliseconds: '-1',
            },
        },
        racyCheckOk: true,
        contentCheckOk: true,
    };
    const response = await fetchInnerTube('/player', payload);
    if (!response.ok) {
        throw new Error(`Player API failed: ${response.status} ${response.statusText}`);
    }
    const playerData = await response.json();
    if (((_a = playerData.playabilityStatus) === null || _a === void 0 ? void 0 : _a.status) === 'LOGIN_REQUIRED') {
        debug(` LOGIN_REQUIRED status, trying next endpoint`);
        // Try the next endpoint for additional data including engagement panels
        const nextPayload = {
            ...sessionData,
            videoId: videoID,
        };
        const nextResponse = await fetchInnerTube('/next', nextPayload);
        if (!nextResponse.ok) {
            throw new Error(`Next API failed: ${nextResponse.status} ${nextResponse.statusText}`);
        }
        const nextData = await nextResponse.json();
        debug(` Next API response keys:`, Object.keys(nextData));
        return { playerData, nextData };
    }
    debug(`Player API success, status:`, (_b = playerData.playabilityStatus) === null || _b === void 0 ? void 0 : _b.status);
    return { playerData, nextData: null };
}
// Extract transcript using proper engagement panel approach (like YouTube.js)
async function getTranscriptFromEngagementPanel(videoID, nextData) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2;
    if (!(nextData === null || nextData === void 0 ? void 0 : nextData.engagementPanels)) {
        debug(` No engagement panels found`);
        return [];
    }
    debug(` Found ${nextData.engagementPanels.length} engagement panels`);
    // Find the transcript panel
    const transcriptPanel = nextData.engagementPanels.find((panel) => {
        var _a;
        return ((_a = panel === null || panel === void 0 ? void 0 : panel.engagementPanelSectionListRenderer) === null || _a === void 0 ? void 0 : _a.panelIdentifier) ===
            'engagement-panel-searchable-transcript';
    });
    if (!transcriptPanel) {
        debug(` No transcript engagement panel found`);
        return [];
    }
    debug(` Found transcript engagement panel`);
    // Extract continuation token for transcript
    const content = (_a = transcriptPanel.engagementPanelSectionListRenderer) === null || _a === void 0 ? void 0 : _a.content;
    // Extract continuation token for transcript API
    // Try multiple ways to find the continuation token
    let continuationItem;
    let token;
    // Method 1: Direct continuationItemRenderer
    continuationItem = content === null || content === void 0 ? void 0 : content.continuationItemRenderer;
    // Check for different token/params structures
    if ((_c = (_b = continuationItem === null || continuationItem === void 0 ? void 0 : continuationItem.continuationEndpoint) === null || _b === void 0 ? void 0 : _b.continuationCommand) === null || _c === void 0 ? void 0 : _c.token) {
        token = continuationItem.continuationEndpoint.continuationCommand.token;
        debug(` Found token via continuationCommand`);
    }
    else if ((_e = (_d = continuationItem === null || continuationItem === void 0 ? void 0 : continuationItem.continuationEndpoint) === null || _d === void 0 ? void 0 : _d.getTranscriptEndpoint) === null || _e === void 0 ? void 0 : _e.params) {
        token = continuationItem.continuationEndpoint.getTranscriptEndpoint.params;
        debug(` Found token via getTranscriptEndpoint`);
    }
    // Method 2: Inside sectionListRenderer
    if (!token && ((_g = (_f = content === null || content === void 0 ? void 0 : content.sectionListRenderer) === null || _f === void 0 ? void 0 : _f.contents) === null || _g === void 0 ? void 0 : _g[0])) {
        continuationItem =
            content.sectionListRenderer.contents[0].continuationItemRenderer;
        if ((_j = (_h = continuationItem === null || continuationItem === void 0 ? void 0 : continuationItem.continuationEndpoint) === null || _h === void 0 ? void 0 : _h.continuationCommand) === null || _j === void 0 ? void 0 : _j.token) {
            token = continuationItem.continuationEndpoint.continuationCommand.token;
        }
    }
    // Method 3: Look for transcriptRenderer with footer
    if (!token && ((_k = content === null || content === void 0 ? void 0 : content.sectionListRenderer) === null || _k === void 0 ? void 0 : _k.contents)) {
        for (const item of content.sectionListRenderer.contents) {
            if (item === null || item === void 0 ? void 0 : item.transcriptRenderer) {
                // Look for footer with continuation
                const footer = item.transcriptRenderer.footer;
                if ((_o = (_m = (_l = footer === null || footer === void 0 ? void 0 : footer.transcriptFooterRenderer) === null || _l === void 0 ? void 0 : _l.languageMenu) === null || _m === void 0 ? void 0 : _m.sortFilterSubMenuRenderer) === null || _o === void 0 ? void 0 : _o.subMenuItems) {
                    // Find English or first available language
                    const menuItems = footer.transcriptFooterRenderer.languageMenu
                        .sortFilterSubMenuRenderer.subMenuItems;
                    const englishItem = menuItems.find((item) => {
                        var _a;
                        return ((_a = item === null || item === void 0 ? void 0 : item.title) === null || _a === void 0 ? void 0 : _a.toLowerCase().includes('english')) ||
                            (item === null || item === void 0 ? void 0 : item.selected) === true;
                    }) || menuItems[0];
                    if ((_q = (_p = englishItem === null || englishItem === void 0 ? void 0 : englishItem.continuation) === null || _p === void 0 ? void 0 : _p.reloadContinuationData) === null || _q === void 0 ? void 0 : _q.continuation) {
                        token =
                            englishItem.continuation.reloadContinuationData.continuation;
                        break;
                    }
                }
            }
        }
    }
    if (!token) {
        debug(` No continuation token found in transcript panel`);
        return [];
    }
    debug(` Found continuation token, calling get_transcript`);
    // Call the get_transcript endpoint
    const sessionData = generateSessionData();
    const transcriptPayload = {
        ...sessionData,
        params: token,
    };
    const transcriptResponse = await fetchInnerTube('/get_transcript', transcriptPayload);
    if (!transcriptResponse.ok) {
        throw new Error(`Transcript API failed: ${transcriptResponse.status} ${transcriptResponse.statusText}`);
    }
    const transcriptData = await transcriptResponse.json();
    debug(` Transcript API response keys:`, Object.keys(transcriptData));
    // Parse transcript segments
    const segments = (_z = (_y = (_x = (_w = (_v = (_u = (_t = (_s = (_r = transcriptData === null || transcriptData === void 0 ? void 0 : transcriptData.actions) === null || _r === void 0 ? void 0 : _r[0]) === null || _s === void 0 ? void 0 : _s.updateEngagementPanelAction) === null || _t === void 0 ? void 0 : _t.content) === null || _u === void 0 ? void 0 : _u.transcriptRenderer) === null || _v === void 0 ? void 0 : _v.content) === null || _w === void 0 ? void 0 : _w.transcriptSearchPanelRenderer) === null || _x === void 0 ? void 0 : _x.body) === null || _y === void 0 ? void 0 : _y.transcriptSegmentListRenderer) === null || _z === void 0 ? void 0 : _z.initialSegments;
    if (!segments || !Array.isArray(segments)) {
        debug(` No transcript segments found`);
        return [];
    }
    debug(` Found ${segments.length} transcript segments`);
    // Successfully parsing transcript segments
    const subtitles = [];
    let debugCount = 0;
    for (const segment of segments) {
        if (segment.transcriptSegmentRenderer) {
            const renderer = segment.transcriptSegmentRenderer;
            // Extract subtitle data
            const startMs = parseInt(renderer.startMs || '0');
            const endMs = parseInt(renderer.endMs || '0');
            // Try different text extraction paths
            let text = '';
            if ((_0 = renderer.snippet) === null || _0 === void 0 ? void 0 : _0.simpleText) {
                text = renderer.snippet.simpleText;
            }
            else if ((_1 = renderer.snippet) === null || _1 === void 0 ? void 0 : _1.runs) {
                text = renderer.snippet.runs.map((run) => run.text).join('');
            }
            else if ((_2 = renderer.snippet) === null || _2 === void 0 ? void 0 : _2.text) {
                text = renderer.snippet.text;
            }
            // Log progress for first few segments
            if (debugCount < 5) {
                debug(` Segment: startMs=${startMs}, endMs=${endMs}, text="${text.substring(0, 50)}${text.length > 50 ? '...' : ''}"`);
                debugCount++;
            }
            if (text.trim()) {
                subtitles.push({
                    start: (startMs / 1000).toString(),
                    dur: ((endMs - startMs) / 1000).toString(),
                    text: he_1.default.decode((0, striptags_1.default)(text)),
                });
            }
        }
    }
    return subtitles;
}
// Fallback: Extract captions from player data (traditional method)
async function getSubtitlesFromCaptions(videoID, playerData, lang = 'en') {
    var _a, _b;
    const captionTracks = (_b = (_a = playerData === null || playerData === void 0 ? void 0 : playerData.captions) === null || _a === void 0 ? void 0 : _a.playerCaptionsTracklistRenderer) === null || _b === void 0 ? void 0 : _b.captionTracks;
    if (!captionTracks || !Array.isArray(captionTracks)) {
        debug(` No caption tracks found in player data`);
        return [];
    }
    debug(` Found ${captionTracks.length} caption tracks`);
    // Find the appropriate subtitle language track
    const subtitle = captionTracks.find((track) => track.vssId === `.${lang}`) ||
        captionTracks.find((track) => track.vssId === `a.${lang}`) ||
        captionTracks.find((track) => { var _a; return (_a = track.vssId) === null || _a === void 0 ? void 0 : _a.includes(`.${lang}`); }) ||
        captionTracks[0]; // fallback to first available
    if (!(subtitle === null || subtitle === void 0 ? void 0 : subtitle.baseUrl)) {
        debug(` No suitable caption track found`);
        return [];
    }
    debug(` Using caption track: ${subtitle.vssId}`);
    // Fetch the caption content
    const captionUrl = subtitle.baseUrl.replace('&fmt=srv3', ''); // Force XML format
    const response = await fetch(captionUrl, {
        headers: {
            'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            Referer: `https://www.youtube.com/watch?v=${videoID}`,
        },
    });
    if (!response.ok) {
        throw new Error(`Caption fetch failed: ${response.status}`);
    }
    const xmlText = await response.text();
    if (!xmlText.trim() || !xmlText.includes('<text')) {
        throw new Error('Caption content is empty or invalid');
    }
    debug(` Caption XML length: ${xmlText.length} characters`);
    // Parse XML captions
    const startRegex = /start="([\d.]+)"/;
    const durRegex = /dur="([\d.]+)"/;
    return extractSubtitlesFromXML(xmlText, startRegex, durRegex);
}
function extractSubtitlesFromXML(transcript, startRegex, durRegex) {
    return transcript
        .replace('<?xml version="1.0" encoding="utf-8" ?><transcript>', '')
        .replace('</transcript>', '')
        .split('</text>')
        .filter((line) => line && line.trim())
        .reduce((acc, line) => {
        const startResult = startRegex.exec(line);
        const durResult = durRegex.exec(line);
        if (!startResult || !durResult) {
            return acc;
        }
        const [, start] = startResult;
        const [, dur] = durResult;
        const htmlText = line
            .replace(/<text.+>/, '')
            .replace(/&amp;/gi, '&')
            .replace(/<\/?[^>]+(>|$)/g, '');
        const decodedText = he_1.default.decode(htmlText);
        const text = (0, striptags_1.default)(decodedText);
        acc.push({
            start,
            dur,
            text,
        });
        return acc;
    }, []);
}
const getVideoDetails = async ({ videoID, lang = 'en', }) => {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23, _24, _25, _26, _27, _28, _29, _30;
    try {
        debug(` Getting video details for ${videoID}, serverless: ${isServerless}`);
        // Get video info using proper InnerTube API
        const { playerData, nextData } = await getVideoInfo(videoID);
        // Extract basic video details
        const videoDetails = playerData === null || playerData === void 0 ? void 0 : playerData.videoDetails;
        // Extract title from multiple possible locations
        let title = 'No title found';
        if (videoDetails === null || videoDetails === void 0 ? void 0 : videoDetails.title) {
            title = videoDetails.title;
        }
        else if ((_k = (_j = (_h = (_g = (_f = (_e = (_d = (_c = (_b = (_a = nextData === null || nextData === void 0 ? void 0 : nextData.contents) === null || _a === void 0 ? void 0 : _a.twoColumnWatchNextResults) === null || _b === void 0 ? void 0 : _b.results) === null || _c === void 0 ? void 0 : _c.results) === null || _d === void 0 ? void 0 : _d.contents) === null || _e === void 0 ? void 0 : _e[0]) === null || _f === void 0 ? void 0 : _f.videoPrimaryInfoRenderer) === null || _g === void 0 ? void 0 : _g.title) === null || _h === void 0 ? void 0 : _h.runs) === null || _j === void 0 ? void 0 : _j[0]) === null || _k === void 0 ? void 0 : _k.text) {
            title =
                nextData.contents.twoColumnWatchNextResults.results.results.contents[0]
                    .videoPrimaryInfoRenderer.title.runs[0].text;
        }
        else if ((_o = (_m = (_l = nextData === null || nextData === void 0 ? void 0 : nextData.metadata) === null || _l === void 0 ? void 0 : _l.videoMetadataRenderer) === null || _m === void 0 ? void 0 : _m.title) === null || _o === void 0 ? void 0 : _o.simpleText) {
            title = nextData.metadata.videoMetadataRenderer.title.simpleText;
        }
        else if ((_p = nextData === null || nextData === void 0 ? void 0 : nextData.videoDetails) === null || _p === void 0 ? void 0 : _p.title) {
            title = nextData.videoDetails.title;
        }
        // Extract description from multiple possible locations
        let description = 'No description found';
        if (videoDetails === null || videoDetails === void 0 ? void 0 : videoDetails.shortDescription) {
            description = videoDetails.shortDescription;
        }
        else if ((_x = (_w = (_v = (_u = (_t = (_s = (_r = (_q = nextData === null || nextData === void 0 ? void 0 : nextData.contents) === null || _q === void 0 ? void 0 : _q.twoColumnWatchNextResults) === null || _r === void 0 ? void 0 : _r.results) === null || _s === void 0 ? void 0 : _s.results) === null || _t === void 0 ? void 0 : _t.contents) === null || _u === void 0 ? void 0 : _u[1]) === null || _v === void 0 ? void 0 : _v.videoSecondaryInfoRenderer) === null || _w === void 0 ? void 0 : _w.description) === null || _x === void 0 ? void 0 : _x.runs) {
            description =
                nextData.contents.twoColumnWatchNextResults.results.results.contents[1].videoSecondaryInfoRenderer.description.runs
                    .map((run) => run.text)
                    .join('');
        }
        else if ((_6 = (_5 = (_4 = (_3 = (_2 = (_1 = (_0 = (_z = (_y = nextData === null || nextData === void 0 ? void 0 : nextData.contents) === null || _y === void 0 ? void 0 : _y.twoColumnWatchNextResults) === null || _z === void 0 ? void 0 : _z.results) === null || _0 === void 0 ? void 0 : _0.results) === null || _1 === void 0 ? void 0 : _1.contents) === null || _2 === void 0 ? void 0 : _2[0]) === null || _3 === void 0 ? void 0 : _3.videoPrimaryInfoRenderer) === null || _4 === void 0 ? void 0 : _4.videoActions) === null || _5 === void 0 ? void 0 : _5.menuRenderer) === null || _6 === void 0 ? void 0 : _6.topLevelButtons) {
            // Look for description in primary info renderer
            const primaryInfo = nextData.contents.twoColumnWatchNextResults.results.results.contents[0]
                .videoPrimaryInfoRenderer;
            if ((_7 = primaryInfo === null || primaryInfo === void 0 ? void 0 : primaryInfo.description) === null || _7 === void 0 ? void 0 : _7.runs) {
                description = primaryInfo.description.runs
                    .map((run) => run.text)
                    .join('');
            }
        }
        else if ((_10 = (_9 = (_8 = nextData === null || nextData === void 0 ? void 0 : nextData.metadata) === null || _8 === void 0 ? void 0 : _8.videoMetadataRenderer) === null || _9 === void 0 ? void 0 : _9.description) === null || _10 === void 0 ? void 0 : _10.simpleText) {
            description =
                nextData.metadata.videoMetadataRenderer.description.simpleText;
        }
        else if ((_11 = nextData === null || nextData === void 0 ? void 0 : nextData.videoDetails) === null || _11 === void 0 ? void 0 : _11.shortDescription) {
            description = nextData.videoDetails.shortDescription;
        }
        // Additional search in the secondary info renderer with alternative path
        if (description === 'No description found' &&
            ((_15 = (_14 = (_13 = (_12 = nextData === null || nextData === void 0 ? void 0 : nextData.contents) === null || _12 === void 0 ? void 0 : _12.twoColumnWatchNextResults) === null || _13 === void 0 ? void 0 : _13.results) === null || _14 === void 0 ? void 0 : _14.results) === null || _15 === void 0 ? void 0 : _15.contents)) {
            for (const content of nextData.contents.twoColumnWatchNextResults.results
                .results.contents) {
                if ((_17 = (_16 = content === null || content === void 0 ? void 0 : content.videoSecondaryInfoRenderer) === null || _16 === void 0 ? void 0 : _16.description) === null || _17 === void 0 ? void 0 : _17.runs) {
                    description = content.videoSecondaryInfoRenderer.description.runs
                        .map((run) => run.text)
                        .join('');
                    break;
                }
                if ((_19 = (_18 = content === null || content === void 0 ? void 0 : content.videoSecondaryInfoRenderer) === null || _18 === void 0 ? void 0 : _18.attributedDescription) === null || _19 === void 0 ? void 0 : _19.content) {
                    description =
                        content.videoSecondaryInfoRenderer.attributedDescription.content;
                    break;
                }
            }
        }
        // Search in engagement panels for description
        if (description === 'No description found' && (nextData === null || nextData === void 0 ? void 0 : nextData.engagementPanels)) {
            for (const panel of nextData.engagementPanels) {
                if ((_22 = (_21 = (_20 = panel === null || panel === void 0 ? void 0 : panel.engagementPanelSectionListRenderer) === null || _20 === void 0 ? void 0 : _20.content) === null || _21 === void 0 ? void 0 : _21.structuredDescriptionContentRenderer) === null || _22 === void 0 ? void 0 : _22.items) {
                    const items = panel.engagementPanelSectionListRenderer.content
                        .structuredDescriptionContentRenderer.items;
                    for (const item of items) {
                        if ((_24 = (_23 = item === null || item === void 0 ? void 0 : item.videoDescriptionHeaderRenderer) === null || _23 === void 0 ? void 0 : _23.description) === null || _24 === void 0 ? void 0 : _24.runs) {
                            description = item.videoDescriptionHeaderRenderer.description.runs
                                .map((run) => run.text)
                                .join('');
                            break;
                        }
                        if ((_26 = (_25 = item === null || item === void 0 ? void 0 : item.expandableVideoDescriptionBodyRenderer) === null || _25 === void 0 ? void 0 : _25.descriptionBodyText) === null || _26 === void 0 ? void 0 : _26.runs) {
                            description =
                                item.expandableVideoDescriptionBodyRenderer.descriptionBodyText.runs
                                    .map((run) => run.text)
                                    .join('');
                            break;
                        }
                    }
                    if (description !== 'No description found')
                        break;
                }
            }
        }
        debug(` Video title: ${title}`);
        debug(` Video description: ${description.substring(0, 100)}${description.length > 100 ? '...' : ''}`);
        // Debug: Show available data structures for description
        if (description === 'No description found') {
            debug(` Description not found, checking available structures...`);
            if ((_30 = (_29 = (_28 = (_27 = nextData === null || nextData === void 0 ? void 0 : nextData.contents) === null || _27 === void 0 ? void 0 : _27.twoColumnWatchNextResults) === null || _28 === void 0 ? void 0 : _28.results) === null || _29 === void 0 ? void 0 : _29.results) === null || _30 === void 0 ? void 0 : _30.contents) {
                nextData.contents.twoColumnWatchNextResults.results.results.contents.forEach((content, index) => {
                    debug(` Content ${index} keys:`, Object.keys(content || {}));
                    if (content === null || content === void 0 ? void 0 : content.videoSecondaryInfoRenderer) {
                        debug(`videoSecondaryInfoRenderer keys:`, Object.keys(content.videoSecondaryInfoRenderer || {}));
                    }
                });
            }
        }
        let subtitles = [];
        // Method 1: Try to get transcript from engagement panel (preferred, like YouTube.js)
        if (nextData) {
            try {
                subtitles = await getTranscriptFromEngagementPanel(videoID, nextData);
                if (subtitles.length > 0) {
                    debug(` Successfully got ${subtitles.length} subtitles from transcript API`);
                }
            }
            catch (error) {
                debug(` Transcript API failed:`, error instanceof Error ? error.message : 'Unknown error');
            }
        }
        // Method 2: Fallback to traditional caption tracks
        if (subtitles.length === 0) {
            try {
                subtitles = await getSubtitlesFromCaptions(videoID, playerData, lang);
                if (subtitles.length > 0) {
                    debug(` Successfully got ${subtitles.length} subtitles from captions`);
                }
            }
            catch (error) {
                debug(` Caption fallback failed:`, error instanceof Error ? error.message : 'Unknown error');
            }
        }
        if (subtitles.length === 0) {
            debug(` No subtitles found for video: ${videoID} (language: ${lang})`);
        }
        return {
            title,
            description,
            subtitles,
        };
    }
    catch (error) {
        debug(`Error in getVideoDetails:`, error);
        throw error;
    }
};
exports.getVideoDetails = getVideoDetails;
const getSubtitles = async ({ videoID, lang = 'en', }) => {
    try {
        debug(` Getting subtitles for ${videoID}, serverless: ${isServerless}`);
        const { playerData, nextData } = await getVideoInfo(videoID);
        // Try transcript API first
        if (nextData) {
            try {
                const subtitles = await getTranscriptFromEngagementPanel(videoID, nextData);
                if (subtitles.length > 0) {
                    return subtitles;
                }
            }
            catch (error) {
                debug(` Transcript API failed:`, error instanceof Error ? error.message : 'Unknown error');
            }
        }
        // Fallback to captions
        return await getSubtitlesFromCaptions(videoID, playerData, lang);
    }
    catch (error) {
        debug('Error getting subtitles:', error);
        throw error;
    }
};
exports.getSubtitles = getSubtitles;
