import type { ElementHandle, Page, BoundingBox, Protocol } from 'puppeteer';
import { type Vector, type TimedVector } from './math';
export { default as installMouseHelper } from './mouse-helper';
export interface BoxOptions {
    /**
     * Percentage of padding to be added inside the element when determining the target point.
     * Example:
     * - `0` = may be anywhere within the element.
     * - `100` = will always be center of element.
     * @default 0
     */
    readonly paddingPercentage?: number;
    /**
     * Destination to move the cursor to, relative to the top-left corner of the element.
     * If specified, `paddingPercentage` is not used.
     * If not specified (default), destination is random point within the `paddingPercentage`.
     * @default undefined (random point)
     */
    readonly destination?: Vector;
}
export interface GetElementOptions {
    /**
     * Time to wait for the selector to appear in milliseconds.
     * Default is to not wait for selector.
     */
    readonly waitForSelector?: number;
}
export interface ScrollOptions {
    /**
     * Scroll speed. 0 to 100. 100 is instant.
     * @default 100
     */
    readonly scrollSpeed?: number;
    /**
   * Time to wait after scrolling.
   * @default 200
   */
    readonly scrollDelay?: number;
}
export interface ScrollIntoViewOptions extends ScrollOptions, GetElementOptions {
    /**
     * Scroll speed (when scrolling occurs). 0 to 100. 100 is instant.
     * @default 100
     */
    readonly scrollSpeed?: number;
    /**
     * Time to wait after scrolling (when scrolling occurs).
     * @default 200
     */
    readonly scrollDelay?: number;
    /**
     * Margin (in px) to add around the element when ensuring it is in the viewport.
     * (Does not take effect if CDP scroll fails.)
     * @default 0
     */
    readonly inViewportMargin?: number;
}
export interface MoveOptions extends BoxOptions, ScrollIntoViewOptions, Pick<PathOptions, 'moveSpeed'> {
    /**
     * Delay after moving the mouse in milliseconds. If `randomizeMoveDelay=true`, delay is randomized from 0 to `moveDelay`.
     * @default 0
     */
    readonly moveDelay?: number;
    /**
     * Randomize delay between actions from `0` to `moveDelay`. See `moveDelay` docs.
     * @default true
     */
    readonly randomizeMoveDelay?: boolean;
    /**
     * Maximum number of attempts to mouse-over the element.
     * @default 10
     */
    readonly maxTries?: number;
    /**
     * Distance from current location to destination that triggers overshoot to
     * occur. (Below this distance, no overshoot will occur).
     * @default 500
     */
    readonly overshootThreshold?: number;
}
export interface ClickOptions extends MoveOptions {
    /**
     * Delay before initiating the click action in milliseconds.
     * @default 0
     */
    readonly hesitate?: number;
    /**
     * Delay between mousedown and mouseup in milliseconds.
     * @default 0
     */
    readonly waitForClick?: number;
    /**
     * @default 2000
     */
    readonly moveDelay?: number;
    /**
     * @default "left"
     */
    readonly button?: Protocol.Input.MouseButton;
    /**
     * @default 1
     */
    readonly clickCount?: number;
}
export interface PathOptions {
    /**
     * Override the spread of the generated path.
     */
    readonly spreadOverride?: number;
    /**
     * Speed of mouse movement.
     * Default is random.
     */
    readonly moveSpeed?: number;
    /**
     * Generate timestamps for each point in the path.
     */
    readonly useTimestamps?: boolean;
}
export interface RandomMoveOptions extends Pick<MoveOptions, 'moveDelay' | 'randomizeMoveDelay' | 'moveSpeed'> {
    /**
     * @default 2000
     */
    readonly moveDelay?: number;
}
export interface MoveToOptions extends PathOptions, Pick<MoveOptions, 'moveDelay' | 'randomizeMoveDelay'> {
    /**
     * @default 0
     */
    readonly moveDelay?: number;
}
export type ScrollToDestination = Partial<Vector> | 'top' | 'bottom' | 'left' | 'right';
export interface GhostCursor {
    toggleRandomMove: (random: boolean) => void;
    click: (selector?: string | ElementHandle, options?: ClickOptions) => Promise<void>;
    move: (selector: string | ElementHandle, options?: MoveOptions) => Promise<void>;
    moveTo: (destination: Vector, options?: MoveToOptions) => Promise<void>;
    scrollIntoView: (selector: ElementHandle, options?: ScrollIntoViewOptions) => Promise<void>;
    scrollTo: (destination: ScrollToDestination, options?: ScrollOptions) => Promise<void>;
    scroll: (delta: Partial<Vector>, options?: ScrollOptions) => Promise<void>;
    getElement: (selector: string | ElementHandle, options?: GetElementOptions) => Promise<ElementHandle<Element>>;
    getLocation: () => Vector;
}
/** Get a random point on a browser window */
export declare const getRandomPagePoint: (page: Page) => Promise<Vector>;
export declare function path(point: Vector, target: Vector, options?: number | PathOptions): Vector[] | TimedVector[];
export declare function path(point: Vector, target: BoundingBox, options?: number | PathOptions): Vector[] | TimedVector[];
export declare const createCursor: (page: Page, 
/**
 * Cursor start position.
 * @default { x: 0, y: 0 }
 */
start?: Vector, 
/**
 * Initially perform random movements.
 * If `move`,`click`, etc. is performed, these random movements end.
 * @default false
 */
performRandomMoves?: boolean, defaultOptions?: {
    /**
     * Default options for the `randomMove` function that occurs when `performRandomMoves=true`
     * @default RandomMoveOptions
     */
    randomMove?: RandomMoveOptions;
    /**
     * Default options for the `move` function
     * @default MoveOptions
     */
    move?: MoveOptions;
    /**
     * Default options for the `moveTo` function
     * @default MoveToOptions
     */
    moveTo?: MoveToOptions;
    /**
     * Default options for the `click` function
     * @default ClickOptions
     */
    click?: ClickOptions;
    /**
    * Default options for the `scrollIntoView`, `scrollTo`, and `scroll` functions
    * @default ScrollIntoViewOptions
    */
    scroll?: ScrollOptions & ScrollIntoViewOptions;
    /**
     * Default options for the `getElement` function
     * @default GetElementOptions
     */
    getElement?: GetElementOptions;
}) => GhostCursor;
