# main.py: main application
#
# Copyright (C) 2022 Upscaler Contributors
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <https://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-only

import asyncio
import sys
from collections.abc import Awaitable
from gettext import gettext as _
from pathlib import Path
from typing import Any, Callable, ClassVar, cast

import gi
from gi.events import GLibEventLoopPolicy

from upscaler import APP_ID, APP_NAME, PROFILE
from upscaler.logger import logger

gi.require_version("Gtk", "4.0")
gi.require_version("Adw", "1")

# ruff: noqa: E402
from gi.repository import Adw, GLib, Gio, Gtk

from upscaler.window import Window


class Application(Adw.Application):
    """Main application singleton class."""

    tasks: ClassVar[set[Awaitable[None]]] = set()

    def __init__(self) -> None:
        """Initialize main application."""
        super().__init__(
            application_id=APP_ID,
            flags=Gio.ApplicationFlags.HANDLES_COMMAND_LINE,
            resource_base_path="/io/gitlab/theevilskeleton/Upscaler",
        )
        self.create_action("about", self._about_action)
        self.create_action(
            "open-output",
            self._open_output,
            param=GLib.VariantType("s"),
        )
        self.file: str | None = None

    def do_activate(self) -> None:
        """
        Call when the application is activated.

        We raise the application's main window, creating it if
        necessary.
        """
        win: Window = cast("Window", self.props.active_window)
        if not win:
            win = Window(application=self)
        win.present()

        if PROFILE == "development":
            win.add_css_class("devel")

        if self.file is not None:
            logger.info(self.file)
            task = asyncio.create_task(win.on_load_file(Path(self.file)))
            self.tasks.add(task)

    def _open_output(self, _action: Gio.SimpleAction, data: GLib.Variant) -> None:
        file_path: str = data.unpack()
        self.open_file_in_external_program(Path(file_path))

    def _about_action(self, *_args: Any) -> None:
        """Open About window."""
        about = Adw.AboutDialog.new_from_appdata(
            "/io/gitlab/theevilskeleton/Upscaler/io.gitlab.theevilskeleton.Upscaler.metainfo.xml",
            "1.6.0",
        )
        about.set_developers(developers_list())
        about.set_artists(("Frostbitten-jello https://github.com/Frostbitten-jello",))
        about.set_copyright(_("Copyright © 2022 Upscaler Contributors"))
        about.set_translator_credits(_("translator-credits"))

        ncnn = '<a href="{url}" title="{url}">Upscayl NCNN</a>'.format(
            url="https://github.com/upscayl/upscayl-ncnn",
        )
        course = '<a href="{url}" title="{url}">CS50x</a>'.format(
            url="https://www.edx.org/course/introduction-computer-science-harvardx-cs50x",
        )

        title = _("Backend")
        body_paragraph_1 = _(
            "{app_name} uses {ncnn} under the hood to upscale and enhance images. Without it, {app_name} wouldn’t exist."
        ).format(app_name=APP_NAME, ncnn=ncnn)
        body_paragraph_2 = _(
            "{app_name} started out as a final project for the final {course} assignment, Harvard University’s introductory course to computer science. However, it was later decided to continue it as free and open-source software. Without CS50x, {app_name} wouldn’t exist either."
        ).format(app_name=APP_NAME, course=course)

        about.set_comments(
            f"<b><big>{title}</big></b>\n\n{body_paragraph_1}\n\n{body_paragraph_2}"
        )

        about.add_link(_("Donate"), "https://dir.floss.fund/view/funding/@tesk.page")

        algorithms_sources = (
            "Real-ESRGAN https://github.com/xinntao/Real-ESRGAN",
            "Real-ESRGAN ncnn Vulkan https://github.com/xinntao/Real-ESRGAN-ncnn-vulkan",
            "Upscayl-NCNN https://github.com/upscayl/upscayl-ncnn",
        )
        about.add_acknowledgement_section(_("Algorithms by"), algorithms_sources)

        inspirations = (
            "Avvie https://github.com/Taiko2k/Avvie",
            "Bottles https://github.com/bottlesdevs/Bottles",
            "Loupe https://gitlab.gnome.org/BrainBlasted/loupe",
            "Totem https://gitlab.gnome.org/GNOME/totem",
            "Fractal https://gitlab.gnome.org/GNOME/fractal",
            "Builder https://gitlab.gnome.org/GNOME/gnome-builder",
        )
        about.add_acknowledgement_section(
            _("Code and Design Borrowed from"),
            inspirations,
        )

        sample_image = (
            "Princess Hinghoi https://safebooru.org/index.php?page=post&s=view&id=3084434",
            "SoyooNG https://pbs.twimg.com/media/FYH38wSaQAECj__?format=png&name=4096x4096",
            "Chiru-san https://i1.sndcdn.com/artworks-xwcv2rKIl9VgI5TD-1gZ3Eg-t500x500.jpg",
        )
        about.add_acknowledgement_section(_("Sample Image from"), sample_image)
        about.add_legal_section(
            title="Upscayl-NCNN",
            copyright=_("Copyright © 2024 Upscayl"),
            license_type=Gtk.License.AGPL_3_0,
            license=None,
        )
        about.add_legal_section(
            title="Real-ESRGAN ncnn Vulkan",
            copyright=_("Copyright © 2021 Xintao Wang"),
            license_type=Gtk.License.MIT_X11,
            license=None,
        )
        about.present(self.props.active_window)

    def open_file_in_external_program(self, path: Path, *_args: Any) -> None:
        """Open file in external program."""
        file = path.open("r")
        fid = file.fileno()
        connection = Gio.bus_get_sync(Gio.BusType.SESSION, None)
        proxy = Gio.DBusProxy.new_sync(
            connection,
            Gio.DBusProxyFlags.NONE,
            None,
            "org.freedesktop.portal.Desktop",
            "/org/freedesktop/portal/desktop",
            "org.freedesktop.portal.OpenURI",
            None,
        )

        try:
            proxy.call_with_unix_fd_list_sync(
                "OpenFile",
                GLib.Variant("(sha{sv})", ("", 0, {"ask": GLib.Variant("b", True)})),
                Gio.DBusCallFlags.NONE,
                -1,
                Gio.UnixFDList.new_from_array([fid]),
                None,
            )
        except Exception as e:
            logger.error(f"Error: {e}")

    def create_action(
        self,
        name: str,
        callback: Callable[[Gio.SimpleAction, GLib.Variant], None],
        shortcuts: tuple[str, ...] | None = None,
        param: GLib.VariantType | None = None,
    ) -> None:
        """
        Add an application action.

        Args:
            name: the name of the action
            callback: the function to be called when the action is
              activated
            shortcuts: an optional list of accelerators
            param: an optional list of parameters for the action
        """
        action = Gio.SimpleAction.new(name, param)
        action.connect("activate", callback)
        self.add_action(action)
        if shortcuts:
            self.set_accels_for_action(f"app.{name}", shortcuts)

    def do_command_line(self, command_line: Gio.ApplicationCommandLine) -> int:
        """Handle command-line interface."""
        args = command_line.get_arguments()
        if len(args) > 1:
            self.file = command_line.create_file_for_arg(args[1]).get_path()
        self.activate()
        return 0


def developers_list() -> tuple[str, ...]:
    r"""
    List of developers and contributors.

    If you have contributed code, feel free to add yourself into the Python list:

    Name only:    \nHari Rana
    Name + URL:   \nHari Rana https://tesk.page
    Name + Email: \nHari Rana <theevilskeleton@riseup.net>
    """
    return (
        "Hari Rana (TheEvilSkeleton) https://tesk.page",
        "Matteo",
        "Onuralp SEZER <thunderbirdtr@fedoraproject.org>",
        "Khaleel Al-Adhami <khaleel.aladhami@gmail.com>",
        "gregorni https://gitlab.gnome.org/gregorni",
        "Clocks https://doomsdayrs.page",
        "Zoey Ahmed https://zoey.blahaj.land",
    )


def main(_version: str) -> int:
    """Run application."""
    asyncio.set_event_loop_policy(GLibEventLoopPolicy())
    app = Application()
    return app.run(sys.argv)
