set(MINIMUM_WINDOWS_SDK_VERSION "10.0.26100.0")

set(MIDI_SDK_VERSION "${VERSION}")
set(MIDI_SDK_NUGET_URL "https://github.com/microsoft/MIDI/releases/download/preview-13/Microsoft.Windows.Devices.Midi2.${MIDI_SDK_VERSION}.nupkg")
set(MIDI_SDK_SHA512 e950cf87ec74df7b8fb8d06c1c09646f5a9f390fa1d19b9906cc79874f52310bd90a80371f9bb089f953794b05d013d602780a5905ba77aa8d8a1a6205d341d8)

message("MIDI2: MIDI SDK Version:   ${MIDI_SDK_VERSION}")
message("MIDI2: MIDI SDK NuGet URL: ${MIDI_SDK_NUGET_URL}")

# Get the MIDI SDK ---------------------------------------------------------------------------
# Grab the NuGet package from the official release location
vcpkg_download_distfile(
    MIDISDK_ARCHIVE
    URLS "${MIDI_SDK_NUGET_URL}"
    FILENAME "Microsoft.Windows.Devices.Midi2.${MIDI_SDK_VERSION}.zip"
    SHA512 ${MIDI_SDK_SHA512}
  )

# NuGet files are just zip files, so we extract it here
vcpkg_extract_source_archive(
    MIDI_SDK_EXTRACTED_FILES
    ARCHIVE "${MIDISDK_ARCHIVE}"
    NO_REMOVE_ONE_LEVEL
  )

set(MIDI2_WINRT_WINMD_SOURCE "${MIDI_SDK_EXTRACTED_FILES}/ref/native/Microsoft.Windows.Devices.Midi2.winmd")

file(INSTALL
    "${MIDI2_WINRT_WINMD_SOURCE}"
    DESTINATION "${CURRENT_PACKAGES_DIR}/share/${PORT}"
  )

set(MIDI2_WINRT_WINMD "${CURRENT_PACKAGES_DIR}/share/${PORT}/Microsoft.Windows.Devices.Midi2.winmd")


set(_MIDI2_HEADERS_ROOT_FOLDER "${CURRENT_PACKAGES_DIR}/include")

# these two bootstrapper files are included in the NuGet package, and are not generated by cppwinrt
# the sub path they are placed in is consistent with the NuGet package folder structure

set(_MIDI2_HEADERS_INIT_FOLDER "${_MIDI2_HEADERS_ROOT_FOLDER}/winmidi/init/")
set(_MIDI2_HEADERS_WINRT_FOLDER "${_MIDI2_HEADERS_ROOT_FOLDER}/winrt/")


file(INSTALL
    FILES
      "${MIDI_SDK_EXTRACTED_FILES}/build/native/include/winmidi/init/Microsoft.Windows.Devices.Midi2.Initialization.hpp"
      "${MIDI_SDK_EXTRACTED_FILES}/build/native/include/winmidi/init/WindowsMidiServicesVersion.h"
    DESTINATION "${_MIDI2_HEADERS_INIT_FOLDER}"
  )

# Find and use the latest SDK, but needs to be a minimum version as defined above.
vcpkg_get_windows_sdk(WINDOWS_SDK_VERSION)

if (WINDOWS_SDK_VERSION VERSION_GREATER_EQUAL "${MINIMUM_WINDOWS_SDK_VERSION}")
  message(STATUS "MIDI2: found Windows SDK: ${WINDOWS_SDK_VERSION}")
else()
  message(FATAL_ERROR "MIDI2: Need a Windows SDK version that is at least ${MINIMUM_WINDOWS_SDK_VERSION}")
endif()

configure_file("${CMAKE_CURRENT_LIST_DIR}/microsoft-windows-devices-midi2-config.cmake"
    "${CURRENT_PACKAGES_DIR}/share/${PORT}/${PORT}-config.cmake"
    @ONLY
  )

file(INSTALL
    "${CMAKE_CURRENT_LIST_DIR}/usage"
    DESTINATION "${CURRENT_PACKAGES_DIR}/share/${PORT}"
)

vcpkg_download_distfile(
    LICENSE_FILE
    URLS "https://github.com/microsoft/MIDI/raw/30f42326e4ec0072cb6fd846a9b1230149fd4888/LICENSE"
    FILENAME microsoft-windows-devices-midi2-LICENSE.txt
    SHA512 1d0688424f69c0e7322aeb720e4e28d9af3b5a7a2dc18b8b198156e377a61a6e05bc824528fca0f8e61ac39b137a028029ff82e5229ad400a3cc22e2bdb687ad
)

vcpkg_install_copyright(
    FILE_LIST "${LICENSE_FILE}"
)

# run the cppwinrt tool against the winmd in our extracted archive
# this requires that it was installed and configured before MIDI was configured
# We need to use the latest version that is available, from a dependency port

message(STATUS "MIDI2: Generating Microsoft.Windows.Devices.Midi2 headers.")
message(STATUS "MIDI2:   Using cppwinrt.exe:   ${CURRENT_HOST_INSTALLED_DIR}/tools/cppwinrt/cppwinrt.exe")
message(STATUS "MIDI2:   Including MIDI winmd: ${MIDI2_WINRT_WINMD}")
message(STATUS "MIDI2:   Using Windows SDK:    ${WINDOWS_SDK_VERSION}")

set(MIDI2_TEMP_HEADER_FOLDER "${CURRENT_PACKAGES_DIR}/midi2_temp")

# this will generate projection headers to the "winrt" subfolder of the provided output folder
vcpkg_execute_required_process(
    COMMAND "${CURRENT_HOST_INSTALLED_DIR}/tools/cppwinrt/cppwinrt.exe"
        -include "Microsoft.Windows.Devices.Midi2"
        -exclude "Windows."
        -reference "${WINDOWS_SDK_VERSION}"
        -reference "${MIDI2_WINRT_WINMD}"
        -output "${MIDI2_TEMP_HEADER_FOLDER}"
        -overwrite
        -optimize
        -verbose
    WORKING_DIRECTORY "${CURRENT_PACKAGES_DIR}"
    LOGNAME "Microsoft.Windows.Devices.Midi2.cppwinrt"
)

# the cppwinrt tool always generates Windows.* headers even when you ask to exclude them. So we need
# to copy only the Midi2 headers to ${CURRENT_PACKAGES_DIR}/include
# Consuming projects expect all winrt headers, system and external, to be in the same include folder
# and installing and configuring cppwinrt package automatically generates the projection headers
# for the system Windows.* types.

file(INSTALL
    "${MIDI2_TEMP_HEADER_FOLDER}/winrt"
    DESTINATION "${CURRENT_PACKAGES_DIR}/include"
    FILES_MATCHING PATTERN "Microsoft.Windows.Devices.Midi2*.h"
)

file(REMOVE_RECURSE "${MIDI2_TEMP_HEADER_FOLDER}")
