-- Copyright 2006-2025 Mitchell. See LICENSE.
-- CMake LPeg lexer.

local lexer = lexer
local P, S = lpeg.P, lpeg.S

local lex = lexer.new(..., {case_insensitive_fold_points = true})

-- Commands.
local word = (lexer.alpha + S('_-')) * (lexer.alnum + S('_-'))^0
local builtin_command = lex:tag(lexer.FUNCTION_BUILTIN, lex:word_match('command', true))
local command = lex:tag(lexer.FUNCTION, word)
lex:add_rule('command', (builtin_command + command) * #P('('))

-- Constants.
local constant = lex:word_match(lexer.CONSTANT_BUILTIN, true)
local module = lex:word_match('module')
lex:add_rule('constant', lex:tag(lexer.CONSTANT_BUILTIN, constant + module))

-- Variables.
local builtin_var = lex:word_match(lexer.VARIABLE_BUILTIN)
local expansion_var = (P('CACHE') + 'ENV') * #P('{')
lex:add_rule('variable', lex:tag(lexer.VARIABLE_BUILTIN, builtin_var + expansion_var))

-- Generator expressions.
lex:add_rule('generator',
	lex:tag(lexer.PREPROCESSOR, lpeg.B('$<') * lex:word_match(lexer.PREPROCESSOR)))

-- Operators.
lex:add_rule('operator',
	lex:tag(lexer.OPERATOR, lex:word_match(lexer.OPERATOR, true) + S('=()${}<>')))

-- Identifiers.
lex:add_rule('identifier', lex:tag(lexer.IDENTIFIER, word))

-- Strings.
local bracket = lpeg.Cmt('[' * lpeg.C(P('=')^0) * '[', function(input, index, eq)
	local _, e = input:find(']' .. eq .. ']', index, true)
	return (e or #input) + 1
end)
local quoted = lexer.range('"')
lex:add_rule('string', lex:tag(lexer.STRING, bracket + quoted))

-- Comments.
local line_comment = lexer.to_eol('#')
local bracket_comment = '#' * bracket
lex:add_rule('comment', lex:tag(lexer.COMMENT, bracket_comment + line_comment))

-- Fold points.
lex:add_fold_point(lexer.FUNCTION_BUILTIN, 'if', 'endif')
lex:add_fold_point(lexer.FUNCTION_BUILTIN, 'foreach', 'endforeach')
lex:add_fold_point(lexer.FUNCTION_BUILTIN, 'while', 'endwhile')
lex:add_fold_point(lexer.FUNCTION_BUILTIN, 'macro', 'endmacro')
lex:add_fold_point(lexer.FUNCTION_BUILTIN, 'function', 'endfunction')
lex:add_fold_point(lexer.OPERATOR, '(', ')')

-- Word lists.
lex:set_word_list('command', {
	-- Scripting commands.
	'block', 'break', 'cmake_host_system_information', 'cmake_language', 'cmake_minimum_required',
	'cmake_parse_arguments', 'cmake_path', 'cmake_policy', 'configure_file', 'continue', 'else',
	'elseif', 'endblock', 'endforeach', 'endfunction', 'endif', 'endmacro', 'endwhile',
	'execute_process', 'file', 'find_file', 'find_library', 'find_package', 'find_path',
	'find_program', 'foreach', 'function', 'get_cmake_property', 'get_directory_property',
	'get_filename_component', 'get_property', 'if', 'include', 'include_guard', 'list', 'macro',
	'mark_as_advanced', 'math', 'message', 'option', 'return', 'separate_arguments', 'set',
	'set_directory_properties', 'set_property', 'site_name', 'string', 'unset', 'variable_watch',
	'while',
	-- Project commands.
	'add_compile_definitions', 'add_compile_options', 'add_custom_command', 'add_custom_target',
	'add_definitions', 'add_dependencies', 'add_executable', 'add_library', 'add_link_options',
	'add_subdirectory', 'add_test', 'aux_source_directory', 'build_command', 'create_test_sourcelist',
	'define_property', 'enable_language', 'enable_testing', 'export', 'fltk_wrap_ui',
	'get_source_file_property', 'get_target_property', 'get_test_property', 'include_directories',
	'include_external_msproject', 'include_regular_expression', 'install', 'link_directories',
	'link_libraries', 'load_cache', 'project', 'remove_definitions', 'set_source_files_properties',
	'set_target_properties', 'set_tests_properties', 'source_group', 'target_compile_definitions',
	'target_compile_features', 'target_compile_options', 'target_include_directories',
	'target_link_directories', 'target_link_libraries', 'target_link_options',
	'target_precompile_headers', 'target_sources', 'try_compile', 'try_run',
	-- CTest commands.
	'ctest_build', 'ctest_configure', 'ctest_coverage', 'ctest_empty_binary_directory',
	'ctest_memcheck', 'ctest_read_custom_files', 'ctest_run_script', 'ctest_sleep', 'ctest_start',
	'ctest_submit', 'ctest_test', 'ctest_update', 'ctest_upload'
})

lex:set_word_list(lexer.CONSTANT_BUILTIN, 'on yes true y off no false n ignore notfound')

lex:set_word_list('module', {
	'AndroidTestUtilities', 'BundleUtilities', 'CheckCCompilerFlag', 'CheckCompilerFlag',
	'CheckCSourceCompiles', 'CheckCSourceRuns', 'CheckCXXCompilerFlag', 'CheckCXXSourceCompiles',
	'CheckCXXSourceRuns', 'CheckCXXSymbolExists', 'CheckFortranCompilerFlag',
	'CheckFortranFunctionExists', 'CheckFortranSourceCompiles', 'CheckFortranSourceRuns',
	'CheckFunctionExists', 'CheckIncludeFileCXX', 'CheckIncludeFile', 'CheckIncludeFiles',
	'CheckIPOSupported', 'CheckLanguage', 'CheckLibraryExists', 'CheckLinkerFlag',
	'CheckOBJCCompilerFlag', 'CheckOBJCSourceCompiles', 'CheckOBJCSourceRuns',
	'CheckOBJCXXCompilerFlag', 'CheckOBJCXXSourceCompiles', 'CheckOBJCXXSourceRuns',
	'CheckPIESupported', 'CheckPrototypeDefinition', 'CheckSourceCompiles', 'CheckSourceRuns',
	'CheckStructHasMember', 'CheckSymbolExists', 'CheckTypeSize', 'CheckVariableExists',
	'CMakeAddFortranSubdirectory', 'CMakeBackwardCompatibilityCXX', 'CMakeDependentOption',
	'CMakeFindDependencyMacro', 'CMakeFindFrameworks', 'CMakeFindPackageMode', 'CMakeGraphVizOptions',
	'CMakePackageConfigHelpers', 'CMakePrintHelpers', 'CMakePrintSystemInformation',
	'CMakePushCheckState', 'CMakeVerifyManifest', 'CPack', 'CPackComponent', 'CPackIFW',
	'CPackIFWConfigureFile', 'CSharpUtilities', 'CTest', 'CTestCoverageCollectGCOV',
	'CTestScriptMode', 'CTestUseLaunchers', 'Dart', 'DeployQt4', 'ExternalData', 'ExternalProject',
	'FeatureSummary', 'FetchContent', 'FindPackageHandleStandardArgs', 'FindPackageMessage',
	'FortranCInterface', 'GenerateExportHeader', 'GetPrerequisites', 'GNUInstallDirs', 'GoogleTest',
	'InstallRequiredSystemLibraries', 'ProcessorCount', 'SelectLibraryConfigurations',
	'SquishTestScript', 'TestBigEndian', 'TestForANSIForScope', 'TestForANSIStreamHeaders',
	'TestForSSTREAM', 'TestForSTDNamespace', 'UseEcos', 'UseJava', 'UseSWIG', 'UsewxWidgets'
})

lex:set_word_list(lexer.VARIABLE_BUILTIN, {
	-- Variables that provide information.
	'CMAKE_AR', 'CMAKE_ARGC', 'CMAKE_ARGV0', 'CMAKE_BINARY_DIR', 'CMAKE_BUILD_TOOL',
	'CMAKE_CACHE_MAJOR_VERSION', 'CMAKE_CACHE_MINOR_VERSION', 'CMAKE_CACHE_PATCH_VERSION',
	'CMAKE_CACHEFILE_DIR', 'CMAKE_CFG_INTDIR', 'CMAKE_COMMAND', 'CMAKE_CPACK_COMMAND',
	'CMAKE_CROSSCOMPILING', 'CMAKE_CROSSCOMPILING_EMULATOR', 'CMAKE_CTEST_COMMAND',
	'CMAKE_CURRENT_BINARY_DIR', 'CMAKE_CURRENT_FUNCTION', 'CMAKE_CURRENT_FUNCTION_LIST_DIR',
	'CMAKE_CURRENT_FUNCTION_LIST_FILE', 'CMAKE_CURRENT_FUNCTION_LIST_LINE', 'CMAKE_CURRENT_LIST_DIR',
	'CMAKE_CURRENT_LIST_FILE', 'CMAKE_CURRENT_LIST_LINE', 'CMAKE_CURRENT_SOURCE_DIR',
	'CMAKE_DEBUG_TARGET_PROPERTIES', 'CMAKE_DIRECTORY_LABELS', 'CMAKE_DL_LIBS', 'CMAKE_DOTNET_SDK',
	'CMAKE_DOTNET_TARGET_FRAMEWORK', 'CMAKE_DOTNET_TARGET_FRAMEWORK_VERSION', 'CMAKE_EDIT_COMMAND',
	'CMAKE_EXECUTABLE_SUFFIX',
	-- 'CMAKE_EXECUTABLE_SUFFIX_<LANG>',
	'CMAKE_EXTRA_GENERATOR', 'CMAKE_EXTRA_SHARED_LIBRARY_SUFFIXES', 'CMAKE_FIND_DEBUG_MODE',
	'CMAKE_FIND_PACKAGE_NAME', 'CMAKE_FIND_PACKAGE_REDIRECTS_DIR',
	'CMAKE_FIND_PACKAGE_SORT_DIRECTION', 'CMAKE_FIND_PACKAGE_SORT_ORDER', 'CMAKE_GENERATOR',
	'CMAKE_GENERATOR_INSTANCE', 'CMAKE_GENERATOR_PLATFORM', 'CMAKE_GENERATOR_TOOLSET',
	'CMAKE_IMPORT_LIBRARY_PREFIX', 'CMAKE_IMPORT_LIBRARY_SUFFIX', 'CMAKE_JOB_POOL_COMPILE',
	'CMAKE_JOB_POOL_LINK', 'CMAKE_JOB_POOL_PRECOMPILE_HEADER', 'CMAKE_JOB_POOLS',
	-- 'CMAKE_<LANG>_COMPILER_AR',
	-- 'CMAKE_<LANG>_COMPILER_FRONTEND_VARIANT',
	-- 'CMAKE_<LANG>_COMPILER_RANLIB',
	-- 'CMAKE_<LANG>_LINK_LIBRARY_SUFFIX',
	'CMAKE_LINK_LIBRARY_SUFFIX', 'CMAKE_LINK_SEARCH_END_STATIC', 'CMAKE_LINK_SEARCH_START_STATIC',
	'CMAKE_MAJOR_VERSION', 'CMAKE_MAKE_PROGRAM', 'CMAKE_MATCH_COUNT',
	-- 'CMAKE_MATCH_<n>',
	'CMAKE_MINIMUM_REQUIRED_VERSION', 'CMAKE_MINOR_VERSION', 'CMAKE_NETRC', 'CMAKE_NETRC_FILE',
	'CMAKE_PARENT_LIST_FILE', 'CMAKE_PATCH_VERSION', 'CMAKE_PROJECT_DESCRIPTION',
	'CMAKE_PROJECT_HOMEPAGE_URL', 'CMAKE_PROJECT_NAME', 'CMAKE_PROJECT_VERSION',
	'CMAKE_PROJECT_VERSION_MAJOR', 'CMAKE_PROJECT_VERSION_MINOR', 'CMAKE_PROJECT_VERSION_PATCH',
	'CMAKE_PROJECT_VERSION_TWEAK', 'CMAKE_RANLIB', 'CMAKE_ROOT', 'CMAKE_RULE_MESSAGES',
	'CMAKE_SCRIPT_MODE_FILE', 'CMAKE_SHARED_LIBRARY_PREFIX', 'CMAKE_SHARED_LIBRARY_SUFFIX',
	'CMAKE_SHARED_MODULE_PREFIX', 'CMAKE_SHARED_MODULE_SUFFIX', 'CMAKE_SIZEOF_VOID_P',
	'CMAKE_SKIP_INSTALL_RULES', 'CMAKE_SKIP_RPATH', 'CMAKE_SOURCE_DIR', 'CMAKE_STATIC_LIBRARY_PREFIX',
	'CMAKE_STATIC_LIBRARY_SUFFIX', 'CMAKE_Swift_MODULE_DIRECTORY', 'CMAKE_Swift_NUM_THREADS',
	'CMAKE_TOOLCHAIN_FILE', 'CMAKE_TWEAK_VERSION', 'CMAKE_VERBOSE_MAKEFILE', 'CMAKE_VERSION',
	'CMAKE_VS_DEVENV_COMMAND', 'CMAKE_VS_MSBUILD_COMMAND', 'CMAKE_VS_NsightTegra_VERSION',
	'CMAKE_VS_NUGET_PACKAGE_RESTORE', 'CMAKE_VS_PLATFORM_NAME', 'CMAKE_VS_PLATFORM_NAME_DEFAULT',
	'CMAKE_VS_PLATFORM_TOOLSET', 'CMAKE_VS_PLATFORM_TOOLSET_CUDA',
	'CMAKE_VS_PLATFORM_TOOLSET_CUDA_CUSTOM_DIR', 'CMAKE_VS_PLATFORM_TOOLSET_HOST_ARCHITECTURE',
	'CMAKE_VS_PLATFORM_TOOLSET_VERSION', 'CMAKE_VS_TARGET_FRAMEWORK_IDENTIFIER',
	'CMAKE_VS_TARGET_FRAMEWORK_TARGETS_VERSION', 'CMAKE_VS_TARGET_FRAMEWORK_VERSION',
	'CMAKE_VS_WINDOWS_TARGET_PLATFORM_VERSION', 'CMAKE_VS_WINDOWS_TARGET_PLATFORM_VERSION_MAXIMUM',
	'CMAKE_XCODE_BUILD_SYSTEM', 'CMAKE_XCODE_PLATFORM_TOOLSET',
	-- '<PROJECT-NAME>_BINARY_DIR',
	-- '<PROJECT-NAME>_DESCRIPTION',
	-- '<PROJECT-NAME>_HOMEPAGE_URL',
	-- '<PROJECT-NAME>_IS_TOP_LEVEL',
	-- '<PROJECT-NAME>_SOURCE_DIR',
	-- '<PROJECT-NAME>_VERSION',
	-- '<PROJECT-NAME>_VERSION_MAJOR',
	-- '<PROJECT-NAME>_VERSION_MINOR',
	-- '<PROJECT-NAME>_VERSION_PATCH',
	-- '<PROJECT-NAME>_VERSION_TWEAK',
	'PROJECT_BINARY_DIR', 'PROJECT_DESCRIPTION', 'PROJECT_HOMEPAGE_URL', 'PROJECT_IS_TOP_LEVEL',
	'PROJECT_NAME', 'PROJECT_SOURCE_DIR', 'PROJECT_VERSION', 'PROJECT_VERSION_MAJOR',
	'PROJECT_VERSION_MINOR', 'PROJECT_VERSION_PATCH', 'PROJECT_VERSION_TWEAK',

	-- Variables that change behavior.
	'BUILD_SHARED_LIBS', 'CMAKE_ABSOLUTE_DESTINATION_FILES', 'CMAKE_APPBUNDLE_PATH',
	'CMAKE_AUTOMOC_RELAXED_MODE', 'CMAKE_BACKWARDS_COMPATIBILITY', 'CMAKE_BUILD_TYPE',
	'CMAKE_CLANG_VFS_OVERLAY', 'CMAKE_CODEBLOCKS_COMPILER_ID',
	'CMAKE_CODEBLOCKS_EXCLUDE_EXTERNAL_FILES', 'CMAKE_CODELITE_USE_TARGETS',
	'CMAKE_COLOR_DIAGNOSTICS', 'CMAKE_COLOR_MAKEFILE', 'CMAKE_CONFIGURATION_TYPES',
	'CMAKE_DEPENDS_IN_PROJECT_ONLY',
	-- 'CMAKE_DISABLE_FIND_PACKAGE_<PackageName>',
	'CMAKE_ECLIPSE_GENERATE_LINKED_RESOURCES', 'CMAKE_ECLIPSE_GENERATE_SOURCE_PROJECT',
	'CMAKE_ECLIPSE_MAKE_ARGUMENTS', 'CMAKE_ECLIPSE_RESOURCE_ENCODING', 'CMAKE_ECLIPSE_VERSION',
	'CMAKE_ERROR_DEPRECATED', 'CMAKE_ERROR_ON_ABSOLUTE_INSTALL_DESTINATION',
	'CMAKE_EXECUTE_PROCESS_COMMAND_ECHO', 'CMAKE_EXPORT_COMPILE_COMMANDS',
	'CMAKE_EXPORT_PACKAGE_REGISTRY', 'CMAKE_EXPORT_NO_PACKAGE_REGISTRY', 'CMAKE_FIND_APPBUNDLE',
	'CMAKE_FIND_FRAMEWORK', 'CMAKE_FIND_LIBRARY_CUSTOM_LIB_SUFFIX', 'CMAKE_FIND_LIBRARY_PREFIXES',
	'CMAKE_FIND_LIBRARY_SUFFIXES', 'CMAKE_FIND_NO_INSTALL_PREFIX',
	'CMAKE_FIND_PACKAGE_NO_PACKAGE_REGISTRY', 'CMAKE_FIND_PACKAGE_NO_SYSTEM_PACKAGE_REGISTRY',
	'CMAKE_FIND_PACKAGE_PREFER_CONFIG', 'CMAKE_FIND_PACKAGE_RESOLVE_SYMLINKS',
	'CMAKE_FIND_PACKAGE_TARGETS_GLOBAL', 'CMAKE_FIND_PACKAGE_WARN_NO_MODULE', 'CMAKE_FIND_ROOT_PATH',
	'CMAKE_FIND_ROOT_PATH_MODE_INCLUDE', 'CMAKE_FIND_ROOT_PATH_MODE_LIBRARY',
	'CMAKE_FIND_ROOT_PATH_MODE_PACKAGE', 'CMAKE_FIND_ROOT_PATH_MODE_PROGRAM',
	'CMAKE_FIND_USE_CMAKE_ENVIRONMENT_PATH', 'CMAKE_FIND_USE_CMAKE_PATH',
	'CMAKE_FIND_USE_CMAKE_SYSTEM_PATH', 'CMAKE_FIND_USE_INSTALL_PREFIX',
	'CMAKE_FIND_USE_PACKAGE_REGISTRY', 'CMAKE_FIND_USE_PACKAGE_ROOT_PATH',
	'CMAKE_FIND_USE_SYSTEM_ENVIRONMENT_PATH', 'CMAKE_FIND_USE_SYSTEM_PACKAGE_REGISTRY',
	'CMAKE_FRAMEWORK_PATH', 'CMAKE_IGNORE_PATH', 'CMAKE_IGNORE_PREFIX_PATH',
	'CMAKE_INCLUDE_DIRECTORIES_BEFORE', 'CMAKE_INCLUDE_DIRECTORIES_PROJECT_BEFORE',
	'CMAKE_INCLUDE_PATH', 'CMAKE_INSTALL_DEFAULT_COMPONENT_NAME',
	'CMAKE_INSTALL_DEFAULT_DIRECTORY_PERMISSIONS', 'CMAKE_INSTALL_MESSAGE', 'CMAKE_INSTALL_PREFIX',
	'CMAKE_INSTALL_PREFIX_INITIALIZED_TO_DEFAULT', 'CMAKE_LIBRARY_PATH',
	'CMAKE_LINK_DIRECTORIES_BEFORE', 'CMAKE_LINK_LIBRARIES_ONLY_TARGETS',
	'CMAKE_MAXIMUM_RECURSION_DEPTH', 'CMAKE_MESSAGE_CONTEXT', 'CMAKE_MESSAGE_CONTEXT_SHOW',
	'CMAKE_MESSAGE_INDENT', 'CMAKE_MESSAGE_LOG_LEVEL', 'CMAKE_MFC_FLAG', 'CMAKE_MODULE_PATH',
	-- 'CMAKE_POLICY_DEFAULT_CMP<NNNN>',
	-- 'CMAKE_POLICY_WARNING_CMP<NNNN>',
	'CMAKE_PREFIX_PATH', 'CMAKE_PROGRAM_PATH', 'CMAKE_PROJECT_INCLUDE',
	'CMAKE_PROJECT_INCLUDE_BEFORE',
	-- 'CMAKE_PROJECT_<PROJECT-NAME>_INCLUDE',
	-- 'CMAKE_PROJECT_<PROJECT-NAME>_INCLUDE_BEFORE',
	'CMAKE_PROJECT_TOP_LEVEL_INCLUDES',
	-- 'CMAKE_REQUIRE_FIND_PACKAGE_<PackageName>',
	'CMAKE_SKIP_INSTALL_ALL_DEPENDENCY', 'CMAKE_STAGING_PREFIX', 'CMAKE_SUBLIME_TEXT_2_ENV_SETTINGS',
	'CMAKE_SUBLIME_TEXT_2_EXCLUDE_BUILD_TREE', 'CMAKE_SUPPRESS_REGENERATION', 'CMAKE_SYSROOT',
	'CMAKE_SYSROOT_COMPILE', 'CMAKE_SYSROOT_LINK', 'CMAKE_SYSTEM_APPBUNDLE_PATH',
	'CMAKE_SYSTEM_FRAMEWORK_PATH', 'CMAKE_SYSTEM_IGNORE_PATH', 'CMAKE_SYSTEM_IGNORE_PREFIX_PATH',
	'CMAKE_SYSTEM_INCLUDE_PATH', 'CMAKE_SYSTEM_LIBRARY_PATH', 'CMAKE_SYSTEM_PREFIX_PATH',
	'CMAKE_SYSTEM_PROGRAM_PATH', 'CMAKE_TLS_CAINFO', 'CMAKE_TLS_VERIFY',
	'CMAKE_USER_MAKE_RULES_OVERRIDE', 'CMAKE_WARN_DEPRECATED',
	'CMAKE_WARN_ON_ABSOLUTE_INSTALL_DESTINATION', 'CMAKE_XCODE_GENERATE_SCHEME',
	'CMAKE_XCODE_GENERATE_TOP_LEVEL_PROJECT_ONLY', 'CMAKE_XCODE_LINK_BUILD_PHASE_MODE',
	'CMAKE_XCODE_SCHEME_ADDRESS_SANITIZER', 'CMAKE_XCODE_SCHEME_ADDRESS_SANITIZER_USE_AFTER_RETURN',
	'CMAKE_XCODE_SCHEME_DEBUG_DOCUMENT_VERSIONING', 'CMAKE_XCODE_SCHEME_DISABLE_MAIN_THREAD_CHECKER',
	'CMAKE_XCODE_SCHEME_DYNAMIC_LIBRARY_LOADS', 'CMAKE_XCODE_SCHEME_DYNAMIC_LINKER_API_USAGE',
	'CMAKE_XCODE_SCHEME_ENABLE_GPU_API_VALIDATION',
	'CMAKE_XCODE_SCHEME_ENABLE_GPU_FRAME_CAPTURE_MODE',
	'CMAKE_XCODE_SCHEME_ENABLE_GPU_SHADER_VALIDATION', 'CMAKE_XCODE_SCHEME_ENVIRONMENT',
	'CMAKE_XCODE_SCHEME_GUARD_MALLOC', 'CMAKE_XCODE_SCHEME_LAUNCH_CONFIGURATION',
	'CMAKE_XCODE_SCHEME_LAUNCH_MODE', 'CMAKE_XCODE_SCHEME_MAIN_THREAD_CHECKER_STOP',
	'CMAKE_XCODE_SCHEME_MALLOC_GUARD_EDGES', 'CMAKE_XCODE_SCHEME_MALLOC_SCRIBBLE',
	'CMAKE_XCODE_SCHEME_MALLOC_STACK', 'CMAKE_XCODE_SCHEME_THREAD_SANITIZER',
	'CMAKE_XCODE_SCHEME_THREAD_SANITIZER_STOP', 'CMAKE_XCODE_SCHEME_UNDEFINED_BEHAVIOUR_SANITIZER',
	'CMAKE_XCODE_SCHEME_UNDEFINED_BEHAVIOUR_SANITIZER_STOP', 'CMAKE_XCODE_SCHEME_WORKING_DIRECTORY',
	'CMAKE_XCODE_SCHEME_ZOMBIE_OBJECTS', 'CMAKE_XCODE_XCCONFIG',
	-- '<PackageName>_ROOT',
	
	-- Variables that describe the system.
	'ANDROID', 'APPLE', 'BORLAND', 'BSD', 'CMAKE_ANDROID_NDK_VERSION', 'CMAKE_CL_64',
	'CMAKE_COMPILER_2005', 'CMAKE_HOST_APPLE', 'CMAKE_HOST_BSD', 'CMAKE_HOST_LINUX',
	'CMAKE_HOST_SOLARIS', 'CMAKE_HOST_SYSTEM', 'CMAKE_HOST_SYSTEM_NAME',
	'CMAKE_HOST_SYSTEM_PROCESSOR', 'CMAKE_HOST_SYSTEM_VERSION', 'CMAKE_HOST_UNIX', 'CMAKE_HOST_WIN32',
	'CMAKE_LIBRARY_ARCHITECTURE', 'CMAKE_LIBRARY_ARCHITECTURE_REGEX', 'CMAKE_OBJECT_PATH_MAX',
	'CMAKE_SYSTEM', 'CMAKE_SYSTEM_NAME', 'CMAKE_SYSTEM_PROCESSOR', 'CMAKE_SYSTEM_VERSION', 'CYGWIN',
	'GHSMULTI', 'IOS', 'LINUX', 'MINGW', 'MSVC', 'MSVC10', 'MSVC11', 'MSVC12', 'MSVC14', 'MSVC60',
	'MSVC70', 'MSVC71', 'MSVC80', 'MSVC90', 'MSVC_IDE', 'MSVC_TOOLSET_VERSION', 'MSVC_VERSION',
	'MSYS', 'UNIX', 'WIN32', 'WINCE', 'WINDOWS_PHONE', 'WINDOWS_STORE', 'XCODE', 'XCODE_VERSION',

	-- Variables that control the build.
	'CMAKE_ADSP_ROOT', 'CMAKE_AIX_EXPORT_ALL_SYMBOLS', 'CMAKE_ANDROID_ANT_ADDITIONAL_OPTIONS',
	'CMAKE_ANDROID_API', 'CMAKE_ANDROID_API_MIN', 'CMAKE_ANDROID_ARCH', 'CMAKE_ANDROID_ARCH_ABI',
	'CMAKE_ANDROID_ARM_MODE', 'CMAKE_ANDROID_ARM_NEON', 'CMAKE_ANDROID_ASSETS_DIRECTORIES',
	'CMAKE_ANDROID_EXCEPTIONS', 'CMAKE_ANDROID_GUI', 'CMAKE_ANDROID_JAR_DEPENDENCIES',
	'CMAKE_ANDROID_JAR_DIRECTORIES', 'CMAKE_ANDROID_JAVA_SOURCE_DIR',
	'CMAKE_ANDROID_NATIVE_LIB_DEPENDENCIES', 'CMAKE_ANDROID_NATIVE_LIB_DIRECTORIES',
	'CMAKE_ANDROID_NDK', 'CMAKE_ANDROID_NDK_DEPRECATED_HEADERS',
	'CMAKE_ANDROID_NDK_TOOLCHAIN_HOST_TAG', 'CMAKE_ANDROID_NDK_TOOLCHAIN_VERSION',
	'CMAKE_ANDROID_PROCESS_MAX', 'CMAKE_ANDROID_PROGUARD', 'CMAKE_ANDROID_PROGUARD_CONFIG_PATH',
	'CMAKE_ANDROID_RTTI', 'CMAKE_ANDROID_SECURE_PROPS_PATH', 'CMAKE_ANDROID_SKIP_ANT_STEP',
	'CMAKE_ANDROID_STANDALONE_TOOLCHAIN', 'CMAKE_ANDROID_STL_TYPE', 'CMAKE_APPLE_SILICON_PROCESSOR',
	'CMAKE_ARCHIVE_OUTPUT_DIRECTORY',
	-- 'CMAKE_ARCHIVE_OUTPUT_DIRECTORY_<CONFIG>',
	'CMAKE_AUTOGEN_ORIGIN_DEPENDS', 'CMAKE_AUTOGEN_PARALLEL', 'CMAKE_AUTOGEN_VERBOSE',
	'CMAKE_AUTOMOC', 'CMAKE_AUTOMOC_COMPILER_PREDEFINES', 'CMAKE_AUTOMOC_DEPEND_FILTERS',
	'CMAKE_AUTOMOC_MACRO_NAMES', 'CMAKE_AUTOMOC_MOC_OPTIONS', 'CMAKE_AUTOMOC_PATH_PREFIX',
	'CMAKE_AUTORCC', 'CMAKE_AUTORCC_OPTIONS', 'CMAKE_AUTOUIC', 'CMAKE_AUTOUIC_OPTIONS',
	'CMAKE_AUTOUIC_SEARCH_PATHS', 'CMAKE_BUILD_RPATH', 'CMAKE_BUILD_RPATH_USE_ORIGIN',
	'CMAKE_BUILD_WITH_INSTALL_NAME_DIR', 'CMAKE_BUILD_WITH_INSTALL_RPATH',
	'CMAKE_COMPILE_PDB_OUTPUT_DIRECTORY',
	-- 'CMAKE_COMPILE_PDB_OUTPUT_DIRECTORY_<CONFIG>',
	'CMAKE_COMPILE_WARNING_AS_ERROR',
	-- 'CMAKE_<CONFIG>_POSTFIX',
	'CMAKE_CROSS_CONFIGS', 'CMAKE_CTEST_ARGUMENTS', 'CMAKE_CUDA_RESOLVE_DEVICE_SYMBOLS',
	'CMAKE_CUDA_RUNTIME_LIBRARY', 'CMAKE_CUDA_SEPARABLE_COMPILATION', 'CMAKE_DEBUG_POSTFIX',
	'CMAKE_DEFAULT_BUILD_TYPE', 'CMAKE_DEFAULT_CONFIGS', 'CMAKE_DEPENDS_USE_COMPILER',
	'CMAKE_DISABLE_PRECOMPILE_HEADERS', 'CMAKE_ENABLE_EXPORTS', 'CMAKE_EXE_LINKER_FLAGS',
	-- 'CMAKE_EXE_LINKER_FLAGS_<CONFIG>',
	-- 'CMAKE_EXE_LINKER_FLAGS_<CONFIG>_INIT',
	'CMAKE_EXE_LINKER_FLAGS_INIT', 'CMAKE_FOLDER', 'CMAKE_Fortran_FORMAT',
	'CMAKE_Fortran_MODULE_DIRECTORY', 'CMAKE_Fortran_PREPROCESS', 'CMAKE_FRAMEWORK',
	-- 'CMAKE_FRAMEWORK_MULTI_CONFIG_POSTFIX_<CONFIG>',
	'CMAKE_GHS_NO_SOURCE_GROUP_FILE', 'CMAKE_GLOBAL_AUTOGEN_TARGET',
	'CMAKE_GLOBAL_AUTOGEN_TARGET_NAME', 'CMAKE_GLOBAL_AUTORCC_TARGET',
	'CMAKE_GLOBAL_AUTORCC_TARGET_NAME', 'CMAKE_GNUtoMS', 'CMAKE_INCLUDE_CURRENT_DIR',
	'CMAKE_INCLUDE_CURRENT_DIR_IN_INTERFACE', 'CMAKE_INSTALL_NAME_DIR',
	'CMAKE_INSTALL_REMOVE_ENVIRONMENT_RPATH', 'CMAKE_INSTALL_RPATH',
	'CMAKE_INSTALL_RPATH_USE_LINK_PATH', 'CMAKE_INTERPROCEDURAL_OPTIMIZATION',
	-- 'CMAKE_INTERPROCEDURAL_OPTIMIZATION_<CONFIG>',
	'CMAKE_IOS_INSTALL_COMBINED',
	-- 'CMAKE_<LANG>_CLANG_TIDY',
	-- 'CMAKE_<LANG>_COMPILER_LAUNCHER',
	-- 'CMAKE_<LANG>_CPPCHECK',
	-- 'CMAKE_<LANG>_CPPLINT',
	-- 'CMAKE_<LANG>_INCLUDE_WHAT_YOU_USE',
	-- 'CMAKE_<LANG>_LINK_GROUP_USING_<FEATURE>',
	-- 'CMAKE_<LANG>_LINK_GROUP_USING_<FEATURE>_SUPPORTED',
	-- 'CMAKE_<LANG>_LINK_LIBRARY_FILE_FLAG',
	-- 'CMAKE_<LANG>_LINK_LIBRARY_FLAG',
	-- 'CMAKE_<LANG>_LINK_LIBRARY_USING_<FEATURE>',
	-- 'CMAKE_<LANG>_LINK_LIBRARY_USING_<FEATURE>_SUPPORTED',
	-- 'CMAKE_<LANG>_LINK_WHAT_YOU_USE_FLAG',
	-- 'CMAKE_<LANG>_LINKER_LAUNCHER',
	-- 'CMAKE_<LANG>_VISIBILITY_PRESET',
	'CMAKE_LIBRARY_OUTPUT_DIRECTORY',
	-- 'CMAKE_LIBRARY_OUTPUT_DIRECTORY_<CONFIG>',
	'CMAKE_LIBRARY_PATH_FLAG', 'CMAKE_LINK_DEF_FILE_FLAG', 'CMAKE_LINK_DEPENDS_NO_SHARED',
	-- 'CMAKE_LINK_GROUP_USING_<FEATURE>',
	-- 'CMAKE_LINK_GROUP_USING_<FEATURE>_SUPPORTED',
	'CMAKE_LINK_INTERFACE_LIBRARIES', 'CMAKE_LINK_LIBRARY_FILE_FLAG', 'CMAKE_LINK_LIBRARY_FLAG',
	-- 'CMAKE_LINK_LIBRARY_USING_<FEATURE>',
	-- 'CMAKE_LINK_LIBRARY_USING_<FEATURE>_SUPPORTED',
	'CMAKE_LINK_WHAT_YOU_USE', 'CMAKE_LINK_WHAT_YOU_USE_CHECK', 'CMAKE_MACOSX_BUNDLE',
	'CMAKE_MACOSX_RPATH',
	-- 'CMAKE_MAP_IMPORTED_CONFIG_<CONFIG>',
	'CMAKE_MODULE_LINKER_FLAGS',
	-- 'CMAKE_MODULE_LINKER_FLAGS_<CONFIG>',
	-- 'CMAKE_MODULE_LINKER_FLAGS_<CONFIG>_INIT',
	'CMAKE_MODULE_LINKER_FLAGS_INIT', 'CMAKE_MSVC_DEBUG_INFORMATION_FORMAT',
	'CMAKE_MSVC_RUNTIME_LIBRARY', 'CMAKE_MSVCIDE_RUN_PATH', 'CMAKE_NINJA_OUTPUT_PATH_PREFIX',
	'CMAKE_NO_BUILTIN_CHRPATH', 'CMAKE_NO_SYSTEM_FROM_IMPORTED', 'CMAKE_OPTIMIZE_DEPENDENCIES',
	'CMAKE_OSX_ARCHITECTURES', 'CMAKE_OSX_DEPLOYMENT_TARGET', 'CMAKE_OSX_SYSROOT',
	'CMAKE_PCH_INSTANTIATE_TEMPLATES', 'CMAKE_PCH_WARN_INVALID', 'CMAKE_PDB_OUTPUT_DIRECTORY',
	-- 'CMAKE_PDB_OUTPUT_DIRECTORY_<CONFIG>',
	'CMAKE_PLATFORM_NO_VERSIONED_SONAME', 'CMAKE_POSITION_INDEPENDENT_CODE',
	'CMAKE_RUNTIME_OUTPUT_DIRECTORY',
	-- 'CMAKE_RUNTIME_OUTPUT_DIRECTORY_<CONFIG>',
	'CMAKE_SHARED_LINKER_FLAGS',
	-- 'CMAKE_SHARED_LINKER_FLAGS_<CONFIG>',
	-- 'CMAKE_SHARED_LINKER_FLAGS_<CONFIG>_INIT',
	'CMAKE_SHARED_LINKER_FLAGS_INIT', 'CMAKE_SKIP_BUILD_RPATH', 'CMAKE_SKIP_INSTALL_RPATH',
	'CMAKE_STATIC_LINKER_FLAGS',
	-- 'CMAKE_STATIC_LINKER_FLAGS_<CONFIG>',
	-- 'CMAKE_STATIC_LINKER_FLAGS_<CONFIG>_INIT',
	'CMAKE_STATIC_LINKER_FLAGS_INIT', 'CMAKE_TASKING_TOOLSET', 'CMAKE_TRY_COMPILE_CONFIGURATION',
	'CMAKE_TRY_COMPILE_NO_PLATFORM_VARIABLES', 'CMAKE_TRY_COMPILE_PLATFORM_VARIABLES',
	'CMAKE_TRY_COMPILE_TARGET_TYPE', 'CMAKE_UNITY_BUILD', 'CMAKE_UNITY_BUILD_BATCH_SIZE',
	'CMAKE_UNITY_BUILD_UNIQUE_ID', 'CMAKE_USE_RELATIVE_PATHS', 'CMAKE_VERIFY_INTERFACE_HEADER_SETS',
	'CMAKE_VISIBILITY_INLINES_HIDDEN', 'CMAKE_VS_GLOBALS',
	'CMAKE_VS_INCLUDE_INSTALL_TO_DEFAULT_BUILD', 'CMAKE_VS_INCLUDE_PACKAGE_TO_DEFAULT_BUILD',
	'CMAKE_VS_JUST_MY_CODE_DEBUGGING', 'CMAKE_VS_NO_COMPILE_BATCHING',
	'CMAKE_VS_SDK_EXCLUDE_DIRECTORIES', 'CMAKE_VS_SDK_EXECUTABLE_DIRECTORIES',
	'CMAKE_VS_SDK_INCLUDE_DIRECTORIES', 'CMAKE_VS_SDK_LIBRARY_DIRECTORIES',
	'CMAKE_VS_SDK_LIBRARY_WINRT_DIRECTORIES', 'CMAKE_VS_SDK_REFERENCE_DIRECTORIES',
	'CMAKE_VS_SDK_SOURCE_DIRECTORIES', 'CMAKE_VS_WINRT_BY_DEFAULT', 'CMAKE_WATCOM_RUNTIME_LIBRARY',
	'CMAKE_WIN32_EXECUTABLE', 'CMAKE_WINDOWS_EXPORT_ALL_SYMBOLS',
	-- 'CMAKE_XCODE_ATTRIBUTE_<an-attribute>',
	'EXECUTABLE_OUTPUT_PATH', 'LIBRARY_OUTPUT_PATH',

	-- Variables for languages.
	'CMAKE_C_COMPILE_FEATURES', 'CMAKE_C_EXTENSIONS', 'CMAKE_C_STANDARD', 'CMAKE_C_STANDARD_REQUIRED',
	'CMAKE_COMPILER_IS_GNUCC', 'CMAKE_COMPILER_IS_GNUCXX', 'CMAKE_COMPILER_IS_GNUG77',
	'CMAKE_CUDA_ARCHITECTURES', 'CMAKE_CUDA_COMPILE_FEATURES', 'CMAKE_CUDA_EXTENSIONS',
	'CMAKE_CUDA_HOST_COMPILER', 'CMAKE_CUDA_STANDARD', 'CMAKE_CUDA_STANDARD_REQUIRED',
	'CMAKE_CUDA_TOOLKIT_INCLUDE_DIRECTORIES', 'CMAKE_CXX_COMPILE_FEATURES', 'CMAKE_CXX_EXTENSIONS',
	'CMAKE_CXX_STANDARD', 'CMAKE_CXX_STANDARD_REQUIRED', 'CMAKE_Fortran_MODDIR_DEFAULT',
	'CMAKE_Fortran_MODDIR_FLAG', 'CMAKE_Fortran_MODOUT_FLAG', 'CMAKE_HIP_ARCHITECTURES',
	'CMAKE_HIP_EXTENSIONS', 'CMAKE_HIP_STANDARD', 'CMAKE_HIP_STANDARD_REQUIRED',
	'CMAKE_ISPC_HEADER_DIRECTORY', 'CMAKE_ISPC_HEADER_SUFFIX', 'CMAKE_ISPC_INSTRUCTION_SETS',
	-- 'CMAKE_<LANG>_ANDROID_TOOLCHAIN_MACHINE',
	-- 'CMAKE_<LANG>_ANDROID_TOOLCHAIN_PREFIX',
	-- 'CMAKE_<LANG>_ANDROID_TOOLCHAIN_SUFFIX',
	-- 'CMAKE_<LANG>_ARCHIVE_APPEND',
	-- 'CMAKE_<LANG>_ARCHIVE_CREATE',
	-- 'CMAKE_<LANG>_ARCHIVE_FINISH',
	-- 'CMAKE_<LANG>_BYTE_ORDER',
	-- 'CMAKE_<LANG>_COMPILE_OBJECT',
	-- 'CMAKE_<LANG>_COMPILER',
	-- 'CMAKE_<LANG>_COMPILER_EXTERNAL_TOOLCHAIN',
	-- 'CMAKE_<LANG>_COMPILER_ID',
	-- 'CMAKE_<LANG>_COMPILER_LOADED',
	-- 'CMAKE_<LANG>_COMPILER_PREDEFINES_COMMAND',
	-- 'CMAKE_<LANG>_COMPILER_TARGET',
	-- 'CMAKE_<LANG>_COMPILER_VERSION',
	-- 'CMAKE_<LANG>_CREATE_SHARED_LIBRARY',
	-- 'CMAKE_<LANG>_CREATE_SHARED_MODULE',
	-- 'CMAKE_<LANG>_CREATE_STATIC_LIBRARY',
	-- 'CMAKE_<LANG>_EXTENSIONS',
	-- 'CMAKE_<LANG>_EXTENSIONS_DEFAULT',
	-- 'CMAKE_<LANG>_FLAGS',
	-- 'CMAKE_<LANG>_FLAGS_<CONFIG>',
	-- 'CMAKE_<LANG>_FLAGS_<CONFIG>_INIT',
	-- 'CMAKE_<LANG>_FLAGS_DEBUG',
	-- 'CMAKE_<LANG>_FLAGS_DEBUG_INIT',
	-- 'CMAKE_<LANG>_FLAGS_INIT',
	-- 'CMAKE_<LANG>_FLAGS_MINSIZEREL',
	-- 'CMAKE_<LANG>_FLAGS_MINSIZEREL_INIT',
	-- 'CMAKE_<LANG>_FLAGS_RELEASE',
	-- 'CMAKE_<LANG>_FLAGS_RELEASE_INIT',
	-- 'CMAKE_<LANG>_FLAGS_RELWITHDEBINFO',
	-- 'CMAKE_<LANG>_FLAGS_RELWITHDEBINFO_INIT',
	-- 'CMAKE_<LANG>_IGNORE_EXTENSIONS',
	-- 'CMAKE_<LANG>_IMPLICIT_INCLUDE_DIRECTORIES',
	-- 'CMAKE_<LANG>_IMPLICIT_LINK_DIRECTORIES',
	-- 'CMAKE_<LANG>_IMPLICIT_LINK_FRAMEWORK_DIRECTORIES',
	-- 'CMAKE_<LANG>_IMPLICIT_LINK_LIBRARIES',
	-- 'CMAKE_<LANG>_LIBRARY_ARCHITECTURE',
	-- 'CMAKE_<LANG>_LINK_EXECUTABLE',
	-- 'CMAKE_<LANG>_LINKER_WRAPPER_FLAG',
	-- 'CMAKE_<LANG>_LINKER_WRAPPER_FLAG_SEP',
	-- 'CMAKE_<LANG>_OUTPUT_EXTENSION',
	-- 'CMAKE_<LANG>_SIMULATE_ID',
	-- 'CMAKE_<LANG>_SIMULATE_VERSION',
	-- 'CMAKE_<LANG>_SIZEOF_DATA_PTR',
	-- 'CMAKE_<LANG>_SOURCE_FILE_EXTENSIONS',
	-- 'CMAKE_<LANG>_STANDARD',
	-- 'CMAKE_<LANG>_STANDARD_DEFAULT',
	-- 'CMAKE_<LANG>_STANDARD_INCLUDE_DIRECTORIES',
	-- 'CMAKE_<LANG>_STANDARD_LIBRARIES',
	-- 'CMAKE_<LANG>_STANDARD_REQUIRED',
	'CMAKE_OBJC_EXTENSIONS', 'CMAKE_OBJC_STANDARD', 'CMAKE_OBJC_STANDARD_REQUIRED',
	'CMAKE_OBJCXX_EXTENSIONS', 'CMAKE_OBJCXX_STANDARD', 'CMAKE_OBJCXX_STANDARD_REQUIRED',
	'CMAKE_Swift_LANGUAGE_VERSION',
	-- 'CMAKE_USER_MAKE_RULES_OVERRIDE_<LANG>',
	
	-- Variables for CTest.
	'CTEST_BINARY_DIRECTORY', 'CTEST_BUILD_COMMAND', 'CTEST_BUILD_NAME', 'CTEST_BZR_COMMAND',
	'CTEST_BZR_UPDATE_OPTIONS', 'CTEST_CHANGE_ID', 'CTEST_CHECKOUT_COMMAND',
	'CTEST_CONFIGURATION_TYPE', 'CTEST_CONFIGURE_COMMAND', 'CTEST_COVERAGE_COMMAND',
	'CTEST_COVERAGE_EXTRA_FLAGS', 'CTEST_CURL_OPTIONS', 'CTEST_CUSTOM_COVERAGE_EXCLUDE',
	'CTEST_CUSTOM_ERROR_EXCEPTION', 'CTEST_CUSTOM_ERROR_MATCH', 'CTEST_CUSTOM_ERROR_POST_CONTEXT',
	'CTEST_CUSTOM_ERROR_PRE_CONTEXT', 'CTEST_CUSTOM_MAXIMUM_FAILED_TEST_OUTPUT_SIZE',
	'CTEST_CUSTOM_MAXIMUM_NUMBER_OF_ERRORS', 'CTEST_CUSTOM_MAXIMUM_NUMBER_OF_WARNINGS',
	'CTEST_CUSTOM_MAXIMUM_PASSED_TEST_OUTPUT_SIZE', 'CTEST_CUSTOM_MEMCHECK_IGNORE',
	'CTEST_CUSTOM_POST_MEMCHECK', 'CTEST_CUSTOM_POST_TEST', 'CTEST_CUSTOM_PRE_MEMCHECK',
	'CTEST_CUSTOM_PRE_TEST', 'CTEST_CUSTOM_TEST_OUTPUT_TRUNCATION', 'CTEST_CUSTOM_TESTS_IGNORE',
	'CTEST_CUSTOM_WARNING_EXCEPTION', 'CTEST_CUSTOM_WARNING_MATCH', 'CTEST_CVS_CHECKOUT',
	'CTEST_CVS_COMMAND', 'CTEST_CVS_UPDATE_OPTIONS', 'CTEST_DROP_LOCATION', 'CTEST_DROP_METHOD',
	'CTEST_DROP_SITE', 'CTEST_DROP_SITE_CDASH', 'CTEST_DROP_SITE_PASSWORD', 'CTEST_DROP_SITE_USER',
	'CTEST_EXTRA_COVERAGE_GLOB', 'CTEST_GIT_COMMAND', 'CTEST_GIT_INIT_SUBMODULES',
	'CTEST_GIT_UPDATE_CUSTOM', 'CTEST_GIT_UPDATE_OPTIONS', 'CTEST_HG_COMMAND',
	'CTEST_HG_UPDATE_OPTIONS', 'CTEST_LABELS_FOR_SUBPROJECTS', 'CTEST_MEMORYCHECK_COMMAND',
	'CTEST_MEMORYCHECK_COMMAND_OPTIONS', 'CTEST_MEMORYCHECK_SANITIZER_OPTIONS',
	'CTEST_MEMORYCHECK_SUPPRESSIONS_FILE', 'CTEST_MEMORYCHECK_TYPE', 'CTEST_NIGHTLY_START_TIME',
	'CTEST_P4_CLIENT', 'CTEST_P4_COMMAND', 'CTEST_P4_OPTIONS', 'CTEST_P4_UPDATE_OPTIONS',
	'CTEST_RESOURCE_SPEC_FILE', 'CTEST_RUN_CURRENT_SCRIPT', 'CTEST_SCP_COMMAND',
	'CTEST_SCRIPT_DIRECTORY', 'CTEST_SITE', 'CTEST_SOURCE_DIRECTORY',
	'CTEST_SUBMIT_INACTIVITY_TIMEOUT', 'CTEST_SUBMIT_URL', 'CTEST_SVN_COMMAND', 'CTEST_SVN_OPTIONS',
	'CTEST_SVN_UPDATE_OPTIONS', 'CTEST_TEST_LOAD', 'CTEST_TEST_TIMEOUT', 'CTEST_TRIGGER_SITE',
	'CTEST_UPDATE_COMMAND', 'CTEST_UPDATE_OPTIONS', 'CTEST_UPDATE_VERSION_ONLY',
	'CTEST_UPDATE_VERSION_OVERRIDE', 'CTEST_USE_LAUNCHERS',

	-- Variables for CPack.
	'CPACK_ABSOLUTE_DESTINATION_FILES', 'CPACK_COMPONENT_INCLUDE_TOPLEVEL_DIRECTORY',
	'CPACK_CUSTOM_INSTALL_VARIABLES', 'CPACK_ERROR_ON_ABSOLUTE_INSTALL_DESTINATION',
	'CPACK_INCLUDE_TOPLEVEL_DIRECTORY', 'CPACK_INSTALL_DEFAULT_DIRECTORY_PERMISSIONS',
	'CPACK_PACKAGING_INSTALL_PREFIX', 'CPACK_SET_DESTDIR',
	'CPACK_WARN_ON_ABSOLUTE_INSTALL_DESTINATION'
})

lex:set_word_list(lexer.PREPROCESSOR, {
	'IF', 'BOOL', -- conditional
	'AND', 'OR', 'NOT', -- logical
	'STREQUAL', 'EQUAL', -- string comparison
	'VERSION_LESS', 'VERSION_GREATER', 'VERSION_EQUAL', 'VERSION_LESS_EQUAL', 'VERSION_GREATER_EQUAL', -- version comparison
	'LOWER_CASE', 'UPPER_CASE', 'MAKE_C_IDENTIFIER', -- string transformations
	'IN_LIST', 'JOIN', 'REMOVE_DUPLICATES', 'FILTER', -- list expressions
	'PATH_EQUAL', 'PATH', -- path expressions (note: cannot use ':' parts)
	'SHELL_PATH', -- shell paths
	'CONFIG', 'OUTPUT_CONFIG', 'COMMAND_CONFIG', -- configuration expressions
	'PLATFORM_ID', -- platform
	'C_COMPILER_VERSION', 'CXX_COMPILER_VERSION', 'CUDA_COMPILER_VERSION', 'OBJC_COMPILER_VERSION',
	'OBJCXX_COMPILER_VERSION', 'Fortran_COMPILER_VERSION', 'HIP_COMPILER_VERSION',
	'ISPC_COMPILER_VERSION', -- compiler version
	'C_COMPILER_ID', 'CXX_COMPILER_ID', 'CUDA_COMPILER_ID', 'OBJC_COMPILER_ID', 'OBJCXX_COMPILER_ID',
	'Fortran_COMPILER_ID', 'HIP_COMPILER_ID', 'ISPC_COMPILER_ID', 'COMPILE_LANGUAGE',
	'COMPILE_LANG_AND_ID', 'COMPILE_FEATURES', -- compile features
	'LINK_LANGUAGE', 'LINK_LANG_AND_ID', -- linker language and ID
	'LINK_LIBRARY', 'LINK_GROUP', -- link features
	'LINK_ONLY', 'DEVICE_LINK', 'HOST_LINK', -- link context
	'TARGET_EXISTS', 'TARGET_NAME_IF_EXISTS', 'TARGET_NAME', 'TARGET_PROPERTY', 'TARGET_OBJECTS',
	'TARGET_POLICY', 'TARGET_FILE', 'TARGET_FILE_BASE_NAME', 'TARGET_FILE_PREFIX',
	'TARGET_FILE_SUFFIX', 'TARGET_FILE_NAME', 'TARGET_FILE_DIR', 'TARGET_LINKER_FILE',
	'TARGET_LINKER_FILE_BASE_NAME', 'TARGET_LINKER_FILE_PREFIX', 'TARGET_LINKER_FILE_SUFFIX',
	'TARGET_LINKER_FILE_NAME', 'TARGET_LINKER_FILE_DIR', 'TARGET_SONAME_FILE',
	'TARGET_SONAME_FILE_NAME', 'TARGET_SONAME_FILE_DIR', 'TARGET_PDB_FILE',
	'TARGET_PDB_FILE_BASE_NAME', 'TARGET_PDB_FILE_NAME', 'TARGET_PDB_FILE_DIR', 'TARGET_BUNDLE_DIR',
	'TARGET_BUNDLE_DIR_NAME', 'TARGET_BUNDLE_CONTENT_DIR', 'TARGET_RUNTIME_DLLS', -- target-dependent expressions
	'INSTALL_INTERFACE', 'BUILD_INTERFACE', 'INSTALL_PREFIX', -- export and install expressions
	'GENEX_EVAL', 'TARGET_GENEX_EVAL', -- multi-level expression evaluation
	'ANGLE-R', 'COMMA', 'SEMICOLON' -- escaped characters
})

lex:set_word_list(lexer.OPERATOR, {
	-- Unary test.
	'EXISTS', 'COMMAND', 'DEFINED',
	-- Binary test.
	'EQUAL', 'LESS', 'LESS_EQUAL', 'GREATER', 'GREATER_EQUAL', 'STREQUAL', 'STRLESS', 'STRLESS_EQUAL',
	'STRGREATER', 'STRGREATER_EQUAL', 'VERSION_EQUAL', 'VERSION_LESS', 'VERSION_LESS_EQUAL',
	'VERSION_GREATER', 'VERSION_GREATER_EQUAL', 'PATH_EQUAL', 'MATCHES',
	-- Logical.
	'NOT', 'AND', 'OR'
})

lexer.property['scintillua.comment'] = '#'

return lex
