/*
 *  yosys -- Yosys Open SYnthesis Suite
 *
 *  Copyright (C) 2012  Claire Xenia Wolf <claire@yosyshq.com>
 *
 *  Permission to use, copy, modify, and/or distribute this software for any
 *  purpose with or without fee is hereby granted, provided that the above
 *  copyright notice and this permission notice appear in all copies.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 *  WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 *  ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 *  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 *  ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 *  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "kernel/yosys.h"

USING_YOSYS_NAMESPACE
PRIVATE_NAMESPACE_BEGIN

static void publish(RTLIL::IdString& id) {
	if (id.begins_with("$")) {
		log_debug("publishing %s\n", id.c_str());
		id = "\\" + id.str();
		log_debug("published %s\n", id.c_str());
	}
}

static void publish_design(RTLIL::Design* design) {
	auto saved_modules = design->modules_;
	design->modules_.clear();
	for (auto& [name, mod] : saved_modules) {
		publish(mod->name);
		design->modules_[mod->name] = mod;
		for (auto* cell : mod->cells()) {
			publish(cell->type);
		}
	}
}


struct ChtypePass : public Pass {
	ChtypePass() : Pass("chtype", "change type of cells in the design") { }
	void help() override
	{
		//   |---v---|---v---|---v---|---v---|---v---|---v---|---v---|---v---|---v---|---v---|
		log("\n");
		log("    chtype [options] [selection]\n");
		log("\n");
		log("Change the types of cells in the design.\n");
		log("\n");
		log("    -set <type>\n");
		log("        set the cell type to the given type\n");
		log("\n");
		log("    -map <old_type> <new_type>\n");
		log("        change cells types that match <old_type> to <new_type>\n");
		log("\n");
		log("    -publish_icells\n");
		log("        change internal cells types to public types\n");
		log("\n");
		log("\n");
	}
	void execute(std::vector<std::string> args, RTLIL::Design *design) override
	{
		IdString set_type;
		dict<IdString, IdString> map_types;
		bool publish_mode = false;

		size_t argidx;
		for (argidx = 1; argidx < args.size(); argidx++)
		{
			if (set_type == IdString() && args[argidx] == "-set" && argidx+1 < args.size()) {
				set_type = RTLIL::escape_id(args[++argidx]);
				continue;
			}
			if (args[argidx] == "-map" && argidx+2 < args.size()) {
				IdString old_type = RTLIL::escape_id(args[++argidx]);
				IdString new_type = RTLIL::escape_id(args[++argidx]);
				map_types[old_type] = new_type;
				continue;
			}
			if (args[argidx] == "-publish_icells") {
				publish_mode = true;
				continue;
			}
			break;
		}
		extra_args(args, argidx, design);

		if (publish_mode)
			publish_design(design);

		for (auto module : design->selected_modules())
		{
			for (auto cell : module->selected_cells())
			{
				if (map_types.count(cell->type)) {
					cell->type = map_types.at(cell->type);
					continue;
				}

				if (set_type != IdString()) {
					cell->type = set_type;
					continue;
				}
			}
		}
	}
} ChtypePass;

PRIVATE_NAMESPACE_END
